/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lightsail.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The parameters for gaining temporary access to one of your Amazon Lightsail instances.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InstanceAccessDetails implements SdkPojo, Serializable,
        ToCopyableBuilder<InstanceAccessDetails.Builder, InstanceAccessDetails> {
    private static final SdkField<String> CERT_KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("certKey").getter(getter(InstanceAccessDetails::certKey)).setter(setter(Builder::certKey))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("certKey").build()).build();

    private static final SdkField<Instant> EXPIRES_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("expiresAt").getter(getter(InstanceAccessDetails::expiresAt)).setter(setter(Builder::expiresAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("expiresAt").build()).build();

    private static final SdkField<String> IP_ADDRESS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ipAddress").getter(getter(InstanceAccessDetails::ipAddress)).setter(setter(Builder::ipAddress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ipAddress").build()).build();

    private static final SdkField<String> PASSWORD_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("password").getter(getter(InstanceAccessDetails::password)).setter(setter(Builder::password))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("password").build()).build();

    private static final SdkField<PasswordData> PASSWORD_DATA_FIELD = SdkField.<PasswordData> builder(MarshallingType.SDK_POJO)
            .memberName("passwordData").getter(getter(InstanceAccessDetails::passwordData)).setter(setter(Builder::passwordData))
            .constructor(PasswordData::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("passwordData").build()).build();

    private static final SdkField<String> PRIVATE_KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("privateKey").getter(getter(InstanceAccessDetails::privateKey)).setter(setter(Builder::privateKey))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("privateKey").build()).build();

    private static final SdkField<String> PROTOCOL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("protocol").getter(getter(InstanceAccessDetails::protocolAsString)).setter(setter(Builder::protocol))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("protocol").build()).build();

    private static final SdkField<String> INSTANCE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("instanceName").getter(getter(InstanceAccessDetails::instanceName)).setter(setter(Builder::instanceName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("instanceName").build()).build();

    private static final SdkField<String> USERNAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("username").getter(getter(InstanceAccessDetails::username)).setter(setter(Builder::username))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("username").build()).build();

    private static final SdkField<List<HostKeyAttributes>> HOST_KEYS_FIELD = SdkField
            .<List<HostKeyAttributes>> builder(MarshallingType.LIST)
            .memberName("hostKeys")
            .getter(getter(InstanceAccessDetails::hostKeys))
            .setter(setter(Builder::hostKeys))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("hostKeys").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<HostKeyAttributes> builder(MarshallingType.SDK_POJO)
                                            .constructor(HostKeyAttributes::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CERT_KEY_FIELD,
            EXPIRES_AT_FIELD, IP_ADDRESS_FIELD, PASSWORD_FIELD, PASSWORD_DATA_FIELD, PRIVATE_KEY_FIELD, PROTOCOL_FIELD,
            INSTANCE_NAME_FIELD, USERNAME_FIELD, HOST_KEYS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String certKey;

    private final Instant expiresAt;

    private final String ipAddress;

    private final String password;

    private final PasswordData passwordData;

    private final String privateKey;

    private final String protocol;

    private final String instanceName;

    private final String username;

    private final List<HostKeyAttributes> hostKeys;

    private InstanceAccessDetails(BuilderImpl builder) {
        this.certKey = builder.certKey;
        this.expiresAt = builder.expiresAt;
        this.ipAddress = builder.ipAddress;
        this.password = builder.password;
        this.passwordData = builder.passwordData;
        this.privateKey = builder.privateKey;
        this.protocol = builder.protocol;
        this.instanceName = builder.instanceName;
        this.username = builder.username;
        this.hostKeys = builder.hostKeys;
    }

    /**
     * <p>
     * For SSH access, the public key to use when accessing your instance For OpenSSH clients (e.g., command line SSH),
     * you should save this value to <code>tempkey-cert.pub</code>.
     * </p>
     * 
     * @return For SSH access, the public key to use when accessing your instance For OpenSSH clients (e.g., command
     *         line SSH), you should save this value to <code>tempkey-cert.pub</code>.
     */
    public final String certKey() {
        return certKey;
    }

    /**
     * <p>
     * For SSH access, the date on which the temporary keys expire.
     * </p>
     * 
     * @return For SSH access, the date on which the temporary keys expire.
     */
    public final Instant expiresAt() {
        return expiresAt;
    }

    /**
     * <p>
     * The public IP address of the Amazon Lightsail instance.
     * </p>
     * 
     * @return The public IP address of the Amazon Lightsail instance.
     */
    public final String ipAddress() {
        return ipAddress;
    }

    /**
     * <p>
     * For RDP access, the password for your Amazon Lightsail instance. Password will be an empty string if the password
     * for your new instance is not ready yet. When you create an instance, it can take up to 15 minutes for the
     * instance to be ready.
     * </p>
     * <note>
     * <p>
     * If you create an instance using any key pair other than the default (<code>LightsailDefaultKeyPair</code>),
     * <code>password</code> will always be an empty string.
     * </p>
     * <p>
     * If you change the Administrator password on the instance, Lightsail will continue to return the original password
     * value. When accessing the instance using RDP, you need to manually enter the Administrator password after
     * changing it from the default.
     * </p>
     * </note>
     * 
     * @return For RDP access, the password for your Amazon Lightsail instance. Password will be an empty string if the
     *         password for your new instance is not ready yet. When you create an instance, it can take up to 15
     *         minutes for the instance to be ready.</p> <note>
     *         <p>
     *         If you create an instance using any key pair other than the default (<code>LightsailDefaultKeyPair</code>
     *         ), <code>password</code> will always be an empty string.
     *         </p>
     *         <p>
     *         If you change the Administrator password on the instance, Lightsail will continue to return the original
     *         password value. When accessing the instance using RDP, you need to manually enter the Administrator
     *         password after changing it from the default.
     *         </p>
     */
    public final String password() {
        return password;
    }

    /**
     * <p>
     * For a Windows Server-based instance, an object with the data you can use to retrieve your password. This is only
     * needed if <code>password</code> is empty and the instance is not new (and therefore the password is not ready
     * yet). When you create an instance, it can take up to 15 minutes for the instance to be ready.
     * </p>
     * 
     * @return For a Windows Server-based instance, an object with the data you can use to retrieve your password. This
     *         is only needed if <code>password</code> is empty and the instance is not new (and therefore the password
     *         is not ready yet). When you create an instance, it can take up to 15 minutes for the instance to be
     *         ready.
     */
    public final PasswordData passwordData() {
        return passwordData;
    }

    /**
     * <p>
     * For SSH access, the temporary private key. For OpenSSH clients (e.g., command line SSH), you should save this
     * value to <code>tempkey</code>).
     * </p>
     * 
     * @return For SSH access, the temporary private key. For OpenSSH clients (e.g., command line SSH), you should save
     *         this value to <code>tempkey</code>).
     */
    public final String privateKey() {
        return privateKey;
    }

    /**
     * <p>
     * The protocol for these Amazon Lightsail instance access details.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #protocol} will
     * return {@link InstanceAccessProtocol#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #protocolAsString}.
     * </p>
     * 
     * @return The protocol for these Amazon Lightsail instance access details.
     * @see InstanceAccessProtocol
     */
    public final InstanceAccessProtocol protocol() {
        return InstanceAccessProtocol.fromValue(protocol);
    }

    /**
     * <p>
     * The protocol for these Amazon Lightsail instance access details.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #protocol} will
     * return {@link InstanceAccessProtocol#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #protocolAsString}.
     * </p>
     * 
     * @return The protocol for these Amazon Lightsail instance access details.
     * @see InstanceAccessProtocol
     */
    public final String protocolAsString() {
        return protocol;
    }

    /**
     * <p>
     * The name of this Amazon Lightsail instance.
     * </p>
     * 
     * @return The name of this Amazon Lightsail instance.
     */
    public final String instanceName() {
        return instanceName;
    }

    /**
     * <p>
     * The user name to use when logging in to the Amazon Lightsail instance.
     * </p>
     * 
     * @return The user name to use when logging in to the Amazon Lightsail instance.
     */
    public final String username() {
        return username;
    }

    /**
     * Returns true if the HostKeys property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasHostKeys() {
        return hostKeys != null && !(hostKeys instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Describes the public SSH host keys or the RDP certificate.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasHostKeys()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Describes the public SSH host keys or the RDP certificate.
     */
    public final List<HostKeyAttributes> hostKeys() {
        return hostKeys;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(certKey());
        hashCode = 31 * hashCode + Objects.hashCode(expiresAt());
        hashCode = 31 * hashCode + Objects.hashCode(ipAddress());
        hashCode = 31 * hashCode + Objects.hashCode(password());
        hashCode = 31 * hashCode + Objects.hashCode(passwordData());
        hashCode = 31 * hashCode + Objects.hashCode(privateKey());
        hashCode = 31 * hashCode + Objects.hashCode(protocolAsString());
        hashCode = 31 * hashCode + Objects.hashCode(instanceName());
        hashCode = 31 * hashCode + Objects.hashCode(username());
        hashCode = 31 * hashCode + Objects.hashCode(hasHostKeys() ? hostKeys() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InstanceAccessDetails)) {
            return false;
        }
        InstanceAccessDetails other = (InstanceAccessDetails) obj;
        return Objects.equals(certKey(), other.certKey()) && Objects.equals(expiresAt(), other.expiresAt())
                && Objects.equals(ipAddress(), other.ipAddress()) && Objects.equals(password(), other.password())
                && Objects.equals(passwordData(), other.passwordData()) && Objects.equals(privateKey(), other.privateKey())
                && Objects.equals(protocolAsString(), other.protocolAsString())
                && Objects.equals(instanceName(), other.instanceName()) && Objects.equals(username(), other.username())
                && hasHostKeys() == other.hasHostKeys() && Objects.equals(hostKeys(), other.hostKeys());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("InstanceAccessDetails").add("CertKey", certKey()).add("ExpiresAt", expiresAt())
                .add("IpAddress", ipAddress()).add("Password", password()).add("PasswordData", passwordData())
                .add("PrivateKey", privateKey()).add("Protocol", protocolAsString()).add("InstanceName", instanceName())
                .add("Username", username()).add("HostKeys", hasHostKeys() ? hostKeys() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "certKey":
            return Optional.ofNullable(clazz.cast(certKey()));
        case "expiresAt":
            return Optional.ofNullable(clazz.cast(expiresAt()));
        case "ipAddress":
            return Optional.ofNullable(clazz.cast(ipAddress()));
        case "password":
            return Optional.ofNullable(clazz.cast(password()));
        case "passwordData":
            return Optional.ofNullable(clazz.cast(passwordData()));
        case "privateKey":
            return Optional.ofNullable(clazz.cast(privateKey()));
        case "protocol":
            return Optional.ofNullable(clazz.cast(protocolAsString()));
        case "instanceName":
            return Optional.ofNullable(clazz.cast(instanceName()));
        case "username":
            return Optional.ofNullable(clazz.cast(username()));
        case "hostKeys":
            return Optional.ofNullable(clazz.cast(hostKeys()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<InstanceAccessDetails, T> g) {
        return obj -> g.apply((InstanceAccessDetails) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InstanceAccessDetails> {
        /**
         * <p>
         * For SSH access, the public key to use when accessing your instance For OpenSSH clients (e.g., command line
         * SSH), you should save this value to <code>tempkey-cert.pub</code>.
         * </p>
         * 
         * @param certKey
         *        For SSH access, the public key to use when accessing your instance For OpenSSH clients (e.g., command
         *        line SSH), you should save this value to <code>tempkey-cert.pub</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder certKey(String certKey);

        /**
         * <p>
         * For SSH access, the date on which the temporary keys expire.
         * </p>
         * 
         * @param expiresAt
         *        For SSH access, the date on which the temporary keys expire.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder expiresAt(Instant expiresAt);

        /**
         * <p>
         * The public IP address of the Amazon Lightsail instance.
         * </p>
         * 
         * @param ipAddress
         *        The public IP address of the Amazon Lightsail instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipAddress(String ipAddress);

        /**
         * <p>
         * For RDP access, the password for your Amazon Lightsail instance. Password will be an empty string if the
         * password for your new instance is not ready yet. When you create an instance, it can take up to 15 minutes
         * for the instance to be ready.
         * </p>
         * <note>
         * <p>
         * If you create an instance using any key pair other than the default (<code>LightsailDefaultKeyPair</code>),
         * <code>password</code> will always be an empty string.
         * </p>
         * <p>
         * If you change the Administrator password on the instance, Lightsail will continue to return the original
         * password value. When accessing the instance using RDP, you need to manually enter the Administrator password
         * after changing it from the default.
         * </p>
         * </note>
         * 
         * @param password
         *        For RDP access, the password for your Amazon Lightsail instance. Password will be an empty string if
         *        the password for your new instance is not ready yet. When you create an instance, it can take up to 15
         *        minutes for the instance to be ready.</p> <note>
         *        <p>
         *        If you create an instance using any key pair other than the default (
         *        <code>LightsailDefaultKeyPair</code>), <code>password</code> will always be an empty string.
         *        </p>
         *        <p>
         *        If you change the Administrator password on the instance, Lightsail will continue to return the
         *        original password value. When accessing the instance using RDP, you need to manually enter the
         *        Administrator password after changing it from the default.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder password(String password);

        /**
         * <p>
         * For a Windows Server-based instance, an object with the data you can use to retrieve your password. This is
         * only needed if <code>password</code> is empty and the instance is not new (and therefore the password is not
         * ready yet). When you create an instance, it can take up to 15 minutes for the instance to be ready.
         * </p>
         * 
         * @param passwordData
         *        For a Windows Server-based instance, an object with the data you can use to retrieve your password.
         *        This is only needed if <code>password</code> is empty and the instance is not new (and therefore the
         *        password is not ready yet). When you create an instance, it can take up to 15 minutes for the instance
         *        to be ready.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder passwordData(PasswordData passwordData);

        /**
         * <p>
         * For a Windows Server-based instance, an object with the data you can use to retrieve your password. This is
         * only needed if <code>password</code> is empty and the instance is not new (and therefore the password is not
         * ready yet). When you create an instance, it can take up to 15 minutes for the instance to be ready.
         * </p>
         * This is a convenience that creates an instance of the {@link PasswordData.Builder} avoiding the need to
         * create one manually via {@link PasswordData#builder()}.
         *
         * When the {@link Consumer} completes, {@link PasswordData.Builder#build()} is called immediately and its
         * result is passed to {@link #passwordData(PasswordData)}.
         * 
         * @param passwordData
         *        a consumer that will call methods on {@link PasswordData.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #passwordData(PasswordData)
         */
        default Builder passwordData(Consumer<PasswordData.Builder> passwordData) {
            return passwordData(PasswordData.builder().applyMutation(passwordData).build());
        }

        /**
         * <p>
         * For SSH access, the temporary private key. For OpenSSH clients (e.g., command line SSH), you should save this
         * value to <code>tempkey</code>).
         * </p>
         * 
         * @param privateKey
         *        For SSH access, the temporary private key. For OpenSSH clients (e.g., command line SSH), you should
         *        save this value to <code>tempkey</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder privateKey(String privateKey);

        /**
         * <p>
         * The protocol for these Amazon Lightsail instance access details.
         * </p>
         * 
         * @param protocol
         *        The protocol for these Amazon Lightsail instance access details.
         * @see InstanceAccessProtocol
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceAccessProtocol
         */
        Builder protocol(String protocol);

        /**
         * <p>
         * The protocol for these Amazon Lightsail instance access details.
         * </p>
         * 
         * @param protocol
         *        The protocol for these Amazon Lightsail instance access details.
         * @see InstanceAccessProtocol
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceAccessProtocol
         */
        Builder protocol(InstanceAccessProtocol protocol);

        /**
         * <p>
         * The name of this Amazon Lightsail instance.
         * </p>
         * 
         * @param instanceName
         *        The name of this Amazon Lightsail instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceName(String instanceName);

        /**
         * <p>
         * The user name to use when logging in to the Amazon Lightsail instance.
         * </p>
         * 
         * @param username
         *        The user name to use when logging in to the Amazon Lightsail instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder username(String username);

        /**
         * <p>
         * Describes the public SSH host keys or the RDP certificate.
         * </p>
         * 
         * @param hostKeys
         *        Describes the public SSH host keys or the RDP certificate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hostKeys(Collection<HostKeyAttributes> hostKeys);

        /**
         * <p>
         * Describes the public SSH host keys or the RDP certificate.
         * </p>
         * 
         * @param hostKeys
         *        Describes the public SSH host keys or the RDP certificate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hostKeys(HostKeyAttributes... hostKeys);

        /**
         * <p>
         * Describes the public SSH host keys or the RDP certificate.
         * </p>
         * This is a convenience that creates an instance of the {@link List<HostKeyAttributes>.Builder} avoiding the
         * need to create one manually via {@link List<HostKeyAttributes>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<HostKeyAttributes>.Builder#build()} is called immediately
         * and its result is passed to {@link #hostKeys(List<HostKeyAttributes>)}.
         * 
         * @param hostKeys
         *        a consumer that will call methods on {@link List<HostKeyAttributes>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #hostKeys(List<HostKeyAttributes>)
         */
        Builder hostKeys(Consumer<HostKeyAttributes.Builder>... hostKeys);
    }

    static final class BuilderImpl implements Builder {
        private String certKey;

        private Instant expiresAt;

        private String ipAddress;

        private String password;

        private PasswordData passwordData;

        private String privateKey;

        private String protocol;

        private String instanceName;

        private String username;

        private List<HostKeyAttributes> hostKeys = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(InstanceAccessDetails model) {
            certKey(model.certKey);
            expiresAt(model.expiresAt);
            ipAddress(model.ipAddress);
            password(model.password);
            passwordData(model.passwordData);
            privateKey(model.privateKey);
            protocol(model.protocol);
            instanceName(model.instanceName);
            username(model.username);
            hostKeys(model.hostKeys);
        }

        public final String getCertKey() {
            return certKey;
        }

        public final void setCertKey(String certKey) {
            this.certKey = certKey;
        }

        @Override
        @Transient
        public final Builder certKey(String certKey) {
            this.certKey = certKey;
            return this;
        }

        public final Instant getExpiresAt() {
            return expiresAt;
        }

        public final void setExpiresAt(Instant expiresAt) {
            this.expiresAt = expiresAt;
        }

        @Override
        @Transient
        public final Builder expiresAt(Instant expiresAt) {
            this.expiresAt = expiresAt;
            return this;
        }

        public final String getIpAddress() {
            return ipAddress;
        }

        public final void setIpAddress(String ipAddress) {
            this.ipAddress = ipAddress;
        }

        @Override
        @Transient
        public final Builder ipAddress(String ipAddress) {
            this.ipAddress = ipAddress;
            return this;
        }

        public final String getPassword() {
            return password;
        }

        public final void setPassword(String password) {
            this.password = password;
        }

        @Override
        @Transient
        public final Builder password(String password) {
            this.password = password;
            return this;
        }

        public final PasswordData.Builder getPasswordData() {
            return passwordData != null ? passwordData.toBuilder() : null;
        }

        public final void setPasswordData(PasswordData.BuilderImpl passwordData) {
            this.passwordData = passwordData != null ? passwordData.build() : null;
        }

        @Override
        @Transient
        public final Builder passwordData(PasswordData passwordData) {
            this.passwordData = passwordData;
            return this;
        }

        public final String getPrivateKey() {
            return privateKey;
        }

        public final void setPrivateKey(String privateKey) {
            this.privateKey = privateKey;
        }

        @Override
        @Transient
        public final Builder privateKey(String privateKey) {
            this.privateKey = privateKey;
            return this;
        }

        public final String getProtocol() {
            return protocol;
        }

        public final void setProtocol(String protocol) {
            this.protocol = protocol;
        }

        @Override
        @Transient
        public final Builder protocol(String protocol) {
            this.protocol = protocol;
            return this;
        }

        @Override
        @Transient
        public final Builder protocol(InstanceAccessProtocol protocol) {
            this.protocol(protocol == null ? null : protocol.toString());
            return this;
        }

        public final String getInstanceName() {
            return instanceName;
        }

        public final void setInstanceName(String instanceName) {
            this.instanceName = instanceName;
        }

        @Override
        @Transient
        public final Builder instanceName(String instanceName) {
            this.instanceName = instanceName;
            return this;
        }

        public final String getUsername() {
            return username;
        }

        public final void setUsername(String username) {
            this.username = username;
        }

        @Override
        @Transient
        public final Builder username(String username) {
            this.username = username;
            return this;
        }

        public final List<HostKeyAttributes.Builder> getHostKeys() {
            List<HostKeyAttributes.Builder> result = HostKeysListCopier.copyToBuilder(this.hostKeys);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setHostKeys(Collection<HostKeyAttributes.BuilderImpl> hostKeys) {
            this.hostKeys = HostKeysListCopier.copyFromBuilder(hostKeys);
        }

        @Override
        @Transient
        public final Builder hostKeys(Collection<HostKeyAttributes> hostKeys) {
            this.hostKeys = HostKeysListCopier.copy(hostKeys);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder hostKeys(HostKeyAttributes... hostKeys) {
            hostKeys(Arrays.asList(hostKeys));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder hostKeys(Consumer<HostKeyAttributes.Builder>... hostKeys) {
            hostKeys(Stream.of(hostKeys).map(c -> HostKeyAttributes.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public InstanceAccessDetails build() {
            return new InstanceAccessDetails(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
