/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lightsail.model;

import java.beans.Transient;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class GetInstanceMetricDataRequest extends LightsailRequest implements
        ToCopyableBuilder<GetInstanceMetricDataRequest.Builder, GetInstanceMetricDataRequest> {
    private static final SdkField<String> INSTANCE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("instanceName").getter(getter(GetInstanceMetricDataRequest::instanceName))
            .setter(setter(Builder::instanceName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("instanceName").build()).build();

    private static final SdkField<String> METRIC_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("metricName").getter(getter(GetInstanceMetricDataRequest::metricNameAsString))
            .setter(setter(Builder::metricName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("metricName").build()).build();

    private static final SdkField<Integer> PERIOD_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("period").getter(getter(GetInstanceMetricDataRequest::period)).setter(setter(Builder::period))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("period").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("startTime").getter(getter(GetInstanceMetricDataRequest::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("startTime").build()).build();

    private static final SdkField<Instant> END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("endTime").getter(getter(GetInstanceMetricDataRequest::endTime)).setter(setter(Builder::endTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("endTime").build()).build();

    private static final SdkField<String> UNIT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("unit")
            .getter(getter(GetInstanceMetricDataRequest::unitAsString)).setter(setter(Builder::unit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("unit").build()).build();

    private static final SdkField<List<String>> STATISTICS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("statistics")
            .getter(getter(GetInstanceMetricDataRequest::statisticsAsStrings))
            .setter(setter(Builder::statisticsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("statistics").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INSTANCE_NAME_FIELD,
            METRIC_NAME_FIELD, PERIOD_FIELD, START_TIME_FIELD, END_TIME_FIELD, UNIT_FIELD, STATISTICS_FIELD));

    private final String instanceName;

    private final String metricName;

    private final Integer period;

    private final Instant startTime;

    private final Instant endTime;

    private final String unit;

    private final List<String> statistics;

    private GetInstanceMetricDataRequest(BuilderImpl builder) {
        super(builder);
        this.instanceName = builder.instanceName;
        this.metricName = builder.metricName;
        this.period = builder.period;
        this.startTime = builder.startTime;
        this.endTime = builder.endTime;
        this.unit = builder.unit;
        this.statistics = builder.statistics;
    }

    /**
     * <p>
     * The name of the instance for which you want to get metrics data.
     * </p>
     * 
     * @return The name of the instance for which you want to get metrics data.
     */
    public final String instanceName() {
        return instanceName;
    }

    /**
     * <p>
     * The metric for which you want to return information.
     * </p>
     * <p>
     * Valid instance metric names are listed below, along with the most useful <code>statistics</code> to include in
     * your request, and the published <code>unit</code> value.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>BurstCapacityPercentage</code> </b> - The percentage of CPU performance available for your instance to
     * burst above its baseline. Your instance continuously accrues and consumes burst capacity. Burst capacity stops
     * accruing when your instance's <code>BurstCapacityPercentage</code> reaches 100%. For more information, see <a
     * href="https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-viewing-instance-burst-capacity">
     * Viewing instance burst capacity in Amazon Lightsail</a>.
     * </p>
     * <p>
     * <code>Statistics</code>: The most useful statistics are <code>Maximum</code> and <code>Average</code>.
     * </p>
     * <p>
     * <code>Unit</code>: The published unit is <code>Percent</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>BurstCapacityTime</code> </b> - The available amount of time for your instance to burst at 100% CPU
     * utilization. Your instance continuously accrues and consumes burst capacity. Burst capacity time stops accruing
     * when your instance's <code>BurstCapacityPercentage</code> metric reaches 100%.
     * </p>
     * <p>
     * Burst capacity time is consumed at the full rate only when your instance operates at 100% CPU utilization. For
     * example, if your instance operates at 50% CPU utilization in the burstable zone for a 5-minute period, then it
     * consumes CPU burst capacity minutes at a 50% rate in that period. Your instance consumed 2 minutes and 30 seconds
     * of CPU burst capacity minutes in the 5-minute period. For more information, see <a
     * href="https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-viewing-instance-burst-capacity"
     * >Viewing instance burst capacity in Amazon Lightsail</a>.
     * </p>
     * <p>
     * <code>Statistics</code>: The most useful statistics are <code>Maximum</code> and <code>Average</code>.
     * </p>
     * <p>
     * <code>Unit</code>: The published unit is <code>Seconds</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>CPUUtilization</code> </b> - The percentage of allocated compute units that are currently in use on the
     * instance. This metric identifies the processing power to run the applications on the instance. Tools in your
     * operating system can show a lower percentage than Lightsail when the instance is not allocated a full processor
     * core.
     * </p>
     * <p>
     * <code>Statistics</code>: The most useful statistics are <code>Maximum</code> and <code>Average</code>.
     * </p>
     * <p>
     * <code>Unit</code>: The published unit is <code>Percent</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>NetworkIn</code> </b> - The number of bytes received on all network interfaces by the instance. This
     * metric identifies the volume of incoming network traffic to the instance. The number reported is the number of
     * bytes received during the period. Because this metric is reported in 5-minute intervals, divide the reported
     * number by 300 to find Bytes/second.
     * </p>
     * <p>
     * <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
     * </p>
     * <p>
     * <code>Unit</code>: The published unit is <code>Bytes</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>NetworkOut</code> </b> - The number of bytes sent out on all network interfaces by the instance. This
     * metric identifies the volume of outgoing network traffic from the instance. The number reported is the number of
     * bytes sent during the period. Because this metric is reported in 5-minute intervals, divide the reported number
     * by 300 to find Bytes/second.
     * </p>
     * <p>
     * <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
     * </p>
     * <p>
     * <code>Unit</code>: The published unit is <code>Bytes</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>StatusCheckFailed</code> </b> - Reports whether the instance passed or failed both the instance status
     * check and the system status check. This metric can be either 0 (passed) or 1 (failed). This metric data is
     * available in 1-minute (60 seconds) granularity.
     * </p>
     * <p>
     * <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
     * </p>
     * <p>
     * <code>Unit</code>: The published unit is <code>Count</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>StatusCheckFailed_Instance</code> </b> - Reports whether the instance passed or failed the instance
     * status check. This metric can be either 0 (passed) or 1 (failed). This metric data is available in 1-minute (60
     * seconds) granularity.
     * </p>
     * <p>
     * <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
     * </p>
     * <p>
     * <code>Unit</code>: The published unit is <code>Count</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>StatusCheckFailed_System</code> </b> - Reports whether the instance passed or failed the system status
     * check. This metric can be either 0 (passed) or 1 (failed). This metric data is available in 1-minute (60 seconds)
     * granularity.
     * </p>
     * <p>
     * <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
     * </p>
     * <p>
     * <code>Unit</code>: The published unit is <code>Count</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #metricName} will
     * return {@link InstanceMetricName#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #metricNameAsString}.
     * </p>
     * 
     * @return The metric for which you want to return information.</p>
     *         <p>
     *         Valid instance metric names are listed below, along with the most useful <code>statistics</code> to
     *         include in your request, and the published <code>unit</code> value.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>BurstCapacityPercentage</code> </b> - The percentage of CPU performance available for your
     *         instance to burst above its baseline. Your instance continuously accrues and consumes burst capacity.
     *         Burst capacity stops accruing when your instance's <code>BurstCapacityPercentage</code> reaches 100%. For
     *         more information, see <a href=
     *         "https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-viewing-instance-burst-capacity"
     *         >Viewing instance burst capacity in Amazon Lightsail</a>.
     *         </p>
     *         <p>
     *         <code>Statistics</code>: The most useful statistics are <code>Maximum</code> and <code>Average</code>.
     *         </p>
     *         <p>
     *         <code>Unit</code>: The published unit is <code>Percent</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>BurstCapacityTime</code> </b> - The available amount of time for your instance to burst at 100%
     *         CPU utilization. Your instance continuously accrues and consumes burst capacity. Burst capacity time
     *         stops accruing when your instance's <code>BurstCapacityPercentage</code> metric reaches 100%.
     *         </p>
     *         <p>
     *         Burst capacity time is consumed at the full rate only when your instance operates at 100% CPU
     *         utilization. For example, if your instance operates at 50% CPU utilization in the burstable zone for a
     *         5-minute period, then it consumes CPU burst capacity minutes at a 50% rate in that period. Your instance
     *         consumed 2 minutes and 30 seconds of CPU burst capacity minutes in the 5-minute period. For more
     *         information, see <a href=
     *         "https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-viewing-instance-burst-capacity"
     *         >Viewing instance burst capacity in Amazon Lightsail</a>.
     *         </p>
     *         <p>
     *         <code>Statistics</code>: The most useful statistics are <code>Maximum</code> and <code>Average</code>.
     *         </p>
     *         <p>
     *         <code>Unit</code>: The published unit is <code>Seconds</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>CPUUtilization</code> </b> - The percentage of allocated compute units that are currently in
     *         use on the instance. This metric identifies the processing power to run the applications on the instance.
     *         Tools in your operating system can show a lower percentage than Lightsail when the instance is not
     *         allocated a full processor core.
     *         </p>
     *         <p>
     *         <code>Statistics</code>: The most useful statistics are <code>Maximum</code> and <code>Average</code>.
     *         </p>
     *         <p>
     *         <code>Unit</code>: The published unit is <code>Percent</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>NetworkIn</code> </b> - The number of bytes received on all network interfaces by the instance.
     *         This metric identifies the volume of incoming network traffic to the instance. The number reported is the
     *         number of bytes received during the period. Because this metric is reported in 5-minute intervals, divide
     *         the reported number by 300 to find Bytes/second.
     *         </p>
     *         <p>
     *         <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
     *         </p>
     *         <p>
     *         <code>Unit</code>: The published unit is <code>Bytes</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>NetworkOut</code> </b> - The number of bytes sent out on all network interfaces by the
     *         instance. This metric identifies the volume of outgoing network traffic from the instance. The number
     *         reported is the number of bytes sent during the period. Because this metric is reported in 5-minute
     *         intervals, divide the reported number by 300 to find Bytes/second.
     *         </p>
     *         <p>
     *         <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
     *         </p>
     *         <p>
     *         <code>Unit</code>: The published unit is <code>Bytes</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>StatusCheckFailed</code> </b> - Reports whether the instance passed or failed both the instance
     *         status check and the system status check. This metric can be either 0 (passed) or 1 (failed). This metric
     *         data is available in 1-minute (60 seconds) granularity.
     *         </p>
     *         <p>
     *         <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
     *         </p>
     *         <p>
     *         <code>Unit</code>: The published unit is <code>Count</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>StatusCheckFailed_Instance</code> </b> - Reports whether the instance passed or failed the
     *         instance status check. This metric can be either 0 (passed) or 1 (failed). This metric data is available
     *         in 1-minute (60 seconds) granularity.
     *         </p>
     *         <p>
     *         <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
     *         </p>
     *         <p>
     *         <code>Unit</code>: The published unit is <code>Count</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>StatusCheckFailed_System</code> </b> - Reports whether the instance passed or failed the system
     *         status check. This metric can be either 0 (passed) or 1 (failed). This metric data is available in
     *         1-minute (60 seconds) granularity.
     *         </p>
     *         <p>
     *         <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
     *         </p>
     *         <p>
     *         <code>Unit</code>: The published unit is <code>Count</code>.
     *         </p>
     *         </li>
     * @see InstanceMetricName
     */
    public final InstanceMetricName metricName() {
        return InstanceMetricName.fromValue(metricName);
    }

    /**
     * <p>
     * The metric for which you want to return information.
     * </p>
     * <p>
     * Valid instance metric names are listed below, along with the most useful <code>statistics</code> to include in
     * your request, and the published <code>unit</code> value.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>BurstCapacityPercentage</code> </b> - The percentage of CPU performance available for your instance to
     * burst above its baseline. Your instance continuously accrues and consumes burst capacity. Burst capacity stops
     * accruing when your instance's <code>BurstCapacityPercentage</code> reaches 100%. For more information, see <a
     * href="https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-viewing-instance-burst-capacity">
     * Viewing instance burst capacity in Amazon Lightsail</a>.
     * </p>
     * <p>
     * <code>Statistics</code>: The most useful statistics are <code>Maximum</code> and <code>Average</code>.
     * </p>
     * <p>
     * <code>Unit</code>: The published unit is <code>Percent</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>BurstCapacityTime</code> </b> - The available amount of time for your instance to burst at 100% CPU
     * utilization. Your instance continuously accrues and consumes burst capacity. Burst capacity time stops accruing
     * when your instance's <code>BurstCapacityPercentage</code> metric reaches 100%.
     * </p>
     * <p>
     * Burst capacity time is consumed at the full rate only when your instance operates at 100% CPU utilization. For
     * example, if your instance operates at 50% CPU utilization in the burstable zone for a 5-minute period, then it
     * consumes CPU burst capacity minutes at a 50% rate in that period. Your instance consumed 2 minutes and 30 seconds
     * of CPU burst capacity minutes in the 5-minute period. For more information, see <a
     * href="https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-viewing-instance-burst-capacity"
     * >Viewing instance burst capacity in Amazon Lightsail</a>.
     * </p>
     * <p>
     * <code>Statistics</code>: The most useful statistics are <code>Maximum</code> and <code>Average</code>.
     * </p>
     * <p>
     * <code>Unit</code>: The published unit is <code>Seconds</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>CPUUtilization</code> </b> - The percentage of allocated compute units that are currently in use on the
     * instance. This metric identifies the processing power to run the applications on the instance. Tools in your
     * operating system can show a lower percentage than Lightsail when the instance is not allocated a full processor
     * core.
     * </p>
     * <p>
     * <code>Statistics</code>: The most useful statistics are <code>Maximum</code> and <code>Average</code>.
     * </p>
     * <p>
     * <code>Unit</code>: The published unit is <code>Percent</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>NetworkIn</code> </b> - The number of bytes received on all network interfaces by the instance. This
     * metric identifies the volume of incoming network traffic to the instance. The number reported is the number of
     * bytes received during the period. Because this metric is reported in 5-minute intervals, divide the reported
     * number by 300 to find Bytes/second.
     * </p>
     * <p>
     * <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
     * </p>
     * <p>
     * <code>Unit</code>: The published unit is <code>Bytes</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>NetworkOut</code> </b> - The number of bytes sent out on all network interfaces by the instance. This
     * metric identifies the volume of outgoing network traffic from the instance. The number reported is the number of
     * bytes sent during the period. Because this metric is reported in 5-minute intervals, divide the reported number
     * by 300 to find Bytes/second.
     * </p>
     * <p>
     * <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
     * </p>
     * <p>
     * <code>Unit</code>: The published unit is <code>Bytes</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>StatusCheckFailed</code> </b> - Reports whether the instance passed or failed both the instance status
     * check and the system status check. This metric can be either 0 (passed) or 1 (failed). This metric data is
     * available in 1-minute (60 seconds) granularity.
     * </p>
     * <p>
     * <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
     * </p>
     * <p>
     * <code>Unit</code>: The published unit is <code>Count</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>StatusCheckFailed_Instance</code> </b> - Reports whether the instance passed or failed the instance
     * status check. This metric can be either 0 (passed) or 1 (failed). This metric data is available in 1-minute (60
     * seconds) granularity.
     * </p>
     * <p>
     * <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
     * </p>
     * <p>
     * <code>Unit</code>: The published unit is <code>Count</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>StatusCheckFailed_System</code> </b> - Reports whether the instance passed or failed the system status
     * check. This metric can be either 0 (passed) or 1 (failed). This metric data is available in 1-minute (60 seconds)
     * granularity.
     * </p>
     * <p>
     * <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
     * </p>
     * <p>
     * <code>Unit</code>: The published unit is <code>Count</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #metricName} will
     * return {@link InstanceMetricName#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #metricNameAsString}.
     * </p>
     * 
     * @return The metric for which you want to return information.</p>
     *         <p>
     *         Valid instance metric names are listed below, along with the most useful <code>statistics</code> to
     *         include in your request, and the published <code>unit</code> value.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>BurstCapacityPercentage</code> </b> - The percentage of CPU performance available for your
     *         instance to burst above its baseline. Your instance continuously accrues and consumes burst capacity.
     *         Burst capacity stops accruing when your instance's <code>BurstCapacityPercentage</code> reaches 100%. For
     *         more information, see <a href=
     *         "https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-viewing-instance-burst-capacity"
     *         >Viewing instance burst capacity in Amazon Lightsail</a>.
     *         </p>
     *         <p>
     *         <code>Statistics</code>: The most useful statistics are <code>Maximum</code> and <code>Average</code>.
     *         </p>
     *         <p>
     *         <code>Unit</code>: The published unit is <code>Percent</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>BurstCapacityTime</code> </b> - The available amount of time for your instance to burst at 100%
     *         CPU utilization. Your instance continuously accrues and consumes burst capacity. Burst capacity time
     *         stops accruing when your instance's <code>BurstCapacityPercentage</code> metric reaches 100%.
     *         </p>
     *         <p>
     *         Burst capacity time is consumed at the full rate only when your instance operates at 100% CPU
     *         utilization. For example, if your instance operates at 50% CPU utilization in the burstable zone for a
     *         5-minute period, then it consumes CPU burst capacity minutes at a 50% rate in that period. Your instance
     *         consumed 2 minutes and 30 seconds of CPU burst capacity minutes in the 5-minute period. For more
     *         information, see <a href=
     *         "https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-viewing-instance-burst-capacity"
     *         >Viewing instance burst capacity in Amazon Lightsail</a>.
     *         </p>
     *         <p>
     *         <code>Statistics</code>: The most useful statistics are <code>Maximum</code> and <code>Average</code>.
     *         </p>
     *         <p>
     *         <code>Unit</code>: The published unit is <code>Seconds</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>CPUUtilization</code> </b> - The percentage of allocated compute units that are currently in
     *         use on the instance. This metric identifies the processing power to run the applications on the instance.
     *         Tools in your operating system can show a lower percentage than Lightsail when the instance is not
     *         allocated a full processor core.
     *         </p>
     *         <p>
     *         <code>Statistics</code>: The most useful statistics are <code>Maximum</code> and <code>Average</code>.
     *         </p>
     *         <p>
     *         <code>Unit</code>: The published unit is <code>Percent</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>NetworkIn</code> </b> - The number of bytes received on all network interfaces by the instance.
     *         This metric identifies the volume of incoming network traffic to the instance. The number reported is the
     *         number of bytes received during the period. Because this metric is reported in 5-minute intervals, divide
     *         the reported number by 300 to find Bytes/second.
     *         </p>
     *         <p>
     *         <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
     *         </p>
     *         <p>
     *         <code>Unit</code>: The published unit is <code>Bytes</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>NetworkOut</code> </b> - The number of bytes sent out on all network interfaces by the
     *         instance. This metric identifies the volume of outgoing network traffic from the instance. The number
     *         reported is the number of bytes sent during the period. Because this metric is reported in 5-minute
     *         intervals, divide the reported number by 300 to find Bytes/second.
     *         </p>
     *         <p>
     *         <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
     *         </p>
     *         <p>
     *         <code>Unit</code>: The published unit is <code>Bytes</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>StatusCheckFailed</code> </b> - Reports whether the instance passed or failed both the instance
     *         status check and the system status check. This metric can be either 0 (passed) or 1 (failed). This metric
     *         data is available in 1-minute (60 seconds) granularity.
     *         </p>
     *         <p>
     *         <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
     *         </p>
     *         <p>
     *         <code>Unit</code>: The published unit is <code>Count</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>StatusCheckFailed_Instance</code> </b> - Reports whether the instance passed or failed the
     *         instance status check. This metric can be either 0 (passed) or 1 (failed). This metric data is available
     *         in 1-minute (60 seconds) granularity.
     *         </p>
     *         <p>
     *         <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
     *         </p>
     *         <p>
     *         <code>Unit</code>: The published unit is <code>Count</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>StatusCheckFailed_System</code> </b> - Reports whether the instance passed or failed the system
     *         status check. This metric can be either 0 (passed) or 1 (failed). This metric data is available in
     *         1-minute (60 seconds) granularity.
     *         </p>
     *         <p>
     *         <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
     *         </p>
     *         <p>
     *         <code>Unit</code>: The published unit is <code>Count</code>.
     *         </p>
     *         </li>
     * @see InstanceMetricName
     */
    public final String metricNameAsString() {
        return metricName;
    }

    /**
     * <p>
     * The granularity, in seconds, of the returned data points.
     * </p>
     * <p>
     * The <code>StatusCheckFailed</code>, <code>StatusCheckFailed_Instance</code>, and
     * <code>StatusCheckFailed_System</code> instance metric data is available in 1-minute (60 seconds) granularity. All
     * other instance metric data is available in 5-minute (300 seconds) granularity.
     * </p>
     * 
     * @return The granularity, in seconds, of the returned data points.</p>
     *         <p>
     *         The <code>StatusCheckFailed</code>, <code>StatusCheckFailed_Instance</code>, and
     *         <code>StatusCheckFailed_System</code> instance metric data is available in 1-minute (60 seconds)
     *         granularity. All other instance metric data is available in 5-minute (300 seconds) granularity.
     */
    public final Integer period() {
        return period;
    }

    /**
     * <p>
     * The start time of the time period.
     * </p>
     * 
     * @return The start time of the time period.
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * The end time of the time period.
     * </p>
     * 
     * @return The end time of the time period.
     */
    public final Instant endTime() {
        return endTime;
    }

    /**
     * <p>
     * The unit for the metric data request. Valid units depend on the metric data being requested. For the valid units
     * to specify with each available metric, see the <code>metricName</code> parameter.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #unit} will return
     * {@link MetricUnit#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #unitAsString}.
     * </p>
     * 
     * @return The unit for the metric data request. Valid units depend on the metric data being requested. For the
     *         valid units to specify with each available metric, see the <code>metricName</code> parameter.
     * @see MetricUnit
     */
    public final MetricUnit unit() {
        return MetricUnit.fromValue(unit);
    }

    /**
     * <p>
     * The unit for the metric data request. Valid units depend on the metric data being requested. For the valid units
     * to specify with each available metric, see the <code>metricName</code> parameter.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #unit} will return
     * {@link MetricUnit#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #unitAsString}.
     * </p>
     * 
     * @return The unit for the metric data request. Valid units depend on the metric data being requested. For the
     *         valid units to specify with each available metric, see the <code>metricName</code> parameter.
     * @see MetricUnit
     */
    public final String unitAsString() {
        return unit;
    }

    /**
     * <p>
     * The statistic for the metric.
     * </p>
     * <p>
     * The following statistics are available:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Minimum</code> - The lowest value observed during the specified period. Use this value to determine low
     * volumes of activity for your application.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Maximum</code> - The highest value observed during the specified period. Use this value to determine high
     * volumes of activity for your application.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Sum</code> - All values submitted for the matching metric added together. You can use this statistic to
     * determine the total volume of a metric.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Average</code> - The value of Sum / SampleCount during the specified period. By comparing this statistic
     * with the Minimum and Maximum values, you can determine the full scope of a metric and how close the average use
     * is to the Minimum and Maximum values. This comparison helps you to know when to increase or decrease your
     * resources.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasStatistics} method.
     * </p>
     * 
     * @return The statistic for the metric.</p>
     *         <p>
     *         The following statistics are available:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Minimum</code> - The lowest value observed during the specified period. Use this value to determine
     *         low volumes of activity for your application.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Maximum</code> - The highest value observed during the specified period. Use this value to
     *         determine high volumes of activity for your application.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Sum</code> - All values submitted for the matching metric added together. You can use this
     *         statistic to determine the total volume of a metric.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Average</code> - The value of Sum / SampleCount during the specified period. By comparing this
     *         statistic with the Minimum and Maximum values, you can determine the full scope of a metric and how close
     *         the average use is to the Minimum and Maximum values. This comparison helps you to know when to increase
     *         or decrease your resources.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
     *         </p>
     *         </li>
     */
    public final List<MetricStatistic> statistics() {
        return MetricStatisticListCopier.copyStringToEnum(statistics);
    }

    /**
     * For responses, this returns true if the service returned a value for the Statistics property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasStatistics() {
        return statistics != null && !(statistics instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The statistic for the metric.
     * </p>
     * <p>
     * The following statistics are available:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Minimum</code> - The lowest value observed during the specified period. Use this value to determine low
     * volumes of activity for your application.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Maximum</code> - The highest value observed during the specified period. Use this value to determine high
     * volumes of activity for your application.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Sum</code> - All values submitted for the matching metric added together. You can use this statistic to
     * determine the total volume of a metric.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Average</code> - The value of Sum / SampleCount during the specified period. By comparing this statistic
     * with the Minimum and Maximum values, you can determine the full scope of a metric and how close the average use
     * is to the Minimum and Maximum values. This comparison helps you to know when to increase or decrease your
     * resources.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasStatistics} method.
     * </p>
     * 
     * @return The statistic for the metric.</p>
     *         <p>
     *         The following statistics are available:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Minimum</code> - The lowest value observed during the specified period. Use this value to determine
     *         low volumes of activity for your application.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Maximum</code> - The highest value observed during the specified period. Use this value to
     *         determine high volumes of activity for your application.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Sum</code> - All values submitted for the matching metric added together. You can use this
     *         statistic to determine the total volume of a metric.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Average</code> - The value of Sum / SampleCount during the specified period. By comparing this
     *         statistic with the Minimum and Maximum values, you can determine the full scope of a metric and how close
     *         the average use is to the Minimum and Maximum values. This comparison helps you to know when to increase
     *         or decrease your resources.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
     *         </p>
     *         </li>
     */
    public final List<String> statisticsAsStrings() {
        return statistics;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(instanceName());
        hashCode = 31 * hashCode + Objects.hashCode(metricNameAsString());
        hashCode = 31 * hashCode + Objects.hashCode(period());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        hashCode = 31 * hashCode + Objects.hashCode(unitAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasStatistics() ? statisticsAsStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GetInstanceMetricDataRequest)) {
            return false;
        }
        GetInstanceMetricDataRequest other = (GetInstanceMetricDataRequest) obj;
        return Objects.equals(instanceName(), other.instanceName())
                && Objects.equals(metricNameAsString(), other.metricNameAsString()) && Objects.equals(period(), other.period())
                && Objects.equals(startTime(), other.startTime()) && Objects.equals(endTime(), other.endTime())
                && Objects.equals(unitAsString(), other.unitAsString()) && hasStatistics() == other.hasStatistics()
                && Objects.equals(statisticsAsStrings(), other.statisticsAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("GetInstanceMetricDataRequest").add("InstanceName", instanceName())
                .add("MetricName", metricNameAsString()).add("Period", period()).add("StartTime", startTime())
                .add("EndTime", endTime()).add("Unit", unitAsString())
                .add("Statistics", hasStatistics() ? statisticsAsStrings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "instanceName":
            return Optional.ofNullable(clazz.cast(instanceName()));
        case "metricName":
            return Optional.ofNullable(clazz.cast(metricNameAsString()));
        case "period":
            return Optional.ofNullable(clazz.cast(period()));
        case "startTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "endTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        case "unit":
            return Optional.ofNullable(clazz.cast(unitAsString()));
        case "statistics":
            return Optional.ofNullable(clazz.cast(statisticsAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<GetInstanceMetricDataRequest, T> g) {
        return obj -> g.apply((GetInstanceMetricDataRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends LightsailRequest.Builder, SdkPojo, CopyableBuilder<Builder, GetInstanceMetricDataRequest> {
        /**
         * <p>
         * The name of the instance for which you want to get metrics data.
         * </p>
         * 
         * @param instanceName
         *        The name of the instance for which you want to get metrics data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceName(String instanceName);

        /**
         * <p>
         * The metric for which you want to return information.
         * </p>
         * <p>
         * Valid instance metric names are listed below, along with the most useful <code>statistics</code> to include
         * in your request, and the published <code>unit</code> value.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>BurstCapacityPercentage</code> </b> - The percentage of CPU performance available for your instance
         * to burst above its baseline. Your instance continuously accrues and consumes burst capacity. Burst capacity
         * stops accruing when your instance's <code>BurstCapacityPercentage</code> reaches 100%. For more information,
         * see <a href=
         * "https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-viewing-instance-burst-capacity"
         * >Viewing instance burst capacity in Amazon Lightsail</a>.
         * </p>
         * <p>
         * <code>Statistics</code>: The most useful statistics are <code>Maximum</code> and <code>Average</code>.
         * </p>
         * <p>
         * <code>Unit</code>: The published unit is <code>Percent</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>BurstCapacityTime</code> </b> - The available amount of time for your instance to burst at 100% CPU
         * utilization. Your instance continuously accrues and consumes burst capacity. Burst capacity time stops
         * accruing when your instance's <code>BurstCapacityPercentage</code> metric reaches 100%.
         * </p>
         * <p>
         * Burst capacity time is consumed at the full rate only when your instance operates at 100% CPU utilization.
         * For example, if your instance operates at 50% CPU utilization in the burstable zone for a 5-minute period,
         * then it consumes CPU burst capacity minutes at a 50% rate in that period. Your instance consumed 2 minutes
         * and 30 seconds of CPU burst capacity minutes in the 5-minute period. For more information, see <a href=
         * "https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-viewing-instance-burst-capacity"
         * >Viewing instance burst capacity in Amazon Lightsail</a>.
         * </p>
         * <p>
         * <code>Statistics</code>: The most useful statistics are <code>Maximum</code> and <code>Average</code>.
         * </p>
         * <p>
         * <code>Unit</code>: The published unit is <code>Seconds</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>CPUUtilization</code> </b> - The percentage of allocated compute units that are currently in use on
         * the instance. This metric identifies the processing power to run the applications on the instance. Tools in
         * your operating system can show a lower percentage than Lightsail when the instance is not allocated a full
         * processor core.
         * </p>
         * <p>
         * <code>Statistics</code>: The most useful statistics are <code>Maximum</code> and <code>Average</code>.
         * </p>
         * <p>
         * <code>Unit</code>: The published unit is <code>Percent</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>NetworkIn</code> </b> - The number of bytes received on all network interfaces by the instance.
         * This metric identifies the volume of incoming network traffic to the instance. The number reported is the
         * number of bytes received during the period. Because this metric is reported in 5-minute intervals, divide the
         * reported number by 300 to find Bytes/second.
         * </p>
         * <p>
         * <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
         * </p>
         * <p>
         * <code>Unit</code>: The published unit is <code>Bytes</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>NetworkOut</code> </b> - The number of bytes sent out on all network interfaces by the instance.
         * This metric identifies the volume of outgoing network traffic from the instance. The number reported is the
         * number of bytes sent during the period. Because this metric is reported in 5-minute intervals, divide the
         * reported number by 300 to find Bytes/second.
         * </p>
         * <p>
         * <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
         * </p>
         * <p>
         * <code>Unit</code>: The published unit is <code>Bytes</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>StatusCheckFailed</code> </b> - Reports whether the instance passed or failed both the instance
         * status check and the system status check. This metric can be either 0 (passed) or 1 (failed). This metric
         * data is available in 1-minute (60 seconds) granularity.
         * </p>
         * <p>
         * <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
         * </p>
         * <p>
         * <code>Unit</code>: The published unit is <code>Count</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>StatusCheckFailed_Instance</code> </b> - Reports whether the instance passed or failed the instance
         * status check. This metric can be either 0 (passed) or 1 (failed). This metric data is available in 1-minute
         * (60 seconds) granularity.
         * </p>
         * <p>
         * <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
         * </p>
         * <p>
         * <code>Unit</code>: The published unit is <code>Count</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>StatusCheckFailed_System</code> </b> - Reports whether the instance passed or failed the system
         * status check. This metric can be either 0 (passed) or 1 (failed). This metric data is available in 1-minute
         * (60 seconds) granularity.
         * </p>
         * <p>
         * <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
         * </p>
         * <p>
         * <code>Unit</code>: The published unit is <code>Count</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param metricName
         *        The metric for which you want to return information.</p>
         *        <p>
         *        Valid instance metric names are listed below, along with the most useful <code>statistics</code> to
         *        include in your request, and the published <code>unit</code> value.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>BurstCapacityPercentage</code> </b> - The percentage of CPU performance available for your
         *        instance to burst above its baseline. Your instance continuously accrues and consumes burst capacity.
         *        Burst capacity stops accruing when your instance's <code>BurstCapacityPercentage</code> reaches 100%.
         *        For more information, see <a href=
         *        "https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-viewing-instance-burst-capacity"
         *        >Viewing instance burst capacity in Amazon Lightsail</a>.
         *        </p>
         *        <p>
         *        <code>Statistics</code>: The most useful statistics are <code>Maximum</code> and <code>Average</code>.
         *        </p>
         *        <p>
         *        <code>Unit</code>: The published unit is <code>Percent</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>BurstCapacityTime</code> </b> - The available amount of time for your instance to burst at
         *        100% CPU utilization. Your instance continuously accrues and consumes burst capacity. Burst capacity
         *        time stops accruing when your instance's <code>BurstCapacityPercentage</code> metric reaches 100%.
         *        </p>
         *        <p>
         *        Burst capacity time is consumed at the full rate only when your instance operates at 100% CPU
         *        utilization. For example, if your instance operates at 50% CPU utilization in the burstable zone for a
         *        5-minute period, then it consumes CPU burst capacity minutes at a 50% rate in that period. Your
         *        instance consumed 2 minutes and 30 seconds of CPU burst capacity minutes in the 5-minute period. For
         *        more information, see <a href=
         *        "https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-viewing-instance-burst-capacity"
         *        >Viewing instance burst capacity in Amazon Lightsail</a>.
         *        </p>
         *        <p>
         *        <code>Statistics</code>: The most useful statistics are <code>Maximum</code> and <code>Average</code>.
         *        </p>
         *        <p>
         *        <code>Unit</code>: The published unit is <code>Seconds</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>CPUUtilization</code> </b> - The percentage of allocated compute units that are currently in
         *        use on the instance. This metric identifies the processing power to run the applications on the
         *        instance. Tools in your operating system can show a lower percentage than Lightsail when the instance
         *        is not allocated a full processor core.
         *        </p>
         *        <p>
         *        <code>Statistics</code>: The most useful statistics are <code>Maximum</code> and <code>Average</code>.
         *        </p>
         *        <p>
         *        <code>Unit</code>: The published unit is <code>Percent</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>NetworkIn</code> </b> - The number of bytes received on all network interfaces by the
         *        instance. This metric identifies the volume of incoming network traffic to the instance. The number
         *        reported is the number of bytes received during the period. Because this metric is reported in
         *        5-minute intervals, divide the reported number by 300 to find Bytes/second.
         *        </p>
         *        <p>
         *        <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
         *        </p>
         *        <p>
         *        <code>Unit</code>: The published unit is <code>Bytes</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>NetworkOut</code> </b> - The number of bytes sent out on all network interfaces by the
         *        instance. This metric identifies the volume of outgoing network traffic from the instance. The number
         *        reported is the number of bytes sent during the period. Because this metric is reported in 5-minute
         *        intervals, divide the reported number by 300 to find Bytes/second.
         *        </p>
         *        <p>
         *        <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
         *        </p>
         *        <p>
         *        <code>Unit</code>: The published unit is <code>Bytes</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>StatusCheckFailed</code> </b> - Reports whether the instance passed or failed both the
         *        instance status check and the system status check. This metric can be either 0 (passed) or 1 (failed).
         *        This metric data is available in 1-minute (60 seconds) granularity.
         *        </p>
         *        <p>
         *        <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
         *        </p>
         *        <p>
         *        <code>Unit</code>: The published unit is <code>Count</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>StatusCheckFailed_Instance</code> </b> - Reports whether the instance passed or failed the
         *        instance status check. This metric can be either 0 (passed) or 1 (failed). This metric data is
         *        available in 1-minute (60 seconds) granularity.
         *        </p>
         *        <p>
         *        <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
         *        </p>
         *        <p>
         *        <code>Unit</code>: The published unit is <code>Count</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>StatusCheckFailed_System</code> </b> - Reports whether the instance passed or failed the
         *        system status check. This metric can be either 0 (passed) or 1 (failed). This metric data is available
         *        in 1-minute (60 seconds) granularity.
         *        </p>
         *        <p>
         *        <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
         *        </p>
         *        <p>
         *        <code>Unit</code>: The published unit is <code>Count</code>.
         *        </p>
         *        </li>
         * @see InstanceMetricName
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceMetricName
         */
        Builder metricName(String metricName);

        /**
         * <p>
         * The metric for which you want to return information.
         * </p>
         * <p>
         * Valid instance metric names are listed below, along with the most useful <code>statistics</code> to include
         * in your request, and the published <code>unit</code> value.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>BurstCapacityPercentage</code> </b> - The percentage of CPU performance available for your instance
         * to burst above its baseline. Your instance continuously accrues and consumes burst capacity. Burst capacity
         * stops accruing when your instance's <code>BurstCapacityPercentage</code> reaches 100%. For more information,
         * see <a href=
         * "https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-viewing-instance-burst-capacity"
         * >Viewing instance burst capacity in Amazon Lightsail</a>.
         * </p>
         * <p>
         * <code>Statistics</code>: The most useful statistics are <code>Maximum</code> and <code>Average</code>.
         * </p>
         * <p>
         * <code>Unit</code>: The published unit is <code>Percent</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>BurstCapacityTime</code> </b> - The available amount of time for your instance to burst at 100% CPU
         * utilization. Your instance continuously accrues and consumes burst capacity. Burst capacity time stops
         * accruing when your instance's <code>BurstCapacityPercentage</code> metric reaches 100%.
         * </p>
         * <p>
         * Burst capacity time is consumed at the full rate only when your instance operates at 100% CPU utilization.
         * For example, if your instance operates at 50% CPU utilization in the burstable zone for a 5-minute period,
         * then it consumes CPU burst capacity minutes at a 50% rate in that period. Your instance consumed 2 minutes
         * and 30 seconds of CPU burst capacity minutes in the 5-minute period. For more information, see <a href=
         * "https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-viewing-instance-burst-capacity"
         * >Viewing instance burst capacity in Amazon Lightsail</a>.
         * </p>
         * <p>
         * <code>Statistics</code>: The most useful statistics are <code>Maximum</code> and <code>Average</code>.
         * </p>
         * <p>
         * <code>Unit</code>: The published unit is <code>Seconds</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>CPUUtilization</code> </b> - The percentage of allocated compute units that are currently in use on
         * the instance. This metric identifies the processing power to run the applications on the instance. Tools in
         * your operating system can show a lower percentage than Lightsail when the instance is not allocated a full
         * processor core.
         * </p>
         * <p>
         * <code>Statistics</code>: The most useful statistics are <code>Maximum</code> and <code>Average</code>.
         * </p>
         * <p>
         * <code>Unit</code>: The published unit is <code>Percent</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>NetworkIn</code> </b> - The number of bytes received on all network interfaces by the instance.
         * This metric identifies the volume of incoming network traffic to the instance. The number reported is the
         * number of bytes received during the period. Because this metric is reported in 5-minute intervals, divide the
         * reported number by 300 to find Bytes/second.
         * </p>
         * <p>
         * <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
         * </p>
         * <p>
         * <code>Unit</code>: The published unit is <code>Bytes</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>NetworkOut</code> </b> - The number of bytes sent out on all network interfaces by the instance.
         * This metric identifies the volume of outgoing network traffic from the instance. The number reported is the
         * number of bytes sent during the period. Because this metric is reported in 5-minute intervals, divide the
         * reported number by 300 to find Bytes/second.
         * </p>
         * <p>
         * <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
         * </p>
         * <p>
         * <code>Unit</code>: The published unit is <code>Bytes</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>StatusCheckFailed</code> </b> - Reports whether the instance passed or failed both the instance
         * status check and the system status check. This metric can be either 0 (passed) or 1 (failed). This metric
         * data is available in 1-minute (60 seconds) granularity.
         * </p>
         * <p>
         * <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
         * </p>
         * <p>
         * <code>Unit</code>: The published unit is <code>Count</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>StatusCheckFailed_Instance</code> </b> - Reports whether the instance passed or failed the instance
         * status check. This metric can be either 0 (passed) or 1 (failed). This metric data is available in 1-minute
         * (60 seconds) granularity.
         * </p>
         * <p>
         * <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
         * </p>
         * <p>
         * <code>Unit</code>: The published unit is <code>Count</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>StatusCheckFailed_System</code> </b> - Reports whether the instance passed or failed the system
         * status check. This metric can be either 0 (passed) or 1 (failed). This metric data is available in 1-minute
         * (60 seconds) granularity.
         * </p>
         * <p>
         * <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
         * </p>
         * <p>
         * <code>Unit</code>: The published unit is <code>Count</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param metricName
         *        The metric for which you want to return information.</p>
         *        <p>
         *        Valid instance metric names are listed below, along with the most useful <code>statistics</code> to
         *        include in your request, and the published <code>unit</code> value.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>BurstCapacityPercentage</code> </b> - The percentage of CPU performance available for your
         *        instance to burst above its baseline. Your instance continuously accrues and consumes burst capacity.
         *        Burst capacity stops accruing when your instance's <code>BurstCapacityPercentage</code> reaches 100%.
         *        For more information, see <a href=
         *        "https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-viewing-instance-burst-capacity"
         *        >Viewing instance burst capacity in Amazon Lightsail</a>.
         *        </p>
         *        <p>
         *        <code>Statistics</code>: The most useful statistics are <code>Maximum</code> and <code>Average</code>.
         *        </p>
         *        <p>
         *        <code>Unit</code>: The published unit is <code>Percent</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>BurstCapacityTime</code> </b> - The available amount of time for your instance to burst at
         *        100% CPU utilization. Your instance continuously accrues and consumes burst capacity. Burst capacity
         *        time stops accruing when your instance's <code>BurstCapacityPercentage</code> metric reaches 100%.
         *        </p>
         *        <p>
         *        Burst capacity time is consumed at the full rate only when your instance operates at 100% CPU
         *        utilization. For example, if your instance operates at 50% CPU utilization in the burstable zone for a
         *        5-minute period, then it consumes CPU burst capacity minutes at a 50% rate in that period. Your
         *        instance consumed 2 minutes and 30 seconds of CPU burst capacity minutes in the 5-minute period. For
         *        more information, see <a href=
         *        "https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-viewing-instance-burst-capacity"
         *        >Viewing instance burst capacity in Amazon Lightsail</a>.
         *        </p>
         *        <p>
         *        <code>Statistics</code>: The most useful statistics are <code>Maximum</code> and <code>Average</code>.
         *        </p>
         *        <p>
         *        <code>Unit</code>: The published unit is <code>Seconds</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>CPUUtilization</code> </b> - The percentage of allocated compute units that are currently in
         *        use on the instance. This metric identifies the processing power to run the applications on the
         *        instance. Tools in your operating system can show a lower percentage than Lightsail when the instance
         *        is not allocated a full processor core.
         *        </p>
         *        <p>
         *        <code>Statistics</code>: The most useful statistics are <code>Maximum</code> and <code>Average</code>.
         *        </p>
         *        <p>
         *        <code>Unit</code>: The published unit is <code>Percent</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>NetworkIn</code> </b> - The number of bytes received on all network interfaces by the
         *        instance. This metric identifies the volume of incoming network traffic to the instance. The number
         *        reported is the number of bytes received during the period. Because this metric is reported in
         *        5-minute intervals, divide the reported number by 300 to find Bytes/second.
         *        </p>
         *        <p>
         *        <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
         *        </p>
         *        <p>
         *        <code>Unit</code>: The published unit is <code>Bytes</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>NetworkOut</code> </b> - The number of bytes sent out on all network interfaces by the
         *        instance. This metric identifies the volume of outgoing network traffic from the instance. The number
         *        reported is the number of bytes sent during the period. Because this metric is reported in 5-minute
         *        intervals, divide the reported number by 300 to find Bytes/second.
         *        </p>
         *        <p>
         *        <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
         *        </p>
         *        <p>
         *        <code>Unit</code>: The published unit is <code>Bytes</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>StatusCheckFailed</code> </b> - Reports whether the instance passed or failed both the
         *        instance status check and the system status check. This metric can be either 0 (passed) or 1 (failed).
         *        This metric data is available in 1-minute (60 seconds) granularity.
         *        </p>
         *        <p>
         *        <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
         *        </p>
         *        <p>
         *        <code>Unit</code>: The published unit is <code>Count</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>StatusCheckFailed_Instance</code> </b> - Reports whether the instance passed or failed the
         *        instance status check. This metric can be either 0 (passed) or 1 (failed). This metric data is
         *        available in 1-minute (60 seconds) granularity.
         *        </p>
         *        <p>
         *        <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
         *        </p>
         *        <p>
         *        <code>Unit</code>: The published unit is <code>Count</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>StatusCheckFailed_System</code> </b> - Reports whether the instance passed or failed the
         *        system status check. This metric can be either 0 (passed) or 1 (failed). This metric data is available
         *        in 1-minute (60 seconds) granularity.
         *        </p>
         *        <p>
         *        <code>Statistics</code>: The most useful statistic is <code>Sum</code>.
         *        </p>
         *        <p>
         *        <code>Unit</code>: The published unit is <code>Count</code>.
         *        </p>
         *        </li>
         * @see InstanceMetricName
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceMetricName
         */
        Builder metricName(InstanceMetricName metricName);

        /**
         * <p>
         * The granularity, in seconds, of the returned data points.
         * </p>
         * <p>
         * The <code>StatusCheckFailed</code>, <code>StatusCheckFailed_Instance</code>, and
         * <code>StatusCheckFailed_System</code> instance metric data is available in 1-minute (60 seconds) granularity.
         * All other instance metric data is available in 5-minute (300 seconds) granularity.
         * </p>
         * 
         * @param period
         *        The granularity, in seconds, of the returned data points.</p>
         *        <p>
         *        The <code>StatusCheckFailed</code>, <code>StatusCheckFailed_Instance</code>, and
         *        <code>StatusCheckFailed_System</code> instance metric data is available in 1-minute (60 seconds)
         *        granularity. All other instance metric data is available in 5-minute (300 seconds) granularity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder period(Integer period);

        /**
         * <p>
         * The start time of the time period.
         * </p>
         * 
         * @param startTime
         *        The start time of the time period.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The end time of the time period.
         * </p>
         * 
         * @param endTime
         *        The end time of the time period.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Instant endTime);

        /**
         * <p>
         * The unit for the metric data request. Valid units depend on the metric data being requested. For the valid
         * units to specify with each available metric, see the <code>metricName</code> parameter.
         * </p>
         * 
         * @param unit
         *        The unit for the metric data request. Valid units depend on the metric data being requested. For the
         *        valid units to specify with each available metric, see the <code>metricName</code> parameter.
         * @see MetricUnit
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricUnit
         */
        Builder unit(String unit);

        /**
         * <p>
         * The unit for the metric data request. Valid units depend on the metric data being requested. For the valid
         * units to specify with each available metric, see the <code>metricName</code> parameter.
         * </p>
         * 
         * @param unit
         *        The unit for the metric data request. Valid units depend on the metric data being requested. For the
         *        valid units to specify with each available metric, see the <code>metricName</code> parameter.
         * @see MetricUnit
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricUnit
         */
        Builder unit(MetricUnit unit);

        /**
         * <p>
         * The statistic for the metric.
         * </p>
         * <p>
         * The following statistics are available:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Minimum</code> - The lowest value observed during the specified period. Use this value to determine low
         * volumes of activity for your application.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Maximum</code> - The highest value observed during the specified period. Use this value to determine
         * high volumes of activity for your application.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Sum</code> - All values submitted for the matching metric added together. You can use this statistic to
         * determine the total volume of a metric.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Average</code> - The value of Sum / SampleCount during the specified period. By comparing this
         * statistic with the Minimum and Maximum values, you can determine the full scope of a metric and how close the
         * average use is to the Minimum and Maximum values. This comparison helps you to know when to increase or
         * decrease your resources.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
         * </p>
         * </li>
         * </ul>
         * 
         * @param statistics
         *        The statistic for the metric.</p>
         *        <p>
         *        The following statistics are available:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Minimum</code> - The lowest value observed during the specified period. Use this value to
         *        determine low volumes of activity for your application.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Maximum</code> - The highest value observed during the specified period. Use this value to
         *        determine high volumes of activity for your application.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Sum</code> - All values submitted for the matching metric added together. You can use this
         *        statistic to determine the total volume of a metric.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Average</code> - The value of Sum / SampleCount during the specified period. By comparing this
         *        statistic with the Minimum and Maximum values, you can determine the full scope of a metric and how
         *        close the average use is to the Minimum and Maximum values. This comparison helps you to know when to
         *        increase or decrease your resources.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statisticsWithStrings(Collection<String> statistics);

        /**
         * <p>
         * The statistic for the metric.
         * </p>
         * <p>
         * The following statistics are available:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Minimum</code> - The lowest value observed during the specified period. Use this value to determine low
         * volumes of activity for your application.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Maximum</code> - The highest value observed during the specified period. Use this value to determine
         * high volumes of activity for your application.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Sum</code> - All values submitted for the matching metric added together. You can use this statistic to
         * determine the total volume of a metric.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Average</code> - The value of Sum / SampleCount during the specified period. By comparing this
         * statistic with the Minimum and Maximum values, you can determine the full scope of a metric and how close the
         * average use is to the Minimum and Maximum values. This comparison helps you to know when to increase or
         * decrease your resources.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
         * </p>
         * </li>
         * </ul>
         * 
         * @param statistics
         *        The statistic for the metric.</p>
         *        <p>
         *        The following statistics are available:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Minimum</code> - The lowest value observed during the specified period. Use this value to
         *        determine low volumes of activity for your application.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Maximum</code> - The highest value observed during the specified period. Use this value to
         *        determine high volumes of activity for your application.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Sum</code> - All values submitted for the matching metric added together. You can use this
         *        statistic to determine the total volume of a metric.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Average</code> - The value of Sum / SampleCount during the specified period. By comparing this
         *        statistic with the Minimum and Maximum values, you can determine the full scope of a metric and how
         *        close the average use is to the Minimum and Maximum values. This comparison helps you to know when to
         *        increase or decrease your resources.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statisticsWithStrings(String... statistics);

        /**
         * <p>
         * The statistic for the metric.
         * </p>
         * <p>
         * The following statistics are available:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Minimum</code> - The lowest value observed during the specified period. Use this value to determine low
         * volumes of activity for your application.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Maximum</code> - The highest value observed during the specified period. Use this value to determine
         * high volumes of activity for your application.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Sum</code> - All values submitted for the matching metric added together. You can use this statistic to
         * determine the total volume of a metric.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Average</code> - The value of Sum / SampleCount during the specified period. By comparing this
         * statistic with the Minimum and Maximum values, you can determine the full scope of a metric and how close the
         * average use is to the Minimum and Maximum values. This comparison helps you to know when to increase or
         * decrease your resources.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
         * </p>
         * </li>
         * </ul>
         * 
         * @param statistics
         *        The statistic for the metric.</p>
         *        <p>
         *        The following statistics are available:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Minimum</code> - The lowest value observed during the specified period. Use this value to
         *        determine low volumes of activity for your application.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Maximum</code> - The highest value observed during the specified period. Use this value to
         *        determine high volumes of activity for your application.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Sum</code> - All values submitted for the matching metric added together. You can use this
         *        statistic to determine the total volume of a metric.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Average</code> - The value of Sum / SampleCount during the specified period. By comparing this
         *        statistic with the Minimum and Maximum values, you can determine the full scope of a metric and how
         *        close the average use is to the Minimum and Maximum values. This comparison helps you to know when to
         *        increase or decrease your resources.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statistics(Collection<MetricStatistic> statistics);

        /**
         * <p>
         * The statistic for the metric.
         * </p>
         * <p>
         * The following statistics are available:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Minimum</code> - The lowest value observed during the specified period. Use this value to determine low
         * volumes of activity for your application.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Maximum</code> - The highest value observed during the specified period. Use this value to determine
         * high volumes of activity for your application.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Sum</code> - All values submitted for the matching metric added together. You can use this statistic to
         * determine the total volume of a metric.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Average</code> - The value of Sum / SampleCount during the specified period. By comparing this
         * statistic with the Minimum and Maximum values, you can determine the full scope of a metric and how close the
         * average use is to the Minimum and Maximum values. This comparison helps you to know when to increase or
         * decrease your resources.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
         * </p>
         * </li>
         * </ul>
         * 
         * @param statistics
         *        The statistic for the metric.</p>
         *        <p>
         *        The following statistics are available:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Minimum</code> - The lowest value observed during the specified period. Use this value to
         *        determine low volumes of activity for your application.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Maximum</code> - The highest value observed during the specified period. Use this value to
         *        determine high volumes of activity for your application.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Sum</code> - All values submitted for the matching metric added together. You can use this
         *        statistic to determine the total volume of a metric.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Average</code> - The value of Sum / SampleCount during the specified period. By comparing this
         *        statistic with the Minimum and Maximum values, you can determine the full scope of a metric and how
         *        close the average use is to the Minimum and Maximum values. This comparison helps you to know when to
         *        increase or decrease your resources.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statistics(MetricStatistic... statistics);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends LightsailRequest.BuilderImpl implements Builder {
        private String instanceName;

        private String metricName;

        private Integer period;

        private Instant startTime;

        private Instant endTime;

        private String unit;

        private List<String> statistics = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(GetInstanceMetricDataRequest model) {
            super(model);
            instanceName(model.instanceName);
            metricName(model.metricName);
            period(model.period);
            startTime(model.startTime);
            endTime(model.endTime);
            unit(model.unit);
            statisticsWithStrings(model.statistics);
        }

        public final String getInstanceName() {
            return instanceName;
        }

        public final void setInstanceName(String instanceName) {
            this.instanceName = instanceName;
        }

        @Override
        @Transient
        public final Builder instanceName(String instanceName) {
            this.instanceName = instanceName;
            return this;
        }

        public final String getMetricName() {
            return metricName;
        }

        public final void setMetricName(String metricName) {
            this.metricName = metricName;
        }

        @Override
        @Transient
        public final Builder metricName(String metricName) {
            this.metricName = metricName;
            return this;
        }

        @Override
        @Transient
        public final Builder metricName(InstanceMetricName metricName) {
            this.metricName(metricName == null ? null : metricName.toString());
            return this;
        }

        public final Integer getPeriod() {
            return period;
        }

        public final void setPeriod(Integer period) {
            this.period = period;
        }

        @Override
        @Transient
        public final Builder period(Integer period) {
            this.period = period;
            return this;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        @Override
        @Transient
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final Instant getEndTime() {
            return endTime;
        }

        public final void setEndTime(Instant endTime) {
            this.endTime = endTime;
        }

        @Override
        @Transient
        public final Builder endTime(Instant endTime) {
            this.endTime = endTime;
            return this;
        }

        public final String getUnit() {
            return unit;
        }

        public final void setUnit(String unit) {
            this.unit = unit;
        }

        @Override
        @Transient
        public final Builder unit(String unit) {
            this.unit = unit;
            return this;
        }

        @Override
        @Transient
        public final Builder unit(MetricUnit unit) {
            this.unit(unit == null ? null : unit.toString());
            return this;
        }

        public final Collection<String> getStatistics() {
            if (statistics instanceof SdkAutoConstructList) {
                return null;
            }
            return statistics;
        }

        public final void setStatistics(Collection<String> statistics) {
            this.statistics = MetricStatisticListCopier.copy(statistics);
        }

        @Override
        @Transient
        public final Builder statisticsWithStrings(Collection<String> statistics) {
            this.statistics = MetricStatisticListCopier.copy(statistics);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder statisticsWithStrings(String... statistics) {
            statisticsWithStrings(Arrays.asList(statistics));
            return this;
        }

        @Override
        @Transient
        public final Builder statistics(Collection<MetricStatistic> statistics) {
            this.statistics = MetricStatisticListCopier.copyEnumToString(statistics);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder statistics(MetricStatistic... statistics) {
            statistics(Arrays.asList(statistics));
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public GetInstanceMetricDataRequest build() {
            return new GetInstanceMetricDataRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
