/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lightsail.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the TLS security policies that are available for Lightsail load balancers.
 * </p>
 * <p>
 * For more information about load balancer TLS security policies, see <a href=
 * "https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-configure-load-balancer-tls-security-policy"
 * >Load balancer TLS security policies</a> in the <i>Amazon Lightsail Developer Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LoadBalancerTlsPolicy implements SdkPojo, Serializable,
        ToCopyableBuilder<LoadBalancerTlsPolicy.Builder, LoadBalancerTlsPolicy> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(LoadBalancerTlsPolicy::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<Boolean> IS_DEFAULT_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("isDefault").getter(getter(LoadBalancerTlsPolicy::isDefault)).setter(setter(Builder::isDefault))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("isDefault").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("description").getter(getter(LoadBalancerTlsPolicy::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<List<String>> PROTOCOLS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("protocols")
            .getter(getter(LoadBalancerTlsPolicy::protocols))
            .setter(setter(Builder::protocols))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("protocols").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> CIPHERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ciphers")
            .getter(getter(LoadBalancerTlsPolicy::ciphers))
            .setter(setter(Builder::ciphers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ciphers").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, IS_DEFAULT_FIELD,
            DESCRIPTION_FIELD, PROTOCOLS_FIELD, CIPHERS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final Boolean isDefault;

    private final String description;

    private final List<String> protocols;

    private final List<String> ciphers;

    private LoadBalancerTlsPolicy(BuilderImpl builder) {
        this.name = builder.name;
        this.isDefault = builder.isDefault;
        this.description = builder.description;
        this.protocols = builder.protocols;
        this.ciphers = builder.ciphers;
    }

    /**
     * <p>
     * The name of the TLS security policy.
     * </p>
     * <p>
     * The following TLS security policy names are possible:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>TLS-2016-08</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TLS-FS-Res-1-2-2019-08</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * You can specify either of these values for the <code>tlsSecurityPolicyName</code> request parameter in the <a
     * href
     * ="https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateLoadBalancer.html">CreateLoadBalancer
     * </a> action, and the <code>attributeValue</code> request parameter in the <a
     * href="https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UpdateLoadBalancerAttribute.html"
     * >UpdateLoadBalancerAttribute</a> action.
     * </p>
     * 
     * @return The name of the TLS security policy.</p>
     *         <p>
     *         The following TLS security policy names are possible:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>TLS-2016-08</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TLS-FS-Res-1-2-2019-08</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         You can specify either of these values for the <code>tlsSecurityPolicyName</code> request parameter in
     *         the <a href="https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateLoadBalancer.html">
     *         CreateLoadBalancer</a> action, and the <code>attributeValue</code> request parameter in the <a
     *         href="https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UpdateLoadBalancerAttribute.html"
     *         >UpdateLoadBalancerAttribute</a> action.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * A Boolean value that indicates whether the TLS security policy is the default.
     * </p>
     * 
     * @return A Boolean value that indicates whether the TLS security policy is the default.
     */
    public final Boolean isDefault() {
        return isDefault;
    }

    /**
     * <p>
     * The description of the TLS security policy.
     * </p>
     * 
     * @return The description of the TLS security policy.
     */
    public final String description() {
        return description;
    }

    /**
     * For responses, this returns true if the service returned a value for the Protocols property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasProtocols() {
        return protocols != null && !(protocols instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The protocols used in a given TLS security policy.
     * </p>
     * <p>
     * The following protocols are possible:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Protocol-TLSv1</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Protocol-TLSv1.1</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Protocol-TLSv1.2</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasProtocols} method.
     * </p>
     * 
     * @return The protocols used in a given TLS security policy.</p>
     *         <p>
     *         The following protocols are possible:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Protocol-TLSv1</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Protocol-TLSv1.1</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Protocol-TLSv1.2</code>
     *         </p>
     *         </li>
     */
    public final List<String> protocols() {
        return protocols;
    }

    /**
     * For responses, this returns true if the service returned a value for the Ciphers property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasCiphers() {
        return ciphers != null && !(ciphers instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The ciphers used by the TLS security policy.
     * </p>
     * <p>
     * The ciphers are listed in order of preference.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCiphers} method.
     * </p>
     * 
     * @return The ciphers used by the TLS security policy.</p>
     *         <p>
     *         The ciphers are listed in order of preference.
     */
    public final List<String> ciphers() {
        return ciphers;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(isDefault());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(hasProtocols() ? protocols() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasCiphers() ? ciphers() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LoadBalancerTlsPolicy)) {
            return false;
        }
        LoadBalancerTlsPolicy other = (LoadBalancerTlsPolicy) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(isDefault(), other.isDefault())
                && Objects.equals(description(), other.description()) && hasProtocols() == other.hasProtocols()
                && Objects.equals(protocols(), other.protocols()) && hasCiphers() == other.hasCiphers()
                && Objects.equals(ciphers(), other.ciphers());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("LoadBalancerTlsPolicy").add("Name", name()).add("IsDefault", isDefault())
                .add("Description", description()).add("Protocols", hasProtocols() ? protocols() : null)
                .add("Ciphers", hasCiphers() ? ciphers() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "isDefault":
            return Optional.ofNullable(clazz.cast(isDefault()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "protocols":
            return Optional.ofNullable(clazz.cast(protocols()));
        case "ciphers":
            return Optional.ofNullable(clazz.cast(ciphers()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<LoadBalancerTlsPolicy, T> g) {
        return obj -> g.apply((LoadBalancerTlsPolicy) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LoadBalancerTlsPolicy> {
        /**
         * <p>
         * The name of the TLS security policy.
         * </p>
         * <p>
         * The following TLS security policy names are possible:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>TLS-2016-08</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TLS-FS-Res-1-2-2019-08</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * You can specify either of these values for the <code>tlsSecurityPolicyName</code> request parameter in the <a
         * href="https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateLoadBalancer.html">
         * CreateLoadBalancer</a> action, and the <code>attributeValue</code> request parameter in the <a
         * href="https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UpdateLoadBalancerAttribute.html"
         * >UpdateLoadBalancerAttribute</a> action.
         * </p>
         * 
         * @param name
         *        The name of the TLS security policy.</p>
         *        <p>
         *        The following TLS security policy names are possible:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>TLS-2016-08</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TLS-FS-Res-1-2-2019-08</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        You can specify either of these values for the <code>tlsSecurityPolicyName</code> request parameter in
         *        the <a
         *        href="https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_CreateLoadBalancer.html"
         *        >CreateLoadBalancer</a> action, and the <code>attributeValue</code> request parameter in the <a
         *        href="https://docs.aws.amazon.com/lightsail/2016-11-28/api-reference/API_UpdateLoadBalancerAttribute.html"
         *        >UpdateLoadBalancerAttribute</a> action.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A Boolean value that indicates whether the TLS security policy is the default.
         * </p>
         * 
         * @param isDefault
         *        A Boolean value that indicates whether the TLS security policy is the default.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isDefault(Boolean isDefault);

        /**
         * <p>
         * The description of the TLS security policy.
         * </p>
         * 
         * @param description
         *        The description of the TLS security policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The protocols used in a given TLS security policy.
         * </p>
         * <p>
         * The following protocols are possible:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Protocol-TLSv1</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Protocol-TLSv1.1</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Protocol-TLSv1.2</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param protocols
         *        The protocols used in a given TLS security policy.</p>
         *        <p>
         *        The following protocols are possible:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Protocol-TLSv1</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Protocol-TLSv1.1</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Protocol-TLSv1.2</code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder protocols(Collection<String> protocols);

        /**
         * <p>
         * The protocols used in a given TLS security policy.
         * </p>
         * <p>
         * The following protocols are possible:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Protocol-TLSv1</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Protocol-TLSv1.1</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Protocol-TLSv1.2</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param protocols
         *        The protocols used in a given TLS security policy.</p>
         *        <p>
         *        The following protocols are possible:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Protocol-TLSv1</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Protocol-TLSv1.1</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Protocol-TLSv1.2</code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder protocols(String... protocols);

        /**
         * <p>
         * The ciphers used by the TLS security policy.
         * </p>
         * <p>
         * The ciphers are listed in order of preference.
         * </p>
         * 
         * @param ciphers
         *        The ciphers used by the TLS security policy.</p>
         *        <p>
         *        The ciphers are listed in order of preference.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ciphers(Collection<String> ciphers);

        /**
         * <p>
         * The ciphers used by the TLS security policy.
         * </p>
         * <p>
         * The ciphers are listed in order of preference.
         * </p>
         * 
         * @param ciphers
         *        The ciphers used by the TLS security policy.</p>
         *        <p>
         *        The ciphers are listed in order of preference.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ciphers(String... ciphers);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private Boolean isDefault;

        private String description;

        private List<String> protocols = DefaultSdkAutoConstructList.getInstance();

        private List<String> ciphers = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(LoadBalancerTlsPolicy model) {
            name(model.name);
            isDefault(model.isDefault);
            description(model.description);
            protocols(model.protocols);
            ciphers(model.ciphers);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Boolean getIsDefault() {
            return isDefault;
        }

        public final void setIsDefault(Boolean isDefault) {
            this.isDefault = isDefault;
        }

        @Override
        public final Builder isDefault(Boolean isDefault) {
            this.isDefault = isDefault;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final Collection<String> getProtocols() {
            if (protocols instanceof SdkAutoConstructList) {
                return null;
            }
            return protocols;
        }

        public final void setProtocols(Collection<String> protocols) {
            this.protocols = StringListCopier.copy(protocols);
        }

        @Override
        public final Builder protocols(Collection<String> protocols) {
            this.protocols = StringListCopier.copy(protocols);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder protocols(String... protocols) {
            protocols(Arrays.asList(protocols));
            return this;
        }

        public final Collection<String> getCiphers() {
            if (ciphers instanceof SdkAutoConstructList) {
                return null;
            }
            return ciphers;
        }

        public final void setCiphers(Collection<String> ciphers) {
            this.ciphers = StringListCopier.copy(ciphers);
        }

        @Override
        public final Builder ciphers(Collection<String> ciphers) {
            this.ciphers = StringListCopier.copy(ciphers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ciphers(String... ciphers) {
            ciphers(Arrays.asList(ciphers));
            return this;
        }

        @Override
        public LoadBalancerTlsPolicy build() {
            return new LoadBalancerTlsPolicy(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
