/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lightsail.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the cache settings of an Amazon Lightsail content delivery network (CDN) distribution.
 * </p>
 * <p>
 * These settings apply only to your distribution's <code>cacheBehaviors</code> (including the
 * <code>defaultCacheBehavior</code>) that have a <code>behavior</code> of <code>cache</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CacheSettings implements SdkPojo, Serializable, ToCopyableBuilder<CacheSettings.Builder, CacheSettings> {
    private static final SdkField<Long> DEFAULT_TTL_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("defaultTTL").getter(getter(CacheSettings::defaultTTL)).setter(setter(Builder::defaultTTL))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("defaultTTL").build()).build();

    private static final SdkField<Long> MINIMUM_TTL_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("minimumTTL").getter(getter(CacheSettings::minimumTTL)).setter(setter(Builder::minimumTTL))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("minimumTTL").build()).build();

    private static final SdkField<Long> MAXIMUM_TTL_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("maximumTTL").getter(getter(CacheSettings::maximumTTL)).setter(setter(Builder::maximumTTL))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maximumTTL").build()).build();

    private static final SdkField<String> ALLOWED_HTTP_METHODS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("allowedHTTPMethods").getter(getter(CacheSettings::allowedHTTPMethods))
            .setter(setter(Builder::allowedHTTPMethods))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("allowedHTTPMethods").build())
            .build();

    private static final SdkField<String> CACHED_HTTP_METHODS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("cachedHTTPMethods").getter(getter(CacheSettings::cachedHTTPMethods))
            .setter(setter(Builder::cachedHTTPMethods))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("cachedHTTPMethods").build()).build();

    private static final SdkField<CookieObject> FORWARDED_COOKIES_FIELD = SdkField
            .<CookieObject> builder(MarshallingType.SDK_POJO).memberName("forwardedCookies")
            .getter(getter(CacheSettings::forwardedCookies)).setter(setter(Builder::forwardedCookies))
            .constructor(CookieObject::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("forwardedCookies").build()).build();

    private static final SdkField<HeaderObject> FORWARDED_HEADERS_FIELD = SdkField
            .<HeaderObject> builder(MarshallingType.SDK_POJO).memberName("forwardedHeaders")
            .getter(getter(CacheSettings::forwardedHeaders)).setter(setter(Builder::forwardedHeaders))
            .constructor(HeaderObject::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("forwardedHeaders").build()).build();

    private static final SdkField<QueryStringObject> FORWARDED_QUERY_STRINGS_FIELD = SdkField
            .<QueryStringObject> builder(MarshallingType.SDK_POJO).memberName("forwardedQueryStrings")
            .getter(getter(CacheSettings::forwardedQueryStrings)).setter(setter(Builder::forwardedQueryStrings))
            .constructor(QueryStringObject::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("forwardedQueryStrings").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DEFAULT_TTL_FIELD,
            MINIMUM_TTL_FIELD, MAXIMUM_TTL_FIELD, ALLOWED_HTTP_METHODS_FIELD, CACHED_HTTP_METHODS_FIELD, FORWARDED_COOKIES_FIELD,
            FORWARDED_HEADERS_FIELD, FORWARDED_QUERY_STRINGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final Long defaultTTL;

    private final Long minimumTTL;

    private final Long maximumTTL;

    private final String allowedHTTPMethods;

    private final String cachedHTTPMethods;

    private final CookieObject forwardedCookies;

    private final HeaderObject forwardedHeaders;

    private final QueryStringObject forwardedQueryStrings;

    private CacheSettings(BuilderImpl builder) {
        this.defaultTTL = builder.defaultTTL;
        this.minimumTTL = builder.minimumTTL;
        this.maximumTTL = builder.maximumTTL;
        this.allowedHTTPMethods = builder.allowedHTTPMethods;
        this.cachedHTTPMethods = builder.cachedHTTPMethods;
        this.forwardedCookies = builder.forwardedCookies;
        this.forwardedHeaders = builder.forwardedHeaders;
        this.forwardedQueryStrings = builder.forwardedQueryStrings;
    }

    /**
     * <p>
     * The default amount of time that objects stay in the distribution's cache before the distribution forwards another
     * request to the origin to determine whether the content has been updated.
     * </p>
     * <note>
     * <p>
     * The value specified applies only when the origin does not add HTTP headers such as
     * <code>Cache-Control max-age</code>, <code>Cache-Control s-maxage</code>, and <code>Expires</code> to objects.
     * </p>
     * </note>
     * 
     * @return The default amount of time that objects stay in the distribution's cache before the distribution forwards
     *         another request to the origin to determine whether the content has been updated.</p> <note>
     *         <p>
     *         The value specified applies only when the origin does not add HTTP headers such as
     *         <code>Cache-Control max-age</code>, <code>Cache-Control s-maxage</code>, and <code>Expires</code> to
     *         objects.
     *         </p>
     */
    public final Long defaultTTL() {
        return defaultTTL;
    }

    /**
     * <p>
     * The minimum amount of time that objects stay in the distribution's cache before the distribution forwards another
     * request to the origin to determine whether the object has been updated.
     * </p>
     * <p>
     * A value of <code>0</code> must be specified for <code>minimumTTL</code> if the distribution is configured to
     * forward all headers to the origin.
     * </p>
     * 
     * @return The minimum amount of time that objects stay in the distribution's cache before the distribution forwards
     *         another request to the origin to determine whether the object has been updated.</p>
     *         <p>
     *         A value of <code>0</code> must be specified for <code>minimumTTL</code> if the distribution is configured
     *         to forward all headers to the origin.
     */
    public final Long minimumTTL() {
        return minimumTTL;
    }

    /**
     * <p>
     * The maximum amount of time that objects stay in the distribution's cache before the distribution forwards another
     * request to the origin to determine whether the object has been updated.
     * </p>
     * <p>
     * The value specified applies only when the origin adds HTTP headers such as <code>Cache-Control max-age</code>,
     * <code>Cache-Control s-maxage</code>, and <code>Expires</code> to objects.
     * </p>
     * 
     * @return The maximum amount of time that objects stay in the distribution's cache before the distribution forwards
     *         another request to the origin to determine whether the object has been updated.</p>
     *         <p>
     *         The value specified applies only when the origin adds HTTP headers such as
     *         <code>Cache-Control max-age</code>, <code>Cache-Control s-maxage</code>, and <code>Expires</code> to
     *         objects.
     */
    public final Long maximumTTL() {
        return maximumTTL;
    }

    /**
     * <p>
     * The HTTP methods that are processed and forwarded to the distribution's origin.
     * </p>
     * <p>
     * You can specify the following options:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>GET,HEAD</code> - The distribution forwards the <code>GET</code> and <code>HEAD</code> methods.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>GET,HEAD,OPTIONS</code> - The distribution forwards the <code>GET</code>, <code>HEAD</code>, and
     * <code>OPTIONS</code> methods.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>GET,HEAD,OPTIONS,PUT,PATCH,POST,DELETE</code> - The distribution forwards the <code>GET</code>,
     * <code>HEAD</code>, <code>OPTIONS</code>, <code>PUT</code>, <code>PATCH</code>, <code>POST</code>, and
     * <code>DELETE</code> methods.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you specify the third option, you might need to restrict access to your distribution's origin so users can't
     * perform operations that you don't want them to. For example, you might not want users to have permission to
     * delete objects from your origin.
     * </p>
     * 
     * @return The HTTP methods that are processed and forwarded to the distribution's origin.</p>
     *         <p>
     *         You can specify the following options:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>GET,HEAD</code> - The distribution forwards the <code>GET</code> and <code>HEAD</code> methods.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>GET,HEAD,OPTIONS</code> - The distribution forwards the <code>GET</code>, <code>HEAD</code>, and
     *         <code>OPTIONS</code> methods.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>GET,HEAD,OPTIONS,PUT,PATCH,POST,DELETE</code> - The distribution forwards the <code>GET</code>,
     *         <code>HEAD</code>, <code>OPTIONS</code>, <code>PUT</code>, <code>PATCH</code>, <code>POST</code>, and
     *         <code>DELETE</code> methods.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you specify the third option, you might need to restrict access to your distribution's origin so users
     *         can't perform operations that you don't want them to. For example, you might not want users to have
     *         permission to delete objects from your origin.
     */
    public final String allowedHTTPMethods() {
        return allowedHTTPMethods;
    }

    /**
     * <p>
     * The HTTP method responses that are cached by your distribution.
     * </p>
     * <p>
     * You can specify the following options:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>GET,HEAD</code> - The distribution caches responses to the <code>GET</code> and <code>HEAD</code> methods.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>GET,HEAD,OPTIONS</code> - The distribution caches responses to the <code>GET</code>, <code>HEAD</code>, and
     * <code>OPTIONS</code> methods.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The HTTP method responses that are cached by your distribution.</p>
     *         <p>
     *         You can specify the following options:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>GET,HEAD</code> - The distribution caches responses to the <code>GET</code> and <code>HEAD</code>
     *         methods.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>GET,HEAD,OPTIONS</code> - The distribution caches responses to the <code>GET</code>,
     *         <code>HEAD</code>, and <code>OPTIONS</code> methods.
     *         </p>
     *         </li>
     */
    public final String cachedHTTPMethods() {
        return cachedHTTPMethods;
    }

    /**
     * <p>
     * An object that describes the cookies that are forwarded to the origin. Your content is cached based on the
     * cookies that are forwarded.
     * </p>
     * 
     * @return An object that describes the cookies that are forwarded to the origin. Your content is cached based on
     *         the cookies that are forwarded.
     */
    public final CookieObject forwardedCookies() {
        return forwardedCookies;
    }

    /**
     * <p>
     * An object that describes the headers that are forwarded to the origin. Your content is cached based on the
     * headers that are forwarded.
     * </p>
     * 
     * @return An object that describes the headers that are forwarded to the origin. Your content is cached based on
     *         the headers that are forwarded.
     */
    public final HeaderObject forwardedHeaders() {
        return forwardedHeaders;
    }

    /**
     * <p>
     * An object that describes the query strings that are forwarded to the origin. Your content is cached based on the
     * query strings that are forwarded.
     * </p>
     * 
     * @return An object that describes the query strings that are forwarded to the origin. Your content is cached based
     *         on the query strings that are forwarded.
     */
    public final QueryStringObject forwardedQueryStrings() {
        return forwardedQueryStrings;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(defaultTTL());
        hashCode = 31 * hashCode + Objects.hashCode(minimumTTL());
        hashCode = 31 * hashCode + Objects.hashCode(maximumTTL());
        hashCode = 31 * hashCode + Objects.hashCode(allowedHTTPMethods());
        hashCode = 31 * hashCode + Objects.hashCode(cachedHTTPMethods());
        hashCode = 31 * hashCode + Objects.hashCode(forwardedCookies());
        hashCode = 31 * hashCode + Objects.hashCode(forwardedHeaders());
        hashCode = 31 * hashCode + Objects.hashCode(forwardedQueryStrings());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CacheSettings)) {
            return false;
        }
        CacheSettings other = (CacheSettings) obj;
        return Objects.equals(defaultTTL(), other.defaultTTL()) && Objects.equals(minimumTTL(), other.minimumTTL())
                && Objects.equals(maximumTTL(), other.maximumTTL())
                && Objects.equals(allowedHTTPMethods(), other.allowedHTTPMethods())
                && Objects.equals(cachedHTTPMethods(), other.cachedHTTPMethods())
                && Objects.equals(forwardedCookies(), other.forwardedCookies())
                && Objects.equals(forwardedHeaders(), other.forwardedHeaders())
                && Objects.equals(forwardedQueryStrings(), other.forwardedQueryStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CacheSettings").add("DefaultTTL", defaultTTL()).add("MinimumTTL", minimumTTL())
                .add("MaximumTTL", maximumTTL()).add("AllowedHTTPMethods", allowedHTTPMethods())
                .add("CachedHTTPMethods", cachedHTTPMethods()).add("ForwardedCookies", forwardedCookies())
                .add("ForwardedHeaders", forwardedHeaders()).add("ForwardedQueryStrings", forwardedQueryStrings()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "defaultTTL":
            return Optional.ofNullable(clazz.cast(defaultTTL()));
        case "minimumTTL":
            return Optional.ofNullable(clazz.cast(minimumTTL()));
        case "maximumTTL":
            return Optional.ofNullable(clazz.cast(maximumTTL()));
        case "allowedHTTPMethods":
            return Optional.ofNullable(clazz.cast(allowedHTTPMethods()));
        case "cachedHTTPMethods":
            return Optional.ofNullable(clazz.cast(cachedHTTPMethods()));
        case "forwardedCookies":
            return Optional.ofNullable(clazz.cast(forwardedCookies()));
        case "forwardedHeaders":
            return Optional.ofNullable(clazz.cast(forwardedHeaders()));
        case "forwardedQueryStrings":
            return Optional.ofNullable(clazz.cast(forwardedQueryStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CacheSettings, T> g) {
        return obj -> g.apply((CacheSettings) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CacheSettings> {
        /**
         * <p>
         * The default amount of time that objects stay in the distribution's cache before the distribution forwards
         * another request to the origin to determine whether the content has been updated.
         * </p>
         * <note>
         * <p>
         * The value specified applies only when the origin does not add HTTP headers such as
         * <code>Cache-Control max-age</code>, <code>Cache-Control s-maxage</code>, and <code>Expires</code> to objects.
         * </p>
         * </note>
         * 
         * @param defaultTTL
         *        The default amount of time that objects stay in the distribution's cache before the distribution
         *        forwards another request to the origin to determine whether the content has been updated.</p> <note>
         *        <p>
         *        The value specified applies only when the origin does not add HTTP headers such as
         *        <code>Cache-Control max-age</code>, <code>Cache-Control s-maxage</code>, and <code>Expires</code> to
         *        objects.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultTTL(Long defaultTTL);

        /**
         * <p>
         * The minimum amount of time that objects stay in the distribution's cache before the distribution forwards
         * another request to the origin to determine whether the object has been updated.
         * </p>
         * <p>
         * A value of <code>0</code> must be specified for <code>minimumTTL</code> if the distribution is configured to
         * forward all headers to the origin.
         * </p>
         * 
         * @param minimumTTL
         *        The minimum amount of time that objects stay in the distribution's cache before the distribution
         *        forwards another request to the origin to determine whether the object has been updated.</p>
         *        <p>
         *        A value of <code>0</code> must be specified for <code>minimumTTL</code> if the distribution is
         *        configured to forward all headers to the origin.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minimumTTL(Long minimumTTL);

        /**
         * <p>
         * The maximum amount of time that objects stay in the distribution's cache before the distribution forwards
         * another request to the origin to determine whether the object has been updated.
         * </p>
         * <p>
         * The value specified applies only when the origin adds HTTP headers such as <code>Cache-Control max-age</code>, <code>Cache-Control s-maxage</code>, and <code>Expires</code> to objects.
         * </p>
         * 
         * @param maximumTTL
         *        The maximum amount of time that objects stay in the distribution's cache before the distribution
         *        forwards another request to the origin to determine whether the object has been updated.</p>
         *        <p>
         *        The value specified applies only when the origin adds HTTP headers such as
         *        <code>Cache-Control max-age</code>, <code>Cache-Control s-maxage</code>, and <code>Expires</code> to
         *        objects.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maximumTTL(Long maximumTTL);

        /**
         * <p>
         * The HTTP methods that are processed and forwarded to the distribution's origin.
         * </p>
         * <p>
         * You can specify the following options:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>GET,HEAD</code> - The distribution forwards the <code>GET</code> and <code>HEAD</code> methods.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>GET,HEAD,OPTIONS</code> - The distribution forwards the <code>GET</code>, <code>HEAD</code>, and
         * <code>OPTIONS</code> methods.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>GET,HEAD,OPTIONS,PUT,PATCH,POST,DELETE</code> - The distribution forwards the <code>GET</code>,
         * <code>HEAD</code>, <code>OPTIONS</code>, <code>PUT</code>, <code>PATCH</code>, <code>POST</code>, and
         * <code>DELETE</code> methods.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you specify the third option, you might need to restrict access to your distribution's origin so users
         * can't perform operations that you don't want them to. For example, you might not want users to have
         * permission to delete objects from your origin.
         * </p>
         * 
         * @param allowedHTTPMethods
         *        The HTTP methods that are processed and forwarded to the distribution's origin.</p>
         *        <p>
         *        You can specify the following options:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>GET,HEAD</code> - The distribution forwards the <code>GET</code> and <code>HEAD</code> methods.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>GET,HEAD,OPTIONS</code> - The distribution forwards the <code>GET</code>, <code>HEAD</code>, and
         *        <code>OPTIONS</code> methods.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>GET,HEAD,OPTIONS,PUT,PATCH,POST,DELETE</code> - The distribution forwards the <code>GET</code>,
         *        <code>HEAD</code>, <code>OPTIONS</code>, <code>PUT</code>, <code>PATCH</code>, <code>POST</code>, and
         *        <code>DELETE</code> methods.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you specify the third option, you might need to restrict access to your distribution's origin so
         *        users can't perform operations that you don't want them to. For example, you might not want users to
         *        have permission to delete objects from your origin.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowedHTTPMethods(String allowedHTTPMethods);

        /**
         * <p>
         * The HTTP method responses that are cached by your distribution.
         * </p>
         * <p>
         * You can specify the following options:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>GET,HEAD</code> - The distribution caches responses to the <code>GET</code> and <code>HEAD</code>
         * methods.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>GET,HEAD,OPTIONS</code> - The distribution caches responses to the <code>GET</code>, <code>HEAD</code>,
         * and <code>OPTIONS</code> methods.
         * </p>
         * </li>
         * </ul>
         * 
         * @param cachedHTTPMethods
         *        The HTTP method responses that are cached by your distribution.</p>
         *        <p>
         *        You can specify the following options:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>GET,HEAD</code> - The distribution caches responses to the <code>GET</code> and
         *        <code>HEAD</code> methods.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>GET,HEAD,OPTIONS</code> - The distribution caches responses to the <code>GET</code>,
         *        <code>HEAD</code>, and <code>OPTIONS</code> methods.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cachedHTTPMethods(String cachedHTTPMethods);

        /**
         * <p>
         * An object that describes the cookies that are forwarded to the origin. Your content is cached based on the
         * cookies that are forwarded.
         * </p>
         * 
         * @param forwardedCookies
         *        An object that describes the cookies that are forwarded to the origin. Your content is cached based on
         *        the cookies that are forwarded.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder forwardedCookies(CookieObject forwardedCookies);

        /**
         * <p>
         * An object that describes the cookies that are forwarded to the origin. Your content is cached based on the
         * cookies that are forwarded.
         * </p>
         * This is a convenience method that creates an instance of the {@link CookieObject.Builder} avoiding the need
         * to create one manually via {@link CookieObject#builder()}.
         *
         * When the {@link Consumer} completes, {@link CookieObject.Builder#build()} is called immediately and its
         * result is passed to {@link #forwardedCookies(CookieObject)}.
         * 
         * @param forwardedCookies
         *        a consumer that will call methods on {@link CookieObject.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #forwardedCookies(CookieObject)
         */
        default Builder forwardedCookies(Consumer<CookieObject.Builder> forwardedCookies) {
            return forwardedCookies(CookieObject.builder().applyMutation(forwardedCookies).build());
        }

        /**
         * <p>
         * An object that describes the headers that are forwarded to the origin. Your content is cached based on the
         * headers that are forwarded.
         * </p>
         * 
         * @param forwardedHeaders
         *        An object that describes the headers that are forwarded to the origin. Your content is cached based on
         *        the headers that are forwarded.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder forwardedHeaders(HeaderObject forwardedHeaders);

        /**
         * <p>
         * An object that describes the headers that are forwarded to the origin. Your content is cached based on the
         * headers that are forwarded.
         * </p>
         * This is a convenience method that creates an instance of the {@link HeaderObject.Builder} avoiding the need
         * to create one manually via {@link HeaderObject#builder()}.
         *
         * When the {@link Consumer} completes, {@link HeaderObject.Builder#build()} is called immediately and its
         * result is passed to {@link #forwardedHeaders(HeaderObject)}.
         * 
         * @param forwardedHeaders
         *        a consumer that will call methods on {@link HeaderObject.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #forwardedHeaders(HeaderObject)
         */
        default Builder forwardedHeaders(Consumer<HeaderObject.Builder> forwardedHeaders) {
            return forwardedHeaders(HeaderObject.builder().applyMutation(forwardedHeaders).build());
        }

        /**
         * <p>
         * An object that describes the query strings that are forwarded to the origin. Your content is cached based on
         * the query strings that are forwarded.
         * </p>
         * 
         * @param forwardedQueryStrings
         *        An object that describes the query strings that are forwarded to the origin. Your content is cached
         *        based on the query strings that are forwarded.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder forwardedQueryStrings(QueryStringObject forwardedQueryStrings);

        /**
         * <p>
         * An object that describes the query strings that are forwarded to the origin. Your content is cached based on
         * the query strings that are forwarded.
         * </p>
         * This is a convenience method that creates an instance of the {@link QueryStringObject.Builder} avoiding the
         * need to create one manually via {@link QueryStringObject#builder()}.
         *
         * When the {@link Consumer} completes, {@link QueryStringObject.Builder#build()} is called immediately and its
         * result is passed to {@link #forwardedQueryStrings(QueryStringObject)}.
         * 
         * @param forwardedQueryStrings
         *        a consumer that will call methods on {@link QueryStringObject.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #forwardedQueryStrings(QueryStringObject)
         */
        default Builder forwardedQueryStrings(Consumer<QueryStringObject.Builder> forwardedQueryStrings) {
            return forwardedQueryStrings(QueryStringObject.builder().applyMutation(forwardedQueryStrings).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private Long defaultTTL;

        private Long minimumTTL;

        private Long maximumTTL;

        private String allowedHTTPMethods;

        private String cachedHTTPMethods;

        private CookieObject forwardedCookies;

        private HeaderObject forwardedHeaders;

        private QueryStringObject forwardedQueryStrings;

        private BuilderImpl() {
        }

        private BuilderImpl(CacheSettings model) {
            defaultTTL(model.defaultTTL);
            minimumTTL(model.minimumTTL);
            maximumTTL(model.maximumTTL);
            allowedHTTPMethods(model.allowedHTTPMethods);
            cachedHTTPMethods(model.cachedHTTPMethods);
            forwardedCookies(model.forwardedCookies);
            forwardedHeaders(model.forwardedHeaders);
            forwardedQueryStrings(model.forwardedQueryStrings);
        }

        public final Long getDefaultTTL() {
            return defaultTTL;
        }

        public final void setDefaultTTL(Long defaultTTL) {
            this.defaultTTL = defaultTTL;
        }

        @Override
        public final Builder defaultTTL(Long defaultTTL) {
            this.defaultTTL = defaultTTL;
            return this;
        }

        public final Long getMinimumTTL() {
            return minimumTTL;
        }

        public final void setMinimumTTL(Long minimumTTL) {
            this.minimumTTL = minimumTTL;
        }

        @Override
        public final Builder minimumTTL(Long minimumTTL) {
            this.minimumTTL = minimumTTL;
            return this;
        }

        public final Long getMaximumTTL() {
            return maximumTTL;
        }

        public final void setMaximumTTL(Long maximumTTL) {
            this.maximumTTL = maximumTTL;
        }

        @Override
        public final Builder maximumTTL(Long maximumTTL) {
            this.maximumTTL = maximumTTL;
            return this;
        }

        public final String getAllowedHTTPMethods() {
            return allowedHTTPMethods;
        }

        public final void setAllowedHTTPMethods(String allowedHTTPMethods) {
            this.allowedHTTPMethods = allowedHTTPMethods;
        }

        @Override
        public final Builder allowedHTTPMethods(String allowedHTTPMethods) {
            this.allowedHTTPMethods = allowedHTTPMethods;
            return this;
        }

        public final String getCachedHTTPMethods() {
            return cachedHTTPMethods;
        }

        public final void setCachedHTTPMethods(String cachedHTTPMethods) {
            this.cachedHTTPMethods = cachedHTTPMethods;
        }

        @Override
        public final Builder cachedHTTPMethods(String cachedHTTPMethods) {
            this.cachedHTTPMethods = cachedHTTPMethods;
            return this;
        }

        public final CookieObject.Builder getForwardedCookies() {
            return forwardedCookies != null ? forwardedCookies.toBuilder() : null;
        }

        public final void setForwardedCookies(CookieObject.BuilderImpl forwardedCookies) {
            this.forwardedCookies = forwardedCookies != null ? forwardedCookies.build() : null;
        }

        @Override
        public final Builder forwardedCookies(CookieObject forwardedCookies) {
            this.forwardedCookies = forwardedCookies;
            return this;
        }

        public final HeaderObject.Builder getForwardedHeaders() {
            return forwardedHeaders != null ? forwardedHeaders.toBuilder() : null;
        }

        public final void setForwardedHeaders(HeaderObject.BuilderImpl forwardedHeaders) {
            this.forwardedHeaders = forwardedHeaders != null ? forwardedHeaders.build() : null;
        }

        @Override
        public final Builder forwardedHeaders(HeaderObject forwardedHeaders) {
            this.forwardedHeaders = forwardedHeaders;
            return this;
        }

        public final QueryStringObject.Builder getForwardedQueryStrings() {
            return forwardedQueryStrings != null ? forwardedQueryStrings.toBuilder() : null;
        }

        public final void setForwardedQueryStrings(QueryStringObject.BuilderImpl forwardedQueryStrings) {
            this.forwardedQueryStrings = forwardedQueryStrings != null ? forwardedQueryStrings.build() : null;
        }

        @Override
        public final Builder forwardedQueryStrings(QueryStringObject forwardedQueryStrings) {
            this.forwardedQueryStrings = forwardedQueryStrings;
            return this;
        }

        @Override
        public CacheSettings build() {
            return new CacheSettings(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
