/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lightsail.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a bundle, which is a set of specs describing your virtual private server (or <i>instance</i>).
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Bundle implements SdkPojo, Serializable, ToCopyableBuilder<Bundle.Builder, Bundle> {
    private static final SdkField<Float> PRICE_FIELD = SdkField.<Float> builder(MarshallingType.FLOAT).memberName("price")
            .getter(getter(Bundle::price)).setter(setter(Builder::price))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("price").build()).build();

    private static final SdkField<Integer> CPU_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("cpuCount").getter(getter(Bundle::cpuCount)).setter(setter(Builder::cpuCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("cpuCount").build()).build();

    private static final SdkField<Integer> DISK_SIZE_IN_GB_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("diskSizeInGb").getter(getter(Bundle::diskSizeInGb)).setter(setter(Builder::diskSizeInGb))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("diskSizeInGb").build()).build();

    private static final SdkField<String> BUNDLE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("bundleId").getter(getter(Bundle::bundleId)).setter(setter(Builder::bundleId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("bundleId").build()).build();

    private static final SdkField<String> INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("instanceType").getter(getter(Bundle::instanceType)).setter(setter(Builder::instanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("instanceType").build()).build();

    private static final SdkField<Boolean> IS_ACTIVE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("isActive").getter(getter(Bundle::isActive)).setter(setter(Builder::isActive))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("isActive").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(Bundle::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<Integer> POWER_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("power")
            .getter(getter(Bundle::power)).setter(setter(Builder::power))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("power").build()).build();

    private static final SdkField<Float> RAM_SIZE_IN_GB_FIELD = SdkField.<Float> builder(MarshallingType.FLOAT)
            .memberName("ramSizeInGb").getter(getter(Bundle::ramSizeInGb)).setter(setter(Builder::ramSizeInGb))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ramSizeInGb").build()).build();

    private static final SdkField<Integer> TRANSFER_PER_MONTH_IN_GB_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("transferPerMonthInGb").getter(getter(Bundle::transferPerMonthInGb))
            .setter(setter(Builder::transferPerMonthInGb))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("transferPerMonthInGb").build())
            .build();

    private static final SdkField<List<String>> SUPPORTED_PLATFORMS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("supportedPlatforms")
            .getter(getter(Bundle::supportedPlatformsAsStrings))
            .setter(setter(Builder::supportedPlatformsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("supportedPlatforms").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> SUPPORTED_APP_CATEGORIES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("supportedAppCategories")
            .getter(getter(Bundle::supportedAppCategoriesAsStrings))
            .setter(setter(Builder::supportedAppCategoriesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("supportedAppCategories").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PRICE_FIELD, CPU_COUNT_FIELD,
            DISK_SIZE_IN_GB_FIELD, BUNDLE_ID_FIELD, INSTANCE_TYPE_FIELD, IS_ACTIVE_FIELD, NAME_FIELD, POWER_FIELD,
            RAM_SIZE_IN_GB_FIELD, TRANSFER_PER_MONTH_IN_GB_FIELD, SUPPORTED_PLATFORMS_FIELD, SUPPORTED_APP_CATEGORIES_FIELD));

    private static final long serialVersionUID = 1L;

    private final Float price;

    private final Integer cpuCount;

    private final Integer diskSizeInGb;

    private final String bundleId;

    private final String instanceType;

    private final Boolean isActive;

    private final String name;

    private final Integer power;

    private final Float ramSizeInGb;

    private final Integer transferPerMonthInGb;

    private final List<String> supportedPlatforms;

    private final List<String> supportedAppCategories;

    private Bundle(BuilderImpl builder) {
        this.price = builder.price;
        this.cpuCount = builder.cpuCount;
        this.diskSizeInGb = builder.diskSizeInGb;
        this.bundleId = builder.bundleId;
        this.instanceType = builder.instanceType;
        this.isActive = builder.isActive;
        this.name = builder.name;
        this.power = builder.power;
        this.ramSizeInGb = builder.ramSizeInGb;
        this.transferPerMonthInGb = builder.transferPerMonthInGb;
        this.supportedPlatforms = builder.supportedPlatforms;
        this.supportedAppCategories = builder.supportedAppCategories;
    }

    /**
     * <p>
     * The price in US dollars (<code>5.0</code>) of the bundle.
     * </p>
     * 
     * @return The price in US dollars (<code>5.0</code>) of the bundle.
     */
    public final Float price() {
        return price;
    }

    /**
     * <p>
     * The number of vCPUs included in the bundle (<code>2</code>).
     * </p>
     * 
     * @return The number of vCPUs included in the bundle (<code>2</code>).
     */
    public final Integer cpuCount() {
        return cpuCount;
    }

    /**
     * <p>
     * The size of the SSD (<code>30</code>).
     * </p>
     * 
     * @return The size of the SSD (<code>30</code>).
     */
    public final Integer diskSizeInGb() {
        return diskSizeInGb;
    }

    /**
     * <p>
     * The bundle ID (<code>micro_1_0</code>).
     * </p>
     * 
     * @return The bundle ID (<code>micro_1_0</code>).
     */
    public final String bundleId() {
        return bundleId;
    }

    /**
     * <p>
     * The Amazon EC2 instance type (<code>t2.micro</code>).
     * </p>
     * 
     * @return The Amazon EC2 instance type (<code>t2.micro</code>).
     */
    public final String instanceType() {
        return instanceType;
    }

    /**
     * <p>
     * A Boolean value indicating whether the bundle is active.
     * </p>
     * 
     * @return A Boolean value indicating whether the bundle is active.
     */
    public final Boolean isActive() {
        return isActive;
    }

    /**
     * <p>
     * A friendly name for the bundle (<code>Micro</code>).
     * </p>
     * 
     * @return A friendly name for the bundle (<code>Micro</code>).
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * A numeric value that represents the power of the bundle (<code>500</code>). You can use the bundle's power value
     * in conjunction with a blueprint's minimum power value to determine whether the blueprint will run on the bundle.
     * For example, you need a bundle with a power value of 500 or more to create an instance that uses a blueprint with
     * a minimum power value of 500.
     * </p>
     * 
     * @return A numeric value that represents the power of the bundle (<code>500</code>). You can use the bundle's
     *         power value in conjunction with a blueprint's minimum power value to determine whether the blueprint will
     *         run on the bundle. For example, you need a bundle with a power value of 500 or more to create an instance
     *         that uses a blueprint with a minimum power value of 500.
     */
    public final Integer power() {
        return power;
    }

    /**
     * <p>
     * The amount of RAM in GB (<code>2.0</code>).
     * </p>
     * 
     * @return The amount of RAM in GB (<code>2.0</code>).
     */
    public final Float ramSizeInGb() {
        return ramSizeInGb;
    }

    /**
     * <p>
     * The data transfer rate per month in GB (<code>2000</code>).
     * </p>
     * 
     * @return The data transfer rate per month in GB (<code>2000</code>).
     */
    public final Integer transferPerMonthInGb() {
        return transferPerMonthInGb;
    }

    /**
     * <p>
     * The operating system platform (Linux/Unix-based or Windows Server-based) that the bundle supports. You can only
     * launch a <code>WINDOWS</code> bundle on a blueprint that supports the <code>WINDOWS</code> platform.
     * <code>LINUX_UNIX</code> blueprints require a <code>LINUX_UNIX</code> bundle.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSupportedPlatforms} method.
     * </p>
     * 
     * @return The operating system platform (Linux/Unix-based or Windows Server-based) that the bundle supports. You
     *         can only launch a <code>WINDOWS</code> bundle on a blueprint that supports the <code>WINDOWS</code>
     *         platform. <code>LINUX_UNIX</code> blueprints require a <code>LINUX_UNIX</code> bundle.
     */
    public final List<InstancePlatform> supportedPlatforms() {
        return InstancePlatformListCopier.copyStringToEnum(supportedPlatforms);
    }

    /**
     * For responses, this returns true if the service returned a value for the SupportedPlatforms property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSupportedPlatforms() {
        return supportedPlatforms != null && !(supportedPlatforms instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The operating system platform (Linux/Unix-based or Windows Server-based) that the bundle supports. You can only
     * launch a <code>WINDOWS</code> bundle on a blueprint that supports the <code>WINDOWS</code> platform.
     * <code>LINUX_UNIX</code> blueprints require a <code>LINUX_UNIX</code> bundle.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSupportedPlatforms} method.
     * </p>
     * 
     * @return The operating system platform (Linux/Unix-based or Windows Server-based) that the bundle supports. You
     *         can only launch a <code>WINDOWS</code> bundle on a blueprint that supports the <code>WINDOWS</code>
     *         platform. <code>LINUX_UNIX</code> blueprints require a <code>LINUX_UNIX</code> bundle.
     */
    public final List<String> supportedPlatformsAsStrings() {
        return supportedPlatforms;
    }

    /**
     * <p>
     * Virtual computer blueprints that are supported by a Lightsail for Research bundle.
     * </p>
     * <important>
     * <p>
     * This parameter only applies to Lightsail for Research resources.
     * </p>
     * </important>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSupportedAppCategories} method.
     * </p>
     * 
     * @return Virtual computer blueprints that are supported by a Lightsail for Research bundle.</p> <important>
     *         <p>
     *         This parameter only applies to Lightsail for Research resources.
     *         </p>
     */
    public final List<AppCategory> supportedAppCategories() {
        return AppCategoryListCopier.copyStringToEnum(supportedAppCategories);
    }

    /**
     * For responses, this returns true if the service returned a value for the SupportedAppCategories property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasSupportedAppCategories() {
        return supportedAppCategories != null && !(supportedAppCategories instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Virtual computer blueprints that are supported by a Lightsail for Research bundle.
     * </p>
     * <important>
     * <p>
     * This parameter only applies to Lightsail for Research resources.
     * </p>
     * </important>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSupportedAppCategories} method.
     * </p>
     * 
     * @return Virtual computer blueprints that are supported by a Lightsail for Research bundle.</p> <important>
     *         <p>
     *         This parameter only applies to Lightsail for Research resources.
     *         </p>
     */
    public final List<String> supportedAppCategoriesAsStrings() {
        return supportedAppCategories;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(price());
        hashCode = 31 * hashCode + Objects.hashCode(cpuCount());
        hashCode = 31 * hashCode + Objects.hashCode(diskSizeInGb());
        hashCode = 31 * hashCode + Objects.hashCode(bundleId());
        hashCode = 31 * hashCode + Objects.hashCode(instanceType());
        hashCode = 31 * hashCode + Objects.hashCode(isActive());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(power());
        hashCode = 31 * hashCode + Objects.hashCode(ramSizeInGb());
        hashCode = 31 * hashCode + Objects.hashCode(transferPerMonthInGb());
        hashCode = 31 * hashCode + Objects.hashCode(hasSupportedPlatforms() ? supportedPlatformsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSupportedAppCategories() ? supportedAppCategoriesAsStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Bundle)) {
            return false;
        }
        Bundle other = (Bundle) obj;
        return Objects.equals(price(), other.price()) && Objects.equals(cpuCount(), other.cpuCount())
                && Objects.equals(diskSizeInGb(), other.diskSizeInGb()) && Objects.equals(bundleId(), other.bundleId())
                && Objects.equals(instanceType(), other.instanceType()) && Objects.equals(isActive(), other.isActive())
                && Objects.equals(name(), other.name()) && Objects.equals(power(), other.power())
                && Objects.equals(ramSizeInGb(), other.ramSizeInGb())
                && Objects.equals(transferPerMonthInGb(), other.transferPerMonthInGb())
                && hasSupportedPlatforms() == other.hasSupportedPlatforms()
                && Objects.equals(supportedPlatformsAsStrings(), other.supportedPlatformsAsStrings())
                && hasSupportedAppCategories() == other.hasSupportedAppCategories()
                && Objects.equals(supportedAppCategoriesAsStrings(), other.supportedAppCategoriesAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Bundle").add("Price", price()).add("CpuCount", cpuCount()).add("DiskSizeInGb", diskSizeInGb())
                .add("BundleId", bundleId()).add("InstanceType", instanceType()).add("IsActive", isActive()).add("Name", name())
                .add("Power", power()).add("RamSizeInGb", ramSizeInGb()).add("TransferPerMonthInGb", transferPerMonthInGb())
                .add("SupportedPlatforms", hasSupportedPlatforms() ? supportedPlatformsAsStrings() : null)
                .add("SupportedAppCategories", hasSupportedAppCategories() ? supportedAppCategoriesAsStrings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "price":
            return Optional.ofNullable(clazz.cast(price()));
        case "cpuCount":
            return Optional.ofNullable(clazz.cast(cpuCount()));
        case "diskSizeInGb":
            return Optional.ofNullable(clazz.cast(diskSizeInGb()));
        case "bundleId":
            return Optional.ofNullable(clazz.cast(bundleId()));
        case "instanceType":
            return Optional.ofNullable(clazz.cast(instanceType()));
        case "isActive":
            return Optional.ofNullable(clazz.cast(isActive()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "power":
            return Optional.ofNullable(clazz.cast(power()));
        case "ramSizeInGb":
            return Optional.ofNullable(clazz.cast(ramSizeInGb()));
        case "transferPerMonthInGb":
            return Optional.ofNullable(clazz.cast(transferPerMonthInGb()));
        case "supportedPlatforms":
            return Optional.ofNullable(clazz.cast(supportedPlatformsAsStrings()));
        case "supportedAppCategories":
            return Optional.ofNullable(clazz.cast(supportedAppCategoriesAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Bundle, T> g) {
        return obj -> g.apply((Bundle) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Bundle> {
        /**
         * <p>
         * The price in US dollars (<code>5.0</code>) of the bundle.
         * </p>
         * 
         * @param price
         *        The price in US dollars (<code>5.0</code>) of the bundle.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder price(Float price);

        /**
         * <p>
         * The number of vCPUs included in the bundle (<code>2</code>).
         * </p>
         * 
         * @param cpuCount
         *        The number of vCPUs included in the bundle (<code>2</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cpuCount(Integer cpuCount);

        /**
         * <p>
         * The size of the SSD (<code>30</code>).
         * </p>
         * 
         * @param diskSizeInGb
         *        The size of the SSD (<code>30</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder diskSizeInGb(Integer diskSizeInGb);

        /**
         * <p>
         * The bundle ID (<code>micro_1_0</code>).
         * </p>
         * 
         * @param bundleId
         *        The bundle ID (<code>micro_1_0</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bundleId(String bundleId);

        /**
         * <p>
         * The Amazon EC2 instance type (<code>t2.micro</code>).
         * </p>
         * 
         * @param instanceType
         *        The Amazon EC2 instance type (<code>t2.micro</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceType(String instanceType);

        /**
         * <p>
         * A Boolean value indicating whether the bundle is active.
         * </p>
         * 
         * @param isActive
         *        A Boolean value indicating whether the bundle is active.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isActive(Boolean isActive);

        /**
         * <p>
         * A friendly name for the bundle (<code>Micro</code>).
         * </p>
         * 
         * @param name
         *        A friendly name for the bundle (<code>Micro</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A numeric value that represents the power of the bundle (<code>500</code>). You can use the bundle's power
         * value in conjunction with a blueprint's minimum power value to determine whether the blueprint will run on
         * the bundle. For example, you need a bundle with a power value of 500 or more to create an instance that uses
         * a blueprint with a minimum power value of 500.
         * </p>
         * 
         * @param power
         *        A numeric value that represents the power of the bundle (<code>500</code>). You can use the bundle's
         *        power value in conjunction with a blueprint's minimum power value to determine whether the blueprint
         *        will run on the bundle. For example, you need a bundle with a power value of 500 or more to create an
         *        instance that uses a blueprint with a minimum power value of 500.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder power(Integer power);

        /**
         * <p>
         * The amount of RAM in GB (<code>2.0</code>).
         * </p>
         * 
         * @param ramSizeInGb
         *        The amount of RAM in GB (<code>2.0</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ramSizeInGb(Float ramSizeInGb);

        /**
         * <p>
         * The data transfer rate per month in GB (<code>2000</code>).
         * </p>
         * 
         * @param transferPerMonthInGb
         *        The data transfer rate per month in GB (<code>2000</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transferPerMonthInGb(Integer transferPerMonthInGb);

        /**
         * <p>
         * The operating system platform (Linux/Unix-based or Windows Server-based) that the bundle supports. You can
         * only launch a <code>WINDOWS</code> bundle on a blueprint that supports the <code>WINDOWS</code> platform.
         * <code>LINUX_UNIX</code> blueprints require a <code>LINUX_UNIX</code> bundle.
         * </p>
         * 
         * @param supportedPlatforms
         *        The operating system platform (Linux/Unix-based or Windows Server-based) that the bundle supports. You
         *        can only launch a <code>WINDOWS</code> bundle on a blueprint that supports the <code>WINDOWS</code>
         *        platform. <code>LINUX_UNIX</code> blueprints require a <code>LINUX_UNIX</code> bundle.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedPlatformsWithStrings(Collection<String> supportedPlatforms);

        /**
         * <p>
         * The operating system platform (Linux/Unix-based or Windows Server-based) that the bundle supports. You can
         * only launch a <code>WINDOWS</code> bundle on a blueprint that supports the <code>WINDOWS</code> platform.
         * <code>LINUX_UNIX</code> blueprints require a <code>LINUX_UNIX</code> bundle.
         * </p>
         * 
         * @param supportedPlatforms
         *        The operating system platform (Linux/Unix-based or Windows Server-based) that the bundle supports. You
         *        can only launch a <code>WINDOWS</code> bundle on a blueprint that supports the <code>WINDOWS</code>
         *        platform. <code>LINUX_UNIX</code> blueprints require a <code>LINUX_UNIX</code> bundle.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedPlatformsWithStrings(String... supportedPlatforms);

        /**
         * <p>
         * The operating system platform (Linux/Unix-based or Windows Server-based) that the bundle supports. You can
         * only launch a <code>WINDOWS</code> bundle on a blueprint that supports the <code>WINDOWS</code> platform.
         * <code>LINUX_UNIX</code> blueprints require a <code>LINUX_UNIX</code> bundle.
         * </p>
         * 
         * @param supportedPlatforms
         *        The operating system platform (Linux/Unix-based or Windows Server-based) that the bundle supports. You
         *        can only launch a <code>WINDOWS</code> bundle on a blueprint that supports the <code>WINDOWS</code>
         *        platform. <code>LINUX_UNIX</code> blueprints require a <code>LINUX_UNIX</code> bundle.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedPlatforms(Collection<InstancePlatform> supportedPlatforms);

        /**
         * <p>
         * The operating system platform (Linux/Unix-based or Windows Server-based) that the bundle supports. You can
         * only launch a <code>WINDOWS</code> bundle on a blueprint that supports the <code>WINDOWS</code> platform.
         * <code>LINUX_UNIX</code> blueprints require a <code>LINUX_UNIX</code> bundle.
         * </p>
         * 
         * @param supportedPlatforms
         *        The operating system platform (Linux/Unix-based or Windows Server-based) that the bundle supports. You
         *        can only launch a <code>WINDOWS</code> bundle on a blueprint that supports the <code>WINDOWS</code>
         *        platform. <code>LINUX_UNIX</code> blueprints require a <code>LINUX_UNIX</code> bundle.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedPlatforms(InstancePlatform... supportedPlatforms);

        /**
         * <p>
         * Virtual computer blueprints that are supported by a Lightsail for Research bundle.
         * </p>
         * <important>
         * <p>
         * This parameter only applies to Lightsail for Research resources.
         * </p>
         * </important>
         * 
         * @param supportedAppCategories
         *        Virtual computer blueprints that are supported by a Lightsail for Research bundle.</p> <important>
         *        <p>
         *        This parameter only applies to Lightsail for Research resources.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedAppCategoriesWithStrings(Collection<String> supportedAppCategories);

        /**
         * <p>
         * Virtual computer blueprints that are supported by a Lightsail for Research bundle.
         * </p>
         * <important>
         * <p>
         * This parameter only applies to Lightsail for Research resources.
         * </p>
         * </important>
         * 
         * @param supportedAppCategories
         *        Virtual computer blueprints that are supported by a Lightsail for Research bundle.</p> <important>
         *        <p>
         *        This parameter only applies to Lightsail for Research resources.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedAppCategoriesWithStrings(String... supportedAppCategories);

        /**
         * <p>
         * Virtual computer blueprints that are supported by a Lightsail for Research bundle.
         * </p>
         * <important>
         * <p>
         * This parameter only applies to Lightsail for Research resources.
         * </p>
         * </important>
         * 
         * @param supportedAppCategories
         *        Virtual computer blueprints that are supported by a Lightsail for Research bundle.</p> <important>
         *        <p>
         *        This parameter only applies to Lightsail for Research resources.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedAppCategories(Collection<AppCategory> supportedAppCategories);

        /**
         * <p>
         * Virtual computer blueprints that are supported by a Lightsail for Research bundle.
         * </p>
         * <important>
         * <p>
         * This parameter only applies to Lightsail for Research resources.
         * </p>
         * </important>
         * 
         * @param supportedAppCategories
         *        Virtual computer blueprints that are supported by a Lightsail for Research bundle.</p> <important>
         *        <p>
         *        This parameter only applies to Lightsail for Research resources.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedAppCategories(AppCategory... supportedAppCategories);
    }

    static final class BuilderImpl implements Builder {
        private Float price;

        private Integer cpuCount;

        private Integer diskSizeInGb;

        private String bundleId;

        private String instanceType;

        private Boolean isActive;

        private String name;

        private Integer power;

        private Float ramSizeInGb;

        private Integer transferPerMonthInGb;

        private List<String> supportedPlatforms = DefaultSdkAutoConstructList.getInstance();

        private List<String> supportedAppCategories = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Bundle model) {
            price(model.price);
            cpuCount(model.cpuCount);
            diskSizeInGb(model.diskSizeInGb);
            bundleId(model.bundleId);
            instanceType(model.instanceType);
            isActive(model.isActive);
            name(model.name);
            power(model.power);
            ramSizeInGb(model.ramSizeInGb);
            transferPerMonthInGb(model.transferPerMonthInGb);
            supportedPlatformsWithStrings(model.supportedPlatforms);
            supportedAppCategoriesWithStrings(model.supportedAppCategories);
        }

        public final Float getPrice() {
            return price;
        }

        public final void setPrice(Float price) {
            this.price = price;
        }

        @Override
        public final Builder price(Float price) {
            this.price = price;
            return this;
        }

        public final Integer getCpuCount() {
            return cpuCount;
        }

        public final void setCpuCount(Integer cpuCount) {
            this.cpuCount = cpuCount;
        }

        @Override
        public final Builder cpuCount(Integer cpuCount) {
            this.cpuCount = cpuCount;
            return this;
        }

        public final Integer getDiskSizeInGb() {
            return diskSizeInGb;
        }

        public final void setDiskSizeInGb(Integer diskSizeInGb) {
            this.diskSizeInGb = diskSizeInGb;
        }

        @Override
        public final Builder diskSizeInGb(Integer diskSizeInGb) {
            this.diskSizeInGb = diskSizeInGb;
            return this;
        }

        public final String getBundleId() {
            return bundleId;
        }

        public final void setBundleId(String bundleId) {
            this.bundleId = bundleId;
        }

        @Override
        public final Builder bundleId(String bundleId) {
            this.bundleId = bundleId;
            return this;
        }

        public final String getInstanceType() {
            return instanceType;
        }

        public final void setInstanceType(String instanceType) {
            this.instanceType = instanceType;
        }

        @Override
        public final Builder instanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        public final Boolean getIsActive() {
            return isActive;
        }

        public final void setIsActive(Boolean isActive) {
            this.isActive = isActive;
        }

        @Override
        public final Builder isActive(Boolean isActive) {
            this.isActive = isActive;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Integer getPower() {
            return power;
        }

        public final void setPower(Integer power) {
            this.power = power;
        }

        @Override
        public final Builder power(Integer power) {
            this.power = power;
            return this;
        }

        public final Float getRamSizeInGb() {
            return ramSizeInGb;
        }

        public final void setRamSizeInGb(Float ramSizeInGb) {
            this.ramSizeInGb = ramSizeInGb;
        }

        @Override
        public final Builder ramSizeInGb(Float ramSizeInGb) {
            this.ramSizeInGb = ramSizeInGb;
            return this;
        }

        public final Integer getTransferPerMonthInGb() {
            return transferPerMonthInGb;
        }

        public final void setTransferPerMonthInGb(Integer transferPerMonthInGb) {
            this.transferPerMonthInGb = transferPerMonthInGb;
        }

        @Override
        public final Builder transferPerMonthInGb(Integer transferPerMonthInGb) {
            this.transferPerMonthInGb = transferPerMonthInGb;
            return this;
        }

        public final Collection<String> getSupportedPlatforms() {
            if (supportedPlatforms instanceof SdkAutoConstructList) {
                return null;
            }
            return supportedPlatforms;
        }

        public final void setSupportedPlatforms(Collection<String> supportedPlatforms) {
            this.supportedPlatforms = InstancePlatformListCopier.copy(supportedPlatforms);
        }

        @Override
        public final Builder supportedPlatformsWithStrings(Collection<String> supportedPlatforms) {
            this.supportedPlatforms = InstancePlatformListCopier.copy(supportedPlatforms);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supportedPlatformsWithStrings(String... supportedPlatforms) {
            supportedPlatformsWithStrings(Arrays.asList(supportedPlatforms));
            return this;
        }

        @Override
        public final Builder supportedPlatforms(Collection<InstancePlatform> supportedPlatforms) {
            this.supportedPlatforms = InstancePlatformListCopier.copyEnumToString(supportedPlatforms);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supportedPlatforms(InstancePlatform... supportedPlatforms) {
            supportedPlatforms(Arrays.asList(supportedPlatforms));
            return this;
        }

        public final Collection<String> getSupportedAppCategories() {
            if (supportedAppCategories instanceof SdkAutoConstructList) {
                return null;
            }
            return supportedAppCategories;
        }

        public final void setSupportedAppCategories(Collection<String> supportedAppCategories) {
            this.supportedAppCategories = AppCategoryListCopier.copy(supportedAppCategories);
        }

        @Override
        public final Builder supportedAppCategoriesWithStrings(Collection<String> supportedAppCategories) {
            this.supportedAppCategories = AppCategoryListCopier.copy(supportedAppCategories);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supportedAppCategoriesWithStrings(String... supportedAppCategories) {
            supportedAppCategoriesWithStrings(Arrays.asList(supportedAppCategories));
            return this;
        }

        @Override
        public final Builder supportedAppCategories(Collection<AppCategory> supportedAppCategories) {
            this.supportedAppCategories = AppCategoryListCopier.copyEnumToString(supportedAppCategories);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supportedAppCategories(AppCategory... supportedAppCategories) {
            supportedAppCategories(Arrays.asList(supportedAppCategories));
            return this;
        }

        @Override
        public Bundle build() {
            return new Bundle(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
