/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lightsail.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes information about ports for an Amazon Lightsail instance.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InstancePortInfo implements SdkPojo, Serializable,
        ToCopyableBuilder<InstancePortInfo.Builder, InstancePortInfo> {
    private static final SdkField<Integer> FROM_PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("fromPort").getter(getter(InstancePortInfo::fromPort)).setter(setter(Builder::fromPort))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("fromPort").build()).build();

    private static final SdkField<Integer> TO_PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("toPort").getter(getter(InstancePortInfo::toPort)).setter(setter(Builder::toPort))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("toPort").build()).build();

    private static final SdkField<String> PROTOCOL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("protocol").getter(getter(InstancePortInfo::protocolAsString)).setter(setter(Builder::protocol))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("protocol").build()).build();

    private static final SdkField<String> ACCESS_FROM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("accessFrom").getter(getter(InstancePortInfo::accessFrom)).setter(setter(Builder::accessFrom))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("accessFrom").build()).build();

    private static final SdkField<String> ACCESS_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("accessType").getter(getter(InstancePortInfo::accessTypeAsString)).setter(setter(Builder::accessType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("accessType").build()).build();

    private static final SdkField<String> COMMON_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("commonName").getter(getter(InstancePortInfo::commonName)).setter(setter(Builder::commonName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("commonName").build()).build();

    private static final SdkField<String> ACCESS_DIRECTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("accessDirection").getter(getter(InstancePortInfo::accessDirectionAsString))
            .setter(setter(Builder::accessDirection))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("accessDirection").build()).build();

    private static final SdkField<List<String>> CIDRS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("cidrs")
            .getter(getter(InstancePortInfo::cidrs))
            .setter(setter(Builder::cidrs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("cidrs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> IPV6_CIDRS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ipv6Cidrs")
            .getter(getter(InstancePortInfo::ipv6Cidrs))
            .setter(setter(Builder::ipv6Cidrs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ipv6Cidrs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> CIDR_LIST_ALIASES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("cidrListAliases")
            .getter(getter(InstancePortInfo::cidrListAliases))
            .setter(setter(Builder::cidrListAliases))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("cidrListAliases").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FROM_PORT_FIELD,
            TO_PORT_FIELD, PROTOCOL_FIELD, ACCESS_FROM_FIELD, ACCESS_TYPE_FIELD, COMMON_NAME_FIELD, ACCESS_DIRECTION_FIELD,
            CIDRS_FIELD, IPV6_CIDRS_FIELD, CIDR_LIST_ALIASES_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer fromPort;

    private final Integer toPort;

    private final String protocol;

    private final String accessFrom;

    private final String accessType;

    private final String commonName;

    private final String accessDirection;

    private final List<String> cidrs;

    private final List<String> ipv6Cidrs;

    private final List<String> cidrListAliases;

    private InstancePortInfo(BuilderImpl builder) {
        this.fromPort = builder.fromPort;
        this.toPort = builder.toPort;
        this.protocol = builder.protocol;
        this.accessFrom = builder.accessFrom;
        this.accessType = builder.accessType;
        this.commonName = builder.commonName;
        this.accessDirection = builder.accessDirection;
        this.cidrs = builder.cidrs;
        this.ipv6Cidrs = builder.ipv6Cidrs;
        this.cidrListAliases = builder.cidrListAliases;
    }

    /**
     * <p>
     * The first port in a range of open ports on an instance.
     * </p>
     * <p>
     * Allowed ports:
     * </p>
     * <ul>
     * <li>
     * <p>
     * TCP and UDP - <code>0</code> to <code>65535</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * ICMP - The ICMP type for IPv4 addresses. For example, specify <code>8</code> as the <code>fromPort</code> (ICMP
     * type), and <code>-1</code> as the <code>toPort</code> (ICMP code), to enable ICMP Ping. For more information, see
     * <a href="https://en.wikipedia.org/wiki/Internet_Control_Message_Protocol#Control_messages">Control Messages</a>
     * on <i>Wikipedia</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * ICMPv6 - The ICMP type for IPv6 addresses. For example, specify <code>128</code> as the <code>fromPort</code>
     * (ICMPv6 type), and <code>0</code> as <code>toPort</code> (ICMPv6 code). For more information, see <a
     * href="https://en.wikipedia.org/wiki/Internet_Control_Message_Protocol_for_IPv6">Internet Control Message Protocol
     * for IPv6</a>.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The first port in a range of open ports on an instance.</p>
     *         <p>
     *         Allowed ports:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         TCP and UDP - <code>0</code> to <code>65535</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ICMP - The ICMP type for IPv4 addresses. For example, specify <code>8</code> as the <code>fromPort</code>
     *         (ICMP type), and <code>-1</code> as the <code>toPort</code> (ICMP code), to enable ICMP Ping. For more
     *         information, see <a
     *         href="https://en.wikipedia.org/wiki/Internet_Control_Message_Protocol#Control_messages">Control
     *         Messages</a> on <i>Wikipedia</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ICMPv6 - The ICMP type for IPv6 addresses. For example, specify <code>128</code> as the
     *         <code>fromPort</code> (ICMPv6 type), and <code>0</code> as <code>toPort</code> (ICMPv6 code). For more
     *         information, see <a
     *         href="https://en.wikipedia.org/wiki/Internet_Control_Message_Protocol_for_IPv6">Internet Control Message
     *         Protocol for IPv6</a>.
     *         </p>
     *         </li>
     */
    public final Integer fromPort() {
        return fromPort;
    }

    /**
     * <p>
     * The last port in a range of open ports on an instance.
     * </p>
     * <p>
     * Allowed ports:
     * </p>
     * <ul>
     * <li>
     * <p>
     * TCP and UDP - <code>0</code> to <code>65535</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * ICMP - The ICMP code for IPv4 addresses. For example, specify <code>8</code> as the <code>fromPort</code> (ICMP
     * type), and <code>-1</code> as the <code>toPort</code> (ICMP code), to enable ICMP Ping. For more information, see
     * <a href="https://en.wikipedia.org/wiki/Internet_Control_Message_Protocol#Control_messages">Control Messages</a>
     * on <i>Wikipedia</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * ICMPv6 - The ICMP code for IPv6 addresses. For example, specify <code>128</code> as the <code>fromPort</code>
     * (ICMPv6 type), and <code>0</code> as <code>toPort</code> (ICMPv6 code). For more information, see <a
     * href="https://en.wikipedia.org/wiki/Internet_Control_Message_Protocol_for_IPv6">Internet Control Message Protocol
     * for IPv6</a>.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The last port in a range of open ports on an instance.</p>
     *         <p>
     *         Allowed ports:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         TCP and UDP - <code>0</code> to <code>65535</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ICMP - The ICMP code for IPv4 addresses. For example, specify <code>8</code> as the <code>fromPort</code>
     *         (ICMP type), and <code>-1</code> as the <code>toPort</code> (ICMP code), to enable ICMP Ping. For more
     *         information, see <a
     *         href="https://en.wikipedia.org/wiki/Internet_Control_Message_Protocol#Control_messages">Control
     *         Messages</a> on <i>Wikipedia</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ICMPv6 - The ICMP code for IPv6 addresses. For example, specify <code>128</code> as the
     *         <code>fromPort</code> (ICMPv6 type), and <code>0</code> as <code>toPort</code> (ICMPv6 code). For more
     *         information, see <a
     *         href="https://en.wikipedia.org/wiki/Internet_Control_Message_Protocol_for_IPv6">Internet Control Message
     *         Protocol for IPv6</a>.
     *         </p>
     *         </li>
     */
    public final Integer toPort() {
        return toPort;
    }

    /**
     * <p>
     * The IP protocol name.
     * </p>
     * <p>
     * The name can be one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>tcp</code> - Transmission Control Protocol (TCP) provides reliable, ordered, and error-checked delivery of
     * streamed data between applications running on hosts communicating by an IP network. If you have an application
     * that doesn't require reliable data stream service, use UDP instead.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>all</code> - All transport layer protocol types. For more general information, see <a
     * href="https://en.wikipedia.org/wiki/Transport_layer">Transport layer</a> on <i>Wikipedia</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>udp</code> - With User Datagram Protocol (UDP), computer applications can send messages (or datagrams) to
     * other hosts on an Internet Protocol (IP) network. Prior communications are not required to set up transmission
     * channels or data paths. Applications that don't require reliable data stream service can use UDP, which provides
     * a connectionless datagram service that emphasizes reduced latency over reliability. If you do require reliable
     * data stream service, use TCP instead.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>icmp</code> - Internet Control Message Protocol (ICMP) is used to send error messages and operational
     * information indicating success or failure when communicating with an instance. For example, an error is indicated
     * when an instance could not be reached. When you specify <code>icmp</code> as the <code>protocol</code>, you must
     * specify the ICMP type using the <code>fromPort</code> parameter, and ICMP code using the <code>toPort</code>
     * parameter.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>icmp6</code> - Internet Control Message Protocol (ICMP) for IPv6. When you specify <code>icmp6</code> as
     * the <code>protocol</code>, you must specify the ICMP type using the <code>fromPort</code> parameter, and ICMP
     * code using the <code>toPort</code> parameter.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #protocol} will
     * return {@link NetworkProtocol#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #protocolAsString}.
     * </p>
     * 
     * @return The IP protocol name.</p>
     *         <p>
     *         The name can be one of the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>tcp</code> - Transmission Control Protocol (TCP) provides reliable, ordered, and error-checked
     *         delivery of streamed data between applications running on hosts communicating by an IP network. If you
     *         have an application that doesn't require reliable data stream service, use UDP instead.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>all</code> - All transport layer protocol types. For more general information, see <a
     *         href="https://en.wikipedia.org/wiki/Transport_layer">Transport layer</a> on <i>Wikipedia</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>udp</code> - With User Datagram Protocol (UDP), computer applications can send messages (or
     *         datagrams) to other hosts on an Internet Protocol (IP) network. Prior communications are not required to
     *         set up transmission channels or data paths. Applications that don't require reliable data stream service
     *         can use UDP, which provides a connectionless datagram service that emphasizes reduced latency over
     *         reliability. If you do require reliable data stream service, use TCP instead.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>icmp</code> - Internet Control Message Protocol (ICMP) is used to send error messages and
     *         operational information indicating success or failure when communicating with an instance. For example,
     *         an error is indicated when an instance could not be reached. When you specify <code>icmp</code> as the
     *         <code>protocol</code>, you must specify the ICMP type using the <code>fromPort</code> parameter, and ICMP
     *         code using the <code>toPort</code> parameter.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>icmp6</code> - Internet Control Message Protocol (ICMP) for IPv6. When you specify
     *         <code>icmp6</code> as the <code>protocol</code>, you must specify the ICMP type using the
     *         <code>fromPort</code> parameter, and ICMP code using the <code>toPort</code> parameter.
     *         </p>
     *         </li>
     * @see NetworkProtocol
     */
    public final NetworkProtocol protocol() {
        return NetworkProtocol.fromValue(protocol);
    }

    /**
     * <p>
     * The IP protocol name.
     * </p>
     * <p>
     * The name can be one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>tcp</code> - Transmission Control Protocol (TCP) provides reliable, ordered, and error-checked delivery of
     * streamed data between applications running on hosts communicating by an IP network. If you have an application
     * that doesn't require reliable data stream service, use UDP instead.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>all</code> - All transport layer protocol types. For more general information, see <a
     * href="https://en.wikipedia.org/wiki/Transport_layer">Transport layer</a> on <i>Wikipedia</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>udp</code> - With User Datagram Protocol (UDP), computer applications can send messages (or datagrams) to
     * other hosts on an Internet Protocol (IP) network. Prior communications are not required to set up transmission
     * channels or data paths. Applications that don't require reliable data stream service can use UDP, which provides
     * a connectionless datagram service that emphasizes reduced latency over reliability. If you do require reliable
     * data stream service, use TCP instead.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>icmp</code> - Internet Control Message Protocol (ICMP) is used to send error messages and operational
     * information indicating success or failure when communicating with an instance. For example, an error is indicated
     * when an instance could not be reached. When you specify <code>icmp</code> as the <code>protocol</code>, you must
     * specify the ICMP type using the <code>fromPort</code> parameter, and ICMP code using the <code>toPort</code>
     * parameter.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>icmp6</code> - Internet Control Message Protocol (ICMP) for IPv6. When you specify <code>icmp6</code> as
     * the <code>protocol</code>, you must specify the ICMP type using the <code>fromPort</code> parameter, and ICMP
     * code using the <code>toPort</code> parameter.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #protocol} will
     * return {@link NetworkProtocol#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #protocolAsString}.
     * </p>
     * 
     * @return The IP protocol name.</p>
     *         <p>
     *         The name can be one of the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>tcp</code> - Transmission Control Protocol (TCP) provides reliable, ordered, and error-checked
     *         delivery of streamed data between applications running on hosts communicating by an IP network. If you
     *         have an application that doesn't require reliable data stream service, use UDP instead.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>all</code> - All transport layer protocol types. For more general information, see <a
     *         href="https://en.wikipedia.org/wiki/Transport_layer">Transport layer</a> on <i>Wikipedia</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>udp</code> - With User Datagram Protocol (UDP), computer applications can send messages (or
     *         datagrams) to other hosts on an Internet Protocol (IP) network. Prior communications are not required to
     *         set up transmission channels or data paths. Applications that don't require reliable data stream service
     *         can use UDP, which provides a connectionless datagram service that emphasizes reduced latency over
     *         reliability. If you do require reliable data stream service, use TCP instead.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>icmp</code> - Internet Control Message Protocol (ICMP) is used to send error messages and
     *         operational information indicating success or failure when communicating with an instance. For example,
     *         an error is indicated when an instance could not be reached. When you specify <code>icmp</code> as the
     *         <code>protocol</code>, you must specify the ICMP type using the <code>fromPort</code> parameter, and ICMP
     *         code using the <code>toPort</code> parameter.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>icmp6</code> - Internet Control Message Protocol (ICMP) for IPv6. When you specify
     *         <code>icmp6</code> as the <code>protocol</code>, you must specify the ICMP type using the
     *         <code>fromPort</code> parameter, and ICMP code using the <code>toPort</code> parameter.
     *         </p>
     *         </li>
     * @see NetworkProtocol
     */
    public final String protocolAsString() {
        return protocol;
    }

    /**
     * <p>
     * The location from which access is allowed. For example, <code>Anywhere (0.0.0.0/0)</code>, or <code>Custom</code>
     * if a specific IP address or range of IP addresses is allowed.
     * </p>
     * 
     * @return The location from which access is allowed. For example, <code>Anywhere (0.0.0.0/0)</code>, or
     *         <code>Custom</code> if a specific IP address or range of IP addresses is allowed.
     */
    public final String accessFrom() {
        return accessFrom;
    }

    /**
     * <p>
     * The type of access (<code>Public</code> or <code>Private</code>).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #accessType} will
     * return {@link PortAccessType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #accessTypeAsString}.
     * </p>
     * 
     * @return The type of access (<code>Public</code> or <code>Private</code>).
     * @see PortAccessType
     */
    public final PortAccessType accessType() {
        return PortAccessType.fromValue(accessType);
    }

    /**
     * <p>
     * The type of access (<code>Public</code> or <code>Private</code>).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #accessType} will
     * return {@link PortAccessType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #accessTypeAsString}.
     * </p>
     * 
     * @return The type of access (<code>Public</code> or <code>Private</code>).
     * @see PortAccessType
     */
    public final String accessTypeAsString() {
        return accessType;
    }

    /**
     * <p>
     * The common name of the port information.
     * </p>
     * 
     * @return The common name of the port information.
     */
    public final String commonName() {
        return commonName;
    }

    /**
     * <p>
     * The access direction (<code>inbound</code> or <code>outbound</code>).
     * </p>
     * <note>
     * <p>
     * Lightsail currently supports only <code>inbound</code> access direction.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #accessDirection}
     * will return {@link AccessDirection#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #accessDirectionAsString}.
     * </p>
     * 
     * @return The access direction (<code>inbound</code> or <code>outbound</code>).</p> <note>
     *         <p>
     *         Lightsail currently supports only <code>inbound</code> access direction.
     *         </p>
     * @see AccessDirection
     */
    public final AccessDirection accessDirection() {
        return AccessDirection.fromValue(accessDirection);
    }

    /**
     * <p>
     * The access direction (<code>inbound</code> or <code>outbound</code>).
     * </p>
     * <note>
     * <p>
     * Lightsail currently supports only <code>inbound</code> access direction.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #accessDirection}
     * will return {@link AccessDirection#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #accessDirectionAsString}.
     * </p>
     * 
     * @return The access direction (<code>inbound</code> or <code>outbound</code>).</p> <note>
     *         <p>
     *         Lightsail currently supports only <code>inbound</code> access direction.
     *         </p>
     * @see AccessDirection
     */
    public final String accessDirectionAsString() {
        return accessDirection;
    }

    /**
     * For responses, this returns true if the service returned a value for the Cidrs property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasCidrs() {
        return cidrs != null && !(cidrs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The IPv4 address, or range of IPv4 addresses (in CIDR notation) that are allowed to connect to an instance
     * through the ports, and the protocol.
     * </p>
     * <note>
     * <p>
     * The <code>ipv6Cidrs</code> parameter lists the IPv6 addresses that are allowed to connect to an instance.
     * </p>
     * </note>
     * <p>
     * For more information about CIDR block notation, see <a
     * href="https://en.wikipedia.org/wiki/Classless_Inter-Domain_Routing#CIDR_notation">Classless Inter-Domain
     * Routing</a> on <i>Wikipedia</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCidrs} method.
     * </p>
     * 
     * @return The IPv4 address, or range of IPv4 addresses (in CIDR notation) that are allowed to connect to an
     *         instance through the ports, and the protocol.</p> <note>
     *         <p>
     *         The <code>ipv6Cidrs</code> parameter lists the IPv6 addresses that are allowed to connect to an instance.
     *         </p>
     *         </note>
     *         <p>
     *         For more information about CIDR block notation, see <a
     *         href="https://en.wikipedia.org/wiki/Classless_Inter-Domain_Routing#CIDR_notation">Classless Inter-Domain
     *         Routing</a> on <i>Wikipedia</i>.
     */
    public final List<String> cidrs() {
        return cidrs;
    }

    /**
     * For responses, this returns true if the service returned a value for the Ipv6Cidrs property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasIpv6Cidrs() {
        return ipv6Cidrs != null && !(ipv6Cidrs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The IPv6 address, or range of IPv6 addresses (in CIDR notation) that are allowed to connect to an instance
     * through the ports, and the protocol. Only devices with an IPv6 address can connect to an instance through IPv6;
     * otherwise, IPv4 should be used.
     * </p>
     * <note>
     * <p>
     * The <code>cidrs</code> parameter lists the IPv4 addresses that are allowed to connect to an instance.
     * </p>
     * </note>
     * <p>
     * For more information about CIDR block notation, see <a
     * href="https://en.wikipedia.org/wiki/Classless_Inter-Domain_Routing#CIDR_notation">Classless Inter-Domain
     * Routing</a> on <i>Wikipedia</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasIpv6Cidrs} method.
     * </p>
     * 
     * @return The IPv6 address, or range of IPv6 addresses (in CIDR notation) that are allowed to connect to an
     *         instance through the ports, and the protocol. Only devices with an IPv6 address can connect to an
     *         instance through IPv6; otherwise, IPv4 should be used.</p> <note>
     *         <p>
     *         The <code>cidrs</code> parameter lists the IPv4 addresses that are allowed to connect to an instance.
     *         </p>
     *         </note>
     *         <p>
     *         For more information about CIDR block notation, see <a
     *         href="https://en.wikipedia.org/wiki/Classless_Inter-Domain_Routing#CIDR_notation">Classless Inter-Domain
     *         Routing</a> on <i>Wikipedia</i>.
     */
    public final List<String> ipv6Cidrs() {
        return ipv6Cidrs;
    }

    /**
     * For responses, this returns true if the service returned a value for the CidrListAliases property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasCidrListAliases() {
        return cidrListAliases != null && !(cidrListAliases instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An alias that defines access for a preconfigured range of IP addresses.
     * </p>
     * <p>
     * The only alias currently supported is <code>lightsail-connect</code>, which allows IP addresses of the
     * browser-based RDP/SSH client in the Lightsail console to connect to your instance.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCidrListAliases} method.
     * </p>
     * 
     * @return An alias that defines access for a preconfigured range of IP addresses.</p>
     *         <p>
     *         The only alias currently supported is <code>lightsail-connect</code>, which allows IP addresses of the
     *         browser-based RDP/SSH client in the Lightsail console to connect to your instance.
     */
    public final List<String> cidrListAliases() {
        return cidrListAliases;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(fromPort());
        hashCode = 31 * hashCode + Objects.hashCode(toPort());
        hashCode = 31 * hashCode + Objects.hashCode(protocolAsString());
        hashCode = 31 * hashCode + Objects.hashCode(accessFrom());
        hashCode = 31 * hashCode + Objects.hashCode(accessTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(commonName());
        hashCode = 31 * hashCode + Objects.hashCode(accessDirectionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasCidrs() ? cidrs() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasIpv6Cidrs() ? ipv6Cidrs() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasCidrListAliases() ? cidrListAliases() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InstancePortInfo)) {
            return false;
        }
        InstancePortInfo other = (InstancePortInfo) obj;
        return Objects.equals(fromPort(), other.fromPort()) && Objects.equals(toPort(), other.toPort())
                && Objects.equals(protocolAsString(), other.protocolAsString())
                && Objects.equals(accessFrom(), other.accessFrom())
                && Objects.equals(accessTypeAsString(), other.accessTypeAsString())
                && Objects.equals(commonName(), other.commonName())
                && Objects.equals(accessDirectionAsString(), other.accessDirectionAsString()) && hasCidrs() == other.hasCidrs()
                && Objects.equals(cidrs(), other.cidrs()) && hasIpv6Cidrs() == other.hasIpv6Cidrs()
                && Objects.equals(ipv6Cidrs(), other.ipv6Cidrs()) && hasCidrListAliases() == other.hasCidrListAliases()
                && Objects.equals(cidrListAliases(), other.cidrListAliases());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("InstancePortInfo").add("FromPort", fromPort()).add("ToPort", toPort())
                .add("Protocol", protocolAsString()).add("AccessFrom", accessFrom()).add("AccessType", accessTypeAsString())
                .add("CommonName", commonName()).add("AccessDirection", accessDirectionAsString())
                .add("Cidrs", hasCidrs() ? cidrs() : null).add("Ipv6Cidrs", hasIpv6Cidrs() ? ipv6Cidrs() : null)
                .add("CidrListAliases", hasCidrListAliases() ? cidrListAliases() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "fromPort":
            return Optional.ofNullable(clazz.cast(fromPort()));
        case "toPort":
            return Optional.ofNullable(clazz.cast(toPort()));
        case "protocol":
            return Optional.ofNullable(clazz.cast(protocolAsString()));
        case "accessFrom":
            return Optional.ofNullable(clazz.cast(accessFrom()));
        case "accessType":
            return Optional.ofNullable(clazz.cast(accessTypeAsString()));
        case "commonName":
            return Optional.ofNullable(clazz.cast(commonName()));
        case "accessDirection":
            return Optional.ofNullable(clazz.cast(accessDirectionAsString()));
        case "cidrs":
            return Optional.ofNullable(clazz.cast(cidrs()));
        case "ipv6Cidrs":
            return Optional.ofNullable(clazz.cast(ipv6Cidrs()));
        case "cidrListAliases":
            return Optional.ofNullable(clazz.cast(cidrListAliases()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<InstancePortInfo, T> g) {
        return obj -> g.apply((InstancePortInfo) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InstancePortInfo> {
        /**
         * <p>
         * The first port in a range of open ports on an instance.
         * </p>
         * <p>
         * Allowed ports:
         * </p>
         * <ul>
         * <li>
         * <p>
         * TCP and UDP - <code>0</code> to <code>65535</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * ICMP - The ICMP type for IPv4 addresses. For example, specify <code>8</code> as the <code>fromPort</code>
         * (ICMP type), and <code>-1</code> as the <code>toPort</code> (ICMP code), to enable ICMP Ping. For more
         * information, see <a
         * href="https://en.wikipedia.org/wiki/Internet_Control_Message_Protocol#Control_messages">Control Messages</a>
         * on <i>Wikipedia</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * ICMPv6 - The ICMP type for IPv6 addresses. For example, specify <code>128</code> as the <code>fromPort</code>
         * (ICMPv6 type), and <code>0</code> as <code>toPort</code> (ICMPv6 code). For more information, see <a
         * href="https://en.wikipedia.org/wiki/Internet_Control_Message_Protocol_for_IPv6">Internet Control Message
         * Protocol for IPv6</a>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param fromPort
         *        The first port in a range of open ports on an instance.</p>
         *        <p>
         *        Allowed ports:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        TCP and UDP - <code>0</code> to <code>65535</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ICMP - The ICMP type for IPv4 addresses. For example, specify <code>8</code> as the
         *        <code>fromPort</code> (ICMP type), and <code>-1</code> as the <code>toPort</code> (ICMP code), to
         *        enable ICMP Ping. For more information, see <a
         *        href="https://en.wikipedia.org/wiki/Internet_Control_Message_Protocol#Control_messages">Control
         *        Messages</a> on <i>Wikipedia</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ICMPv6 - The ICMP type for IPv6 addresses. For example, specify <code>128</code> as the
         *        <code>fromPort</code> (ICMPv6 type), and <code>0</code> as <code>toPort</code> (ICMPv6 code). For more
         *        information, see <a
         *        href="https://en.wikipedia.org/wiki/Internet_Control_Message_Protocol_for_IPv6">Internet Control
         *        Message Protocol for IPv6</a>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fromPort(Integer fromPort);

        /**
         * <p>
         * The last port in a range of open ports on an instance.
         * </p>
         * <p>
         * Allowed ports:
         * </p>
         * <ul>
         * <li>
         * <p>
         * TCP and UDP - <code>0</code> to <code>65535</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * ICMP - The ICMP code for IPv4 addresses. For example, specify <code>8</code> as the <code>fromPort</code>
         * (ICMP type), and <code>-1</code> as the <code>toPort</code> (ICMP code), to enable ICMP Ping. For more
         * information, see <a
         * href="https://en.wikipedia.org/wiki/Internet_Control_Message_Protocol#Control_messages">Control Messages</a>
         * on <i>Wikipedia</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * ICMPv6 - The ICMP code for IPv6 addresses. For example, specify <code>128</code> as the <code>fromPort</code>
         * (ICMPv6 type), and <code>0</code> as <code>toPort</code> (ICMPv6 code). For more information, see <a
         * href="https://en.wikipedia.org/wiki/Internet_Control_Message_Protocol_for_IPv6">Internet Control Message
         * Protocol for IPv6</a>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param toPort
         *        The last port in a range of open ports on an instance.</p>
         *        <p>
         *        Allowed ports:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        TCP and UDP - <code>0</code> to <code>65535</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ICMP - The ICMP code for IPv4 addresses. For example, specify <code>8</code> as the
         *        <code>fromPort</code> (ICMP type), and <code>-1</code> as the <code>toPort</code> (ICMP code), to
         *        enable ICMP Ping. For more information, see <a
         *        href="https://en.wikipedia.org/wiki/Internet_Control_Message_Protocol#Control_messages">Control
         *        Messages</a> on <i>Wikipedia</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ICMPv6 - The ICMP code for IPv6 addresses. For example, specify <code>128</code> as the
         *        <code>fromPort</code> (ICMPv6 type), and <code>0</code> as <code>toPort</code> (ICMPv6 code). For more
         *        information, see <a
         *        href="https://en.wikipedia.org/wiki/Internet_Control_Message_Protocol_for_IPv6">Internet Control
         *        Message Protocol for IPv6</a>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder toPort(Integer toPort);

        /**
         * <p>
         * The IP protocol name.
         * </p>
         * <p>
         * The name can be one of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>tcp</code> - Transmission Control Protocol (TCP) provides reliable, ordered, and error-checked delivery
         * of streamed data between applications running on hosts communicating by an IP network. If you have an
         * application that doesn't require reliable data stream service, use UDP instead.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>all</code> - All transport layer protocol types. For more general information, see <a
         * href="https://en.wikipedia.org/wiki/Transport_layer">Transport layer</a> on <i>Wikipedia</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>udp</code> - With User Datagram Protocol (UDP), computer applications can send messages (or datagrams)
         * to other hosts on an Internet Protocol (IP) network. Prior communications are not required to set up
         * transmission channels or data paths. Applications that don't require reliable data stream service can use
         * UDP, which provides a connectionless datagram service that emphasizes reduced latency over reliability. If
         * you do require reliable data stream service, use TCP instead.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>icmp</code> - Internet Control Message Protocol (ICMP) is used to send error messages and operational
         * information indicating success or failure when communicating with an instance. For example, an error is
         * indicated when an instance could not be reached. When you specify <code>icmp</code> as the
         * <code>protocol</code>, you must specify the ICMP type using the <code>fromPort</code> parameter, and ICMP
         * code using the <code>toPort</code> parameter.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>icmp6</code> - Internet Control Message Protocol (ICMP) for IPv6. When you specify <code>icmp6</code>
         * as the <code>protocol</code>, you must specify the ICMP type using the <code>fromPort</code> parameter, and
         * ICMP code using the <code>toPort</code> parameter.
         * </p>
         * </li>
         * </ul>
         * 
         * @param protocol
         *        The IP protocol name.</p>
         *        <p>
         *        The name can be one of the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>tcp</code> - Transmission Control Protocol (TCP) provides reliable, ordered, and error-checked
         *        delivery of streamed data between applications running on hosts communicating by an IP network. If you
         *        have an application that doesn't require reliable data stream service, use UDP instead.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>all</code> - All transport layer protocol types. For more general information, see <a
         *        href="https://en.wikipedia.org/wiki/Transport_layer">Transport layer</a> on <i>Wikipedia</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>udp</code> - With User Datagram Protocol (UDP), computer applications can send messages (or
         *        datagrams) to other hosts on an Internet Protocol (IP) network. Prior communications are not required
         *        to set up transmission channels or data paths. Applications that don't require reliable data stream
         *        service can use UDP, which provides a connectionless datagram service that emphasizes reduced latency
         *        over reliability. If you do require reliable data stream service, use TCP instead.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>icmp</code> - Internet Control Message Protocol (ICMP) is used to send error messages and
         *        operational information indicating success or failure when communicating with an instance. For
         *        example, an error is indicated when an instance could not be reached. When you specify
         *        <code>icmp</code> as the <code>protocol</code>, you must specify the ICMP type using the
         *        <code>fromPort</code> parameter, and ICMP code using the <code>toPort</code> parameter.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>icmp6</code> - Internet Control Message Protocol (ICMP) for IPv6. When you specify
         *        <code>icmp6</code> as the <code>protocol</code>, you must specify the ICMP type using the
         *        <code>fromPort</code> parameter, and ICMP code using the <code>toPort</code> parameter.
         *        </p>
         *        </li>
         * @see NetworkProtocol
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NetworkProtocol
         */
        Builder protocol(String protocol);

        /**
         * <p>
         * The IP protocol name.
         * </p>
         * <p>
         * The name can be one of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>tcp</code> - Transmission Control Protocol (TCP) provides reliable, ordered, and error-checked delivery
         * of streamed data between applications running on hosts communicating by an IP network. If you have an
         * application that doesn't require reliable data stream service, use UDP instead.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>all</code> - All transport layer protocol types. For more general information, see <a
         * href="https://en.wikipedia.org/wiki/Transport_layer">Transport layer</a> on <i>Wikipedia</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>udp</code> - With User Datagram Protocol (UDP), computer applications can send messages (or datagrams)
         * to other hosts on an Internet Protocol (IP) network. Prior communications are not required to set up
         * transmission channels or data paths. Applications that don't require reliable data stream service can use
         * UDP, which provides a connectionless datagram service that emphasizes reduced latency over reliability. If
         * you do require reliable data stream service, use TCP instead.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>icmp</code> - Internet Control Message Protocol (ICMP) is used to send error messages and operational
         * information indicating success or failure when communicating with an instance. For example, an error is
         * indicated when an instance could not be reached. When you specify <code>icmp</code> as the
         * <code>protocol</code>, you must specify the ICMP type using the <code>fromPort</code> parameter, and ICMP
         * code using the <code>toPort</code> parameter.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>icmp6</code> - Internet Control Message Protocol (ICMP) for IPv6. When you specify <code>icmp6</code>
         * as the <code>protocol</code>, you must specify the ICMP type using the <code>fromPort</code> parameter, and
         * ICMP code using the <code>toPort</code> parameter.
         * </p>
         * </li>
         * </ul>
         * 
         * @param protocol
         *        The IP protocol name.</p>
         *        <p>
         *        The name can be one of the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>tcp</code> - Transmission Control Protocol (TCP) provides reliable, ordered, and error-checked
         *        delivery of streamed data between applications running on hosts communicating by an IP network. If you
         *        have an application that doesn't require reliable data stream service, use UDP instead.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>all</code> - All transport layer protocol types. For more general information, see <a
         *        href="https://en.wikipedia.org/wiki/Transport_layer">Transport layer</a> on <i>Wikipedia</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>udp</code> - With User Datagram Protocol (UDP), computer applications can send messages (or
         *        datagrams) to other hosts on an Internet Protocol (IP) network. Prior communications are not required
         *        to set up transmission channels or data paths. Applications that don't require reliable data stream
         *        service can use UDP, which provides a connectionless datagram service that emphasizes reduced latency
         *        over reliability. If you do require reliable data stream service, use TCP instead.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>icmp</code> - Internet Control Message Protocol (ICMP) is used to send error messages and
         *        operational information indicating success or failure when communicating with an instance. For
         *        example, an error is indicated when an instance could not be reached. When you specify
         *        <code>icmp</code> as the <code>protocol</code>, you must specify the ICMP type using the
         *        <code>fromPort</code> parameter, and ICMP code using the <code>toPort</code> parameter.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>icmp6</code> - Internet Control Message Protocol (ICMP) for IPv6. When you specify
         *        <code>icmp6</code> as the <code>protocol</code>, you must specify the ICMP type using the
         *        <code>fromPort</code> parameter, and ICMP code using the <code>toPort</code> parameter.
         *        </p>
         *        </li>
         * @see NetworkProtocol
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NetworkProtocol
         */
        Builder protocol(NetworkProtocol protocol);

        /**
         * <p>
         * The location from which access is allowed. For example, <code>Anywhere (0.0.0.0/0)</code>, or
         * <code>Custom</code> if a specific IP address or range of IP addresses is allowed.
         * </p>
         * 
         * @param accessFrom
         *        The location from which access is allowed. For example, <code>Anywhere (0.0.0.0/0)</code>, or
         *        <code>Custom</code> if a specific IP address or range of IP addresses is allowed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessFrom(String accessFrom);

        /**
         * <p>
         * The type of access (<code>Public</code> or <code>Private</code>).
         * </p>
         * 
         * @param accessType
         *        The type of access (<code>Public</code> or <code>Private</code>).
         * @see PortAccessType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PortAccessType
         */
        Builder accessType(String accessType);

        /**
         * <p>
         * The type of access (<code>Public</code> or <code>Private</code>).
         * </p>
         * 
         * @param accessType
         *        The type of access (<code>Public</code> or <code>Private</code>).
         * @see PortAccessType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PortAccessType
         */
        Builder accessType(PortAccessType accessType);

        /**
         * <p>
         * The common name of the port information.
         * </p>
         * 
         * @param commonName
         *        The common name of the port information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder commonName(String commonName);

        /**
         * <p>
         * The access direction (<code>inbound</code> or <code>outbound</code>).
         * </p>
         * <note>
         * <p>
         * Lightsail currently supports only <code>inbound</code> access direction.
         * </p>
         * </note>
         * 
         * @param accessDirection
         *        The access direction (<code>inbound</code> or <code>outbound</code>).</p> <note>
         *        <p>
         *        Lightsail currently supports only <code>inbound</code> access direction.
         *        </p>
         * @see AccessDirection
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AccessDirection
         */
        Builder accessDirection(String accessDirection);

        /**
         * <p>
         * The access direction (<code>inbound</code> or <code>outbound</code>).
         * </p>
         * <note>
         * <p>
         * Lightsail currently supports only <code>inbound</code> access direction.
         * </p>
         * </note>
         * 
         * @param accessDirection
         *        The access direction (<code>inbound</code> or <code>outbound</code>).</p> <note>
         *        <p>
         *        Lightsail currently supports only <code>inbound</code> access direction.
         *        </p>
         * @see AccessDirection
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AccessDirection
         */
        Builder accessDirection(AccessDirection accessDirection);

        /**
         * <p>
         * The IPv4 address, or range of IPv4 addresses (in CIDR notation) that are allowed to connect to an instance
         * through the ports, and the protocol.
         * </p>
         * <note>
         * <p>
         * The <code>ipv6Cidrs</code> parameter lists the IPv6 addresses that are allowed to connect to an instance.
         * </p>
         * </note>
         * <p>
         * For more information about CIDR block notation, see <a
         * href="https://en.wikipedia.org/wiki/Classless_Inter-Domain_Routing#CIDR_notation">Classless Inter-Domain
         * Routing</a> on <i>Wikipedia</i>.
         * </p>
         * 
         * @param cidrs
         *        The IPv4 address, or range of IPv4 addresses (in CIDR notation) that are allowed to connect to an
         *        instance through the ports, and the protocol.</p> <note>
         *        <p>
         *        The <code>ipv6Cidrs</code> parameter lists the IPv6 addresses that are allowed to connect to an
         *        instance.
         *        </p>
         *        </note>
         *        <p>
         *        For more information about CIDR block notation, see <a
         *        href="https://en.wikipedia.org/wiki/Classless_Inter-Domain_Routing#CIDR_notation">Classless
         *        Inter-Domain Routing</a> on <i>Wikipedia</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cidrs(Collection<String> cidrs);

        /**
         * <p>
         * The IPv4 address, or range of IPv4 addresses (in CIDR notation) that are allowed to connect to an instance
         * through the ports, and the protocol.
         * </p>
         * <note>
         * <p>
         * The <code>ipv6Cidrs</code> parameter lists the IPv6 addresses that are allowed to connect to an instance.
         * </p>
         * </note>
         * <p>
         * For more information about CIDR block notation, see <a
         * href="https://en.wikipedia.org/wiki/Classless_Inter-Domain_Routing#CIDR_notation">Classless Inter-Domain
         * Routing</a> on <i>Wikipedia</i>.
         * </p>
         * 
         * @param cidrs
         *        The IPv4 address, or range of IPv4 addresses (in CIDR notation) that are allowed to connect to an
         *        instance through the ports, and the protocol.</p> <note>
         *        <p>
         *        The <code>ipv6Cidrs</code> parameter lists the IPv6 addresses that are allowed to connect to an
         *        instance.
         *        </p>
         *        </note>
         *        <p>
         *        For more information about CIDR block notation, see <a
         *        href="https://en.wikipedia.org/wiki/Classless_Inter-Domain_Routing#CIDR_notation">Classless
         *        Inter-Domain Routing</a> on <i>Wikipedia</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cidrs(String... cidrs);

        /**
         * <p>
         * The IPv6 address, or range of IPv6 addresses (in CIDR notation) that are allowed to connect to an instance
         * through the ports, and the protocol. Only devices with an IPv6 address can connect to an instance through
         * IPv6; otherwise, IPv4 should be used.
         * </p>
         * <note>
         * <p>
         * The <code>cidrs</code> parameter lists the IPv4 addresses that are allowed to connect to an instance.
         * </p>
         * </note>
         * <p>
         * For more information about CIDR block notation, see <a
         * href="https://en.wikipedia.org/wiki/Classless_Inter-Domain_Routing#CIDR_notation">Classless Inter-Domain
         * Routing</a> on <i>Wikipedia</i>.
         * </p>
         * 
         * @param ipv6Cidrs
         *        The IPv6 address, or range of IPv6 addresses (in CIDR notation) that are allowed to connect to an
         *        instance through the ports, and the protocol. Only devices with an IPv6 address can connect to an
         *        instance through IPv6; otherwise, IPv4 should be used.</p> <note>
         *        <p>
         *        The <code>cidrs</code> parameter lists the IPv4 addresses that are allowed to connect to an instance.
         *        </p>
         *        </note>
         *        <p>
         *        For more information about CIDR block notation, see <a
         *        href="https://en.wikipedia.org/wiki/Classless_Inter-Domain_Routing#CIDR_notation">Classless
         *        Inter-Domain Routing</a> on <i>Wikipedia</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipv6Cidrs(Collection<String> ipv6Cidrs);

        /**
         * <p>
         * The IPv6 address, or range of IPv6 addresses (in CIDR notation) that are allowed to connect to an instance
         * through the ports, and the protocol. Only devices with an IPv6 address can connect to an instance through
         * IPv6; otherwise, IPv4 should be used.
         * </p>
         * <note>
         * <p>
         * The <code>cidrs</code> parameter lists the IPv4 addresses that are allowed to connect to an instance.
         * </p>
         * </note>
         * <p>
         * For more information about CIDR block notation, see <a
         * href="https://en.wikipedia.org/wiki/Classless_Inter-Domain_Routing#CIDR_notation">Classless Inter-Domain
         * Routing</a> on <i>Wikipedia</i>.
         * </p>
         * 
         * @param ipv6Cidrs
         *        The IPv6 address, or range of IPv6 addresses (in CIDR notation) that are allowed to connect to an
         *        instance through the ports, and the protocol. Only devices with an IPv6 address can connect to an
         *        instance through IPv6; otherwise, IPv4 should be used.</p> <note>
         *        <p>
         *        The <code>cidrs</code> parameter lists the IPv4 addresses that are allowed to connect to an instance.
         *        </p>
         *        </note>
         *        <p>
         *        For more information about CIDR block notation, see <a
         *        href="https://en.wikipedia.org/wiki/Classless_Inter-Domain_Routing#CIDR_notation">Classless
         *        Inter-Domain Routing</a> on <i>Wikipedia</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipv6Cidrs(String... ipv6Cidrs);

        /**
         * <p>
         * An alias that defines access for a preconfigured range of IP addresses.
         * </p>
         * <p>
         * The only alias currently supported is <code>lightsail-connect</code>, which allows IP addresses of the
         * browser-based RDP/SSH client in the Lightsail console to connect to your instance.
         * </p>
         * 
         * @param cidrListAliases
         *        An alias that defines access for a preconfigured range of IP addresses.</p>
         *        <p>
         *        The only alias currently supported is <code>lightsail-connect</code>, which allows IP addresses of the
         *        browser-based RDP/SSH client in the Lightsail console to connect to your instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cidrListAliases(Collection<String> cidrListAliases);

        /**
         * <p>
         * An alias that defines access for a preconfigured range of IP addresses.
         * </p>
         * <p>
         * The only alias currently supported is <code>lightsail-connect</code>, which allows IP addresses of the
         * browser-based RDP/SSH client in the Lightsail console to connect to your instance.
         * </p>
         * 
         * @param cidrListAliases
         *        An alias that defines access for a preconfigured range of IP addresses.</p>
         *        <p>
         *        The only alias currently supported is <code>lightsail-connect</code>, which allows IP addresses of the
         *        browser-based RDP/SSH client in the Lightsail console to connect to your instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cidrListAliases(String... cidrListAliases);
    }

    static final class BuilderImpl implements Builder {
        private Integer fromPort;

        private Integer toPort;

        private String protocol;

        private String accessFrom;

        private String accessType;

        private String commonName;

        private String accessDirection;

        private List<String> cidrs = DefaultSdkAutoConstructList.getInstance();

        private List<String> ipv6Cidrs = DefaultSdkAutoConstructList.getInstance();

        private List<String> cidrListAliases = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(InstancePortInfo model) {
            fromPort(model.fromPort);
            toPort(model.toPort);
            protocol(model.protocol);
            accessFrom(model.accessFrom);
            accessType(model.accessType);
            commonName(model.commonName);
            accessDirection(model.accessDirection);
            cidrs(model.cidrs);
            ipv6Cidrs(model.ipv6Cidrs);
            cidrListAliases(model.cidrListAliases);
        }

        public final Integer getFromPort() {
            return fromPort;
        }

        public final void setFromPort(Integer fromPort) {
            this.fromPort = fromPort;
        }

        @Override
        public final Builder fromPort(Integer fromPort) {
            this.fromPort = fromPort;
            return this;
        }

        public final Integer getToPort() {
            return toPort;
        }

        public final void setToPort(Integer toPort) {
            this.toPort = toPort;
        }

        @Override
        public final Builder toPort(Integer toPort) {
            this.toPort = toPort;
            return this;
        }

        public final String getProtocol() {
            return protocol;
        }

        public final void setProtocol(String protocol) {
            this.protocol = protocol;
        }

        @Override
        public final Builder protocol(String protocol) {
            this.protocol = protocol;
            return this;
        }

        @Override
        public final Builder protocol(NetworkProtocol protocol) {
            this.protocol(protocol == null ? null : protocol.toString());
            return this;
        }

        public final String getAccessFrom() {
            return accessFrom;
        }

        public final void setAccessFrom(String accessFrom) {
            this.accessFrom = accessFrom;
        }

        @Override
        public final Builder accessFrom(String accessFrom) {
            this.accessFrom = accessFrom;
            return this;
        }

        public final String getAccessType() {
            return accessType;
        }

        public final void setAccessType(String accessType) {
            this.accessType = accessType;
        }

        @Override
        public final Builder accessType(String accessType) {
            this.accessType = accessType;
            return this;
        }

        @Override
        public final Builder accessType(PortAccessType accessType) {
            this.accessType(accessType == null ? null : accessType.toString());
            return this;
        }

        public final String getCommonName() {
            return commonName;
        }

        public final void setCommonName(String commonName) {
            this.commonName = commonName;
        }

        @Override
        public final Builder commonName(String commonName) {
            this.commonName = commonName;
            return this;
        }

        public final String getAccessDirection() {
            return accessDirection;
        }

        public final void setAccessDirection(String accessDirection) {
            this.accessDirection = accessDirection;
        }

        @Override
        public final Builder accessDirection(String accessDirection) {
            this.accessDirection = accessDirection;
            return this;
        }

        @Override
        public final Builder accessDirection(AccessDirection accessDirection) {
            this.accessDirection(accessDirection == null ? null : accessDirection.toString());
            return this;
        }

        public final Collection<String> getCidrs() {
            if (cidrs instanceof SdkAutoConstructList) {
                return null;
            }
            return cidrs;
        }

        public final void setCidrs(Collection<String> cidrs) {
            this.cidrs = StringListCopier.copy(cidrs);
        }

        @Override
        public final Builder cidrs(Collection<String> cidrs) {
            this.cidrs = StringListCopier.copy(cidrs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder cidrs(String... cidrs) {
            cidrs(Arrays.asList(cidrs));
            return this;
        }

        public final Collection<String> getIpv6Cidrs() {
            if (ipv6Cidrs instanceof SdkAutoConstructList) {
                return null;
            }
            return ipv6Cidrs;
        }

        public final void setIpv6Cidrs(Collection<String> ipv6Cidrs) {
            this.ipv6Cidrs = StringListCopier.copy(ipv6Cidrs);
        }

        @Override
        public final Builder ipv6Cidrs(Collection<String> ipv6Cidrs) {
            this.ipv6Cidrs = StringListCopier.copy(ipv6Cidrs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ipv6Cidrs(String... ipv6Cidrs) {
            ipv6Cidrs(Arrays.asList(ipv6Cidrs));
            return this;
        }

        public final Collection<String> getCidrListAliases() {
            if (cidrListAliases instanceof SdkAutoConstructList) {
                return null;
            }
            return cidrListAliases;
        }

        public final void setCidrListAliases(Collection<String> cidrListAliases) {
            this.cidrListAliases = StringListCopier.copy(cidrListAliases);
        }

        @Override
        public final Builder cidrListAliases(Collection<String> cidrListAliases) {
            this.cidrListAliases = StringListCopier.copy(cidrListAliases);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder cidrListAliases(String... cidrListAliases) {
            cidrListAliases(Arrays.asList(cidrListAliases));
            return this;
        }

        @Override
        public InstancePortInfo build() {
            return new InstancePortInfo(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
