/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lightsail.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an Amazon Lightsail content delivery network (CDN) distribution.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LightsailDistribution implements SdkPojo, Serializable,
        ToCopyableBuilder<LightsailDistribution.Builder, LightsailDistribution> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(LightsailDistribution::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(LightsailDistribution::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<String> SUPPORT_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("supportCode").getter(getter(LightsailDistribution::supportCode)).setter(setter(Builder::supportCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("supportCode").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt").getter(getter(LightsailDistribution::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build()).build();

    private static final SdkField<ResourceLocation> LOCATION_FIELD = SdkField
            .<ResourceLocation> builder(MarshallingType.SDK_POJO).memberName("location")
            .getter(getter(LightsailDistribution::location)).setter(setter(Builder::location))
            .constructor(ResourceLocation::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("location").build()).build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("resourceType").getter(getter(LightsailDistribution::resourceTypeAsString))
            .setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceType").build()).build();

    private static final SdkField<List<String>> ALTERNATIVE_DOMAIN_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("alternativeDomainNames")
            .getter(getter(LightsailDistribution::alternativeDomainNames))
            .setter(setter(Builder::alternativeDomainNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("alternativeDomainNames").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(LightsailDistribution::status)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<Boolean> IS_ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("isEnabled").getter(getter(LightsailDistribution::isEnabled)).setter(setter(Builder::isEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("isEnabled").build()).build();

    private static final SdkField<String> DOMAIN_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("domainName").getter(getter(LightsailDistribution::domainName)).setter(setter(Builder::domainName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("domainName").build()).build();

    private static final SdkField<String> BUNDLE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("bundleId").getter(getter(LightsailDistribution::bundleId)).setter(setter(Builder::bundleId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("bundleId").build()).build();

    private static final SdkField<String> CERTIFICATE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("certificateName").getter(getter(LightsailDistribution::certificateName))
            .setter(setter(Builder::certificateName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("certificateName").build()).build();

    private static final SdkField<Origin> ORIGIN_FIELD = SdkField.<Origin> builder(MarshallingType.SDK_POJO).memberName("origin")
            .getter(getter(LightsailDistribution::origin)).setter(setter(Builder::origin)).constructor(Origin::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("origin").build()).build();

    private static final SdkField<String> ORIGIN_PUBLIC_DNS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("originPublicDNS").getter(getter(LightsailDistribution::originPublicDNS))
            .setter(setter(Builder::originPublicDNS))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("originPublicDNS").build()).build();

    private static final SdkField<CacheBehavior> DEFAULT_CACHE_BEHAVIOR_FIELD = SdkField
            .<CacheBehavior> builder(MarshallingType.SDK_POJO).memberName("defaultCacheBehavior")
            .getter(getter(LightsailDistribution::defaultCacheBehavior)).setter(setter(Builder::defaultCacheBehavior))
            .constructor(CacheBehavior::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("defaultCacheBehavior").build())
            .build();

    private static final SdkField<CacheSettings> CACHE_BEHAVIOR_SETTINGS_FIELD = SdkField
            .<CacheSettings> builder(MarshallingType.SDK_POJO).memberName("cacheBehaviorSettings")
            .getter(getter(LightsailDistribution::cacheBehaviorSettings)).setter(setter(Builder::cacheBehaviorSettings))
            .constructor(CacheSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("cacheBehaviorSettings").build())
            .build();

    private static final SdkField<List<CacheBehaviorPerPath>> CACHE_BEHAVIORS_FIELD = SdkField
            .<List<CacheBehaviorPerPath>> builder(MarshallingType.LIST)
            .memberName("cacheBehaviors")
            .getter(getter(LightsailDistribution::cacheBehaviors))
            .setter(setter(Builder::cacheBehaviors))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("cacheBehaviors").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<CacheBehaviorPerPath> builder(MarshallingType.SDK_POJO)
                                            .constructor(CacheBehaviorPerPath::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> ABLE_TO_UPDATE_BUNDLE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("ableToUpdateBundle").getter(getter(LightsailDistribution::ableToUpdateBundle))
            .setter(setter(Builder::ableToUpdateBundle))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ableToUpdateBundle").build())
            .build();

    private static final SdkField<String> IP_ADDRESS_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ipAddressType").getter(getter(LightsailDistribution::ipAddressTypeAsString))
            .setter(setter(Builder::ipAddressType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ipAddressType").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("tags")
            .getter(getter(LightsailDistribution::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> VIEWER_MINIMUM_TLS_PROTOCOL_VERSION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("viewerMinimumTlsProtocolVersion")
            .getter(getter(LightsailDistribution::viewerMinimumTlsProtocolVersion))
            .setter(setter(Builder::viewerMinimumTlsProtocolVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("viewerMinimumTlsProtocolVersion")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, ARN_FIELD,
            SUPPORT_CODE_FIELD, CREATED_AT_FIELD, LOCATION_FIELD, RESOURCE_TYPE_FIELD, ALTERNATIVE_DOMAIN_NAMES_FIELD,
            STATUS_FIELD, IS_ENABLED_FIELD, DOMAIN_NAME_FIELD, BUNDLE_ID_FIELD, CERTIFICATE_NAME_FIELD, ORIGIN_FIELD,
            ORIGIN_PUBLIC_DNS_FIELD, DEFAULT_CACHE_BEHAVIOR_FIELD, CACHE_BEHAVIOR_SETTINGS_FIELD, CACHE_BEHAVIORS_FIELD,
            ABLE_TO_UPDATE_BUNDLE_FIELD, IP_ADDRESS_TYPE_FIELD, TAGS_FIELD, VIEWER_MINIMUM_TLS_PROTOCOL_VERSION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String arn;

    private final String supportCode;

    private final Instant createdAt;

    private final ResourceLocation location;

    private final String resourceType;

    private final List<String> alternativeDomainNames;

    private final String status;

    private final Boolean isEnabled;

    private final String domainName;

    private final String bundleId;

    private final String certificateName;

    private final Origin origin;

    private final String originPublicDNS;

    private final CacheBehavior defaultCacheBehavior;

    private final CacheSettings cacheBehaviorSettings;

    private final List<CacheBehaviorPerPath> cacheBehaviors;

    private final Boolean ableToUpdateBundle;

    private final String ipAddressType;

    private final List<Tag> tags;

    private final String viewerMinimumTlsProtocolVersion;

    private LightsailDistribution(BuilderImpl builder) {
        this.name = builder.name;
        this.arn = builder.arn;
        this.supportCode = builder.supportCode;
        this.createdAt = builder.createdAt;
        this.location = builder.location;
        this.resourceType = builder.resourceType;
        this.alternativeDomainNames = builder.alternativeDomainNames;
        this.status = builder.status;
        this.isEnabled = builder.isEnabled;
        this.domainName = builder.domainName;
        this.bundleId = builder.bundleId;
        this.certificateName = builder.certificateName;
        this.origin = builder.origin;
        this.originPublicDNS = builder.originPublicDNS;
        this.defaultCacheBehavior = builder.defaultCacheBehavior;
        this.cacheBehaviorSettings = builder.cacheBehaviorSettings;
        this.cacheBehaviors = builder.cacheBehaviors;
        this.ableToUpdateBundle = builder.ableToUpdateBundle;
        this.ipAddressType = builder.ipAddressType;
        this.tags = builder.tags;
        this.viewerMinimumTlsProtocolVersion = builder.viewerMinimumTlsProtocolVersion;
    }

    /**
     * <p>
     * The name of the distribution.
     * </p>
     * 
     * @return The name of the distribution.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the distribution.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the distribution.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The support code. Include this code in your email to support when you have questions about your Lightsail
     * distribution. This code enables our support team to look up your Lightsail information more easily.
     * </p>
     * 
     * @return The support code. Include this code in your email to support when you have questions about your Lightsail
     *         distribution. This code enables our support team to look up your Lightsail information more easily.
     */
    public final String supportCode() {
        return supportCode;
    }

    /**
     * <p>
     * The timestamp when the distribution was created.
     * </p>
     * 
     * @return The timestamp when the distribution was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * An object that describes the location of the distribution, such as the Amazon Web Services Region and
     * Availability Zone.
     * </p>
     * <note>
     * <p>
     * Lightsail distributions are global resources that can reference an origin in any Amazon Web Services Region, and
     * distribute its content globally. However, all distributions are located in the <code>us-east-1</code> Region.
     * </p>
     * </note>
     * 
     * @return An object that describes the location of the distribution, such as the Amazon Web Services Region and
     *         Availability Zone.</p> <note>
     *         <p>
     *         Lightsail distributions are global resources that can reference an origin in any Amazon Web Services
     *         Region, and distribute its content globally. However, all distributions are located in the
     *         <code>us-east-1</code> Region.
     *         </p>
     */
    public final ResourceLocation location() {
        return location;
    }

    /**
     * <p>
     * The Lightsail resource type (<code>Distribution</code>).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The Lightsail resource type (<code>Distribution</code>).
     * @see ResourceType
     */
    public final ResourceType resourceType() {
        return ResourceType.fromValue(resourceType);
    }

    /**
     * <p>
     * The Lightsail resource type (<code>Distribution</code>).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The Lightsail resource type (<code>Distribution</code>).
     * @see ResourceType
     */
    public final String resourceTypeAsString() {
        return resourceType;
    }

    /**
     * For responses, this returns true if the service returned a value for the AlternativeDomainNames property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasAlternativeDomainNames() {
        return alternativeDomainNames != null && !(alternativeDomainNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The alternate domain names of the distribution.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAlternativeDomainNames} method.
     * </p>
     * 
     * @return The alternate domain names of the distribution.
     */
    public final List<String> alternativeDomainNames() {
        return alternativeDomainNames;
    }

    /**
     * <p>
     * The status of the distribution.
     * </p>
     * 
     * @return The status of the distribution.
     */
    public final String status() {
        return status;
    }

    /**
     * <p>
     * Indicates whether the distribution is enabled.
     * </p>
     * 
     * @return Indicates whether the distribution is enabled.
     */
    public final Boolean isEnabled() {
        return isEnabled;
    }

    /**
     * <p>
     * The domain name of the distribution.
     * </p>
     * 
     * @return The domain name of the distribution.
     */
    public final String domainName() {
        return domainName;
    }

    /**
     * <p>
     * The ID of the bundle currently applied to the distribution.
     * </p>
     * 
     * @return The ID of the bundle currently applied to the distribution.
     */
    public final String bundleId() {
        return bundleId;
    }

    /**
     * <p>
     * The name of the SSL/TLS certificate attached to the distribution, if any.
     * </p>
     * 
     * @return The name of the SSL/TLS certificate attached to the distribution, if any.
     */
    public final String certificateName() {
        return certificateName;
    }

    /**
     * <p>
     * An object that describes the origin resource of the distribution, such as a Lightsail instance, bucket, or load
     * balancer.
     * </p>
     * <p>
     * The distribution pulls, caches, and serves content from the origin.
     * </p>
     * 
     * @return An object that describes the origin resource of the distribution, such as a Lightsail instance, bucket,
     *         or load balancer.</p>
     *         <p>
     *         The distribution pulls, caches, and serves content from the origin.
     */
    public final Origin origin() {
        return origin;
    }

    /**
     * <p>
     * The public DNS of the origin.
     * </p>
     * 
     * @return The public DNS of the origin.
     */
    public final String originPublicDNS() {
        return originPublicDNS;
    }

    /**
     * <p>
     * An object that describes the default cache behavior of the distribution.
     * </p>
     * 
     * @return An object that describes the default cache behavior of the distribution.
     */
    public final CacheBehavior defaultCacheBehavior() {
        return defaultCacheBehavior;
    }

    /**
     * <p>
     * An object that describes the cache behavior settings of the distribution.
     * </p>
     * 
     * @return An object that describes the cache behavior settings of the distribution.
     */
    public final CacheSettings cacheBehaviorSettings() {
        return cacheBehaviorSettings;
    }

    /**
     * For responses, this returns true if the service returned a value for the CacheBehaviors property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasCacheBehaviors() {
        return cacheBehaviors != null && !(cacheBehaviors instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of objects that describe the per-path cache behavior of the distribution.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCacheBehaviors} method.
     * </p>
     * 
     * @return An array of objects that describe the per-path cache behavior of the distribution.
     */
    public final List<CacheBehaviorPerPath> cacheBehaviors() {
        return cacheBehaviors;
    }

    /**
     * <p>
     * Indicates whether the bundle that is currently applied to your distribution, specified using the
     * <code>distributionName</code> parameter, can be changed to another bundle.
     * </p>
     * <p>
     * Use the <code>UpdateDistributionBundle</code> action to change your distribution's bundle.
     * </p>
     * 
     * @return Indicates whether the bundle that is currently applied to your distribution, specified using the
     *         <code>distributionName</code> parameter, can be changed to another bundle.</p>
     *         <p>
     *         Use the <code>UpdateDistributionBundle</code> action to change your distribution's bundle.
     */
    public final Boolean ableToUpdateBundle() {
        return ableToUpdateBundle;
    }

    /**
     * <p>
     * The IP address type of the distribution.
     * </p>
     * <p>
     * The possible values are <code>ipv4</code> for IPv4 only, and <code>dualstack</code> for IPv4 and IPv6.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #ipAddressType}
     * will return {@link IpAddressType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #ipAddressTypeAsString}.
     * </p>
     * 
     * @return The IP address type of the distribution.</p>
     *         <p>
     *         The possible values are <code>ipv4</code> for IPv4 only, and <code>dualstack</code> for IPv4 and IPv6.
     * @see IpAddressType
     */
    public final IpAddressType ipAddressType() {
        return IpAddressType.fromValue(ipAddressType);
    }

    /**
     * <p>
     * The IP address type of the distribution.
     * </p>
     * <p>
     * The possible values are <code>ipv4</code> for IPv4 only, and <code>dualstack</code> for IPv4 and IPv6.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #ipAddressType}
     * will return {@link IpAddressType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #ipAddressTypeAsString}.
     * </p>
     * 
     * @return The IP address type of the distribution.</p>
     *         <p>
     *         The possible values are <code>ipv4</code> for IPv4 only, and <code>dualstack</code> for IPv4 and IPv6.
     * @see IpAddressType
     */
    public final String ipAddressTypeAsString() {
        return ipAddressType;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tag keys and optional values for the resource. For more information about tags in Lightsail, see the <a
     * href="https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-tags">Amazon Lightsail Developer
     * Guide</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tag keys and optional values for the resource. For more information about tags in Lightsail, see the
     *         <a href="https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-tags">Amazon Lightsail
     *         Developer Guide</a>.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * The minimum TLS protocol version that the distribution can use to communicate with viewers.
     * </p>
     * 
     * @return The minimum TLS protocol version that the distribution can use to communicate with viewers.
     */
    public final String viewerMinimumTlsProtocolVersion() {
        return viewerMinimumTlsProtocolVersion;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(supportCode());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(location());
        hashCode = 31 * hashCode + Objects.hashCode(resourceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasAlternativeDomainNames() ? alternativeDomainNames() : null);
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(isEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(domainName());
        hashCode = 31 * hashCode + Objects.hashCode(bundleId());
        hashCode = 31 * hashCode + Objects.hashCode(certificateName());
        hashCode = 31 * hashCode + Objects.hashCode(origin());
        hashCode = 31 * hashCode + Objects.hashCode(originPublicDNS());
        hashCode = 31 * hashCode + Objects.hashCode(defaultCacheBehavior());
        hashCode = 31 * hashCode + Objects.hashCode(cacheBehaviorSettings());
        hashCode = 31 * hashCode + Objects.hashCode(hasCacheBehaviors() ? cacheBehaviors() : null);
        hashCode = 31 * hashCode + Objects.hashCode(ableToUpdateBundle());
        hashCode = 31 * hashCode + Objects.hashCode(ipAddressTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(viewerMinimumTlsProtocolVersion());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LightsailDistribution)) {
            return false;
        }
        LightsailDistribution other = (LightsailDistribution) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(arn(), other.arn())
                && Objects.equals(supportCode(), other.supportCode()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(location(), other.location())
                && Objects.equals(resourceTypeAsString(), other.resourceTypeAsString())
                && hasAlternativeDomainNames() == other.hasAlternativeDomainNames()
                && Objects.equals(alternativeDomainNames(), other.alternativeDomainNames())
                && Objects.equals(status(), other.status()) && Objects.equals(isEnabled(), other.isEnabled())
                && Objects.equals(domainName(), other.domainName()) && Objects.equals(bundleId(), other.bundleId())
                && Objects.equals(certificateName(), other.certificateName()) && Objects.equals(origin(), other.origin())
                && Objects.equals(originPublicDNS(), other.originPublicDNS())
                && Objects.equals(defaultCacheBehavior(), other.defaultCacheBehavior())
                && Objects.equals(cacheBehaviorSettings(), other.cacheBehaviorSettings())
                && hasCacheBehaviors() == other.hasCacheBehaviors() && Objects.equals(cacheBehaviors(), other.cacheBehaviors())
                && Objects.equals(ableToUpdateBundle(), other.ableToUpdateBundle())
                && Objects.equals(ipAddressTypeAsString(), other.ipAddressTypeAsString()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags())
                && Objects.equals(viewerMinimumTlsProtocolVersion(), other.viewerMinimumTlsProtocolVersion());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("LightsailDistribution").add("Name", name()).add("Arn", arn()).add("SupportCode", supportCode())
                .add("CreatedAt", createdAt()).add("Location", location()).add("ResourceType", resourceTypeAsString())
                .add("AlternativeDomainNames", hasAlternativeDomainNames() ? alternativeDomainNames() : null)
                .add("Status", status()).add("IsEnabled", isEnabled()).add("DomainName", domainName())
                .add("BundleId", bundleId()).add("CertificateName", certificateName()).add("Origin", origin())
                .add("OriginPublicDNS", originPublicDNS()).add("DefaultCacheBehavior", defaultCacheBehavior())
                .add("CacheBehaviorSettings", cacheBehaviorSettings())
                .add("CacheBehaviors", hasCacheBehaviors() ? cacheBehaviors() : null)
                .add("AbleToUpdateBundle", ableToUpdateBundle()).add("IpAddressType", ipAddressTypeAsString())
                .add("Tags", hasTags() ? tags() : null).add("ViewerMinimumTlsProtocolVersion", viewerMinimumTlsProtocolVersion())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "supportCode":
            return Optional.ofNullable(clazz.cast(supportCode()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "location":
            return Optional.ofNullable(clazz.cast(location()));
        case "resourceType":
            return Optional.ofNullable(clazz.cast(resourceTypeAsString()));
        case "alternativeDomainNames":
            return Optional.ofNullable(clazz.cast(alternativeDomainNames()));
        case "status":
            return Optional.ofNullable(clazz.cast(status()));
        case "isEnabled":
            return Optional.ofNullable(clazz.cast(isEnabled()));
        case "domainName":
            return Optional.ofNullable(clazz.cast(domainName()));
        case "bundleId":
            return Optional.ofNullable(clazz.cast(bundleId()));
        case "certificateName":
            return Optional.ofNullable(clazz.cast(certificateName()));
        case "origin":
            return Optional.ofNullable(clazz.cast(origin()));
        case "originPublicDNS":
            return Optional.ofNullable(clazz.cast(originPublicDNS()));
        case "defaultCacheBehavior":
            return Optional.ofNullable(clazz.cast(defaultCacheBehavior()));
        case "cacheBehaviorSettings":
            return Optional.ofNullable(clazz.cast(cacheBehaviorSettings()));
        case "cacheBehaviors":
            return Optional.ofNullable(clazz.cast(cacheBehaviors()));
        case "ableToUpdateBundle":
            return Optional.ofNullable(clazz.cast(ableToUpdateBundle()));
        case "ipAddressType":
            return Optional.ofNullable(clazz.cast(ipAddressTypeAsString()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "viewerMinimumTlsProtocolVersion":
            return Optional.ofNullable(clazz.cast(viewerMinimumTlsProtocolVersion()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<LightsailDistribution, T> g) {
        return obj -> g.apply((LightsailDistribution) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LightsailDistribution> {
        /**
         * <p>
         * The name of the distribution.
         * </p>
         * 
         * @param name
         *        The name of the distribution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the distribution.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the distribution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The support code. Include this code in your email to support when you have questions about your Lightsail
         * distribution. This code enables our support team to look up your Lightsail information more easily.
         * </p>
         * 
         * @param supportCode
         *        The support code. Include this code in your email to support when you have questions about your
         *        Lightsail distribution. This code enables our support team to look up your Lightsail information more
         *        easily.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportCode(String supportCode);

        /**
         * <p>
         * The timestamp when the distribution was created.
         * </p>
         * 
         * @param createdAt
         *        The timestamp when the distribution was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * An object that describes the location of the distribution, such as the Amazon Web Services Region and
         * Availability Zone.
         * </p>
         * <note>
         * <p>
         * Lightsail distributions are global resources that can reference an origin in any Amazon Web Services Region,
         * and distribute its content globally. However, all distributions are located in the <code>us-east-1</code>
         * Region.
         * </p>
         * </note>
         * 
         * @param location
         *        An object that describes the location of the distribution, such as the Amazon Web Services Region and
         *        Availability Zone.</p> <note>
         *        <p>
         *        Lightsail distributions are global resources that can reference an origin in any Amazon Web Services
         *        Region, and distribute its content globally. However, all distributions are located in the
         *        <code>us-east-1</code> Region.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder location(ResourceLocation location);

        /**
         * <p>
         * An object that describes the location of the distribution, such as the Amazon Web Services Region and
         * Availability Zone.
         * </p>
         * <note>
         * <p>
         * Lightsail distributions are global resources that can reference an origin in any Amazon Web Services Region,
         * and distribute its content globally. However, all distributions are located in the <code>us-east-1</code>
         * Region.
         * </p>
         * </note> This is a convenience method that creates an instance of the {@link ResourceLocation.Builder}
         * avoiding the need to create one manually via {@link ResourceLocation#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ResourceLocation.Builder#build()} is called immediately and its
         * result is passed to {@link #location(ResourceLocation)}.
         * 
         * @param location
         *        a consumer that will call methods on {@link ResourceLocation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #location(ResourceLocation)
         */
        default Builder location(Consumer<ResourceLocation.Builder> location) {
            return location(ResourceLocation.builder().applyMutation(location).build());
        }

        /**
         * <p>
         * The Lightsail resource type (<code>Distribution</code>).
         * </p>
         * 
         * @param resourceType
         *        The Lightsail resource type (<code>Distribution</code>).
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * The Lightsail resource type (<code>Distribution</code>).
         * </p>
         * 
         * @param resourceType
         *        The Lightsail resource type (<code>Distribution</code>).
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder resourceType(ResourceType resourceType);

        /**
         * <p>
         * The alternate domain names of the distribution.
         * </p>
         * 
         * @param alternativeDomainNames
         *        The alternate domain names of the distribution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alternativeDomainNames(Collection<String> alternativeDomainNames);

        /**
         * <p>
         * The alternate domain names of the distribution.
         * </p>
         * 
         * @param alternativeDomainNames
         *        The alternate domain names of the distribution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alternativeDomainNames(String... alternativeDomainNames);

        /**
         * <p>
         * The status of the distribution.
         * </p>
         * 
         * @param status
         *        The status of the distribution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(String status);

        /**
         * <p>
         * Indicates whether the distribution is enabled.
         * </p>
         * 
         * @param isEnabled
         *        Indicates whether the distribution is enabled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isEnabled(Boolean isEnabled);

        /**
         * <p>
         * The domain name of the distribution.
         * </p>
         * 
         * @param domainName
         *        The domain name of the distribution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainName(String domainName);

        /**
         * <p>
         * The ID of the bundle currently applied to the distribution.
         * </p>
         * 
         * @param bundleId
         *        The ID of the bundle currently applied to the distribution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bundleId(String bundleId);

        /**
         * <p>
         * The name of the SSL/TLS certificate attached to the distribution, if any.
         * </p>
         * 
         * @param certificateName
         *        The name of the SSL/TLS certificate attached to the distribution, if any.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder certificateName(String certificateName);

        /**
         * <p>
         * An object that describes the origin resource of the distribution, such as a Lightsail instance, bucket, or
         * load balancer.
         * </p>
         * <p>
         * The distribution pulls, caches, and serves content from the origin.
         * </p>
         * 
         * @param origin
         *        An object that describes the origin resource of the distribution, such as a Lightsail instance,
         *        bucket, or load balancer.</p>
         *        <p>
         *        The distribution pulls, caches, and serves content from the origin.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder origin(Origin origin);

        /**
         * <p>
         * An object that describes the origin resource of the distribution, such as a Lightsail instance, bucket, or
         * load balancer.
         * </p>
         * <p>
         * The distribution pulls, caches, and serves content from the origin.
         * </p>
         * This is a convenience method that creates an instance of the {@link Origin.Builder} avoiding the need to
         * create one manually via {@link Origin#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Origin.Builder#build()} is called immediately and its result is
         * passed to {@link #origin(Origin)}.
         * 
         * @param origin
         *        a consumer that will call methods on {@link Origin.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #origin(Origin)
         */
        default Builder origin(Consumer<Origin.Builder> origin) {
            return origin(Origin.builder().applyMutation(origin).build());
        }

        /**
         * <p>
         * The public DNS of the origin.
         * </p>
         * 
         * @param originPublicDNS
         *        The public DNS of the origin.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder originPublicDNS(String originPublicDNS);

        /**
         * <p>
         * An object that describes the default cache behavior of the distribution.
         * </p>
         * 
         * @param defaultCacheBehavior
         *        An object that describes the default cache behavior of the distribution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultCacheBehavior(CacheBehavior defaultCacheBehavior);

        /**
         * <p>
         * An object that describes the default cache behavior of the distribution.
         * </p>
         * This is a convenience method that creates an instance of the {@link CacheBehavior.Builder} avoiding the need
         * to create one manually via {@link CacheBehavior#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CacheBehavior.Builder#build()} is called immediately and its
         * result is passed to {@link #defaultCacheBehavior(CacheBehavior)}.
         * 
         * @param defaultCacheBehavior
         *        a consumer that will call methods on {@link CacheBehavior.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #defaultCacheBehavior(CacheBehavior)
         */
        default Builder defaultCacheBehavior(Consumer<CacheBehavior.Builder> defaultCacheBehavior) {
            return defaultCacheBehavior(CacheBehavior.builder().applyMutation(defaultCacheBehavior).build());
        }

        /**
         * <p>
         * An object that describes the cache behavior settings of the distribution.
         * </p>
         * 
         * @param cacheBehaviorSettings
         *        An object that describes the cache behavior settings of the distribution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cacheBehaviorSettings(CacheSettings cacheBehaviorSettings);

        /**
         * <p>
         * An object that describes the cache behavior settings of the distribution.
         * </p>
         * This is a convenience method that creates an instance of the {@link CacheSettings.Builder} avoiding the need
         * to create one manually via {@link CacheSettings#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CacheSettings.Builder#build()} is called immediately and its
         * result is passed to {@link #cacheBehaviorSettings(CacheSettings)}.
         * 
         * @param cacheBehaviorSettings
         *        a consumer that will call methods on {@link CacheSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #cacheBehaviorSettings(CacheSettings)
         */
        default Builder cacheBehaviorSettings(Consumer<CacheSettings.Builder> cacheBehaviorSettings) {
            return cacheBehaviorSettings(CacheSettings.builder().applyMutation(cacheBehaviorSettings).build());
        }

        /**
         * <p>
         * An array of objects that describe the per-path cache behavior of the distribution.
         * </p>
         * 
         * @param cacheBehaviors
         *        An array of objects that describe the per-path cache behavior of the distribution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cacheBehaviors(Collection<CacheBehaviorPerPath> cacheBehaviors);

        /**
         * <p>
         * An array of objects that describe the per-path cache behavior of the distribution.
         * </p>
         * 
         * @param cacheBehaviors
         *        An array of objects that describe the per-path cache behavior of the distribution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cacheBehaviors(CacheBehaviorPerPath... cacheBehaviors);

        /**
         * <p>
         * An array of objects that describe the per-path cache behavior of the distribution.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.lightsail.model.CacheBehaviorPerPath.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.lightsail.model.CacheBehaviorPerPath#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.lightsail.model.CacheBehaviorPerPath.Builder#build()} is called
         * immediately and its result is passed to {@link #cacheBehaviors(List<CacheBehaviorPerPath>)}.
         * 
         * @param cacheBehaviors
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.lightsail.model.CacheBehaviorPerPath.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #cacheBehaviors(java.util.Collection<CacheBehaviorPerPath>)
         */
        Builder cacheBehaviors(Consumer<CacheBehaviorPerPath.Builder>... cacheBehaviors);

        /**
         * <p>
         * Indicates whether the bundle that is currently applied to your distribution, specified using the
         * <code>distributionName</code> parameter, can be changed to another bundle.
         * </p>
         * <p>
         * Use the <code>UpdateDistributionBundle</code> action to change your distribution's bundle.
         * </p>
         * 
         * @param ableToUpdateBundle
         *        Indicates whether the bundle that is currently applied to your distribution, specified using the
         *        <code>distributionName</code> parameter, can be changed to another bundle.</p>
         *        <p>
         *        Use the <code>UpdateDistributionBundle</code> action to change your distribution's bundle.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ableToUpdateBundle(Boolean ableToUpdateBundle);

        /**
         * <p>
         * The IP address type of the distribution.
         * </p>
         * <p>
         * The possible values are <code>ipv4</code> for IPv4 only, and <code>dualstack</code> for IPv4 and IPv6.
         * </p>
         * 
         * @param ipAddressType
         *        The IP address type of the distribution.</p>
         *        <p>
         *        The possible values are <code>ipv4</code> for IPv4 only, and <code>dualstack</code> for IPv4 and IPv6.
         * @see IpAddressType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpAddressType
         */
        Builder ipAddressType(String ipAddressType);

        /**
         * <p>
         * The IP address type of the distribution.
         * </p>
         * <p>
         * The possible values are <code>ipv4</code> for IPv4 only, and <code>dualstack</code> for IPv4 and IPv6.
         * </p>
         * 
         * @param ipAddressType
         *        The IP address type of the distribution.</p>
         *        <p>
         *        The possible values are <code>ipv4</code> for IPv4 only, and <code>dualstack</code> for IPv4 and IPv6.
         * @see IpAddressType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpAddressType
         */
        Builder ipAddressType(IpAddressType ipAddressType);

        /**
         * <p>
         * The tag keys and optional values for the resource. For more information about tags in Lightsail, see the <a
         * href="https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-tags">Amazon Lightsail
         * Developer Guide</a>.
         * </p>
         * 
         * @param tags
         *        The tag keys and optional values for the resource. For more information about tags in Lightsail, see
         *        the <a href="https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-tags">Amazon
         *        Lightsail Developer Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The tag keys and optional values for the resource. For more information about tags in Lightsail, see the <a
         * href="https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-tags">Amazon Lightsail
         * Developer Guide</a>.
         * </p>
         * 
         * @param tags
         *        The tag keys and optional values for the resource. For more information about tags in Lightsail, see
         *        the <a href="https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-tags">Amazon
         *        Lightsail Developer Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The tag keys and optional values for the resource. For more information about tags in Lightsail, see the <a
         * href="https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-tags">Amazon Lightsail
         * Developer Guide</a>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.lightsail.model.Tag.Builder} avoiding the need to create one manually
         * via {@link software.amazon.awssdk.services.lightsail.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.lightsail.model.Tag.Builder#build()} is called immediately and its
         * result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.lightsail.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * The minimum TLS protocol version that the distribution can use to communicate with viewers.
         * </p>
         * 
         * @param viewerMinimumTlsProtocolVersion
         *        The minimum TLS protocol version that the distribution can use to communicate with viewers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder viewerMinimumTlsProtocolVersion(String viewerMinimumTlsProtocolVersion);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String arn;

        private String supportCode;

        private Instant createdAt;

        private ResourceLocation location;

        private String resourceType;

        private List<String> alternativeDomainNames = DefaultSdkAutoConstructList.getInstance();

        private String status;

        private Boolean isEnabled;

        private String domainName;

        private String bundleId;

        private String certificateName;

        private Origin origin;

        private String originPublicDNS;

        private CacheBehavior defaultCacheBehavior;

        private CacheSettings cacheBehaviorSettings;

        private List<CacheBehaviorPerPath> cacheBehaviors = DefaultSdkAutoConstructList.getInstance();

        private Boolean ableToUpdateBundle;

        private String ipAddressType;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String viewerMinimumTlsProtocolVersion;

        private BuilderImpl() {
        }

        private BuilderImpl(LightsailDistribution model) {
            name(model.name);
            arn(model.arn);
            supportCode(model.supportCode);
            createdAt(model.createdAt);
            location(model.location);
            resourceType(model.resourceType);
            alternativeDomainNames(model.alternativeDomainNames);
            status(model.status);
            isEnabled(model.isEnabled);
            domainName(model.domainName);
            bundleId(model.bundleId);
            certificateName(model.certificateName);
            origin(model.origin);
            originPublicDNS(model.originPublicDNS);
            defaultCacheBehavior(model.defaultCacheBehavior);
            cacheBehaviorSettings(model.cacheBehaviorSettings);
            cacheBehaviors(model.cacheBehaviors);
            ableToUpdateBundle(model.ableToUpdateBundle);
            ipAddressType(model.ipAddressType);
            tags(model.tags);
            viewerMinimumTlsProtocolVersion(model.viewerMinimumTlsProtocolVersion);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getSupportCode() {
            return supportCode;
        }

        public final void setSupportCode(String supportCode) {
            this.supportCode = supportCode;
        }

        @Override
        public final Builder supportCode(String supportCode) {
            this.supportCode = supportCode;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final ResourceLocation.Builder getLocation() {
            return location != null ? location.toBuilder() : null;
        }

        public final void setLocation(ResourceLocation.BuilderImpl location) {
            this.location = location != null ? location.build() : null;
        }

        @Override
        public final Builder location(ResourceLocation location) {
            this.location = location;
            return this;
        }

        public final String getResourceType() {
            return resourceType;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        @Override
        public final Builder resourceType(ResourceType resourceType) {
            this.resourceType(resourceType == null ? null : resourceType.toString());
            return this;
        }

        public final Collection<String> getAlternativeDomainNames() {
            if (alternativeDomainNames instanceof SdkAutoConstructList) {
                return null;
            }
            return alternativeDomainNames;
        }

        public final void setAlternativeDomainNames(Collection<String> alternativeDomainNames) {
            this.alternativeDomainNames = StringListCopier.copy(alternativeDomainNames);
        }

        @Override
        public final Builder alternativeDomainNames(Collection<String> alternativeDomainNames) {
            this.alternativeDomainNames = StringListCopier.copy(alternativeDomainNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder alternativeDomainNames(String... alternativeDomainNames) {
            alternativeDomainNames(Arrays.asList(alternativeDomainNames));
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        public final Boolean getIsEnabled() {
            return isEnabled;
        }

        public final void setIsEnabled(Boolean isEnabled) {
            this.isEnabled = isEnabled;
        }

        @Override
        public final Builder isEnabled(Boolean isEnabled) {
            this.isEnabled = isEnabled;
            return this;
        }

        public final String getDomainName() {
            return domainName;
        }

        public final void setDomainName(String domainName) {
            this.domainName = domainName;
        }

        @Override
        public final Builder domainName(String domainName) {
            this.domainName = domainName;
            return this;
        }

        public final String getBundleId() {
            return bundleId;
        }

        public final void setBundleId(String bundleId) {
            this.bundleId = bundleId;
        }

        @Override
        public final Builder bundleId(String bundleId) {
            this.bundleId = bundleId;
            return this;
        }

        public final String getCertificateName() {
            return certificateName;
        }

        public final void setCertificateName(String certificateName) {
            this.certificateName = certificateName;
        }

        @Override
        public final Builder certificateName(String certificateName) {
            this.certificateName = certificateName;
            return this;
        }

        public final Origin.Builder getOrigin() {
            return origin != null ? origin.toBuilder() : null;
        }

        public final void setOrigin(Origin.BuilderImpl origin) {
            this.origin = origin != null ? origin.build() : null;
        }

        @Override
        public final Builder origin(Origin origin) {
            this.origin = origin;
            return this;
        }

        public final String getOriginPublicDNS() {
            return originPublicDNS;
        }

        public final void setOriginPublicDNS(String originPublicDNS) {
            this.originPublicDNS = originPublicDNS;
        }

        @Override
        public final Builder originPublicDNS(String originPublicDNS) {
            this.originPublicDNS = originPublicDNS;
            return this;
        }

        public final CacheBehavior.Builder getDefaultCacheBehavior() {
            return defaultCacheBehavior != null ? defaultCacheBehavior.toBuilder() : null;
        }

        public final void setDefaultCacheBehavior(CacheBehavior.BuilderImpl defaultCacheBehavior) {
            this.defaultCacheBehavior = defaultCacheBehavior != null ? defaultCacheBehavior.build() : null;
        }

        @Override
        public final Builder defaultCacheBehavior(CacheBehavior defaultCacheBehavior) {
            this.defaultCacheBehavior = defaultCacheBehavior;
            return this;
        }

        public final CacheSettings.Builder getCacheBehaviorSettings() {
            return cacheBehaviorSettings != null ? cacheBehaviorSettings.toBuilder() : null;
        }

        public final void setCacheBehaviorSettings(CacheSettings.BuilderImpl cacheBehaviorSettings) {
            this.cacheBehaviorSettings = cacheBehaviorSettings != null ? cacheBehaviorSettings.build() : null;
        }

        @Override
        public final Builder cacheBehaviorSettings(CacheSettings cacheBehaviorSettings) {
            this.cacheBehaviorSettings = cacheBehaviorSettings;
            return this;
        }

        public final List<CacheBehaviorPerPath.Builder> getCacheBehaviors() {
            List<CacheBehaviorPerPath.Builder> result = CacheBehaviorListCopier.copyToBuilder(this.cacheBehaviors);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setCacheBehaviors(Collection<CacheBehaviorPerPath.BuilderImpl> cacheBehaviors) {
            this.cacheBehaviors = CacheBehaviorListCopier.copyFromBuilder(cacheBehaviors);
        }

        @Override
        public final Builder cacheBehaviors(Collection<CacheBehaviorPerPath> cacheBehaviors) {
            this.cacheBehaviors = CacheBehaviorListCopier.copy(cacheBehaviors);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder cacheBehaviors(CacheBehaviorPerPath... cacheBehaviors) {
            cacheBehaviors(Arrays.asList(cacheBehaviors));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder cacheBehaviors(Consumer<CacheBehaviorPerPath.Builder>... cacheBehaviors) {
            cacheBehaviors(Stream.of(cacheBehaviors).map(c -> CacheBehaviorPerPath.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Boolean getAbleToUpdateBundle() {
            return ableToUpdateBundle;
        }

        public final void setAbleToUpdateBundle(Boolean ableToUpdateBundle) {
            this.ableToUpdateBundle = ableToUpdateBundle;
        }

        @Override
        public final Builder ableToUpdateBundle(Boolean ableToUpdateBundle) {
            this.ableToUpdateBundle = ableToUpdateBundle;
            return this;
        }

        public final String getIpAddressType() {
            return ipAddressType;
        }

        public final void setIpAddressType(String ipAddressType) {
            this.ipAddressType = ipAddressType;
        }

        @Override
        public final Builder ipAddressType(String ipAddressType) {
            this.ipAddressType = ipAddressType;
            return this;
        }

        @Override
        public final Builder ipAddressType(IpAddressType ipAddressType) {
            this.ipAddressType(ipAddressType == null ? null : ipAddressType.toString());
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getViewerMinimumTlsProtocolVersion() {
            return viewerMinimumTlsProtocolVersion;
        }

        public final void setViewerMinimumTlsProtocolVersion(String viewerMinimumTlsProtocolVersion) {
            this.viewerMinimumTlsProtocolVersion = viewerMinimumTlsProtocolVersion;
        }

        @Override
        public final Builder viewerMinimumTlsProtocolVersion(String viewerMinimumTlsProtocolVersion) {
            this.viewerMinimumTlsProtocolVersion = viewerMinimumTlsProtocolVersion;
            return this;
        }

        @Override
        public LightsailDistribution build() {
            return new LightsailDistribution(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
