/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lightsail.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the hardware for the instance.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InstanceHardware implements SdkPojo, Serializable,
        ToCopyableBuilder<InstanceHardware.Builder, InstanceHardware> {
    private static final SdkField<Integer> CPU_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(InstanceHardware::cpuCount)).setter(setter(Builder::cpuCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("cpuCount").build()).build();

    private static final SdkField<List<Disk>> DISKS_FIELD = SdkField
            .<List<Disk>> builder(MarshallingType.LIST)
            .getter(getter(InstanceHardware::disks))
            .setter(setter(Builder::disks))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("disks").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Disk> builder(MarshallingType.SDK_POJO)
                                            .constructor(Disk::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Float> RAM_SIZE_IN_GB_FIELD = SdkField.<Float> builder(MarshallingType.FLOAT)
            .getter(getter(InstanceHardware::ramSizeInGb)).setter(setter(Builder::ramSizeInGb))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ramSizeInGb").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CPU_COUNT_FIELD, DISKS_FIELD,
            RAM_SIZE_IN_GB_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer cpuCount;

    private final List<Disk> disks;

    private final Float ramSizeInGb;

    private InstanceHardware(BuilderImpl builder) {
        this.cpuCount = builder.cpuCount;
        this.disks = builder.disks;
        this.ramSizeInGb = builder.ramSizeInGb;
    }

    /**
     * <p>
     * The number of vCPUs the instance has.
     * </p>
     * 
     * @return The number of vCPUs the instance has.
     */
    public Integer cpuCount() {
        return cpuCount;
    }

    /**
     * <p>
     * The disks attached to the instance.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The disks attached to the instance.
     */
    public List<Disk> disks() {
        return disks;
    }

    /**
     * <p>
     * The amount of RAM in GB on the instance (e.g., <code>1.0</code>).
     * </p>
     * 
     * @return The amount of RAM in GB on the instance (e.g., <code>1.0</code>).
     */
    public Float ramSizeInGb() {
        return ramSizeInGb;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(cpuCount());
        hashCode = 31 * hashCode + Objects.hashCode(disks());
        hashCode = 31 * hashCode + Objects.hashCode(ramSizeInGb());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InstanceHardware)) {
            return false;
        }
        InstanceHardware other = (InstanceHardware) obj;
        return Objects.equals(cpuCount(), other.cpuCount()) && Objects.equals(disks(), other.disks())
                && Objects.equals(ramSizeInGb(), other.ramSizeInGb());
    }

    @Override
    public String toString() {
        return ToString.builder("InstanceHardware").add("CpuCount", cpuCount()).add("Disks", disks())
                .add("RamSizeInGb", ramSizeInGb()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "cpuCount":
            return Optional.ofNullable(clazz.cast(cpuCount()));
        case "disks":
            return Optional.ofNullable(clazz.cast(disks()));
        case "ramSizeInGb":
            return Optional.ofNullable(clazz.cast(ramSizeInGb()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<InstanceHardware, T> g) {
        return obj -> g.apply((InstanceHardware) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InstanceHardware> {
        /**
         * <p>
         * The number of vCPUs the instance has.
         * </p>
         * 
         * @param cpuCount
         *        The number of vCPUs the instance has.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cpuCount(Integer cpuCount);

        /**
         * <p>
         * The disks attached to the instance.
         * </p>
         * 
         * @param disks
         *        The disks attached to the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder disks(Collection<Disk> disks);

        /**
         * <p>
         * The disks attached to the instance.
         * </p>
         * 
         * @param disks
         *        The disks attached to the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder disks(Disk... disks);

        /**
         * <p>
         * The disks attached to the instance.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Disk>.Builder} avoiding the need to create
         * one manually via {@link List<Disk>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Disk>.Builder#build()} is called immediately and its result
         * is passed to {@link #disks(List<Disk>)}.
         * 
         * @param disks
         *        a consumer that will call methods on {@link List<Disk>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #disks(List<Disk>)
         */
        Builder disks(Consumer<Disk.Builder>... disks);

        /**
         * <p>
         * The amount of RAM in GB on the instance (e.g., <code>1.0</code>).
         * </p>
         * 
         * @param ramSizeInGb
         *        The amount of RAM in GB on the instance (e.g., <code>1.0</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ramSizeInGb(Float ramSizeInGb);
    }

    static final class BuilderImpl implements Builder {
        private Integer cpuCount;

        private List<Disk> disks = DefaultSdkAutoConstructList.getInstance();

        private Float ramSizeInGb;

        private BuilderImpl() {
        }

        private BuilderImpl(InstanceHardware model) {
            cpuCount(model.cpuCount);
            disks(model.disks);
            ramSizeInGb(model.ramSizeInGb);
        }

        public final Integer getCpuCount() {
            return cpuCount;
        }

        @Override
        public final Builder cpuCount(Integer cpuCount) {
            this.cpuCount = cpuCount;
            return this;
        }

        public final void setCpuCount(Integer cpuCount) {
            this.cpuCount = cpuCount;
        }

        public final Collection<Disk.Builder> getDisks() {
            return disks != null ? disks.stream().map(Disk::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder disks(Collection<Disk> disks) {
            this.disks = DiskListCopier.copy(disks);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder disks(Disk... disks) {
            disks(Arrays.asList(disks));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder disks(Consumer<Disk.Builder>... disks) {
            disks(Stream.of(disks).map(c -> Disk.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setDisks(Collection<Disk.BuilderImpl> disks) {
            this.disks = DiskListCopier.copyFromBuilder(disks);
        }

        public final Float getRamSizeInGb() {
            return ramSizeInGb;
        }

        @Override
        public final Builder ramSizeInGb(Float ramSizeInGb) {
            this.ramSizeInGb = ramSizeInGb;
            return this;
        }

        public final void setRamSizeInGb(Float ramSizeInGb) {
            this.ramSizeInGb = ramSizeInGb;
        }

        @Override
        public InstanceHardware build() {
            return new InstanceHardware(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
