/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lightsail.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.runtime.TypeConverter;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the Lightsail load balancer.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LoadBalancer implements SdkPojo, Serializable, ToCopyableBuilder<LoadBalancer.Builder, LoadBalancer> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(LoadBalancer::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(LoadBalancer::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<String> SUPPORT_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(LoadBalancer::supportCode)).setter(setter(Builder::supportCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("supportCode").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(LoadBalancer::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build()).build();

    private static final SdkField<ResourceLocation> LOCATION_FIELD = SdkField
            .<ResourceLocation> builder(MarshallingType.SDK_POJO).getter(getter(LoadBalancer::location))
            .setter(setter(Builder::location)).constructor(ResourceLocation::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("location").build()).build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(LoadBalancer::resourceTypeAsString)).setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceType").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .getter(getter(LoadBalancer::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> DNS_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(LoadBalancer::dnsName)).setter(setter(Builder::dnsName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dnsName").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(LoadBalancer::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("state").build()).build();

    private static final SdkField<String> PROTOCOL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(LoadBalancer::protocolAsString)).setter(setter(Builder::protocol))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("protocol").build()).build();

    private static final SdkField<List<Integer>> PUBLIC_PORTS_FIELD = SdkField
            .<List<Integer>> builder(MarshallingType.LIST)
            .getter(getter(LoadBalancer::publicPorts))
            .setter(setter(Builder::publicPorts))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("publicPorts").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Integer> builder(MarshallingType.INTEGER)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> HEALTH_CHECK_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(LoadBalancer::healthCheckPath)).setter(setter(Builder::healthCheckPath))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("healthCheckPath").build()).build();

    private static final SdkField<Integer> INSTANCE_PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(LoadBalancer::instancePort)).setter(setter(Builder::instancePort))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("instancePort").build()).build();

    private static final SdkField<List<InstanceHealthSummary>> INSTANCE_HEALTH_SUMMARY_FIELD = SdkField
            .<List<InstanceHealthSummary>> builder(MarshallingType.LIST)
            .getter(getter(LoadBalancer::instanceHealthSummary))
            .setter(setter(Builder::instanceHealthSummary))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("instanceHealthSummary").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<InstanceHealthSummary> builder(MarshallingType.SDK_POJO)
                                            .constructor(InstanceHealthSummary::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<LoadBalancerTlsCertificateSummary>> TLS_CERTIFICATE_SUMMARIES_FIELD = SdkField
            .<List<LoadBalancerTlsCertificateSummary>> builder(MarshallingType.LIST)
            .getter(getter(LoadBalancer::tlsCertificateSummaries))
            .setter(setter(Builder::tlsCertificateSummaries))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tlsCertificateSummaries").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<LoadBalancerTlsCertificateSummary> builder(MarshallingType.SDK_POJO)
                                            .constructor(LoadBalancerTlsCertificateSummary::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> CONFIGURATION_OPTIONS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(LoadBalancer::configurationOptionsAsStrings))
            .setter(setter(Builder::configurationOptionsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("configurationOptions").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, ARN_FIELD,
            SUPPORT_CODE_FIELD, CREATED_AT_FIELD, LOCATION_FIELD, RESOURCE_TYPE_FIELD, TAGS_FIELD, DNS_NAME_FIELD, STATE_FIELD,
            PROTOCOL_FIELD, PUBLIC_PORTS_FIELD, HEALTH_CHECK_PATH_FIELD, INSTANCE_PORT_FIELD, INSTANCE_HEALTH_SUMMARY_FIELD,
            TLS_CERTIFICATE_SUMMARIES_FIELD, CONFIGURATION_OPTIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String arn;

    private final String supportCode;

    private final Instant createdAt;

    private final ResourceLocation location;

    private final String resourceType;

    private final List<Tag> tags;

    private final String dnsName;

    private final String state;

    private final String protocol;

    private final List<Integer> publicPorts;

    private final String healthCheckPath;

    private final Integer instancePort;

    private final List<InstanceHealthSummary> instanceHealthSummary;

    private final List<LoadBalancerTlsCertificateSummary> tlsCertificateSummaries;

    private final Map<String, String> configurationOptions;

    private LoadBalancer(BuilderImpl builder) {
        this.name = builder.name;
        this.arn = builder.arn;
        this.supportCode = builder.supportCode;
        this.createdAt = builder.createdAt;
        this.location = builder.location;
        this.resourceType = builder.resourceType;
        this.tags = builder.tags;
        this.dnsName = builder.dnsName;
        this.state = builder.state;
        this.protocol = builder.protocol;
        this.publicPorts = builder.publicPorts;
        this.healthCheckPath = builder.healthCheckPath;
        this.instancePort = builder.instancePort;
        this.instanceHealthSummary = builder.instanceHealthSummary;
        this.tlsCertificateSummaries = builder.tlsCertificateSummaries;
        this.configurationOptions = builder.configurationOptions;
    }

    /**
     * <p>
     * The name of the load balancer (e.g., <code>my-load-balancer</code>).
     * </p>
     * 
     * @return The name of the load balancer (e.g., <code>my-load-balancer</code>).
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the load balancer.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the load balancer.
     */
    public String arn() {
        return arn;
    }

    /**
     * <p>
     * The support code. Include this code in your email to support when you have questions about your Lightsail load
     * balancer. This code enables our support team to look up your Lightsail information more easily.
     * </p>
     * 
     * @return The support code. Include this code in your email to support when you have questions about your Lightsail
     *         load balancer. This code enables our support team to look up your Lightsail information more easily.
     */
    public String supportCode() {
        return supportCode;
    }

    /**
     * <p>
     * The date when your load balancer was created.
     * </p>
     * 
     * @return The date when your load balancer was created.
     */
    public Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The AWS Region where your load balancer was created (e.g., <code>us-east-2a</code>). Lightsail automatically
     * creates your load balancer across Availability Zones.
     * </p>
     * 
     * @return The AWS Region where your load balancer was created (e.g., <code>us-east-2a</code>). Lightsail
     *         automatically creates your load balancer across Availability Zones.
     */
    public ResourceLocation location() {
        return location;
    }

    /**
     * <p>
     * The resource type (e.g., <code>LoadBalancer</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The resource type (e.g., <code>LoadBalancer</code>.
     * @see ResourceType
     */
    public ResourceType resourceType() {
        return ResourceType.fromValue(resourceType);
    }

    /**
     * <p>
     * The resource type (e.g., <code>LoadBalancer</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The resource type (e.g., <code>LoadBalancer</code>.
     * @see ResourceType
     */
    public String resourceTypeAsString() {
        return resourceType;
    }

    /**
     * <p>
     * The tag keys and optional values for the resource. For more information about tags in Lightsail, see the <a
     * href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev Guide</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The tag keys and optional values for the resource. For more information about tags in Lightsail, see the
     *         <a href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev
     *         Guide</a>.
     */
    public List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * The DNS name of your Lightsail load balancer.
     * </p>
     * 
     * @return The DNS name of your Lightsail load balancer.
     */
    public String dnsName() {
        return dnsName;
    }

    /**
     * <p>
     * The status of your load balancer. Valid values are below.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link LoadBalancerState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The status of your load balancer. Valid values are below.
     * @see LoadBalancerState
     */
    public LoadBalancerState state() {
        return LoadBalancerState.fromValue(state);
    }

    /**
     * <p>
     * The status of your load balancer. Valid values are below.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link LoadBalancerState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The status of your load balancer. Valid values are below.
     * @see LoadBalancerState
     */
    public String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The protocol you have enabled for your load balancer. Valid values are below.
     * </p>
     * <p>
     * You can't just have <code>HTTP_HTTPS</code>, but you can have just <code>HTTP</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #protocol} will
     * return {@link LoadBalancerProtocol#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #protocolAsString}.
     * </p>
     * 
     * @return The protocol you have enabled for your load balancer. Valid values are below.</p>
     *         <p>
     *         You can't just have <code>HTTP_HTTPS</code>, but you can have just <code>HTTP</code>.
     * @see LoadBalancerProtocol
     */
    public LoadBalancerProtocol protocol() {
        return LoadBalancerProtocol.fromValue(protocol);
    }

    /**
     * <p>
     * The protocol you have enabled for your load balancer. Valid values are below.
     * </p>
     * <p>
     * You can't just have <code>HTTP_HTTPS</code>, but you can have just <code>HTTP</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #protocol} will
     * return {@link LoadBalancerProtocol#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #protocolAsString}.
     * </p>
     * 
     * @return The protocol you have enabled for your load balancer. Valid values are below.</p>
     *         <p>
     *         You can't just have <code>HTTP_HTTPS</code>, but you can have just <code>HTTP</code>.
     * @see LoadBalancerProtocol
     */
    public String protocolAsString() {
        return protocol;
    }

    /**
     * <p>
     * An array of public port settings for your load balancer. For HTTP, use port 80. For HTTPS, use port 443.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return An array of public port settings for your load balancer. For HTTP, use port 80. For HTTPS, use port 443.
     */
    public List<Integer> publicPorts() {
        return publicPorts;
    }

    /**
     * <p>
     * The path you specified to perform your health checks. If no path is specified, the load balancer tries to make a
     * request to the default (root) page.
     * </p>
     * 
     * @return The path you specified to perform your health checks. If no path is specified, the load balancer tries to
     *         make a request to the default (root) page.
     */
    public String healthCheckPath() {
        return healthCheckPath;
    }

    /**
     * <p>
     * The port where the load balancer will direct traffic to your Lightsail instances. For HTTP traffic, it's port 80.
     * For HTTPS traffic, it's port 443.
     * </p>
     * 
     * @return The port where the load balancer will direct traffic to your Lightsail instances. For HTTP traffic, it's
     *         port 80. For HTTPS traffic, it's port 443.
     */
    public Integer instancePort() {
        return instancePort;
    }

    /**
     * <p>
     * An array of InstanceHealthSummary objects describing the health of the load balancer.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return An array of InstanceHealthSummary objects describing the health of the load balancer.
     */
    public List<InstanceHealthSummary> instanceHealthSummary() {
        return instanceHealthSummary;
    }

    /**
     * <p>
     * An array of LoadBalancerTlsCertificateSummary objects that provide additional information about the SSL/TLS
     * certificates. For example, if <code>true</code>, the certificate is attached to the load balancer.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return An array of LoadBalancerTlsCertificateSummary objects that provide additional information about the
     *         SSL/TLS certificates. For example, if <code>true</code>, the certificate is attached to the load
     *         balancer.
     */
    public List<LoadBalancerTlsCertificateSummary> tlsCertificateSummaries() {
        return tlsCertificateSummaries;
    }

    /**
     * <p>
     * A string to string map of the configuration options for your load balancer. Valid values are listed below.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A string to string map of the configuration options for your load balancer. Valid values are listed
     *         below.
     */
    public Map<LoadBalancerAttributeName, String> configurationOptions() {
        return TypeConverter.convert(configurationOptions, LoadBalancerAttributeName::fromValue, Function.identity(),
                (k, v) -> !Objects.equals(k, LoadBalancerAttributeName.UNKNOWN_TO_SDK_VERSION));
    }

    /**
     * <p>
     * A string to string map of the configuration options for your load balancer. Valid values are listed below.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A string to string map of the configuration options for your load balancer. Valid values are listed
     *         below.
     */
    public Map<String, String> configurationOptionsAsStrings() {
        return configurationOptions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(supportCode());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(location());
        hashCode = 31 * hashCode + Objects.hashCode(resourceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        hashCode = 31 * hashCode + Objects.hashCode(dnsName());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(protocolAsString());
        hashCode = 31 * hashCode + Objects.hashCode(publicPorts());
        hashCode = 31 * hashCode + Objects.hashCode(healthCheckPath());
        hashCode = 31 * hashCode + Objects.hashCode(instancePort());
        hashCode = 31 * hashCode + Objects.hashCode(instanceHealthSummary());
        hashCode = 31 * hashCode + Objects.hashCode(tlsCertificateSummaries());
        hashCode = 31 * hashCode + Objects.hashCode(configurationOptionsAsStrings());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LoadBalancer)) {
            return false;
        }
        LoadBalancer other = (LoadBalancer) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(arn(), other.arn())
                && Objects.equals(supportCode(), other.supportCode()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(location(), other.location())
                && Objects.equals(resourceTypeAsString(), other.resourceTypeAsString()) && Objects.equals(tags(), other.tags())
                && Objects.equals(dnsName(), other.dnsName()) && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(protocolAsString(), other.protocolAsString())
                && Objects.equals(publicPorts(), other.publicPorts())
                && Objects.equals(healthCheckPath(), other.healthCheckPath())
                && Objects.equals(instancePort(), other.instancePort())
                && Objects.equals(instanceHealthSummary(), other.instanceHealthSummary())
                && Objects.equals(tlsCertificateSummaries(), other.tlsCertificateSummaries())
                && Objects.equals(configurationOptionsAsStrings(), other.configurationOptionsAsStrings());
    }

    @Override
    public String toString() {
        return ToString.builder("LoadBalancer").add("Name", name()).add("Arn", arn()).add("SupportCode", supportCode())
                .add("CreatedAt", createdAt()).add("Location", location()).add("ResourceType", resourceTypeAsString())
                .add("Tags", tags()).add("DnsName", dnsName()).add("State", stateAsString()).add("Protocol", protocolAsString())
                .add("PublicPorts", publicPorts()).add("HealthCheckPath", healthCheckPath()).add("InstancePort", instancePort())
                .add("InstanceHealthSummary", instanceHealthSummary()).add("TlsCertificateSummaries", tlsCertificateSummaries())
                .add("ConfigurationOptions", configurationOptionsAsStrings()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "supportCode":
            return Optional.ofNullable(clazz.cast(supportCode()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "location":
            return Optional.ofNullable(clazz.cast(location()));
        case "resourceType":
            return Optional.ofNullable(clazz.cast(resourceTypeAsString()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "dnsName":
            return Optional.ofNullable(clazz.cast(dnsName()));
        case "state":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "protocol":
            return Optional.ofNullable(clazz.cast(protocolAsString()));
        case "publicPorts":
            return Optional.ofNullable(clazz.cast(publicPorts()));
        case "healthCheckPath":
            return Optional.ofNullable(clazz.cast(healthCheckPath()));
        case "instancePort":
            return Optional.ofNullable(clazz.cast(instancePort()));
        case "instanceHealthSummary":
            return Optional.ofNullable(clazz.cast(instanceHealthSummary()));
        case "tlsCertificateSummaries":
            return Optional.ofNullable(clazz.cast(tlsCertificateSummaries()));
        case "configurationOptions":
            return Optional.ofNullable(clazz.cast(configurationOptionsAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<LoadBalancer, T> g) {
        return obj -> g.apply((LoadBalancer) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LoadBalancer> {
        /**
         * <p>
         * The name of the load balancer (e.g., <code>my-load-balancer</code>).
         * </p>
         * 
         * @param name
         *        The name of the load balancer (e.g., <code>my-load-balancer</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the load balancer.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the load balancer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The support code. Include this code in your email to support when you have questions about your Lightsail
         * load balancer. This code enables our support team to look up your Lightsail information more easily.
         * </p>
         * 
         * @param supportCode
         *        The support code. Include this code in your email to support when you have questions about your
         *        Lightsail load balancer. This code enables our support team to look up your Lightsail information more
         *        easily.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportCode(String supportCode);

        /**
         * <p>
         * The date when your load balancer was created.
         * </p>
         * 
         * @param createdAt
         *        The date when your load balancer was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The AWS Region where your load balancer was created (e.g., <code>us-east-2a</code>). Lightsail automatically
         * creates your load balancer across Availability Zones.
         * </p>
         * 
         * @param location
         *        The AWS Region where your load balancer was created (e.g., <code>us-east-2a</code>). Lightsail
         *        automatically creates your load balancer across Availability Zones.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder location(ResourceLocation location);

        /**
         * <p>
         * The AWS Region where your load balancer was created (e.g., <code>us-east-2a</code>). Lightsail automatically
         * creates your load balancer across Availability Zones.
         * </p>
         * This is a convenience that creates an instance of the {@link ResourceLocation.Builder} avoiding the need to
         * create one manually via {@link ResourceLocation#builder()}.
         *
         * When the {@link Consumer} completes, {@link ResourceLocation.Builder#build()} is called immediately and its
         * result is passed to {@link #location(ResourceLocation)}.
         * 
         * @param location
         *        a consumer that will call methods on {@link ResourceLocation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #location(ResourceLocation)
         */
        default Builder location(Consumer<ResourceLocation.Builder> location) {
            return location(ResourceLocation.builder().applyMutation(location).build());
        }

        /**
         * <p>
         * The resource type (e.g., <code>LoadBalancer</code>.
         * </p>
         * 
         * @param resourceType
         *        The resource type (e.g., <code>LoadBalancer</code>.
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * The resource type (e.g., <code>LoadBalancer</code>.
         * </p>
         * 
         * @param resourceType
         *        The resource type (e.g., <code>LoadBalancer</code>.
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder resourceType(ResourceType resourceType);

        /**
         * <p>
         * The tag keys and optional values for the resource. For more information about tags in Lightsail, see the <a
         * href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev Guide</a>.
         * </p>
         * 
         * @param tags
         *        The tag keys and optional values for the resource. For more information about tags in Lightsail, see
         *        the <a href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev
         *        Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The tag keys and optional values for the resource. For more information about tags in Lightsail, see the <a
         * href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev Guide</a>.
         * </p>
         * 
         * @param tags
         *        The tag keys and optional values for the resource. For more information about tags in Lightsail, see
         *        the <a href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev
         *        Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The tag keys and optional values for the resource. For more information about tags in Lightsail, see the <a
         * href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev Guide</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * The DNS name of your Lightsail load balancer.
         * </p>
         * 
         * @param dnsName
         *        The DNS name of your Lightsail load balancer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsName(String dnsName);

        /**
         * <p>
         * The status of your load balancer. Valid values are below.
         * </p>
         * 
         * @param state
         *        The status of your load balancer. Valid values are below.
         * @see LoadBalancerState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LoadBalancerState
         */
        Builder state(String state);

        /**
         * <p>
         * The status of your load balancer. Valid values are below.
         * </p>
         * 
         * @param state
         *        The status of your load balancer. Valid values are below.
         * @see LoadBalancerState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LoadBalancerState
         */
        Builder state(LoadBalancerState state);

        /**
         * <p>
         * The protocol you have enabled for your load balancer. Valid values are below.
         * </p>
         * <p>
         * You can't just have <code>HTTP_HTTPS</code>, but you can have just <code>HTTP</code>.
         * </p>
         * 
         * @param protocol
         *        The protocol you have enabled for your load balancer. Valid values are below.</p>
         *        <p>
         *        You can't just have <code>HTTP_HTTPS</code>, but you can have just <code>HTTP</code>.
         * @see LoadBalancerProtocol
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LoadBalancerProtocol
         */
        Builder protocol(String protocol);

        /**
         * <p>
         * The protocol you have enabled for your load balancer. Valid values are below.
         * </p>
         * <p>
         * You can't just have <code>HTTP_HTTPS</code>, but you can have just <code>HTTP</code>.
         * </p>
         * 
         * @param protocol
         *        The protocol you have enabled for your load balancer. Valid values are below.</p>
         *        <p>
         *        You can't just have <code>HTTP_HTTPS</code>, but you can have just <code>HTTP</code>.
         * @see LoadBalancerProtocol
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LoadBalancerProtocol
         */
        Builder protocol(LoadBalancerProtocol protocol);

        /**
         * <p>
         * An array of public port settings for your load balancer. For HTTP, use port 80. For HTTPS, use port 443.
         * </p>
         * 
         * @param publicPorts
         *        An array of public port settings for your load balancer. For HTTP, use port 80. For HTTPS, use port
         *        443.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder publicPorts(Collection<Integer> publicPorts);

        /**
         * <p>
         * An array of public port settings for your load balancer. For HTTP, use port 80. For HTTPS, use port 443.
         * </p>
         * 
         * @param publicPorts
         *        An array of public port settings for your load balancer. For HTTP, use port 80. For HTTPS, use port
         *        443.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder publicPorts(Integer... publicPorts);

        /**
         * <p>
         * The path you specified to perform your health checks. If no path is specified, the load balancer tries to
         * make a request to the default (root) page.
         * </p>
         * 
         * @param healthCheckPath
         *        The path you specified to perform your health checks. If no path is specified, the load balancer tries
         *        to make a request to the default (root) page.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder healthCheckPath(String healthCheckPath);

        /**
         * <p>
         * The port where the load balancer will direct traffic to your Lightsail instances. For HTTP traffic, it's port
         * 80. For HTTPS traffic, it's port 443.
         * </p>
         * 
         * @param instancePort
         *        The port where the load balancer will direct traffic to your Lightsail instances. For HTTP traffic,
         *        it's port 80. For HTTPS traffic, it's port 443.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instancePort(Integer instancePort);

        /**
         * <p>
         * An array of InstanceHealthSummary objects describing the health of the load balancer.
         * </p>
         * 
         * @param instanceHealthSummary
         *        An array of InstanceHealthSummary objects describing the health of the load balancer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceHealthSummary(Collection<InstanceHealthSummary> instanceHealthSummary);

        /**
         * <p>
         * An array of InstanceHealthSummary objects describing the health of the load balancer.
         * </p>
         * 
         * @param instanceHealthSummary
         *        An array of InstanceHealthSummary objects describing the health of the load balancer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceHealthSummary(InstanceHealthSummary... instanceHealthSummary);

        /**
         * <p>
         * An array of InstanceHealthSummary objects describing the health of the load balancer.
         * </p>
         * This is a convenience that creates an instance of the {@link List<InstanceHealthSummary>.Builder} avoiding
         * the need to create one manually via {@link List<InstanceHealthSummary>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<InstanceHealthSummary>.Builder#build()} is called
         * immediately and its result is passed to {@link #instanceHealthSummary(List<InstanceHealthSummary>)}.
         * 
         * @param instanceHealthSummary
         *        a consumer that will call methods on {@link List<InstanceHealthSummary>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #instanceHealthSummary(List<InstanceHealthSummary>)
         */
        Builder instanceHealthSummary(Consumer<InstanceHealthSummary.Builder>... instanceHealthSummary);

        /**
         * <p>
         * An array of LoadBalancerTlsCertificateSummary objects that provide additional information about the SSL/TLS
         * certificates. For example, if <code>true</code>, the certificate is attached to the load balancer.
         * </p>
         * 
         * @param tlsCertificateSummaries
         *        An array of LoadBalancerTlsCertificateSummary objects that provide additional information about the
         *        SSL/TLS certificates. For example, if <code>true</code>, the certificate is attached to the load
         *        balancer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tlsCertificateSummaries(Collection<LoadBalancerTlsCertificateSummary> tlsCertificateSummaries);

        /**
         * <p>
         * An array of LoadBalancerTlsCertificateSummary objects that provide additional information about the SSL/TLS
         * certificates. For example, if <code>true</code>, the certificate is attached to the load balancer.
         * </p>
         * 
         * @param tlsCertificateSummaries
         *        An array of LoadBalancerTlsCertificateSummary objects that provide additional information about the
         *        SSL/TLS certificates. For example, if <code>true</code>, the certificate is attached to the load
         *        balancer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tlsCertificateSummaries(LoadBalancerTlsCertificateSummary... tlsCertificateSummaries);

        /**
         * <p>
         * An array of LoadBalancerTlsCertificateSummary objects that provide additional information about the SSL/TLS
         * certificates. For example, if <code>true</code>, the certificate is attached to the load balancer.
         * </p>
         * This is a convenience that creates an instance of the {@link List<LoadBalancerTlsCertificateSummary>.Builder}
         * avoiding the need to create one manually via {@link List<LoadBalancerTlsCertificateSummary>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<LoadBalancerTlsCertificateSummary>.Builder#build()} is
         * called immediately and its result is passed to {@link
         * #tlsCertificateSummaries(List<LoadBalancerTlsCertificateSummary>)}.
         * 
         * @param tlsCertificateSummaries
         *        a consumer that will call methods on {@link List<LoadBalancerTlsCertificateSummary>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tlsCertificateSummaries(List<LoadBalancerTlsCertificateSummary>)
         */
        Builder tlsCertificateSummaries(Consumer<LoadBalancerTlsCertificateSummary.Builder>... tlsCertificateSummaries);

        /**
         * <p>
         * A string to string map of the configuration options for your load balancer. Valid values are listed below.
         * </p>
         * 
         * @param configurationOptions
         *        A string to string map of the configuration options for your load balancer. Valid values are listed
         *        below.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurationOptionsWithStrings(Map<String, String> configurationOptions);

        /**
         * <p>
         * A string to string map of the configuration options for your load balancer. Valid values are listed below.
         * </p>
         * 
         * @param configurationOptions
         *        A string to string map of the configuration options for your load balancer. Valid values are listed
         *        below.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurationOptions(Map<LoadBalancerAttributeName, String> configurationOptions);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String arn;

        private String supportCode;

        private Instant createdAt;

        private ResourceLocation location;

        private String resourceType;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String dnsName;

        private String state;

        private String protocol;

        private List<Integer> publicPorts = DefaultSdkAutoConstructList.getInstance();

        private String healthCheckPath;

        private Integer instancePort;

        private List<InstanceHealthSummary> instanceHealthSummary = DefaultSdkAutoConstructList.getInstance();

        private List<LoadBalancerTlsCertificateSummary> tlsCertificateSummaries = DefaultSdkAutoConstructList.getInstance();

        private Map<String, String> configurationOptions = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(LoadBalancer model) {
            name(model.name);
            arn(model.arn);
            supportCode(model.supportCode);
            createdAt(model.createdAt);
            location(model.location);
            resourceType(model.resourceType);
            tags(model.tags);
            dnsName(model.dnsName);
            state(model.state);
            protocol(model.protocol);
            publicPorts(model.publicPorts);
            healthCheckPath(model.healthCheckPath);
            instancePort(model.instancePort);
            instanceHealthSummary(model.instanceHealthSummary);
            tlsCertificateSummaries(model.tlsCertificateSummaries);
            configurationOptionsWithStrings(model.configurationOptions);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getArn() {
            return arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        public final String getSupportCode() {
            return supportCode;
        }

        @Override
        public final Builder supportCode(String supportCode) {
            this.supportCode = supportCode;
            return this;
        }

        public final void setSupportCode(String supportCode) {
            this.supportCode = supportCode;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        public final ResourceLocation.Builder getLocation() {
            return location != null ? location.toBuilder() : null;
        }

        @Override
        public final Builder location(ResourceLocation location) {
            this.location = location;
            return this;
        }

        public final void setLocation(ResourceLocation.BuilderImpl location) {
            this.location = location != null ? location.build() : null;
        }

        public final String getResourceTypeAsString() {
            return resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        @Override
        public final Builder resourceType(ResourceType resourceType) {
            this.resourceType(resourceType.toString());
            return this;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        public final Collection<Tag.Builder> getTags() {
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        public final String getDnsName() {
            return dnsName;
        }

        @Override
        public final Builder dnsName(String dnsName) {
            this.dnsName = dnsName;
            return this;
        }

        public final void setDnsName(String dnsName) {
            this.dnsName = dnsName;
        }

        public final String getStateAsString() {
            return state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(LoadBalancerState state) {
            this.state(state.toString());
            return this;
        }

        public final void setState(String state) {
            this.state = state;
        }

        public final String getProtocolAsString() {
            return protocol;
        }

        @Override
        public final Builder protocol(String protocol) {
            this.protocol = protocol;
            return this;
        }

        @Override
        public final Builder protocol(LoadBalancerProtocol protocol) {
            this.protocol(protocol.toString());
            return this;
        }

        public final void setProtocol(String protocol) {
            this.protocol = protocol;
        }

        public final Collection<Integer> getPublicPorts() {
            return publicPorts;
        }

        @Override
        public final Builder publicPorts(Collection<Integer> publicPorts) {
            this.publicPorts = PortListCopier.copy(publicPorts);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder publicPorts(Integer... publicPorts) {
            publicPorts(Arrays.asList(publicPorts));
            return this;
        }

        public final void setPublicPorts(Collection<Integer> publicPorts) {
            this.publicPorts = PortListCopier.copy(publicPorts);
        }

        public final String getHealthCheckPath() {
            return healthCheckPath;
        }

        @Override
        public final Builder healthCheckPath(String healthCheckPath) {
            this.healthCheckPath = healthCheckPath;
            return this;
        }

        public final void setHealthCheckPath(String healthCheckPath) {
            this.healthCheckPath = healthCheckPath;
        }

        public final Integer getInstancePort() {
            return instancePort;
        }

        @Override
        public final Builder instancePort(Integer instancePort) {
            this.instancePort = instancePort;
            return this;
        }

        public final void setInstancePort(Integer instancePort) {
            this.instancePort = instancePort;
        }

        public final Collection<InstanceHealthSummary.Builder> getInstanceHealthSummary() {
            return instanceHealthSummary != null ? instanceHealthSummary.stream().map(InstanceHealthSummary::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder instanceHealthSummary(Collection<InstanceHealthSummary> instanceHealthSummary) {
            this.instanceHealthSummary = InstanceHealthSummaryListCopier.copy(instanceHealthSummary);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceHealthSummary(InstanceHealthSummary... instanceHealthSummary) {
            instanceHealthSummary(Arrays.asList(instanceHealthSummary));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceHealthSummary(Consumer<InstanceHealthSummary.Builder>... instanceHealthSummary) {
            instanceHealthSummary(Stream.of(instanceHealthSummary)
                    .map(c -> InstanceHealthSummary.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setInstanceHealthSummary(Collection<InstanceHealthSummary.BuilderImpl> instanceHealthSummary) {
            this.instanceHealthSummary = InstanceHealthSummaryListCopier.copyFromBuilder(instanceHealthSummary);
        }

        public final Collection<LoadBalancerTlsCertificateSummary.Builder> getTlsCertificateSummaries() {
            return tlsCertificateSummaries != null ? tlsCertificateSummaries.stream()
                    .map(LoadBalancerTlsCertificateSummary::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tlsCertificateSummaries(Collection<LoadBalancerTlsCertificateSummary> tlsCertificateSummaries) {
            this.tlsCertificateSummaries = LoadBalancerTlsCertificateSummaryListCopier.copy(tlsCertificateSummaries);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tlsCertificateSummaries(LoadBalancerTlsCertificateSummary... tlsCertificateSummaries) {
            tlsCertificateSummaries(Arrays.asList(tlsCertificateSummaries));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tlsCertificateSummaries(
                Consumer<LoadBalancerTlsCertificateSummary.Builder>... tlsCertificateSummaries) {
            tlsCertificateSummaries(Stream.of(tlsCertificateSummaries)
                    .map(c -> LoadBalancerTlsCertificateSummary.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTlsCertificateSummaries(
                Collection<LoadBalancerTlsCertificateSummary.BuilderImpl> tlsCertificateSummaries) {
            this.tlsCertificateSummaries = LoadBalancerTlsCertificateSummaryListCopier.copyFromBuilder(tlsCertificateSummaries);
        }

        public final Map<String, String> getConfigurationOptionsAsStrings() {
            return configurationOptions;
        }

        @Override
        public final Builder configurationOptionsWithStrings(Map<String, String> configurationOptions) {
            this.configurationOptions = LoadBalancerConfigurationOptionsCopier.copy(configurationOptions);
            return this;
        }

        @Override
        public final Builder configurationOptions(Map<LoadBalancerAttributeName, String> configurationOptions) {
            this.configurationOptions = LoadBalancerConfigurationOptionsCopier.copyEnumToString(configurationOptions);
            return this;
        }

        public final void setConfigurationOptionsWithStrings(Map<String, String> configurationOptions) {
            this.configurationOptions = LoadBalancerConfigurationOptionsCopier.copy(configurationOptions);
        }

        @Override
        public LoadBalancer build() {
            return new LoadBalancer(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
