/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lightsail.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the synchronization status of the Amazon Simple Storage Service (Amazon S3) account-level block public
 * access (BPA) feature for your Lightsail buckets.
 * </p>
 * <p>
 * The account-level BPA feature of Amazon S3 provides centralized controls to limit public access to all Amazon S3
 * buckets in an account. BPA can make all Amazon S3 buckets in an Amazon Web Services account private regardless of the
 * individual bucket and object permissions that are configured. Lightsail buckets take into account the Amazon S3
 * account-level BPA configuration when allowing or denying public access. To do this, Lightsail periodically fetches
 * the account-level BPA configuration from Amazon S3. When the account-level BPA status is <code>InSync</code>, the
 * Amazon S3 account-level BPA configuration is synchronized and it applies to your Lightsail buckets. For more
 * information about Amazon Simple Storage Service account-level BPA and how it affects Lightsail buckets, see <a
 * href="https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-block-public-access-for-buckets">Block
 * public access for buckets in Amazon Lightsail</a> in the <i>Amazon Lightsail Developer Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AccountLevelBpaSync implements SdkPojo, Serializable,
        ToCopyableBuilder<AccountLevelBpaSync.Builder, AccountLevelBpaSync> {
    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(AccountLevelBpaSync::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<Instant> LAST_SYNCED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastSyncedAt").getter(getter(AccountLevelBpaSync::lastSyncedAt)).setter(setter(Builder::lastSyncedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastSyncedAt").build()).build();

    private static final SdkField<String> MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("message")
            .getter(getter(AccountLevelBpaSync::messageAsString)).setter(setter(Builder::message))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("message").build()).build();

    private static final SdkField<Boolean> BPA_IMPACTS_LIGHTSAIL_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("bpaImpactsLightsail").getter(getter(AccountLevelBpaSync::bpaImpactsLightsail))
            .setter(setter(Builder::bpaImpactsLightsail))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("bpaImpactsLightsail").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STATUS_FIELD,
            LAST_SYNCED_AT_FIELD, MESSAGE_FIELD, BPA_IMPACTS_LIGHTSAIL_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String status;

    private final Instant lastSyncedAt;

    private final String message;

    private final Boolean bpaImpactsLightsail;

    private AccountLevelBpaSync(BuilderImpl builder) {
        this.status = builder.status;
        this.lastSyncedAt = builder.lastSyncedAt;
        this.message = builder.message;
        this.bpaImpactsLightsail = builder.bpaImpactsLightsail;
    }

    /**
     * <p>
     * The status of the account-level BPA synchronization.
     * </p>
     * <p>
     * The following statuses are possible:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>InSync</code> - Account-level BPA is synchronized. The Amazon S3 account-level BPA configuration applies to
     * your Lightsail buckets.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NeverSynced</code> - Synchronization has not yet happened. The Amazon S3 account-level BPA configuration
     * does not apply to your Lightsail buckets.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Failed</code> - Synchronization failed. The Amazon S3 account-level BPA configuration does not apply to
     * your Lightsail buckets.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Defaulted</code> - Synchronization failed and account-level BPA for your Lightsail buckets is defaulted to
     * <i>active</i>.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * You might need to complete further actions if the status is <code>Failed</code> or <code>Defaulted</code>. The
     * <code>message</code> parameter provides more information for those statuses.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link AccountLevelBpaSyncStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the account-level BPA synchronization.</p>
     *         <p>
     *         The following statuses are possible:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>InSync</code> - Account-level BPA is synchronized. The Amazon S3 account-level BPA configuration
     *         applies to your Lightsail buckets.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NeverSynced</code> - Synchronization has not yet happened. The Amazon S3 account-level BPA
     *         configuration does not apply to your Lightsail buckets.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Failed</code> - Synchronization failed. The Amazon S3 account-level BPA configuration does not
     *         apply to your Lightsail buckets.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Defaulted</code> - Synchronization failed and account-level BPA for your Lightsail buckets is
     *         defaulted to <i>active</i>.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         You might need to complete further actions if the status is <code>Failed</code> or <code>Defaulted</code>
     *         . The <code>message</code> parameter provides more information for those statuses.
     *         </p>
     * @see AccountLevelBpaSyncStatus
     */
    public final AccountLevelBpaSyncStatus status() {
        return AccountLevelBpaSyncStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the account-level BPA synchronization.
     * </p>
     * <p>
     * The following statuses are possible:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>InSync</code> - Account-level BPA is synchronized. The Amazon S3 account-level BPA configuration applies to
     * your Lightsail buckets.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NeverSynced</code> - Synchronization has not yet happened. The Amazon S3 account-level BPA configuration
     * does not apply to your Lightsail buckets.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Failed</code> - Synchronization failed. The Amazon S3 account-level BPA configuration does not apply to
     * your Lightsail buckets.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Defaulted</code> - Synchronization failed and account-level BPA for your Lightsail buckets is defaulted to
     * <i>active</i>.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * You might need to complete further actions if the status is <code>Failed</code> or <code>Defaulted</code>. The
     * <code>message</code> parameter provides more information for those statuses.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link AccountLevelBpaSyncStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the account-level BPA synchronization.</p>
     *         <p>
     *         The following statuses are possible:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>InSync</code> - Account-level BPA is synchronized. The Amazon S3 account-level BPA configuration
     *         applies to your Lightsail buckets.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NeverSynced</code> - Synchronization has not yet happened. The Amazon S3 account-level BPA
     *         configuration does not apply to your Lightsail buckets.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Failed</code> - Synchronization failed. The Amazon S3 account-level BPA configuration does not
     *         apply to your Lightsail buckets.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Defaulted</code> - Synchronization failed and account-level BPA for your Lightsail buckets is
     *         defaulted to <i>active</i>.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         You might need to complete further actions if the status is <code>Failed</code> or <code>Defaulted</code>
     *         . The <code>message</code> parameter provides more information for those statuses.
     *         </p>
     * @see AccountLevelBpaSyncStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The timestamp of when the account-level BPA configuration was last synchronized. This value is null when the
     * account-level BPA configuration has not been synchronized.
     * </p>
     * 
     * @return The timestamp of when the account-level BPA configuration was last synchronized. This value is null when
     *         the account-level BPA configuration has not been synchronized.
     */
    public final Instant lastSyncedAt() {
        return lastSyncedAt;
    }

    /**
     * <p>
     * A message that provides a reason for a <code>Failed</code> or <code>Defaulted</code> synchronization status.
     * </p>
     * <p>
     * The following messages are possible:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>SYNC_ON_HOLD</code> - The synchronization has not yet happened. This status message occurs immediately
     * after you create your first Lightsail bucket. This status message should change after the first synchronization
     * happens, approximately 1 hour after the first bucket is created.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DEFAULTED_FOR_SLR_MISSING</code> - The synchronization failed because the required service-linked role is
     * missing from your Amazon Web Services account. The account-level BPA configuration for your Lightsail buckets is
     * defaulted to <i>active</i> until the synchronization can occur. This means that all your buckets are private and
     * not publicly accessible. For more information about how to create the required service-linked role to allow
     * synchronization, see <a
     * href="https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-using-service-linked-roles">Using
     * Service-Linked Roles for Amazon Lightsail</a> in the <i>Amazon Lightsail Developer Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DEFAULTED_FOR_SLR_MISSING_ON_HOLD</code> - The synchronization failed because the required service-linked
     * role is missing from your Amazon Web Services account. Account-level BPA is not yet configured for your Lightsail
     * buckets. Therefore, only the bucket access permissions and individual object access permissions apply to your
     * Lightsail buckets. For more information about how to create the required service-linked role to allow
     * synchronization, see <a
     * href="https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-using-service-linked-roles">Using
     * Service-Linked Roles for Amazon Lightsail</a> in the <i>Amazon Lightsail Developer Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Unknown</code> - The reason that synchronization failed is unknown. Contact Amazon Web Services Support for
     * more information.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #message} will
     * return {@link BPAStatusMessage#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #messageAsString}.
     * </p>
     * 
     * @return A message that provides a reason for a <code>Failed</code> or <code>Defaulted</code> synchronization
     *         status.</p>
     *         <p>
     *         The following messages are possible:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>SYNC_ON_HOLD</code> - The synchronization has not yet happened. This status message occurs
     *         immediately after you create your first Lightsail bucket. This status message should change after the
     *         first synchronization happens, approximately 1 hour after the first bucket is created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DEFAULTED_FOR_SLR_MISSING</code> - The synchronization failed because the required service-linked
     *         role is missing from your Amazon Web Services account. The account-level BPA configuration for your
     *         Lightsail buckets is defaulted to <i>active</i> until the synchronization can occur. This means that all
     *         your buckets are private and not publicly accessible. For more information about how to create the
     *         required service-linked role to allow synchronization, see <a href=
     *         "https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-using-service-linked-roles"
     *         >Using Service-Linked Roles for Amazon Lightsail</a> in the <i>Amazon Lightsail Developer Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DEFAULTED_FOR_SLR_MISSING_ON_HOLD</code> - The synchronization failed because the required
     *         service-linked role is missing from your Amazon Web Services account. Account-level BPA is not yet
     *         configured for your Lightsail buckets. Therefore, only the bucket access permissions and individual
     *         object access permissions apply to your Lightsail buckets. For more information about how to create the
     *         required service-linked role to allow synchronization, see <a href=
     *         "https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-using-service-linked-roles"
     *         >Using Service-Linked Roles for Amazon Lightsail</a> in the <i>Amazon Lightsail Developer Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Unknown</code> - The reason that synchronization failed is unknown. Contact Amazon Web Services
     *         Support for more information.
     *         </p>
     *         </li>
     * @see BPAStatusMessage
     */
    public final BPAStatusMessage message() {
        return BPAStatusMessage.fromValue(message);
    }

    /**
     * <p>
     * A message that provides a reason for a <code>Failed</code> or <code>Defaulted</code> synchronization status.
     * </p>
     * <p>
     * The following messages are possible:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>SYNC_ON_HOLD</code> - The synchronization has not yet happened. This status message occurs immediately
     * after you create your first Lightsail bucket. This status message should change after the first synchronization
     * happens, approximately 1 hour after the first bucket is created.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DEFAULTED_FOR_SLR_MISSING</code> - The synchronization failed because the required service-linked role is
     * missing from your Amazon Web Services account. The account-level BPA configuration for your Lightsail buckets is
     * defaulted to <i>active</i> until the synchronization can occur. This means that all your buckets are private and
     * not publicly accessible. For more information about how to create the required service-linked role to allow
     * synchronization, see <a
     * href="https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-using-service-linked-roles">Using
     * Service-Linked Roles for Amazon Lightsail</a> in the <i>Amazon Lightsail Developer Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DEFAULTED_FOR_SLR_MISSING_ON_HOLD</code> - The synchronization failed because the required service-linked
     * role is missing from your Amazon Web Services account. Account-level BPA is not yet configured for your Lightsail
     * buckets. Therefore, only the bucket access permissions and individual object access permissions apply to your
     * Lightsail buckets. For more information about how to create the required service-linked role to allow
     * synchronization, see <a
     * href="https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-using-service-linked-roles">Using
     * Service-Linked Roles for Amazon Lightsail</a> in the <i>Amazon Lightsail Developer Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Unknown</code> - The reason that synchronization failed is unknown. Contact Amazon Web Services Support for
     * more information.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #message} will
     * return {@link BPAStatusMessage#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #messageAsString}.
     * </p>
     * 
     * @return A message that provides a reason for a <code>Failed</code> or <code>Defaulted</code> synchronization
     *         status.</p>
     *         <p>
     *         The following messages are possible:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>SYNC_ON_HOLD</code> - The synchronization has not yet happened. This status message occurs
     *         immediately after you create your first Lightsail bucket. This status message should change after the
     *         first synchronization happens, approximately 1 hour after the first bucket is created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DEFAULTED_FOR_SLR_MISSING</code> - The synchronization failed because the required service-linked
     *         role is missing from your Amazon Web Services account. The account-level BPA configuration for your
     *         Lightsail buckets is defaulted to <i>active</i> until the synchronization can occur. This means that all
     *         your buckets are private and not publicly accessible. For more information about how to create the
     *         required service-linked role to allow synchronization, see <a href=
     *         "https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-using-service-linked-roles"
     *         >Using Service-Linked Roles for Amazon Lightsail</a> in the <i>Amazon Lightsail Developer Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DEFAULTED_FOR_SLR_MISSING_ON_HOLD</code> - The synchronization failed because the required
     *         service-linked role is missing from your Amazon Web Services account. Account-level BPA is not yet
     *         configured for your Lightsail buckets. Therefore, only the bucket access permissions and individual
     *         object access permissions apply to your Lightsail buckets. For more information about how to create the
     *         required service-linked role to allow synchronization, see <a href=
     *         "https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-using-service-linked-roles"
     *         >Using Service-Linked Roles for Amazon Lightsail</a> in the <i>Amazon Lightsail Developer Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Unknown</code> - The reason that synchronization failed is unknown. Contact Amazon Web Services
     *         Support for more information.
     *         </p>
     *         </li>
     * @see BPAStatusMessage
     */
    public final String messageAsString() {
        return message;
    }

    /**
     * <p>
     * A Boolean value that indicates whether account-level block public access is affecting your Lightsail buckets.
     * </p>
     * 
     * @return A Boolean value that indicates whether account-level block public access is affecting your Lightsail
     *         buckets.
     */
    public final Boolean bpaImpactsLightsail() {
        return bpaImpactsLightsail;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(lastSyncedAt());
        hashCode = 31 * hashCode + Objects.hashCode(messageAsString());
        hashCode = 31 * hashCode + Objects.hashCode(bpaImpactsLightsail());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AccountLevelBpaSync)) {
            return false;
        }
        AccountLevelBpaSync other = (AccountLevelBpaSync) obj;
        return Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(lastSyncedAt(), other.lastSyncedAt())
                && Objects.equals(messageAsString(), other.messageAsString())
                && Objects.equals(bpaImpactsLightsail(), other.bpaImpactsLightsail());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AccountLevelBpaSync").add("Status", statusAsString()).add("LastSyncedAt", lastSyncedAt())
                .add("Message", messageAsString()).add("BpaImpactsLightsail", bpaImpactsLightsail()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "lastSyncedAt":
            return Optional.ofNullable(clazz.cast(lastSyncedAt()));
        case "message":
            return Optional.ofNullable(clazz.cast(messageAsString()));
        case "bpaImpactsLightsail":
            return Optional.ofNullable(clazz.cast(bpaImpactsLightsail()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("status", STATUS_FIELD);
        map.put("lastSyncedAt", LAST_SYNCED_AT_FIELD);
        map.put("message", MESSAGE_FIELD);
        map.put("bpaImpactsLightsail", BPA_IMPACTS_LIGHTSAIL_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<AccountLevelBpaSync, T> g) {
        return obj -> g.apply((AccountLevelBpaSync) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AccountLevelBpaSync> {
        /**
         * <p>
         * The status of the account-level BPA synchronization.
         * </p>
         * <p>
         * The following statuses are possible:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>InSync</code> - Account-level BPA is synchronized. The Amazon S3 account-level BPA configuration
         * applies to your Lightsail buckets.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NeverSynced</code> - Synchronization has not yet happened. The Amazon S3 account-level BPA
         * configuration does not apply to your Lightsail buckets.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Failed</code> - Synchronization failed. The Amazon S3 account-level BPA configuration does not apply to
         * your Lightsail buckets.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Defaulted</code> - Synchronization failed and account-level BPA for your Lightsail buckets is defaulted
         * to <i>active</i>.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * You might need to complete further actions if the status is <code>Failed</code> or <code>Defaulted</code>.
         * The <code>message</code> parameter provides more information for those statuses.
         * </p>
         * </note>
         * 
         * @param status
         *        The status of the account-level BPA synchronization.</p>
         *        <p>
         *        The following statuses are possible:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>InSync</code> - Account-level BPA is synchronized. The Amazon S3 account-level BPA configuration
         *        applies to your Lightsail buckets.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NeverSynced</code> - Synchronization has not yet happened. The Amazon S3 account-level BPA
         *        configuration does not apply to your Lightsail buckets.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Failed</code> - Synchronization failed. The Amazon S3 account-level BPA configuration does not
         *        apply to your Lightsail buckets.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Defaulted</code> - Synchronization failed and account-level BPA for your Lightsail buckets is
         *        defaulted to <i>active</i>.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        You might need to complete further actions if the status is <code>Failed</code> or
         *        <code>Defaulted</code>. The <code>message</code> parameter provides more information for those
         *        statuses.
         *        </p>
         * @see AccountLevelBpaSyncStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AccountLevelBpaSyncStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the account-level BPA synchronization.
         * </p>
         * <p>
         * The following statuses are possible:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>InSync</code> - Account-level BPA is synchronized. The Amazon S3 account-level BPA configuration
         * applies to your Lightsail buckets.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NeverSynced</code> - Synchronization has not yet happened. The Amazon S3 account-level BPA
         * configuration does not apply to your Lightsail buckets.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Failed</code> - Synchronization failed. The Amazon S3 account-level BPA configuration does not apply to
         * your Lightsail buckets.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Defaulted</code> - Synchronization failed and account-level BPA for your Lightsail buckets is defaulted
         * to <i>active</i>.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * You might need to complete further actions if the status is <code>Failed</code> or <code>Defaulted</code>.
         * The <code>message</code> parameter provides more information for those statuses.
         * </p>
         * </note>
         * 
         * @param status
         *        The status of the account-level BPA synchronization.</p>
         *        <p>
         *        The following statuses are possible:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>InSync</code> - Account-level BPA is synchronized. The Amazon S3 account-level BPA configuration
         *        applies to your Lightsail buckets.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NeverSynced</code> - Synchronization has not yet happened. The Amazon S3 account-level BPA
         *        configuration does not apply to your Lightsail buckets.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Failed</code> - Synchronization failed. The Amazon S3 account-level BPA configuration does not
         *        apply to your Lightsail buckets.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Defaulted</code> - Synchronization failed and account-level BPA for your Lightsail buckets is
         *        defaulted to <i>active</i>.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        You might need to complete further actions if the status is <code>Failed</code> or
         *        <code>Defaulted</code>. The <code>message</code> parameter provides more information for those
         *        statuses.
         *        </p>
         * @see AccountLevelBpaSyncStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AccountLevelBpaSyncStatus
         */
        Builder status(AccountLevelBpaSyncStatus status);

        /**
         * <p>
         * The timestamp of when the account-level BPA configuration was last synchronized. This value is null when the
         * account-level BPA configuration has not been synchronized.
         * </p>
         * 
         * @param lastSyncedAt
         *        The timestamp of when the account-level BPA configuration was last synchronized. This value is null
         *        when the account-level BPA configuration has not been synchronized.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastSyncedAt(Instant lastSyncedAt);

        /**
         * <p>
         * A message that provides a reason for a <code>Failed</code> or <code>Defaulted</code> synchronization status.
         * </p>
         * <p>
         * The following messages are possible:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SYNC_ON_HOLD</code> - The synchronization has not yet happened. This status message occurs immediately
         * after you create your first Lightsail bucket. This status message should change after the first
         * synchronization happens, approximately 1 hour after the first bucket is created.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DEFAULTED_FOR_SLR_MISSING</code> - The synchronization failed because the required service-linked role
         * is missing from your Amazon Web Services account. The account-level BPA configuration for your Lightsail
         * buckets is defaulted to <i>active</i> until the synchronization can occur. This means that all your buckets
         * are private and not publicly accessible. For more information about how to create the required service-linked
         * role to allow synchronization, see <a
         * href="https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-using-service-linked-roles"
         * >Using Service-Linked Roles for Amazon Lightsail</a> in the <i>Amazon Lightsail Developer Guide</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DEFAULTED_FOR_SLR_MISSING_ON_HOLD</code> - The synchronization failed because the required
         * service-linked role is missing from your Amazon Web Services account. Account-level BPA is not yet configured
         * for your Lightsail buckets. Therefore, only the bucket access permissions and individual object access
         * permissions apply to your Lightsail buckets. For more information about how to create the required
         * service-linked role to allow synchronization, see <a
         * href="https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-using-service-linked-roles"
         * >Using Service-Linked Roles for Amazon Lightsail</a> in the <i>Amazon Lightsail Developer Guide</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Unknown</code> - The reason that synchronization failed is unknown. Contact Amazon Web Services Support
         * for more information.
         * </p>
         * </li>
         * </ul>
         * 
         * @param message
         *        A message that provides a reason for a <code>Failed</code> or <code>Defaulted</code> synchronization
         *        status.</p>
         *        <p>
         *        The following messages are possible:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SYNC_ON_HOLD</code> - The synchronization has not yet happened. This status message occurs
         *        immediately after you create your first Lightsail bucket. This status message should change after the
         *        first synchronization happens, approximately 1 hour after the first bucket is created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DEFAULTED_FOR_SLR_MISSING</code> - The synchronization failed because the required
         *        service-linked role is missing from your Amazon Web Services account. The account-level BPA
         *        configuration for your Lightsail buckets is defaulted to <i>active</i> until the synchronization can
         *        occur. This means that all your buckets are private and not publicly accessible. For more information
         *        about how to create the required service-linked role to allow synchronization, see <a href=
         *        "https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-using-service-linked-roles"
         *        >Using Service-Linked Roles for Amazon Lightsail</a> in the <i>Amazon Lightsail Developer Guide</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DEFAULTED_FOR_SLR_MISSING_ON_HOLD</code> - The synchronization failed because the required
         *        service-linked role is missing from your Amazon Web Services account. Account-level BPA is not yet
         *        configured for your Lightsail buckets. Therefore, only the bucket access permissions and individual
         *        object access permissions apply to your Lightsail buckets. For more information about how to create
         *        the required service-linked role to allow synchronization, see <a href=
         *        "https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-using-service-linked-roles"
         *        >Using Service-Linked Roles for Amazon Lightsail</a> in the <i>Amazon Lightsail Developer Guide</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Unknown</code> - The reason that synchronization failed is unknown. Contact Amazon Web Services
         *        Support for more information.
         *        </p>
         *        </li>
         * @see BPAStatusMessage
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BPAStatusMessage
         */
        Builder message(String message);

        /**
         * <p>
         * A message that provides a reason for a <code>Failed</code> or <code>Defaulted</code> synchronization status.
         * </p>
         * <p>
         * The following messages are possible:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SYNC_ON_HOLD</code> - The synchronization has not yet happened. This status message occurs immediately
         * after you create your first Lightsail bucket. This status message should change after the first
         * synchronization happens, approximately 1 hour after the first bucket is created.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DEFAULTED_FOR_SLR_MISSING</code> - The synchronization failed because the required service-linked role
         * is missing from your Amazon Web Services account. The account-level BPA configuration for your Lightsail
         * buckets is defaulted to <i>active</i> until the synchronization can occur. This means that all your buckets
         * are private and not publicly accessible. For more information about how to create the required service-linked
         * role to allow synchronization, see <a
         * href="https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-using-service-linked-roles"
         * >Using Service-Linked Roles for Amazon Lightsail</a> in the <i>Amazon Lightsail Developer Guide</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DEFAULTED_FOR_SLR_MISSING_ON_HOLD</code> - The synchronization failed because the required
         * service-linked role is missing from your Amazon Web Services account. Account-level BPA is not yet configured
         * for your Lightsail buckets. Therefore, only the bucket access permissions and individual object access
         * permissions apply to your Lightsail buckets. For more information about how to create the required
         * service-linked role to allow synchronization, see <a
         * href="https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-using-service-linked-roles"
         * >Using Service-Linked Roles for Amazon Lightsail</a> in the <i>Amazon Lightsail Developer Guide</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Unknown</code> - The reason that synchronization failed is unknown. Contact Amazon Web Services Support
         * for more information.
         * </p>
         * </li>
         * </ul>
         * 
         * @param message
         *        A message that provides a reason for a <code>Failed</code> or <code>Defaulted</code> synchronization
         *        status.</p>
         *        <p>
         *        The following messages are possible:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SYNC_ON_HOLD</code> - The synchronization has not yet happened. This status message occurs
         *        immediately after you create your first Lightsail bucket. This status message should change after the
         *        first synchronization happens, approximately 1 hour after the first bucket is created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DEFAULTED_FOR_SLR_MISSING</code> - The synchronization failed because the required
         *        service-linked role is missing from your Amazon Web Services account. The account-level BPA
         *        configuration for your Lightsail buckets is defaulted to <i>active</i> until the synchronization can
         *        occur. This means that all your buckets are private and not publicly accessible. For more information
         *        about how to create the required service-linked role to allow synchronization, see <a href=
         *        "https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-using-service-linked-roles"
         *        >Using Service-Linked Roles for Amazon Lightsail</a> in the <i>Amazon Lightsail Developer Guide</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DEFAULTED_FOR_SLR_MISSING_ON_HOLD</code> - The synchronization failed because the required
         *        service-linked role is missing from your Amazon Web Services account. Account-level BPA is not yet
         *        configured for your Lightsail buckets. Therefore, only the bucket access permissions and individual
         *        object access permissions apply to your Lightsail buckets. For more information about how to create
         *        the required service-linked role to allow synchronization, see <a href=
         *        "https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-using-service-linked-roles"
         *        >Using Service-Linked Roles for Amazon Lightsail</a> in the <i>Amazon Lightsail Developer Guide</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Unknown</code> - The reason that synchronization failed is unknown. Contact Amazon Web Services
         *        Support for more information.
         *        </p>
         *        </li>
         * @see BPAStatusMessage
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BPAStatusMessage
         */
        Builder message(BPAStatusMessage message);

        /**
         * <p>
         * A Boolean value that indicates whether account-level block public access is affecting your Lightsail buckets.
         * </p>
         * 
         * @param bpaImpactsLightsail
         *        A Boolean value that indicates whether account-level block public access is affecting your Lightsail
         *        buckets.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bpaImpactsLightsail(Boolean bpaImpactsLightsail);
    }

    static final class BuilderImpl implements Builder {
        private String status;

        private Instant lastSyncedAt;

        private String message;

        private Boolean bpaImpactsLightsail;

        private BuilderImpl() {
        }

        private BuilderImpl(AccountLevelBpaSync model) {
            status(model.status);
            lastSyncedAt(model.lastSyncedAt);
            message(model.message);
            bpaImpactsLightsail(model.bpaImpactsLightsail);
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(AccountLevelBpaSyncStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Instant getLastSyncedAt() {
            return lastSyncedAt;
        }

        public final void setLastSyncedAt(Instant lastSyncedAt) {
            this.lastSyncedAt = lastSyncedAt;
        }

        @Override
        public final Builder lastSyncedAt(Instant lastSyncedAt) {
            this.lastSyncedAt = lastSyncedAt;
            return this;
        }

        public final String getMessage() {
            return message;
        }

        public final void setMessage(String message) {
            this.message = message;
        }

        @Override
        public final Builder message(String message) {
            this.message = message;
            return this;
        }

        @Override
        public final Builder message(BPAStatusMessage message) {
            this.message(message == null ? null : message.toString());
            return this;
        }

        public final Boolean getBpaImpactsLightsail() {
            return bpaImpactsLightsail;
        }

        public final void setBpaImpactsLightsail(Boolean bpaImpactsLightsail) {
            this.bpaImpactsLightsail = bpaImpactsLightsail;
        }

        @Override
        public final Builder bpaImpactsLightsail(Boolean bpaImpactsLightsail) {
            this.bpaImpactsLightsail = bpaImpactsLightsail;
            return this;
        }

        @Override
        public AccountLevelBpaSync build() {
            return new AccountLevelBpaSync(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
