/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lightsail.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an alarm.
 * </p>
 * <p>
 * An alarm is a way to monitor your Lightsail resource metrics. For more information, see <a
 * href="https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-alarms">Alarms in Amazon
 * Lightsail</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Alarm implements SdkPojo, Serializable, ToCopyableBuilder<Alarm.Builder, Alarm> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(Alarm::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(Alarm::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt").getter(getter(Alarm::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build()).build();

    private static final SdkField<ResourceLocation> LOCATION_FIELD = SdkField
            .<ResourceLocation> builder(MarshallingType.SDK_POJO).memberName("location").getter(getter(Alarm::location))
            .setter(setter(Builder::location)).constructor(ResourceLocation::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("location").build()).build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("resourceType").getter(getter(Alarm::resourceTypeAsString)).setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceType").build()).build();

    private static final SdkField<String> SUPPORT_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("supportCode").getter(getter(Alarm::supportCode)).setter(setter(Builder::supportCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("supportCode").build()).build();

    private static final SdkField<MonitoredResourceInfo> MONITORED_RESOURCE_INFO_FIELD = SdkField
            .<MonitoredResourceInfo> builder(MarshallingType.SDK_POJO).memberName("monitoredResourceInfo")
            .getter(getter(Alarm::monitoredResourceInfo)).setter(setter(Builder::monitoredResourceInfo))
            .constructor(MonitoredResourceInfo::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("monitoredResourceInfo").build())
            .build();

    private static final SdkField<String> COMPARISON_OPERATOR_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("comparisonOperator").getter(getter(Alarm::comparisonOperatorAsString))
            .setter(setter(Builder::comparisonOperator))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("comparisonOperator").build())
            .build();

    private static final SdkField<Integer> EVALUATION_PERIODS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("evaluationPeriods").getter(getter(Alarm::evaluationPeriods)).setter(setter(Builder::evaluationPeriods))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("evaluationPeriods").build()).build();

    private static final SdkField<Integer> PERIOD_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("period").getter(getter(Alarm::period)).setter(setter(Builder::period))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("period").build()).build();

    private static final SdkField<Double> THRESHOLD_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("threshold").getter(getter(Alarm::threshold)).setter(setter(Builder::threshold))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("threshold").build()).build();

    private static final SdkField<Integer> DATAPOINTS_TO_ALARM_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("datapointsToAlarm").getter(getter(Alarm::datapointsToAlarm)).setter(setter(Builder::datapointsToAlarm))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("datapointsToAlarm").build()).build();

    private static final SdkField<String> TREAT_MISSING_DATA_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("treatMissingData").getter(getter(Alarm::treatMissingDataAsString))
            .setter(setter(Builder::treatMissingData))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("treatMissingData").build()).build();

    private static final SdkField<String> STATISTIC_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("statistic").getter(getter(Alarm::statisticAsString)).setter(setter(Builder::statistic))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("statistic").build()).build();

    private static final SdkField<String> METRIC_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("metricName").getter(getter(Alarm::metricNameAsString)).setter(setter(Builder::metricName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("metricName").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("state")
            .getter(getter(Alarm::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("state").build()).build();

    private static final SdkField<String> UNIT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("unit")
            .getter(getter(Alarm::unitAsString)).setter(setter(Builder::unit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("unit").build()).build();

    private static final SdkField<List<String>> CONTACT_PROTOCOLS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("contactProtocols")
            .getter(getter(Alarm::contactProtocolsAsStrings))
            .setter(setter(Builder::contactProtocolsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("contactProtocols").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> NOTIFICATION_TRIGGERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("notificationTriggers")
            .getter(getter(Alarm::notificationTriggersAsStrings))
            .setter(setter(Builder::notificationTriggersWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("notificationTriggers").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> NOTIFICATION_ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("notificationEnabled").getter(getter(Alarm::notificationEnabled))
            .setter(setter(Builder::notificationEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("notificationEnabled").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, ARN_FIELD,
            CREATED_AT_FIELD, LOCATION_FIELD, RESOURCE_TYPE_FIELD, SUPPORT_CODE_FIELD, MONITORED_RESOURCE_INFO_FIELD,
            COMPARISON_OPERATOR_FIELD, EVALUATION_PERIODS_FIELD, PERIOD_FIELD, THRESHOLD_FIELD, DATAPOINTS_TO_ALARM_FIELD,
            TREAT_MISSING_DATA_FIELD, STATISTIC_FIELD, METRIC_NAME_FIELD, STATE_FIELD, UNIT_FIELD, CONTACT_PROTOCOLS_FIELD,
            NOTIFICATION_TRIGGERS_FIELD, NOTIFICATION_ENABLED_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String arn;

    private final Instant createdAt;

    private final ResourceLocation location;

    private final String resourceType;

    private final String supportCode;

    private final MonitoredResourceInfo monitoredResourceInfo;

    private final String comparisonOperator;

    private final Integer evaluationPeriods;

    private final Integer period;

    private final Double threshold;

    private final Integer datapointsToAlarm;

    private final String treatMissingData;

    private final String statistic;

    private final String metricName;

    private final String state;

    private final String unit;

    private final List<String> contactProtocols;

    private final List<String> notificationTriggers;

    private final Boolean notificationEnabled;

    private Alarm(BuilderImpl builder) {
        this.name = builder.name;
        this.arn = builder.arn;
        this.createdAt = builder.createdAt;
        this.location = builder.location;
        this.resourceType = builder.resourceType;
        this.supportCode = builder.supportCode;
        this.monitoredResourceInfo = builder.monitoredResourceInfo;
        this.comparisonOperator = builder.comparisonOperator;
        this.evaluationPeriods = builder.evaluationPeriods;
        this.period = builder.period;
        this.threshold = builder.threshold;
        this.datapointsToAlarm = builder.datapointsToAlarm;
        this.treatMissingData = builder.treatMissingData;
        this.statistic = builder.statistic;
        this.metricName = builder.metricName;
        this.state = builder.state;
        this.unit = builder.unit;
        this.contactProtocols = builder.contactProtocols;
        this.notificationTriggers = builder.notificationTriggers;
        this.notificationEnabled = builder.notificationEnabled;
    }

    /**
     * <p>
     * The name of the alarm.
     * </p>
     * 
     * @return The name of the alarm.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the alarm.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the alarm.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The timestamp when the alarm was created.
     * </p>
     * 
     * @return The timestamp when the alarm was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * An object that lists information about the location of the alarm.
     * </p>
     * 
     * @return An object that lists information about the location of the alarm.
     */
    public final ResourceLocation location() {
        return location;
    }

    /**
     * <p>
     * The Lightsail resource type of the alarm.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The Lightsail resource type of the alarm.
     * @see ResourceType
     */
    public final ResourceType resourceType() {
        return ResourceType.fromValue(resourceType);
    }

    /**
     * <p>
     * The Lightsail resource type of the alarm.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The Lightsail resource type of the alarm.
     * @see ResourceType
     */
    public final String resourceTypeAsString() {
        return resourceType;
    }

    /**
     * <p>
     * The support code. Include this code in your email to support when you have questions about your Lightsail alarm.
     * This code enables our support team to look up your Lightsail information more easily.
     * </p>
     * 
     * @return The support code. Include this code in your email to support when you have questions about your Lightsail
     *         alarm. This code enables our support team to look up your Lightsail information more easily.
     */
    public final String supportCode() {
        return supportCode;
    }

    /**
     * <p>
     * An object that lists information about the resource monitored by the alarm.
     * </p>
     * 
     * @return An object that lists information about the resource monitored by the alarm.
     */
    public final MonitoredResourceInfo monitoredResourceInfo() {
        return monitoredResourceInfo;
    }

    /**
     * <p>
     * The arithmetic operation used when comparing the specified statistic and threshold.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #comparisonOperator} will return {@link ComparisonOperator#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #comparisonOperatorAsString}.
     * </p>
     * 
     * @return The arithmetic operation used when comparing the specified statistic and threshold.
     * @see ComparisonOperator
     */
    public final ComparisonOperator comparisonOperator() {
        return ComparisonOperator.fromValue(comparisonOperator);
    }

    /**
     * <p>
     * The arithmetic operation used when comparing the specified statistic and threshold.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #comparisonOperator} will return {@link ComparisonOperator#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #comparisonOperatorAsString}.
     * </p>
     * 
     * @return The arithmetic operation used when comparing the specified statistic and threshold.
     * @see ComparisonOperator
     */
    public final String comparisonOperatorAsString() {
        return comparisonOperator;
    }

    /**
     * <p>
     * The number of periods over which data is compared to the specified threshold.
     * </p>
     * 
     * @return The number of periods over which data is compared to the specified threshold.
     */
    public final Integer evaluationPeriods() {
        return evaluationPeriods;
    }

    /**
     * <p>
     * The period, in seconds, over which the statistic is applied.
     * </p>
     * 
     * @return The period, in seconds, over which the statistic is applied.
     */
    public final Integer period() {
        return period;
    }

    /**
     * <p>
     * The value against which the specified statistic is compared.
     * </p>
     * 
     * @return The value against which the specified statistic is compared.
     */
    public final Double threshold() {
        return threshold;
    }

    /**
     * <p>
     * The number of data points that must not within the specified threshold to trigger the alarm.
     * </p>
     * 
     * @return The number of data points that must not within the specified threshold to trigger the alarm.
     */
    public final Integer datapointsToAlarm() {
        return datapointsToAlarm;
    }

    /**
     * <p>
     * Specifies how the alarm handles missing data points.
     * </p>
     * <p>
     * An alarm can treat missing data in the following ways:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>breaching</code> - Assume the missing data is not within the threshold. Missing data counts towards the
     * number of times the metric is not within the threshold.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>notBreaching</code> - Assume the missing data is within the threshold. Missing data does not count towards
     * the number of times the metric is not within the threshold.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ignore</code> - Ignore the missing data. Maintains the current alarm state.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>missing</code> - Missing data is treated as missing.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #treatMissingData}
     * will return {@link TreatMissingData#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #treatMissingDataAsString}.
     * </p>
     * 
     * @return Specifies how the alarm handles missing data points.</p>
     *         <p>
     *         An alarm can treat missing data in the following ways:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>breaching</code> - Assume the missing data is not within the threshold. Missing data counts towards
     *         the number of times the metric is not within the threshold.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>notBreaching</code> - Assume the missing data is within the threshold. Missing data does not count
     *         towards the number of times the metric is not within the threshold.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ignore</code> - Ignore the missing data. Maintains the current alarm state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>missing</code> - Missing data is treated as missing.
     *         </p>
     *         </li>
     * @see TreatMissingData
     */
    public final TreatMissingData treatMissingData() {
        return TreatMissingData.fromValue(treatMissingData);
    }

    /**
     * <p>
     * Specifies how the alarm handles missing data points.
     * </p>
     * <p>
     * An alarm can treat missing data in the following ways:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>breaching</code> - Assume the missing data is not within the threshold. Missing data counts towards the
     * number of times the metric is not within the threshold.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>notBreaching</code> - Assume the missing data is within the threshold. Missing data does not count towards
     * the number of times the metric is not within the threshold.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ignore</code> - Ignore the missing data. Maintains the current alarm state.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>missing</code> - Missing data is treated as missing.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #treatMissingData}
     * will return {@link TreatMissingData#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #treatMissingDataAsString}.
     * </p>
     * 
     * @return Specifies how the alarm handles missing data points.</p>
     *         <p>
     *         An alarm can treat missing data in the following ways:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>breaching</code> - Assume the missing data is not within the threshold. Missing data counts towards
     *         the number of times the metric is not within the threshold.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>notBreaching</code> - Assume the missing data is within the threshold. Missing data does not count
     *         towards the number of times the metric is not within the threshold.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ignore</code> - Ignore the missing data. Maintains the current alarm state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>missing</code> - Missing data is treated as missing.
     *         </p>
     *         </li>
     * @see TreatMissingData
     */
    public final String treatMissingDataAsString() {
        return treatMissingData;
    }

    /**
     * <p>
     * The statistic for the metric associated with the alarm.
     * </p>
     * <p>
     * The following statistics are available:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Minimum</code> - The lowest value observed during the specified period. Use this value to determine low
     * volumes of activity for your application.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Maximum</code> - The highest value observed during the specified period. Use this value to determine high
     * volumes of activity for your application.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Sum</code> - All values submitted for the matching metric added together. You can use this statistic to
     * determine the total volume of a metric.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Average</code> - The value of Sum / SampleCount during the specified period. By comparing this statistic
     * with the Minimum and Maximum values, you can determine the full scope of a metric and how close the average use
     * is to the Minimum and Maximum values. This comparison helps you to know when to increase or decrease your
     * resources.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statistic} will
     * return {@link MetricStatistic#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statisticAsString}.
     * </p>
     * 
     * @return The statistic for the metric associated with the alarm.</p>
     *         <p>
     *         The following statistics are available:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Minimum</code> - The lowest value observed during the specified period. Use this value to determine
     *         low volumes of activity for your application.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Maximum</code> - The highest value observed during the specified period. Use this value to
     *         determine high volumes of activity for your application.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Sum</code> - All values submitted for the matching metric added together. You can use this
     *         statistic to determine the total volume of a metric.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Average</code> - The value of Sum / SampleCount during the specified period. By comparing this
     *         statistic with the Minimum and Maximum values, you can determine the full scope of a metric and how close
     *         the average use is to the Minimum and Maximum values. This comparison helps you to know when to increase
     *         or decrease your resources.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
     *         </p>
     *         </li>
     * @see MetricStatistic
     */
    public final MetricStatistic statistic() {
        return MetricStatistic.fromValue(statistic);
    }

    /**
     * <p>
     * The statistic for the metric associated with the alarm.
     * </p>
     * <p>
     * The following statistics are available:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Minimum</code> - The lowest value observed during the specified period. Use this value to determine low
     * volumes of activity for your application.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Maximum</code> - The highest value observed during the specified period. Use this value to determine high
     * volumes of activity for your application.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Sum</code> - All values submitted for the matching metric added together. You can use this statistic to
     * determine the total volume of a metric.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Average</code> - The value of Sum / SampleCount during the specified period. By comparing this statistic
     * with the Minimum and Maximum values, you can determine the full scope of a metric and how close the average use
     * is to the Minimum and Maximum values. This comparison helps you to know when to increase or decrease your
     * resources.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statistic} will
     * return {@link MetricStatistic#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statisticAsString}.
     * </p>
     * 
     * @return The statistic for the metric associated with the alarm.</p>
     *         <p>
     *         The following statistics are available:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Minimum</code> - The lowest value observed during the specified period. Use this value to determine
     *         low volumes of activity for your application.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Maximum</code> - The highest value observed during the specified period. Use this value to
     *         determine high volumes of activity for your application.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Sum</code> - All values submitted for the matching metric added together. You can use this
     *         statistic to determine the total volume of a metric.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Average</code> - The value of Sum / SampleCount during the specified period. By comparing this
     *         statistic with the Minimum and Maximum values, you can determine the full scope of a metric and how close
     *         the average use is to the Minimum and Maximum values. This comparison helps you to know when to increase
     *         or decrease your resources.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
     *         </p>
     *         </li>
     * @see MetricStatistic
     */
    public final String statisticAsString() {
        return statistic;
    }

    /**
     * <p>
     * The name of the metric associated with the alarm.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #metricName} will
     * return {@link MetricName#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #metricNameAsString}.
     * </p>
     * 
     * @return The name of the metric associated with the alarm.
     * @see MetricName
     */
    public final MetricName metricName() {
        return MetricName.fromValue(metricName);
    }

    /**
     * <p>
     * The name of the metric associated with the alarm.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #metricName} will
     * return {@link MetricName#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #metricNameAsString}.
     * </p>
     * 
     * @return The name of the metric associated with the alarm.
     * @see MetricName
     */
    public final String metricNameAsString() {
        return metricName;
    }

    /**
     * <p>
     * The current state of the alarm.
     * </p>
     * <p>
     * An alarm has the following possible states:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ALARM</code> - The metric is outside of the defined threshold.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INSUFFICIENT_DATA</code> - The alarm has just started, the metric is not available, or not enough data is
     * available for the metric to determine the alarm state.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OK</code> - The metric is within the defined threshold.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link AlarmState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of the alarm.</p>
     *         <p>
     *         An alarm has the following possible states:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ALARM</code> - The metric is outside of the defined threshold.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INSUFFICIENT_DATA</code> - The alarm has just started, the metric is not available, or not enough
     *         data is available for the metric to determine the alarm state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>OK</code> - The metric is within the defined threshold.
     *         </p>
     *         </li>
     * @see AlarmState
     */
    public final AlarmState state() {
        return AlarmState.fromValue(state);
    }

    /**
     * <p>
     * The current state of the alarm.
     * </p>
     * <p>
     * An alarm has the following possible states:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ALARM</code> - The metric is outside of the defined threshold.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INSUFFICIENT_DATA</code> - The alarm has just started, the metric is not available, or not enough data is
     * available for the metric to determine the alarm state.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OK</code> - The metric is within the defined threshold.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link AlarmState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of the alarm.</p>
     *         <p>
     *         An alarm has the following possible states:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ALARM</code> - The metric is outside of the defined threshold.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INSUFFICIENT_DATA</code> - The alarm has just started, the metric is not available, or not enough
     *         data is available for the metric to determine the alarm state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>OK</code> - The metric is within the defined threshold.
     *         </p>
     *         </li>
     * @see AlarmState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The unit of the metric associated with the alarm.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #unit} will return
     * {@link MetricUnit#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #unitAsString}.
     * </p>
     * 
     * @return The unit of the metric associated with the alarm.
     * @see MetricUnit
     */
    public final MetricUnit unit() {
        return MetricUnit.fromValue(unit);
    }

    /**
     * <p>
     * The unit of the metric associated with the alarm.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #unit} will return
     * {@link MetricUnit#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #unitAsString}.
     * </p>
     * 
     * @return The unit of the metric associated with the alarm.
     * @see MetricUnit
     */
    public final String unitAsString() {
        return unit;
    }

    /**
     * <p>
     * The contact protocols for the alarm, such as <code>Email</code>, <code>SMS</code> (text messaging), or both.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasContactProtocols} method.
     * </p>
     * 
     * @return The contact protocols for the alarm, such as <code>Email</code>, <code>SMS</code> (text messaging), or
     *         both.
     */
    public final List<ContactProtocol> contactProtocols() {
        return ContactProtocolsListCopier.copyStringToEnum(contactProtocols);
    }

    /**
     * For responses, this returns true if the service returned a value for the ContactProtocols property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasContactProtocols() {
        return contactProtocols != null && !(contactProtocols instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The contact protocols for the alarm, such as <code>Email</code>, <code>SMS</code> (text messaging), or both.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasContactProtocols} method.
     * </p>
     * 
     * @return The contact protocols for the alarm, such as <code>Email</code>, <code>SMS</code> (text messaging), or
     *         both.
     */
    public final List<String> contactProtocolsAsStrings() {
        return contactProtocols;
    }

    /**
     * <p>
     * The alarm states that trigger a notification.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNotificationTriggers} method.
     * </p>
     * 
     * @return The alarm states that trigger a notification.
     */
    public final List<AlarmState> notificationTriggers() {
        return NotificationTriggerListCopier.copyStringToEnum(notificationTriggers);
    }

    /**
     * For responses, this returns true if the service returned a value for the NotificationTriggers property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasNotificationTriggers() {
        return notificationTriggers != null && !(notificationTriggers instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The alarm states that trigger a notification.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNotificationTriggers} method.
     * </p>
     * 
     * @return The alarm states that trigger a notification.
     */
    public final List<String> notificationTriggersAsStrings() {
        return notificationTriggers;
    }

    /**
     * <p>
     * Indicates whether the alarm is enabled.
     * </p>
     * 
     * @return Indicates whether the alarm is enabled.
     */
    public final Boolean notificationEnabled() {
        return notificationEnabled;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(location());
        hashCode = 31 * hashCode + Objects.hashCode(resourceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(supportCode());
        hashCode = 31 * hashCode + Objects.hashCode(monitoredResourceInfo());
        hashCode = 31 * hashCode + Objects.hashCode(comparisonOperatorAsString());
        hashCode = 31 * hashCode + Objects.hashCode(evaluationPeriods());
        hashCode = 31 * hashCode + Objects.hashCode(period());
        hashCode = 31 * hashCode + Objects.hashCode(threshold());
        hashCode = 31 * hashCode + Objects.hashCode(datapointsToAlarm());
        hashCode = 31 * hashCode + Objects.hashCode(treatMissingDataAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statisticAsString());
        hashCode = 31 * hashCode + Objects.hashCode(metricNameAsString());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(unitAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasContactProtocols() ? contactProtocolsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasNotificationTriggers() ? notificationTriggersAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(notificationEnabled());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Alarm)) {
            return false;
        }
        Alarm other = (Alarm) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(arn(), other.arn())
                && Objects.equals(createdAt(), other.createdAt()) && Objects.equals(location(), other.location())
                && Objects.equals(resourceTypeAsString(), other.resourceTypeAsString())
                && Objects.equals(supportCode(), other.supportCode())
                && Objects.equals(monitoredResourceInfo(), other.monitoredResourceInfo())
                && Objects.equals(comparisonOperatorAsString(), other.comparisonOperatorAsString())
                && Objects.equals(evaluationPeriods(), other.evaluationPeriods()) && Objects.equals(period(), other.period())
                && Objects.equals(threshold(), other.threshold())
                && Objects.equals(datapointsToAlarm(), other.datapointsToAlarm())
                && Objects.equals(treatMissingDataAsString(), other.treatMissingDataAsString())
                && Objects.equals(statisticAsString(), other.statisticAsString())
                && Objects.equals(metricNameAsString(), other.metricNameAsString())
                && Objects.equals(stateAsString(), other.stateAsString()) && Objects.equals(unitAsString(), other.unitAsString())
                && hasContactProtocols() == other.hasContactProtocols()
                && Objects.equals(contactProtocolsAsStrings(), other.contactProtocolsAsStrings())
                && hasNotificationTriggers() == other.hasNotificationTriggers()
                && Objects.equals(notificationTriggersAsStrings(), other.notificationTriggersAsStrings())
                && Objects.equals(notificationEnabled(), other.notificationEnabled());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Alarm").add("Name", name()).add("Arn", arn()).add("CreatedAt", createdAt())
                .add("Location", location()).add("ResourceType", resourceTypeAsString()).add("SupportCode", supportCode())
                .add("MonitoredResourceInfo", monitoredResourceInfo()).add("ComparisonOperator", comparisonOperatorAsString())
                .add("EvaluationPeriods", evaluationPeriods()).add("Period", period()).add("Threshold", threshold())
                .add("DatapointsToAlarm", datapointsToAlarm()).add("TreatMissingData", treatMissingDataAsString())
                .add("Statistic", statisticAsString()).add("MetricName", metricNameAsString()).add("State", stateAsString())
                .add("Unit", unitAsString()).add("ContactProtocols", hasContactProtocols() ? contactProtocolsAsStrings() : null)
                .add("NotificationTriggers", hasNotificationTriggers() ? notificationTriggersAsStrings() : null)
                .add("NotificationEnabled", notificationEnabled()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "location":
            return Optional.ofNullable(clazz.cast(location()));
        case "resourceType":
            return Optional.ofNullable(clazz.cast(resourceTypeAsString()));
        case "supportCode":
            return Optional.ofNullable(clazz.cast(supportCode()));
        case "monitoredResourceInfo":
            return Optional.ofNullable(clazz.cast(monitoredResourceInfo()));
        case "comparisonOperator":
            return Optional.ofNullable(clazz.cast(comparisonOperatorAsString()));
        case "evaluationPeriods":
            return Optional.ofNullable(clazz.cast(evaluationPeriods()));
        case "period":
            return Optional.ofNullable(clazz.cast(period()));
        case "threshold":
            return Optional.ofNullable(clazz.cast(threshold()));
        case "datapointsToAlarm":
            return Optional.ofNullable(clazz.cast(datapointsToAlarm()));
        case "treatMissingData":
            return Optional.ofNullable(clazz.cast(treatMissingDataAsString()));
        case "statistic":
            return Optional.ofNullable(clazz.cast(statisticAsString()));
        case "metricName":
            return Optional.ofNullable(clazz.cast(metricNameAsString()));
        case "state":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "unit":
            return Optional.ofNullable(clazz.cast(unitAsString()));
        case "contactProtocols":
            return Optional.ofNullable(clazz.cast(contactProtocolsAsStrings()));
        case "notificationTriggers":
            return Optional.ofNullable(clazz.cast(notificationTriggersAsStrings()));
        case "notificationEnabled":
            return Optional.ofNullable(clazz.cast(notificationEnabled()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("name", NAME_FIELD);
        map.put("arn", ARN_FIELD);
        map.put("createdAt", CREATED_AT_FIELD);
        map.put("location", LOCATION_FIELD);
        map.put("resourceType", RESOURCE_TYPE_FIELD);
        map.put("supportCode", SUPPORT_CODE_FIELD);
        map.put("monitoredResourceInfo", MONITORED_RESOURCE_INFO_FIELD);
        map.put("comparisonOperator", COMPARISON_OPERATOR_FIELD);
        map.put("evaluationPeriods", EVALUATION_PERIODS_FIELD);
        map.put("period", PERIOD_FIELD);
        map.put("threshold", THRESHOLD_FIELD);
        map.put("datapointsToAlarm", DATAPOINTS_TO_ALARM_FIELD);
        map.put("treatMissingData", TREAT_MISSING_DATA_FIELD);
        map.put("statistic", STATISTIC_FIELD);
        map.put("metricName", METRIC_NAME_FIELD);
        map.put("state", STATE_FIELD);
        map.put("unit", UNIT_FIELD);
        map.put("contactProtocols", CONTACT_PROTOCOLS_FIELD);
        map.put("notificationTriggers", NOTIFICATION_TRIGGERS_FIELD);
        map.put("notificationEnabled", NOTIFICATION_ENABLED_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Alarm, T> g) {
        return obj -> g.apply((Alarm) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Alarm> {
        /**
         * <p>
         * The name of the alarm.
         * </p>
         * 
         * @param name
         *        The name of the alarm.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the alarm.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the alarm.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The timestamp when the alarm was created.
         * </p>
         * 
         * @param createdAt
         *        The timestamp when the alarm was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * An object that lists information about the location of the alarm.
         * </p>
         * 
         * @param location
         *        An object that lists information about the location of the alarm.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder location(ResourceLocation location);

        /**
         * <p>
         * An object that lists information about the location of the alarm.
         * </p>
         * This is a convenience method that creates an instance of the {@link ResourceLocation.Builder} avoiding the
         * need to create one manually via {@link ResourceLocation#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ResourceLocation.Builder#build()} is called immediately and its
         * result is passed to {@link #location(ResourceLocation)}.
         * 
         * @param location
         *        a consumer that will call methods on {@link ResourceLocation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #location(ResourceLocation)
         */
        default Builder location(Consumer<ResourceLocation.Builder> location) {
            return location(ResourceLocation.builder().applyMutation(location).build());
        }

        /**
         * <p>
         * The Lightsail resource type of the alarm.
         * </p>
         * 
         * @param resourceType
         *        The Lightsail resource type of the alarm.
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * The Lightsail resource type of the alarm.
         * </p>
         * 
         * @param resourceType
         *        The Lightsail resource type of the alarm.
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder resourceType(ResourceType resourceType);

        /**
         * <p>
         * The support code. Include this code in your email to support when you have questions about your Lightsail
         * alarm. This code enables our support team to look up your Lightsail information more easily.
         * </p>
         * 
         * @param supportCode
         *        The support code. Include this code in your email to support when you have questions about your
         *        Lightsail alarm. This code enables our support team to look up your Lightsail information more easily.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportCode(String supportCode);

        /**
         * <p>
         * An object that lists information about the resource monitored by the alarm.
         * </p>
         * 
         * @param monitoredResourceInfo
         *        An object that lists information about the resource monitored by the alarm.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder monitoredResourceInfo(MonitoredResourceInfo monitoredResourceInfo);

        /**
         * <p>
         * An object that lists information about the resource monitored by the alarm.
         * </p>
         * This is a convenience method that creates an instance of the {@link MonitoredResourceInfo.Builder} avoiding
         * the need to create one manually via {@link MonitoredResourceInfo#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link MonitoredResourceInfo.Builder#build()} is called immediately and
         * its result is passed to {@link #monitoredResourceInfo(MonitoredResourceInfo)}.
         * 
         * @param monitoredResourceInfo
         *        a consumer that will call methods on {@link MonitoredResourceInfo.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #monitoredResourceInfo(MonitoredResourceInfo)
         */
        default Builder monitoredResourceInfo(Consumer<MonitoredResourceInfo.Builder> monitoredResourceInfo) {
            return monitoredResourceInfo(MonitoredResourceInfo.builder().applyMutation(monitoredResourceInfo).build());
        }

        /**
         * <p>
         * The arithmetic operation used when comparing the specified statistic and threshold.
         * </p>
         * 
         * @param comparisonOperator
         *        The arithmetic operation used when comparing the specified statistic and threshold.
         * @see ComparisonOperator
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComparisonOperator
         */
        Builder comparisonOperator(String comparisonOperator);

        /**
         * <p>
         * The arithmetic operation used when comparing the specified statistic and threshold.
         * </p>
         * 
         * @param comparisonOperator
         *        The arithmetic operation used when comparing the specified statistic and threshold.
         * @see ComparisonOperator
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComparisonOperator
         */
        Builder comparisonOperator(ComparisonOperator comparisonOperator);

        /**
         * <p>
         * The number of periods over which data is compared to the specified threshold.
         * </p>
         * 
         * @param evaluationPeriods
         *        The number of periods over which data is compared to the specified threshold.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder evaluationPeriods(Integer evaluationPeriods);

        /**
         * <p>
         * The period, in seconds, over which the statistic is applied.
         * </p>
         * 
         * @param period
         *        The period, in seconds, over which the statistic is applied.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder period(Integer period);

        /**
         * <p>
         * The value against which the specified statistic is compared.
         * </p>
         * 
         * @param threshold
         *        The value against which the specified statistic is compared.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder threshold(Double threshold);

        /**
         * <p>
         * The number of data points that must not within the specified threshold to trigger the alarm.
         * </p>
         * 
         * @param datapointsToAlarm
         *        The number of data points that must not within the specified threshold to trigger the alarm.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder datapointsToAlarm(Integer datapointsToAlarm);

        /**
         * <p>
         * Specifies how the alarm handles missing data points.
         * </p>
         * <p>
         * An alarm can treat missing data in the following ways:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>breaching</code> - Assume the missing data is not within the threshold. Missing data counts towards the
         * number of times the metric is not within the threshold.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>notBreaching</code> - Assume the missing data is within the threshold. Missing data does not count
         * towards the number of times the metric is not within the threshold.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ignore</code> - Ignore the missing data. Maintains the current alarm state.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>missing</code> - Missing data is treated as missing.
         * </p>
         * </li>
         * </ul>
         * 
         * @param treatMissingData
         *        Specifies how the alarm handles missing data points.</p>
         *        <p>
         *        An alarm can treat missing data in the following ways:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>breaching</code> - Assume the missing data is not within the threshold. Missing data counts
         *        towards the number of times the metric is not within the threshold.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>notBreaching</code> - Assume the missing data is within the threshold. Missing data does not
         *        count towards the number of times the metric is not within the threshold.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ignore</code> - Ignore the missing data. Maintains the current alarm state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>missing</code> - Missing data is treated as missing.
         *        </p>
         *        </li>
         * @see TreatMissingData
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TreatMissingData
         */
        Builder treatMissingData(String treatMissingData);

        /**
         * <p>
         * Specifies how the alarm handles missing data points.
         * </p>
         * <p>
         * An alarm can treat missing data in the following ways:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>breaching</code> - Assume the missing data is not within the threshold. Missing data counts towards the
         * number of times the metric is not within the threshold.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>notBreaching</code> - Assume the missing data is within the threshold. Missing data does not count
         * towards the number of times the metric is not within the threshold.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ignore</code> - Ignore the missing data. Maintains the current alarm state.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>missing</code> - Missing data is treated as missing.
         * </p>
         * </li>
         * </ul>
         * 
         * @param treatMissingData
         *        Specifies how the alarm handles missing data points.</p>
         *        <p>
         *        An alarm can treat missing data in the following ways:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>breaching</code> - Assume the missing data is not within the threshold. Missing data counts
         *        towards the number of times the metric is not within the threshold.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>notBreaching</code> - Assume the missing data is within the threshold. Missing data does not
         *        count towards the number of times the metric is not within the threshold.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ignore</code> - Ignore the missing data. Maintains the current alarm state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>missing</code> - Missing data is treated as missing.
         *        </p>
         *        </li>
         * @see TreatMissingData
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TreatMissingData
         */
        Builder treatMissingData(TreatMissingData treatMissingData);

        /**
         * <p>
         * The statistic for the metric associated with the alarm.
         * </p>
         * <p>
         * The following statistics are available:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Minimum</code> - The lowest value observed during the specified period. Use this value to determine low
         * volumes of activity for your application.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Maximum</code> - The highest value observed during the specified period. Use this value to determine
         * high volumes of activity for your application.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Sum</code> - All values submitted for the matching metric added together. You can use this statistic to
         * determine the total volume of a metric.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Average</code> - The value of Sum / SampleCount during the specified period. By comparing this
         * statistic with the Minimum and Maximum values, you can determine the full scope of a metric and how close the
         * average use is to the Minimum and Maximum values. This comparison helps you to know when to increase or
         * decrease your resources.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
         * </p>
         * </li>
         * </ul>
         * 
         * @param statistic
         *        The statistic for the metric associated with the alarm.</p>
         *        <p>
         *        The following statistics are available:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Minimum</code> - The lowest value observed during the specified period. Use this value to
         *        determine low volumes of activity for your application.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Maximum</code> - The highest value observed during the specified period. Use this value to
         *        determine high volumes of activity for your application.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Sum</code> - All values submitted for the matching metric added together. You can use this
         *        statistic to determine the total volume of a metric.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Average</code> - The value of Sum / SampleCount during the specified period. By comparing this
         *        statistic with the Minimum and Maximum values, you can determine the full scope of a metric and how
         *        close the average use is to the Minimum and Maximum values. This comparison helps you to know when to
         *        increase or decrease your resources.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
         *        </p>
         *        </li>
         * @see MetricStatistic
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricStatistic
         */
        Builder statistic(String statistic);

        /**
         * <p>
         * The statistic for the metric associated with the alarm.
         * </p>
         * <p>
         * The following statistics are available:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Minimum</code> - The lowest value observed during the specified period. Use this value to determine low
         * volumes of activity for your application.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Maximum</code> - The highest value observed during the specified period. Use this value to determine
         * high volumes of activity for your application.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Sum</code> - All values submitted for the matching metric added together. You can use this statistic to
         * determine the total volume of a metric.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Average</code> - The value of Sum / SampleCount during the specified period. By comparing this
         * statistic with the Minimum and Maximum values, you can determine the full scope of a metric and how close the
         * average use is to the Minimum and Maximum values. This comparison helps you to know when to increase or
         * decrease your resources.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
         * </p>
         * </li>
         * </ul>
         * 
         * @param statistic
         *        The statistic for the metric associated with the alarm.</p>
         *        <p>
         *        The following statistics are available:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Minimum</code> - The lowest value observed during the specified period. Use this value to
         *        determine low volumes of activity for your application.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Maximum</code> - The highest value observed during the specified period. Use this value to
         *        determine high volumes of activity for your application.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Sum</code> - All values submitted for the matching metric added together. You can use this
         *        statistic to determine the total volume of a metric.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Average</code> - The value of Sum / SampleCount during the specified period. By comparing this
         *        statistic with the Minimum and Maximum values, you can determine the full scope of a metric and how
         *        close the average use is to the Minimum and Maximum values. This comparison helps you to know when to
         *        increase or decrease your resources.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SampleCount</code> - The count, or number, of data points used for the statistical calculation.
         *        </p>
         *        </li>
         * @see MetricStatistic
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricStatistic
         */
        Builder statistic(MetricStatistic statistic);

        /**
         * <p>
         * The name of the metric associated with the alarm.
         * </p>
         * 
         * @param metricName
         *        The name of the metric associated with the alarm.
         * @see MetricName
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricName
         */
        Builder metricName(String metricName);

        /**
         * <p>
         * The name of the metric associated with the alarm.
         * </p>
         * 
         * @param metricName
         *        The name of the metric associated with the alarm.
         * @see MetricName
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricName
         */
        Builder metricName(MetricName metricName);

        /**
         * <p>
         * The current state of the alarm.
         * </p>
         * <p>
         * An alarm has the following possible states:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ALARM</code> - The metric is outside of the defined threshold.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INSUFFICIENT_DATA</code> - The alarm has just started, the metric is not available, or not enough data
         * is available for the metric to determine the alarm state.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OK</code> - The metric is within the defined threshold.
         * </p>
         * </li>
         * </ul>
         * 
         * @param state
         *        The current state of the alarm.</p>
         *        <p>
         *        An alarm has the following possible states:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ALARM</code> - The metric is outside of the defined threshold.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INSUFFICIENT_DATA</code> - The alarm has just started, the metric is not available, or not
         *        enough data is available for the metric to determine the alarm state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OK</code> - The metric is within the defined threshold.
         *        </p>
         *        </li>
         * @see AlarmState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AlarmState
         */
        Builder state(String state);

        /**
         * <p>
         * The current state of the alarm.
         * </p>
         * <p>
         * An alarm has the following possible states:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ALARM</code> - The metric is outside of the defined threshold.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INSUFFICIENT_DATA</code> - The alarm has just started, the metric is not available, or not enough data
         * is available for the metric to determine the alarm state.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OK</code> - The metric is within the defined threshold.
         * </p>
         * </li>
         * </ul>
         * 
         * @param state
         *        The current state of the alarm.</p>
         *        <p>
         *        An alarm has the following possible states:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ALARM</code> - The metric is outside of the defined threshold.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INSUFFICIENT_DATA</code> - The alarm has just started, the metric is not available, or not
         *        enough data is available for the metric to determine the alarm state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OK</code> - The metric is within the defined threshold.
         *        </p>
         *        </li>
         * @see AlarmState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AlarmState
         */
        Builder state(AlarmState state);

        /**
         * <p>
         * The unit of the metric associated with the alarm.
         * </p>
         * 
         * @param unit
         *        The unit of the metric associated with the alarm.
         * @see MetricUnit
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricUnit
         */
        Builder unit(String unit);

        /**
         * <p>
         * The unit of the metric associated with the alarm.
         * </p>
         * 
         * @param unit
         *        The unit of the metric associated with the alarm.
         * @see MetricUnit
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricUnit
         */
        Builder unit(MetricUnit unit);

        /**
         * <p>
         * The contact protocols for the alarm, such as <code>Email</code>, <code>SMS</code> (text messaging), or both.
         * </p>
         * 
         * @param contactProtocols
         *        The contact protocols for the alarm, such as <code>Email</code>, <code>SMS</code> (text messaging), or
         *        both.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder contactProtocolsWithStrings(Collection<String> contactProtocols);

        /**
         * <p>
         * The contact protocols for the alarm, such as <code>Email</code>, <code>SMS</code> (text messaging), or both.
         * </p>
         * 
         * @param contactProtocols
         *        The contact protocols for the alarm, such as <code>Email</code>, <code>SMS</code> (text messaging), or
         *        both.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder contactProtocolsWithStrings(String... contactProtocols);

        /**
         * <p>
         * The contact protocols for the alarm, such as <code>Email</code>, <code>SMS</code> (text messaging), or both.
         * </p>
         * 
         * @param contactProtocols
         *        The contact protocols for the alarm, such as <code>Email</code>, <code>SMS</code> (text messaging), or
         *        both.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder contactProtocols(Collection<ContactProtocol> contactProtocols);

        /**
         * <p>
         * The contact protocols for the alarm, such as <code>Email</code>, <code>SMS</code> (text messaging), or both.
         * </p>
         * 
         * @param contactProtocols
         *        The contact protocols for the alarm, such as <code>Email</code>, <code>SMS</code> (text messaging), or
         *        both.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder contactProtocols(ContactProtocol... contactProtocols);

        /**
         * <p>
         * The alarm states that trigger a notification.
         * </p>
         * 
         * @param notificationTriggers
         *        The alarm states that trigger a notification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationTriggersWithStrings(Collection<String> notificationTriggers);

        /**
         * <p>
         * The alarm states that trigger a notification.
         * </p>
         * 
         * @param notificationTriggers
         *        The alarm states that trigger a notification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationTriggersWithStrings(String... notificationTriggers);

        /**
         * <p>
         * The alarm states that trigger a notification.
         * </p>
         * 
         * @param notificationTriggers
         *        The alarm states that trigger a notification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationTriggers(Collection<AlarmState> notificationTriggers);

        /**
         * <p>
         * The alarm states that trigger a notification.
         * </p>
         * 
         * @param notificationTriggers
         *        The alarm states that trigger a notification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationTriggers(AlarmState... notificationTriggers);

        /**
         * <p>
         * Indicates whether the alarm is enabled.
         * </p>
         * 
         * @param notificationEnabled
         *        Indicates whether the alarm is enabled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationEnabled(Boolean notificationEnabled);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String arn;

        private Instant createdAt;

        private ResourceLocation location;

        private String resourceType;

        private String supportCode;

        private MonitoredResourceInfo monitoredResourceInfo;

        private String comparisonOperator;

        private Integer evaluationPeriods;

        private Integer period;

        private Double threshold;

        private Integer datapointsToAlarm;

        private String treatMissingData;

        private String statistic;

        private String metricName;

        private String state;

        private String unit;

        private List<String> contactProtocols = DefaultSdkAutoConstructList.getInstance();

        private List<String> notificationTriggers = DefaultSdkAutoConstructList.getInstance();

        private Boolean notificationEnabled;

        private BuilderImpl() {
        }

        private BuilderImpl(Alarm model) {
            name(model.name);
            arn(model.arn);
            createdAt(model.createdAt);
            location(model.location);
            resourceType(model.resourceType);
            supportCode(model.supportCode);
            monitoredResourceInfo(model.monitoredResourceInfo);
            comparisonOperator(model.comparisonOperator);
            evaluationPeriods(model.evaluationPeriods);
            period(model.period);
            threshold(model.threshold);
            datapointsToAlarm(model.datapointsToAlarm);
            treatMissingData(model.treatMissingData);
            statistic(model.statistic);
            metricName(model.metricName);
            state(model.state);
            unit(model.unit);
            contactProtocolsWithStrings(model.contactProtocols);
            notificationTriggersWithStrings(model.notificationTriggers);
            notificationEnabled(model.notificationEnabled);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final ResourceLocation.Builder getLocation() {
            return location != null ? location.toBuilder() : null;
        }

        public final void setLocation(ResourceLocation.BuilderImpl location) {
            this.location = location != null ? location.build() : null;
        }

        @Override
        public final Builder location(ResourceLocation location) {
            this.location = location;
            return this;
        }

        public final String getResourceType() {
            return resourceType;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        @Override
        public final Builder resourceType(ResourceType resourceType) {
            this.resourceType(resourceType == null ? null : resourceType.toString());
            return this;
        }

        public final String getSupportCode() {
            return supportCode;
        }

        public final void setSupportCode(String supportCode) {
            this.supportCode = supportCode;
        }

        @Override
        public final Builder supportCode(String supportCode) {
            this.supportCode = supportCode;
            return this;
        }

        public final MonitoredResourceInfo.Builder getMonitoredResourceInfo() {
            return monitoredResourceInfo != null ? monitoredResourceInfo.toBuilder() : null;
        }

        public final void setMonitoredResourceInfo(MonitoredResourceInfo.BuilderImpl monitoredResourceInfo) {
            this.monitoredResourceInfo = monitoredResourceInfo != null ? monitoredResourceInfo.build() : null;
        }

        @Override
        public final Builder monitoredResourceInfo(MonitoredResourceInfo monitoredResourceInfo) {
            this.monitoredResourceInfo = monitoredResourceInfo;
            return this;
        }

        public final String getComparisonOperator() {
            return comparisonOperator;
        }

        public final void setComparisonOperator(String comparisonOperator) {
            this.comparisonOperator = comparisonOperator;
        }

        @Override
        public final Builder comparisonOperator(String comparisonOperator) {
            this.comparisonOperator = comparisonOperator;
            return this;
        }

        @Override
        public final Builder comparisonOperator(ComparisonOperator comparisonOperator) {
            this.comparisonOperator(comparisonOperator == null ? null : comparisonOperator.toString());
            return this;
        }

        public final Integer getEvaluationPeriods() {
            return evaluationPeriods;
        }

        public final void setEvaluationPeriods(Integer evaluationPeriods) {
            this.evaluationPeriods = evaluationPeriods;
        }

        @Override
        public final Builder evaluationPeriods(Integer evaluationPeriods) {
            this.evaluationPeriods = evaluationPeriods;
            return this;
        }

        public final Integer getPeriod() {
            return period;
        }

        public final void setPeriod(Integer period) {
            this.period = period;
        }

        @Override
        public final Builder period(Integer period) {
            this.period = period;
            return this;
        }

        public final Double getThreshold() {
            return threshold;
        }

        public final void setThreshold(Double threshold) {
            this.threshold = threshold;
        }

        @Override
        public final Builder threshold(Double threshold) {
            this.threshold = threshold;
            return this;
        }

        public final Integer getDatapointsToAlarm() {
            return datapointsToAlarm;
        }

        public final void setDatapointsToAlarm(Integer datapointsToAlarm) {
            this.datapointsToAlarm = datapointsToAlarm;
        }

        @Override
        public final Builder datapointsToAlarm(Integer datapointsToAlarm) {
            this.datapointsToAlarm = datapointsToAlarm;
            return this;
        }

        public final String getTreatMissingData() {
            return treatMissingData;
        }

        public final void setTreatMissingData(String treatMissingData) {
            this.treatMissingData = treatMissingData;
        }

        @Override
        public final Builder treatMissingData(String treatMissingData) {
            this.treatMissingData = treatMissingData;
            return this;
        }

        @Override
        public final Builder treatMissingData(TreatMissingData treatMissingData) {
            this.treatMissingData(treatMissingData == null ? null : treatMissingData.toString());
            return this;
        }

        public final String getStatistic() {
            return statistic;
        }

        public final void setStatistic(String statistic) {
            this.statistic = statistic;
        }

        @Override
        public final Builder statistic(String statistic) {
            this.statistic = statistic;
            return this;
        }

        @Override
        public final Builder statistic(MetricStatistic statistic) {
            this.statistic(statistic == null ? null : statistic.toString());
            return this;
        }

        public final String getMetricName() {
            return metricName;
        }

        public final void setMetricName(String metricName) {
            this.metricName = metricName;
        }

        @Override
        public final Builder metricName(String metricName) {
            this.metricName = metricName;
            return this;
        }

        @Override
        public final Builder metricName(MetricName metricName) {
            this.metricName(metricName == null ? null : metricName.toString());
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(AlarmState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final String getUnit() {
            return unit;
        }

        public final void setUnit(String unit) {
            this.unit = unit;
        }

        @Override
        public final Builder unit(String unit) {
            this.unit = unit;
            return this;
        }

        @Override
        public final Builder unit(MetricUnit unit) {
            this.unit(unit == null ? null : unit.toString());
            return this;
        }

        public final Collection<String> getContactProtocols() {
            if (contactProtocols instanceof SdkAutoConstructList) {
                return null;
            }
            return contactProtocols;
        }

        public final void setContactProtocols(Collection<String> contactProtocols) {
            this.contactProtocols = ContactProtocolsListCopier.copy(contactProtocols);
        }

        @Override
        public final Builder contactProtocolsWithStrings(Collection<String> contactProtocols) {
            this.contactProtocols = ContactProtocolsListCopier.copy(contactProtocols);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder contactProtocolsWithStrings(String... contactProtocols) {
            contactProtocolsWithStrings(Arrays.asList(contactProtocols));
            return this;
        }

        @Override
        public final Builder contactProtocols(Collection<ContactProtocol> contactProtocols) {
            this.contactProtocols = ContactProtocolsListCopier.copyEnumToString(contactProtocols);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder contactProtocols(ContactProtocol... contactProtocols) {
            contactProtocols(Arrays.asList(contactProtocols));
            return this;
        }

        public final Collection<String> getNotificationTriggers() {
            if (notificationTriggers instanceof SdkAutoConstructList) {
                return null;
            }
            return notificationTriggers;
        }

        public final void setNotificationTriggers(Collection<String> notificationTriggers) {
            this.notificationTriggers = NotificationTriggerListCopier.copy(notificationTriggers);
        }

        @Override
        public final Builder notificationTriggersWithStrings(Collection<String> notificationTriggers) {
            this.notificationTriggers = NotificationTriggerListCopier.copy(notificationTriggers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder notificationTriggersWithStrings(String... notificationTriggers) {
            notificationTriggersWithStrings(Arrays.asList(notificationTriggers));
            return this;
        }

        @Override
        public final Builder notificationTriggers(Collection<AlarmState> notificationTriggers) {
            this.notificationTriggers = NotificationTriggerListCopier.copyEnumToString(notificationTriggers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder notificationTriggers(AlarmState... notificationTriggers) {
            notificationTriggers(Arrays.asList(notificationTriggers));
            return this;
        }

        public final Boolean getNotificationEnabled() {
            return notificationEnabled;
        }

        public final void setNotificationEnabled(Boolean notificationEnabled) {
            this.notificationEnabled = notificationEnabled;
        }

        @Override
        public final Builder notificationEnabled(Boolean notificationEnabled) {
            this.notificationEnabled = notificationEnabled;
            return this;
        }

        @Override
        public Alarm build() {
            return new Alarm(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
