/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lightsail.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a block storage disk.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Disk implements SdkPojo, Serializable, ToCopyableBuilder<Disk.Builder, Disk> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(Disk::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(Disk::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<String> SUPPORT_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("supportCode").getter(getter(Disk::supportCode)).setter(setter(Builder::supportCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("supportCode").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt").getter(getter(Disk::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build()).build();

    private static final SdkField<ResourceLocation> LOCATION_FIELD = SdkField
            .<ResourceLocation> builder(MarshallingType.SDK_POJO).memberName("location").getter(getter(Disk::location))
            .setter(setter(Builder::location)).constructor(ResourceLocation::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("location").build()).build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("resourceType").getter(getter(Disk::resourceTypeAsString)).setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceType").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("tags")
            .getter(getter(Disk::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<AddOn>> ADD_ONS_FIELD = SdkField
            .<List<AddOn>> builder(MarshallingType.LIST)
            .memberName("addOns")
            .getter(getter(Disk::addOns))
            .setter(setter(Builder::addOns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("addOns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AddOn> builder(MarshallingType.SDK_POJO)
                                            .constructor(AddOn::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> SIZE_IN_GB_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("sizeInGb").getter(getter(Disk::sizeInGb)).setter(setter(Builder::sizeInGb))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sizeInGb").build()).build();

    private static final SdkField<Boolean> IS_SYSTEM_DISK_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("isSystemDisk").getter(getter(Disk::isSystemDisk)).setter(setter(Builder::isSystemDisk))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("isSystemDisk").build()).build();

    private static final SdkField<Integer> IOPS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("iops")
            .getter(getter(Disk::iops)).setter(setter(Builder::iops))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("iops").build()).build();

    private static final SdkField<String> PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("path")
            .getter(getter(Disk::path)).setter(setter(Builder::path))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("path").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("state")
            .getter(getter(Disk::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("state").build()).build();

    private static final SdkField<String> ATTACHED_TO_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("attachedTo").getter(getter(Disk::attachedTo)).setter(setter(Builder::attachedTo))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("attachedTo").build()).build();

    private static final SdkField<Boolean> IS_ATTACHED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("isAttached").getter(getter(Disk::isAttached)).setter(setter(Builder::isAttached))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("isAttached").build()).build();

    private static final SdkField<String> ATTACHMENT_STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("attachmentState").getter(getter(Disk::attachmentState)).setter(setter(Builder::attachmentState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("attachmentState").build()).build();

    private static final SdkField<Integer> GB_IN_USE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("gbInUse").getter(getter(Disk::gbInUse)).setter(setter(Builder::gbInUse))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("gbInUse").build()).build();

    private static final SdkField<String> AUTO_MOUNT_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("autoMountStatus").getter(getter(Disk::autoMountStatusAsString)).setter(setter(Builder::autoMountStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("autoMountStatus").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, ARN_FIELD,
            SUPPORT_CODE_FIELD, CREATED_AT_FIELD, LOCATION_FIELD, RESOURCE_TYPE_FIELD, TAGS_FIELD, ADD_ONS_FIELD,
            SIZE_IN_GB_FIELD, IS_SYSTEM_DISK_FIELD, IOPS_FIELD, PATH_FIELD, STATE_FIELD, ATTACHED_TO_FIELD, IS_ATTACHED_FIELD,
            ATTACHMENT_STATE_FIELD, GB_IN_USE_FIELD, AUTO_MOUNT_STATUS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String arn;

    private final String supportCode;

    private final Instant createdAt;

    private final ResourceLocation location;

    private final String resourceType;

    private final List<Tag> tags;

    private final List<AddOn> addOns;

    private final Integer sizeInGb;

    private final Boolean isSystemDisk;

    private final Integer iops;

    private final String path;

    private final String state;

    private final String attachedTo;

    private final Boolean isAttached;

    private final String attachmentState;

    private final Integer gbInUse;

    private final String autoMountStatus;

    private Disk(BuilderImpl builder) {
        this.name = builder.name;
        this.arn = builder.arn;
        this.supportCode = builder.supportCode;
        this.createdAt = builder.createdAt;
        this.location = builder.location;
        this.resourceType = builder.resourceType;
        this.tags = builder.tags;
        this.addOns = builder.addOns;
        this.sizeInGb = builder.sizeInGb;
        this.isSystemDisk = builder.isSystemDisk;
        this.iops = builder.iops;
        this.path = builder.path;
        this.state = builder.state;
        this.attachedTo = builder.attachedTo;
        this.isAttached = builder.isAttached;
        this.attachmentState = builder.attachmentState;
        this.gbInUse = builder.gbInUse;
        this.autoMountStatus = builder.autoMountStatus;
    }

    /**
     * <p>
     * The unique name of the disk.
     * </p>
     * 
     * @return The unique name of the disk.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the disk.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the disk.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The support code. Include this code in your email to support when you have questions about an instance or another
     * resource in Lightsail. This code enables our support team to look up your Lightsail information more easily.
     * </p>
     * 
     * @return The support code. Include this code in your email to support when you have questions about an instance or
     *         another resource in Lightsail. This code enables our support team to look up your Lightsail information
     *         more easily.
     */
    public final String supportCode() {
        return supportCode;
    }

    /**
     * <p>
     * The date when the disk was created.
     * </p>
     * 
     * @return The date when the disk was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The AWS Region and Availability Zone where the disk is located.
     * </p>
     * 
     * @return The AWS Region and Availability Zone where the disk is located.
     */
    public final ResourceLocation location() {
        return location;
    }

    /**
     * <p>
     * The Lightsail resource type (<code>Disk</code>).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The Lightsail resource type (<code>Disk</code>).
     * @see ResourceType
     */
    public final ResourceType resourceType() {
        return ResourceType.fromValue(resourceType);
    }

    /**
     * <p>
     * The Lightsail resource type (<code>Disk</code>).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The Lightsail resource type (<code>Disk</code>).
     * @see ResourceType
     */
    public final String resourceTypeAsString() {
        return resourceType;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tag keys and optional values for the resource. For more information about tags in Lightsail, see the <a
     * href="https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-tags">Amazon Lightsail Developer
     * Guide</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tag keys and optional values for the resource. For more information about tags in Lightsail, see the
     *         <a href="https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-tags">Amazon Lightsail
     *         Developer Guide</a>.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * For responses, this returns true if the service returned a value for the AddOns property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAddOns() {
        return addOns != null && !(addOns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of objects representing the add-ons enabled on the disk.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAddOns} method.
     * </p>
     * 
     * @return An array of objects representing the add-ons enabled on the disk.
     */
    public final List<AddOn> addOns() {
        return addOns;
    }

    /**
     * <p>
     * The size of the disk in GB.
     * </p>
     * 
     * @return The size of the disk in GB.
     */
    public final Integer sizeInGb() {
        return sizeInGb;
    }

    /**
     * <p>
     * A Boolean value indicating whether this disk is a system disk (has an operating system loaded on it).
     * </p>
     * 
     * @return A Boolean value indicating whether this disk is a system disk (has an operating system loaded on it).
     */
    public final Boolean isSystemDisk() {
        return isSystemDisk;
    }

    /**
     * <p>
     * The input/output operations per second (IOPS) of the disk.
     * </p>
     * 
     * @return The input/output operations per second (IOPS) of the disk.
     */
    public final Integer iops() {
        return iops;
    }

    /**
     * <p>
     * The disk path.
     * </p>
     * 
     * @return The disk path.
     */
    public final String path() {
        return path;
    }

    /**
     * <p>
     * Describes the status of the disk.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link DiskState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return Describes the status of the disk.
     * @see DiskState
     */
    public final DiskState state() {
        return DiskState.fromValue(state);
    }

    /**
     * <p>
     * Describes the status of the disk.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link DiskState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return Describes the status of the disk.
     * @see DiskState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The resources to which the disk is attached.
     * </p>
     * 
     * @return The resources to which the disk is attached.
     */
    public final String attachedTo() {
        return attachedTo;
    }

    /**
     * <p>
     * A Boolean value indicating whether the disk is attached.
     * </p>
     * 
     * @return A Boolean value indicating whether the disk is attached.
     */
    public final Boolean isAttached() {
        return isAttached;
    }

    /**
     * <p>
     * (Discontinued) The attachment state of the disk.
     * </p>
     * <note>
     * <p>
     * In releases prior to November 14, 2017, this parameter returned <code>attached</code> for system disks in the API
     * response. It is now discontinued, but still included in the response. Use <code>isAttached</code> instead.
     * </p>
     * </note>
     * 
     * @return (Discontinued) The attachment state of the disk.</p> <note>
     *         <p>
     *         In releases prior to November 14, 2017, this parameter returned <code>attached</code> for system disks in
     *         the API response. It is now discontinued, but still included in the response. Use <code>isAttached</code>
     *         instead.
     *         </p>
     * @deprecated
     */
    @Deprecated
    public final String attachmentState() {
        return attachmentState;
    }

    /**
     * <p>
     * (Discontinued) The number of GB in use by the disk.
     * </p>
     * <note>
     * <p>
     * In releases prior to November 14, 2017, this parameter was not included in the API response. It is now
     * discontinued.
     * </p>
     * </note>
     * 
     * @return (Discontinued) The number of GB in use by the disk.</p> <note>
     *         <p>
     *         In releases prior to November 14, 2017, this parameter was not included in the API response. It is now
     *         discontinued.
     *         </p>
     * @deprecated
     */
    @Deprecated
    public final Integer gbInUse() {
        return gbInUse;
    }

    /**
     * <p>
     * The status of automatically mounting a storage disk to a virtual computer.
     * </p>
     * <important>
     * <p>
     * This parameter only applies to Lightsail for Research resources.
     * </p>
     * </important>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #autoMountStatus}
     * will return {@link AutoMountStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #autoMountStatusAsString}.
     * </p>
     * 
     * @return The status of automatically mounting a storage disk to a virtual computer.</p> <important>
     *         <p>
     *         This parameter only applies to Lightsail for Research resources.
     *         </p>
     * @see AutoMountStatus
     */
    public final AutoMountStatus autoMountStatus() {
        return AutoMountStatus.fromValue(autoMountStatus);
    }

    /**
     * <p>
     * The status of automatically mounting a storage disk to a virtual computer.
     * </p>
     * <important>
     * <p>
     * This parameter only applies to Lightsail for Research resources.
     * </p>
     * </important>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #autoMountStatus}
     * will return {@link AutoMountStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #autoMountStatusAsString}.
     * </p>
     * 
     * @return The status of automatically mounting a storage disk to a virtual computer.</p> <important>
     *         <p>
     *         This parameter only applies to Lightsail for Research resources.
     *         </p>
     * @see AutoMountStatus
     */
    public final String autoMountStatusAsString() {
        return autoMountStatus;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(supportCode());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(location());
        hashCode = 31 * hashCode + Objects.hashCode(resourceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasAddOns() ? addOns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(sizeInGb());
        hashCode = 31 * hashCode + Objects.hashCode(isSystemDisk());
        hashCode = 31 * hashCode + Objects.hashCode(iops());
        hashCode = 31 * hashCode + Objects.hashCode(path());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(attachedTo());
        hashCode = 31 * hashCode + Objects.hashCode(isAttached());
        hashCode = 31 * hashCode + Objects.hashCode(attachmentState());
        hashCode = 31 * hashCode + Objects.hashCode(gbInUse());
        hashCode = 31 * hashCode + Objects.hashCode(autoMountStatusAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Disk)) {
            return false;
        }
        Disk other = (Disk) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(arn(), other.arn())
                && Objects.equals(supportCode(), other.supportCode()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(location(), other.location())
                && Objects.equals(resourceTypeAsString(), other.resourceTypeAsString()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && hasAddOns() == other.hasAddOns()
                && Objects.equals(addOns(), other.addOns()) && Objects.equals(sizeInGb(), other.sizeInGb())
                && Objects.equals(isSystemDisk(), other.isSystemDisk()) && Objects.equals(iops(), other.iops())
                && Objects.equals(path(), other.path()) && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(attachedTo(), other.attachedTo()) && Objects.equals(isAttached(), other.isAttached())
                && Objects.equals(attachmentState(), other.attachmentState()) && Objects.equals(gbInUse(), other.gbInUse())
                && Objects.equals(autoMountStatusAsString(), other.autoMountStatusAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Disk").add("Name", name()).add("Arn", arn()).add("SupportCode", supportCode())
                .add("CreatedAt", createdAt()).add("Location", location()).add("ResourceType", resourceTypeAsString())
                .add("Tags", hasTags() ? tags() : null).add("AddOns", hasAddOns() ? addOns() : null).add("SizeInGb", sizeInGb())
                .add("IsSystemDisk", isSystemDisk()).add("Iops", iops()).add("Path", path()).add("State", stateAsString())
                .add("AttachedTo", attachedTo()).add("IsAttached", isAttached()).add("AttachmentState", attachmentState())
                .add("GbInUse", gbInUse()).add("AutoMountStatus", autoMountStatusAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "supportCode":
            return Optional.ofNullable(clazz.cast(supportCode()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "location":
            return Optional.ofNullable(clazz.cast(location()));
        case "resourceType":
            return Optional.ofNullable(clazz.cast(resourceTypeAsString()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "addOns":
            return Optional.ofNullable(clazz.cast(addOns()));
        case "sizeInGb":
            return Optional.ofNullable(clazz.cast(sizeInGb()));
        case "isSystemDisk":
            return Optional.ofNullable(clazz.cast(isSystemDisk()));
        case "iops":
            return Optional.ofNullable(clazz.cast(iops()));
        case "path":
            return Optional.ofNullable(clazz.cast(path()));
        case "state":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "attachedTo":
            return Optional.ofNullable(clazz.cast(attachedTo()));
        case "isAttached":
            return Optional.ofNullable(clazz.cast(isAttached()));
        case "attachmentState":
            return Optional.ofNullable(clazz.cast(attachmentState()));
        case "gbInUse":
            return Optional.ofNullable(clazz.cast(gbInUse()));
        case "autoMountStatus":
            return Optional.ofNullable(clazz.cast(autoMountStatusAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("name", NAME_FIELD);
        map.put("arn", ARN_FIELD);
        map.put("supportCode", SUPPORT_CODE_FIELD);
        map.put("createdAt", CREATED_AT_FIELD);
        map.put("location", LOCATION_FIELD);
        map.put("resourceType", RESOURCE_TYPE_FIELD);
        map.put("tags", TAGS_FIELD);
        map.put("addOns", ADD_ONS_FIELD);
        map.put("sizeInGb", SIZE_IN_GB_FIELD);
        map.put("isSystemDisk", IS_SYSTEM_DISK_FIELD);
        map.put("iops", IOPS_FIELD);
        map.put("path", PATH_FIELD);
        map.put("state", STATE_FIELD);
        map.put("attachedTo", ATTACHED_TO_FIELD);
        map.put("isAttached", IS_ATTACHED_FIELD);
        map.put("attachmentState", ATTACHMENT_STATE_FIELD);
        map.put("gbInUse", GB_IN_USE_FIELD);
        map.put("autoMountStatus", AUTO_MOUNT_STATUS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Disk, T> g) {
        return obj -> g.apply((Disk) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Disk> {
        /**
         * <p>
         * The unique name of the disk.
         * </p>
         * 
         * @param name
         *        The unique name of the disk.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the disk.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the disk.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The support code. Include this code in your email to support when you have questions about an instance or
         * another resource in Lightsail. This code enables our support team to look up your Lightsail information more
         * easily.
         * </p>
         * 
         * @param supportCode
         *        The support code. Include this code in your email to support when you have questions about an instance
         *        or another resource in Lightsail. This code enables our support team to look up your Lightsail
         *        information more easily.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportCode(String supportCode);

        /**
         * <p>
         * The date when the disk was created.
         * </p>
         * 
         * @param createdAt
         *        The date when the disk was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The AWS Region and Availability Zone where the disk is located.
         * </p>
         * 
         * @param location
         *        The AWS Region and Availability Zone where the disk is located.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder location(ResourceLocation location);

        /**
         * <p>
         * The AWS Region and Availability Zone where the disk is located.
         * </p>
         * This is a convenience method that creates an instance of the {@link ResourceLocation.Builder} avoiding the
         * need to create one manually via {@link ResourceLocation#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ResourceLocation.Builder#build()} is called immediately and its
         * result is passed to {@link #location(ResourceLocation)}.
         * 
         * @param location
         *        a consumer that will call methods on {@link ResourceLocation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #location(ResourceLocation)
         */
        default Builder location(Consumer<ResourceLocation.Builder> location) {
            return location(ResourceLocation.builder().applyMutation(location).build());
        }

        /**
         * <p>
         * The Lightsail resource type (<code>Disk</code>).
         * </p>
         * 
         * @param resourceType
         *        The Lightsail resource type (<code>Disk</code>).
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * The Lightsail resource type (<code>Disk</code>).
         * </p>
         * 
         * @param resourceType
         *        The Lightsail resource type (<code>Disk</code>).
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder resourceType(ResourceType resourceType);

        /**
         * <p>
         * The tag keys and optional values for the resource. For more information about tags in Lightsail, see the <a
         * href="https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-tags">Amazon Lightsail
         * Developer Guide</a>.
         * </p>
         * 
         * @param tags
         *        The tag keys and optional values for the resource. For more information about tags in Lightsail, see
         *        the <a href="https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-tags">Amazon
         *        Lightsail Developer Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The tag keys and optional values for the resource. For more information about tags in Lightsail, see the <a
         * href="https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-tags">Amazon Lightsail
         * Developer Guide</a>.
         * </p>
         * 
         * @param tags
         *        The tag keys and optional values for the resource. For more information about tags in Lightsail, see
         *        the <a href="https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-tags">Amazon
         *        Lightsail Developer Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The tag keys and optional values for the resource. For more information about tags in Lightsail, see the <a
         * href="https://lightsail.aws.amazon.com/ls/docs/en_us/articles/amazon-lightsail-tags">Amazon Lightsail
         * Developer Guide</a>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.lightsail.model.Tag.Builder} avoiding the need to create one manually
         * via {@link software.amazon.awssdk.services.lightsail.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.lightsail.model.Tag.Builder#build()} is called immediately and its
         * result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.lightsail.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * An array of objects representing the add-ons enabled on the disk.
         * </p>
         * 
         * @param addOns
         *        An array of objects representing the add-ons enabled on the disk.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder addOns(Collection<AddOn> addOns);

        /**
         * <p>
         * An array of objects representing the add-ons enabled on the disk.
         * </p>
         * 
         * @param addOns
         *        An array of objects representing the add-ons enabled on the disk.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder addOns(AddOn... addOns);

        /**
         * <p>
         * An array of objects representing the add-ons enabled on the disk.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.lightsail.model.AddOn.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.lightsail.model.AddOn#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.lightsail.model.AddOn.Builder#build()} is called immediately and its
         * result is passed to {@link #addOns(List<AddOn>)}.
         * 
         * @param addOns
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.lightsail.model.AddOn.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #addOns(java.util.Collection<AddOn>)
         */
        Builder addOns(Consumer<AddOn.Builder>... addOns);

        /**
         * <p>
         * The size of the disk in GB.
         * </p>
         * 
         * @param sizeInGb
         *        The size of the disk in GB.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sizeInGb(Integer sizeInGb);

        /**
         * <p>
         * A Boolean value indicating whether this disk is a system disk (has an operating system loaded on it).
         * </p>
         * 
         * @param isSystemDisk
         *        A Boolean value indicating whether this disk is a system disk (has an operating system loaded on it).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isSystemDisk(Boolean isSystemDisk);

        /**
         * <p>
         * The input/output operations per second (IOPS) of the disk.
         * </p>
         * 
         * @param iops
         *        The input/output operations per second (IOPS) of the disk.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder iops(Integer iops);

        /**
         * <p>
         * The disk path.
         * </p>
         * 
         * @param path
         *        The disk path.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder path(String path);

        /**
         * <p>
         * Describes the status of the disk.
         * </p>
         * 
         * @param state
         *        Describes the status of the disk.
         * @see DiskState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DiskState
         */
        Builder state(String state);

        /**
         * <p>
         * Describes the status of the disk.
         * </p>
         * 
         * @param state
         *        Describes the status of the disk.
         * @see DiskState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DiskState
         */
        Builder state(DiskState state);

        /**
         * <p>
         * The resources to which the disk is attached.
         * </p>
         * 
         * @param attachedTo
         *        The resources to which the disk is attached.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attachedTo(String attachedTo);

        /**
         * <p>
         * A Boolean value indicating whether the disk is attached.
         * </p>
         * 
         * @param isAttached
         *        A Boolean value indicating whether the disk is attached.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isAttached(Boolean isAttached);

        /**
         * <p>
         * (Discontinued) The attachment state of the disk.
         * </p>
         * <note>
         * <p>
         * In releases prior to November 14, 2017, this parameter returned <code>attached</code> for system disks in the
         * API response. It is now discontinued, but still included in the response. Use <code>isAttached</code>
         * instead.
         * </p>
         * </note>
         * 
         * @param attachmentState
         *        (Discontinued) The attachment state of the disk.</p> <note>
         *        <p>
         *        In releases prior to November 14, 2017, this parameter returned <code>attached</code> for system disks
         *        in the API response. It is now discontinued, but still included in the response. Use
         *        <code>isAttached</code> instead.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         * @deprecated
         */
        @Deprecated
        Builder attachmentState(String attachmentState);

        /**
         * <p>
         * (Discontinued) The number of GB in use by the disk.
         * </p>
         * <note>
         * <p>
         * In releases prior to November 14, 2017, this parameter was not included in the API response. It is now
         * discontinued.
         * </p>
         * </note>
         * 
         * @param gbInUse
         *        (Discontinued) The number of GB in use by the disk.</p> <note>
         *        <p>
         *        In releases prior to November 14, 2017, this parameter was not included in the API response. It is now
         *        discontinued.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         * @deprecated
         */
        @Deprecated
        Builder gbInUse(Integer gbInUse);

        /**
         * <p>
         * The status of automatically mounting a storage disk to a virtual computer.
         * </p>
         * <important>
         * <p>
         * This parameter only applies to Lightsail for Research resources.
         * </p>
         * </important>
         * 
         * @param autoMountStatus
         *        The status of automatically mounting a storage disk to a virtual computer.</p> <important>
         *        <p>
         *        This parameter only applies to Lightsail for Research resources.
         *        </p>
         * @see AutoMountStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AutoMountStatus
         */
        Builder autoMountStatus(String autoMountStatus);

        /**
         * <p>
         * The status of automatically mounting a storage disk to a virtual computer.
         * </p>
         * <important>
         * <p>
         * This parameter only applies to Lightsail for Research resources.
         * </p>
         * </important>
         * 
         * @param autoMountStatus
         *        The status of automatically mounting a storage disk to a virtual computer.</p> <important>
         *        <p>
         *        This parameter only applies to Lightsail for Research resources.
         *        </p>
         * @see AutoMountStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AutoMountStatus
         */
        Builder autoMountStatus(AutoMountStatus autoMountStatus);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String arn;

        private String supportCode;

        private Instant createdAt;

        private ResourceLocation location;

        private String resourceType;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private List<AddOn> addOns = DefaultSdkAutoConstructList.getInstance();

        private Integer sizeInGb;

        private Boolean isSystemDisk;

        private Integer iops;

        private String path;

        private String state;

        private String attachedTo;

        private Boolean isAttached;

        private String attachmentState;

        private Integer gbInUse;

        private String autoMountStatus;

        private BuilderImpl() {
        }

        private BuilderImpl(Disk model) {
            name(model.name);
            arn(model.arn);
            supportCode(model.supportCode);
            createdAt(model.createdAt);
            location(model.location);
            resourceType(model.resourceType);
            tags(model.tags);
            addOns(model.addOns);
            sizeInGb(model.sizeInGb);
            isSystemDisk(model.isSystemDisk);
            iops(model.iops);
            path(model.path);
            state(model.state);
            attachedTo(model.attachedTo);
            isAttached(model.isAttached);
            attachmentState(model.attachmentState);
            gbInUse(model.gbInUse);
            autoMountStatus(model.autoMountStatus);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getSupportCode() {
            return supportCode;
        }

        public final void setSupportCode(String supportCode) {
            this.supportCode = supportCode;
        }

        @Override
        public final Builder supportCode(String supportCode) {
            this.supportCode = supportCode;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final ResourceLocation.Builder getLocation() {
            return location != null ? location.toBuilder() : null;
        }

        public final void setLocation(ResourceLocation.BuilderImpl location) {
            this.location = location != null ? location.build() : null;
        }

        @Override
        public final Builder location(ResourceLocation location) {
            this.location = location;
            return this;
        }

        public final String getResourceType() {
            return resourceType;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        @Override
        public final Builder resourceType(ResourceType resourceType) {
            this.resourceType(resourceType == null ? null : resourceType.toString());
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<AddOn.Builder> getAddOns() {
            List<AddOn.Builder> result = AddOnListCopier.copyToBuilder(this.addOns);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAddOns(Collection<AddOn.BuilderImpl> addOns) {
            this.addOns = AddOnListCopier.copyFromBuilder(addOns);
        }

        @Override
        public final Builder addOns(Collection<AddOn> addOns) {
            this.addOns = AddOnListCopier.copy(addOns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder addOns(AddOn... addOns) {
            addOns(Arrays.asList(addOns));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder addOns(Consumer<AddOn.Builder>... addOns) {
            addOns(Stream.of(addOns).map(c -> AddOn.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Integer getSizeInGb() {
            return sizeInGb;
        }

        public final void setSizeInGb(Integer sizeInGb) {
            this.sizeInGb = sizeInGb;
        }

        @Override
        public final Builder sizeInGb(Integer sizeInGb) {
            this.sizeInGb = sizeInGb;
            return this;
        }

        public final Boolean getIsSystemDisk() {
            return isSystemDisk;
        }

        public final void setIsSystemDisk(Boolean isSystemDisk) {
            this.isSystemDisk = isSystemDisk;
        }

        @Override
        public final Builder isSystemDisk(Boolean isSystemDisk) {
            this.isSystemDisk = isSystemDisk;
            return this;
        }

        public final Integer getIops() {
            return iops;
        }

        public final void setIops(Integer iops) {
            this.iops = iops;
        }

        @Override
        public final Builder iops(Integer iops) {
            this.iops = iops;
            return this;
        }

        public final String getPath() {
            return path;
        }

        public final void setPath(String path) {
            this.path = path;
        }

        @Override
        public final Builder path(String path) {
            this.path = path;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(DiskState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final String getAttachedTo() {
            return attachedTo;
        }

        public final void setAttachedTo(String attachedTo) {
            this.attachedTo = attachedTo;
        }

        @Override
        public final Builder attachedTo(String attachedTo) {
            this.attachedTo = attachedTo;
            return this;
        }

        public final Boolean getIsAttached() {
            return isAttached;
        }

        public final void setIsAttached(Boolean isAttached) {
            this.isAttached = isAttached;
        }

        @Override
        public final Builder isAttached(Boolean isAttached) {
            this.isAttached = isAttached;
            return this;
        }

        @Deprecated
        public final String getAttachmentState() {
            return attachmentState;
        }

        @Deprecated
        public final void setAttachmentState(String attachmentState) {
            this.attachmentState = attachmentState;
        }

        @Override
        @Deprecated
        public final Builder attachmentState(String attachmentState) {
            this.attachmentState = attachmentState;
            return this;
        }

        @Deprecated
        public final Integer getGbInUse() {
            return gbInUse;
        }

        @Deprecated
        public final void setGbInUse(Integer gbInUse) {
            this.gbInUse = gbInUse;
        }

        @Override
        @Deprecated
        public final Builder gbInUse(Integer gbInUse) {
            this.gbInUse = gbInUse;
            return this;
        }

        public final String getAutoMountStatus() {
            return autoMountStatus;
        }

        public final void setAutoMountStatus(String autoMountStatus) {
            this.autoMountStatus = autoMountStatus;
        }

        @Override
        public final Builder autoMountStatus(String autoMountStatus) {
            this.autoMountStatus = autoMountStatus;
            return this;
        }

        @Override
        public final Builder autoMountStatus(AutoMountStatus autoMountStatus) {
            this.autoMountStatus(autoMountStatus == null ? null : autoMountStatus.toString());
            return this;
        }

        @Override
        public Disk build() {
            return new Disk(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
