/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lightsail.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the query string parameters that an Amazon Lightsail content delivery network (CDN) distribution to bases
 * caching on.
 * </p>
 * <p>
 * For the query strings that you specify, your distribution caches separate versions of the specified content based on
 * the query string values in viewer requests.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class QueryStringObject implements SdkPojo, Serializable,
        ToCopyableBuilder<QueryStringObject.Builder, QueryStringObject> {
    private static final SdkField<Boolean> OPTION_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("option").getter(getter(QueryStringObject::option)).setter(setter(Builder::option))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("option").build()).build();

    private static final SdkField<List<String>> QUERY_STRINGS_ALLOW_LIST_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("queryStringsAllowList")
            .getter(getter(QueryStringObject::queryStringsAllowList))
            .setter(setter(Builder::queryStringsAllowList))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("queryStringsAllowList").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(OPTION_FIELD,
            QUERY_STRINGS_ALLOW_LIST_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Boolean option;

    private final List<String> queryStringsAllowList;

    private QueryStringObject(BuilderImpl builder) {
        this.option = builder.option;
        this.queryStringsAllowList = builder.queryStringsAllowList;
    }

    /**
     * <p>
     * Indicates whether the distribution forwards and caches based on query strings.
     * </p>
     * 
     * @return Indicates whether the distribution forwards and caches based on query strings.
     */
    public final Boolean option() {
        return option;
    }

    /**
     * For responses, this returns true if the service returned a value for the QueryStringsAllowList property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasQueryStringsAllowList() {
        return queryStringsAllowList != null && !(queryStringsAllowList instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The specific query strings that the distribution forwards to the origin.
     * </p>
     * <p>
     * Your distribution will cache content based on the specified query strings.
     * </p>
     * <p>
     * If the <code>option</code> parameter is true, then your distribution forwards all query strings, regardless of
     * what you specify using the <code>queryStringsAllowList</code> parameter.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasQueryStringsAllowList} method.
     * </p>
     * 
     * @return The specific query strings that the distribution forwards to the origin.</p>
     *         <p>
     *         Your distribution will cache content based on the specified query strings.
     *         </p>
     *         <p>
     *         If the <code>option</code> parameter is true, then your distribution forwards all query strings,
     *         regardless of what you specify using the <code>queryStringsAllowList</code> parameter.
     */
    public final List<String> queryStringsAllowList() {
        return queryStringsAllowList;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(option());
        hashCode = 31 * hashCode + Objects.hashCode(hasQueryStringsAllowList() ? queryStringsAllowList() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof QueryStringObject)) {
            return false;
        }
        QueryStringObject other = (QueryStringObject) obj;
        return Objects.equals(option(), other.option()) && hasQueryStringsAllowList() == other.hasQueryStringsAllowList()
                && Objects.equals(queryStringsAllowList(), other.queryStringsAllowList());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("QueryStringObject").add("Option", option())
                .add("QueryStringsAllowList", hasQueryStringsAllowList() ? queryStringsAllowList() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "option":
            return Optional.ofNullable(clazz.cast(option()));
        case "queryStringsAllowList":
            return Optional.ofNullable(clazz.cast(queryStringsAllowList()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("option", OPTION_FIELD);
        map.put("queryStringsAllowList", QUERY_STRINGS_ALLOW_LIST_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<QueryStringObject, T> g) {
        return obj -> g.apply((QueryStringObject) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, QueryStringObject> {
        /**
         * <p>
         * Indicates whether the distribution forwards and caches based on query strings.
         * </p>
         * 
         * @param option
         *        Indicates whether the distribution forwards and caches based on query strings.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder option(Boolean option);

        /**
         * <p>
         * The specific query strings that the distribution forwards to the origin.
         * </p>
         * <p>
         * Your distribution will cache content based on the specified query strings.
         * </p>
         * <p>
         * If the <code>option</code> parameter is true, then your distribution forwards all query strings, regardless
         * of what you specify using the <code>queryStringsAllowList</code> parameter.
         * </p>
         * 
         * @param queryStringsAllowList
         *        The specific query strings that the distribution forwards to the origin.</p>
         *        <p>
         *        Your distribution will cache content based on the specified query strings.
         *        </p>
         *        <p>
         *        If the <code>option</code> parameter is true, then your distribution forwards all query strings,
         *        regardless of what you specify using the <code>queryStringsAllowList</code> parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder queryStringsAllowList(Collection<String> queryStringsAllowList);

        /**
         * <p>
         * The specific query strings that the distribution forwards to the origin.
         * </p>
         * <p>
         * Your distribution will cache content based on the specified query strings.
         * </p>
         * <p>
         * If the <code>option</code> parameter is true, then your distribution forwards all query strings, regardless
         * of what you specify using the <code>queryStringsAllowList</code> parameter.
         * </p>
         * 
         * @param queryStringsAllowList
         *        The specific query strings that the distribution forwards to the origin.</p>
         *        <p>
         *        Your distribution will cache content based on the specified query strings.
         *        </p>
         *        <p>
         *        If the <code>option</code> parameter is true, then your distribution forwards all query strings,
         *        regardless of what you specify using the <code>queryStringsAllowList</code> parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder queryStringsAllowList(String... queryStringsAllowList);
    }

    static final class BuilderImpl implements Builder {
        private Boolean option;

        private List<String> queryStringsAllowList = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(QueryStringObject model) {
            option(model.option);
            queryStringsAllowList(model.queryStringsAllowList);
        }

        public final Boolean getOption() {
            return option;
        }

        public final void setOption(Boolean option) {
            this.option = option;
        }

        @Override
        public final Builder option(Boolean option) {
            this.option = option;
            return this;
        }

        public final Collection<String> getQueryStringsAllowList() {
            if (queryStringsAllowList instanceof SdkAutoConstructList) {
                return null;
            }
            return queryStringsAllowList;
        }

        public final void setQueryStringsAllowList(Collection<String> queryStringsAllowList) {
            this.queryStringsAllowList = StringListCopier.copy(queryStringsAllowList);
        }

        @Override
        public final Builder queryStringsAllowList(Collection<String> queryStringsAllowList) {
            this.queryStringsAllowList = StringListCopier.copy(queryStringsAllowList);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder queryStringsAllowList(String... queryStringsAllowList) {
            queryStringsAllowList(Arrays.asList(queryStringsAllowList));
            return this;
        }

        @Override
        public QueryStringObject build() {
            return new QueryStringObject(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
