/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lightsail.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Returns information that was submitted during the <code>SetupInstanceHttps</code> request. Email information is
 * redacted for privacy.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SetupRequest implements SdkPojo, Serializable, ToCopyableBuilder<SetupRequest.Builder, SetupRequest> {
    private static final SdkField<String> INSTANCE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("instanceName").getter(getter(SetupRequest::instanceName)).setter(setter(Builder::instanceName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("instanceName").build()).build();

    private static final SdkField<List<String>> DOMAIN_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("domainNames")
            .getter(getter(SetupRequest::domainNames))
            .setter(setter(Builder::domainNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("domainNames").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> CERTIFICATE_PROVIDER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("certificateProvider").getter(getter(SetupRequest::certificateProviderAsString))
            .setter(setter(Builder::certificateProvider))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("certificateProvider").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INSTANCE_NAME_FIELD,
            DOMAIN_NAMES_FIELD, CERTIFICATE_PROVIDER_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String instanceName;

    private final List<String> domainNames;

    private final String certificateProvider;

    private SetupRequest(BuilderImpl builder) {
        this.instanceName = builder.instanceName;
        this.domainNames = builder.domainNames;
        this.certificateProvider = builder.certificateProvider;
    }

    /**
     * <p>
     * The name of the Lightsail instance.
     * </p>
     * 
     * @return The name of the Lightsail instance.
     */
    public final String instanceName() {
        return instanceName;
    }

    /**
     * For responses, this returns true if the service returned a value for the DomainNames property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasDomainNames() {
        return domainNames != null && !(domainNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The name of the domain and subdomains that the SSL/TLS certificate secures.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDomainNames} method.
     * </p>
     * 
     * @return The name of the domain and subdomains that the SSL/TLS certificate secures.
     */
    public final List<String> domainNames() {
        return domainNames;
    }

    /**
     * <p>
     * The Certificate Authority (CA) that issues the SSL/TLS certificate.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #certificateProvider} will return {@link CertificateProvider#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #certificateProviderAsString}.
     * </p>
     * 
     * @return The Certificate Authority (CA) that issues the SSL/TLS certificate.
     * @see CertificateProvider
     */
    public final CertificateProvider certificateProvider() {
        return CertificateProvider.fromValue(certificateProvider);
    }

    /**
     * <p>
     * The Certificate Authority (CA) that issues the SSL/TLS certificate.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #certificateProvider} will return {@link CertificateProvider#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #certificateProviderAsString}.
     * </p>
     * 
     * @return The Certificate Authority (CA) that issues the SSL/TLS certificate.
     * @see CertificateProvider
     */
    public final String certificateProviderAsString() {
        return certificateProvider;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(instanceName());
        hashCode = 31 * hashCode + Objects.hashCode(hasDomainNames() ? domainNames() : null);
        hashCode = 31 * hashCode + Objects.hashCode(certificateProviderAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SetupRequest)) {
            return false;
        }
        SetupRequest other = (SetupRequest) obj;
        return Objects.equals(instanceName(), other.instanceName()) && hasDomainNames() == other.hasDomainNames()
                && Objects.equals(domainNames(), other.domainNames())
                && Objects.equals(certificateProviderAsString(), other.certificateProviderAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SetupRequest").add("InstanceName", instanceName())
                .add("DomainNames", hasDomainNames() ? domainNames() : null)
                .add("CertificateProvider", certificateProviderAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "instanceName":
            return Optional.ofNullable(clazz.cast(instanceName()));
        case "domainNames":
            return Optional.ofNullable(clazz.cast(domainNames()));
        case "certificateProvider":
            return Optional.ofNullable(clazz.cast(certificateProviderAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("instanceName", INSTANCE_NAME_FIELD);
        map.put("domainNames", DOMAIN_NAMES_FIELD);
        map.put("certificateProvider", CERTIFICATE_PROVIDER_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<SetupRequest, T> g) {
        return obj -> g.apply((SetupRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SetupRequest> {
        /**
         * <p>
         * The name of the Lightsail instance.
         * </p>
         * 
         * @param instanceName
         *        The name of the Lightsail instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceName(String instanceName);

        /**
         * <p>
         * The name of the domain and subdomains that the SSL/TLS certificate secures.
         * </p>
         * 
         * @param domainNames
         *        The name of the domain and subdomains that the SSL/TLS certificate secures.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainNames(Collection<String> domainNames);

        /**
         * <p>
         * The name of the domain and subdomains that the SSL/TLS certificate secures.
         * </p>
         * 
         * @param domainNames
         *        The name of the domain and subdomains that the SSL/TLS certificate secures.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainNames(String... domainNames);

        /**
         * <p>
         * The Certificate Authority (CA) that issues the SSL/TLS certificate.
         * </p>
         * 
         * @param certificateProvider
         *        The Certificate Authority (CA) that issues the SSL/TLS certificate.
         * @see CertificateProvider
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CertificateProvider
         */
        Builder certificateProvider(String certificateProvider);

        /**
         * <p>
         * The Certificate Authority (CA) that issues the SSL/TLS certificate.
         * </p>
         * 
         * @param certificateProvider
         *        The Certificate Authority (CA) that issues the SSL/TLS certificate.
         * @see CertificateProvider
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CertificateProvider
         */
        Builder certificateProvider(CertificateProvider certificateProvider);
    }

    static final class BuilderImpl implements Builder {
        private String instanceName;

        private List<String> domainNames = DefaultSdkAutoConstructList.getInstance();

        private String certificateProvider;

        private BuilderImpl() {
        }

        private BuilderImpl(SetupRequest model) {
            instanceName(model.instanceName);
            domainNames(model.domainNames);
            certificateProvider(model.certificateProvider);
        }

        public final String getInstanceName() {
            return instanceName;
        }

        public final void setInstanceName(String instanceName) {
            this.instanceName = instanceName;
        }

        @Override
        public final Builder instanceName(String instanceName) {
            this.instanceName = instanceName;
            return this;
        }

        public final Collection<String> getDomainNames() {
            if (domainNames instanceof SdkAutoConstructList) {
                return null;
            }
            return domainNames;
        }

        public final void setDomainNames(Collection<String> domainNames) {
            this.domainNames = SetupDomainNameListCopier.copy(domainNames);
        }

        @Override
        public final Builder domainNames(Collection<String> domainNames) {
            this.domainNames = SetupDomainNameListCopier.copy(domainNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder domainNames(String... domainNames) {
            domainNames(Arrays.asList(domainNames));
            return this;
        }

        public final String getCertificateProvider() {
            return certificateProvider;
        }

        public final void setCertificateProvider(String certificateProvider) {
            this.certificateProvider = certificateProvider;
        }

        @Override
        public final Builder certificateProvider(String certificateProvider) {
            this.certificateProvider = certificateProvider;
            return this;
        }

        @Override
        public final Builder certificateProvider(CertificateProvider certificateProvider) {
            this.certificateProvider(certificateProvider == null ? null : certificateProvider.toString());
            return this;
        }

        @Override
        public SetupRequest build() {
            return new SetupRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
