/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lightsail.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the public SSH host keys or the RDP certificate.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class HostKeyAttributes implements SdkPojo, Serializable,
        ToCopyableBuilder<HostKeyAttributes.Builder, HostKeyAttributes> {
    private static final SdkField<String> ALGORITHM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(HostKeyAttributes::algorithm)).setter(setter(Builder::algorithm))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("algorithm").build()).build();

    private static final SdkField<String> PUBLIC_KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(HostKeyAttributes::publicKey)).setter(setter(Builder::publicKey))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("publicKey").build()).build();

    private static final SdkField<Instant> WITNESSED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(HostKeyAttributes::witnessedAt)).setter(setter(Builder::witnessedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("witnessedAt").build()).build();

    private static final SdkField<String> FINGERPRINT_SHA1_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(HostKeyAttributes::fingerprintSHA1)).setter(setter(Builder::fingerprintSHA1))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("fingerprintSHA1").build()).build();

    private static final SdkField<String> FINGERPRINT_SHA256_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(HostKeyAttributes::fingerprintSHA256)).setter(setter(Builder::fingerprintSHA256))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("fingerprintSHA256").build()).build();

    private static final SdkField<Instant> NOT_VALID_BEFORE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(HostKeyAttributes::notValidBefore)).setter(setter(Builder::notValidBefore))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("notValidBefore").build()).build();

    private static final SdkField<Instant> NOT_VALID_AFTER_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(HostKeyAttributes::notValidAfter)).setter(setter(Builder::notValidAfter))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("notValidAfter").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ALGORITHM_FIELD,
            PUBLIC_KEY_FIELD, WITNESSED_AT_FIELD, FINGERPRINT_SHA1_FIELD, FINGERPRINT_SHA256_FIELD, NOT_VALID_BEFORE_FIELD,
            NOT_VALID_AFTER_FIELD));

    private static final long serialVersionUID = 1L;

    private final String algorithm;

    private final String publicKey;

    private final Instant witnessedAt;

    private final String fingerprintSHA1;

    private final String fingerprintSHA256;

    private final Instant notValidBefore;

    private final Instant notValidAfter;

    private HostKeyAttributes(BuilderImpl builder) {
        this.algorithm = builder.algorithm;
        this.publicKey = builder.publicKey;
        this.witnessedAt = builder.witnessedAt;
        this.fingerprintSHA1 = builder.fingerprintSHA1;
        this.fingerprintSHA256 = builder.fingerprintSHA256;
        this.notValidBefore = builder.notValidBefore;
        this.notValidAfter = builder.notValidAfter;
    }

    /**
     * <p>
     * The SSH host key algorithm or the RDP certificate format.
     * </p>
     * <p>
     * For SSH host keys, the algorithm may be <code>ssh-rsa</code>, <code>ecdsa-sha2-nistp256</code>,
     * <code>ssh-ed25519</code>, etc. For RDP certificates, the algorithm is always <code>x509-cert</code>.
     * </p>
     * 
     * @return The SSH host key algorithm or the RDP certificate format.</p>
     *         <p>
     *         For SSH host keys, the algorithm may be <code>ssh-rsa</code>, <code>ecdsa-sha2-nistp256</code>,
     *         <code>ssh-ed25519</code>, etc. For RDP certificates, the algorithm is always <code>x509-cert</code>.
     */
    public String algorithm() {
        return algorithm;
    }

    /**
     * <p>
     * The public SSH host key or the RDP certificate.
     * </p>
     * 
     * @return The public SSH host key or the RDP certificate.
     */
    public String publicKey() {
        return publicKey;
    }

    /**
     * <p>
     * The time that the SSH host key or RDP certificate was recorded by Lightsail.
     * </p>
     * 
     * @return The time that the SSH host key or RDP certificate was recorded by Lightsail.
     */
    public Instant witnessedAt() {
        return witnessedAt;
    }

    /**
     * <p>
     * The SHA-1 fingerprint of the returned SSH host key or RDP certificate.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Example of an SHA-1 SSH fingerprint:
     * </p>
     * <p>
     * <code>SHA1:1CHH6FaAaXjtFOsR/t83vf91SR0</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Example of an SHA-1 RDP fingerprint:
     * </p>
     * <p>
     * <code>af:34:51:fe:09:f0:e0:da:b8:4e:56:ca:60:c2:10:ff:38:06:db:45</code>
     * </p>
     * </li>
     * </ul>
     * 
     * @return The SHA-1 fingerprint of the returned SSH host key or RDP certificate.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Example of an SHA-1 SSH fingerprint:
     *         </p>
     *         <p>
     *         <code>SHA1:1CHH6FaAaXjtFOsR/t83vf91SR0</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Example of an SHA-1 RDP fingerprint:
     *         </p>
     *         <p>
     *         <code>af:34:51:fe:09:f0:e0:da:b8:4e:56:ca:60:c2:10:ff:38:06:db:45</code>
     *         </p>
     *         </li>
     */
    public String fingerprintSHA1() {
        return fingerprintSHA1;
    }

    /**
     * <p>
     * The SHA-256 fingerprint of the returned SSH host key or RDP certificate.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Example of an SHA-256 SSH fingerprint:
     * </p>
     * <p>
     * <code>SHA256:KTsMnRBh1IhD17HpdfsbzeGA4jOijm5tyXsMjKVbB8o</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Example of an SHA-256 RDP fingerprint:
     * </p>
     * <p>
     * <code>03:9b:36:9f:4b:de:4e:61:70:fc:7c:c9:78:e7:d2:1a:1c:25:a8:0c:91:f6:7c:e4:d6:a0:85:c8:b4:53:99:68</code>
     * </p>
     * </li>
     * </ul>
     * 
     * @return The SHA-256 fingerprint of the returned SSH host key or RDP certificate.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Example of an SHA-256 SSH fingerprint:
     *         </p>
     *         <p>
     *         <code>SHA256:KTsMnRBh1IhD17HpdfsbzeGA4jOijm5tyXsMjKVbB8o</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Example of an SHA-256 RDP fingerprint:
     *         </p>
     *         <p>
     *         <code>03:9b:36:9f:4b:de:4e:61:70:fc:7c:c9:78:e7:d2:1a:1c:25:a8:0c:91:f6:7c:e4:d6:a0:85:c8:b4:53:99:68</code>
     *         </p>
     *         </li>
     */
    public String fingerprintSHA256() {
        return fingerprintSHA256;
    }

    /**
     * <p>
     * The returned RDP certificate is valid after this point in time.
     * </p>
     * <p>
     * This value is listed only for RDP certificates.
     * </p>
     * 
     * @return The returned RDP certificate is valid after this point in time.</p>
     *         <p>
     *         This value is listed only for RDP certificates.
     */
    public Instant notValidBefore() {
        return notValidBefore;
    }

    /**
     * <p>
     * The returned RDP certificate is not valid after this point in time.
     * </p>
     * <p>
     * This value is listed only for RDP certificates.
     * </p>
     * 
     * @return The returned RDP certificate is not valid after this point in time.</p>
     *         <p>
     *         This value is listed only for RDP certificates.
     */
    public Instant notValidAfter() {
        return notValidAfter;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(algorithm());
        hashCode = 31 * hashCode + Objects.hashCode(publicKey());
        hashCode = 31 * hashCode + Objects.hashCode(witnessedAt());
        hashCode = 31 * hashCode + Objects.hashCode(fingerprintSHA1());
        hashCode = 31 * hashCode + Objects.hashCode(fingerprintSHA256());
        hashCode = 31 * hashCode + Objects.hashCode(notValidBefore());
        hashCode = 31 * hashCode + Objects.hashCode(notValidAfter());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof HostKeyAttributes)) {
            return false;
        }
        HostKeyAttributes other = (HostKeyAttributes) obj;
        return Objects.equals(algorithm(), other.algorithm()) && Objects.equals(publicKey(), other.publicKey())
                && Objects.equals(witnessedAt(), other.witnessedAt())
                && Objects.equals(fingerprintSHA1(), other.fingerprintSHA1())
                && Objects.equals(fingerprintSHA256(), other.fingerprintSHA256())
                && Objects.equals(notValidBefore(), other.notValidBefore())
                && Objects.equals(notValidAfter(), other.notValidAfter());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("HostKeyAttributes").add("Algorithm", algorithm()).add("PublicKey", publicKey())
                .add("WitnessedAt", witnessedAt()).add("FingerprintSHA1", fingerprintSHA1())
                .add("FingerprintSHA256", fingerprintSHA256()).add("NotValidBefore", notValidBefore())
                .add("NotValidAfter", notValidAfter()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "algorithm":
            return Optional.ofNullable(clazz.cast(algorithm()));
        case "publicKey":
            return Optional.ofNullable(clazz.cast(publicKey()));
        case "witnessedAt":
            return Optional.ofNullable(clazz.cast(witnessedAt()));
        case "fingerprintSHA1":
            return Optional.ofNullable(clazz.cast(fingerprintSHA1()));
        case "fingerprintSHA256":
            return Optional.ofNullable(clazz.cast(fingerprintSHA256()));
        case "notValidBefore":
            return Optional.ofNullable(clazz.cast(notValidBefore()));
        case "notValidAfter":
            return Optional.ofNullable(clazz.cast(notValidAfter()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<HostKeyAttributes, T> g) {
        return obj -> g.apply((HostKeyAttributes) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, HostKeyAttributes> {
        /**
         * <p>
         * The SSH host key algorithm or the RDP certificate format.
         * </p>
         * <p>
         * For SSH host keys, the algorithm may be <code>ssh-rsa</code>, <code>ecdsa-sha2-nistp256</code>,
         * <code>ssh-ed25519</code>, etc. For RDP certificates, the algorithm is always <code>x509-cert</code>.
         * </p>
         * 
         * @param algorithm
         *        The SSH host key algorithm or the RDP certificate format.</p>
         *        <p>
         *        For SSH host keys, the algorithm may be <code>ssh-rsa</code>, <code>ecdsa-sha2-nistp256</code>,
         *        <code>ssh-ed25519</code>, etc. For RDP certificates, the algorithm is always <code>x509-cert</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder algorithm(String algorithm);

        /**
         * <p>
         * The public SSH host key or the RDP certificate.
         * </p>
         * 
         * @param publicKey
         *        The public SSH host key or the RDP certificate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder publicKey(String publicKey);

        /**
         * <p>
         * The time that the SSH host key or RDP certificate was recorded by Lightsail.
         * </p>
         * 
         * @param witnessedAt
         *        The time that the SSH host key or RDP certificate was recorded by Lightsail.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder witnessedAt(Instant witnessedAt);

        /**
         * <p>
         * The SHA-1 fingerprint of the returned SSH host key or RDP certificate.
         * </p>
         * <ul>
         * <li>
         * <p>
         * Example of an SHA-1 SSH fingerprint:
         * </p>
         * <p>
         * <code>SHA1:1CHH6FaAaXjtFOsR/t83vf91SR0</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Example of an SHA-1 RDP fingerprint:
         * </p>
         * <p>
         * <code>af:34:51:fe:09:f0:e0:da:b8:4e:56:ca:60:c2:10:ff:38:06:db:45</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param fingerprintSHA1
         *        The SHA-1 fingerprint of the returned SSH host key or RDP certificate.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Example of an SHA-1 SSH fingerprint:
         *        </p>
         *        <p>
         *        <code>SHA1:1CHH6FaAaXjtFOsR/t83vf91SR0</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Example of an SHA-1 RDP fingerprint:
         *        </p>
         *        <p>
         *        <code>af:34:51:fe:09:f0:e0:da:b8:4e:56:ca:60:c2:10:ff:38:06:db:45</code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fingerprintSHA1(String fingerprintSHA1);

        /**
         * <p>
         * The SHA-256 fingerprint of the returned SSH host key or RDP certificate.
         * </p>
         * <ul>
         * <li>
         * <p>
         * Example of an SHA-256 SSH fingerprint:
         * </p>
         * <p>
         * <code>SHA256:KTsMnRBh1IhD17HpdfsbzeGA4jOijm5tyXsMjKVbB8o</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Example of an SHA-256 RDP fingerprint:
         * </p>
         * <p>
         * <code>03:9b:36:9f:4b:de:4e:61:70:fc:7c:c9:78:e7:d2:1a:1c:25:a8:0c:91:f6:7c:e4:d6:a0:85:c8:b4:53:99:68</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param fingerprintSHA256
         *        The SHA-256 fingerprint of the returned SSH host key or RDP certificate.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Example of an SHA-256 SSH fingerprint:
         *        </p>
         *        <p>
         *        <code>SHA256:KTsMnRBh1IhD17HpdfsbzeGA4jOijm5tyXsMjKVbB8o</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Example of an SHA-256 RDP fingerprint:
         *        </p>
         *        <p>
         *        <code>03:9b:36:9f:4b:de:4e:61:70:fc:7c:c9:78:e7:d2:1a:1c:25:a8:0c:91:f6:7c:e4:d6:a0:85:c8:b4:53:99:68</code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fingerprintSHA256(String fingerprintSHA256);

        /**
         * <p>
         * The returned RDP certificate is valid after this point in time.
         * </p>
         * <p>
         * This value is listed only for RDP certificates.
         * </p>
         * 
         * @param notValidBefore
         *        The returned RDP certificate is valid after this point in time.</p>
         *        <p>
         *        This value is listed only for RDP certificates.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notValidBefore(Instant notValidBefore);

        /**
         * <p>
         * The returned RDP certificate is not valid after this point in time.
         * </p>
         * <p>
         * This value is listed only for RDP certificates.
         * </p>
         * 
         * @param notValidAfter
         *        The returned RDP certificate is not valid after this point in time.</p>
         *        <p>
         *        This value is listed only for RDP certificates.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notValidAfter(Instant notValidAfter);
    }

    static final class BuilderImpl implements Builder {
        private String algorithm;

        private String publicKey;

        private Instant witnessedAt;

        private String fingerprintSHA1;

        private String fingerprintSHA256;

        private Instant notValidBefore;

        private Instant notValidAfter;

        private BuilderImpl() {
        }

        private BuilderImpl(HostKeyAttributes model) {
            algorithm(model.algorithm);
            publicKey(model.publicKey);
            witnessedAt(model.witnessedAt);
            fingerprintSHA1(model.fingerprintSHA1);
            fingerprintSHA256(model.fingerprintSHA256);
            notValidBefore(model.notValidBefore);
            notValidAfter(model.notValidAfter);
        }

        public final String getAlgorithm() {
            return algorithm;
        }

        @Override
        public final Builder algorithm(String algorithm) {
            this.algorithm = algorithm;
            return this;
        }

        public final void setAlgorithm(String algorithm) {
            this.algorithm = algorithm;
        }

        public final String getPublicKey() {
            return publicKey;
        }

        @Override
        public final Builder publicKey(String publicKey) {
            this.publicKey = publicKey;
            return this;
        }

        public final void setPublicKey(String publicKey) {
            this.publicKey = publicKey;
        }

        public final Instant getWitnessedAt() {
            return witnessedAt;
        }

        @Override
        public final Builder witnessedAt(Instant witnessedAt) {
            this.witnessedAt = witnessedAt;
            return this;
        }

        public final void setWitnessedAt(Instant witnessedAt) {
            this.witnessedAt = witnessedAt;
        }

        public final String getFingerprintSHA1() {
            return fingerprintSHA1;
        }

        @Override
        public final Builder fingerprintSHA1(String fingerprintSHA1) {
            this.fingerprintSHA1 = fingerprintSHA1;
            return this;
        }

        public final void setFingerprintSHA1(String fingerprintSHA1) {
            this.fingerprintSHA1 = fingerprintSHA1;
        }

        public final String getFingerprintSHA256() {
            return fingerprintSHA256;
        }

        @Override
        public final Builder fingerprintSHA256(String fingerprintSHA256) {
            this.fingerprintSHA256 = fingerprintSHA256;
            return this;
        }

        public final void setFingerprintSHA256(String fingerprintSHA256) {
            this.fingerprintSHA256 = fingerprintSHA256;
        }

        public final Instant getNotValidBefore() {
            return notValidBefore;
        }

        @Override
        public final Builder notValidBefore(Instant notValidBefore) {
            this.notValidBefore = notValidBefore;
            return this;
        }

        public final void setNotValidBefore(Instant notValidBefore) {
            this.notValidBefore = notValidBefore;
        }

        public final Instant getNotValidAfter() {
            return notValidAfter;
        }

        @Override
        public final Builder notValidAfter(Instant notValidAfter) {
            this.notValidAfter = notValidAfter;
            return this;
        }

        public final void setNotValidAfter(Instant notValidAfter) {
            this.notValidAfter = notValidAfter;
        }

        @Override
        public HostKeyAttributes build() {
            return new HostKeyAttributes(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
