/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lightsail.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an event for a database.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RelationalDatabaseEvent implements SdkPojo, Serializable,
        ToCopyableBuilder<RelationalDatabaseEvent.Builder, RelationalDatabaseEvent> {
    private static final SdkField<String> RESOURCE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(RelationalDatabaseEvent::resource)).setter(setter(Builder::resource))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resource").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(RelationalDatabaseEvent::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build()).build();

    private static final SdkField<String> MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(RelationalDatabaseEvent::message)).setter(setter(Builder::message))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("message").build()).build();

    private static final SdkField<List<String>> EVENT_CATEGORIES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(RelationalDatabaseEvent::eventCategories))
            .setter(setter(Builder::eventCategories))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("eventCategories").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RESOURCE_FIELD,
            CREATED_AT_FIELD, MESSAGE_FIELD, EVENT_CATEGORIES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String resource;

    private final Instant createdAt;

    private final String message;

    private final List<String> eventCategories;

    private RelationalDatabaseEvent(BuilderImpl builder) {
        this.resource = builder.resource;
        this.createdAt = builder.createdAt;
        this.message = builder.message;
        this.eventCategories = builder.eventCategories;
    }

    /**
     * <p>
     * The database that the database event relates to.
     * </p>
     * 
     * @return The database that the database event relates to.
     */
    public String resource() {
        return resource;
    }

    /**
     * <p>
     * The timestamp when the database event was created.
     * </p>
     * 
     * @return The timestamp when the database event was created.
     */
    public Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The message of the database event.
     * </p>
     * 
     * @return The message of the database event.
     */
    public String message() {
        return message;
    }

    /**
     * <p>
     * The category that the database event belongs to.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The category that the database event belongs to.
     */
    public List<String> eventCategories() {
        return eventCategories;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(resource());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(message());
        hashCode = 31 * hashCode + Objects.hashCode(eventCategories());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RelationalDatabaseEvent)) {
            return false;
        }
        RelationalDatabaseEvent other = (RelationalDatabaseEvent) obj;
        return Objects.equals(resource(), other.resource()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(message(), other.message()) && Objects.equals(eventCategories(), other.eventCategories());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("RelationalDatabaseEvent").add("Resource", resource()).add("CreatedAt", createdAt())
                .add("Message", message()).add("EventCategories", eventCategories()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "resource":
            return Optional.ofNullable(clazz.cast(resource()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "message":
            return Optional.ofNullable(clazz.cast(message()));
        case "eventCategories":
            return Optional.ofNullable(clazz.cast(eventCategories()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RelationalDatabaseEvent, T> g) {
        return obj -> g.apply((RelationalDatabaseEvent) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RelationalDatabaseEvent> {
        /**
         * <p>
         * The database that the database event relates to.
         * </p>
         * 
         * @param resource
         *        The database that the database event relates to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resource(String resource);

        /**
         * <p>
         * The timestamp when the database event was created.
         * </p>
         * 
         * @param createdAt
         *        The timestamp when the database event was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The message of the database event.
         * </p>
         * 
         * @param message
         *        The message of the database event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder message(String message);

        /**
         * <p>
         * The category that the database event belongs to.
         * </p>
         * 
         * @param eventCategories
         *        The category that the database event belongs to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventCategories(Collection<String> eventCategories);

        /**
         * <p>
         * The category that the database event belongs to.
         * </p>
         * 
         * @param eventCategories
         *        The category that the database event belongs to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventCategories(String... eventCategories);
    }

    static final class BuilderImpl implements Builder {
        private String resource;

        private Instant createdAt;

        private String message;

        private List<String> eventCategories = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(RelationalDatabaseEvent model) {
            resource(model.resource);
            createdAt(model.createdAt);
            message(model.message);
            eventCategories(model.eventCategories);
        }

        public final String getResource() {
            return resource;
        }

        @Override
        public final Builder resource(String resource) {
            this.resource = resource;
            return this;
        }

        public final void setResource(String resource) {
            this.resource = resource;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        public final String getMessage() {
            return message;
        }

        @Override
        public final Builder message(String message) {
            this.message = message;
            return this;
        }

        public final void setMessage(String message) {
            this.message = message;
        }

        public final Collection<String> getEventCategories() {
            return eventCategories;
        }

        @Override
        public final Builder eventCategories(Collection<String> eventCategories) {
            this.eventCategories = StringListCopier.copy(eventCategories);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder eventCategories(String... eventCategories) {
            eventCategories(Arrays.asList(eventCategories));
            return this;
        }

        public final void setEventCategories(Collection<String> eventCategories) {
            this.eventCategories = StringListCopier.copy(eventCategories);
        }

        @Override
        public RelationalDatabaseEvent build() {
            return new RelationalDatabaseEvent(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
