/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lightsail.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the snapshot of the virtual private server, or <i>instance</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InstanceSnapshot implements SdkPojo, Serializable,
        ToCopyableBuilder<InstanceSnapshot.Builder, InstanceSnapshot> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceSnapshot::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceSnapshot::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<String> SUPPORT_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceSnapshot::supportCode)).setter(setter(Builder::supportCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("supportCode").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(InstanceSnapshot::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build()).build();

    private static final SdkField<ResourceLocation> LOCATION_FIELD = SdkField
            .<ResourceLocation> builder(MarshallingType.SDK_POJO).getter(getter(InstanceSnapshot::location))
            .setter(setter(Builder::location)).constructor(ResourceLocation::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("location").build()).build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceSnapshot::resourceTypeAsString)).setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceType").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .getter(getter(InstanceSnapshot::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceSnapshot::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("state").build()).build();

    private static final SdkField<String> PROGRESS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceSnapshot::progress)).setter(setter(Builder::progress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("progress").build()).build();

    private static final SdkField<List<Disk>> FROM_ATTACHED_DISKS_FIELD = SdkField
            .<List<Disk>> builder(MarshallingType.LIST)
            .getter(getter(InstanceSnapshot::fromAttachedDisks))
            .setter(setter(Builder::fromAttachedDisks))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("fromAttachedDisks").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Disk> builder(MarshallingType.SDK_POJO)
                                            .constructor(Disk::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> FROM_INSTANCE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceSnapshot::fromInstanceName)).setter(setter(Builder::fromInstanceName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("fromInstanceName").build()).build();

    private static final SdkField<String> FROM_INSTANCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceSnapshot::fromInstanceArn)).setter(setter(Builder::fromInstanceArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("fromInstanceArn").build()).build();

    private static final SdkField<String> FROM_BLUEPRINT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceSnapshot::fromBlueprintId)).setter(setter(Builder::fromBlueprintId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("fromBlueprintId").build()).build();

    private static final SdkField<String> FROM_BUNDLE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceSnapshot::fromBundleId)).setter(setter(Builder::fromBundleId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("fromBundleId").build()).build();

    private static final SdkField<Integer> SIZE_IN_GB_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(InstanceSnapshot::sizeInGb)).setter(setter(Builder::sizeInGb))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sizeInGb").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, ARN_FIELD,
            SUPPORT_CODE_FIELD, CREATED_AT_FIELD, LOCATION_FIELD, RESOURCE_TYPE_FIELD, TAGS_FIELD, STATE_FIELD, PROGRESS_FIELD,
            FROM_ATTACHED_DISKS_FIELD, FROM_INSTANCE_NAME_FIELD, FROM_INSTANCE_ARN_FIELD, FROM_BLUEPRINT_ID_FIELD,
            FROM_BUNDLE_ID_FIELD, SIZE_IN_GB_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String arn;

    private final String supportCode;

    private final Instant createdAt;

    private final ResourceLocation location;

    private final String resourceType;

    private final List<Tag> tags;

    private final String state;

    private final String progress;

    private final List<Disk> fromAttachedDisks;

    private final String fromInstanceName;

    private final String fromInstanceArn;

    private final String fromBlueprintId;

    private final String fromBundleId;

    private final Integer sizeInGb;

    private InstanceSnapshot(BuilderImpl builder) {
        this.name = builder.name;
        this.arn = builder.arn;
        this.supportCode = builder.supportCode;
        this.createdAt = builder.createdAt;
        this.location = builder.location;
        this.resourceType = builder.resourceType;
        this.tags = builder.tags;
        this.state = builder.state;
        this.progress = builder.progress;
        this.fromAttachedDisks = builder.fromAttachedDisks;
        this.fromInstanceName = builder.fromInstanceName;
        this.fromInstanceArn = builder.fromInstanceArn;
        this.fromBlueprintId = builder.fromBlueprintId;
        this.fromBundleId = builder.fromBundleId;
        this.sizeInGb = builder.sizeInGb;
    }

    /**
     * <p>
     * The name of the snapshot.
     * </p>
     * 
     * @return The name of the snapshot.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the snapshot (e.g.,
     * <code>arn:aws:lightsail:us-east-2:123456789101:InstanceSnapshot/d23b5706-3322-4d83-81e5-12345EXAMPLE</code>).
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the snapshot (e.g.,
     *         <code>arn:aws:lightsail:us-east-2:123456789101:InstanceSnapshot/d23b5706-3322-4d83-81e5-12345EXAMPLE</code>
     *         ).
     */
    public String arn() {
        return arn;
    }

    /**
     * <p>
     * The support code. Include this code in your email to support when you have questions about an instance or another
     * resource in Lightsail. This code enables our support team to look up your Lightsail information more easily.
     * </p>
     * 
     * @return The support code. Include this code in your email to support when you have questions about an instance or
     *         another resource in Lightsail. This code enables our support team to look up your Lightsail information
     *         more easily.
     */
    public String supportCode() {
        return supportCode;
    }

    /**
     * <p>
     * The timestamp when the snapshot was created (e.g., <code>1479907467.024</code>).
     * </p>
     * 
     * @return The timestamp when the snapshot was created (e.g., <code>1479907467.024</code>).
     */
    public Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The region name and Availability Zone where you created the snapshot.
     * </p>
     * 
     * @return The region name and Availability Zone where you created the snapshot.
     */
    public ResourceLocation location() {
        return location;
    }

    /**
     * <p>
     * The type of resource (usually <code>InstanceSnapshot</code>).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The type of resource (usually <code>InstanceSnapshot</code>).
     * @see ResourceType
     */
    public ResourceType resourceType() {
        return ResourceType.fromValue(resourceType);
    }

    /**
     * <p>
     * The type of resource (usually <code>InstanceSnapshot</code>).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The type of resource (usually <code>InstanceSnapshot</code>).
     * @see ResourceType
     */
    public String resourceTypeAsString() {
        return resourceType;
    }

    /**
     * <p>
     * The tag keys and optional values for the resource. For more information about tags in Lightsail, see the <a
     * href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev Guide</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The tag keys and optional values for the resource. For more information about tags in Lightsail, see the
     *         <a href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev
     *         Guide</a>.
     */
    public List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * The state the snapshot is in.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link InstanceSnapshotState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state the snapshot is in.
     * @see InstanceSnapshotState
     */
    public InstanceSnapshotState state() {
        return InstanceSnapshotState.fromValue(state);
    }

    /**
     * <p>
     * The state the snapshot is in.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link InstanceSnapshotState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state the snapshot is in.
     * @see InstanceSnapshotState
     */
    public String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The progress of the snapshot.
     * </p>
     * 
     * @return The progress of the snapshot.
     */
    public String progress() {
        return progress;
    }

    /**
     * <p>
     * An array of disk objects containing information about all block storage disks.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return An array of disk objects containing information about all block storage disks.
     */
    public List<Disk> fromAttachedDisks() {
        return fromAttachedDisks;
    }

    /**
     * <p>
     * The instance from which the snapshot was created.
     * </p>
     * 
     * @return The instance from which the snapshot was created.
     */
    public String fromInstanceName() {
        return fromInstanceName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the instance from which the snapshot was created (e.g.,
     * <code>arn:aws:lightsail:us-east-2:123456789101:Instance/64b8404c-ccb1-430b-8daf-12345EXAMPLE</code>).
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the instance from which the snapshot was created (e.g.,
     *         <code>arn:aws:lightsail:us-east-2:123456789101:Instance/64b8404c-ccb1-430b-8daf-12345EXAMPLE</code>).
     */
    public String fromInstanceArn() {
        return fromInstanceArn;
    }

    /**
     * <p>
     * The blueprint ID from which you created the snapshot (e.g., <code>os_debian_8_3</code>). A blueprint is a virtual
     * private server (or <i>instance</i>) image used to create instances quickly.
     * </p>
     * 
     * @return The blueprint ID from which you created the snapshot (e.g., <code>os_debian_8_3</code>). A blueprint is a
     *         virtual private server (or <i>instance</i>) image used to create instances quickly.
     */
    public String fromBlueprintId() {
        return fromBlueprintId;
    }

    /**
     * <p>
     * The bundle ID from which you created the snapshot (e.g., <code>micro_1_0</code>).
     * </p>
     * 
     * @return The bundle ID from which you created the snapshot (e.g., <code>micro_1_0</code>).
     */
    public String fromBundleId() {
        return fromBundleId;
    }

    /**
     * <p>
     * The size in GB of the SSD.
     * </p>
     * 
     * @return The size in GB of the SSD.
     */
    public Integer sizeInGb() {
        return sizeInGb;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(supportCode());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(location());
        hashCode = 31 * hashCode + Objects.hashCode(resourceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(progress());
        hashCode = 31 * hashCode + Objects.hashCode(fromAttachedDisks());
        hashCode = 31 * hashCode + Objects.hashCode(fromInstanceName());
        hashCode = 31 * hashCode + Objects.hashCode(fromInstanceArn());
        hashCode = 31 * hashCode + Objects.hashCode(fromBlueprintId());
        hashCode = 31 * hashCode + Objects.hashCode(fromBundleId());
        hashCode = 31 * hashCode + Objects.hashCode(sizeInGb());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InstanceSnapshot)) {
            return false;
        }
        InstanceSnapshot other = (InstanceSnapshot) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(arn(), other.arn())
                && Objects.equals(supportCode(), other.supportCode()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(location(), other.location())
                && Objects.equals(resourceTypeAsString(), other.resourceTypeAsString()) && Objects.equals(tags(), other.tags())
                && Objects.equals(stateAsString(), other.stateAsString()) && Objects.equals(progress(), other.progress())
                && Objects.equals(fromAttachedDisks(), other.fromAttachedDisks())
                && Objects.equals(fromInstanceName(), other.fromInstanceName())
                && Objects.equals(fromInstanceArn(), other.fromInstanceArn())
                && Objects.equals(fromBlueprintId(), other.fromBlueprintId())
                && Objects.equals(fromBundleId(), other.fromBundleId()) && Objects.equals(sizeInGb(), other.sizeInGb());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("InstanceSnapshot").add("Name", name()).add("Arn", arn()).add("SupportCode", supportCode())
                .add("CreatedAt", createdAt()).add("Location", location()).add("ResourceType", resourceTypeAsString())
                .add("Tags", tags()).add("State", stateAsString()).add("Progress", progress())
                .add("FromAttachedDisks", fromAttachedDisks()).add("FromInstanceName", fromInstanceName())
                .add("FromInstanceArn", fromInstanceArn()).add("FromBlueprintId", fromBlueprintId())
                .add("FromBundleId", fromBundleId()).add("SizeInGb", sizeInGb()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "supportCode":
            return Optional.ofNullable(clazz.cast(supportCode()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "location":
            return Optional.ofNullable(clazz.cast(location()));
        case "resourceType":
            return Optional.ofNullable(clazz.cast(resourceTypeAsString()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "state":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "progress":
            return Optional.ofNullable(clazz.cast(progress()));
        case "fromAttachedDisks":
            return Optional.ofNullable(clazz.cast(fromAttachedDisks()));
        case "fromInstanceName":
            return Optional.ofNullable(clazz.cast(fromInstanceName()));
        case "fromInstanceArn":
            return Optional.ofNullable(clazz.cast(fromInstanceArn()));
        case "fromBlueprintId":
            return Optional.ofNullable(clazz.cast(fromBlueprintId()));
        case "fromBundleId":
            return Optional.ofNullable(clazz.cast(fromBundleId()));
        case "sizeInGb":
            return Optional.ofNullable(clazz.cast(sizeInGb()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<InstanceSnapshot, T> g) {
        return obj -> g.apply((InstanceSnapshot) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InstanceSnapshot> {
        /**
         * <p>
         * The name of the snapshot.
         * </p>
         * 
         * @param name
         *        The name of the snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the snapshot (e.g.,
         * <code>arn:aws:lightsail:us-east-2:123456789101:InstanceSnapshot/d23b5706-3322-4d83-81e5-12345EXAMPLE</code>).
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the snapshot (e.g.,
         *        <code>arn:aws:lightsail:us-east-2:123456789101:InstanceSnapshot/d23b5706-3322-4d83-81e5-12345EXAMPLE</code>
         *        ).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The support code. Include this code in your email to support when you have questions about an instance or
         * another resource in Lightsail. This code enables our support team to look up your Lightsail information more
         * easily.
         * </p>
         * 
         * @param supportCode
         *        The support code. Include this code in your email to support when you have questions about an instance
         *        or another resource in Lightsail. This code enables our support team to look up your Lightsail
         *        information more easily.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportCode(String supportCode);

        /**
         * <p>
         * The timestamp when the snapshot was created (e.g., <code>1479907467.024</code>).
         * </p>
         * 
         * @param createdAt
         *        The timestamp when the snapshot was created (e.g., <code>1479907467.024</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The region name and Availability Zone where you created the snapshot.
         * </p>
         * 
         * @param location
         *        The region name and Availability Zone where you created the snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder location(ResourceLocation location);

        /**
         * <p>
         * The region name and Availability Zone where you created the snapshot.
         * </p>
         * This is a convenience that creates an instance of the {@link ResourceLocation.Builder} avoiding the need to
         * create one manually via {@link ResourceLocation#builder()}.
         *
         * When the {@link Consumer} completes, {@link ResourceLocation.Builder#build()} is called immediately and its
         * result is passed to {@link #location(ResourceLocation)}.
         * 
         * @param location
         *        a consumer that will call methods on {@link ResourceLocation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #location(ResourceLocation)
         */
        default Builder location(Consumer<ResourceLocation.Builder> location) {
            return location(ResourceLocation.builder().applyMutation(location).build());
        }

        /**
         * <p>
         * The type of resource (usually <code>InstanceSnapshot</code>).
         * </p>
         * 
         * @param resourceType
         *        The type of resource (usually <code>InstanceSnapshot</code>).
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * The type of resource (usually <code>InstanceSnapshot</code>).
         * </p>
         * 
         * @param resourceType
         *        The type of resource (usually <code>InstanceSnapshot</code>).
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder resourceType(ResourceType resourceType);

        /**
         * <p>
         * The tag keys and optional values for the resource. For more information about tags in Lightsail, see the <a
         * href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev Guide</a>.
         * </p>
         * 
         * @param tags
         *        The tag keys and optional values for the resource. For more information about tags in Lightsail, see
         *        the <a href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev
         *        Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The tag keys and optional values for the resource. For more information about tags in Lightsail, see the <a
         * href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev Guide</a>.
         * </p>
         * 
         * @param tags
         *        The tag keys and optional values for the resource. For more information about tags in Lightsail, see
         *        the <a href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev
         *        Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The tag keys and optional values for the resource. For more information about tags in Lightsail, see the <a
         * href="https://lightsail.aws.amazon.com/ls/docs/en/articles/amazon-lightsail-tags">Lightsail Dev Guide</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * The state the snapshot is in.
         * </p>
         * 
         * @param state
         *        The state the snapshot is in.
         * @see InstanceSnapshotState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceSnapshotState
         */
        Builder state(String state);

        /**
         * <p>
         * The state the snapshot is in.
         * </p>
         * 
         * @param state
         *        The state the snapshot is in.
         * @see InstanceSnapshotState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceSnapshotState
         */
        Builder state(InstanceSnapshotState state);

        /**
         * <p>
         * The progress of the snapshot.
         * </p>
         * 
         * @param progress
         *        The progress of the snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder progress(String progress);

        /**
         * <p>
         * An array of disk objects containing information about all block storage disks.
         * </p>
         * 
         * @param fromAttachedDisks
         *        An array of disk objects containing information about all block storage disks.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fromAttachedDisks(Collection<Disk> fromAttachedDisks);

        /**
         * <p>
         * An array of disk objects containing information about all block storage disks.
         * </p>
         * 
         * @param fromAttachedDisks
         *        An array of disk objects containing information about all block storage disks.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fromAttachedDisks(Disk... fromAttachedDisks);

        /**
         * <p>
         * An array of disk objects containing information about all block storage disks.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Disk>.Builder} avoiding the need to create
         * one manually via {@link List<Disk>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Disk>.Builder#build()} is called immediately and its result
         * is passed to {@link #fromAttachedDisks(List<Disk>)}.
         * 
         * @param fromAttachedDisks
         *        a consumer that will call methods on {@link List<Disk>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #fromAttachedDisks(List<Disk>)
         */
        Builder fromAttachedDisks(Consumer<Disk.Builder>... fromAttachedDisks);

        /**
         * <p>
         * The instance from which the snapshot was created.
         * </p>
         * 
         * @param fromInstanceName
         *        The instance from which the snapshot was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fromInstanceName(String fromInstanceName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the instance from which the snapshot was created (e.g.,
         * <code>arn:aws:lightsail:us-east-2:123456789101:Instance/64b8404c-ccb1-430b-8daf-12345EXAMPLE</code>).
         * </p>
         * 
         * @param fromInstanceArn
         *        The Amazon Resource Name (ARN) of the instance from which the snapshot was created (e.g.,
         *        <code>arn:aws:lightsail:us-east-2:123456789101:Instance/64b8404c-ccb1-430b-8daf-12345EXAMPLE</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fromInstanceArn(String fromInstanceArn);

        /**
         * <p>
         * The blueprint ID from which you created the snapshot (e.g., <code>os_debian_8_3</code>). A blueprint is a
         * virtual private server (or <i>instance</i>) image used to create instances quickly.
         * </p>
         * 
         * @param fromBlueprintId
         *        The blueprint ID from which you created the snapshot (e.g., <code>os_debian_8_3</code>). A blueprint
         *        is a virtual private server (or <i>instance</i>) image used to create instances quickly.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fromBlueprintId(String fromBlueprintId);

        /**
         * <p>
         * The bundle ID from which you created the snapshot (e.g., <code>micro_1_0</code>).
         * </p>
         * 
         * @param fromBundleId
         *        The bundle ID from which you created the snapshot (e.g., <code>micro_1_0</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fromBundleId(String fromBundleId);

        /**
         * <p>
         * The size in GB of the SSD.
         * </p>
         * 
         * @param sizeInGb
         *        The size in GB of the SSD.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sizeInGb(Integer sizeInGb);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String arn;

        private String supportCode;

        private Instant createdAt;

        private ResourceLocation location;

        private String resourceType;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String state;

        private String progress;

        private List<Disk> fromAttachedDisks = DefaultSdkAutoConstructList.getInstance();

        private String fromInstanceName;

        private String fromInstanceArn;

        private String fromBlueprintId;

        private String fromBundleId;

        private Integer sizeInGb;

        private BuilderImpl() {
        }

        private BuilderImpl(InstanceSnapshot model) {
            name(model.name);
            arn(model.arn);
            supportCode(model.supportCode);
            createdAt(model.createdAt);
            location(model.location);
            resourceType(model.resourceType);
            tags(model.tags);
            state(model.state);
            progress(model.progress);
            fromAttachedDisks(model.fromAttachedDisks);
            fromInstanceName(model.fromInstanceName);
            fromInstanceArn(model.fromInstanceArn);
            fromBlueprintId(model.fromBlueprintId);
            fromBundleId(model.fromBundleId);
            sizeInGb(model.sizeInGb);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getArn() {
            return arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        public final String getSupportCode() {
            return supportCode;
        }

        @Override
        public final Builder supportCode(String supportCode) {
            this.supportCode = supportCode;
            return this;
        }

        public final void setSupportCode(String supportCode) {
            this.supportCode = supportCode;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        public final ResourceLocation.Builder getLocation() {
            return location != null ? location.toBuilder() : null;
        }

        @Override
        public final Builder location(ResourceLocation location) {
            this.location = location;
            return this;
        }

        public final void setLocation(ResourceLocation.BuilderImpl location) {
            this.location = location != null ? location.build() : null;
        }

        public final String getResourceTypeAsString() {
            return resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        @Override
        public final Builder resourceType(ResourceType resourceType) {
            this.resourceType(resourceType.toString());
            return this;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        public final Collection<Tag.Builder> getTags() {
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        public final String getStateAsString() {
            return state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(InstanceSnapshotState state) {
            this.state(state.toString());
            return this;
        }

        public final void setState(String state) {
            this.state = state;
        }

        public final String getProgress() {
            return progress;
        }

        @Override
        public final Builder progress(String progress) {
            this.progress = progress;
            return this;
        }

        public final void setProgress(String progress) {
            this.progress = progress;
        }

        public final Collection<Disk.Builder> getFromAttachedDisks() {
            return fromAttachedDisks != null ? fromAttachedDisks.stream().map(Disk::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder fromAttachedDisks(Collection<Disk> fromAttachedDisks) {
            this.fromAttachedDisks = DiskListCopier.copy(fromAttachedDisks);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder fromAttachedDisks(Disk... fromAttachedDisks) {
            fromAttachedDisks(Arrays.asList(fromAttachedDisks));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder fromAttachedDisks(Consumer<Disk.Builder>... fromAttachedDisks) {
            fromAttachedDisks(Stream.of(fromAttachedDisks).map(c -> Disk.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setFromAttachedDisks(Collection<Disk.BuilderImpl> fromAttachedDisks) {
            this.fromAttachedDisks = DiskListCopier.copyFromBuilder(fromAttachedDisks);
        }

        public final String getFromInstanceName() {
            return fromInstanceName;
        }

        @Override
        public final Builder fromInstanceName(String fromInstanceName) {
            this.fromInstanceName = fromInstanceName;
            return this;
        }

        public final void setFromInstanceName(String fromInstanceName) {
            this.fromInstanceName = fromInstanceName;
        }

        public final String getFromInstanceArn() {
            return fromInstanceArn;
        }

        @Override
        public final Builder fromInstanceArn(String fromInstanceArn) {
            this.fromInstanceArn = fromInstanceArn;
            return this;
        }

        public final void setFromInstanceArn(String fromInstanceArn) {
            this.fromInstanceArn = fromInstanceArn;
        }

        public final String getFromBlueprintId() {
            return fromBlueprintId;
        }

        @Override
        public final Builder fromBlueprintId(String fromBlueprintId) {
            this.fromBlueprintId = fromBlueprintId;
            return this;
        }

        public final void setFromBlueprintId(String fromBlueprintId) {
            this.fromBlueprintId = fromBlueprintId;
        }

        public final String getFromBundleId() {
            return fromBundleId;
        }

        @Override
        public final Builder fromBundleId(String fromBundleId) {
            this.fromBundleId = fromBundleId;
            return this;
        }

        public final void setFromBundleId(String fromBundleId) {
            this.fromBundleId = fromBundleId;
        }

        public final Integer getSizeInGb() {
            return sizeInGb;
        }

        @Override
        public final Builder sizeInGb(Integer sizeInGb) {
            this.sizeInGb = sizeInGb;
            return this;
        }

        public final void setSizeInGb(Integer sizeInGb) {
            this.sizeInGb = sizeInGb;
        }

        @Override
        public InstanceSnapshot build() {
            return new InstanceSnapshot(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
