/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.location;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.location.model.AccessDeniedException;
import software.amazon.awssdk.services.location.model.AssociateTrackerConsumerRequest;
import software.amazon.awssdk.services.location.model.AssociateTrackerConsumerResponse;
import software.amazon.awssdk.services.location.model.BatchDeleteGeofenceRequest;
import software.amazon.awssdk.services.location.model.BatchDeleteGeofenceResponse;
import software.amazon.awssdk.services.location.model.BatchEvaluateGeofencesRequest;
import software.amazon.awssdk.services.location.model.BatchEvaluateGeofencesResponse;
import software.amazon.awssdk.services.location.model.BatchGetDevicePositionRequest;
import software.amazon.awssdk.services.location.model.BatchGetDevicePositionResponse;
import software.amazon.awssdk.services.location.model.BatchPutGeofenceRequest;
import software.amazon.awssdk.services.location.model.BatchPutGeofenceResponse;
import software.amazon.awssdk.services.location.model.BatchUpdateDevicePositionRequest;
import software.amazon.awssdk.services.location.model.BatchUpdateDevicePositionResponse;
import software.amazon.awssdk.services.location.model.ConflictException;
import software.amazon.awssdk.services.location.model.CreateGeofenceCollectionRequest;
import software.amazon.awssdk.services.location.model.CreateGeofenceCollectionResponse;
import software.amazon.awssdk.services.location.model.CreateMapRequest;
import software.amazon.awssdk.services.location.model.CreateMapResponse;
import software.amazon.awssdk.services.location.model.CreatePlaceIndexRequest;
import software.amazon.awssdk.services.location.model.CreatePlaceIndexResponse;
import software.amazon.awssdk.services.location.model.CreateTrackerRequest;
import software.amazon.awssdk.services.location.model.CreateTrackerResponse;
import software.amazon.awssdk.services.location.model.DeleteGeofenceCollectionRequest;
import software.amazon.awssdk.services.location.model.DeleteGeofenceCollectionResponse;
import software.amazon.awssdk.services.location.model.DeleteMapRequest;
import software.amazon.awssdk.services.location.model.DeleteMapResponse;
import software.amazon.awssdk.services.location.model.DeletePlaceIndexRequest;
import software.amazon.awssdk.services.location.model.DeletePlaceIndexResponse;
import software.amazon.awssdk.services.location.model.DeleteTrackerRequest;
import software.amazon.awssdk.services.location.model.DeleteTrackerResponse;
import software.amazon.awssdk.services.location.model.DescribeGeofenceCollectionRequest;
import software.amazon.awssdk.services.location.model.DescribeGeofenceCollectionResponse;
import software.amazon.awssdk.services.location.model.DescribeMapRequest;
import software.amazon.awssdk.services.location.model.DescribeMapResponse;
import software.amazon.awssdk.services.location.model.DescribePlaceIndexRequest;
import software.amazon.awssdk.services.location.model.DescribePlaceIndexResponse;
import software.amazon.awssdk.services.location.model.DescribeTrackerRequest;
import software.amazon.awssdk.services.location.model.DescribeTrackerResponse;
import software.amazon.awssdk.services.location.model.DisassociateTrackerConsumerRequest;
import software.amazon.awssdk.services.location.model.DisassociateTrackerConsumerResponse;
import software.amazon.awssdk.services.location.model.GetDevicePositionHistoryRequest;
import software.amazon.awssdk.services.location.model.GetDevicePositionHistoryResponse;
import software.amazon.awssdk.services.location.model.GetDevicePositionRequest;
import software.amazon.awssdk.services.location.model.GetDevicePositionResponse;
import software.amazon.awssdk.services.location.model.GetGeofenceRequest;
import software.amazon.awssdk.services.location.model.GetGeofenceResponse;
import software.amazon.awssdk.services.location.model.GetMapGlyphsRequest;
import software.amazon.awssdk.services.location.model.GetMapGlyphsResponse;
import software.amazon.awssdk.services.location.model.GetMapSpritesRequest;
import software.amazon.awssdk.services.location.model.GetMapSpritesResponse;
import software.amazon.awssdk.services.location.model.GetMapStyleDescriptorRequest;
import software.amazon.awssdk.services.location.model.GetMapStyleDescriptorResponse;
import software.amazon.awssdk.services.location.model.GetMapTileRequest;
import software.amazon.awssdk.services.location.model.GetMapTileResponse;
import software.amazon.awssdk.services.location.model.InternalServerException;
import software.amazon.awssdk.services.location.model.ListGeofenceCollectionsRequest;
import software.amazon.awssdk.services.location.model.ListGeofenceCollectionsResponse;
import software.amazon.awssdk.services.location.model.ListGeofencesRequest;
import software.amazon.awssdk.services.location.model.ListGeofencesResponse;
import software.amazon.awssdk.services.location.model.ListMapsRequest;
import software.amazon.awssdk.services.location.model.ListMapsResponse;
import software.amazon.awssdk.services.location.model.ListPlaceIndexesRequest;
import software.amazon.awssdk.services.location.model.ListPlaceIndexesResponse;
import software.amazon.awssdk.services.location.model.ListTrackerConsumersRequest;
import software.amazon.awssdk.services.location.model.ListTrackerConsumersResponse;
import software.amazon.awssdk.services.location.model.ListTrackersRequest;
import software.amazon.awssdk.services.location.model.ListTrackersResponse;
import software.amazon.awssdk.services.location.model.LocationException;
import software.amazon.awssdk.services.location.model.LocationRequest;
import software.amazon.awssdk.services.location.model.PutGeofenceRequest;
import software.amazon.awssdk.services.location.model.PutGeofenceResponse;
import software.amazon.awssdk.services.location.model.ResourceNotFoundException;
import software.amazon.awssdk.services.location.model.SearchPlaceIndexForPositionRequest;
import software.amazon.awssdk.services.location.model.SearchPlaceIndexForPositionResponse;
import software.amazon.awssdk.services.location.model.SearchPlaceIndexForTextRequest;
import software.amazon.awssdk.services.location.model.SearchPlaceIndexForTextResponse;
import software.amazon.awssdk.services.location.model.ThrottlingException;
import software.amazon.awssdk.services.location.model.ValidationException;
import software.amazon.awssdk.services.location.paginators.GetDevicePositionHistoryPublisher;
import software.amazon.awssdk.services.location.paginators.ListGeofenceCollectionsPublisher;
import software.amazon.awssdk.services.location.paginators.ListGeofencesPublisher;
import software.amazon.awssdk.services.location.paginators.ListMapsPublisher;
import software.amazon.awssdk.services.location.paginators.ListPlaceIndexesPublisher;
import software.amazon.awssdk.services.location.paginators.ListTrackerConsumersPublisher;
import software.amazon.awssdk.services.location.paginators.ListTrackersPublisher;
import software.amazon.awssdk.services.location.transform.AssociateTrackerConsumerRequestMarshaller;
import software.amazon.awssdk.services.location.transform.BatchDeleteGeofenceRequestMarshaller;
import software.amazon.awssdk.services.location.transform.BatchEvaluateGeofencesRequestMarshaller;
import software.amazon.awssdk.services.location.transform.BatchGetDevicePositionRequestMarshaller;
import software.amazon.awssdk.services.location.transform.BatchPutGeofenceRequestMarshaller;
import software.amazon.awssdk.services.location.transform.BatchUpdateDevicePositionRequestMarshaller;
import software.amazon.awssdk.services.location.transform.CreateGeofenceCollectionRequestMarshaller;
import software.amazon.awssdk.services.location.transform.CreateMapRequestMarshaller;
import software.amazon.awssdk.services.location.transform.CreatePlaceIndexRequestMarshaller;
import software.amazon.awssdk.services.location.transform.CreateTrackerRequestMarshaller;
import software.amazon.awssdk.services.location.transform.DeleteGeofenceCollectionRequestMarshaller;
import software.amazon.awssdk.services.location.transform.DeleteMapRequestMarshaller;
import software.amazon.awssdk.services.location.transform.DeletePlaceIndexRequestMarshaller;
import software.amazon.awssdk.services.location.transform.DeleteTrackerRequestMarshaller;
import software.amazon.awssdk.services.location.transform.DescribeGeofenceCollectionRequestMarshaller;
import software.amazon.awssdk.services.location.transform.DescribeMapRequestMarshaller;
import software.amazon.awssdk.services.location.transform.DescribePlaceIndexRequestMarshaller;
import software.amazon.awssdk.services.location.transform.DescribeTrackerRequestMarshaller;
import software.amazon.awssdk.services.location.transform.DisassociateTrackerConsumerRequestMarshaller;
import software.amazon.awssdk.services.location.transform.GetDevicePositionHistoryRequestMarshaller;
import software.amazon.awssdk.services.location.transform.GetDevicePositionRequestMarshaller;
import software.amazon.awssdk.services.location.transform.GetGeofenceRequestMarshaller;
import software.amazon.awssdk.services.location.transform.GetMapGlyphsRequestMarshaller;
import software.amazon.awssdk.services.location.transform.GetMapSpritesRequestMarshaller;
import software.amazon.awssdk.services.location.transform.GetMapStyleDescriptorRequestMarshaller;
import software.amazon.awssdk.services.location.transform.GetMapTileRequestMarshaller;
import software.amazon.awssdk.services.location.transform.ListGeofenceCollectionsRequestMarshaller;
import software.amazon.awssdk.services.location.transform.ListGeofencesRequestMarshaller;
import software.amazon.awssdk.services.location.transform.ListMapsRequestMarshaller;
import software.amazon.awssdk.services.location.transform.ListPlaceIndexesRequestMarshaller;
import software.amazon.awssdk.services.location.transform.ListTrackerConsumersRequestMarshaller;
import software.amazon.awssdk.services.location.transform.ListTrackersRequestMarshaller;
import software.amazon.awssdk.services.location.transform.PutGeofenceRequestMarshaller;
import software.amazon.awssdk.services.location.transform.SearchPlaceIndexForPositionRequestMarshaller;
import software.amazon.awssdk.services.location.transform.SearchPlaceIndexForTextRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link LocationAsyncClient}.
 *
 * @see LocationAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultLocationAsyncClient implements LocationAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultLocationAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultLocationAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates an association between a geofence collection and a tracker resource. This allows the tracker resource to
     * communicate location data to the linked geofence collection.
     * </p>
     *
     * @param associateTrackerConsumerRequest
     * @return A Java Future containing the result of the AssociateTrackerConsumer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>ConflictException The request was unsuccessful due to a conflict.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.AssociateTrackerConsumer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/AssociateTrackerConsumer"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateTrackerConsumerResponse> associateTrackerConsumer(
            AssociateTrackerConsumerRequest associateTrackerConsumerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateTrackerConsumerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateTrackerConsumer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateTrackerConsumerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateTrackerConsumerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "tracking.";
            String resolvedHostExpression = "tracking.";

            CompletableFuture<AssociateTrackerConsumerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateTrackerConsumerRequest, AssociateTrackerConsumerResponse>()
                            .withOperationName("AssociateTrackerConsumer")
                            .withMarshaller(new AssociateTrackerConsumerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(associateTrackerConsumerRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = associateTrackerConsumerRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<AssociateTrackerConsumerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a batch of geofences from a geofence collection.
     * </p>
     * <note>
     * <p>
     * This action deletes the resource permanently. You can't undo this action.
     * </p>
     * </note>
     *
     * @param batchDeleteGeofenceRequest
     * @return A Java Future containing the result of the BatchDeleteGeofence operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.BatchDeleteGeofence
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/BatchDeleteGeofence" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchDeleteGeofenceResponse> batchDeleteGeofence(
            BatchDeleteGeofenceRequest batchDeleteGeofenceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchDeleteGeofenceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchDeleteGeofence");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchDeleteGeofenceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchDeleteGeofenceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "geofencing.";
            String resolvedHostExpression = "geofencing.";

            CompletableFuture<BatchDeleteGeofenceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchDeleteGeofenceRequest, BatchDeleteGeofenceResponse>()
                            .withOperationName("BatchDeleteGeofence")
                            .withMarshaller(new BatchDeleteGeofenceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(batchDeleteGeofenceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = batchDeleteGeofenceRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<BatchDeleteGeofenceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Used in geofence monitoring. Evaluates device positions against the position of geofences in a given geofence
     * collection.
     * </p>
     *
     * @param batchEvaluateGeofencesRequest
     * @return A Java Future containing the result of the BatchEvaluateGeofences operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.BatchEvaluateGeofences
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/BatchEvaluateGeofences"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchEvaluateGeofencesResponse> batchEvaluateGeofences(
            BatchEvaluateGeofencesRequest batchEvaluateGeofencesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchEvaluateGeofencesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchEvaluateGeofences");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchEvaluateGeofencesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchEvaluateGeofencesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "geofencing.";
            String resolvedHostExpression = "geofencing.";

            CompletableFuture<BatchEvaluateGeofencesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchEvaluateGeofencesRequest, BatchEvaluateGeofencesResponse>()
                            .withOperationName("BatchEvaluateGeofences")
                            .withMarshaller(new BatchEvaluateGeofencesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(batchEvaluateGeofencesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = batchEvaluateGeofencesRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<BatchEvaluateGeofencesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * A batch request to retrieve device positions.
     * </p>
     * <note>
     * <p>
     * The response will return the device positions from the last 24 hours.
     * </p>
     * </note>
     *
     * @param batchGetDevicePositionRequest
     * @return A Java Future containing the result of the BatchGetDevicePosition operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.BatchGetDevicePosition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/BatchGetDevicePosition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchGetDevicePositionResponse> batchGetDevicePosition(
            BatchGetDevicePositionRequest batchGetDevicePositionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchGetDevicePositionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchGetDevicePosition");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchGetDevicePositionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchGetDevicePositionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "tracking.";
            String resolvedHostExpression = "tracking.";

            CompletableFuture<BatchGetDevicePositionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchGetDevicePositionRequest, BatchGetDevicePositionResponse>()
                            .withOperationName("BatchGetDevicePosition")
                            .withMarshaller(new BatchGetDevicePositionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(batchGetDevicePositionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = batchGetDevicePositionRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<BatchGetDevicePositionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * A batch request for storing geofences into a given geofence collection.
     * </p>
     *
     * @param batchPutGeofenceRequest
     * @return A Java Future containing the result of the BatchPutGeofence operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.BatchPutGeofence
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/BatchPutGeofence" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<BatchPutGeofenceResponse> batchPutGeofence(BatchPutGeofenceRequest batchPutGeofenceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchPutGeofenceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchPutGeofence");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchPutGeofenceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchPutGeofenceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "geofencing.";
            String resolvedHostExpression = "geofencing.";

            CompletableFuture<BatchPutGeofenceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchPutGeofenceRequest, BatchPutGeofenceResponse>()
                            .withOperationName("BatchPutGeofence")
                            .withMarshaller(new BatchPutGeofenceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(batchPutGeofenceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = batchPutGeofenceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<BatchPutGeofenceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Uploads a position update for one or more devices to a tracker resource. The data is used for API queries
     * requesting the device position and position history.
     * </p>
     * <note>
     * <p>
     * Limitation — Location data is sampled at a fixed rate of 1 position per 30 second interval, and retained for 1
     * year before it is deleted.
     * </p>
     * </note>
     *
     * @param batchUpdateDevicePositionRequest
     * @return A Java Future containing the result of the BatchUpdateDevicePosition operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.BatchUpdateDevicePosition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/BatchUpdateDevicePosition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchUpdateDevicePositionResponse> batchUpdateDevicePosition(
            BatchUpdateDevicePositionRequest batchUpdateDevicePositionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchUpdateDevicePositionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchUpdateDevicePosition");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchUpdateDevicePositionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchUpdateDevicePositionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "tracking.";
            String resolvedHostExpression = "tracking.";

            CompletableFuture<BatchUpdateDevicePositionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchUpdateDevicePositionRequest, BatchUpdateDevicePositionResponse>()
                            .withOperationName("BatchUpdateDevicePosition")
                            .withMarshaller(new BatchUpdateDevicePositionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(batchUpdateDevicePositionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = batchUpdateDevicePositionRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<BatchUpdateDevicePositionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a geofence collection, which manages and stores geofences.
     * </p>
     *
     * @param createGeofenceCollectionRequest
     * @return A Java Future containing the result of the CreateGeofenceCollection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ConflictException The request was unsuccessful due to a conflict.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.CreateGeofenceCollection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/CreateGeofenceCollection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateGeofenceCollectionResponse> createGeofenceCollection(
            CreateGeofenceCollectionRequest createGeofenceCollectionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createGeofenceCollectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateGeofenceCollection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateGeofenceCollectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateGeofenceCollectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "geofencing.";
            String resolvedHostExpression = "geofencing.";

            CompletableFuture<CreateGeofenceCollectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateGeofenceCollectionRequest, CreateGeofenceCollectionResponse>()
                            .withOperationName("CreateGeofenceCollection")
                            .withMarshaller(new CreateGeofenceCollectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(createGeofenceCollectionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createGeofenceCollectionRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<CreateGeofenceCollectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a map resource in your AWS account, which provides map tiles of different styles sourced from global
     * location data providers.
     * </p>
     * <note>
     * <p>
     * By using Maps, you agree that AWS may transmit your API queries to your selected third party provider for
     * processing, which may be outside the AWS region you are currently using. For more information, see the <a
     * href="https://aws.amazon.com/service-terms/">AWS Service Terms</a> for Amazon Location Service.
     * </p>
     * </note>
     *
     * @param createMapRequest
     * @return A Java Future containing the result of the CreateMap operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ConflictException The request was unsuccessful due to a conflict.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.CreateMap
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/CreateMap" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateMapResponse> createMap(CreateMapRequest createMapRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMapRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMap");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateMapResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateMapResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "maps.";
            String resolvedHostExpression = "maps.";

            CompletableFuture<CreateMapResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateMapRequest, CreateMapResponse>().withOperationName("CreateMap")
                            .withMarshaller(new CreateMapRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .hostPrefixExpression(resolvedHostExpression).withInput(createMapRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createMapRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateMapResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a Place index resource in your AWS account, which supports Places functions with geospatial data sourced
     * from your chosen data provider.
     * </p>
     * <note>
     * <p>
     * By using Places, you agree that AWS may transmit your API queries to your selected third party provider for
     * processing, which may be outside the AWS region you are currently using.
     * </p>
     * <p>
     * Because of licensing limitations, you may not use HERE to store results for locations in Japan. For more
     * information, see the <a href="https://aws.amazon.com/service-terms/">AWS Service Terms</a> for Amazon Location
     * Service.
     * </p>
     * </note>
     *
     * @param createPlaceIndexRequest
     * @return A Java Future containing the result of the CreatePlaceIndex operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ConflictException The request was unsuccessful due to a conflict.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.CreatePlaceIndex
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/CreatePlaceIndex" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePlaceIndexResponse> createPlaceIndex(CreatePlaceIndexRequest createPlaceIndexRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPlaceIndexRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePlaceIndex");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePlaceIndexResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreatePlaceIndexResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "places.";
            String resolvedHostExpression = "places.";

            CompletableFuture<CreatePlaceIndexResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePlaceIndexRequest, CreatePlaceIndexResponse>()
                            .withOperationName("CreatePlaceIndex")
                            .withMarshaller(new CreatePlaceIndexRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(createPlaceIndexRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createPlaceIndexRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreatePlaceIndexResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a tracker resource in your AWS account, which lets you retrieve current and historical location of
     * devices.
     * </p>
     *
     * @param createTrackerRequest
     * @return A Java Future containing the result of the CreateTracker operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ConflictException The request was unsuccessful due to a conflict.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.CreateTracker
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/CreateTracker" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateTrackerResponse> createTracker(CreateTrackerRequest createTrackerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTrackerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTracker");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateTrackerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateTrackerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "tracking.";
            String resolvedHostExpression = "tracking.";

            CompletableFuture<CreateTrackerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateTrackerRequest, CreateTrackerResponse>()
                            .withOperationName("CreateTracker")
                            .withMarshaller(new CreateTrackerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(createTrackerRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createTrackerRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateTrackerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a geofence collection from your AWS account.
     * </p>
     * <note>
     * <p>
     * This action deletes the resource permanently. You can't undo this action. If the geofence collection is the
     * target of a tracker resource, the devices will no longer be monitored.
     * </p>
     * </note>
     *
     * @param deleteGeofenceCollectionRequest
     * @return A Java Future containing the result of the DeleteGeofenceCollection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.DeleteGeofenceCollection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/DeleteGeofenceCollection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteGeofenceCollectionResponse> deleteGeofenceCollection(
            DeleteGeofenceCollectionRequest deleteGeofenceCollectionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteGeofenceCollectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteGeofenceCollection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteGeofenceCollectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteGeofenceCollectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "geofencing.";
            String resolvedHostExpression = "geofencing.";

            CompletableFuture<DeleteGeofenceCollectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteGeofenceCollectionRequest, DeleteGeofenceCollectionResponse>()
                            .withOperationName("DeleteGeofenceCollection")
                            .withMarshaller(new DeleteGeofenceCollectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(deleteGeofenceCollectionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteGeofenceCollectionRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DeleteGeofenceCollectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a map resource from your AWS account.
     * </p>
     * <note>
     * <p>
     * This action deletes the resource permanently. You cannot undo this action. If the map is being used in an
     * application, the map may not render.
     * </p>
     * </note>
     *
     * @param deleteMapRequest
     * @return A Java Future containing the result of the DeleteMap operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.DeleteMap
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/DeleteMap" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteMapResponse> deleteMap(DeleteMapRequest deleteMapRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMapRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMap");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteMapResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteMapResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "maps.";
            String resolvedHostExpression = "maps.";

            CompletableFuture<DeleteMapResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteMapRequest, DeleteMapResponse>().withOperationName("DeleteMap")
                            .withMarshaller(new DeleteMapRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .hostPrefixExpression(resolvedHostExpression).withInput(deleteMapRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteMapRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteMapResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a Place index resource from your AWS account.
     * </p>
     * <note>
     * <p>
     * This action deletes the resource permanently. You cannot undo this action.
     * </p>
     * </note>
     *
     * @param deletePlaceIndexRequest
     * @return A Java Future containing the result of the DeletePlaceIndex operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.DeletePlaceIndex
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/DeletePlaceIndex" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePlaceIndexResponse> deletePlaceIndex(DeletePlaceIndexRequest deletePlaceIndexRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePlaceIndexRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePlaceIndex");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePlaceIndexResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeletePlaceIndexResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "places.";
            String resolvedHostExpression = "places.";

            CompletableFuture<DeletePlaceIndexResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePlaceIndexRequest, DeletePlaceIndexResponse>()
                            .withOperationName("DeletePlaceIndex")
                            .withMarshaller(new DeletePlaceIndexRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(deletePlaceIndexRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deletePlaceIndexRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeletePlaceIndexResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a tracker resource from your AWS account.
     * </p>
     * <note>
     * <p>
     * This action deletes the resource permanently. You can't undo this action. If the tracker resource is in use, you
     * may encounter an error. Make sure that the target resource is not a dependency for your applications.
     * </p>
     * </note>
     *
     * @param deleteTrackerRequest
     * @return A Java Future containing the result of the DeleteTracker operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.DeleteTracker
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/DeleteTracker" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteTrackerResponse> deleteTracker(DeleteTrackerRequest deleteTrackerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTrackerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTracker");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteTrackerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteTrackerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "tracking.";
            String resolvedHostExpression = "tracking.";

            CompletableFuture<DeleteTrackerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteTrackerRequest, DeleteTrackerResponse>()
                            .withOperationName("DeleteTracker")
                            .withMarshaller(new DeleteTrackerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(deleteTrackerRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteTrackerRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteTrackerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the geofence collection details.
     * </p>
     *
     * @param describeGeofenceCollectionRequest
     * @return A Java Future containing the result of the DescribeGeofenceCollection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.DescribeGeofenceCollection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/DescribeGeofenceCollection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeGeofenceCollectionResponse> describeGeofenceCollection(
            DescribeGeofenceCollectionRequest describeGeofenceCollectionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeGeofenceCollectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeGeofenceCollection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeGeofenceCollectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeGeofenceCollectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "geofencing.";
            String resolvedHostExpression = "geofencing.";

            CompletableFuture<DescribeGeofenceCollectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeGeofenceCollectionRequest, DescribeGeofenceCollectionResponse>()
                            .withOperationName("DescribeGeofenceCollection")
                            .withMarshaller(new DescribeGeofenceCollectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(describeGeofenceCollectionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeGeofenceCollectionRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DescribeGeofenceCollectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the map resource details.
     * </p>
     *
     * @param describeMapRequest
     * @return A Java Future containing the result of the DescribeMap operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.DescribeMap
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/DescribeMap" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeMapResponse> describeMap(DescribeMapRequest describeMapRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeMapRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeMap");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeMapResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeMapResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "maps.";
            String resolvedHostExpression = "maps.";

            CompletableFuture<DescribeMapResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeMapRequest, DescribeMapResponse>()
                            .withOperationName("DescribeMap").withMarshaller(new DescribeMapRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(describeMapRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeMapRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DescribeMapResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the Place index resource details.
     * </p>
     *
     * @param describePlaceIndexRequest
     * @return A Java Future containing the result of the DescribePlaceIndex operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.DescribePlaceIndex
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/DescribePlaceIndex" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribePlaceIndexResponse> describePlaceIndex(DescribePlaceIndexRequest describePlaceIndexRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describePlaceIndexRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribePlaceIndex");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribePlaceIndexResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribePlaceIndexResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "places.";
            String resolvedHostExpression = "places.";

            CompletableFuture<DescribePlaceIndexResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribePlaceIndexRequest, DescribePlaceIndexResponse>()
                            .withOperationName("DescribePlaceIndex")
                            .withMarshaller(new DescribePlaceIndexRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(describePlaceIndexRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describePlaceIndexRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DescribePlaceIndexResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the tracker resource details.
     * </p>
     *
     * @param describeTrackerRequest
     * @return A Java Future containing the result of the DescribeTracker operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.DescribeTracker
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/DescribeTracker" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeTrackerResponse> describeTracker(DescribeTrackerRequest describeTrackerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeTrackerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTracker");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeTrackerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeTrackerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "tracking.";
            String resolvedHostExpression = "tracking.";

            CompletableFuture<DescribeTrackerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeTrackerRequest, DescribeTrackerResponse>()
                            .withOperationName("DescribeTracker")
                            .withMarshaller(new DescribeTrackerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(describeTrackerRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeTrackerRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DescribeTrackerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the association bewteen a tracker resource and a geofence collection.
     * </p>
     * <note>
     * <p>
     * Once you unlink a tracker resource from a geofence collection, the tracker positions will no longer be
     * automatically evaluated against geofences.
     * </p>
     * </note>
     *
     * @param disassociateTrackerConsumerRequest
     * @return A Java Future containing the result of the DisassociateTrackerConsumer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.DisassociateTrackerConsumer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/DisassociateTrackerConsumer"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateTrackerConsumerResponse> disassociateTrackerConsumer(
            DisassociateTrackerConsumerRequest disassociateTrackerConsumerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateTrackerConsumerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateTrackerConsumer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateTrackerConsumerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateTrackerConsumerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "tracking.";
            String resolvedHostExpression = "tracking.";

            CompletableFuture<DisassociateTrackerConsumerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateTrackerConsumerRequest, DisassociateTrackerConsumerResponse>()
                            .withOperationName("DisassociateTrackerConsumer")
                            .withMarshaller(new DisassociateTrackerConsumerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(disassociateTrackerConsumerRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = disassociateTrackerConsumerRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DisassociateTrackerConsumerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the latest device position.
     * </p>
     * <note>
     * <p>
     * Limitation — Device positions are deleted after one year.
     * </p>
     * </note>
     *
     * @param getDevicePositionRequest
     * @return A Java Future containing the result of the GetDevicePosition operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.GetDevicePosition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/GetDevicePosition" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDevicePositionResponse> getDevicePosition(GetDevicePositionRequest getDevicePositionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDevicePositionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDevicePosition");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDevicePositionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDevicePositionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "tracking.";
            String resolvedHostExpression = "tracking.";

            CompletableFuture<GetDevicePositionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDevicePositionRequest, GetDevicePositionResponse>()
                            .withOperationName("GetDevicePosition")
                            .withMarshaller(new GetDevicePositionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(getDevicePositionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getDevicePositionRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetDevicePositionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the device position history from a tracker resource within a specified range of time.
     * </p>
     * <note>
     * <p>
     * Limitation — Device positions are deleted after one year.
     * </p>
     * </note>
     *
     * @param getDevicePositionHistoryRequest
     * @return A Java Future containing the result of the GetDevicePositionHistory operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.GetDevicePositionHistory
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/GetDevicePositionHistory"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDevicePositionHistoryResponse> getDevicePositionHistory(
            GetDevicePositionHistoryRequest getDevicePositionHistoryRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDevicePositionHistoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDevicePositionHistory");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDevicePositionHistoryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDevicePositionHistoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "tracking.";
            String resolvedHostExpression = "tracking.";

            CompletableFuture<GetDevicePositionHistoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDevicePositionHistoryRequest, GetDevicePositionHistoryResponse>()
                            .withOperationName("GetDevicePositionHistory")
                            .withMarshaller(new GetDevicePositionHistoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(getDevicePositionHistoryRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getDevicePositionHistoryRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<GetDevicePositionHistoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the device position history from a tracker resource within a specified range of time.
     * </p>
     * <note>
     * <p>
     * Limitation — Device positions are deleted after one year.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #getDevicePositionHistory(software.amazon.awssdk.services.location.model.GetDevicePositionHistoryRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.location.paginators.GetDevicePositionHistoryPublisher publisher = client.getDevicePositionHistoryPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.location.paginators.GetDevicePositionHistoryPublisher publisher = client.getDevicePositionHistoryPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.location.model.GetDevicePositionHistoryResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.location.model.GetDevicePositionHistoryResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getDevicePositionHistory(software.amazon.awssdk.services.location.model.GetDevicePositionHistoryRequest)}
     * operation.</b>
     * </p>
     *
     * @param getDevicePositionHistoryRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.GetDevicePositionHistory
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/GetDevicePositionHistory"
     *      target="_top">AWS API Documentation</a>
     */
    public GetDevicePositionHistoryPublisher getDevicePositionHistoryPaginator(
            GetDevicePositionHistoryRequest getDevicePositionHistoryRequest) {
        return new GetDevicePositionHistoryPublisher(this, applyPaginatorUserAgent(getDevicePositionHistoryRequest));
    }

    /**
     * <p>
     * Retrieves the geofence details from a geofence collection.
     * </p>
     *
     * @param getGeofenceRequest
     * @return A Java Future containing the result of the GetGeofence operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.GetGeofence
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/GetGeofence" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetGeofenceResponse> getGeofence(GetGeofenceRequest getGeofenceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getGeofenceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetGeofence");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetGeofenceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetGeofenceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "geofencing.";
            String resolvedHostExpression = "geofencing.";

            CompletableFuture<GetGeofenceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetGeofenceRequest, GetGeofenceResponse>()
                            .withOperationName("GetGeofence").withMarshaller(new GetGeofenceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(getGeofenceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getGeofenceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetGeofenceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves glyphs used to display labels on a map.
     * </p>
     *
     * @param getMapGlyphsRequest
     * @return A Java Future containing the result of the GetMapGlyphs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.GetMapGlyphs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/GetMapGlyphs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetMapGlyphsResponse> getMapGlyphs(GetMapGlyphsRequest getMapGlyphsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMapGlyphsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMapGlyphs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(false).build();

            HttpResponseHandler<GetMapGlyphsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetMapGlyphsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "maps.";
            String resolvedHostExpression = "maps.";

            CompletableFuture<GetMapGlyphsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMapGlyphsRequest, GetMapGlyphsResponse>()
                            .withOperationName("GetMapGlyphs").withMarshaller(new GetMapGlyphsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(getMapGlyphsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getMapGlyphsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetMapGlyphsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the sprite sheet corresponding to a map resource. The sprite sheet is a PNG image paired with a JSON
     * document describing the offsets of individual icons that will be displayed on a rendered map.
     * </p>
     *
     * @param getMapSpritesRequest
     * @return A Java Future containing the result of the GetMapSprites operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.GetMapSprites
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/GetMapSprites" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetMapSpritesResponse> getMapSprites(GetMapSpritesRequest getMapSpritesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMapSpritesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMapSprites");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(false).build();

            HttpResponseHandler<GetMapSpritesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetMapSpritesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "maps.";
            String resolvedHostExpression = "maps.";

            CompletableFuture<GetMapSpritesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMapSpritesRequest, GetMapSpritesResponse>()
                            .withOperationName("GetMapSprites")
                            .withMarshaller(new GetMapSpritesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(getMapSpritesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getMapSpritesRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetMapSpritesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the map style descriptor from a map resource.
     * </p>
     * <p>
     * The style descriptor contains speciﬁcations on how features render on a map. For example, what data to display,
     * what order to display the data in, and the style for the data. Style descriptors follow the Mapbox Style
     * Specification.
     * </p>
     *
     * @param getMapStyleDescriptorRequest
     * @return A Java Future containing the result of the GetMapStyleDescriptor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.GetMapStyleDescriptor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/GetMapStyleDescriptor"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetMapStyleDescriptorResponse> getMapStyleDescriptor(
            GetMapStyleDescriptorRequest getMapStyleDescriptorRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMapStyleDescriptorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMapStyleDescriptor");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(false).build();

            HttpResponseHandler<GetMapStyleDescriptorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetMapStyleDescriptorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "maps.";
            String resolvedHostExpression = "maps.";

            CompletableFuture<GetMapStyleDescriptorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMapStyleDescriptorRequest, GetMapStyleDescriptorResponse>()
                            .withOperationName("GetMapStyleDescriptor")
                            .withMarshaller(new GetMapStyleDescriptorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(getMapStyleDescriptorRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getMapStyleDescriptorRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<GetMapStyleDescriptorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a vector data tile from the map resource. Map tiles are used by clients to render a map. They are
     * addressed using a grid arrangement with an X coordinate, Y coordinate, and Z (zoom) level.
     * </p>
     * <p>
     * The origin (0, 0) is the top left of the map. Increasing the zoom level by 1 doubles both the X and Y dimensions,
     * so a tile containing data for the entire world at (0/0/0) will be split into 4 tiles at zoom 1 (1/0/0, 1/0/1,
     * 1/1/0, 1/1/1).
     * </p>
     *
     * @param getMapTileRequest
     * @return A Java Future containing the result of the GetMapTile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.GetMapTile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/GetMapTile" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetMapTileResponse> getMapTile(GetMapTileRequest getMapTileRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMapTileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMapTile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(false).build();

            HttpResponseHandler<GetMapTileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetMapTileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "maps.";
            String resolvedHostExpression = "maps.";

            CompletableFuture<GetMapTileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMapTileRequest, GetMapTileResponse>().withOperationName("GetMapTile")
                            .withMarshaller(new GetMapTileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(getMapTileRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getMapTileRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetMapTileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists geofence collections in your AWS account.
     * </p>
     *
     * @param listGeofenceCollectionsRequest
     * @return A Java Future containing the result of the ListGeofenceCollections operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.ListGeofenceCollections
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/ListGeofenceCollections"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListGeofenceCollectionsResponse> listGeofenceCollections(
            ListGeofenceCollectionsRequest listGeofenceCollectionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGeofenceCollectionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGeofenceCollections");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListGeofenceCollectionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListGeofenceCollectionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "geofencing.";
            String resolvedHostExpression = "geofencing.";

            CompletableFuture<ListGeofenceCollectionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListGeofenceCollectionsRequest, ListGeofenceCollectionsResponse>()
                            .withOperationName("ListGeofenceCollections")
                            .withMarshaller(new ListGeofenceCollectionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(listGeofenceCollectionsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listGeofenceCollectionsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListGeofenceCollectionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists geofence collections in your AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listGeofenceCollections(software.amazon.awssdk.services.location.model.ListGeofenceCollectionsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.location.paginators.ListGeofenceCollectionsPublisher publisher = client.listGeofenceCollectionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.location.paginators.ListGeofenceCollectionsPublisher publisher = client.listGeofenceCollectionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.location.model.ListGeofenceCollectionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.location.model.ListGeofenceCollectionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listGeofenceCollections(software.amazon.awssdk.services.location.model.ListGeofenceCollectionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listGeofenceCollectionsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.ListGeofenceCollections
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/ListGeofenceCollections"
     *      target="_top">AWS API Documentation</a>
     */
    public ListGeofenceCollectionsPublisher listGeofenceCollectionsPaginator(
            ListGeofenceCollectionsRequest listGeofenceCollectionsRequest) {
        return new ListGeofenceCollectionsPublisher(this, applyPaginatorUserAgent(listGeofenceCollectionsRequest));
    }

    /**
     * <p>
     * Lists geofences stored in a given geofence collection.
     * </p>
     *
     * @param listGeofencesRequest
     * @return A Java Future containing the result of the ListGeofences operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.ListGeofences
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/ListGeofences" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListGeofencesResponse> listGeofences(ListGeofencesRequest listGeofencesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGeofencesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGeofences");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListGeofencesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListGeofencesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "geofencing.";
            String resolvedHostExpression = "geofencing.";

            CompletableFuture<ListGeofencesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListGeofencesRequest, ListGeofencesResponse>()
                            .withOperationName("ListGeofences")
                            .withMarshaller(new ListGeofencesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(listGeofencesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listGeofencesRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListGeofencesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists geofences stored in a given geofence collection.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listGeofences(software.amazon.awssdk.services.location.model.ListGeofencesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.location.paginators.ListGeofencesPublisher publisher = client.listGeofencesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.location.paginators.ListGeofencesPublisher publisher = client.listGeofencesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.location.model.ListGeofencesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.location.model.ListGeofencesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listGeofences(software.amazon.awssdk.services.location.model.ListGeofencesRequest)} operation.</b>
     * </p>
     *
     * @param listGeofencesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.ListGeofences
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/ListGeofences" target="_top">AWS API
     *      Documentation</a>
     */
    public ListGeofencesPublisher listGeofencesPaginator(ListGeofencesRequest listGeofencesRequest) {
        return new ListGeofencesPublisher(this, applyPaginatorUserAgent(listGeofencesRequest));
    }

    /**
     * <p>
     * Lists map resources in your AWS account.
     * </p>
     *
     * @param listMapsRequest
     * @return A Java Future containing the result of the ListMaps operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.ListMaps
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/ListMaps" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListMapsResponse> listMaps(ListMapsRequest listMapsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMapsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMaps");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListMapsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListMapsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "maps.";
            String resolvedHostExpression = "maps.";

            CompletableFuture<ListMapsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListMapsRequest, ListMapsResponse>().withOperationName("ListMaps")
                            .withMarshaller(new ListMapsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .hostPrefixExpression(resolvedHostExpression).withInput(listMapsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listMapsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListMapsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists map resources in your AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listMaps(software.amazon.awssdk.services.location.model.ListMapsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.location.paginators.ListMapsPublisher publisher = client.listMapsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.location.paginators.ListMapsPublisher publisher = client.listMapsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.location.model.ListMapsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.location.model.ListMapsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listMaps(software.amazon.awssdk.services.location.model.ListMapsRequest)} operation.</b>
     * </p>
     *
     * @param listMapsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.ListMaps
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/ListMaps" target="_top">AWS API
     *      Documentation</a>
     */
    public ListMapsPublisher listMapsPaginator(ListMapsRequest listMapsRequest) {
        return new ListMapsPublisher(this, applyPaginatorUserAgent(listMapsRequest));
    }

    /**
     * <p>
     * Lists Place index resources in your AWS account.
     * </p>
     *
     * @param listPlaceIndexesRequest
     * @return A Java Future containing the result of the ListPlaceIndexes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.ListPlaceIndexes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/ListPlaceIndexes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListPlaceIndexesResponse> listPlaceIndexes(ListPlaceIndexesRequest listPlaceIndexesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPlaceIndexesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPlaceIndexes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPlaceIndexesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPlaceIndexesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "places.";
            String resolvedHostExpression = "places.";

            CompletableFuture<ListPlaceIndexesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPlaceIndexesRequest, ListPlaceIndexesResponse>()
                            .withOperationName("ListPlaceIndexes")
                            .withMarshaller(new ListPlaceIndexesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(listPlaceIndexesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listPlaceIndexesRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListPlaceIndexesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists Place index resources in your AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listPlaceIndexes(software.amazon.awssdk.services.location.model.ListPlaceIndexesRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.location.paginators.ListPlaceIndexesPublisher publisher = client.listPlaceIndexesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.location.paginators.ListPlaceIndexesPublisher publisher = client.listPlaceIndexesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.location.model.ListPlaceIndexesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.location.model.ListPlaceIndexesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPlaceIndexes(software.amazon.awssdk.services.location.model.ListPlaceIndexesRequest)} operation.</b>
     * </p>
     *
     * @param listPlaceIndexesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.ListPlaceIndexes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/ListPlaceIndexes" target="_top">AWS API
     *      Documentation</a>
     */
    public ListPlaceIndexesPublisher listPlaceIndexesPaginator(ListPlaceIndexesRequest listPlaceIndexesRequest) {
        return new ListPlaceIndexesPublisher(this, applyPaginatorUserAgent(listPlaceIndexesRequest));
    }

    /**
     * <p>
     * Lists geofence collections currently associated to the given tracker resource.
     * </p>
     *
     * @param listTrackerConsumersRequest
     * @return A Java Future containing the result of the ListTrackerConsumers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.ListTrackerConsumers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/ListTrackerConsumers" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTrackerConsumersResponse> listTrackerConsumers(
            ListTrackerConsumersRequest listTrackerConsumersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTrackerConsumersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTrackerConsumers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTrackerConsumersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTrackerConsumersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "tracking.";
            String resolvedHostExpression = "tracking.";

            CompletableFuture<ListTrackerConsumersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTrackerConsumersRequest, ListTrackerConsumersResponse>()
                            .withOperationName("ListTrackerConsumers")
                            .withMarshaller(new ListTrackerConsumersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(listTrackerConsumersRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listTrackerConsumersRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListTrackerConsumersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists geofence collections currently associated to the given tracker resource.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listTrackerConsumers(software.amazon.awssdk.services.location.model.ListTrackerConsumersRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.location.paginators.ListTrackerConsumersPublisher publisher = client.listTrackerConsumersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.location.paginators.ListTrackerConsumersPublisher publisher = client.listTrackerConsumersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.location.model.ListTrackerConsumersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.location.model.ListTrackerConsumersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTrackerConsumers(software.amazon.awssdk.services.location.model.ListTrackerConsumersRequest)}
     * operation.</b>
     * </p>
     *
     * @param listTrackerConsumersRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.ListTrackerConsumers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/ListTrackerConsumers" target="_top">AWS
     *      API Documentation</a>
     */
    public ListTrackerConsumersPublisher listTrackerConsumersPaginator(ListTrackerConsumersRequest listTrackerConsumersRequest) {
        return new ListTrackerConsumersPublisher(this, applyPaginatorUserAgent(listTrackerConsumersRequest));
    }

    /**
     * <p>
     * Lists tracker resources in your AWS account.
     * </p>
     *
     * @param listTrackersRequest
     * @return A Java Future containing the result of the ListTrackers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.ListTrackers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/ListTrackers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTrackersResponse> listTrackers(ListTrackersRequest listTrackersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTrackersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTrackers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTrackersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListTrackersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "tracking.";
            String resolvedHostExpression = "tracking.";

            CompletableFuture<ListTrackersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTrackersRequest, ListTrackersResponse>()
                            .withOperationName("ListTrackers").withMarshaller(new ListTrackersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(listTrackersRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listTrackersRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListTrackersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists tracker resources in your AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listTrackers(software.amazon.awssdk.services.location.model.ListTrackersRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.location.paginators.ListTrackersPublisher publisher = client.listTrackersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.location.paginators.ListTrackersPublisher publisher = client.listTrackersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.location.model.ListTrackersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.location.model.ListTrackersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTrackers(software.amazon.awssdk.services.location.model.ListTrackersRequest)} operation.</b>
     * </p>
     *
     * @param listTrackersRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.ListTrackers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/ListTrackers" target="_top">AWS API
     *      Documentation</a>
     */
    public ListTrackersPublisher listTrackersPaginator(ListTrackersRequest listTrackersRequest) {
        return new ListTrackersPublisher(this, applyPaginatorUserAgent(listTrackersRequest));
    }

    /**
     * <p>
     * Stores a geofence to a given geofence collection, or updates the geometry of an existing geofence if a geofence
     * ID is included in the request.
     * </p>
     *
     * @param putGeofenceRequest
     * @return A Java Future containing the result of the PutGeofence operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>ConflictException The request was unsuccessful due to a conflict.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.PutGeofence
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/PutGeofence" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutGeofenceResponse> putGeofence(PutGeofenceRequest putGeofenceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putGeofenceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutGeofence");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutGeofenceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutGeofenceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "geofencing.";
            String resolvedHostExpression = "geofencing.";

            CompletableFuture<PutGeofenceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutGeofenceRequest, PutGeofenceResponse>()
                            .withOperationName("PutGeofence").withMarshaller(new PutGeofenceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(putGeofenceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = putGeofenceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<PutGeofenceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Reverse geocodes a given coordinate and returns a legible address. Allows you to search for Places or points of
     * interest near a given position.
     * </p>
     * <note>
     * <p>
     * By using Places, you agree that AWS may transmit your API queries to your selected third party provider for
     * processing, which may be outside the AWS region you are currently using.
     * </p>
     * <p>
     * Because of licensing limitations, you may not use HERE to store results for locations in Japan. For more
     * information, see the <a href="https://aws.amazon.com/service-terms/">AWS Service Terms</a> for Amazon Location
     * Service.
     * </p>
     * </note>
     *
     * @param searchPlaceIndexForPositionRequest
     * @return A Java Future containing the result of the SearchPlaceIndexForPosition operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.SearchPlaceIndexForPosition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/SearchPlaceIndexForPosition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SearchPlaceIndexForPositionResponse> searchPlaceIndexForPosition(
            SearchPlaceIndexForPositionRequest searchPlaceIndexForPositionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchPlaceIndexForPositionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchPlaceIndexForPosition");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SearchPlaceIndexForPositionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SearchPlaceIndexForPositionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "places.";
            String resolvedHostExpression = "places.";

            CompletableFuture<SearchPlaceIndexForPositionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SearchPlaceIndexForPositionRequest, SearchPlaceIndexForPositionResponse>()
                            .withOperationName("SearchPlaceIndexForPosition")
                            .withMarshaller(new SearchPlaceIndexForPositionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(searchPlaceIndexForPositionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = searchPlaceIndexForPositionRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<SearchPlaceIndexForPositionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Geocodes free-form text, such as an address, name, city, or region to allow you to search for Places or points of
     * interest.
     * </p>
     * <p>
     * Includes the option to apply additional parameters to narrow your list of results.
     * </p>
     * <note>
     * <p>
     * You can search for places near a given position using <code>BiasPosition</code>, or filter results within a
     * bounding box using <code>FilterBBox</code>. Providing both parameters simultaneously returns an error.
     * </p>
     * </note> <note>
     * <p>
     * By using Places, you agree that AWS may transmit your API queries to your selected third party provider for
     * processing, which may be outside the AWS region you are currently using.
     * </p>
     * <p>
     * Also, when using HERE as your data provider, you may not (a) use HERE Places for Asset Management, or (b) select
     * the <code>Storage</code> option for the <code>IntendedUse</code> parameter when requesting Places in Japan. For
     * more information, see the <a href="https://aws.amazon.com/service-terms/">AWS Service Terms</a> for Amazon
     * Location Service.
     * </p>
     * </note>
     *
     * @param searchPlaceIndexForTextRequest
     * @return A Java Future containing the result of the SearchPlaceIndexForText operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied due to insufficient access or permission. Check with an
     *         administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.SearchPlaceIndexForText
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/SearchPlaceIndexForText"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SearchPlaceIndexForTextResponse> searchPlaceIndexForText(
            SearchPlaceIndexForTextRequest searchPlaceIndexForTextRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchPlaceIndexForTextRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchPlaceIndexForText");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SearchPlaceIndexForTextResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SearchPlaceIndexForTextResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "places.";
            String resolvedHostExpression = "places.";

            CompletableFuture<SearchPlaceIndexForTextResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SearchPlaceIndexForTextRequest, SearchPlaceIndexForTextResponse>()
                            .withOperationName("SearchPlaceIndexForText")
                            .withMarshaller(new SearchPlaceIndexForTextRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(searchPlaceIndexForTextRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = searchPlaceIndexForTextRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<SearchPlaceIndexForTextResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(LocationException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends LocationRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
