/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.location;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.location.model.AccessDeniedException;
import software.amazon.awssdk.services.location.model.AssociateTrackerConsumerRequest;
import software.amazon.awssdk.services.location.model.AssociateTrackerConsumerResponse;
import software.amazon.awssdk.services.location.model.BatchDeleteDevicePositionHistoryRequest;
import software.amazon.awssdk.services.location.model.BatchDeleteDevicePositionHistoryResponse;
import software.amazon.awssdk.services.location.model.BatchDeleteGeofenceRequest;
import software.amazon.awssdk.services.location.model.BatchDeleteGeofenceResponse;
import software.amazon.awssdk.services.location.model.BatchEvaluateGeofencesRequest;
import software.amazon.awssdk.services.location.model.BatchEvaluateGeofencesResponse;
import software.amazon.awssdk.services.location.model.BatchGetDevicePositionRequest;
import software.amazon.awssdk.services.location.model.BatchGetDevicePositionResponse;
import software.amazon.awssdk.services.location.model.BatchPutGeofenceRequest;
import software.amazon.awssdk.services.location.model.BatchPutGeofenceResponse;
import software.amazon.awssdk.services.location.model.BatchUpdateDevicePositionRequest;
import software.amazon.awssdk.services.location.model.BatchUpdateDevicePositionResponse;
import software.amazon.awssdk.services.location.model.CalculateRouteMatrixRequest;
import software.amazon.awssdk.services.location.model.CalculateRouteMatrixResponse;
import software.amazon.awssdk.services.location.model.CalculateRouteRequest;
import software.amazon.awssdk.services.location.model.CalculateRouteResponse;
import software.amazon.awssdk.services.location.model.ConflictException;
import software.amazon.awssdk.services.location.model.CreateGeofenceCollectionRequest;
import software.amazon.awssdk.services.location.model.CreateGeofenceCollectionResponse;
import software.amazon.awssdk.services.location.model.CreateKeyRequest;
import software.amazon.awssdk.services.location.model.CreateKeyResponse;
import software.amazon.awssdk.services.location.model.CreateMapRequest;
import software.amazon.awssdk.services.location.model.CreateMapResponse;
import software.amazon.awssdk.services.location.model.CreatePlaceIndexRequest;
import software.amazon.awssdk.services.location.model.CreatePlaceIndexResponse;
import software.amazon.awssdk.services.location.model.CreateRouteCalculatorRequest;
import software.amazon.awssdk.services.location.model.CreateRouteCalculatorResponse;
import software.amazon.awssdk.services.location.model.CreateTrackerRequest;
import software.amazon.awssdk.services.location.model.CreateTrackerResponse;
import software.amazon.awssdk.services.location.model.DeleteGeofenceCollectionRequest;
import software.amazon.awssdk.services.location.model.DeleteGeofenceCollectionResponse;
import software.amazon.awssdk.services.location.model.DeleteKeyRequest;
import software.amazon.awssdk.services.location.model.DeleteKeyResponse;
import software.amazon.awssdk.services.location.model.DeleteMapRequest;
import software.amazon.awssdk.services.location.model.DeleteMapResponse;
import software.amazon.awssdk.services.location.model.DeletePlaceIndexRequest;
import software.amazon.awssdk.services.location.model.DeletePlaceIndexResponse;
import software.amazon.awssdk.services.location.model.DeleteRouteCalculatorRequest;
import software.amazon.awssdk.services.location.model.DeleteRouteCalculatorResponse;
import software.amazon.awssdk.services.location.model.DeleteTrackerRequest;
import software.amazon.awssdk.services.location.model.DeleteTrackerResponse;
import software.amazon.awssdk.services.location.model.DescribeGeofenceCollectionRequest;
import software.amazon.awssdk.services.location.model.DescribeGeofenceCollectionResponse;
import software.amazon.awssdk.services.location.model.DescribeKeyRequest;
import software.amazon.awssdk.services.location.model.DescribeKeyResponse;
import software.amazon.awssdk.services.location.model.DescribeMapRequest;
import software.amazon.awssdk.services.location.model.DescribeMapResponse;
import software.amazon.awssdk.services.location.model.DescribePlaceIndexRequest;
import software.amazon.awssdk.services.location.model.DescribePlaceIndexResponse;
import software.amazon.awssdk.services.location.model.DescribeRouteCalculatorRequest;
import software.amazon.awssdk.services.location.model.DescribeRouteCalculatorResponse;
import software.amazon.awssdk.services.location.model.DescribeTrackerRequest;
import software.amazon.awssdk.services.location.model.DescribeTrackerResponse;
import software.amazon.awssdk.services.location.model.DisassociateTrackerConsumerRequest;
import software.amazon.awssdk.services.location.model.DisassociateTrackerConsumerResponse;
import software.amazon.awssdk.services.location.model.GetDevicePositionHistoryRequest;
import software.amazon.awssdk.services.location.model.GetDevicePositionHistoryResponse;
import software.amazon.awssdk.services.location.model.GetDevicePositionRequest;
import software.amazon.awssdk.services.location.model.GetDevicePositionResponse;
import software.amazon.awssdk.services.location.model.GetGeofenceRequest;
import software.amazon.awssdk.services.location.model.GetGeofenceResponse;
import software.amazon.awssdk.services.location.model.GetMapGlyphsRequest;
import software.amazon.awssdk.services.location.model.GetMapGlyphsResponse;
import software.amazon.awssdk.services.location.model.GetMapSpritesRequest;
import software.amazon.awssdk.services.location.model.GetMapSpritesResponse;
import software.amazon.awssdk.services.location.model.GetMapStyleDescriptorRequest;
import software.amazon.awssdk.services.location.model.GetMapStyleDescriptorResponse;
import software.amazon.awssdk.services.location.model.GetMapTileRequest;
import software.amazon.awssdk.services.location.model.GetMapTileResponse;
import software.amazon.awssdk.services.location.model.GetPlaceRequest;
import software.amazon.awssdk.services.location.model.GetPlaceResponse;
import software.amazon.awssdk.services.location.model.InternalServerException;
import software.amazon.awssdk.services.location.model.ListDevicePositionsRequest;
import software.amazon.awssdk.services.location.model.ListDevicePositionsResponse;
import software.amazon.awssdk.services.location.model.ListGeofenceCollectionsRequest;
import software.amazon.awssdk.services.location.model.ListGeofenceCollectionsResponse;
import software.amazon.awssdk.services.location.model.ListGeofencesRequest;
import software.amazon.awssdk.services.location.model.ListGeofencesResponse;
import software.amazon.awssdk.services.location.model.ListKeysRequest;
import software.amazon.awssdk.services.location.model.ListKeysResponse;
import software.amazon.awssdk.services.location.model.ListMapsRequest;
import software.amazon.awssdk.services.location.model.ListMapsResponse;
import software.amazon.awssdk.services.location.model.ListPlaceIndexesRequest;
import software.amazon.awssdk.services.location.model.ListPlaceIndexesResponse;
import software.amazon.awssdk.services.location.model.ListRouteCalculatorsRequest;
import software.amazon.awssdk.services.location.model.ListRouteCalculatorsResponse;
import software.amazon.awssdk.services.location.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.location.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.location.model.ListTrackerConsumersRequest;
import software.amazon.awssdk.services.location.model.ListTrackerConsumersResponse;
import software.amazon.awssdk.services.location.model.ListTrackersRequest;
import software.amazon.awssdk.services.location.model.ListTrackersResponse;
import software.amazon.awssdk.services.location.model.LocationException;
import software.amazon.awssdk.services.location.model.PutGeofenceRequest;
import software.amazon.awssdk.services.location.model.PutGeofenceResponse;
import software.amazon.awssdk.services.location.model.ResourceNotFoundException;
import software.amazon.awssdk.services.location.model.SearchPlaceIndexForPositionRequest;
import software.amazon.awssdk.services.location.model.SearchPlaceIndexForPositionResponse;
import software.amazon.awssdk.services.location.model.SearchPlaceIndexForSuggestionsRequest;
import software.amazon.awssdk.services.location.model.SearchPlaceIndexForSuggestionsResponse;
import software.amazon.awssdk.services.location.model.SearchPlaceIndexForTextRequest;
import software.amazon.awssdk.services.location.model.SearchPlaceIndexForTextResponse;
import software.amazon.awssdk.services.location.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.location.model.TagResourceRequest;
import software.amazon.awssdk.services.location.model.TagResourceResponse;
import software.amazon.awssdk.services.location.model.ThrottlingException;
import software.amazon.awssdk.services.location.model.UntagResourceRequest;
import software.amazon.awssdk.services.location.model.UntagResourceResponse;
import software.amazon.awssdk.services.location.model.UpdateGeofenceCollectionRequest;
import software.amazon.awssdk.services.location.model.UpdateGeofenceCollectionResponse;
import software.amazon.awssdk.services.location.model.UpdateKeyRequest;
import software.amazon.awssdk.services.location.model.UpdateKeyResponse;
import software.amazon.awssdk.services.location.model.UpdateMapRequest;
import software.amazon.awssdk.services.location.model.UpdateMapResponse;
import software.amazon.awssdk.services.location.model.UpdatePlaceIndexRequest;
import software.amazon.awssdk.services.location.model.UpdatePlaceIndexResponse;
import software.amazon.awssdk.services.location.model.UpdateRouteCalculatorRequest;
import software.amazon.awssdk.services.location.model.UpdateRouteCalculatorResponse;
import software.amazon.awssdk.services.location.model.UpdateTrackerRequest;
import software.amazon.awssdk.services.location.model.UpdateTrackerResponse;
import software.amazon.awssdk.services.location.model.ValidationException;
import software.amazon.awssdk.services.location.transform.AssociateTrackerConsumerRequestMarshaller;
import software.amazon.awssdk.services.location.transform.BatchDeleteDevicePositionHistoryRequestMarshaller;
import software.amazon.awssdk.services.location.transform.BatchDeleteGeofenceRequestMarshaller;
import software.amazon.awssdk.services.location.transform.BatchEvaluateGeofencesRequestMarshaller;
import software.amazon.awssdk.services.location.transform.BatchGetDevicePositionRequestMarshaller;
import software.amazon.awssdk.services.location.transform.BatchPutGeofenceRequestMarshaller;
import software.amazon.awssdk.services.location.transform.BatchUpdateDevicePositionRequestMarshaller;
import software.amazon.awssdk.services.location.transform.CalculateRouteMatrixRequestMarshaller;
import software.amazon.awssdk.services.location.transform.CalculateRouteRequestMarshaller;
import software.amazon.awssdk.services.location.transform.CreateGeofenceCollectionRequestMarshaller;
import software.amazon.awssdk.services.location.transform.CreateKeyRequestMarshaller;
import software.amazon.awssdk.services.location.transform.CreateMapRequestMarshaller;
import software.amazon.awssdk.services.location.transform.CreatePlaceIndexRequestMarshaller;
import software.amazon.awssdk.services.location.transform.CreateRouteCalculatorRequestMarshaller;
import software.amazon.awssdk.services.location.transform.CreateTrackerRequestMarshaller;
import software.amazon.awssdk.services.location.transform.DeleteGeofenceCollectionRequestMarshaller;
import software.amazon.awssdk.services.location.transform.DeleteKeyRequestMarshaller;
import software.amazon.awssdk.services.location.transform.DeleteMapRequestMarshaller;
import software.amazon.awssdk.services.location.transform.DeletePlaceIndexRequestMarshaller;
import software.amazon.awssdk.services.location.transform.DeleteRouteCalculatorRequestMarshaller;
import software.amazon.awssdk.services.location.transform.DeleteTrackerRequestMarshaller;
import software.amazon.awssdk.services.location.transform.DescribeGeofenceCollectionRequestMarshaller;
import software.amazon.awssdk.services.location.transform.DescribeKeyRequestMarshaller;
import software.amazon.awssdk.services.location.transform.DescribeMapRequestMarshaller;
import software.amazon.awssdk.services.location.transform.DescribePlaceIndexRequestMarshaller;
import software.amazon.awssdk.services.location.transform.DescribeRouteCalculatorRequestMarshaller;
import software.amazon.awssdk.services.location.transform.DescribeTrackerRequestMarshaller;
import software.amazon.awssdk.services.location.transform.DisassociateTrackerConsumerRequestMarshaller;
import software.amazon.awssdk.services.location.transform.GetDevicePositionHistoryRequestMarshaller;
import software.amazon.awssdk.services.location.transform.GetDevicePositionRequestMarshaller;
import software.amazon.awssdk.services.location.transform.GetGeofenceRequestMarshaller;
import software.amazon.awssdk.services.location.transform.GetMapGlyphsRequestMarshaller;
import software.amazon.awssdk.services.location.transform.GetMapSpritesRequestMarshaller;
import software.amazon.awssdk.services.location.transform.GetMapStyleDescriptorRequestMarshaller;
import software.amazon.awssdk.services.location.transform.GetMapTileRequestMarshaller;
import software.amazon.awssdk.services.location.transform.GetPlaceRequestMarshaller;
import software.amazon.awssdk.services.location.transform.ListDevicePositionsRequestMarshaller;
import software.amazon.awssdk.services.location.transform.ListGeofenceCollectionsRequestMarshaller;
import software.amazon.awssdk.services.location.transform.ListGeofencesRequestMarshaller;
import software.amazon.awssdk.services.location.transform.ListKeysRequestMarshaller;
import software.amazon.awssdk.services.location.transform.ListMapsRequestMarshaller;
import software.amazon.awssdk.services.location.transform.ListPlaceIndexesRequestMarshaller;
import software.amazon.awssdk.services.location.transform.ListRouteCalculatorsRequestMarshaller;
import software.amazon.awssdk.services.location.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.location.transform.ListTrackerConsumersRequestMarshaller;
import software.amazon.awssdk.services.location.transform.ListTrackersRequestMarshaller;
import software.amazon.awssdk.services.location.transform.PutGeofenceRequestMarshaller;
import software.amazon.awssdk.services.location.transform.SearchPlaceIndexForPositionRequestMarshaller;
import software.amazon.awssdk.services.location.transform.SearchPlaceIndexForSuggestionsRequestMarshaller;
import software.amazon.awssdk.services.location.transform.SearchPlaceIndexForTextRequestMarshaller;
import software.amazon.awssdk.services.location.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.location.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.location.transform.UpdateGeofenceCollectionRequestMarshaller;
import software.amazon.awssdk.services.location.transform.UpdateKeyRequestMarshaller;
import software.amazon.awssdk.services.location.transform.UpdateMapRequestMarshaller;
import software.amazon.awssdk.services.location.transform.UpdatePlaceIndexRequestMarshaller;
import software.amazon.awssdk.services.location.transform.UpdateRouteCalculatorRequestMarshaller;
import software.amazon.awssdk.services.location.transform.UpdateTrackerRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link LocationAsyncClient}.
 *
 * @see LocationAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultLocationAsyncClient implements LocationAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultLocationAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final LocationServiceClientConfiguration serviceClientConfiguration;

    protected DefaultLocationAsyncClient(LocationServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates an association between a geofence collection and a tracker resource. This allows the tracker resource to
     * communicate location data to the linked geofence collection.
     * </p>
     * <p>
     * You can associate up to five geofence collections to each tracker resource.
     * </p>
     * <note>
     * <p>
     * Currently not supported — Cross-account configurations, such as creating associations between a tracker resource
     * in one account and a geofence collection in another account.
     * </p>
     * </note>
     *
     * @param associateTrackerConsumerRequest
     * @return A Java Future containing the result of the AssociateTrackerConsumer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>ConflictException The request was unsuccessful because of a conflict.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ServiceQuotaExceededException The operation was denied because the request would exceed the maximum
     *         <a href="https://docs.aws.amazon.com/location/latest/developerguide/location-quotas.html">quota</a> set
     *         for Amazon Location Service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.AssociateTrackerConsumer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/AssociateTrackerConsumer"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateTrackerConsumerResponse> associateTrackerConsumer(
            AssociateTrackerConsumerRequest associateTrackerConsumerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateTrackerConsumerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateTrackerConsumer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateTrackerConsumerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateTrackerConsumerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "tracking.";
            String resolvedHostExpression = "tracking.";

            CompletableFuture<AssociateTrackerConsumerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateTrackerConsumerRequest, AssociateTrackerConsumerResponse>()
                            .withOperationName("AssociateTrackerConsumer")
                            .withMarshaller(new AssociateTrackerConsumerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(associateTrackerConsumerRequest));
            CompletableFuture<AssociateTrackerConsumerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the position history of one or more devices from a tracker resource.
     * </p>
     *
     * @param batchDeleteDevicePositionHistoryRequest
     * @return A Java Future containing the result of the BatchDeleteDevicePositionHistory operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.BatchDeleteDevicePositionHistory
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/BatchDeleteDevicePositionHistory"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchDeleteDevicePositionHistoryResponse> batchDeleteDevicePositionHistory(
            BatchDeleteDevicePositionHistoryRequest batchDeleteDevicePositionHistoryRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                batchDeleteDevicePositionHistoryRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchDeleteDevicePositionHistory");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchDeleteDevicePositionHistoryResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, BatchDeleteDevicePositionHistoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "tracking.";
            String resolvedHostExpression = "tracking.";

            CompletableFuture<BatchDeleteDevicePositionHistoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchDeleteDevicePositionHistoryRequest, BatchDeleteDevicePositionHistoryResponse>()
                            .withOperationName("BatchDeleteDevicePositionHistory")
                            .withMarshaller(new BatchDeleteDevicePositionHistoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(batchDeleteDevicePositionHistoryRequest));
            CompletableFuture<BatchDeleteDevicePositionHistoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a batch of geofences from a geofence collection.
     * </p>
     * <note>
     * <p>
     * This operation deletes the resource permanently.
     * </p>
     * </note>
     *
     * @param batchDeleteGeofenceRequest
     * @return A Java Future containing the result of the BatchDeleteGeofence operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.BatchDeleteGeofence
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/BatchDeleteGeofence" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchDeleteGeofenceResponse> batchDeleteGeofence(
            BatchDeleteGeofenceRequest batchDeleteGeofenceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchDeleteGeofenceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchDeleteGeofence");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchDeleteGeofenceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchDeleteGeofenceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "geofencing.";
            String resolvedHostExpression = "geofencing.";

            CompletableFuture<BatchDeleteGeofenceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchDeleteGeofenceRequest, BatchDeleteGeofenceResponse>()
                            .withOperationName("BatchDeleteGeofence")
                            .withMarshaller(new BatchDeleteGeofenceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(batchDeleteGeofenceRequest));
            CompletableFuture<BatchDeleteGeofenceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Evaluates device positions against the geofence geometries from a given geofence collection.
     * </p>
     * <p>
     * This operation always returns an empty response because geofences are asynchronously evaluated. The evaluation
     * determines if the device has entered or exited a geofenced area, and then publishes one of the following events
     * to Amazon EventBridge:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ENTER</code> if Amazon Location determines that the tracked device has entered a geofenced area.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>EXIT</code> if Amazon Location determines that the tracked device has exited a geofenced area.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The last geofence that a device was observed within is tracked for 30 days after the most recent device position
     * update.
     * </p>
     * </note> <note>
     * <p>
     * Geofence evaluation uses the given device position. It does not account for the optional <code>Accuracy</code> of
     * a <code>DevicePositionUpdate</code>.
     * </p>
     * </note> <note>
     * <p>
     * The <code>DeviceID</code> is used as a string to represent the device. You do not need to have a
     * <code>Tracker</code> associated with the <code>DeviceID</code>.
     * </p>
     * </note>
     *
     * @param batchEvaluateGeofencesRequest
     * @return A Java Future containing the result of the BatchEvaluateGeofences operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.BatchEvaluateGeofences
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/BatchEvaluateGeofences"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchEvaluateGeofencesResponse> batchEvaluateGeofences(
            BatchEvaluateGeofencesRequest batchEvaluateGeofencesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchEvaluateGeofencesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchEvaluateGeofences");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchEvaluateGeofencesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchEvaluateGeofencesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "geofencing.";
            String resolvedHostExpression = "geofencing.";

            CompletableFuture<BatchEvaluateGeofencesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchEvaluateGeofencesRequest, BatchEvaluateGeofencesResponse>()
                            .withOperationName("BatchEvaluateGeofences")
                            .withMarshaller(new BatchEvaluateGeofencesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(batchEvaluateGeofencesRequest));
            CompletableFuture<BatchEvaluateGeofencesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the latest device positions for requested devices.
     * </p>
     *
     * @param batchGetDevicePositionRequest
     * @return A Java Future containing the result of the BatchGetDevicePosition operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.BatchGetDevicePosition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/BatchGetDevicePosition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchGetDevicePositionResponse> batchGetDevicePosition(
            BatchGetDevicePositionRequest batchGetDevicePositionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchGetDevicePositionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchGetDevicePosition");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchGetDevicePositionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchGetDevicePositionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "tracking.";
            String resolvedHostExpression = "tracking.";

            CompletableFuture<BatchGetDevicePositionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchGetDevicePositionRequest, BatchGetDevicePositionResponse>()
                            .withOperationName("BatchGetDevicePosition")
                            .withMarshaller(new BatchGetDevicePositionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(batchGetDevicePositionRequest));
            CompletableFuture<BatchGetDevicePositionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * A batch request for storing geofence geometries into a given geofence collection, or updates the geometry of an
     * existing geofence if a geofence ID is included in the request.
     * </p>
     *
     * @param batchPutGeofenceRequest
     * @return A Java Future containing the result of the BatchPutGeofence operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.BatchPutGeofence
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/BatchPutGeofence" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<BatchPutGeofenceResponse> batchPutGeofence(BatchPutGeofenceRequest batchPutGeofenceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchPutGeofenceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchPutGeofence");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchPutGeofenceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchPutGeofenceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "geofencing.";
            String resolvedHostExpression = "geofencing.";

            CompletableFuture<BatchPutGeofenceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchPutGeofenceRequest, BatchPutGeofenceResponse>()
                            .withOperationName("BatchPutGeofence")
                            .withMarshaller(new BatchPutGeofenceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(batchPutGeofenceRequest));
            CompletableFuture<BatchPutGeofenceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Uploads position update data for one or more devices to a tracker resource (up to 10 devices per batch). Amazon
     * Location uses the data when it reports the last known device position and position history. Amazon Location
     * retains location data for 30 days.
     * </p>
     * <note>
     * <p>
     * Position updates are handled based on the <code>PositionFiltering</code> property of the tracker. When
     * <code>PositionFiltering</code> is set to <code>TimeBased</code>, updates are evaluated against linked geofence
     * collections, and location data is stored at a maximum of one position per 30 second interval. If your update
     * frequency is more often than every 30 seconds, only one update per 30 seconds is stored for each unique device
     * ID.
     * </p>
     * <p>
     * When <code>PositionFiltering</code> is set to <code>DistanceBased</code> filtering, location data is stored and
     * evaluated against linked geofence collections only if the device has moved more than 30 m (98.4 ft).
     * </p>
     * <p>
     * When <code>PositionFiltering</code> is set to <code>AccuracyBased</code> filtering, location data is stored and
     * evaluated against linked geofence collections only if the device has moved more than the measured accuracy. For
     * example, if two consecutive updates from a device have a horizontal accuracy of 5 m and 10 m, the second update
     * is neither stored or evaluated if the device has moved less than 15 m. If <code>PositionFiltering</code> is set
     * to <code>AccuracyBased</code> filtering, Amazon Location uses the default value <code>{ "Horizontal": 0}</code>
     * when accuracy is not provided on a <code>DevicePositionUpdate</code>.
     * </p>
     * </note>
     *
     * @param batchUpdateDevicePositionRequest
     * @return A Java Future containing the result of the BatchUpdateDevicePosition operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.BatchUpdateDevicePosition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/BatchUpdateDevicePosition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchUpdateDevicePositionResponse> batchUpdateDevicePosition(
            BatchUpdateDevicePositionRequest batchUpdateDevicePositionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchUpdateDevicePositionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchUpdateDevicePosition");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchUpdateDevicePositionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchUpdateDevicePositionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "tracking.";
            String resolvedHostExpression = "tracking.";

            CompletableFuture<BatchUpdateDevicePositionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchUpdateDevicePositionRequest, BatchUpdateDevicePositionResponse>()
                            .withOperationName("BatchUpdateDevicePosition")
                            .withMarshaller(new BatchUpdateDevicePositionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(batchUpdateDevicePositionRequest));
            CompletableFuture<BatchUpdateDevicePositionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <a href="https://docs.aws.amazon.com/location/latest/developerguide/calculate-route.html">Calculates a route</a>
     * given the following required parameters: <code>DeparturePosition</code> and <code>DestinationPosition</code>.
     * Requires that you first <a
     * href="https://docs.aws.amazon.com/location-routes/latest/APIReference/API_CreateRouteCalculator.html">create a
     * route calculator resource</a>.
     * </p>
     * <p>
     * By default, a request that doesn't specify a departure time uses the best time of day to travel with the best
     * traffic conditions when calculating the route.
     * </p>
     * <p>
     * Additional options include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/location/latest/developerguide/departure-time.html">Specifying a departure
     * time</a> using either <code>DepartureTime</code> or <code>DepartNow</code>. This calculates a route based on
     * predictive traffic data at the given time.
     * </p>
     * <note>
     * <p>
     * You can't specify both <code>DepartureTime</code> and <code>DepartNow</code> in a single request. Specifying both
     * parameters returns a validation error.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/location/latest/developerguide/travel-mode.html">Specifying a travel
     * mode</a> using TravelMode sets the transportation mode used to calculate the routes. This also lets you specify
     * additional route preferences in <code>CarModeOptions</code> if traveling by <code>Car</code>, or
     * <code>TruckModeOptions</code> if traveling by <code>Truck</code>.
     * </p>
     * <note>
     * <p>
     * If you specify <code>walking</code> for the travel mode and your data provider is Esri, the start and destination
     * must be within 40km.
     * </p>
     * </note></li>
     * </ul>
     *
     * @param calculateRouteRequest
     * @return A Java Future containing the result of the CalculateRoute operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.CalculateRoute
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/CalculateRoute" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CalculateRouteResponse> calculateRoute(CalculateRouteRequest calculateRouteRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, calculateRouteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CalculateRoute");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CalculateRouteResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CalculateRouteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "routes.";
            String resolvedHostExpression = "routes.";

            CompletableFuture<CalculateRouteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CalculateRouteRequest, CalculateRouteResponse>()
                            .withOperationName("CalculateRoute")
                            .withMarshaller(new CalculateRouteRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(calculateRouteRequest));
            CompletableFuture<CalculateRouteResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <a href="https://docs.aws.amazon.com/location/latest/developerguide/calculate-route-matrix.html"> Calculates a
     * route matrix</a> given the following required parameters: <code>DeparturePositions</code> and
     * <code>DestinationPositions</code>. <code>CalculateRouteMatrix</code> calculates routes and returns the travel
     * time and travel distance from each departure position to each destination position in the request. For example,
     * given departure positions A and B, and destination positions X and Y, <code>CalculateRouteMatrix</code> will
     * return time and distance for routes from A to X, A to Y, B to X, and B to Y (in that order). The number of
     * results returned (and routes calculated) will be the number of <code>DeparturePositions</code> times the number
     * of <code>DestinationPositions</code>.
     * </p>
     * <note>
     * <p>
     * Your account is charged for each route calculated, not the number of requests.
     * </p>
     * </note>
     * <p>
     * Requires that you first <a
     * href="https://docs.aws.amazon.com/location-routes/latest/APIReference/API_CreateRouteCalculator.html">create a
     * route calculator resource</a>.
     * </p>
     * <p>
     * By default, a request that doesn't specify a departure time uses the best time of day to travel with the best
     * traffic conditions when calculating routes.
     * </p>
     * <p>
     * Additional options include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/location/latest/developerguide/departure-time.html"> Specifying a departure
     * time</a> using either <code>DepartureTime</code> or <code>DepartNow</code>. This calculates routes based on
     * predictive traffic data at the given time.
     * </p>
     * <note>
     * <p>
     * You can't specify both <code>DepartureTime</code> and <code>DepartNow</code> in a single request. Specifying both
     * parameters returns a validation error.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/location/latest/developerguide/travel-mode.html">Specifying a travel
     * mode</a> using TravelMode sets the transportation mode used to calculate the routes. This also lets you specify
     * additional route preferences in <code>CarModeOptions</code> if traveling by <code>Car</code>, or
     * <code>TruckModeOptions</code> if traveling by <code>Truck</code>.
     * </p>
     * </li>
     * </ul>
     *
     * @param calculateRouteMatrixRequest
     * @return A Java Future containing the result of the CalculateRouteMatrix operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.CalculateRouteMatrix
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/CalculateRouteMatrix" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CalculateRouteMatrixResponse> calculateRouteMatrix(
            CalculateRouteMatrixRequest calculateRouteMatrixRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, calculateRouteMatrixRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CalculateRouteMatrix");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CalculateRouteMatrixResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CalculateRouteMatrixResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "routes.";
            String resolvedHostExpression = "routes.";

            CompletableFuture<CalculateRouteMatrixResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CalculateRouteMatrixRequest, CalculateRouteMatrixResponse>()
                            .withOperationName("CalculateRouteMatrix")
                            .withMarshaller(new CalculateRouteMatrixRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(calculateRouteMatrixRequest));
            CompletableFuture<CalculateRouteMatrixResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a geofence collection, which manages and stores geofences.
     * </p>
     *
     * @param createGeofenceCollectionRequest
     * @return A Java Future containing the result of the CreateGeofenceCollection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ConflictException The request was unsuccessful because of a conflict.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ServiceQuotaExceededException The operation was denied because the request would exceed the maximum
     *         <a href="https://docs.aws.amazon.com/location/latest/developerguide/location-quotas.html">quota</a> set
     *         for Amazon Location Service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.CreateGeofenceCollection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/CreateGeofenceCollection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateGeofenceCollectionResponse> createGeofenceCollection(
            CreateGeofenceCollectionRequest createGeofenceCollectionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createGeofenceCollectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateGeofenceCollection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateGeofenceCollectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateGeofenceCollectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "geofencing.";
            String resolvedHostExpression = "geofencing.";

            CompletableFuture<CreateGeofenceCollectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateGeofenceCollectionRequest, CreateGeofenceCollectionResponse>()
                            .withOperationName("CreateGeofenceCollection")
                            .withMarshaller(new CreateGeofenceCollectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(createGeofenceCollectionRequest));
            CompletableFuture<CreateGeofenceCollectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an API key resource in your Amazon Web Services account, which lets you grant actions for Amazon Location
     * resources to the API key bearer.
     * </p>
     * <note>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/location/latest/developerguide/using-apikeys.html">Using API keys</a>.
     * </p>
     * </note>
     *
     * @param createKeyRequest
     * @return A Java Future containing the result of the CreateKey operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ConflictException The request was unsuccessful because of a conflict.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ServiceQuotaExceededException The operation was denied because the request would exceed the maximum
     *         <a href="https://docs.aws.amazon.com/location/latest/developerguide/location-quotas.html">quota</a> set
     *         for Amazon Location Service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.CreateKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/CreateKey" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateKeyResponse> createKey(CreateKeyRequest createKeyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createKeyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateKey");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateKeyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateKeyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "metadata.";
            String resolvedHostExpression = "metadata.";

            CompletableFuture<CreateKeyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateKeyRequest, CreateKeyResponse>().withOperationName("CreateKey")
                            .withMarshaller(new CreateKeyRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .hostPrefixExpression(resolvedHostExpression).withInput(createKeyRequest));
            CompletableFuture<CreateKeyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a map resource in your Amazon Web Services account, which provides map tiles of different styles sourced
     * from global location data providers.
     * </p>
     * <note>
     * <p>
     * If your application is tracking or routing assets you use in your business, such as delivery vehicles or
     * employees, you must not use Esri as your geolocation provider. See section 82 of the <a
     * href="http://aws.amazon.com/service-terms">Amazon Web Services service terms</a> for more details.
     * </p>
     * </note>
     *
     * @param createMapRequest
     * @return A Java Future containing the result of the CreateMap operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ConflictException The request was unsuccessful because of a conflict.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ServiceQuotaExceededException The operation was denied because the request would exceed the maximum
     *         <a href="https://docs.aws.amazon.com/location/latest/developerguide/location-quotas.html">quota</a> set
     *         for Amazon Location Service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.CreateMap
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/CreateMap" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateMapResponse> createMap(CreateMapRequest createMapRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMapRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMap");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateMapResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateMapResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "maps.";
            String resolvedHostExpression = "maps.";

            CompletableFuture<CreateMapResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateMapRequest, CreateMapResponse>().withOperationName("CreateMap")
                            .withMarshaller(new CreateMapRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .hostPrefixExpression(resolvedHostExpression).withInput(createMapRequest));
            CompletableFuture<CreateMapResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a place index resource in your Amazon Web Services account. Use a place index resource to geocode
     * addresses and other text queries by using the <code>SearchPlaceIndexForText</code> operation, and reverse geocode
     * coordinates by using the <code>SearchPlaceIndexForPosition</code> operation, and enable autosuggestions by using
     * the <code>SearchPlaceIndexForSuggestions</code> operation.
     * </p>
     * <note>
     * <p>
     * If your application is tracking or routing assets you use in your business, such as delivery vehicles or
     * employees, you must not use Esri as your geolocation provider. See section 82 of the <a
     * href="http://aws.amazon.com/service-terms">Amazon Web Services service terms</a> for more details.
     * </p>
     * </note>
     *
     * @param createPlaceIndexRequest
     * @return A Java Future containing the result of the CreatePlaceIndex operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ConflictException The request was unsuccessful because of a conflict.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ServiceQuotaExceededException The operation was denied because the request would exceed the maximum
     *         <a href="https://docs.aws.amazon.com/location/latest/developerguide/location-quotas.html">quota</a> set
     *         for Amazon Location Service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.CreatePlaceIndex
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/CreatePlaceIndex" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePlaceIndexResponse> createPlaceIndex(CreatePlaceIndexRequest createPlaceIndexRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPlaceIndexRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePlaceIndex");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePlaceIndexResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreatePlaceIndexResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "places.";
            String resolvedHostExpression = "places.";

            CompletableFuture<CreatePlaceIndexResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePlaceIndexRequest, CreatePlaceIndexResponse>()
                            .withOperationName("CreatePlaceIndex")
                            .withMarshaller(new CreatePlaceIndexRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(createPlaceIndexRequest));
            CompletableFuture<CreatePlaceIndexResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a route calculator resource in your Amazon Web Services account.
     * </p>
     * <p>
     * You can send requests to a route calculator resource to estimate travel time, distance, and get directions. A
     * route calculator sources traffic and road network data from your chosen data provider.
     * </p>
     * <note>
     * <p>
     * If your application is tracking or routing assets you use in your business, such as delivery vehicles or
     * employees, you must not use Esri as your geolocation provider. See section 82 of the <a
     * href="http://aws.amazon.com/service-terms">Amazon Web Services service terms</a> for more details.
     * </p>
     * </note>
     *
     * @param createRouteCalculatorRequest
     * @return A Java Future containing the result of the CreateRouteCalculator operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ConflictException The request was unsuccessful because of a conflict.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ServiceQuotaExceededException The operation was denied because the request would exceed the maximum
     *         <a href="https://docs.aws.amazon.com/location/latest/developerguide/location-quotas.html">quota</a> set
     *         for Amazon Location Service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.CreateRouteCalculator
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/CreateRouteCalculator"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateRouteCalculatorResponse> createRouteCalculator(
            CreateRouteCalculatorRequest createRouteCalculatorRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRouteCalculatorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRouteCalculator");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateRouteCalculatorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateRouteCalculatorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "routes.";
            String resolvedHostExpression = "routes.";

            CompletableFuture<CreateRouteCalculatorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateRouteCalculatorRequest, CreateRouteCalculatorResponse>()
                            .withOperationName("CreateRouteCalculator")
                            .withMarshaller(new CreateRouteCalculatorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(createRouteCalculatorRequest));
            CompletableFuture<CreateRouteCalculatorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a tracker resource in your Amazon Web Services account, which lets you retrieve current and historical
     * location of devices.
     * </p>
     *
     * @param createTrackerRequest
     * @return A Java Future containing the result of the CreateTracker operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ConflictException The request was unsuccessful because of a conflict.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ServiceQuotaExceededException The operation was denied because the request would exceed the maximum
     *         <a href="https://docs.aws.amazon.com/location/latest/developerguide/location-quotas.html">quota</a> set
     *         for Amazon Location Service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.CreateTracker
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/CreateTracker" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateTrackerResponse> createTracker(CreateTrackerRequest createTrackerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTrackerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTracker");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateTrackerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateTrackerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "tracking.";
            String resolvedHostExpression = "tracking.";

            CompletableFuture<CreateTrackerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateTrackerRequest, CreateTrackerResponse>()
                            .withOperationName("CreateTracker")
                            .withMarshaller(new CreateTrackerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(createTrackerRequest));
            CompletableFuture<CreateTrackerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a geofence collection from your Amazon Web Services account.
     * </p>
     * <note>
     * <p>
     * This operation deletes the resource permanently. If the geofence collection is the target of a tracker resource,
     * the devices will no longer be monitored.
     * </p>
     * </note>
     *
     * @param deleteGeofenceCollectionRequest
     * @return A Java Future containing the result of the DeleteGeofenceCollection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.DeleteGeofenceCollection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/DeleteGeofenceCollection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteGeofenceCollectionResponse> deleteGeofenceCollection(
            DeleteGeofenceCollectionRequest deleteGeofenceCollectionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteGeofenceCollectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteGeofenceCollection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteGeofenceCollectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteGeofenceCollectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "geofencing.";
            String resolvedHostExpression = "geofencing.";

            CompletableFuture<DeleteGeofenceCollectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteGeofenceCollectionRequest, DeleteGeofenceCollectionResponse>()
                            .withOperationName("DeleteGeofenceCollection")
                            .withMarshaller(new DeleteGeofenceCollectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(deleteGeofenceCollectionRequest));
            CompletableFuture<DeleteGeofenceCollectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified API key. The API key must have been deactivated more than 90 days previously.
     * </p>
     *
     * @param deleteKeyRequest
     * @return A Java Future containing the result of the DeleteKey operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.DeleteKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/DeleteKey" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteKeyResponse> deleteKey(DeleteKeyRequest deleteKeyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteKeyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteKey");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteKeyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteKeyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "metadata.";
            String resolvedHostExpression = "metadata.";

            CompletableFuture<DeleteKeyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteKeyRequest, DeleteKeyResponse>().withOperationName("DeleteKey")
                            .withMarshaller(new DeleteKeyRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .hostPrefixExpression(resolvedHostExpression).withInput(deleteKeyRequest));
            CompletableFuture<DeleteKeyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a map resource from your Amazon Web Services account.
     * </p>
     * <note>
     * <p>
     * This operation deletes the resource permanently. If the map is being used in an application, the map may not
     * render.
     * </p>
     * </note>
     *
     * @param deleteMapRequest
     * @return A Java Future containing the result of the DeleteMap operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.DeleteMap
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/DeleteMap" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteMapResponse> deleteMap(DeleteMapRequest deleteMapRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMapRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMap");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteMapResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteMapResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "maps.";
            String resolvedHostExpression = "maps.";

            CompletableFuture<DeleteMapResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteMapRequest, DeleteMapResponse>().withOperationName("DeleteMap")
                            .withMarshaller(new DeleteMapRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .hostPrefixExpression(resolvedHostExpression).withInput(deleteMapRequest));
            CompletableFuture<DeleteMapResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a place index resource from your Amazon Web Services account.
     * </p>
     * <note>
     * <p>
     * This operation deletes the resource permanently.
     * </p>
     * </note>
     *
     * @param deletePlaceIndexRequest
     * @return A Java Future containing the result of the DeletePlaceIndex operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.DeletePlaceIndex
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/DeletePlaceIndex" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePlaceIndexResponse> deletePlaceIndex(DeletePlaceIndexRequest deletePlaceIndexRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePlaceIndexRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePlaceIndex");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePlaceIndexResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeletePlaceIndexResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "places.";
            String resolvedHostExpression = "places.";

            CompletableFuture<DeletePlaceIndexResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePlaceIndexRequest, DeletePlaceIndexResponse>()
                            .withOperationName("DeletePlaceIndex")
                            .withMarshaller(new DeletePlaceIndexRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(deletePlaceIndexRequest));
            CompletableFuture<DeletePlaceIndexResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a route calculator resource from your Amazon Web Services account.
     * </p>
     * <note>
     * <p>
     * This operation deletes the resource permanently.
     * </p>
     * </note>
     *
     * @param deleteRouteCalculatorRequest
     * @return A Java Future containing the result of the DeleteRouteCalculator operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.DeleteRouteCalculator
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/DeleteRouteCalculator"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteRouteCalculatorResponse> deleteRouteCalculator(
            DeleteRouteCalculatorRequest deleteRouteCalculatorRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRouteCalculatorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRouteCalculator");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteRouteCalculatorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteRouteCalculatorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "routes.";
            String resolvedHostExpression = "routes.";

            CompletableFuture<DeleteRouteCalculatorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteRouteCalculatorRequest, DeleteRouteCalculatorResponse>()
                            .withOperationName("DeleteRouteCalculator")
                            .withMarshaller(new DeleteRouteCalculatorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(deleteRouteCalculatorRequest));
            CompletableFuture<DeleteRouteCalculatorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a tracker resource from your Amazon Web Services account.
     * </p>
     * <note>
     * <p>
     * This operation deletes the resource permanently. If the tracker resource is in use, you may encounter an error.
     * Make sure that the target resource isn't a dependency for your applications.
     * </p>
     * </note>
     *
     * @param deleteTrackerRequest
     * @return A Java Future containing the result of the DeleteTracker operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.DeleteTracker
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/DeleteTracker" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteTrackerResponse> deleteTracker(DeleteTrackerRequest deleteTrackerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTrackerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTracker");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteTrackerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteTrackerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "tracking.";
            String resolvedHostExpression = "tracking.";

            CompletableFuture<DeleteTrackerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteTrackerRequest, DeleteTrackerResponse>()
                            .withOperationName("DeleteTracker")
                            .withMarshaller(new DeleteTrackerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(deleteTrackerRequest));
            CompletableFuture<DeleteTrackerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the geofence collection details.
     * </p>
     *
     * @param describeGeofenceCollectionRequest
     * @return A Java Future containing the result of the DescribeGeofenceCollection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.DescribeGeofenceCollection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/DescribeGeofenceCollection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeGeofenceCollectionResponse> describeGeofenceCollection(
            DescribeGeofenceCollectionRequest describeGeofenceCollectionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeGeofenceCollectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeGeofenceCollection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeGeofenceCollectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeGeofenceCollectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "geofencing.";
            String resolvedHostExpression = "geofencing.";

            CompletableFuture<DescribeGeofenceCollectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeGeofenceCollectionRequest, DescribeGeofenceCollectionResponse>()
                            .withOperationName("DescribeGeofenceCollection")
                            .withMarshaller(new DescribeGeofenceCollectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(describeGeofenceCollectionRequest));
            CompletableFuture<DescribeGeofenceCollectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the API key resource details.
     * </p>
     *
     * @param describeKeyRequest
     * @return A Java Future containing the result of the DescribeKey operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.DescribeKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/DescribeKey" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeKeyResponse> describeKey(DescribeKeyRequest describeKeyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeKeyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeKey");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeKeyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeKeyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "metadata.";
            String resolvedHostExpression = "metadata.";

            CompletableFuture<DescribeKeyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeKeyRequest, DescribeKeyResponse>()
                            .withOperationName("DescribeKey").withMarshaller(new DescribeKeyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(describeKeyRequest));
            CompletableFuture<DescribeKeyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the map resource details.
     * </p>
     *
     * @param describeMapRequest
     * @return A Java Future containing the result of the DescribeMap operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.DescribeMap
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/DescribeMap" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeMapResponse> describeMap(DescribeMapRequest describeMapRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeMapRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeMap");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeMapResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeMapResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "maps.";
            String resolvedHostExpression = "maps.";

            CompletableFuture<DescribeMapResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeMapRequest, DescribeMapResponse>()
                            .withOperationName("DescribeMap").withMarshaller(new DescribeMapRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(describeMapRequest));
            CompletableFuture<DescribeMapResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the place index resource details.
     * </p>
     *
     * @param describePlaceIndexRequest
     * @return A Java Future containing the result of the DescribePlaceIndex operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.DescribePlaceIndex
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/DescribePlaceIndex" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribePlaceIndexResponse> describePlaceIndex(DescribePlaceIndexRequest describePlaceIndexRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describePlaceIndexRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribePlaceIndex");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribePlaceIndexResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribePlaceIndexResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "places.";
            String resolvedHostExpression = "places.";

            CompletableFuture<DescribePlaceIndexResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribePlaceIndexRequest, DescribePlaceIndexResponse>()
                            .withOperationName("DescribePlaceIndex")
                            .withMarshaller(new DescribePlaceIndexRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(describePlaceIndexRequest));
            CompletableFuture<DescribePlaceIndexResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the route calculator resource details.
     * </p>
     *
     * @param describeRouteCalculatorRequest
     * @return A Java Future containing the result of the DescribeRouteCalculator operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.DescribeRouteCalculator
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/DescribeRouteCalculator"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeRouteCalculatorResponse> describeRouteCalculator(
            DescribeRouteCalculatorRequest describeRouteCalculatorRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeRouteCalculatorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeRouteCalculator");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeRouteCalculatorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeRouteCalculatorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "routes.";
            String resolvedHostExpression = "routes.";

            CompletableFuture<DescribeRouteCalculatorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeRouteCalculatorRequest, DescribeRouteCalculatorResponse>()
                            .withOperationName("DescribeRouteCalculator")
                            .withMarshaller(new DescribeRouteCalculatorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(describeRouteCalculatorRequest));
            CompletableFuture<DescribeRouteCalculatorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the tracker resource details.
     * </p>
     *
     * @param describeTrackerRequest
     * @return A Java Future containing the result of the DescribeTracker operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.DescribeTracker
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/DescribeTracker" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeTrackerResponse> describeTracker(DescribeTrackerRequest describeTrackerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeTrackerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTracker");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeTrackerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeTrackerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "tracking.";
            String resolvedHostExpression = "tracking.";

            CompletableFuture<DescribeTrackerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeTrackerRequest, DescribeTrackerResponse>()
                            .withOperationName("DescribeTracker")
                            .withMarshaller(new DescribeTrackerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(describeTrackerRequest));
            CompletableFuture<DescribeTrackerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the association between a tracker resource and a geofence collection.
     * </p>
     * <note>
     * <p>
     * Once you unlink a tracker resource from a geofence collection, the tracker positions will no longer be
     * automatically evaluated against geofences.
     * </p>
     * </note>
     *
     * @param disassociateTrackerConsumerRequest
     * @return A Java Future containing the result of the DisassociateTrackerConsumer operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.DisassociateTrackerConsumer
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/DisassociateTrackerConsumer"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateTrackerConsumerResponse> disassociateTrackerConsumer(
            DisassociateTrackerConsumerRequest disassociateTrackerConsumerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateTrackerConsumerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateTrackerConsumer");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateTrackerConsumerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateTrackerConsumerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "tracking.";
            String resolvedHostExpression = "tracking.";

            CompletableFuture<DisassociateTrackerConsumerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateTrackerConsumerRequest, DisassociateTrackerConsumerResponse>()
                            .withOperationName("DisassociateTrackerConsumer")
                            .withMarshaller(new DisassociateTrackerConsumerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(disassociateTrackerConsumerRequest));
            CompletableFuture<DisassociateTrackerConsumerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a device's most recent position according to its sample time.
     * </p>
     * <note>
     * <p>
     * Device positions are deleted after 30 days.
     * </p>
     * </note>
     *
     * @param getDevicePositionRequest
     * @return A Java Future containing the result of the GetDevicePosition operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.GetDevicePosition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/GetDevicePosition" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDevicePositionResponse> getDevicePosition(GetDevicePositionRequest getDevicePositionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDevicePositionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDevicePosition");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDevicePositionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDevicePositionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "tracking.";
            String resolvedHostExpression = "tracking.";

            CompletableFuture<GetDevicePositionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDevicePositionRequest, GetDevicePositionResponse>()
                            .withOperationName("GetDevicePosition")
                            .withMarshaller(new GetDevicePositionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(getDevicePositionRequest));
            CompletableFuture<GetDevicePositionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the device position history from a tracker resource within a specified range of time.
     * </p>
     * <note>
     * <p>
     * Device positions are deleted after 30 days.
     * </p>
     * </note>
     *
     * @param getDevicePositionHistoryRequest
     * @return A Java Future containing the result of the GetDevicePositionHistory operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.GetDevicePositionHistory
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/GetDevicePositionHistory"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDevicePositionHistoryResponse> getDevicePositionHistory(
            GetDevicePositionHistoryRequest getDevicePositionHistoryRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDevicePositionHistoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDevicePositionHistory");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDevicePositionHistoryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDevicePositionHistoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "tracking.";
            String resolvedHostExpression = "tracking.";

            CompletableFuture<GetDevicePositionHistoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDevicePositionHistoryRequest, GetDevicePositionHistoryResponse>()
                            .withOperationName("GetDevicePositionHistory")
                            .withMarshaller(new GetDevicePositionHistoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(getDevicePositionHistoryRequest));
            CompletableFuture<GetDevicePositionHistoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the geofence details from a geofence collection.
     * </p>
     *
     * @param getGeofenceRequest
     * @return A Java Future containing the result of the GetGeofence operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.GetGeofence
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/GetGeofence" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetGeofenceResponse> getGeofence(GetGeofenceRequest getGeofenceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getGeofenceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetGeofence");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetGeofenceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetGeofenceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "geofencing.";
            String resolvedHostExpression = "geofencing.";

            CompletableFuture<GetGeofenceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetGeofenceRequest, GetGeofenceResponse>()
                            .withOperationName("GetGeofence").withMarshaller(new GetGeofenceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(getGeofenceRequest));
            CompletableFuture<GetGeofenceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves glyphs used to display labels on a map.
     * </p>
     *
     * @param getMapGlyphsRequest
     * @return A Java Future containing the result of the GetMapGlyphs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.GetMapGlyphs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/GetMapGlyphs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetMapGlyphsResponse> getMapGlyphs(GetMapGlyphsRequest getMapGlyphsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMapGlyphsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMapGlyphs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(false).build();

            HttpResponseHandler<GetMapGlyphsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetMapGlyphsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "maps.";
            String resolvedHostExpression = "maps.";

            CompletableFuture<GetMapGlyphsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMapGlyphsRequest, GetMapGlyphsResponse>()
                            .withOperationName("GetMapGlyphs").withMarshaller(new GetMapGlyphsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(getMapGlyphsRequest));
            CompletableFuture<GetMapGlyphsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the sprite sheet corresponding to a map resource. The sprite sheet is a PNG image paired with a JSON
     * document describing the offsets of individual icons that will be displayed on a rendered map.
     * </p>
     *
     * @param getMapSpritesRequest
     * @return A Java Future containing the result of the GetMapSprites operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.GetMapSprites
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/GetMapSprites" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetMapSpritesResponse> getMapSprites(GetMapSpritesRequest getMapSpritesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMapSpritesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMapSprites");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(false).build();

            HttpResponseHandler<GetMapSpritesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetMapSpritesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "maps.";
            String resolvedHostExpression = "maps.";

            CompletableFuture<GetMapSpritesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMapSpritesRequest, GetMapSpritesResponse>()
                            .withOperationName("GetMapSprites")
                            .withMarshaller(new GetMapSpritesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(getMapSpritesRequest));
            CompletableFuture<GetMapSpritesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the map style descriptor from a map resource.
     * </p>
     * <p>
     * The style descriptor contains speciﬁcations on how features render on a map. For example, what data to display,
     * what order to display the data in, and the style for the data. Style descriptors follow the Mapbox Style
     * Specification.
     * </p>
     *
     * @param getMapStyleDescriptorRequest
     * @return A Java Future containing the result of the GetMapStyleDescriptor operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.GetMapStyleDescriptor
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/GetMapStyleDescriptor"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetMapStyleDescriptorResponse> getMapStyleDescriptor(
            GetMapStyleDescriptorRequest getMapStyleDescriptorRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMapStyleDescriptorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMapStyleDescriptor");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(false).build();

            HttpResponseHandler<GetMapStyleDescriptorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetMapStyleDescriptorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "maps.";
            String resolvedHostExpression = "maps.";

            CompletableFuture<GetMapStyleDescriptorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMapStyleDescriptorRequest, GetMapStyleDescriptorResponse>()
                            .withOperationName("GetMapStyleDescriptor")
                            .withMarshaller(new GetMapStyleDescriptorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(getMapStyleDescriptorRequest));
            CompletableFuture<GetMapStyleDescriptorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a vector data tile from the map resource. Map tiles are used by clients to render a map. they're
     * addressed using a grid arrangement with an X coordinate, Y coordinate, and Z (zoom) level.
     * </p>
     * <p>
     * The origin (0, 0) is the top left of the map. Increasing the zoom level by 1 doubles both the X and Y dimensions,
     * so a tile containing data for the entire world at (0/0/0) will be split into 4 tiles at zoom 1 (1/0/0, 1/0/1,
     * 1/1/0, 1/1/1).
     * </p>
     *
     * @param getMapTileRequest
     * @return A Java Future containing the result of the GetMapTile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.GetMapTile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/GetMapTile" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetMapTileResponse> getMapTile(GetMapTileRequest getMapTileRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMapTileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMapTile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(false).build();

            HttpResponseHandler<GetMapTileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetMapTileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "maps.";
            String resolvedHostExpression = "maps.";

            CompletableFuture<GetMapTileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMapTileRequest, GetMapTileResponse>().withOperationName("GetMapTile")
                            .withMarshaller(new GetMapTileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(getMapTileRequest));
            CompletableFuture<GetMapTileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Finds a place by its unique ID. A <code>PlaceId</code> is returned by other search operations.
     * </p>
     * <note>
     * <p>
     * A PlaceId is valid only if all of the following are the same in the original search request and the call to
     * <code>GetPlace</code>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Customer Amazon Web Services account
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon Web Services Region
     * </p>
     * </li>
     * <li>
     * <p>
     * Data provider specified in the place index resource
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param getPlaceRequest
     * @return A Java Future containing the result of the GetPlace operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.GetPlace
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/GetPlace" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetPlaceResponse> getPlace(GetPlaceRequest getPlaceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPlaceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPlace");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPlaceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetPlaceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "places.";
            String resolvedHostExpression = "places.";

            CompletableFuture<GetPlaceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPlaceRequest, GetPlaceResponse>().withOperationName("GetPlace")
                            .withMarshaller(new GetPlaceRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .hostPrefixExpression(resolvedHostExpression).withInput(getPlaceRequest));
            CompletableFuture<GetPlaceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * A batch request to retrieve all device positions.
     * </p>
     *
     * @param listDevicePositionsRequest
     * @return A Java Future containing the result of the ListDevicePositions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.ListDevicePositions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/ListDevicePositions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDevicePositionsResponse> listDevicePositions(
            ListDevicePositionsRequest listDevicePositionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDevicePositionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDevicePositions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDevicePositionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDevicePositionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "tracking.";
            String resolvedHostExpression = "tracking.";

            CompletableFuture<ListDevicePositionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDevicePositionsRequest, ListDevicePositionsResponse>()
                            .withOperationName("ListDevicePositions")
                            .withMarshaller(new ListDevicePositionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(listDevicePositionsRequest));
            CompletableFuture<ListDevicePositionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists geofence collections in your Amazon Web Services account.
     * </p>
     *
     * @param listGeofenceCollectionsRequest
     * @return A Java Future containing the result of the ListGeofenceCollections operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.ListGeofenceCollections
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/ListGeofenceCollections"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListGeofenceCollectionsResponse> listGeofenceCollections(
            ListGeofenceCollectionsRequest listGeofenceCollectionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGeofenceCollectionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGeofenceCollections");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListGeofenceCollectionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListGeofenceCollectionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "geofencing.";
            String resolvedHostExpression = "geofencing.";

            CompletableFuture<ListGeofenceCollectionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListGeofenceCollectionsRequest, ListGeofenceCollectionsResponse>()
                            .withOperationName("ListGeofenceCollections")
                            .withMarshaller(new ListGeofenceCollectionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(listGeofenceCollectionsRequest));
            CompletableFuture<ListGeofenceCollectionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists geofences stored in a given geofence collection.
     * </p>
     *
     * @param listGeofencesRequest
     * @return A Java Future containing the result of the ListGeofences operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.ListGeofences
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/ListGeofences" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListGeofencesResponse> listGeofences(ListGeofencesRequest listGeofencesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGeofencesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGeofences");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListGeofencesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListGeofencesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "geofencing.";
            String resolvedHostExpression = "geofencing.";

            CompletableFuture<ListGeofencesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListGeofencesRequest, ListGeofencesResponse>()
                            .withOperationName("ListGeofences")
                            .withMarshaller(new ListGeofencesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(listGeofencesRequest));
            CompletableFuture<ListGeofencesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists API key resources in your Amazon Web Services account.
     * </p>
     *
     * @param listKeysRequest
     * @return A Java Future containing the result of the ListKeys operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.ListKeys
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/ListKeys" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListKeysResponse> listKeys(ListKeysRequest listKeysRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listKeysRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListKeys");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListKeysResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListKeysResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "metadata.";
            String resolvedHostExpression = "metadata.";

            CompletableFuture<ListKeysResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListKeysRequest, ListKeysResponse>().withOperationName("ListKeys")
                            .withMarshaller(new ListKeysRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .hostPrefixExpression(resolvedHostExpression).withInput(listKeysRequest));
            CompletableFuture<ListKeysResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists map resources in your Amazon Web Services account.
     * </p>
     *
     * @param listMapsRequest
     * @return A Java Future containing the result of the ListMaps operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.ListMaps
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/ListMaps" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListMapsResponse> listMaps(ListMapsRequest listMapsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMapsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMaps");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListMapsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListMapsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "maps.";
            String resolvedHostExpression = "maps.";

            CompletableFuture<ListMapsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListMapsRequest, ListMapsResponse>().withOperationName("ListMaps")
                            .withMarshaller(new ListMapsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .hostPrefixExpression(resolvedHostExpression).withInput(listMapsRequest));
            CompletableFuture<ListMapsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists place index resources in your Amazon Web Services account.
     * </p>
     *
     * @param listPlaceIndexesRequest
     * @return A Java Future containing the result of the ListPlaceIndexes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.ListPlaceIndexes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/ListPlaceIndexes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListPlaceIndexesResponse> listPlaceIndexes(ListPlaceIndexesRequest listPlaceIndexesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPlaceIndexesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPlaceIndexes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPlaceIndexesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPlaceIndexesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "places.";
            String resolvedHostExpression = "places.";

            CompletableFuture<ListPlaceIndexesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPlaceIndexesRequest, ListPlaceIndexesResponse>()
                            .withOperationName("ListPlaceIndexes")
                            .withMarshaller(new ListPlaceIndexesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(listPlaceIndexesRequest));
            CompletableFuture<ListPlaceIndexesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists route calculator resources in your Amazon Web Services account.
     * </p>
     *
     * @param listRouteCalculatorsRequest
     * @return A Java Future containing the result of the ListRouteCalculators operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.ListRouteCalculators
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/ListRouteCalculators" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListRouteCalculatorsResponse> listRouteCalculators(
            ListRouteCalculatorsRequest listRouteCalculatorsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRouteCalculatorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRouteCalculators");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRouteCalculatorsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListRouteCalculatorsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "routes.";
            String resolvedHostExpression = "routes.";

            CompletableFuture<ListRouteCalculatorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRouteCalculatorsRequest, ListRouteCalculatorsResponse>()
                            .withOperationName("ListRouteCalculators")
                            .withMarshaller(new ListRouteCalculatorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(listRouteCalculatorsRequest));
            CompletableFuture<ListRouteCalculatorsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of tags that are applied to the specified Amazon Location resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "metadata.";
            String resolvedHostExpression = "metadata.";

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists geofence collections currently associated to the given tracker resource.
     * </p>
     *
     * @param listTrackerConsumersRequest
     * @return A Java Future containing the result of the ListTrackerConsumers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.ListTrackerConsumers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/ListTrackerConsumers" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTrackerConsumersResponse> listTrackerConsumers(
            ListTrackerConsumersRequest listTrackerConsumersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTrackerConsumersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTrackerConsumers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTrackerConsumersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTrackerConsumersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "tracking.";
            String resolvedHostExpression = "tracking.";

            CompletableFuture<ListTrackerConsumersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTrackerConsumersRequest, ListTrackerConsumersResponse>()
                            .withOperationName("ListTrackerConsumers")
                            .withMarshaller(new ListTrackerConsumersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(listTrackerConsumersRequest));
            CompletableFuture<ListTrackerConsumersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists tracker resources in your Amazon Web Services account.
     * </p>
     *
     * @param listTrackersRequest
     * @return A Java Future containing the result of the ListTrackers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.ListTrackers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/ListTrackers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTrackersResponse> listTrackers(ListTrackersRequest listTrackersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTrackersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTrackers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTrackersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListTrackersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "tracking.";
            String resolvedHostExpression = "tracking.";

            CompletableFuture<ListTrackersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTrackersRequest, ListTrackersResponse>()
                            .withOperationName("ListTrackers").withMarshaller(new ListTrackersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(listTrackersRequest));
            CompletableFuture<ListTrackersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stores a geofence geometry in a given geofence collection, or updates the geometry of an existing geofence if a
     * geofence ID is included in the request.
     * </p>
     *
     * @param putGeofenceRequest
     * @return A Java Future containing the result of the PutGeofence operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>ConflictException The request was unsuccessful because of a conflict.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.PutGeofence
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/PutGeofence" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutGeofenceResponse> putGeofence(PutGeofenceRequest putGeofenceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putGeofenceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutGeofence");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutGeofenceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutGeofenceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "geofencing.";
            String resolvedHostExpression = "geofencing.";

            CompletableFuture<PutGeofenceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutGeofenceRequest, PutGeofenceResponse>()
                            .withOperationName("PutGeofence").withMarshaller(new PutGeofenceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(putGeofenceRequest));
            CompletableFuture<PutGeofenceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Reverse geocodes a given coordinate and returns a legible address. Allows you to search for Places or points of
     * interest near a given position.
     * </p>
     *
     * @param searchPlaceIndexForPositionRequest
     * @return A Java Future containing the result of the SearchPlaceIndexForPosition operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.SearchPlaceIndexForPosition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/SearchPlaceIndexForPosition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SearchPlaceIndexForPositionResponse> searchPlaceIndexForPosition(
            SearchPlaceIndexForPositionRequest searchPlaceIndexForPositionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchPlaceIndexForPositionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchPlaceIndexForPosition");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SearchPlaceIndexForPositionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SearchPlaceIndexForPositionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "places.";
            String resolvedHostExpression = "places.";

            CompletableFuture<SearchPlaceIndexForPositionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SearchPlaceIndexForPositionRequest, SearchPlaceIndexForPositionResponse>()
                            .withOperationName("SearchPlaceIndexForPosition")
                            .withMarshaller(new SearchPlaceIndexForPositionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(searchPlaceIndexForPositionRequest));
            CompletableFuture<SearchPlaceIndexForPositionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Generates suggestions for addresses and points of interest based on partial or misspelled free-form text. This
     * operation is also known as autocomplete, autosuggest, or fuzzy matching.
     * </p>
     * <p>
     * Optional parameters let you narrow your search results by bounding box or country, or bias your search toward a
     * specific position on the globe.
     * </p>
     * <note>
     * <p>
     * You can search for suggested place names near a specified position by using <code>BiasPosition</code>, or filter
     * results within a bounding box by using <code>FilterBBox</code>. These parameters are mutually exclusive; using
     * both <code>BiasPosition</code> and <code>FilterBBox</code> in the same command returns an error.
     * </p>
     * </note>
     *
     * @param searchPlaceIndexForSuggestionsRequest
     * @return A Java Future containing the result of the SearchPlaceIndexForSuggestions operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.SearchPlaceIndexForSuggestions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/SearchPlaceIndexForSuggestions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SearchPlaceIndexForSuggestionsResponse> searchPlaceIndexForSuggestions(
            SearchPlaceIndexForSuggestionsRequest searchPlaceIndexForSuggestionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                searchPlaceIndexForSuggestionsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchPlaceIndexForSuggestions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SearchPlaceIndexForSuggestionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SearchPlaceIndexForSuggestionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "places.";
            String resolvedHostExpression = "places.";

            CompletableFuture<SearchPlaceIndexForSuggestionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SearchPlaceIndexForSuggestionsRequest, SearchPlaceIndexForSuggestionsResponse>()
                            .withOperationName("SearchPlaceIndexForSuggestions")
                            .withMarshaller(new SearchPlaceIndexForSuggestionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(searchPlaceIndexForSuggestionsRequest));
            CompletableFuture<SearchPlaceIndexForSuggestionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Geocodes free-form text, such as an address, name, city, or region to allow you to search for Places or points of
     * interest.
     * </p>
     * <p>
     * Optional parameters let you narrow your search results by bounding box or country, or bias your search toward a
     * specific position on the globe.
     * </p>
     * <note>
     * <p>
     * You can search for places near a given position using <code>BiasPosition</code>, or filter results within a
     * bounding box using <code>FilterBBox</code>. Providing both parameters simultaneously returns an error.
     * </p>
     * </note>
     * <p>
     * Search results are returned in order of highest to lowest relevance.
     * </p>
     *
     * @param searchPlaceIndexForTextRequest
     * @return A Java Future containing the result of the SearchPlaceIndexForText operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.SearchPlaceIndexForText
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/SearchPlaceIndexForText"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SearchPlaceIndexForTextResponse> searchPlaceIndexForText(
            SearchPlaceIndexForTextRequest searchPlaceIndexForTextRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchPlaceIndexForTextRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchPlaceIndexForText");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SearchPlaceIndexForTextResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SearchPlaceIndexForTextResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "places.";
            String resolvedHostExpression = "places.";

            CompletableFuture<SearchPlaceIndexForTextResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SearchPlaceIndexForTextRequest, SearchPlaceIndexForTextResponse>()
                            .withOperationName("SearchPlaceIndexForText")
                            .withMarshaller(new SearchPlaceIndexForTextRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(searchPlaceIndexForTextRequest));
            CompletableFuture<SearchPlaceIndexForTextResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Assigns one or more tags (key-value pairs) to the specified Amazon Location Service resource.
     * </p>
     * <p>
     * Tags can help you organize and categorize your resources. You can also use them to scope user permissions, by
     * granting a user permission to access or change only resources with certain tag values.
     * </p>
     * <p>
     * You can use the <code>TagResource</code> operation with an Amazon Location Service resource that already has
     * tags. If you specify a new tag key for the resource, this tag is appended to the tags already associated with the
     * resource. If you specify a tag key that's already associated with the resource, the new tag value that you
     * specify replaces the previous value for that tag.
     * </p>
     * <p>
     * You can associate up to 50 tags with a resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "metadata.";
            String resolvedHostExpression = "metadata.";

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes one or more tags from the specified Amazon Location resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "metadata.";
            String resolvedHostExpression = "metadata.";

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the specified properties of a given geofence collection.
     * </p>
     *
     * @param updateGeofenceCollectionRequest
     * @return A Java Future containing the result of the UpdateGeofenceCollection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.UpdateGeofenceCollection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/UpdateGeofenceCollection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateGeofenceCollectionResponse> updateGeofenceCollection(
            UpdateGeofenceCollectionRequest updateGeofenceCollectionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateGeofenceCollectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateGeofenceCollection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateGeofenceCollectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateGeofenceCollectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "geofencing.";
            String resolvedHostExpression = "geofencing.";

            CompletableFuture<UpdateGeofenceCollectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateGeofenceCollectionRequest, UpdateGeofenceCollectionResponse>()
                            .withOperationName("UpdateGeofenceCollection")
                            .withMarshaller(new UpdateGeofenceCollectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(updateGeofenceCollectionRequest));
            CompletableFuture<UpdateGeofenceCollectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the specified properties of a given API key resource.
     * </p>
     *
     * @param updateKeyRequest
     * @return A Java Future containing the result of the UpdateKey operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.UpdateKey
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/UpdateKey" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateKeyResponse> updateKey(UpdateKeyRequest updateKeyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateKeyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateKey");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateKeyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateKeyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "metadata.";
            String resolvedHostExpression = "metadata.";

            CompletableFuture<UpdateKeyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateKeyRequest, UpdateKeyResponse>().withOperationName("UpdateKey")
                            .withMarshaller(new UpdateKeyRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .hostPrefixExpression(resolvedHostExpression).withInput(updateKeyRequest));
            CompletableFuture<UpdateKeyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the specified properties of a given map resource.
     * </p>
     *
     * @param updateMapRequest
     * @return A Java Future containing the result of the UpdateMap operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.UpdateMap
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/UpdateMap" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateMapResponse> updateMap(UpdateMapRequest updateMapRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateMapRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateMap");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateMapResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateMapResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "maps.";
            String resolvedHostExpression = "maps.";

            CompletableFuture<UpdateMapResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateMapRequest, UpdateMapResponse>().withOperationName("UpdateMap")
                            .withMarshaller(new UpdateMapRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .hostPrefixExpression(resolvedHostExpression).withInput(updateMapRequest));
            CompletableFuture<UpdateMapResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the specified properties of a given place index resource.
     * </p>
     *
     * @param updatePlaceIndexRequest
     * @return A Java Future containing the result of the UpdatePlaceIndex operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.UpdatePlaceIndex
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/UpdatePlaceIndex" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdatePlaceIndexResponse> updatePlaceIndex(UpdatePlaceIndexRequest updatePlaceIndexRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePlaceIndexRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePlaceIndex");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdatePlaceIndexResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdatePlaceIndexResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "places.";
            String resolvedHostExpression = "places.";

            CompletableFuture<UpdatePlaceIndexResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdatePlaceIndexRequest, UpdatePlaceIndexResponse>()
                            .withOperationName("UpdatePlaceIndex")
                            .withMarshaller(new UpdatePlaceIndexRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(updatePlaceIndexRequest));
            CompletableFuture<UpdatePlaceIndexResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the specified properties for a given route calculator resource.
     * </p>
     *
     * @param updateRouteCalculatorRequest
     * @return A Java Future containing the result of the UpdateRouteCalculator operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.UpdateRouteCalculator
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/UpdateRouteCalculator"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateRouteCalculatorResponse> updateRouteCalculator(
            UpdateRouteCalculatorRequest updateRouteCalculatorRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRouteCalculatorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRouteCalculator");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateRouteCalculatorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateRouteCalculatorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "routes.";
            String resolvedHostExpression = "routes.";

            CompletableFuture<UpdateRouteCalculatorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateRouteCalculatorRequest, UpdateRouteCalculatorResponse>()
                            .withOperationName("UpdateRouteCalculator")
                            .withMarshaller(new UpdateRouteCalculatorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(updateRouteCalculatorRequest));
            CompletableFuture<UpdateRouteCalculatorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the specified properties of a given tracker resource.
     * </p>
     *
     * @param updateTrackerRequest
     * @return A Java Future containing the result of the UpdateTracker operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request has failed to process because of an unknown server error,
     *         exception, or failure.</li>
     *         <li>ResourceNotFoundException The resource that you've entered was not found in your AWS account.</li>
     *         <li>AccessDeniedException The request was denied because of insufficient access or permissions. Check
     *         with an administrator to verify your permissions.</li>
     *         <li>ValidationException The input failed to meet the constraints specified by the AWS service.</li>
     *         <li>ThrottlingException The request was denied because of request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LocationException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LocationAsyncClient.UpdateTracker
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/location-2020-11-19/UpdateTracker" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateTrackerResponse> updateTracker(UpdateTrackerRequest updateTrackerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateTrackerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Location");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTracker");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateTrackerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateTrackerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            String hostPrefix = "tracking.";
            String resolvedHostExpression = "tracking.";

            CompletableFuture<UpdateTrackerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateTrackerRequest, UpdateTrackerResponse>()
                            .withOperationName("UpdateTracker")
                            .withMarshaller(new UpdateTrackerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).hostPrefixExpression(resolvedHostExpression)
                            .withInput(updateTrackerRequest));
            CompletableFuture<UpdateTrackerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final LocationServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(LocationException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
