/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.location.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CalculateRouteRequest extends LocationRequest implements
        ToCopyableBuilder<CalculateRouteRequest.Builder, CalculateRouteRequest> {
    private static final SdkField<Instant> ARRIVAL_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("ArrivalTime")
            .getter(getter(CalculateRouteRequest::arrivalTime))
            .setter(setter(Builder::arrivalTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ArrivalTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> CALCULATOR_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CalculatorName").getter(getter(CalculateRouteRequest::calculatorName))
            .setter(setter(Builder::calculatorName))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("CalculatorName").build()).build();

    private static final SdkField<CalculateRouteCarModeOptions> CAR_MODE_OPTIONS_FIELD = SdkField
            .<CalculateRouteCarModeOptions> builder(MarshallingType.SDK_POJO).memberName("CarModeOptions")
            .getter(getter(CalculateRouteRequest::carModeOptions)).setter(setter(Builder::carModeOptions))
            .constructor(CalculateRouteCarModeOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CarModeOptions").build()).build();

    private static final SdkField<Boolean> DEPART_NOW_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("DepartNow").getter(getter(CalculateRouteRequest::departNow)).setter(setter(Builder::departNow))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DepartNow").build()).build();

    private static final SdkField<List<Double>> DEPARTURE_POSITION_FIELD = SdkField
            .<List<Double>> builder(MarshallingType.LIST)
            .memberName("DeparturePosition")
            .getter(getter(CalculateRouteRequest::departurePosition))
            .setter(setter(Builder::departurePosition))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeparturePosition").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Double> builder(MarshallingType.DOUBLE)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> DEPARTURE_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("DepartureTime")
            .getter(getter(CalculateRouteRequest::departureTime))
            .setter(setter(Builder::departureTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DepartureTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<List<Double>> DESTINATION_POSITION_FIELD = SdkField
            .<List<Double>> builder(MarshallingType.LIST)
            .memberName("DestinationPosition")
            .getter(getter(CalculateRouteRequest::destinationPosition))
            .setter(setter(Builder::destinationPosition))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DestinationPosition").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Double> builder(MarshallingType.DOUBLE)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> DISTANCE_UNIT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DistanceUnit").getter(getter(CalculateRouteRequest::distanceUnitAsString))
            .setter(setter(Builder::distanceUnit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DistanceUnit").build()).build();

    private static final SdkField<Boolean> INCLUDE_LEG_GEOMETRY_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("IncludeLegGeometry").getter(getter(CalculateRouteRequest::includeLegGeometry))
            .setter(setter(Builder::includeLegGeometry))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IncludeLegGeometry").build())
            .build();

    private static final SdkField<String> KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Key")
            .getter(getter(CalculateRouteRequest::key)).setter(setter(Builder::key))
            .traits(LocationTrait.builder().location(MarshallLocation.QUERY_PARAM).locationName("key").build()).build();

    private static final SdkField<String> OPTIMIZE_FOR_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OptimizeFor").getter(getter(CalculateRouteRequest::optimizeForAsString))
            .setter(setter(Builder::optimizeFor))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OptimizeFor").build()).build();

    private static final SdkField<String> TRAVEL_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TravelMode").getter(getter(CalculateRouteRequest::travelModeAsString))
            .setter(setter(Builder::travelMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TravelMode").build()).build();

    private static final SdkField<CalculateRouteTruckModeOptions> TRUCK_MODE_OPTIONS_FIELD = SdkField
            .<CalculateRouteTruckModeOptions> builder(MarshallingType.SDK_POJO).memberName("TruckModeOptions")
            .getter(getter(CalculateRouteRequest::truckModeOptions)).setter(setter(Builder::truckModeOptions))
            .constructor(CalculateRouteTruckModeOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TruckModeOptions").build()).build();

    private static final SdkField<List<List<Double>>> WAYPOINT_POSITIONS_FIELD = SdkField
            .<List<List<Double>>> builder(MarshallingType.LIST)
            .memberName("WaypointPositions")
            .getter(getter(CalculateRouteRequest::waypointPositions))
            .setter(setter(Builder::waypointPositions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WaypointPositions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<List<Double>> builder(MarshallingType.LIST)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build(),
                                                    ListTrait
                                                            .builder()
                                                            .memberLocationName(null)
                                                            .memberFieldInfo(
                                                                    SdkField.<Double> builder(MarshallingType.DOUBLE)
                                                                            .traits(LocationTrait.builder()
                                                                                    .location(MarshallLocation.PAYLOAD)
                                                                                    .locationName("member").build()).build())
                                                            .build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ARRIVAL_TIME_FIELD,
            CALCULATOR_NAME_FIELD, CAR_MODE_OPTIONS_FIELD, DEPART_NOW_FIELD, DEPARTURE_POSITION_FIELD, DEPARTURE_TIME_FIELD,
            DESTINATION_POSITION_FIELD, DISTANCE_UNIT_FIELD, INCLUDE_LEG_GEOMETRY_FIELD, KEY_FIELD, OPTIMIZE_FOR_FIELD,
            TRAVEL_MODE_FIELD, TRUCK_MODE_OPTIONS_FIELD, WAYPOINT_POSITIONS_FIELD));

    private final Instant arrivalTime;

    private final String calculatorName;

    private final CalculateRouteCarModeOptions carModeOptions;

    private final Boolean departNow;

    private final List<Double> departurePosition;

    private final Instant departureTime;

    private final List<Double> destinationPosition;

    private final String distanceUnit;

    private final Boolean includeLegGeometry;

    private final String key;

    private final String optimizeFor;

    private final String travelMode;

    private final CalculateRouteTruckModeOptions truckModeOptions;

    private final List<List<Double>> waypointPositions;

    private CalculateRouteRequest(BuilderImpl builder) {
        super(builder);
        this.arrivalTime = builder.arrivalTime;
        this.calculatorName = builder.calculatorName;
        this.carModeOptions = builder.carModeOptions;
        this.departNow = builder.departNow;
        this.departurePosition = builder.departurePosition;
        this.departureTime = builder.departureTime;
        this.destinationPosition = builder.destinationPosition;
        this.distanceUnit = builder.distanceUnit;
        this.includeLegGeometry = builder.includeLegGeometry;
        this.key = builder.key;
        this.optimizeFor = builder.optimizeFor;
        this.travelMode = builder.travelMode;
        this.truckModeOptions = builder.truckModeOptions;
        this.waypointPositions = builder.waypointPositions;
    }

    /**
     * <p>
     * Specifies the desired time of arrival. Uses the given time to calculate the route. Otherwise, the best time of
     * day to travel with the best traffic conditions is used to calculate the route.
     * </p>
     * <note>
     * <p>
     * ArrivalTime is not supported Esri.
     * </p>
     * </note>
     * 
     * @return Specifies the desired time of arrival. Uses the given time to calculate the route. Otherwise, the best
     *         time of day to travel with the best traffic conditions is used to calculate the route.</p> <note>
     *         <p>
     *         ArrivalTime is not supported Esri.
     *         </p>
     */
    public final Instant arrivalTime() {
        return arrivalTime;
    }

    /**
     * <p>
     * The name of the route calculator resource that you want to use to calculate the route.
     * </p>
     * 
     * @return The name of the route calculator resource that you want to use to calculate the route.
     */
    public final String calculatorName() {
        return calculatorName;
    }

    /**
     * <p>
     * Specifies route preferences when traveling by <code>Car</code>, such as avoiding routes that use ferries or
     * tolls.
     * </p>
     * <p>
     * Requirements: <code>TravelMode</code> must be specified as <code>Car</code>.
     * </p>
     * 
     * @return Specifies route preferences when traveling by <code>Car</code>, such as avoiding routes that use ferries
     *         or tolls.</p>
     *         <p>
     *         Requirements: <code>TravelMode</code> must be specified as <code>Car</code>.
     */
    public final CalculateRouteCarModeOptions carModeOptions() {
        return carModeOptions;
    }

    /**
     * <p>
     * Sets the time of departure as the current time. Uses the current time to calculate a route. Otherwise, the best
     * time of day to travel with the best traffic conditions is used to calculate the route.
     * </p>
     * <p>
     * Default Value: <code>false</code>
     * </p>
     * <p>
     * Valid Values: <code>false</code> | <code>true</code>
     * </p>
     * 
     * @return Sets the time of departure as the current time. Uses the current time to calculate a route. Otherwise,
     *         the best time of day to travel with the best traffic conditions is used to calculate the route.</p>
     *         <p>
     *         Default Value: <code>false</code>
     *         </p>
     *         <p>
     *         Valid Values: <code>false</code> | <code>true</code>
     */
    public final Boolean departNow() {
        return departNow;
    }

    /**
     * For responses, this returns true if the service returned a value for the DeparturePosition property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasDeparturePosition() {
        return departurePosition != null && !(departurePosition instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The start position for the route. Defined in <a
     * href="https://earth-info.nga.mil/index.php?dir=wgs84&amp;action=wgs84">World Geodetic System (WGS 84)</a> format:
     * <code>[longitude, latitude]</code>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For example, <code>[-123.115, 49.285]</code>
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * If you specify a departure that's not located on a road, Amazon Location <a
     * href="https://docs.aws.amazon.com/location/latest/developerguide/snap-to-nearby-road.html">moves the position to
     * the nearest road</a>. If Esri is the provider for your route calculator, specifying a route that is longer than
     * 400 km returns a <code>400 RoutesValidationException</code> error.
     * </p>
     * </note>
     * <p>
     * Valid Values: <code>[-180 to 180,-90 to 90]</code>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDeparturePosition} method.
     * </p>
     * 
     * @return The start position for the route. Defined in <a
     *         href="https://earth-info.nga.mil/index.php?dir=wgs84&amp;action=wgs84">World Geodetic System (WGS 84)</a>
     *         format: <code>[longitude, latitude]</code>.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For example, <code>[-123.115, 49.285]</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         If you specify a departure that's not located on a road, Amazon Location <a
     *         href="https://docs.aws.amazon.com/location/latest/developerguide/snap-to-nearby-road.html">moves the
     *         position to the nearest road</a>. If Esri is the provider for your route calculator, specifying a route
     *         that is longer than 400 km returns a <code>400 RoutesValidationException</code> error.
     *         </p>
     *         </note>
     *         <p>
     *         Valid Values: <code>[-180 to 180,-90 to 90]</code>
     */
    public final List<Double> departurePosition() {
        return departurePosition;
    }

    /**
     * <p>
     * Specifies the desired time of departure. Uses the given time to calculate the route. Otherwise, the best time of
     * day to travel with the best traffic conditions is used to calculate the route.
     * </p>
     * <ul>
     * <li>
     * <p>
     * In <a href="https://www.iso.org/iso-8601-date-and-time-format.html">ISO 8601</a> format:
     * <code>YYYY-MM-DDThh:mm:ss.sssZ</code>. For example, <code>2020–07-2T12:15:20.000Z+01:00</code>
     * </p>
     * </li>
     * </ul>
     * 
     * @return Specifies the desired time of departure. Uses the given time to calculate the route. Otherwise, the best
     *         time of day to travel with the best traffic conditions is used to calculate the route.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         In <a href="https://www.iso.org/iso-8601-date-and-time-format.html">ISO 8601</a> format:
     *         <code>YYYY-MM-DDThh:mm:ss.sssZ</code>. For example, <code>2020–07-2T12:15:20.000Z+01:00</code>
     *         </p>
     *         </li>
     */
    public final Instant departureTime() {
        return departureTime;
    }

    /**
     * For responses, this returns true if the service returned a value for the DestinationPosition property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasDestinationPosition() {
        return destinationPosition != null && !(destinationPosition instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The finish position for the route. Defined in <a
     * href="https://earth-info.nga.mil/index.php?dir=wgs84&amp;action=wgs84">World Geodetic System (WGS 84)</a> format:
     * <code>[longitude, latitude]</code>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For example, <code>[-122.339, 47.615]</code>
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * If you specify a destination that's not located on a road, Amazon Location <a
     * href="https://docs.aws.amazon.com/location/latest/developerguide/snap-to-nearby-road.html">moves the position to
     * the nearest road</a>.
     * </p>
     * </note>
     * <p>
     * Valid Values: <code>[-180 to 180,-90 to 90]</code>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDestinationPosition} method.
     * </p>
     * 
     * @return The finish position for the route. Defined in <a
     *         href="https://earth-info.nga.mil/index.php?dir=wgs84&amp;action=wgs84">World Geodetic System (WGS 84)</a>
     *         format: <code>[longitude, latitude]</code>.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For example, <code>[-122.339, 47.615]</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         If you specify a destination that's not located on a road, Amazon Location <a
     *         href="https://docs.aws.amazon.com/location/latest/developerguide/snap-to-nearby-road.html">moves the
     *         position to the nearest road</a>.
     *         </p>
     *         </note>
     *         <p>
     *         Valid Values: <code>[-180 to 180,-90 to 90]</code>
     */
    public final List<Double> destinationPosition() {
        return destinationPosition;
    }

    /**
     * <p>
     * Set the unit system to specify the distance.
     * </p>
     * <p>
     * Default Value: <code>Kilometers</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #distanceUnit} will
     * return {@link DistanceUnit#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #distanceUnitAsString}.
     * </p>
     * 
     * @return Set the unit system to specify the distance.</p>
     *         <p>
     *         Default Value: <code>Kilometers</code>
     * @see DistanceUnit
     */
    public final DistanceUnit distanceUnit() {
        return DistanceUnit.fromValue(distanceUnit);
    }

    /**
     * <p>
     * Set the unit system to specify the distance.
     * </p>
     * <p>
     * Default Value: <code>Kilometers</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #distanceUnit} will
     * return {@link DistanceUnit#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #distanceUnitAsString}.
     * </p>
     * 
     * @return Set the unit system to specify the distance.</p>
     *         <p>
     *         Default Value: <code>Kilometers</code>
     * @see DistanceUnit
     */
    public final String distanceUnitAsString() {
        return distanceUnit;
    }

    /**
     * <p>
     * Set to include the geometry details in the result for each path between a pair of positions.
     * </p>
     * <p>
     * Default Value: <code>false</code>
     * </p>
     * <p>
     * Valid Values: <code>false</code> | <code>true</code>
     * </p>
     * 
     * @return Set to include the geometry details in the result for each path between a pair of positions.</p>
     *         <p>
     *         Default Value: <code>false</code>
     *         </p>
     *         <p>
     *         Valid Values: <code>false</code> | <code>true</code>
     */
    public final Boolean includeLegGeometry() {
        return includeLegGeometry;
    }

    /**
     * <p>
     * The optional <a href="https://docs.aws.amazon.com/location/latest/developerguide/using-apikeys.html">API key</a>
     * to authorize the request.
     * </p>
     * 
     * @return The optional <a href="https://docs.aws.amazon.com/location/latest/developerguide/using-apikeys.html">API
     *         key</a> to authorize the request.
     */
    public final String key() {
        return key;
    }

    /**
     * <p>
     * Specifies the distance to optimize for when calculating a route.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #optimizeFor} will
     * return {@link OptimizationMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #optimizeForAsString}.
     * </p>
     * 
     * @return Specifies the distance to optimize for when calculating a route.
     * @see OptimizationMode
     */
    public final OptimizationMode optimizeFor() {
        return OptimizationMode.fromValue(optimizeFor);
    }

    /**
     * <p>
     * Specifies the distance to optimize for when calculating a route.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #optimizeFor} will
     * return {@link OptimizationMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #optimizeForAsString}.
     * </p>
     * 
     * @return Specifies the distance to optimize for when calculating a route.
     * @see OptimizationMode
     */
    public final String optimizeForAsString() {
        return optimizeFor;
    }

    /**
     * <p>
     * Specifies the mode of transport when calculating a route. Used in estimating the speed of travel and road
     * compatibility. You can choose <code>Car</code>, <code>Truck</code>, <code>Walking</code>, <code>Bicycle</code> or
     * <code>Motorcycle</code> as options for the <code>TravelMode</code>.
     * </p>
     * <note>
     * <p>
     * <code>Bicycle</code> and <code>Motorcycle</code> are only valid when using Grab as a data provider, and only
     * within Southeast Asia.
     * </p>
     * <p>
     * <code>Truck</code> is not available for Grab.
     * </p>
     * <p>
     * For more details on the using Grab for routing, including areas of coverage, see <a
     * href="https://docs.aws.amazon.com/location/latest/developerguide/grab.html">GrabMaps</a> in the <i>Amazon
     * Location Service Developer Guide</i>.
     * </p>
     * </note>
     * <p>
     * The <code>TravelMode</code> you specify also determines how you specify route preferences:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If traveling by <code>Car</code> use the <code>CarModeOptions</code> parameter.
     * </p>
     * </li>
     * <li>
     * <p>
     * If traveling by <code>Truck</code> use the <code>TruckModeOptions</code> parameter.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Default Value: <code>Car</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #travelMode} will
     * return {@link TravelMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #travelModeAsString}.
     * </p>
     * 
     * @return Specifies the mode of transport when calculating a route. Used in estimating the speed of travel and road
     *         compatibility. You can choose <code>Car</code>, <code>Truck</code>, <code>Walking</code>,
     *         <code>Bicycle</code> or <code>Motorcycle</code> as options for the <code>TravelMode</code>.</p> <note>
     *         <p>
     *         <code>Bicycle</code> and <code>Motorcycle</code> are only valid when using Grab as a data provider, and
     *         only within Southeast Asia.
     *         </p>
     *         <p>
     *         <code>Truck</code> is not available for Grab.
     *         </p>
     *         <p>
     *         For more details on the using Grab for routing, including areas of coverage, see <a
     *         href="https://docs.aws.amazon.com/location/latest/developerguide/grab.html">GrabMaps</a> in the <i>Amazon
     *         Location Service Developer Guide</i>.
     *         </p>
     *         </note>
     *         <p>
     *         The <code>TravelMode</code> you specify also determines how you specify route preferences:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If traveling by <code>Car</code> use the <code>CarModeOptions</code> parameter.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If traveling by <code>Truck</code> use the <code>TruckModeOptions</code> parameter.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Default Value: <code>Car</code>
     * @see TravelMode
     */
    public final TravelMode travelMode() {
        return TravelMode.fromValue(travelMode);
    }

    /**
     * <p>
     * Specifies the mode of transport when calculating a route. Used in estimating the speed of travel and road
     * compatibility. You can choose <code>Car</code>, <code>Truck</code>, <code>Walking</code>, <code>Bicycle</code> or
     * <code>Motorcycle</code> as options for the <code>TravelMode</code>.
     * </p>
     * <note>
     * <p>
     * <code>Bicycle</code> and <code>Motorcycle</code> are only valid when using Grab as a data provider, and only
     * within Southeast Asia.
     * </p>
     * <p>
     * <code>Truck</code> is not available for Grab.
     * </p>
     * <p>
     * For more details on the using Grab for routing, including areas of coverage, see <a
     * href="https://docs.aws.amazon.com/location/latest/developerguide/grab.html">GrabMaps</a> in the <i>Amazon
     * Location Service Developer Guide</i>.
     * </p>
     * </note>
     * <p>
     * The <code>TravelMode</code> you specify also determines how you specify route preferences:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If traveling by <code>Car</code> use the <code>CarModeOptions</code> parameter.
     * </p>
     * </li>
     * <li>
     * <p>
     * If traveling by <code>Truck</code> use the <code>TruckModeOptions</code> parameter.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Default Value: <code>Car</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #travelMode} will
     * return {@link TravelMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #travelModeAsString}.
     * </p>
     * 
     * @return Specifies the mode of transport when calculating a route. Used in estimating the speed of travel and road
     *         compatibility. You can choose <code>Car</code>, <code>Truck</code>, <code>Walking</code>,
     *         <code>Bicycle</code> or <code>Motorcycle</code> as options for the <code>TravelMode</code>.</p> <note>
     *         <p>
     *         <code>Bicycle</code> and <code>Motorcycle</code> are only valid when using Grab as a data provider, and
     *         only within Southeast Asia.
     *         </p>
     *         <p>
     *         <code>Truck</code> is not available for Grab.
     *         </p>
     *         <p>
     *         For more details on the using Grab for routing, including areas of coverage, see <a
     *         href="https://docs.aws.amazon.com/location/latest/developerguide/grab.html">GrabMaps</a> in the <i>Amazon
     *         Location Service Developer Guide</i>.
     *         </p>
     *         </note>
     *         <p>
     *         The <code>TravelMode</code> you specify also determines how you specify route preferences:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If traveling by <code>Car</code> use the <code>CarModeOptions</code> parameter.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If traveling by <code>Truck</code> use the <code>TruckModeOptions</code> parameter.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Default Value: <code>Car</code>
     * @see TravelMode
     */
    public final String travelModeAsString() {
        return travelMode;
    }

    /**
     * <p>
     * Specifies route preferences when traveling by <code>Truck</code>, such as avoiding routes that use ferries or
     * tolls, and truck specifications to consider when choosing an optimal road.
     * </p>
     * <p>
     * Requirements: <code>TravelMode</code> must be specified as <code>Truck</code>.
     * </p>
     * 
     * @return Specifies route preferences when traveling by <code>Truck</code>, such as avoiding routes that use
     *         ferries or tolls, and truck specifications to consider when choosing an optimal road.</p>
     *         <p>
     *         Requirements: <code>TravelMode</code> must be specified as <code>Truck</code>.
     */
    public final CalculateRouteTruckModeOptions truckModeOptions() {
        return truckModeOptions;
    }

    /**
     * For responses, this returns true if the service returned a value for the WaypointPositions property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasWaypointPositions() {
        return waypointPositions != null && !(waypointPositions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies an ordered list of up to 23 intermediate positions to include along a route between the departure
     * position and destination position.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For example, from the <code>DeparturePosition</code> <code>[-123.115, 49.285]</code>, the route follows the order
     * that the waypoint positions are given <code>[[-122.757, 49.0021],[-122.349, 47.620]]</code>
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * If you specify a waypoint position that's not located on a road, Amazon Location <a
     * href="https://docs.aws.amazon.com/location/latest/developerguide/snap-to-nearby-road.html">moves the position to
     * the nearest road</a>.
     * </p>
     * <p>
     * Specifying more than 23 waypoints returns a <code>400 ValidationException</code> error.
     * </p>
     * <p>
     * If Esri is the provider for your route calculator, specifying a route that is longer than 400 km returns a
     * <code>400 RoutesValidationException</code> error.
     * </p>
     * </note>
     * <p>
     * Valid Values: <code>[-180 to 180,-90 to 90]</code>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasWaypointPositions} method.
     * </p>
     * 
     * @return Specifies an ordered list of up to 23 intermediate positions to include along a route between the
     *         departure position and destination position. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         For example, from the <code>DeparturePosition</code> <code>[-123.115, 49.285]</code>, the route follows
     *         the order that the waypoint positions are given <code>[[-122.757, 49.0021],[-122.349, 47.620]]</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         If you specify a waypoint position that's not located on a road, Amazon Location <a
     *         href="https://docs.aws.amazon.com/location/latest/developerguide/snap-to-nearby-road.html">moves the
     *         position to the nearest road</a>.
     *         </p>
     *         <p>
     *         Specifying more than 23 waypoints returns a <code>400 ValidationException</code> error.
     *         </p>
     *         <p>
     *         If Esri is the provider for your route calculator, specifying a route that is longer than 400 km returns
     *         a <code>400 RoutesValidationException</code> error.
     *         </p>
     *         </note>
     *         <p>
     *         Valid Values: <code>[-180 to 180,-90 to 90]</code>
     */
    public final List<List<Double>> waypointPositions() {
        return waypointPositions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(arrivalTime());
        hashCode = 31 * hashCode + Objects.hashCode(calculatorName());
        hashCode = 31 * hashCode + Objects.hashCode(carModeOptions());
        hashCode = 31 * hashCode + Objects.hashCode(departNow());
        hashCode = 31 * hashCode + Objects.hashCode(hasDeparturePosition() ? departurePosition() : null);
        hashCode = 31 * hashCode + Objects.hashCode(departureTime());
        hashCode = 31 * hashCode + Objects.hashCode(hasDestinationPosition() ? destinationPosition() : null);
        hashCode = 31 * hashCode + Objects.hashCode(distanceUnitAsString());
        hashCode = 31 * hashCode + Objects.hashCode(includeLegGeometry());
        hashCode = 31 * hashCode + Objects.hashCode(key());
        hashCode = 31 * hashCode + Objects.hashCode(optimizeForAsString());
        hashCode = 31 * hashCode + Objects.hashCode(travelModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(truckModeOptions());
        hashCode = 31 * hashCode + Objects.hashCode(hasWaypointPositions() ? waypointPositions() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CalculateRouteRequest)) {
            return false;
        }
        CalculateRouteRequest other = (CalculateRouteRequest) obj;
        return Objects.equals(arrivalTime(), other.arrivalTime()) && Objects.equals(calculatorName(), other.calculatorName())
                && Objects.equals(carModeOptions(), other.carModeOptions()) && Objects.equals(departNow(), other.departNow())
                && hasDeparturePosition() == other.hasDeparturePosition()
                && Objects.equals(departurePosition(), other.departurePosition())
                && Objects.equals(departureTime(), other.departureTime())
                && hasDestinationPosition() == other.hasDestinationPosition()
                && Objects.equals(destinationPosition(), other.destinationPosition())
                && Objects.equals(distanceUnitAsString(), other.distanceUnitAsString())
                && Objects.equals(includeLegGeometry(), other.includeLegGeometry()) && Objects.equals(key(), other.key())
                && Objects.equals(optimizeForAsString(), other.optimizeForAsString())
                && Objects.equals(travelModeAsString(), other.travelModeAsString())
                && Objects.equals(truckModeOptions(), other.truckModeOptions())
                && hasWaypointPositions() == other.hasWaypointPositions()
                && Objects.equals(waypointPositions(), other.waypointPositions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CalculateRouteRequest").add("ArrivalTime", arrivalTime())
                .add("CalculatorName", calculatorName()).add("CarModeOptions", carModeOptions()).add("DepartNow", departNow())
                .add("DeparturePosition", departurePosition() == null ? null : "*** Sensitive Data Redacted ***")
                .add("DepartureTime", departureTime())
                .add("DestinationPosition", destinationPosition() == null ? null : "*** Sensitive Data Redacted ***")
                .add("DistanceUnit", distanceUnitAsString()).add("IncludeLegGeometry", includeLegGeometry())
                .add("Key", key() == null ? null : "*** Sensitive Data Redacted ***").add("OptimizeFor", optimizeForAsString())
                .add("TravelMode", travelModeAsString()).add("TruckModeOptions", truckModeOptions())
                .add("WaypointPositions", waypointPositions() == null ? null : "*** Sensitive Data Redacted ***").build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ArrivalTime":
            return Optional.ofNullable(clazz.cast(arrivalTime()));
        case "CalculatorName":
            return Optional.ofNullable(clazz.cast(calculatorName()));
        case "CarModeOptions":
            return Optional.ofNullable(clazz.cast(carModeOptions()));
        case "DepartNow":
            return Optional.ofNullable(clazz.cast(departNow()));
        case "DeparturePosition":
            return Optional.ofNullable(clazz.cast(departurePosition()));
        case "DepartureTime":
            return Optional.ofNullable(clazz.cast(departureTime()));
        case "DestinationPosition":
            return Optional.ofNullable(clazz.cast(destinationPosition()));
        case "DistanceUnit":
            return Optional.ofNullable(clazz.cast(distanceUnitAsString()));
        case "IncludeLegGeometry":
            return Optional.ofNullable(clazz.cast(includeLegGeometry()));
        case "Key":
            return Optional.ofNullable(clazz.cast(key()));
        case "OptimizeFor":
            return Optional.ofNullable(clazz.cast(optimizeForAsString()));
        case "TravelMode":
            return Optional.ofNullable(clazz.cast(travelModeAsString()));
        case "TruckModeOptions":
            return Optional.ofNullable(clazz.cast(truckModeOptions()));
        case "WaypointPositions":
            return Optional.ofNullable(clazz.cast(waypointPositions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CalculateRouteRequest, T> g) {
        return obj -> g.apply((CalculateRouteRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends LocationRequest.Builder, SdkPojo, CopyableBuilder<Builder, CalculateRouteRequest> {
        /**
         * <p>
         * Specifies the desired time of arrival. Uses the given time to calculate the route. Otherwise, the best time
         * of day to travel with the best traffic conditions is used to calculate the route.
         * </p>
         * <note>
         * <p>
         * ArrivalTime is not supported Esri.
         * </p>
         * </note>
         * 
         * @param arrivalTime
         *        Specifies the desired time of arrival. Uses the given time to calculate the route. Otherwise, the best
         *        time of day to travel with the best traffic conditions is used to calculate the route.</p> <note>
         *        <p>
         *        ArrivalTime is not supported Esri.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arrivalTime(Instant arrivalTime);

        /**
         * <p>
         * The name of the route calculator resource that you want to use to calculate the route.
         * </p>
         * 
         * @param calculatorName
         *        The name of the route calculator resource that you want to use to calculate the route.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder calculatorName(String calculatorName);

        /**
         * <p>
         * Specifies route preferences when traveling by <code>Car</code>, such as avoiding routes that use ferries or
         * tolls.
         * </p>
         * <p>
         * Requirements: <code>TravelMode</code> must be specified as <code>Car</code>.
         * </p>
         * 
         * @param carModeOptions
         *        Specifies route preferences when traveling by <code>Car</code>, such as avoiding routes that use
         *        ferries or tolls.</p>
         *        <p>
         *        Requirements: <code>TravelMode</code> must be specified as <code>Car</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder carModeOptions(CalculateRouteCarModeOptions carModeOptions);

        /**
         * <p>
         * Specifies route preferences when traveling by <code>Car</code>, such as avoiding routes that use ferries or
         * tolls.
         * </p>
         * <p>
         * Requirements: <code>TravelMode</code> must be specified as <code>Car</code>.
         * </p>
         * This is a convenience method that creates an instance of the {@link CalculateRouteCarModeOptions.Builder}
         * avoiding the need to create one manually via {@link CalculateRouteCarModeOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CalculateRouteCarModeOptions.Builder#build()} is called
         * immediately and its result is passed to {@link #carModeOptions(CalculateRouteCarModeOptions)}.
         * 
         * @param carModeOptions
         *        a consumer that will call methods on {@link CalculateRouteCarModeOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #carModeOptions(CalculateRouteCarModeOptions)
         */
        default Builder carModeOptions(Consumer<CalculateRouteCarModeOptions.Builder> carModeOptions) {
            return carModeOptions(CalculateRouteCarModeOptions.builder().applyMutation(carModeOptions).build());
        }

        /**
         * <p>
         * Sets the time of departure as the current time. Uses the current time to calculate a route. Otherwise, the
         * best time of day to travel with the best traffic conditions is used to calculate the route.
         * </p>
         * <p>
         * Default Value: <code>false</code>
         * </p>
         * <p>
         * Valid Values: <code>false</code> | <code>true</code>
         * </p>
         * 
         * @param departNow
         *        Sets the time of departure as the current time. Uses the current time to calculate a route. Otherwise,
         *        the best time of day to travel with the best traffic conditions is used to calculate the route.</p>
         *        <p>
         *        Default Value: <code>false</code>
         *        </p>
         *        <p>
         *        Valid Values: <code>false</code> | <code>true</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder departNow(Boolean departNow);

        /**
         * <p>
         * The start position for the route. Defined in <a
         * href="https://earth-info.nga.mil/index.php?dir=wgs84&amp;action=wgs84">World Geodetic System (WGS 84)</a>
         * format: <code>[longitude, latitude]</code>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For example, <code>[-123.115, 49.285]</code>
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * If you specify a departure that's not located on a road, Amazon Location <a
         * href="https://docs.aws.amazon.com/location/latest/developerguide/snap-to-nearby-road.html">moves the position
         * to the nearest road</a>. If Esri is the provider for your route calculator, specifying a route that is longer
         * than 400 km returns a <code>400 RoutesValidationException</code> error.
         * </p>
         * </note>
         * <p>
         * Valid Values: <code>[-180 to 180,-90 to 90]</code>
         * </p>
         * 
         * @param departurePosition
         *        The start position for the route. Defined in <a
         *        href="https://earth-info.nga.mil/index.php?dir=wgs84&amp;action=wgs84">World Geodetic System (WGS
         *        84)</a> format: <code>[longitude, latitude]</code>.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For example, <code>[-123.115, 49.285]</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        If you specify a departure that's not located on a road, Amazon Location <a
         *        href="https://docs.aws.amazon.com/location/latest/developerguide/snap-to-nearby-road.html">moves the
         *        position to the nearest road</a>. If Esri is the provider for your route calculator, specifying a
         *        route that is longer than 400 km returns a <code>400 RoutesValidationException</code> error.
         *        </p>
         *        </note>
         *        <p>
         *        Valid Values: <code>[-180 to 180,-90 to 90]</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder departurePosition(Collection<Double> departurePosition);

        /**
         * <p>
         * The start position for the route. Defined in <a
         * href="https://earth-info.nga.mil/index.php?dir=wgs84&amp;action=wgs84">World Geodetic System (WGS 84)</a>
         * format: <code>[longitude, latitude]</code>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For example, <code>[-123.115, 49.285]</code>
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * If you specify a departure that's not located on a road, Amazon Location <a
         * href="https://docs.aws.amazon.com/location/latest/developerguide/snap-to-nearby-road.html">moves the position
         * to the nearest road</a>. If Esri is the provider for your route calculator, specifying a route that is longer
         * than 400 km returns a <code>400 RoutesValidationException</code> error.
         * </p>
         * </note>
         * <p>
         * Valid Values: <code>[-180 to 180,-90 to 90]</code>
         * </p>
         * 
         * @param departurePosition
         *        The start position for the route. Defined in <a
         *        href="https://earth-info.nga.mil/index.php?dir=wgs84&amp;action=wgs84">World Geodetic System (WGS
         *        84)</a> format: <code>[longitude, latitude]</code>.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For example, <code>[-123.115, 49.285]</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        If you specify a departure that's not located on a road, Amazon Location <a
         *        href="https://docs.aws.amazon.com/location/latest/developerguide/snap-to-nearby-road.html">moves the
         *        position to the nearest road</a>. If Esri is the provider for your route calculator, specifying a
         *        route that is longer than 400 km returns a <code>400 RoutesValidationException</code> error.
         *        </p>
         *        </note>
         *        <p>
         *        Valid Values: <code>[-180 to 180,-90 to 90]</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder departurePosition(Double... departurePosition);

        /**
         * <p>
         * Specifies the desired time of departure. Uses the given time to calculate the route. Otherwise, the best time
         * of day to travel with the best traffic conditions is used to calculate the route.
         * </p>
         * <ul>
         * <li>
         * <p>
         * In <a href="https://www.iso.org/iso-8601-date-and-time-format.html">ISO 8601</a> format:
         * <code>YYYY-MM-DDThh:mm:ss.sssZ</code>. For example, <code>2020–07-2T12:15:20.000Z+01:00</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param departureTime
         *        Specifies the desired time of departure. Uses the given time to calculate the route. Otherwise, the
         *        best time of day to travel with the best traffic conditions is used to calculate the route.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        In <a href="https://www.iso.org/iso-8601-date-and-time-format.html">ISO 8601</a> format:
         *        <code>YYYY-MM-DDThh:mm:ss.sssZ</code>. For example, <code>2020–07-2T12:15:20.000Z+01:00</code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder departureTime(Instant departureTime);

        /**
         * <p>
         * The finish position for the route. Defined in <a
         * href="https://earth-info.nga.mil/index.php?dir=wgs84&amp;action=wgs84">World Geodetic System (WGS 84)</a>
         * format: <code>[longitude, latitude]</code>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For example, <code>[-122.339, 47.615]</code>
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * If you specify a destination that's not located on a road, Amazon Location <a
         * href="https://docs.aws.amazon.com/location/latest/developerguide/snap-to-nearby-road.html">moves the position
         * to the nearest road</a>.
         * </p>
         * </note>
         * <p>
         * Valid Values: <code>[-180 to 180,-90 to 90]</code>
         * </p>
         * 
         * @param destinationPosition
         *        The finish position for the route. Defined in <a
         *        href="https://earth-info.nga.mil/index.php?dir=wgs84&amp;action=wgs84">World Geodetic System (WGS
         *        84)</a> format: <code>[longitude, latitude]</code>.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For example, <code>[-122.339, 47.615]</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        If you specify a destination that's not located on a road, Amazon Location <a
         *        href="https://docs.aws.amazon.com/location/latest/developerguide/snap-to-nearby-road.html">moves the
         *        position to the nearest road</a>.
         *        </p>
         *        </note>
         *        <p>
         *        Valid Values: <code>[-180 to 180,-90 to 90]</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinationPosition(Collection<Double> destinationPosition);

        /**
         * <p>
         * The finish position for the route. Defined in <a
         * href="https://earth-info.nga.mil/index.php?dir=wgs84&amp;action=wgs84">World Geodetic System (WGS 84)</a>
         * format: <code>[longitude, latitude]</code>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For example, <code>[-122.339, 47.615]</code>
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * If you specify a destination that's not located on a road, Amazon Location <a
         * href="https://docs.aws.amazon.com/location/latest/developerguide/snap-to-nearby-road.html">moves the position
         * to the nearest road</a>.
         * </p>
         * </note>
         * <p>
         * Valid Values: <code>[-180 to 180,-90 to 90]</code>
         * </p>
         * 
         * @param destinationPosition
         *        The finish position for the route. Defined in <a
         *        href="https://earth-info.nga.mil/index.php?dir=wgs84&amp;action=wgs84">World Geodetic System (WGS
         *        84)</a> format: <code>[longitude, latitude]</code>.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For example, <code>[-122.339, 47.615]</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        If you specify a destination that's not located on a road, Amazon Location <a
         *        href="https://docs.aws.amazon.com/location/latest/developerguide/snap-to-nearby-road.html">moves the
         *        position to the nearest road</a>.
         *        </p>
         *        </note>
         *        <p>
         *        Valid Values: <code>[-180 to 180,-90 to 90]</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinationPosition(Double... destinationPosition);

        /**
         * <p>
         * Set the unit system to specify the distance.
         * </p>
         * <p>
         * Default Value: <code>Kilometers</code>
         * </p>
         * 
         * @param distanceUnit
         *        Set the unit system to specify the distance.</p>
         *        <p>
         *        Default Value: <code>Kilometers</code>
         * @see DistanceUnit
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DistanceUnit
         */
        Builder distanceUnit(String distanceUnit);

        /**
         * <p>
         * Set the unit system to specify the distance.
         * </p>
         * <p>
         * Default Value: <code>Kilometers</code>
         * </p>
         * 
         * @param distanceUnit
         *        Set the unit system to specify the distance.</p>
         *        <p>
         *        Default Value: <code>Kilometers</code>
         * @see DistanceUnit
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DistanceUnit
         */
        Builder distanceUnit(DistanceUnit distanceUnit);

        /**
         * <p>
         * Set to include the geometry details in the result for each path between a pair of positions.
         * </p>
         * <p>
         * Default Value: <code>false</code>
         * </p>
         * <p>
         * Valid Values: <code>false</code> | <code>true</code>
         * </p>
         * 
         * @param includeLegGeometry
         *        Set to include the geometry details in the result for each path between a pair of positions.</p>
         *        <p>
         *        Default Value: <code>false</code>
         *        </p>
         *        <p>
         *        Valid Values: <code>false</code> | <code>true</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder includeLegGeometry(Boolean includeLegGeometry);

        /**
         * <p>
         * The optional <a href="https://docs.aws.amazon.com/location/latest/developerguide/using-apikeys.html">API
         * key</a> to authorize the request.
         * </p>
         * 
         * @param key
         *        The optional <a
         *        href="https://docs.aws.amazon.com/location/latest/developerguide/using-apikeys.html">API key</a> to
         *        authorize the request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder key(String key);

        /**
         * <p>
         * Specifies the distance to optimize for when calculating a route.
         * </p>
         * 
         * @param optimizeFor
         *        Specifies the distance to optimize for when calculating a route.
         * @see OptimizationMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OptimizationMode
         */
        Builder optimizeFor(String optimizeFor);

        /**
         * <p>
         * Specifies the distance to optimize for when calculating a route.
         * </p>
         * 
         * @param optimizeFor
         *        Specifies the distance to optimize for when calculating a route.
         * @see OptimizationMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OptimizationMode
         */
        Builder optimizeFor(OptimizationMode optimizeFor);

        /**
         * <p>
         * Specifies the mode of transport when calculating a route. Used in estimating the speed of travel and road
         * compatibility. You can choose <code>Car</code>, <code>Truck</code>, <code>Walking</code>,
         * <code>Bicycle</code> or <code>Motorcycle</code> as options for the <code>TravelMode</code>.
         * </p>
         * <note>
         * <p>
         * <code>Bicycle</code> and <code>Motorcycle</code> are only valid when using Grab as a data provider, and only
         * within Southeast Asia.
         * </p>
         * <p>
         * <code>Truck</code> is not available for Grab.
         * </p>
         * <p>
         * For more details on the using Grab for routing, including areas of coverage, see <a
         * href="https://docs.aws.amazon.com/location/latest/developerguide/grab.html">GrabMaps</a> in the <i>Amazon
         * Location Service Developer Guide</i>.
         * </p>
         * </note>
         * <p>
         * The <code>TravelMode</code> you specify also determines how you specify route preferences:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If traveling by <code>Car</code> use the <code>CarModeOptions</code> parameter.
         * </p>
         * </li>
         * <li>
         * <p>
         * If traveling by <code>Truck</code> use the <code>TruckModeOptions</code> parameter.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default Value: <code>Car</code>
         * </p>
         * 
         * @param travelMode
         *        Specifies the mode of transport when calculating a route. Used in estimating the speed of travel and
         *        road compatibility. You can choose <code>Car</code>, <code>Truck</code>, <code>Walking</code>,
         *        <code>Bicycle</code> or <code>Motorcycle</code> as options for the <code>TravelMode</code>.</p> <note>
         *        <p>
         *        <code>Bicycle</code> and <code>Motorcycle</code> are only valid when using Grab as a data provider,
         *        and only within Southeast Asia.
         *        </p>
         *        <p>
         *        <code>Truck</code> is not available for Grab.
         *        </p>
         *        <p>
         *        For more details on the using Grab for routing, including areas of coverage, see <a
         *        href="https://docs.aws.amazon.com/location/latest/developerguide/grab.html">GrabMaps</a> in the
         *        <i>Amazon Location Service Developer Guide</i>.
         *        </p>
         *        </note>
         *        <p>
         *        The <code>TravelMode</code> you specify also determines how you specify route preferences:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If traveling by <code>Car</code> use the <code>CarModeOptions</code> parameter.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If traveling by <code>Truck</code> use the <code>TruckModeOptions</code> parameter.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default Value: <code>Car</code>
         * @see TravelMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TravelMode
         */
        Builder travelMode(String travelMode);

        /**
         * <p>
         * Specifies the mode of transport when calculating a route. Used in estimating the speed of travel and road
         * compatibility. You can choose <code>Car</code>, <code>Truck</code>, <code>Walking</code>,
         * <code>Bicycle</code> or <code>Motorcycle</code> as options for the <code>TravelMode</code>.
         * </p>
         * <note>
         * <p>
         * <code>Bicycle</code> and <code>Motorcycle</code> are only valid when using Grab as a data provider, and only
         * within Southeast Asia.
         * </p>
         * <p>
         * <code>Truck</code> is not available for Grab.
         * </p>
         * <p>
         * For more details on the using Grab for routing, including areas of coverage, see <a
         * href="https://docs.aws.amazon.com/location/latest/developerguide/grab.html">GrabMaps</a> in the <i>Amazon
         * Location Service Developer Guide</i>.
         * </p>
         * </note>
         * <p>
         * The <code>TravelMode</code> you specify also determines how you specify route preferences:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If traveling by <code>Car</code> use the <code>CarModeOptions</code> parameter.
         * </p>
         * </li>
         * <li>
         * <p>
         * If traveling by <code>Truck</code> use the <code>TruckModeOptions</code> parameter.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default Value: <code>Car</code>
         * </p>
         * 
         * @param travelMode
         *        Specifies the mode of transport when calculating a route. Used in estimating the speed of travel and
         *        road compatibility. You can choose <code>Car</code>, <code>Truck</code>, <code>Walking</code>,
         *        <code>Bicycle</code> or <code>Motorcycle</code> as options for the <code>TravelMode</code>.</p> <note>
         *        <p>
         *        <code>Bicycle</code> and <code>Motorcycle</code> are only valid when using Grab as a data provider,
         *        and only within Southeast Asia.
         *        </p>
         *        <p>
         *        <code>Truck</code> is not available for Grab.
         *        </p>
         *        <p>
         *        For more details on the using Grab for routing, including areas of coverage, see <a
         *        href="https://docs.aws.amazon.com/location/latest/developerguide/grab.html">GrabMaps</a> in the
         *        <i>Amazon Location Service Developer Guide</i>.
         *        </p>
         *        </note>
         *        <p>
         *        The <code>TravelMode</code> you specify also determines how you specify route preferences:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If traveling by <code>Car</code> use the <code>CarModeOptions</code> parameter.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If traveling by <code>Truck</code> use the <code>TruckModeOptions</code> parameter.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default Value: <code>Car</code>
         * @see TravelMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TravelMode
         */
        Builder travelMode(TravelMode travelMode);

        /**
         * <p>
         * Specifies route preferences when traveling by <code>Truck</code>, such as avoiding routes that use ferries or
         * tolls, and truck specifications to consider when choosing an optimal road.
         * </p>
         * <p>
         * Requirements: <code>TravelMode</code> must be specified as <code>Truck</code>.
         * </p>
         * 
         * @param truckModeOptions
         *        Specifies route preferences when traveling by <code>Truck</code>, such as avoiding routes that use
         *        ferries or tolls, and truck specifications to consider when choosing an optimal road.</p>
         *        <p>
         *        Requirements: <code>TravelMode</code> must be specified as <code>Truck</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder truckModeOptions(CalculateRouteTruckModeOptions truckModeOptions);

        /**
         * <p>
         * Specifies route preferences when traveling by <code>Truck</code>, such as avoiding routes that use ferries or
         * tolls, and truck specifications to consider when choosing an optimal road.
         * </p>
         * <p>
         * Requirements: <code>TravelMode</code> must be specified as <code>Truck</code>.
         * </p>
         * This is a convenience method that creates an instance of the {@link CalculateRouteTruckModeOptions.Builder}
         * avoiding the need to create one manually via {@link CalculateRouteTruckModeOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CalculateRouteTruckModeOptions.Builder#build()} is called
         * immediately and its result is passed to {@link #truckModeOptions(CalculateRouteTruckModeOptions)}.
         * 
         * @param truckModeOptions
         *        a consumer that will call methods on {@link CalculateRouteTruckModeOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #truckModeOptions(CalculateRouteTruckModeOptions)
         */
        default Builder truckModeOptions(Consumer<CalculateRouteTruckModeOptions.Builder> truckModeOptions) {
            return truckModeOptions(CalculateRouteTruckModeOptions.builder().applyMutation(truckModeOptions).build());
        }

        /**
         * <p>
         * Specifies an ordered list of up to 23 intermediate positions to include along a route between the departure
         * position and destination position.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For example, from the <code>DeparturePosition</code> <code>[-123.115, 49.285]</code>, the route follows the
         * order that the waypoint positions are given <code>[[-122.757, 49.0021],[-122.349, 47.620]]</code>
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * If you specify a waypoint position that's not located on a road, Amazon Location <a
         * href="https://docs.aws.amazon.com/location/latest/developerguide/snap-to-nearby-road.html">moves the position
         * to the nearest road</a>.
         * </p>
         * <p>
         * Specifying more than 23 waypoints returns a <code>400 ValidationException</code> error.
         * </p>
         * <p>
         * If Esri is the provider for your route calculator, specifying a route that is longer than 400 km returns a
         * <code>400 RoutesValidationException</code> error.
         * </p>
         * </note>
         * <p>
         * Valid Values: <code>[-180 to 180,-90 to 90]</code>
         * </p>
         * 
         * @param waypointPositions
         *        Specifies an ordered list of up to 23 intermediate positions to include along a route between the
         *        departure position and destination position. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For example, from the <code>DeparturePosition</code> <code>[-123.115, 49.285]</code>, the route
         *        follows the order that the waypoint positions are given
         *        <code>[[-122.757, 49.0021],[-122.349, 47.620]]</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        If you specify a waypoint position that's not located on a road, Amazon Location <a
         *        href="https://docs.aws.amazon.com/location/latest/developerguide/snap-to-nearby-road.html">moves the
         *        position to the nearest road</a>.
         *        </p>
         *        <p>
         *        Specifying more than 23 waypoints returns a <code>400 ValidationException</code> error.
         *        </p>
         *        <p>
         *        If Esri is the provider for your route calculator, specifying a route that is longer than 400 km
         *        returns a <code>400 RoutesValidationException</code> error.
         *        </p>
         *        </note>
         *        <p>
         *        Valid Values: <code>[-180 to 180,-90 to 90]</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder waypointPositions(Collection<? extends Collection<Double>> waypointPositions);

        /**
         * <p>
         * Specifies an ordered list of up to 23 intermediate positions to include along a route between the departure
         * position and destination position.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For example, from the <code>DeparturePosition</code> <code>[-123.115, 49.285]</code>, the route follows the
         * order that the waypoint positions are given <code>[[-122.757, 49.0021],[-122.349, 47.620]]</code>
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * If you specify a waypoint position that's not located on a road, Amazon Location <a
         * href="https://docs.aws.amazon.com/location/latest/developerguide/snap-to-nearby-road.html">moves the position
         * to the nearest road</a>.
         * </p>
         * <p>
         * Specifying more than 23 waypoints returns a <code>400 ValidationException</code> error.
         * </p>
         * <p>
         * If Esri is the provider for your route calculator, specifying a route that is longer than 400 km returns a
         * <code>400 RoutesValidationException</code> error.
         * </p>
         * </note>
         * <p>
         * Valid Values: <code>[-180 to 180,-90 to 90]</code>
         * </p>
         * 
         * @param waypointPositions
         *        Specifies an ordered list of up to 23 intermediate positions to include along a route between the
         *        departure position and destination position. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For example, from the <code>DeparturePosition</code> <code>[-123.115, 49.285]</code>, the route
         *        follows the order that the waypoint positions are given
         *        <code>[[-122.757, 49.0021],[-122.349, 47.620]]</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        If you specify a waypoint position that's not located on a road, Amazon Location <a
         *        href="https://docs.aws.amazon.com/location/latest/developerguide/snap-to-nearby-road.html">moves the
         *        position to the nearest road</a>.
         *        </p>
         *        <p>
         *        Specifying more than 23 waypoints returns a <code>400 ValidationException</code> error.
         *        </p>
         *        <p>
         *        If Esri is the provider for your route calculator, specifying a route that is longer than 400 km
         *        returns a <code>400 RoutesValidationException</code> error.
         *        </p>
         *        </note>
         *        <p>
         *        Valid Values: <code>[-180 to 180,-90 to 90]</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder waypointPositions(Collection<Double>... waypointPositions);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends LocationRequest.BuilderImpl implements Builder {
        private Instant arrivalTime;

        private String calculatorName;

        private CalculateRouteCarModeOptions carModeOptions;

        private Boolean departNow;

        private List<Double> departurePosition = DefaultSdkAutoConstructList.getInstance();

        private Instant departureTime;

        private List<Double> destinationPosition = DefaultSdkAutoConstructList.getInstance();

        private String distanceUnit;

        private Boolean includeLegGeometry;

        private String key;

        private String optimizeFor;

        private String travelMode;

        private CalculateRouteTruckModeOptions truckModeOptions;

        private List<List<Double>> waypointPositions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CalculateRouteRequest model) {
            super(model);
            arrivalTime(model.arrivalTime);
            calculatorName(model.calculatorName);
            carModeOptions(model.carModeOptions);
            departNow(model.departNow);
            departurePosition(model.departurePosition);
            departureTime(model.departureTime);
            destinationPosition(model.destinationPosition);
            distanceUnit(model.distanceUnit);
            includeLegGeometry(model.includeLegGeometry);
            key(model.key);
            optimizeFor(model.optimizeFor);
            travelMode(model.travelMode);
            truckModeOptions(model.truckModeOptions);
            waypointPositions(model.waypointPositions);
        }

        public final Instant getArrivalTime() {
            return arrivalTime;
        }

        public final void setArrivalTime(Instant arrivalTime) {
            this.arrivalTime = arrivalTime;
        }

        @Override
        public final Builder arrivalTime(Instant arrivalTime) {
            this.arrivalTime = arrivalTime;
            return this;
        }

        public final String getCalculatorName() {
            return calculatorName;
        }

        public final void setCalculatorName(String calculatorName) {
            this.calculatorName = calculatorName;
        }

        @Override
        public final Builder calculatorName(String calculatorName) {
            this.calculatorName = calculatorName;
            return this;
        }

        public final CalculateRouteCarModeOptions.Builder getCarModeOptions() {
            return carModeOptions != null ? carModeOptions.toBuilder() : null;
        }

        public final void setCarModeOptions(CalculateRouteCarModeOptions.BuilderImpl carModeOptions) {
            this.carModeOptions = carModeOptions != null ? carModeOptions.build() : null;
        }

        @Override
        public final Builder carModeOptions(CalculateRouteCarModeOptions carModeOptions) {
            this.carModeOptions = carModeOptions;
            return this;
        }

        public final Boolean getDepartNow() {
            return departNow;
        }

        public final void setDepartNow(Boolean departNow) {
            this.departNow = departNow;
        }

        @Override
        public final Builder departNow(Boolean departNow) {
            this.departNow = departNow;
            return this;
        }

        public final Collection<Double> getDeparturePosition() {
            if (departurePosition instanceof SdkAutoConstructList) {
                return null;
            }
            return departurePosition;
        }

        public final void setDeparturePosition(Collection<Double> departurePosition) {
            this.departurePosition = PositionCopier.copy(departurePosition);
        }

        @Override
        public final Builder departurePosition(Collection<Double> departurePosition) {
            this.departurePosition = PositionCopier.copy(departurePosition);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder departurePosition(Double... departurePosition) {
            departurePosition(Arrays.asList(departurePosition));
            return this;
        }

        public final Instant getDepartureTime() {
            return departureTime;
        }

        public final void setDepartureTime(Instant departureTime) {
            this.departureTime = departureTime;
        }

        @Override
        public final Builder departureTime(Instant departureTime) {
            this.departureTime = departureTime;
            return this;
        }

        public final Collection<Double> getDestinationPosition() {
            if (destinationPosition instanceof SdkAutoConstructList) {
                return null;
            }
            return destinationPosition;
        }

        public final void setDestinationPosition(Collection<Double> destinationPosition) {
            this.destinationPosition = PositionCopier.copy(destinationPosition);
        }

        @Override
        public final Builder destinationPosition(Collection<Double> destinationPosition) {
            this.destinationPosition = PositionCopier.copy(destinationPosition);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder destinationPosition(Double... destinationPosition) {
            destinationPosition(Arrays.asList(destinationPosition));
            return this;
        }

        public final String getDistanceUnit() {
            return distanceUnit;
        }

        public final void setDistanceUnit(String distanceUnit) {
            this.distanceUnit = distanceUnit;
        }

        @Override
        public final Builder distanceUnit(String distanceUnit) {
            this.distanceUnit = distanceUnit;
            return this;
        }

        @Override
        public final Builder distanceUnit(DistanceUnit distanceUnit) {
            this.distanceUnit(distanceUnit == null ? null : distanceUnit.toString());
            return this;
        }

        public final Boolean getIncludeLegGeometry() {
            return includeLegGeometry;
        }

        public final void setIncludeLegGeometry(Boolean includeLegGeometry) {
            this.includeLegGeometry = includeLegGeometry;
        }

        @Override
        public final Builder includeLegGeometry(Boolean includeLegGeometry) {
            this.includeLegGeometry = includeLegGeometry;
            return this;
        }

        public final String getKey() {
            return key;
        }

        public final void setKey(String key) {
            this.key = key;
        }

        @Override
        public final Builder key(String key) {
            this.key = key;
            return this;
        }

        public final String getOptimizeFor() {
            return optimizeFor;
        }

        public final void setOptimizeFor(String optimizeFor) {
            this.optimizeFor = optimizeFor;
        }

        @Override
        public final Builder optimizeFor(String optimizeFor) {
            this.optimizeFor = optimizeFor;
            return this;
        }

        @Override
        public final Builder optimizeFor(OptimizationMode optimizeFor) {
            this.optimizeFor(optimizeFor == null ? null : optimizeFor.toString());
            return this;
        }

        public final String getTravelMode() {
            return travelMode;
        }

        public final void setTravelMode(String travelMode) {
            this.travelMode = travelMode;
        }

        @Override
        public final Builder travelMode(String travelMode) {
            this.travelMode = travelMode;
            return this;
        }

        @Override
        public final Builder travelMode(TravelMode travelMode) {
            this.travelMode(travelMode == null ? null : travelMode.toString());
            return this;
        }

        public final CalculateRouteTruckModeOptions.Builder getTruckModeOptions() {
            return truckModeOptions != null ? truckModeOptions.toBuilder() : null;
        }

        public final void setTruckModeOptions(CalculateRouteTruckModeOptions.BuilderImpl truckModeOptions) {
            this.truckModeOptions = truckModeOptions != null ? truckModeOptions.build() : null;
        }

        @Override
        public final Builder truckModeOptions(CalculateRouteTruckModeOptions truckModeOptions) {
            this.truckModeOptions = truckModeOptions;
            return this;
        }

        public final Collection<? extends Collection<Double>> getWaypointPositions() {
            if (waypointPositions instanceof SdkAutoConstructList) {
                return null;
            }
            return waypointPositions;
        }

        public final void setWaypointPositions(Collection<? extends Collection<Double>> waypointPositions) {
            this.waypointPositions = CalculateRouteRequestWaypointPositionsListCopier.copy(waypointPositions);
        }

        @Override
        public final Builder waypointPositions(Collection<? extends Collection<Double>> waypointPositions) {
            this.waypointPositions = CalculateRouteRequestWaypointPositionsListCopier.copy(waypointPositions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder waypointPositions(Collection<Double>... waypointPositions) {
            waypointPositions(Arrays.asList(waypointPositions));
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CalculateRouteRequest build() {
            return new CalculateRouteRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
