/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.location.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CalculateRouteMatrixRequest extends LocationRequest implements
        ToCopyableBuilder<CalculateRouteMatrixRequest.Builder, CalculateRouteMatrixRequest> {
    private static final SdkField<String> CALCULATOR_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CalculatorName").getter(getter(CalculateRouteMatrixRequest::calculatorName))
            .setter(setter(Builder::calculatorName))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("CalculatorName").build()).build();

    private static final SdkField<CalculateRouteCarModeOptions> CAR_MODE_OPTIONS_FIELD = SdkField
            .<CalculateRouteCarModeOptions> builder(MarshallingType.SDK_POJO).memberName("CarModeOptions")
            .getter(getter(CalculateRouteMatrixRequest::carModeOptions)).setter(setter(Builder::carModeOptions))
            .constructor(CalculateRouteCarModeOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CarModeOptions").build()).build();

    private static final SdkField<Boolean> DEPART_NOW_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("DepartNow").getter(getter(CalculateRouteMatrixRequest::departNow)).setter(setter(Builder::departNow))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DepartNow").build()).build();

    private static final SdkField<List<List<Double>>> DEPARTURE_POSITIONS_FIELD = SdkField
            .<List<List<Double>>> builder(MarshallingType.LIST)
            .memberName("DeparturePositions")
            .getter(getter(CalculateRouteMatrixRequest::departurePositions))
            .setter(setter(Builder::departurePositions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeparturePositions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<List<Double>> builder(MarshallingType.LIST)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build(),
                                                    ListTrait
                                                            .builder()
                                                            .memberLocationName(null)
                                                            .memberFieldInfo(
                                                                    SdkField.<Double> builder(MarshallingType.DOUBLE)
                                                                            .traits(LocationTrait.builder()
                                                                                    .location(MarshallLocation.PAYLOAD)
                                                                                    .locationName("member").build()).build())
                                                            .build()).build()).build()).build();

    private static final SdkField<Instant> DEPARTURE_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("DepartureTime")
            .getter(getter(CalculateRouteMatrixRequest::departureTime))
            .setter(setter(Builder::departureTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DepartureTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<List<List<Double>>> DESTINATION_POSITIONS_FIELD = SdkField
            .<List<List<Double>>> builder(MarshallingType.LIST)
            .memberName("DestinationPositions")
            .getter(getter(CalculateRouteMatrixRequest::destinationPositions))
            .setter(setter(Builder::destinationPositions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DestinationPositions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<List<Double>> builder(MarshallingType.LIST)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build(),
                                                    ListTrait
                                                            .builder()
                                                            .memberLocationName(null)
                                                            .memberFieldInfo(
                                                                    SdkField.<Double> builder(MarshallingType.DOUBLE)
                                                                            .traits(LocationTrait.builder()
                                                                                    .location(MarshallLocation.PAYLOAD)
                                                                                    .locationName("member").build()).build())
                                                            .build()).build()).build()).build();

    private static final SdkField<String> DISTANCE_UNIT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DistanceUnit").getter(getter(CalculateRouteMatrixRequest::distanceUnitAsString))
            .setter(setter(Builder::distanceUnit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DistanceUnit").build()).build();

    private static final SdkField<String> KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Key")
            .getter(getter(CalculateRouteMatrixRequest::key)).setter(setter(Builder::key))
            .traits(LocationTrait.builder().location(MarshallLocation.QUERY_PARAM).locationName("key").build()).build();

    private static final SdkField<String> TRAVEL_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TravelMode").getter(getter(CalculateRouteMatrixRequest::travelModeAsString))
            .setter(setter(Builder::travelMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TravelMode").build()).build();

    private static final SdkField<CalculateRouteTruckModeOptions> TRUCK_MODE_OPTIONS_FIELD = SdkField
            .<CalculateRouteTruckModeOptions> builder(MarshallingType.SDK_POJO).memberName("TruckModeOptions")
            .getter(getter(CalculateRouteMatrixRequest::truckModeOptions)).setter(setter(Builder::truckModeOptions))
            .constructor(CalculateRouteTruckModeOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TruckModeOptions").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CALCULATOR_NAME_FIELD,
            CAR_MODE_OPTIONS_FIELD, DEPART_NOW_FIELD, DEPARTURE_POSITIONS_FIELD, DEPARTURE_TIME_FIELD,
            DESTINATION_POSITIONS_FIELD, DISTANCE_UNIT_FIELD, KEY_FIELD, TRAVEL_MODE_FIELD, TRUCK_MODE_OPTIONS_FIELD));

    private final String calculatorName;

    private final CalculateRouteCarModeOptions carModeOptions;

    private final Boolean departNow;

    private final List<List<Double>> departurePositions;

    private final Instant departureTime;

    private final List<List<Double>> destinationPositions;

    private final String distanceUnit;

    private final String key;

    private final String travelMode;

    private final CalculateRouteTruckModeOptions truckModeOptions;

    private CalculateRouteMatrixRequest(BuilderImpl builder) {
        super(builder);
        this.calculatorName = builder.calculatorName;
        this.carModeOptions = builder.carModeOptions;
        this.departNow = builder.departNow;
        this.departurePositions = builder.departurePositions;
        this.departureTime = builder.departureTime;
        this.destinationPositions = builder.destinationPositions;
        this.distanceUnit = builder.distanceUnit;
        this.key = builder.key;
        this.travelMode = builder.travelMode;
        this.truckModeOptions = builder.truckModeOptions;
    }

    /**
     * <p>
     * The name of the route calculator resource that you want to use to calculate the route matrix.
     * </p>
     * 
     * @return The name of the route calculator resource that you want to use to calculate the route matrix.
     */
    public final String calculatorName() {
        return calculatorName;
    }

    /**
     * <p>
     * Specifies route preferences when traveling by <code>Car</code>, such as avoiding routes that use ferries or
     * tolls.
     * </p>
     * <p>
     * Requirements: <code>TravelMode</code> must be specified as <code>Car</code>.
     * </p>
     * 
     * @return Specifies route preferences when traveling by <code>Car</code>, such as avoiding routes that use ferries
     *         or tolls.</p>
     *         <p>
     *         Requirements: <code>TravelMode</code> must be specified as <code>Car</code>.
     */
    public final CalculateRouteCarModeOptions carModeOptions() {
        return carModeOptions;
    }

    /**
     * <p>
     * Sets the time of departure as the current time. Uses the current time to calculate the route matrix. You can't
     * set both <code>DepartureTime</code> and <code>DepartNow</code>. If neither is set, the best time of day to travel
     * with the best traffic conditions is used to calculate the route matrix.
     * </p>
     * <p>
     * Default Value: <code>false</code>
     * </p>
     * <p>
     * Valid Values: <code>false</code> | <code>true</code>
     * </p>
     * 
     * @return Sets the time of departure as the current time. Uses the current time to calculate the route matrix. You
     *         can't set both <code>DepartureTime</code> and <code>DepartNow</code>. If neither is set, the best time of
     *         day to travel with the best traffic conditions is used to calculate the route matrix.</p>
     *         <p>
     *         Default Value: <code>false</code>
     *         </p>
     *         <p>
     *         Valid Values: <code>false</code> | <code>true</code>
     */
    public final Boolean departNow() {
        return departNow;
    }

    /**
     * For responses, this returns true if the service returned a value for the DeparturePositions property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasDeparturePositions() {
        return departurePositions != null && !(departurePositions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of departure (origin) positions for the route matrix. An array of points, each of which is itself a
     * 2-value array defined in <a href="https://earth-info.nga.mil/GandG/wgs84/index.html">WGS 84</a> format:
     * <code>[longitude, latitude]</code>. For example, <code>[-123.115, 49.285]</code>.
     * </p>
     * <important>
     * <p>
     * Depending on the data provider selected in the route calculator resource there may be additional restrictions on
     * the inputs you can choose. See <a href=
     * "https://docs.aws.amazon.com/location/latest/developerguide/calculate-route-matrix.html#matrix-routing-position-limits"
     * > Position restrictions</a> in the <i>Amazon Location Service Developer Guide</i>.
     * </p>
     * </important> <note>
     * <p>
     * For route calculators that use Esri as the data provider, if you specify a departure that's not located on a
     * road, Amazon Location <a
     * href="https://docs.aws.amazon.com/location/latest/developerguide/snap-to-nearby-road.html"> moves the position to
     * the nearest road</a>. The snapped value is available in the result in <code>SnappedDeparturePositions</code>.
     * </p>
     * </note>
     * <p>
     * Valid Values: <code>[-180 to 180,-90 to 90]</code>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDeparturePositions} method.
     * </p>
     * 
     * @return The list of departure (origin) positions for the route matrix. An array of points, each of which is
     *         itself a 2-value array defined in <a href="https://earth-info.nga.mil/GandG/wgs84/index.html">WGS 84</a>
     *         format: <code>[longitude, latitude]</code>. For example, <code>[-123.115, 49.285]</code>.</p> <important>
     *         <p>
     *         Depending on the data provider selected in the route calculator resource there may be additional
     *         restrictions on the inputs you can choose. See <a href=
     *         "https://docs.aws.amazon.com/location/latest/developerguide/calculate-route-matrix.html#matrix-routing-position-limits"
     *         > Position restrictions</a> in the <i>Amazon Location Service Developer Guide</i>.
     *         </p>
     *         </important> <note>
     *         <p>
     *         For route calculators that use Esri as the data provider, if you specify a departure that's not located
     *         on a road, Amazon Location <a
     *         href="https://docs.aws.amazon.com/location/latest/developerguide/snap-to-nearby-road.html"> moves the
     *         position to the nearest road</a>. The snapped value is available in the result in
     *         <code>SnappedDeparturePositions</code>.
     *         </p>
     *         </note>
     *         <p>
     *         Valid Values: <code>[-180 to 180,-90 to 90]</code>
     */
    public final List<List<Double>> departurePositions() {
        return departurePositions;
    }

    /**
     * <p>
     * Specifies the desired time of departure. Uses the given time to calculate the route matrix. You can't set both
     * <code>DepartureTime</code> and <code>DepartNow</code>. If neither is set, the best time of day to travel with the
     * best traffic conditions is used to calculate the route matrix.
     * </p>
     * <note>
     * <p>
     * Setting a departure time in the past returns a <code>400 ValidationException</code> error.
     * </p>
     * </note>
     * <ul>
     * <li>
     * <p>
     * In <a href="https://www.iso.org/iso-8601-date-and-time-format.html">ISO 8601</a> format:
     * <code>YYYY-MM-DDThh:mm:ss.sssZ</code>. For example, <code>2020–07-2T12:15:20.000Z+01:00</code>
     * </p>
     * </li>
     * </ul>
     * 
     * @return Specifies the desired time of departure. Uses the given time to calculate the route matrix. You can't set
     *         both <code>DepartureTime</code> and <code>DepartNow</code>. If neither is set, the best time of day to
     *         travel with the best traffic conditions is used to calculate the route matrix.</p> <note>
     *         <p>
     *         Setting a departure time in the past returns a <code>400 ValidationException</code> error.
     *         </p>
     *         </note>
     *         <ul>
     *         <li>
     *         <p>
     *         In <a href="https://www.iso.org/iso-8601-date-and-time-format.html">ISO 8601</a> format:
     *         <code>YYYY-MM-DDThh:mm:ss.sssZ</code>. For example, <code>2020–07-2T12:15:20.000Z+01:00</code>
     *         </p>
     *         </li>
     */
    public final Instant departureTime() {
        return departureTime;
    }

    /**
     * For responses, this returns true if the service returned a value for the DestinationPositions property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasDestinationPositions() {
        return destinationPositions != null && !(destinationPositions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of destination positions for the route matrix. An array of points, each of which is itself a 2-value
     * array defined in <a href="https://earth-info.nga.mil/GandG/wgs84/index.html">WGS 84</a> format:
     * <code>[longitude, latitude]</code>. For example, <code>[-122.339, 47.615]</code>
     * </p>
     * <important>
     * <p>
     * Depending on the data provider selected in the route calculator resource there may be additional restrictions on
     * the inputs you can choose. See <a href=
     * "https://docs.aws.amazon.com/location/latest/developerguide/calculate-route-matrix.html#matrix-routing-position-limits"
     * > Position restrictions</a> in the <i>Amazon Location Service Developer Guide</i>.
     * </p>
     * </important> <note>
     * <p>
     * For route calculators that use Esri as the data provider, if you specify a destination that's not located on a
     * road, Amazon Location <a
     * href="https://docs.aws.amazon.com/location/latest/developerguide/snap-to-nearby-road.html"> moves the position to
     * the nearest road</a>. The snapped value is available in the result in <code>SnappedDestinationPositions</code>.
     * </p>
     * </note>
     * <p>
     * Valid Values: <code>[-180 to 180,-90 to 90]</code>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDestinationPositions} method.
     * </p>
     * 
     * @return The list of destination positions for the route matrix. An array of points, each of which is itself a
     *         2-value array defined in <a href="https://earth-info.nga.mil/GandG/wgs84/index.html">WGS 84</a> format:
     *         <code>[longitude, latitude]</code>. For example, <code>[-122.339, 47.615]</code> </p> <important>
     *         <p>
     *         Depending on the data provider selected in the route calculator resource there may be additional
     *         restrictions on the inputs you can choose. See <a href=
     *         "https://docs.aws.amazon.com/location/latest/developerguide/calculate-route-matrix.html#matrix-routing-position-limits"
     *         > Position restrictions</a> in the <i>Amazon Location Service Developer Guide</i>.
     *         </p>
     *         </important> <note>
     *         <p>
     *         For route calculators that use Esri as the data provider, if you specify a destination that's not located
     *         on a road, Amazon Location <a
     *         href="https://docs.aws.amazon.com/location/latest/developerguide/snap-to-nearby-road.html"> moves the
     *         position to the nearest road</a>. The snapped value is available in the result in
     *         <code>SnappedDestinationPositions</code>.
     *         </p>
     *         </note>
     *         <p>
     *         Valid Values: <code>[-180 to 180,-90 to 90]</code>
     */
    public final List<List<Double>> destinationPositions() {
        return destinationPositions;
    }

    /**
     * <p>
     * Set the unit system to specify the distance.
     * </p>
     * <p>
     * Default Value: <code>Kilometers</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #distanceUnit} will
     * return {@link DistanceUnit#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #distanceUnitAsString}.
     * </p>
     * 
     * @return Set the unit system to specify the distance.</p>
     *         <p>
     *         Default Value: <code>Kilometers</code>
     * @see DistanceUnit
     */
    public final DistanceUnit distanceUnit() {
        return DistanceUnit.fromValue(distanceUnit);
    }

    /**
     * <p>
     * Set the unit system to specify the distance.
     * </p>
     * <p>
     * Default Value: <code>Kilometers</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #distanceUnit} will
     * return {@link DistanceUnit#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #distanceUnitAsString}.
     * </p>
     * 
     * @return Set the unit system to specify the distance.</p>
     *         <p>
     *         Default Value: <code>Kilometers</code>
     * @see DistanceUnit
     */
    public final String distanceUnitAsString() {
        return distanceUnit;
    }

    /**
     * <p>
     * The optional <a href="https://docs.aws.amazon.com/location/latest/developerguide/using-apikeys.html">API key</a>
     * to authorize the request.
     * </p>
     * 
     * @return The optional <a href="https://docs.aws.amazon.com/location/latest/developerguide/using-apikeys.html">API
     *         key</a> to authorize the request.
     */
    public final String key() {
        return key;
    }

    /**
     * <p>
     * Specifies the mode of transport when calculating a route. Used in estimating the speed of travel and road
     * compatibility.
     * </p>
     * <p>
     * The <code>TravelMode</code> you specify also determines how you specify route preferences:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If traveling by <code>Car</code> use the <code>CarModeOptions</code> parameter.
     * </p>
     * </li>
     * <li>
     * <p>
     * If traveling by <code>Truck</code> use the <code>TruckModeOptions</code> parameter.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * <code>Bicycle</code> or <code>Motorcycle</code> are only valid when using <code>Grab</code> as a data provider,
     * and only within Southeast Asia.
     * </p>
     * <p>
     * <code>Truck</code> is not available for Grab.
     * </p>
     * <p>
     * For more information about using Grab as a data provider, see <a
     * href="https://docs.aws.amazon.com/location/latest/developerguide/grab.html">GrabMaps</a> in the <i>Amazon
     * Location Service Developer Guide</i>.
     * </p>
     * </note>
     * <p>
     * Default Value: <code>Car</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #travelMode} will
     * return {@link TravelMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #travelModeAsString}.
     * </p>
     * 
     * @return Specifies the mode of transport when calculating a route. Used in estimating the speed of travel and road
     *         compatibility.</p>
     *         <p>
     *         The <code>TravelMode</code> you specify also determines how you specify route preferences:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If traveling by <code>Car</code> use the <code>CarModeOptions</code> parameter.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If traveling by <code>Truck</code> use the <code>TruckModeOptions</code> parameter.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         <code>Bicycle</code> or <code>Motorcycle</code> are only valid when using <code>Grab</code> as a data
     *         provider, and only within Southeast Asia.
     *         </p>
     *         <p>
     *         <code>Truck</code> is not available for Grab.
     *         </p>
     *         <p>
     *         For more information about using Grab as a data provider, see <a
     *         href="https://docs.aws.amazon.com/location/latest/developerguide/grab.html">GrabMaps</a> in the <i>Amazon
     *         Location Service Developer Guide</i>.
     *         </p>
     *         </note>
     *         <p>
     *         Default Value: <code>Car</code>
     * @see TravelMode
     */
    public final TravelMode travelMode() {
        return TravelMode.fromValue(travelMode);
    }

    /**
     * <p>
     * Specifies the mode of transport when calculating a route. Used in estimating the speed of travel and road
     * compatibility.
     * </p>
     * <p>
     * The <code>TravelMode</code> you specify also determines how you specify route preferences:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If traveling by <code>Car</code> use the <code>CarModeOptions</code> parameter.
     * </p>
     * </li>
     * <li>
     * <p>
     * If traveling by <code>Truck</code> use the <code>TruckModeOptions</code> parameter.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * <code>Bicycle</code> or <code>Motorcycle</code> are only valid when using <code>Grab</code> as a data provider,
     * and only within Southeast Asia.
     * </p>
     * <p>
     * <code>Truck</code> is not available for Grab.
     * </p>
     * <p>
     * For more information about using Grab as a data provider, see <a
     * href="https://docs.aws.amazon.com/location/latest/developerguide/grab.html">GrabMaps</a> in the <i>Amazon
     * Location Service Developer Guide</i>.
     * </p>
     * </note>
     * <p>
     * Default Value: <code>Car</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #travelMode} will
     * return {@link TravelMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #travelModeAsString}.
     * </p>
     * 
     * @return Specifies the mode of transport when calculating a route. Used in estimating the speed of travel and road
     *         compatibility.</p>
     *         <p>
     *         The <code>TravelMode</code> you specify also determines how you specify route preferences:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If traveling by <code>Car</code> use the <code>CarModeOptions</code> parameter.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If traveling by <code>Truck</code> use the <code>TruckModeOptions</code> parameter.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         <code>Bicycle</code> or <code>Motorcycle</code> are only valid when using <code>Grab</code> as a data
     *         provider, and only within Southeast Asia.
     *         </p>
     *         <p>
     *         <code>Truck</code> is not available for Grab.
     *         </p>
     *         <p>
     *         For more information about using Grab as a data provider, see <a
     *         href="https://docs.aws.amazon.com/location/latest/developerguide/grab.html">GrabMaps</a> in the <i>Amazon
     *         Location Service Developer Guide</i>.
     *         </p>
     *         </note>
     *         <p>
     *         Default Value: <code>Car</code>
     * @see TravelMode
     */
    public final String travelModeAsString() {
        return travelMode;
    }

    /**
     * <p>
     * Specifies route preferences when traveling by <code>Truck</code>, such as avoiding routes that use ferries or
     * tolls, and truck specifications to consider when choosing an optimal road.
     * </p>
     * <p>
     * Requirements: <code>TravelMode</code> must be specified as <code>Truck</code>.
     * </p>
     * 
     * @return Specifies route preferences when traveling by <code>Truck</code>, such as avoiding routes that use
     *         ferries or tolls, and truck specifications to consider when choosing an optimal road.</p>
     *         <p>
     *         Requirements: <code>TravelMode</code> must be specified as <code>Truck</code>.
     */
    public final CalculateRouteTruckModeOptions truckModeOptions() {
        return truckModeOptions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(calculatorName());
        hashCode = 31 * hashCode + Objects.hashCode(carModeOptions());
        hashCode = 31 * hashCode + Objects.hashCode(departNow());
        hashCode = 31 * hashCode + Objects.hashCode(hasDeparturePositions() ? departurePositions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(departureTime());
        hashCode = 31 * hashCode + Objects.hashCode(hasDestinationPositions() ? destinationPositions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(distanceUnitAsString());
        hashCode = 31 * hashCode + Objects.hashCode(key());
        hashCode = 31 * hashCode + Objects.hashCode(travelModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(truckModeOptions());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CalculateRouteMatrixRequest)) {
            return false;
        }
        CalculateRouteMatrixRequest other = (CalculateRouteMatrixRequest) obj;
        return Objects.equals(calculatorName(), other.calculatorName())
                && Objects.equals(carModeOptions(), other.carModeOptions()) && Objects.equals(departNow(), other.departNow())
                && hasDeparturePositions() == other.hasDeparturePositions()
                && Objects.equals(departurePositions(), other.departurePositions())
                && Objects.equals(departureTime(), other.departureTime())
                && hasDestinationPositions() == other.hasDestinationPositions()
                && Objects.equals(destinationPositions(), other.destinationPositions())
                && Objects.equals(distanceUnitAsString(), other.distanceUnitAsString()) && Objects.equals(key(), other.key())
                && Objects.equals(travelModeAsString(), other.travelModeAsString())
                && Objects.equals(truckModeOptions(), other.truckModeOptions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CalculateRouteMatrixRequest").add("CalculatorName", calculatorName())
                .add("CarModeOptions", carModeOptions()).add("DepartNow", departNow())
                .add("DeparturePositions", departurePositions() == null ? null : "*** Sensitive Data Redacted ***")
                .add("DepartureTime", departureTime())
                .add("DestinationPositions", destinationPositions() == null ? null : "*** Sensitive Data Redacted ***")
                .add("DistanceUnit", distanceUnitAsString()).add("Key", key() == null ? null : "*** Sensitive Data Redacted ***")
                .add("TravelMode", travelModeAsString()).add("TruckModeOptions", truckModeOptions()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CalculatorName":
            return Optional.ofNullable(clazz.cast(calculatorName()));
        case "CarModeOptions":
            return Optional.ofNullable(clazz.cast(carModeOptions()));
        case "DepartNow":
            return Optional.ofNullable(clazz.cast(departNow()));
        case "DeparturePositions":
            return Optional.ofNullable(clazz.cast(departurePositions()));
        case "DepartureTime":
            return Optional.ofNullable(clazz.cast(departureTime()));
        case "DestinationPositions":
            return Optional.ofNullable(clazz.cast(destinationPositions()));
        case "DistanceUnit":
            return Optional.ofNullable(clazz.cast(distanceUnitAsString()));
        case "Key":
            return Optional.ofNullable(clazz.cast(key()));
        case "TravelMode":
            return Optional.ofNullable(clazz.cast(travelModeAsString()));
        case "TruckModeOptions":
            return Optional.ofNullable(clazz.cast(truckModeOptions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CalculateRouteMatrixRequest, T> g) {
        return obj -> g.apply((CalculateRouteMatrixRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends LocationRequest.Builder, SdkPojo, CopyableBuilder<Builder, CalculateRouteMatrixRequest> {
        /**
         * <p>
         * The name of the route calculator resource that you want to use to calculate the route matrix.
         * </p>
         * 
         * @param calculatorName
         *        The name of the route calculator resource that you want to use to calculate the route matrix.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder calculatorName(String calculatorName);

        /**
         * <p>
         * Specifies route preferences when traveling by <code>Car</code>, such as avoiding routes that use ferries or
         * tolls.
         * </p>
         * <p>
         * Requirements: <code>TravelMode</code> must be specified as <code>Car</code>.
         * </p>
         * 
         * @param carModeOptions
         *        Specifies route preferences when traveling by <code>Car</code>, such as avoiding routes that use
         *        ferries or tolls.</p>
         *        <p>
         *        Requirements: <code>TravelMode</code> must be specified as <code>Car</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder carModeOptions(CalculateRouteCarModeOptions carModeOptions);

        /**
         * <p>
         * Specifies route preferences when traveling by <code>Car</code>, such as avoiding routes that use ferries or
         * tolls.
         * </p>
         * <p>
         * Requirements: <code>TravelMode</code> must be specified as <code>Car</code>.
         * </p>
         * This is a convenience method that creates an instance of the {@link CalculateRouteCarModeOptions.Builder}
         * avoiding the need to create one manually via {@link CalculateRouteCarModeOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CalculateRouteCarModeOptions.Builder#build()} is called
         * immediately and its result is passed to {@link #carModeOptions(CalculateRouteCarModeOptions)}.
         * 
         * @param carModeOptions
         *        a consumer that will call methods on {@link CalculateRouteCarModeOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #carModeOptions(CalculateRouteCarModeOptions)
         */
        default Builder carModeOptions(Consumer<CalculateRouteCarModeOptions.Builder> carModeOptions) {
            return carModeOptions(CalculateRouteCarModeOptions.builder().applyMutation(carModeOptions).build());
        }

        /**
         * <p>
         * Sets the time of departure as the current time. Uses the current time to calculate the route matrix. You
         * can't set both <code>DepartureTime</code> and <code>DepartNow</code>. If neither is set, the best time of day
         * to travel with the best traffic conditions is used to calculate the route matrix.
         * </p>
         * <p>
         * Default Value: <code>false</code>
         * </p>
         * <p>
         * Valid Values: <code>false</code> | <code>true</code>
         * </p>
         * 
         * @param departNow
         *        Sets the time of departure as the current time. Uses the current time to calculate the route matrix.
         *        You can't set both <code>DepartureTime</code> and <code>DepartNow</code>. If neither is set, the best
         *        time of day to travel with the best traffic conditions is used to calculate the route matrix.</p>
         *        <p>
         *        Default Value: <code>false</code>
         *        </p>
         *        <p>
         *        Valid Values: <code>false</code> | <code>true</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder departNow(Boolean departNow);

        /**
         * <p>
         * The list of departure (origin) positions for the route matrix. An array of points, each of which is itself a
         * 2-value array defined in <a href="https://earth-info.nga.mil/GandG/wgs84/index.html">WGS 84</a> format:
         * <code>[longitude, latitude]</code>. For example, <code>[-123.115, 49.285]</code>.
         * </p>
         * <important>
         * <p>
         * Depending on the data provider selected in the route calculator resource there may be additional restrictions
         * on the inputs you can choose. See <a href=
         * "https://docs.aws.amazon.com/location/latest/developerguide/calculate-route-matrix.html#matrix-routing-position-limits"
         * > Position restrictions</a> in the <i>Amazon Location Service Developer Guide</i>.
         * </p>
         * </important> <note>
         * <p>
         * For route calculators that use Esri as the data provider, if you specify a departure that's not located on a
         * road, Amazon Location <a
         * href="https://docs.aws.amazon.com/location/latest/developerguide/snap-to-nearby-road.html"> moves the
         * position to the nearest road</a>. The snapped value is available in the result in
         * <code>SnappedDeparturePositions</code>.
         * </p>
         * </note>
         * <p>
         * Valid Values: <code>[-180 to 180,-90 to 90]</code>
         * </p>
         * 
         * @param departurePositions
         *        The list of departure (origin) positions for the route matrix. An array of points, each of which is
         *        itself a 2-value array defined in <a href="https://earth-info.nga.mil/GandG/wgs84/index.html">WGS
         *        84</a> format: <code>[longitude, latitude]</code>. For example, <code>[-123.115, 49.285]</code>.</p>
         *        <important>
         *        <p>
         *        Depending on the data provider selected in the route calculator resource there may be additional
         *        restrictions on the inputs you can choose. See <a href=
         *        "https://docs.aws.amazon.com/location/latest/developerguide/calculate-route-matrix.html#matrix-routing-position-limits"
         *        > Position restrictions</a> in the <i>Amazon Location Service Developer Guide</i>.
         *        </p>
         *        </important> <note>
         *        <p>
         *        For route calculators that use Esri as the data provider, if you specify a departure that's not
         *        located on a road, Amazon Location <a
         *        href="https://docs.aws.amazon.com/location/latest/developerguide/snap-to-nearby-road.html"> moves the
         *        position to the nearest road</a>. The snapped value is available in the result in
         *        <code>SnappedDeparturePositions</code>.
         *        </p>
         *        </note>
         *        <p>
         *        Valid Values: <code>[-180 to 180,-90 to 90]</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder departurePositions(Collection<? extends Collection<Double>> departurePositions);

        /**
         * <p>
         * The list of departure (origin) positions for the route matrix. An array of points, each of which is itself a
         * 2-value array defined in <a href="https://earth-info.nga.mil/GandG/wgs84/index.html">WGS 84</a> format:
         * <code>[longitude, latitude]</code>. For example, <code>[-123.115, 49.285]</code>.
         * </p>
         * <important>
         * <p>
         * Depending on the data provider selected in the route calculator resource there may be additional restrictions
         * on the inputs you can choose. See <a href=
         * "https://docs.aws.amazon.com/location/latest/developerguide/calculate-route-matrix.html#matrix-routing-position-limits"
         * > Position restrictions</a> in the <i>Amazon Location Service Developer Guide</i>.
         * </p>
         * </important> <note>
         * <p>
         * For route calculators that use Esri as the data provider, if you specify a departure that's not located on a
         * road, Amazon Location <a
         * href="https://docs.aws.amazon.com/location/latest/developerguide/snap-to-nearby-road.html"> moves the
         * position to the nearest road</a>. The snapped value is available in the result in
         * <code>SnappedDeparturePositions</code>.
         * </p>
         * </note>
         * <p>
         * Valid Values: <code>[-180 to 180,-90 to 90]</code>
         * </p>
         * 
         * @param departurePositions
         *        The list of departure (origin) positions for the route matrix. An array of points, each of which is
         *        itself a 2-value array defined in <a href="https://earth-info.nga.mil/GandG/wgs84/index.html">WGS
         *        84</a> format: <code>[longitude, latitude]</code>. For example, <code>[-123.115, 49.285]</code>.</p>
         *        <important>
         *        <p>
         *        Depending on the data provider selected in the route calculator resource there may be additional
         *        restrictions on the inputs you can choose. See <a href=
         *        "https://docs.aws.amazon.com/location/latest/developerguide/calculate-route-matrix.html#matrix-routing-position-limits"
         *        > Position restrictions</a> in the <i>Amazon Location Service Developer Guide</i>.
         *        </p>
         *        </important> <note>
         *        <p>
         *        For route calculators that use Esri as the data provider, if you specify a departure that's not
         *        located on a road, Amazon Location <a
         *        href="https://docs.aws.amazon.com/location/latest/developerguide/snap-to-nearby-road.html"> moves the
         *        position to the nearest road</a>. The snapped value is available in the result in
         *        <code>SnappedDeparturePositions</code>.
         *        </p>
         *        </note>
         *        <p>
         *        Valid Values: <code>[-180 to 180,-90 to 90]</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder departurePositions(Collection<Double>... departurePositions);

        /**
         * <p>
         * Specifies the desired time of departure. Uses the given time to calculate the route matrix. You can't set
         * both <code>DepartureTime</code> and <code>DepartNow</code>. If neither is set, the best time of day to travel
         * with the best traffic conditions is used to calculate the route matrix.
         * </p>
         * <note>
         * <p>
         * Setting a departure time in the past returns a <code>400 ValidationException</code> error.
         * </p>
         * </note>
         * <ul>
         * <li>
         * <p>
         * In <a href="https://www.iso.org/iso-8601-date-and-time-format.html">ISO 8601</a> format:
         * <code>YYYY-MM-DDThh:mm:ss.sssZ</code>. For example, <code>2020–07-2T12:15:20.000Z+01:00</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param departureTime
         *        Specifies the desired time of departure. Uses the given time to calculate the route matrix. You can't
         *        set both <code>DepartureTime</code> and <code>DepartNow</code>. If neither is set, the best time of
         *        day to travel with the best traffic conditions is used to calculate the route matrix.</p> <note>
         *        <p>
         *        Setting a departure time in the past returns a <code>400 ValidationException</code> error.
         *        </p>
         *        </note>
         *        <ul>
         *        <li>
         *        <p>
         *        In <a href="https://www.iso.org/iso-8601-date-and-time-format.html">ISO 8601</a> format:
         *        <code>YYYY-MM-DDThh:mm:ss.sssZ</code>. For example, <code>2020–07-2T12:15:20.000Z+01:00</code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder departureTime(Instant departureTime);

        /**
         * <p>
         * The list of destination positions for the route matrix. An array of points, each of which is itself a 2-value
         * array defined in <a href="https://earth-info.nga.mil/GandG/wgs84/index.html">WGS 84</a> format:
         * <code>[longitude, latitude]</code>. For example, <code>[-122.339, 47.615]</code>
         * </p>
         * <important>
         * <p>
         * Depending on the data provider selected in the route calculator resource there may be additional restrictions
         * on the inputs you can choose. See <a href=
         * "https://docs.aws.amazon.com/location/latest/developerguide/calculate-route-matrix.html#matrix-routing-position-limits"
         * > Position restrictions</a> in the <i>Amazon Location Service Developer Guide</i>.
         * </p>
         * </important> <note>
         * <p>
         * For route calculators that use Esri as the data provider, if you specify a destination that's not located on
         * a road, Amazon Location <a
         * href="https://docs.aws.amazon.com/location/latest/developerguide/snap-to-nearby-road.html"> moves the
         * position to the nearest road</a>. The snapped value is available in the result in
         * <code>SnappedDestinationPositions</code>.
         * </p>
         * </note>
         * <p>
         * Valid Values: <code>[-180 to 180,-90 to 90]</code>
         * </p>
         * 
         * @param destinationPositions
         *        The list of destination positions for the route matrix. An array of points, each of which is itself a
         *        2-value array defined in <a href="https://earth-info.nga.mil/GandG/wgs84/index.html">WGS 84</a>
         *        format: <code>[longitude, latitude]</code>. For example, <code>[-122.339, 47.615]</code> </p>
         *        <important>
         *        <p>
         *        Depending on the data provider selected in the route calculator resource there may be additional
         *        restrictions on the inputs you can choose. See <a href=
         *        "https://docs.aws.amazon.com/location/latest/developerguide/calculate-route-matrix.html#matrix-routing-position-limits"
         *        > Position restrictions</a> in the <i>Amazon Location Service Developer Guide</i>.
         *        </p>
         *        </important> <note>
         *        <p>
         *        For route calculators that use Esri as the data provider, if you specify a destination that's not
         *        located on a road, Amazon Location <a
         *        href="https://docs.aws.amazon.com/location/latest/developerguide/snap-to-nearby-road.html"> moves the
         *        position to the nearest road</a>. The snapped value is available in the result in
         *        <code>SnappedDestinationPositions</code>.
         *        </p>
         *        </note>
         *        <p>
         *        Valid Values: <code>[-180 to 180,-90 to 90]</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinationPositions(Collection<? extends Collection<Double>> destinationPositions);

        /**
         * <p>
         * The list of destination positions for the route matrix. An array of points, each of which is itself a 2-value
         * array defined in <a href="https://earth-info.nga.mil/GandG/wgs84/index.html">WGS 84</a> format:
         * <code>[longitude, latitude]</code>. For example, <code>[-122.339, 47.615]</code>
         * </p>
         * <important>
         * <p>
         * Depending on the data provider selected in the route calculator resource there may be additional restrictions
         * on the inputs you can choose. See <a href=
         * "https://docs.aws.amazon.com/location/latest/developerguide/calculate-route-matrix.html#matrix-routing-position-limits"
         * > Position restrictions</a> in the <i>Amazon Location Service Developer Guide</i>.
         * </p>
         * </important> <note>
         * <p>
         * For route calculators that use Esri as the data provider, if you specify a destination that's not located on
         * a road, Amazon Location <a
         * href="https://docs.aws.amazon.com/location/latest/developerguide/snap-to-nearby-road.html"> moves the
         * position to the nearest road</a>. The snapped value is available in the result in
         * <code>SnappedDestinationPositions</code>.
         * </p>
         * </note>
         * <p>
         * Valid Values: <code>[-180 to 180,-90 to 90]</code>
         * </p>
         * 
         * @param destinationPositions
         *        The list of destination positions for the route matrix. An array of points, each of which is itself a
         *        2-value array defined in <a href="https://earth-info.nga.mil/GandG/wgs84/index.html">WGS 84</a>
         *        format: <code>[longitude, latitude]</code>. For example, <code>[-122.339, 47.615]</code> </p>
         *        <important>
         *        <p>
         *        Depending on the data provider selected in the route calculator resource there may be additional
         *        restrictions on the inputs you can choose. See <a href=
         *        "https://docs.aws.amazon.com/location/latest/developerguide/calculate-route-matrix.html#matrix-routing-position-limits"
         *        > Position restrictions</a> in the <i>Amazon Location Service Developer Guide</i>.
         *        </p>
         *        </important> <note>
         *        <p>
         *        For route calculators that use Esri as the data provider, if you specify a destination that's not
         *        located on a road, Amazon Location <a
         *        href="https://docs.aws.amazon.com/location/latest/developerguide/snap-to-nearby-road.html"> moves the
         *        position to the nearest road</a>. The snapped value is available in the result in
         *        <code>SnappedDestinationPositions</code>.
         *        </p>
         *        </note>
         *        <p>
         *        Valid Values: <code>[-180 to 180,-90 to 90]</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinationPositions(Collection<Double>... destinationPositions);

        /**
         * <p>
         * Set the unit system to specify the distance.
         * </p>
         * <p>
         * Default Value: <code>Kilometers</code>
         * </p>
         * 
         * @param distanceUnit
         *        Set the unit system to specify the distance.</p>
         *        <p>
         *        Default Value: <code>Kilometers</code>
         * @see DistanceUnit
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DistanceUnit
         */
        Builder distanceUnit(String distanceUnit);

        /**
         * <p>
         * Set the unit system to specify the distance.
         * </p>
         * <p>
         * Default Value: <code>Kilometers</code>
         * </p>
         * 
         * @param distanceUnit
         *        Set the unit system to specify the distance.</p>
         *        <p>
         *        Default Value: <code>Kilometers</code>
         * @see DistanceUnit
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DistanceUnit
         */
        Builder distanceUnit(DistanceUnit distanceUnit);

        /**
         * <p>
         * The optional <a href="https://docs.aws.amazon.com/location/latest/developerguide/using-apikeys.html">API
         * key</a> to authorize the request.
         * </p>
         * 
         * @param key
         *        The optional <a
         *        href="https://docs.aws.amazon.com/location/latest/developerguide/using-apikeys.html">API key</a> to
         *        authorize the request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder key(String key);

        /**
         * <p>
         * Specifies the mode of transport when calculating a route. Used in estimating the speed of travel and road
         * compatibility.
         * </p>
         * <p>
         * The <code>TravelMode</code> you specify also determines how you specify route preferences:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If traveling by <code>Car</code> use the <code>CarModeOptions</code> parameter.
         * </p>
         * </li>
         * <li>
         * <p>
         * If traveling by <code>Truck</code> use the <code>TruckModeOptions</code> parameter.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * <code>Bicycle</code> or <code>Motorcycle</code> are only valid when using <code>Grab</code> as a data
         * provider, and only within Southeast Asia.
         * </p>
         * <p>
         * <code>Truck</code> is not available for Grab.
         * </p>
         * <p>
         * For more information about using Grab as a data provider, see <a
         * href="https://docs.aws.amazon.com/location/latest/developerguide/grab.html">GrabMaps</a> in the <i>Amazon
         * Location Service Developer Guide</i>.
         * </p>
         * </note>
         * <p>
         * Default Value: <code>Car</code>
         * </p>
         * 
         * @param travelMode
         *        Specifies the mode of transport when calculating a route. Used in estimating the speed of travel and
         *        road compatibility.</p>
         *        <p>
         *        The <code>TravelMode</code> you specify also determines how you specify route preferences:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If traveling by <code>Car</code> use the <code>CarModeOptions</code> parameter.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If traveling by <code>Truck</code> use the <code>TruckModeOptions</code> parameter.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        <code>Bicycle</code> or <code>Motorcycle</code> are only valid when using <code>Grab</code> as a data
         *        provider, and only within Southeast Asia.
         *        </p>
         *        <p>
         *        <code>Truck</code> is not available for Grab.
         *        </p>
         *        <p>
         *        For more information about using Grab as a data provider, see <a
         *        href="https://docs.aws.amazon.com/location/latest/developerguide/grab.html">GrabMaps</a> in the
         *        <i>Amazon Location Service Developer Guide</i>.
         *        </p>
         *        </note>
         *        <p>
         *        Default Value: <code>Car</code>
         * @see TravelMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TravelMode
         */
        Builder travelMode(String travelMode);

        /**
         * <p>
         * Specifies the mode of transport when calculating a route. Used in estimating the speed of travel and road
         * compatibility.
         * </p>
         * <p>
         * The <code>TravelMode</code> you specify also determines how you specify route preferences:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If traveling by <code>Car</code> use the <code>CarModeOptions</code> parameter.
         * </p>
         * </li>
         * <li>
         * <p>
         * If traveling by <code>Truck</code> use the <code>TruckModeOptions</code> parameter.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * <code>Bicycle</code> or <code>Motorcycle</code> are only valid when using <code>Grab</code> as a data
         * provider, and only within Southeast Asia.
         * </p>
         * <p>
         * <code>Truck</code> is not available for Grab.
         * </p>
         * <p>
         * For more information about using Grab as a data provider, see <a
         * href="https://docs.aws.amazon.com/location/latest/developerguide/grab.html">GrabMaps</a> in the <i>Amazon
         * Location Service Developer Guide</i>.
         * </p>
         * </note>
         * <p>
         * Default Value: <code>Car</code>
         * </p>
         * 
         * @param travelMode
         *        Specifies the mode of transport when calculating a route. Used in estimating the speed of travel and
         *        road compatibility.</p>
         *        <p>
         *        The <code>TravelMode</code> you specify also determines how you specify route preferences:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If traveling by <code>Car</code> use the <code>CarModeOptions</code> parameter.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If traveling by <code>Truck</code> use the <code>TruckModeOptions</code> parameter.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        <code>Bicycle</code> or <code>Motorcycle</code> are only valid when using <code>Grab</code> as a data
         *        provider, and only within Southeast Asia.
         *        </p>
         *        <p>
         *        <code>Truck</code> is not available for Grab.
         *        </p>
         *        <p>
         *        For more information about using Grab as a data provider, see <a
         *        href="https://docs.aws.amazon.com/location/latest/developerguide/grab.html">GrabMaps</a> in the
         *        <i>Amazon Location Service Developer Guide</i>.
         *        </p>
         *        </note>
         *        <p>
         *        Default Value: <code>Car</code>
         * @see TravelMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TravelMode
         */
        Builder travelMode(TravelMode travelMode);

        /**
         * <p>
         * Specifies route preferences when traveling by <code>Truck</code>, such as avoiding routes that use ferries or
         * tolls, and truck specifications to consider when choosing an optimal road.
         * </p>
         * <p>
         * Requirements: <code>TravelMode</code> must be specified as <code>Truck</code>.
         * </p>
         * 
         * @param truckModeOptions
         *        Specifies route preferences when traveling by <code>Truck</code>, such as avoiding routes that use
         *        ferries or tolls, and truck specifications to consider when choosing an optimal road.</p>
         *        <p>
         *        Requirements: <code>TravelMode</code> must be specified as <code>Truck</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder truckModeOptions(CalculateRouteTruckModeOptions truckModeOptions);

        /**
         * <p>
         * Specifies route preferences when traveling by <code>Truck</code>, such as avoiding routes that use ferries or
         * tolls, and truck specifications to consider when choosing an optimal road.
         * </p>
         * <p>
         * Requirements: <code>TravelMode</code> must be specified as <code>Truck</code>.
         * </p>
         * This is a convenience method that creates an instance of the {@link CalculateRouteTruckModeOptions.Builder}
         * avoiding the need to create one manually via {@link CalculateRouteTruckModeOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CalculateRouteTruckModeOptions.Builder#build()} is called
         * immediately and its result is passed to {@link #truckModeOptions(CalculateRouteTruckModeOptions)}.
         * 
         * @param truckModeOptions
         *        a consumer that will call methods on {@link CalculateRouteTruckModeOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #truckModeOptions(CalculateRouteTruckModeOptions)
         */
        default Builder truckModeOptions(Consumer<CalculateRouteTruckModeOptions.Builder> truckModeOptions) {
            return truckModeOptions(CalculateRouteTruckModeOptions.builder().applyMutation(truckModeOptions).build());
        }

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends LocationRequest.BuilderImpl implements Builder {
        private String calculatorName;

        private CalculateRouteCarModeOptions carModeOptions;

        private Boolean departNow;

        private List<List<Double>> departurePositions = DefaultSdkAutoConstructList.getInstance();

        private Instant departureTime;

        private List<List<Double>> destinationPositions = DefaultSdkAutoConstructList.getInstance();

        private String distanceUnit;

        private String key;

        private String travelMode;

        private CalculateRouteTruckModeOptions truckModeOptions;

        private BuilderImpl() {
        }

        private BuilderImpl(CalculateRouteMatrixRequest model) {
            super(model);
            calculatorName(model.calculatorName);
            carModeOptions(model.carModeOptions);
            departNow(model.departNow);
            departurePositions(model.departurePositions);
            departureTime(model.departureTime);
            destinationPositions(model.destinationPositions);
            distanceUnit(model.distanceUnit);
            key(model.key);
            travelMode(model.travelMode);
            truckModeOptions(model.truckModeOptions);
        }

        public final String getCalculatorName() {
            return calculatorName;
        }

        public final void setCalculatorName(String calculatorName) {
            this.calculatorName = calculatorName;
        }

        @Override
        public final Builder calculatorName(String calculatorName) {
            this.calculatorName = calculatorName;
            return this;
        }

        public final CalculateRouteCarModeOptions.Builder getCarModeOptions() {
            return carModeOptions != null ? carModeOptions.toBuilder() : null;
        }

        public final void setCarModeOptions(CalculateRouteCarModeOptions.BuilderImpl carModeOptions) {
            this.carModeOptions = carModeOptions != null ? carModeOptions.build() : null;
        }

        @Override
        public final Builder carModeOptions(CalculateRouteCarModeOptions carModeOptions) {
            this.carModeOptions = carModeOptions;
            return this;
        }

        public final Boolean getDepartNow() {
            return departNow;
        }

        public final void setDepartNow(Boolean departNow) {
            this.departNow = departNow;
        }

        @Override
        public final Builder departNow(Boolean departNow) {
            this.departNow = departNow;
            return this;
        }

        public final Collection<? extends Collection<Double>> getDeparturePositions() {
            if (departurePositions instanceof SdkAutoConstructList) {
                return null;
            }
            return departurePositions;
        }

        public final void setDeparturePositions(Collection<? extends Collection<Double>> departurePositions) {
            this.departurePositions = CalculateRouteMatrixRequestDeparturePositionsListCopier.copy(departurePositions);
        }

        @Override
        public final Builder departurePositions(Collection<? extends Collection<Double>> departurePositions) {
            this.departurePositions = CalculateRouteMatrixRequestDeparturePositionsListCopier.copy(departurePositions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder departurePositions(Collection<Double>... departurePositions) {
            departurePositions(Arrays.asList(departurePositions));
            return this;
        }

        public final Instant getDepartureTime() {
            return departureTime;
        }

        public final void setDepartureTime(Instant departureTime) {
            this.departureTime = departureTime;
        }

        @Override
        public final Builder departureTime(Instant departureTime) {
            this.departureTime = departureTime;
            return this;
        }

        public final Collection<? extends Collection<Double>> getDestinationPositions() {
            if (destinationPositions instanceof SdkAutoConstructList) {
                return null;
            }
            return destinationPositions;
        }

        public final void setDestinationPositions(Collection<? extends Collection<Double>> destinationPositions) {
            this.destinationPositions = CalculateRouteMatrixRequestDestinationPositionsListCopier.copy(destinationPositions);
        }

        @Override
        public final Builder destinationPositions(Collection<? extends Collection<Double>> destinationPositions) {
            this.destinationPositions = CalculateRouteMatrixRequestDestinationPositionsListCopier.copy(destinationPositions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder destinationPositions(Collection<Double>... destinationPositions) {
            destinationPositions(Arrays.asList(destinationPositions));
            return this;
        }

        public final String getDistanceUnit() {
            return distanceUnit;
        }

        public final void setDistanceUnit(String distanceUnit) {
            this.distanceUnit = distanceUnit;
        }

        @Override
        public final Builder distanceUnit(String distanceUnit) {
            this.distanceUnit = distanceUnit;
            return this;
        }

        @Override
        public final Builder distanceUnit(DistanceUnit distanceUnit) {
            this.distanceUnit(distanceUnit == null ? null : distanceUnit.toString());
            return this;
        }

        public final String getKey() {
            return key;
        }

        public final void setKey(String key) {
            this.key = key;
        }

        @Override
        public final Builder key(String key) {
            this.key = key;
            return this;
        }

        public final String getTravelMode() {
            return travelMode;
        }

        public final void setTravelMode(String travelMode) {
            this.travelMode = travelMode;
        }

        @Override
        public final Builder travelMode(String travelMode) {
            this.travelMode = travelMode;
            return this;
        }

        @Override
        public final Builder travelMode(TravelMode travelMode) {
            this.travelMode(travelMode == null ? null : travelMode.toString());
            return this;
        }

        public final CalculateRouteTruckModeOptions.Builder getTruckModeOptions() {
            return truckModeOptions != null ? truckModeOptions.toBuilder() : null;
        }

        public final void setTruckModeOptions(CalculateRouteTruckModeOptions.BuilderImpl truckModeOptions) {
            this.truckModeOptions = truckModeOptions != null ? truckModeOptions.build() : null;
        }

        @Override
        public final Builder truckModeOptions(CalculateRouteTruckModeOptions truckModeOptions) {
            this.truckModeOptions = truckModeOptions;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CalculateRouteMatrixRequest build() {
            return new CalculateRouteMatrixRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
