/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.location.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A forecasted event represents a geofence event in relation to the requested device state, that may occur given the
 * provided device state and time horizon.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ForecastedEvent implements SdkPojo, Serializable, ToCopyableBuilder<ForecastedEvent.Builder, ForecastedEvent> {
    private static final SdkField<String> EVENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EventId").getter(getter(ForecastedEvent::eventId)).setter(setter(Builder::eventId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventId").build()).build();

    private static final SdkField<String> GEOFENCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("GeofenceId").getter(getter(ForecastedEvent::geofenceId)).setter(setter(Builder::geofenceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GeofenceId").build()).build();

    private static final SdkField<Boolean> IS_DEVICE_IN_GEOFENCE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("IsDeviceInGeofence").getter(getter(ForecastedEvent::isDeviceInGeofence))
            .setter(setter(Builder::isDeviceInGeofence))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IsDeviceInGeofence").build())
            .build();

    private static final SdkField<Double> NEAREST_DISTANCE_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("NearestDistance").getter(getter(ForecastedEvent::nearestDistance))
            .setter(setter(Builder::nearestDistance))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NearestDistance").build()).build();

    private static final SdkField<String> EVENT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EventType").getter(getter(ForecastedEvent::eventTypeAsString)).setter(setter(Builder::eventType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventType").build()).build();

    private static final SdkField<Instant> FORECASTED_BREACH_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("ForecastedBreachTime")
            .getter(getter(ForecastedEvent::forecastedBreachTime))
            .setter(setter(Builder::forecastedBreachTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ForecastedBreachTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Map<String, String>> GEOFENCE_PROPERTIES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("GeofenceProperties")
            .getter(getter(ForecastedEvent::geofenceProperties))
            .setter(setter(Builder::geofenceProperties))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GeofenceProperties").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EVENT_ID_FIELD,
            GEOFENCE_ID_FIELD, IS_DEVICE_IN_GEOFENCE_FIELD, NEAREST_DISTANCE_FIELD, EVENT_TYPE_FIELD,
            FORECASTED_BREACH_TIME_FIELD, GEOFENCE_PROPERTIES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String eventId;

    private final String geofenceId;

    private final Boolean isDeviceInGeofence;

    private final Double nearestDistance;

    private final String eventType;

    private final Instant forecastedBreachTime;

    private final Map<String, String> geofenceProperties;

    private ForecastedEvent(BuilderImpl builder) {
        this.eventId = builder.eventId;
        this.geofenceId = builder.geofenceId;
        this.isDeviceInGeofence = builder.isDeviceInGeofence;
        this.nearestDistance = builder.nearestDistance;
        this.eventType = builder.eventType;
        this.forecastedBreachTime = builder.forecastedBreachTime;
        this.geofenceProperties = builder.geofenceProperties;
    }

    /**
     * <p>
     * The forecasted event identifier.
     * </p>
     * 
     * @return The forecasted event identifier.
     */
    public final String eventId() {
        return eventId;
    }

    /**
     * <p>
     * The geofence identifier pertaining to the forecasted event.
     * </p>
     * 
     * @return The geofence identifier pertaining to the forecasted event.
     */
    public final String geofenceId() {
        return geofenceId;
    }

    /**
     * <p>
     * Indicates if the device is located within the geofence.
     * </p>
     * 
     * @return Indicates if the device is located within the geofence.
     */
    public final Boolean isDeviceInGeofence() {
        return isDeviceInGeofence;
    }

    /**
     * <p>
     * The closest distance from the device's position to the geofence.
     * </p>
     * 
     * @return The closest distance from the device's position to the geofence.
     */
    public final Double nearestDistance() {
        return nearestDistance;
    }

    /**
     * <p>
     * The event type, forecasting three states for which a device can be in relative to a geofence:
     * </p>
     * <p>
     * <code>ENTER</code>: If a device is outside of a geofence, but would breach the fence if the device is moving at
     * its current speed within time horizon window.
     * </p>
     * <p>
     * <code>EXIT</code>: If a device is inside of a geofence, but would breach the fence if the device is moving at its
     * current speed within time horizon window.
     * </p>
     * <p>
     * <code>IDLE</code>: If a device is inside of a geofence, and the device is not moving.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #eventType} will
     * return {@link ForecastedGeofenceEventType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #eventTypeAsString}.
     * </p>
     * 
     * @return The event type, forecasting three states for which a device can be in relative to a geofence:</p>
     *         <p>
     *         <code>ENTER</code>: If a device is outside of a geofence, but would breach the fence if the device is
     *         moving at its current speed within time horizon window.
     *         </p>
     *         <p>
     *         <code>EXIT</code>: If a device is inside of a geofence, but would breach the fence if the device is
     *         moving at its current speed within time horizon window.
     *         </p>
     *         <p>
     *         <code>IDLE</code>: If a device is inside of a geofence, and the device is not moving.
     * @see ForecastedGeofenceEventType
     */
    public final ForecastedGeofenceEventType eventType() {
        return ForecastedGeofenceEventType.fromValue(eventType);
    }

    /**
     * <p>
     * The event type, forecasting three states for which a device can be in relative to a geofence:
     * </p>
     * <p>
     * <code>ENTER</code>: If a device is outside of a geofence, but would breach the fence if the device is moving at
     * its current speed within time horizon window.
     * </p>
     * <p>
     * <code>EXIT</code>: If a device is inside of a geofence, but would breach the fence if the device is moving at its
     * current speed within time horizon window.
     * </p>
     * <p>
     * <code>IDLE</code>: If a device is inside of a geofence, and the device is not moving.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #eventType} will
     * return {@link ForecastedGeofenceEventType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #eventTypeAsString}.
     * </p>
     * 
     * @return The event type, forecasting three states for which a device can be in relative to a geofence:</p>
     *         <p>
     *         <code>ENTER</code>: If a device is outside of a geofence, but would breach the fence if the device is
     *         moving at its current speed within time horizon window.
     *         </p>
     *         <p>
     *         <code>EXIT</code>: If a device is inside of a geofence, but would breach the fence if the device is
     *         moving at its current speed within time horizon window.
     *         </p>
     *         <p>
     *         <code>IDLE</code>: If a device is inside of a geofence, and the device is not moving.
     * @see ForecastedGeofenceEventType
     */
    public final String eventTypeAsString() {
        return eventType;
    }

    /**
     * <p>
     * The forecasted time the device will breach the geofence in <a
     * href="https://www.iso.org/iso-8601-date-and-time-format.html">ISO 8601</a> format:
     * <code>YYYY-MM-DDThh:mm:ss.sssZ</code>
     * </p>
     * 
     * @return The forecasted time the device will breach the geofence in <a
     *         href="https://www.iso.org/iso-8601-date-and-time-format.html">ISO 8601</a> format:
     *         <code>YYYY-MM-DDThh:mm:ss.sssZ</code>
     */
    public final Instant forecastedBreachTime() {
        return forecastedBreachTime;
    }

    /**
     * For responses, this returns true if the service returned a value for the GeofenceProperties property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasGeofenceProperties() {
        return geofenceProperties != null && !(geofenceProperties instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The geofence properties.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasGeofenceProperties} method.
     * </p>
     * 
     * @return The geofence properties.
     */
    public final Map<String, String> geofenceProperties() {
        return geofenceProperties;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(eventId());
        hashCode = 31 * hashCode + Objects.hashCode(geofenceId());
        hashCode = 31 * hashCode + Objects.hashCode(isDeviceInGeofence());
        hashCode = 31 * hashCode + Objects.hashCode(nearestDistance());
        hashCode = 31 * hashCode + Objects.hashCode(eventTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(forecastedBreachTime());
        hashCode = 31 * hashCode + Objects.hashCode(hasGeofenceProperties() ? geofenceProperties() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ForecastedEvent)) {
            return false;
        }
        ForecastedEvent other = (ForecastedEvent) obj;
        return Objects.equals(eventId(), other.eventId()) && Objects.equals(geofenceId(), other.geofenceId())
                && Objects.equals(isDeviceInGeofence(), other.isDeviceInGeofence())
                && Objects.equals(nearestDistance(), other.nearestDistance())
                && Objects.equals(eventTypeAsString(), other.eventTypeAsString())
                && Objects.equals(forecastedBreachTime(), other.forecastedBreachTime())
                && hasGeofenceProperties() == other.hasGeofenceProperties()
                && Objects.equals(geofenceProperties(), other.geofenceProperties());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ForecastedEvent").add("EventId", eventId()).add("GeofenceId", geofenceId())
                .add("IsDeviceInGeofence", isDeviceInGeofence()).add("NearestDistance", nearestDistance())
                .add("EventType", eventTypeAsString()).add("ForecastedBreachTime", forecastedBreachTime())
                .add("GeofenceProperties", geofenceProperties() == null ? null : "*** Sensitive Data Redacted ***").build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "EventId":
            return Optional.ofNullable(clazz.cast(eventId()));
        case "GeofenceId":
            return Optional.ofNullable(clazz.cast(geofenceId()));
        case "IsDeviceInGeofence":
            return Optional.ofNullable(clazz.cast(isDeviceInGeofence()));
        case "NearestDistance":
            return Optional.ofNullable(clazz.cast(nearestDistance()));
        case "EventType":
            return Optional.ofNullable(clazz.cast(eventTypeAsString()));
        case "ForecastedBreachTime":
            return Optional.ofNullable(clazz.cast(forecastedBreachTime()));
        case "GeofenceProperties":
            return Optional.ofNullable(clazz.cast(geofenceProperties()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("EventId", EVENT_ID_FIELD);
        map.put("GeofenceId", GEOFENCE_ID_FIELD);
        map.put("IsDeviceInGeofence", IS_DEVICE_IN_GEOFENCE_FIELD);
        map.put("NearestDistance", NEAREST_DISTANCE_FIELD);
        map.put("EventType", EVENT_TYPE_FIELD);
        map.put("ForecastedBreachTime", FORECASTED_BREACH_TIME_FIELD);
        map.put("GeofenceProperties", GEOFENCE_PROPERTIES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ForecastedEvent, T> g) {
        return obj -> g.apply((ForecastedEvent) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ForecastedEvent> {
        /**
         * <p>
         * The forecasted event identifier.
         * </p>
         * 
         * @param eventId
         *        The forecasted event identifier.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventId(String eventId);

        /**
         * <p>
         * The geofence identifier pertaining to the forecasted event.
         * </p>
         * 
         * @param geofenceId
         *        The geofence identifier pertaining to the forecasted event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder geofenceId(String geofenceId);

        /**
         * <p>
         * Indicates if the device is located within the geofence.
         * </p>
         * 
         * @param isDeviceInGeofence
         *        Indicates if the device is located within the geofence.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isDeviceInGeofence(Boolean isDeviceInGeofence);

        /**
         * <p>
         * The closest distance from the device's position to the geofence.
         * </p>
         * 
         * @param nearestDistance
         *        The closest distance from the device's position to the geofence.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nearestDistance(Double nearestDistance);

        /**
         * <p>
         * The event type, forecasting three states for which a device can be in relative to a geofence:
         * </p>
         * <p>
         * <code>ENTER</code>: If a device is outside of a geofence, but would breach the fence if the device is moving
         * at its current speed within time horizon window.
         * </p>
         * <p>
         * <code>EXIT</code>: If a device is inside of a geofence, but would breach the fence if the device is moving at
         * its current speed within time horizon window.
         * </p>
         * <p>
         * <code>IDLE</code>: If a device is inside of a geofence, and the device is not moving.
         * </p>
         * 
         * @param eventType
         *        The event type, forecasting three states for which a device can be in relative to a geofence:</p>
         *        <p>
         *        <code>ENTER</code>: If a device is outside of a geofence, but would breach the fence if the device is
         *        moving at its current speed within time horizon window.
         *        </p>
         *        <p>
         *        <code>EXIT</code>: If a device is inside of a geofence, but would breach the fence if the device is
         *        moving at its current speed within time horizon window.
         *        </p>
         *        <p>
         *        <code>IDLE</code>: If a device is inside of a geofence, and the device is not moving.
         * @see ForecastedGeofenceEventType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ForecastedGeofenceEventType
         */
        Builder eventType(String eventType);

        /**
         * <p>
         * The event type, forecasting three states for which a device can be in relative to a geofence:
         * </p>
         * <p>
         * <code>ENTER</code>: If a device is outside of a geofence, but would breach the fence if the device is moving
         * at its current speed within time horizon window.
         * </p>
         * <p>
         * <code>EXIT</code>: If a device is inside of a geofence, but would breach the fence if the device is moving at
         * its current speed within time horizon window.
         * </p>
         * <p>
         * <code>IDLE</code>: If a device is inside of a geofence, and the device is not moving.
         * </p>
         * 
         * @param eventType
         *        The event type, forecasting three states for which a device can be in relative to a geofence:</p>
         *        <p>
         *        <code>ENTER</code>: If a device is outside of a geofence, but would breach the fence if the device is
         *        moving at its current speed within time horizon window.
         *        </p>
         *        <p>
         *        <code>EXIT</code>: If a device is inside of a geofence, but would breach the fence if the device is
         *        moving at its current speed within time horizon window.
         *        </p>
         *        <p>
         *        <code>IDLE</code>: If a device is inside of a geofence, and the device is not moving.
         * @see ForecastedGeofenceEventType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ForecastedGeofenceEventType
         */
        Builder eventType(ForecastedGeofenceEventType eventType);

        /**
         * <p>
         * The forecasted time the device will breach the geofence in <a
         * href="https://www.iso.org/iso-8601-date-and-time-format.html">ISO 8601</a> format:
         * <code>YYYY-MM-DDThh:mm:ss.sssZ</code>
         * </p>
         * 
         * @param forecastedBreachTime
         *        The forecasted time the device will breach the geofence in <a
         *        href="https://www.iso.org/iso-8601-date-and-time-format.html">ISO 8601</a> format:
         *        <code>YYYY-MM-DDThh:mm:ss.sssZ</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder forecastedBreachTime(Instant forecastedBreachTime);

        /**
         * <p>
         * The geofence properties.
         * </p>
         * 
         * @param geofenceProperties
         *        The geofence properties.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder geofenceProperties(Map<String, String> geofenceProperties);
    }

    static final class BuilderImpl implements Builder {
        private String eventId;

        private String geofenceId;

        private Boolean isDeviceInGeofence;

        private Double nearestDistance;

        private String eventType;

        private Instant forecastedBreachTime;

        private Map<String, String> geofenceProperties = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ForecastedEvent model) {
            eventId(model.eventId);
            geofenceId(model.geofenceId);
            isDeviceInGeofence(model.isDeviceInGeofence);
            nearestDistance(model.nearestDistance);
            eventType(model.eventType);
            forecastedBreachTime(model.forecastedBreachTime);
            geofenceProperties(model.geofenceProperties);
        }

        public final String getEventId() {
            return eventId;
        }

        public final void setEventId(String eventId) {
            this.eventId = eventId;
        }

        @Override
        public final Builder eventId(String eventId) {
            this.eventId = eventId;
            return this;
        }

        public final String getGeofenceId() {
            return geofenceId;
        }

        public final void setGeofenceId(String geofenceId) {
            this.geofenceId = geofenceId;
        }

        @Override
        public final Builder geofenceId(String geofenceId) {
            this.geofenceId = geofenceId;
            return this;
        }

        public final Boolean getIsDeviceInGeofence() {
            return isDeviceInGeofence;
        }

        public final void setIsDeviceInGeofence(Boolean isDeviceInGeofence) {
            this.isDeviceInGeofence = isDeviceInGeofence;
        }

        @Override
        public final Builder isDeviceInGeofence(Boolean isDeviceInGeofence) {
            this.isDeviceInGeofence = isDeviceInGeofence;
            return this;
        }

        public final Double getNearestDistance() {
            return nearestDistance;
        }

        public final void setNearestDistance(Double nearestDistance) {
            this.nearestDistance = nearestDistance;
        }

        @Override
        public final Builder nearestDistance(Double nearestDistance) {
            this.nearestDistance = nearestDistance;
            return this;
        }

        public final String getEventType() {
            return eventType;
        }

        public final void setEventType(String eventType) {
            this.eventType = eventType;
        }

        @Override
        public final Builder eventType(String eventType) {
            this.eventType = eventType;
            return this;
        }

        @Override
        public final Builder eventType(ForecastedGeofenceEventType eventType) {
            this.eventType(eventType == null ? null : eventType.toString());
            return this;
        }

        public final Instant getForecastedBreachTime() {
            return forecastedBreachTime;
        }

        public final void setForecastedBreachTime(Instant forecastedBreachTime) {
            this.forecastedBreachTime = forecastedBreachTime;
        }

        @Override
        public final Builder forecastedBreachTime(Instant forecastedBreachTime) {
            this.forecastedBreachTime = forecastedBreachTime;
            return this;
        }

        public final Map<String, String> getGeofenceProperties() {
            if (geofenceProperties instanceof SdkAutoConstructMap) {
                return null;
            }
            return geofenceProperties;
        }

        public final void setGeofenceProperties(Map<String, String> geofenceProperties) {
            this.geofenceProperties = PropertyMapCopier.copy(geofenceProperties);
        }

        @Override
        public final Builder geofenceProperties(Map<String, String> geofenceProperties) {
            this.geofenceProperties = PropertyMapCopier.copy(geofenceProperties);
            return this;
        }

        @Override
        public ForecastedEvent build() {
            return new ForecastedEvent(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
