/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.location.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the calculated route's details for each path between a pair of positions. The number of legs returned
 * corresponds to one fewer than the total number of positions in the request.
 * </p>
 * <p>
 * For example, a route with a departure position and destination position returns one leg with the positions <a
 * href="https://docs.aws.amazon.com/location/latest/developerguide/snap-to-nearby-road.html">snapped to a nearby
 * road</a>:
 * </p>
 * <ul>
 * <li>
 * <p>
 * The <code>StartPosition</code> is the departure position.
 * </p>
 * </li>
 * <li>
 * <p>
 * The <code>EndPosition</code> is the destination position.
 * </p>
 * </li>
 * </ul>
 * <p>
 * A route with a waypoint between the departure and destination position returns two legs with the positions snapped to
 * a nearby road:
 * </p>
 * <ul>
 * <li>
 * <p>
 * Leg 1: The <code>StartPosition</code> is the departure position . The <code>EndPosition</code> is the waypoint
 * positon.
 * </p>
 * </li>
 * <li>
 * <p>
 * Leg 2: The <code>StartPosition</code> is the waypoint position. The <code>EndPosition</code> is the destination
 * position.
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Leg implements SdkPojo, Serializable, ToCopyableBuilder<Leg.Builder, Leg> {
    private static final SdkField<List<Double>> START_POSITION_FIELD = SdkField
            .<List<Double>> builder(MarshallingType.LIST)
            .memberName("StartPosition")
            .getter(getter(Leg::startPosition))
            .setter(setter(Builder::startPosition))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartPosition").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Double> builder(MarshallingType.DOUBLE)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Double>> END_POSITION_FIELD = SdkField
            .<List<Double>> builder(MarshallingType.LIST)
            .memberName("EndPosition")
            .getter(getter(Leg::endPosition))
            .setter(setter(Builder::endPosition))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndPosition").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Double> builder(MarshallingType.DOUBLE)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Double> DISTANCE_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("Distance").getter(getter(Leg::distance)).setter(setter(Builder::distance))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Distance").build()).build();

    private static final SdkField<Double> DURATION_SECONDS_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("DurationSeconds").getter(getter(Leg::durationSeconds)).setter(setter(Builder::durationSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DurationSeconds").build()).build();

    private static final SdkField<LegGeometry> GEOMETRY_FIELD = SdkField.<LegGeometry> builder(MarshallingType.SDK_POJO)
            .memberName("Geometry").getter(getter(Leg::geometry)).setter(setter(Builder::geometry))
            .constructor(LegGeometry::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Geometry").build()).build();

    private static final SdkField<List<Step>> STEPS_FIELD = SdkField
            .<List<Step>> builder(MarshallingType.LIST)
            .memberName("Steps")
            .getter(getter(Leg::steps))
            .setter(setter(Builder::steps))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Steps").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Step> builder(MarshallingType.SDK_POJO)
                                            .constructor(Step::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(START_POSITION_FIELD,
            END_POSITION_FIELD, DISTANCE_FIELD, DURATION_SECONDS_FIELD, GEOMETRY_FIELD, STEPS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final List<Double> startPosition;

    private final List<Double> endPosition;

    private final Double distance;

    private final Double durationSeconds;

    private final LegGeometry geometry;

    private final List<Step> steps;

    private Leg(BuilderImpl builder) {
        this.startPosition = builder.startPosition;
        this.endPosition = builder.endPosition;
        this.distance = builder.distance;
        this.durationSeconds = builder.durationSeconds;
        this.geometry = builder.geometry;
        this.steps = builder.steps;
    }

    /**
     * For responses, this returns true if the service returned a value for the StartPosition property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasStartPosition() {
        return startPosition != null && !(startPosition instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The starting position of the leg. Follows the format <code>[longitude,latitude]</code>.
     * </p>
     * <note>
     * <p>
     * If the <code>StartPosition</code> isn't located on a road, it's <a
     * href="https://docs.aws.amazon.com/location/latest/developerguide/snap-to-nearby-road.html">snapped to a nearby
     * road</a>.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasStartPosition} method.
     * </p>
     * 
     * @return The starting position of the leg. Follows the format <code>[longitude,latitude]</code>.</p> <note>
     *         <p>
     *         If the <code>StartPosition</code> isn't located on a road, it's <a
     *         href="https://docs.aws.amazon.com/location/latest/developerguide/snap-to-nearby-road.html">snapped to a
     *         nearby road</a>.
     *         </p>
     */
    public final List<Double> startPosition() {
        return startPosition;
    }

    /**
     * For responses, this returns true if the service returned a value for the EndPosition property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEndPosition() {
        return endPosition != null && !(endPosition instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The terminating position of the leg. Follows the format <code>[longitude,latitude]</code>.
     * </p>
     * <note>
     * <p>
     * If the <code>EndPosition</code> isn't located on a road, it's <a
     * href="https://docs.aws.amazon.com/location/latest/developerguide/nap-to-nearby-road.html">snapped to a nearby
     * road</a>.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEndPosition} method.
     * </p>
     * 
     * @return The terminating position of the leg. Follows the format <code>[longitude,latitude]</code>.</p> <note>
     *         <p>
     *         If the <code>EndPosition</code> isn't located on a road, it's <a
     *         href="https://docs.aws.amazon.com/location/latest/developerguide/nap-to-nearby-road.html">snapped to a
     *         nearby road</a>.
     *         </p>
     */
    public final List<Double> endPosition() {
        return endPosition;
    }

    /**
     * <p>
     * The distance between the leg's <code>StartPosition</code> and <code>EndPosition</code> along a calculated route.
     * </p>
     * <ul>
     * <li>
     * <p>
     * The default measurement is <code>Kilometers</code> unless the request specifies a <code>DistanceUnit</code> of
     * <code>Miles</code>.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The distance between the leg's <code>StartPosition</code> and <code>EndPosition</code> along a calculated
     *         route. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The default measurement is <code>Kilometers</code> unless the request specifies a
     *         <code>DistanceUnit</code> of <code>Miles</code>.
     *         </p>
     *         </li>
     */
    public final Double distance() {
        return distance;
    }

    /**
     * <p>
     * The estimated travel time between the leg's <code>StartPosition</code> and <code>EndPosition</code>. The travel
     * mode and departure time that you specify in the request determines the calculated time.
     * </p>
     * 
     * @return The estimated travel time between the leg's <code>StartPosition</code> and <code>EndPosition</code>. The
     *         travel mode and departure time that you specify in the request determines the calculated time.
     */
    public final Double durationSeconds() {
        return durationSeconds;
    }

    /**
     * <p>
     * Contains the calculated route's path as a linestring geometry.
     * </p>
     * 
     * @return Contains the calculated route's path as a linestring geometry.
     */
    public final LegGeometry geometry() {
        return geometry;
    }

    /**
     * For responses, this returns true if the service returned a value for the Steps property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasSteps() {
        return steps != null && !(steps instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Contains a list of steps, which represent subsections of a leg. Each step provides instructions for how to move
     * to the next step in the leg such as the step's start position, end position, travel distance, travel duration,
     * and geometry offset.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSteps} method.
     * </p>
     * 
     * @return Contains a list of steps, which represent subsections of a leg. Each step provides instructions for how
     *         to move to the next step in the leg such as the step's start position, end position, travel distance,
     *         travel duration, and geometry offset.
     */
    public final List<Step> steps() {
        return steps;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasStartPosition() ? startPosition() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasEndPosition() ? endPosition() : null);
        hashCode = 31 * hashCode + Objects.hashCode(distance());
        hashCode = 31 * hashCode + Objects.hashCode(durationSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(geometry());
        hashCode = 31 * hashCode + Objects.hashCode(hasSteps() ? steps() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Leg)) {
            return false;
        }
        Leg other = (Leg) obj;
        return hasStartPosition() == other.hasStartPosition() && Objects.equals(startPosition(), other.startPosition())
                && hasEndPosition() == other.hasEndPosition() && Objects.equals(endPosition(), other.endPosition())
                && Objects.equals(distance(), other.distance()) && Objects.equals(durationSeconds(), other.durationSeconds())
                && Objects.equals(geometry(), other.geometry()) && hasSteps() == other.hasSteps()
                && Objects.equals(steps(), other.steps());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Leg").add("StartPosition", startPosition() == null ? null : "*** Sensitive Data Redacted ***")
                .add("EndPosition", endPosition() == null ? null : "*** Sensitive Data Redacted ***").add("Distance", distance())
                .add("DurationSeconds", durationSeconds()).add("Geometry", geometry()).add("Steps", hasSteps() ? steps() : null)
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StartPosition":
            return Optional.ofNullable(clazz.cast(startPosition()));
        case "EndPosition":
            return Optional.ofNullable(clazz.cast(endPosition()));
        case "Distance":
            return Optional.ofNullable(clazz.cast(distance()));
        case "DurationSeconds":
            return Optional.ofNullable(clazz.cast(durationSeconds()));
        case "Geometry":
            return Optional.ofNullable(clazz.cast(geometry()));
        case "Steps":
            return Optional.ofNullable(clazz.cast(steps()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("StartPosition", START_POSITION_FIELD);
        map.put("EndPosition", END_POSITION_FIELD);
        map.put("Distance", DISTANCE_FIELD);
        map.put("DurationSeconds", DURATION_SECONDS_FIELD);
        map.put("Geometry", GEOMETRY_FIELD);
        map.put("Steps", STEPS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Leg, T> g) {
        return obj -> g.apply((Leg) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Leg> {
        /**
         * <p>
         * The starting position of the leg. Follows the format <code>[longitude,latitude]</code>.
         * </p>
         * <note>
         * <p>
         * If the <code>StartPosition</code> isn't located on a road, it's <a
         * href="https://docs.aws.amazon.com/location/latest/developerguide/snap-to-nearby-road.html">snapped to a
         * nearby road</a>.
         * </p>
         * </note>
         * 
         * @param startPosition
         *        The starting position of the leg. Follows the format <code>[longitude,latitude]</code>.</p> <note>
         *        <p>
         *        If the <code>StartPosition</code> isn't located on a road, it's <a
         *        href="https://docs.aws.amazon.com/location/latest/developerguide/snap-to-nearby-road.html">snapped to
         *        a nearby road</a>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startPosition(Collection<Double> startPosition);

        /**
         * <p>
         * The starting position of the leg. Follows the format <code>[longitude,latitude]</code>.
         * </p>
         * <note>
         * <p>
         * If the <code>StartPosition</code> isn't located on a road, it's <a
         * href="https://docs.aws.amazon.com/location/latest/developerguide/snap-to-nearby-road.html">snapped to a
         * nearby road</a>.
         * </p>
         * </note>
         * 
         * @param startPosition
         *        The starting position of the leg. Follows the format <code>[longitude,latitude]</code>.</p> <note>
         *        <p>
         *        If the <code>StartPosition</code> isn't located on a road, it's <a
         *        href="https://docs.aws.amazon.com/location/latest/developerguide/snap-to-nearby-road.html">snapped to
         *        a nearby road</a>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startPosition(Double... startPosition);

        /**
         * <p>
         * The terminating position of the leg. Follows the format <code>[longitude,latitude]</code>.
         * </p>
         * <note>
         * <p>
         * If the <code>EndPosition</code> isn't located on a road, it's <a
         * href="https://docs.aws.amazon.com/location/latest/developerguide/nap-to-nearby-road.html">snapped to a nearby
         * road</a>.
         * </p>
         * </note>
         * 
         * @param endPosition
         *        The terminating position of the leg. Follows the format <code>[longitude,latitude]</code>.</p> <note>
         *        <p>
         *        If the <code>EndPosition</code> isn't located on a road, it's <a
         *        href="https://docs.aws.amazon.com/location/latest/developerguide/nap-to-nearby-road.html">snapped to a
         *        nearby road</a>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endPosition(Collection<Double> endPosition);

        /**
         * <p>
         * The terminating position of the leg. Follows the format <code>[longitude,latitude]</code>.
         * </p>
         * <note>
         * <p>
         * If the <code>EndPosition</code> isn't located on a road, it's <a
         * href="https://docs.aws.amazon.com/location/latest/developerguide/nap-to-nearby-road.html">snapped to a nearby
         * road</a>.
         * </p>
         * </note>
         * 
         * @param endPosition
         *        The terminating position of the leg. Follows the format <code>[longitude,latitude]</code>.</p> <note>
         *        <p>
         *        If the <code>EndPosition</code> isn't located on a road, it's <a
         *        href="https://docs.aws.amazon.com/location/latest/developerguide/nap-to-nearby-road.html">snapped to a
         *        nearby road</a>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endPosition(Double... endPosition);

        /**
         * <p>
         * The distance between the leg's <code>StartPosition</code> and <code>EndPosition</code> along a calculated
         * route.
         * </p>
         * <ul>
         * <li>
         * <p>
         * The default measurement is <code>Kilometers</code> unless the request specifies a <code>DistanceUnit</code>
         * of <code>Miles</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param distance
         *        The distance between the leg's <code>StartPosition</code> and <code>EndPosition</code> along a
         *        calculated route. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        The default measurement is <code>Kilometers</code> unless the request specifies a
         *        <code>DistanceUnit</code> of <code>Miles</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder distance(Double distance);

        /**
         * <p>
         * The estimated travel time between the leg's <code>StartPosition</code> and <code>EndPosition</code>. The
         * travel mode and departure time that you specify in the request determines the calculated time.
         * </p>
         * 
         * @param durationSeconds
         *        The estimated travel time between the leg's <code>StartPosition</code> and <code>EndPosition</code>.
         *        The travel mode and departure time that you specify in the request determines the calculated time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder durationSeconds(Double durationSeconds);

        /**
         * <p>
         * Contains the calculated route's path as a linestring geometry.
         * </p>
         * 
         * @param geometry
         *        Contains the calculated route's path as a linestring geometry.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder geometry(LegGeometry geometry);

        /**
         * <p>
         * Contains the calculated route's path as a linestring geometry.
         * </p>
         * This is a convenience method that creates an instance of the {@link LegGeometry.Builder} avoiding the need to
         * create one manually via {@link LegGeometry#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link LegGeometry.Builder#build()} is called immediately and its result
         * is passed to {@link #geometry(LegGeometry)}.
         * 
         * @param geometry
         *        a consumer that will call methods on {@link LegGeometry.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #geometry(LegGeometry)
         */
        default Builder geometry(Consumer<LegGeometry.Builder> geometry) {
            return geometry(LegGeometry.builder().applyMutation(geometry).build());
        }

        /**
         * <p>
         * Contains a list of steps, which represent subsections of a leg. Each step provides instructions for how to
         * move to the next step in the leg such as the step's start position, end position, travel distance, travel
         * duration, and geometry offset.
         * </p>
         * 
         * @param steps
         *        Contains a list of steps, which represent subsections of a leg. Each step provides instructions for
         *        how to move to the next step in the leg such as the step's start position, end position, travel
         *        distance, travel duration, and geometry offset.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder steps(Collection<Step> steps);

        /**
         * <p>
         * Contains a list of steps, which represent subsections of a leg. Each step provides instructions for how to
         * move to the next step in the leg such as the step's start position, end position, travel distance, travel
         * duration, and geometry offset.
         * </p>
         * 
         * @param steps
         *        Contains a list of steps, which represent subsections of a leg. Each step provides instructions for
         *        how to move to the next step in the leg such as the step's start position, end position, travel
         *        distance, travel duration, and geometry offset.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder steps(Step... steps);

        /**
         * <p>
         * Contains a list of steps, which represent subsections of a leg. Each step provides instructions for how to
         * move to the next step in the leg such as the step's start position, end position, travel distance, travel
         * duration, and geometry offset.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.location.model.Step.Builder} avoiding the need to create one manually
         * via {@link software.amazon.awssdk.services.location.model.Step#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.location.model.Step.Builder#build()} is called immediately and its
         * result is passed to {@link #steps(List<Step>)}.
         * 
         * @param steps
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.location.model.Step.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #steps(java.util.Collection<Step>)
         */
        Builder steps(Consumer<Step.Builder>... steps);
    }

    static final class BuilderImpl implements Builder {
        private List<Double> startPosition = DefaultSdkAutoConstructList.getInstance();

        private List<Double> endPosition = DefaultSdkAutoConstructList.getInstance();

        private Double distance;

        private Double durationSeconds;

        private LegGeometry geometry;

        private List<Step> steps = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Leg model) {
            startPosition(model.startPosition);
            endPosition(model.endPosition);
            distance(model.distance);
            durationSeconds(model.durationSeconds);
            geometry(model.geometry);
            steps(model.steps);
        }

        public final Collection<Double> getStartPosition() {
            if (startPosition instanceof SdkAutoConstructList) {
                return null;
            }
            return startPosition;
        }

        public final void setStartPosition(Collection<Double> startPosition) {
            this.startPosition = PositionCopier.copy(startPosition);
        }

        @Override
        public final Builder startPosition(Collection<Double> startPosition) {
            this.startPosition = PositionCopier.copy(startPosition);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder startPosition(Double... startPosition) {
            startPosition(Arrays.asList(startPosition));
            return this;
        }

        public final Collection<Double> getEndPosition() {
            if (endPosition instanceof SdkAutoConstructList) {
                return null;
            }
            return endPosition;
        }

        public final void setEndPosition(Collection<Double> endPosition) {
            this.endPosition = PositionCopier.copy(endPosition);
        }

        @Override
        public final Builder endPosition(Collection<Double> endPosition) {
            this.endPosition = PositionCopier.copy(endPosition);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder endPosition(Double... endPosition) {
            endPosition(Arrays.asList(endPosition));
            return this;
        }

        public final Double getDistance() {
            return distance;
        }

        public final void setDistance(Double distance) {
            this.distance = distance;
        }

        @Override
        public final Builder distance(Double distance) {
            this.distance = distance;
            return this;
        }

        public final Double getDurationSeconds() {
            return durationSeconds;
        }

        public final void setDurationSeconds(Double durationSeconds) {
            this.durationSeconds = durationSeconds;
        }

        @Override
        public final Builder durationSeconds(Double durationSeconds) {
            this.durationSeconds = durationSeconds;
            return this;
        }

        public final LegGeometry.Builder getGeometry() {
            return geometry != null ? geometry.toBuilder() : null;
        }

        public final void setGeometry(LegGeometry.BuilderImpl geometry) {
            this.geometry = geometry != null ? geometry.build() : null;
        }

        @Override
        public final Builder geometry(LegGeometry geometry) {
            this.geometry = geometry;
            return this;
        }

        public final List<Step.Builder> getSteps() {
            List<Step.Builder> result = StepListCopier.copyToBuilder(this.steps);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSteps(Collection<Step.BuilderImpl> steps) {
            this.steps = StepListCopier.copyFromBuilder(steps);
        }

        @Override
        public final Builder steps(Collection<Step> steps) {
            this.steps = StepListCopier.copy(steps);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder steps(Step... steps) {
            steps(Arrays.asList(steps));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder steps(Consumer<Step.Builder>... steps) {
            steps(Stream.of(steps).map(c -> Step.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public Leg build() {
            return new Leg(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
