/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.location.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies the map tile style selected from an available provider.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MapConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<MapConfiguration.Builder, MapConfiguration> {
    private static final SdkField<String> STYLE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Style")
            .getter(getter(MapConfiguration::style)).setter(setter(Builder::style))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Style").build()).build();

    private static final SdkField<String> POLITICAL_VIEW_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PoliticalView").getter(getter(MapConfiguration::politicalView)).setter(setter(Builder::politicalView))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PoliticalView").build()).build();

    private static final SdkField<List<String>> CUSTOM_LAYERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("CustomLayers")
            .getter(getter(MapConfiguration::customLayers))
            .setter(setter(Builder::customLayers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomLayers").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STYLE_FIELD,
            POLITICAL_VIEW_FIELD, CUSTOM_LAYERS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String style;

    private final String politicalView;

    private final List<String> customLayers;

    private MapConfiguration(BuilderImpl builder) {
        this.style = builder.style;
        this.politicalView = builder.politicalView;
        this.customLayers = builder.customLayers;
    }

    /**
     * <p>
     * Specifies the map style selected from an available data provider.
     * </p>
     * <p>
     * Valid <a href="https://docs.aws.amazon.com/location/previous/developerguide/esri.html">Esri map styles</a>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>VectorEsriDarkGrayCanvas</code> – The Esri Dark Gray Canvas map style. A vector basemap with a dark gray,
     * neutral background with minimal colors, labels, and features that's designed to draw attention to your thematic
     * content.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RasterEsriImagery</code> – The Esri Imagery map style. A raster basemap that provides one meter or better
     * satellite and aerial imagery in many parts of the world and lower resolution satellite imagery worldwide.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>VectorEsriLightGrayCanvas</code> – The Esri Light Gray Canvas map style, which provides a detailed vector
     * basemap with a light gray, neutral background style with minimal colors, labels, and features that's designed to
     * draw attention to your thematic content.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>VectorEsriTopographic</code> – The Esri Light map style, which provides a detailed vector basemap with a
     * classic Esri map style.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>VectorEsriStreets</code> – The Esri Street Map style, which provides a detailed vector basemap for the
     * world symbolized with a classic Esri street map style. The vector tile layer is similar in content and style to
     * the World Street Map raster map.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>VectorEsriNavigation</code> – The Esri Navigation map style, which provides a detailed basemap for the
     * world symbolized with a custom navigation map style that's designed for use during the day in mobile devices.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Valid <a href="https://docs.aws.amazon.com/location/previous/developerguide/HERE.html">HERE Technologies map
     * styles</a>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>VectorHereContrast</code> – The HERE Contrast (Berlin) map style is a high contrast detailed base map of
     * the world that blends 3D and 2D rendering.
     * </p>
     * <note>
     * <p>
     * The <code>VectorHereContrast</code> style has been renamed from <code>VectorHereBerlin</code>.
     * <code>VectorHereBerlin</code> has been deprecated, but will continue to work in applications that use it.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * <code>VectorHereExplore</code> – A default HERE map style containing a neutral, global map and its features
     * including roads, buildings, landmarks, and water features. It also now includes a fully designed map of Japan.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>VectorHereExploreTruck</code> – A global map containing truck restrictions and attributes (e.g. width /
     * height / HAZMAT) symbolized with highlighted segments and icons on top of HERE Explore to support use cases
     * within transport and logistics.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RasterHereExploreSatellite</code> – A global map containing high resolution satellite imagery.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>HybridHereExploreSatellite</code> – A global map displaying the road network, street names, and city labels
     * over satellite imagery. This style will automatically retrieve both raster and vector tiles, and your charges
     * will be based on total tiles retrieved.
     * </p>
     * <note>
     * <p>
     * Hybrid styles use both vector and raster tiles when rendering the map that you see. This means that more tiles
     * are retrieved than when using either vector or raster tiles alone. Your charges will include all tiles retrieved.
     * </p>
     * </note></li>
     * </ul>
     * <p>
     * Valid <a href="https://docs.aws.amazon.com/location/previous/developerguide/grab.html">GrabMaps map styles</a>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>VectorGrabStandardLight</code> – The Grab Standard Light map style provides a basemap with detailed land
     * use coloring, area names, roads, landmarks, and points of interest covering Southeast Asia.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>VectorGrabStandardDark</code> – The Grab Standard Dark map style provides a dark variation of the standard
     * basemap covering Southeast Asia.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * Grab provides maps only for countries in Southeast Asia, and is only available in the Asia Pacific (Singapore)
     * Region (<code>ap-southeast-1</code>). For more information, see <a
     * href="https://docs.aws.amazon.com/location/previous/developerguide/grab.html#grab-coverage-area">GrabMaps
     * countries and area covered</a>.
     * </p>
     * </note>
     * <p>
     * Valid <a href="https://docs.aws.amazon.com/location/previous/developerguide/open-data.html">Open Data map
     * styles</a>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>VectorOpenDataStandardLight</code> – The Open Data Standard Light map style provides a detailed basemap for
     * the world suitable for website and mobile application use. The map includes highways major roads, minor roads,
     * railways, water features, cities, parks, landmarks, building footprints, and administrative boundaries.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>VectorOpenDataStandardDark</code> – Open Data Standard Dark is a dark-themed map style that provides a
     * detailed basemap for the world suitable for website and mobile application use. The map includes highways major
     * roads, minor roads, railways, water features, cities, parks, landmarks, building footprints, and administrative
     * boundaries.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>VectorOpenDataVisualizationLight</code> – The Open Data Visualization Light map style is a light-themed
     * style with muted colors and fewer features that aids in understanding overlaid data.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>VectorOpenDataVisualizationDark</code> – The Open Data Visualization Dark map style is a dark-themed style
     * with muted colors and fewer features that aids in understanding overlaid data.
     * </p>
     * </li>
     * </ul>
     * 
     * @return Specifies the map style selected from an available data provider.</p>
     *         <p>
     *         Valid <a href="https://docs.aws.amazon.com/location/previous/developerguide/esri.html">Esri map
     *         styles</a>:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>VectorEsriDarkGrayCanvas</code> – The Esri Dark Gray Canvas map style. A vector basemap with a dark
     *         gray, neutral background with minimal colors, labels, and features that's designed to draw attention to
     *         your thematic content.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RasterEsriImagery</code> – The Esri Imagery map style. A raster basemap that provides one meter or
     *         better satellite and aerial imagery in many parts of the world and lower resolution satellite imagery
     *         worldwide.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>VectorEsriLightGrayCanvas</code> – The Esri Light Gray Canvas map style, which provides a detailed
     *         vector basemap with a light gray, neutral background style with minimal colors, labels, and features
     *         that's designed to draw attention to your thematic content.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>VectorEsriTopographic</code> – The Esri Light map style, which provides a detailed vector basemap
     *         with a classic Esri map style.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>VectorEsriStreets</code> – The Esri Street Map style, which provides a detailed vector basemap for
     *         the world symbolized with a classic Esri street map style. The vector tile layer is similar in content
     *         and style to the World Street Map raster map.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>VectorEsriNavigation</code> – The Esri Navigation map style, which provides a detailed basemap for
     *         the world symbolized with a custom navigation map style that's designed for use during the day in mobile
     *         devices.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Valid <a href="https://docs.aws.amazon.com/location/previous/developerguide/HERE.html">HERE Technologies
     *         map styles</a>:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>VectorHereContrast</code> – The HERE Contrast (Berlin) map style is a high contrast detailed base
     *         map of the world that blends 3D and 2D rendering.
     *         </p>
     *         <note>
     *         <p>
     *         The <code>VectorHereContrast</code> style has been renamed from <code>VectorHereBerlin</code>.
     *         <code>VectorHereBerlin</code> has been deprecated, but will continue to work in applications that use it.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         <code>VectorHereExplore</code> – A default HERE map style containing a neutral, global map and its
     *         features including roads, buildings, landmarks, and water features. It also now includes a fully designed
     *         map of Japan.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>VectorHereExploreTruck</code> – A global map containing truck restrictions and attributes (e.g.
     *         width / height / HAZMAT) symbolized with highlighted segments and icons on top of HERE Explore to support
     *         use cases within transport and logistics.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RasterHereExploreSatellite</code> – A global map containing high resolution satellite imagery.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>HybridHereExploreSatellite</code> – A global map displaying the road network, street names, and
     *         city labels over satellite imagery. This style will automatically retrieve both raster and vector tiles,
     *         and your charges will be based on total tiles retrieved.
     *         </p>
     *         <note>
     *         <p>
     *         Hybrid styles use both vector and raster tiles when rendering the map that you see. This means that more
     *         tiles are retrieved than when using either vector or raster tiles alone. Your charges will include all
     *         tiles retrieved.
     *         </p>
     *         </note></li>
     *         </ul>
     *         <p>
     *         Valid <a href="https://docs.aws.amazon.com/location/previous/developerguide/grab.html">GrabMaps map
     *         styles</a>:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>VectorGrabStandardLight</code> – The Grab Standard Light map style provides a basemap with detailed
     *         land use coloring, area names, roads, landmarks, and points of interest covering Southeast Asia.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>VectorGrabStandardDark</code> – The Grab Standard Dark map style provides a dark variation of the
     *         standard basemap covering Southeast Asia.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         Grab provides maps only for countries in Southeast Asia, and is only available in the Asia Pacific
     *         (Singapore) Region (<code>ap-southeast-1</code>). For more information, see <a
     *         href="https://docs.aws.amazon.com/location/previous/developerguide/grab.html#grab-coverage-area">GrabMaps
     *         countries and area covered</a>.
     *         </p>
     *         </note>
     *         <p>
     *         Valid <a href="https://docs.aws.amazon.com/location/previous/developerguide/open-data.html">Open Data map
     *         styles</a>:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>VectorOpenDataStandardLight</code> – The Open Data Standard Light map style provides a detailed
     *         basemap for the world suitable for website and mobile application use. The map includes highways major
     *         roads, minor roads, railways, water features, cities, parks, landmarks, building footprints, and
     *         administrative boundaries.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>VectorOpenDataStandardDark</code> – Open Data Standard Dark is a dark-themed map style that
     *         provides a detailed basemap for the world suitable for website and mobile application use. The map
     *         includes highways major roads, minor roads, railways, water features, cities, parks, landmarks, building
     *         footprints, and administrative boundaries.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>VectorOpenDataVisualizationLight</code> – The Open Data Visualization Light map style is a
     *         light-themed style with muted colors and fewer features that aids in understanding overlaid data.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>VectorOpenDataVisualizationDark</code> – The Open Data Visualization Dark map style is a
     *         dark-themed style with muted colors and fewer features that aids in understanding overlaid data.
     *         </p>
     *         </li>
     */
    public final String style() {
        return style;
    }

    /**
     * <p>
     * Specifies the political view for the style. Leave unset to not use a political view, or, for styles that support
     * specific political views, you can choose a view, such as <code>IND</code> for the Indian view.
     * </p>
     * <p>
     * Default is unset.
     * </p>
     * <note>
     * <p>
     * Not all map resources or styles support political view styles. See <a
     * href="https://docs.aws.amazon.com/location/previous/developerguide/map-concepts.html#political-views">Political
     * views</a> for more information.
     * </p>
     * </note>
     * 
     * @return Specifies the political view for the style. Leave unset to not use a political view, or, for styles that
     *         support specific political views, you can choose a view, such as <code>IND</code> for the Indian
     *         view.</p>
     *         <p>
     *         Default is unset.
     *         </p>
     *         <note>
     *         <p>
     *         Not all map resources or styles support political view styles. See <a
     *         href="https://docs.aws.amazon.com/location/previous/developerguide/map-concepts.html#political-views"
     *         >Political views</a> for more information.
     *         </p>
     */
    public final String politicalView() {
        return politicalView;
    }

    /**
     * For responses, this returns true if the service returned a value for the CustomLayers property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasCustomLayers() {
        return customLayers != null && !(customLayers instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies the custom layers for the style. Leave unset to not enable any custom layer, or, for styles that
     * support custom layers, you can enable layer(s), such as POI layer for the VectorEsriNavigation style. Default is
     * <code>unset</code>.
     * </p>
     * <note>
     * <p>
     * Not all map resources or styles support custom layers. See Custom Layers for more information.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCustomLayers} method.
     * </p>
     * 
     * @return Specifies the custom layers for the style. Leave unset to not enable any custom layer, or, for styles
     *         that support custom layers, you can enable layer(s), such as POI layer for the VectorEsriNavigation
     *         style. Default is <code>unset</code>.</p> <note>
     *         <p>
     *         Not all map resources or styles support custom layers. See Custom Layers for more information.
     *         </p>
     */
    public final List<String> customLayers() {
        return customLayers;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(style());
        hashCode = 31 * hashCode + Objects.hashCode(politicalView());
        hashCode = 31 * hashCode + Objects.hashCode(hasCustomLayers() ? customLayers() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MapConfiguration)) {
            return false;
        }
        MapConfiguration other = (MapConfiguration) obj;
        return Objects.equals(style(), other.style()) && Objects.equals(politicalView(), other.politicalView())
                && hasCustomLayers() == other.hasCustomLayers() && Objects.equals(customLayers(), other.customLayers());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("MapConfiguration").add("Style", style())
                .add("PoliticalView", politicalView() == null ? null : "*** Sensitive Data Redacted ***")
                .add("CustomLayers", hasCustomLayers() ? customLayers() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Style":
            return Optional.ofNullable(clazz.cast(style()));
        case "PoliticalView":
            return Optional.ofNullable(clazz.cast(politicalView()));
        case "CustomLayers":
            return Optional.ofNullable(clazz.cast(customLayers()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Style", STYLE_FIELD);
        map.put("PoliticalView", POLITICAL_VIEW_FIELD);
        map.put("CustomLayers", CUSTOM_LAYERS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<MapConfiguration, T> g) {
        return obj -> g.apply((MapConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MapConfiguration> {
        /**
         * <p>
         * Specifies the map style selected from an available data provider.
         * </p>
         * <p>
         * Valid <a href="https://docs.aws.amazon.com/location/previous/developerguide/esri.html">Esri map styles</a>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>VectorEsriDarkGrayCanvas</code> – The Esri Dark Gray Canvas map style. A vector basemap with a dark
         * gray, neutral background with minimal colors, labels, and features that's designed to draw attention to your
         * thematic content.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RasterEsriImagery</code> – The Esri Imagery map style. A raster basemap that provides one meter or
         * better satellite and aerial imagery in many parts of the world and lower resolution satellite imagery
         * worldwide.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>VectorEsriLightGrayCanvas</code> – The Esri Light Gray Canvas map style, which provides a detailed
         * vector basemap with a light gray, neutral background style with minimal colors, labels, and features that's
         * designed to draw attention to your thematic content.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>VectorEsriTopographic</code> – The Esri Light map style, which provides a detailed vector basemap with
         * a classic Esri map style.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>VectorEsriStreets</code> – The Esri Street Map style, which provides a detailed vector basemap for the
         * world symbolized with a classic Esri street map style. The vector tile layer is similar in content and style
         * to the World Street Map raster map.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>VectorEsriNavigation</code> – The Esri Navigation map style, which provides a detailed basemap for the
         * world symbolized with a custom navigation map style that's designed for use during the day in mobile devices.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Valid <a href="https://docs.aws.amazon.com/location/previous/developerguide/HERE.html">HERE Technologies map
         * styles</a>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>VectorHereContrast</code> – The HERE Contrast (Berlin) map style is a high contrast detailed base map
         * of the world that blends 3D and 2D rendering.
         * </p>
         * <note>
         * <p>
         * The <code>VectorHereContrast</code> style has been renamed from <code>VectorHereBerlin</code>.
         * <code>VectorHereBerlin</code> has been deprecated, but will continue to work in applications that use it.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * <code>VectorHereExplore</code> – A default HERE map style containing a neutral, global map and its features
         * including roads, buildings, landmarks, and water features. It also now includes a fully designed map of
         * Japan.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>VectorHereExploreTruck</code> – A global map containing truck restrictions and attributes (e.g. width /
         * height / HAZMAT) symbolized with highlighted segments and icons on top of HERE Explore to support use cases
         * within transport and logistics.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RasterHereExploreSatellite</code> – A global map containing high resolution satellite imagery.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>HybridHereExploreSatellite</code> – A global map displaying the road network, street names, and city
         * labels over satellite imagery. This style will automatically retrieve both raster and vector tiles, and your
         * charges will be based on total tiles retrieved.
         * </p>
         * <note>
         * <p>
         * Hybrid styles use both vector and raster tiles when rendering the map that you see. This means that more
         * tiles are retrieved than when using either vector or raster tiles alone. Your charges will include all tiles
         * retrieved.
         * </p>
         * </note></li>
         * </ul>
         * <p>
         * Valid <a href="https://docs.aws.amazon.com/location/previous/developerguide/grab.html">GrabMaps map
         * styles</a>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>VectorGrabStandardLight</code> – The Grab Standard Light map style provides a basemap with detailed
         * land use coloring, area names, roads, landmarks, and points of interest covering Southeast Asia.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>VectorGrabStandardDark</code> – The Grab Standard Dark map style provides a dark variation of the
         * standard basemap covering Southeast Asia.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * Grab provides maps only for countries in Southeast Asia, and is only available in the Asia Pacific
         * (Singapore) Region (<code>ap-southeast-1</code>). For more information, see <a
         * href="https://docs.aws.amazon.com/location/previous/developerguide/grab.html#grab-coverage-area">GrabMaps
         * countries and area covered</a>.
         * </p>
         * </note>
         * <p>
         * Valid <a href="https://docs.aws.amazon.com/location/previous/developerguide/open-data.html">Open Data map
         * styles</a>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>VectorOpenDataStandardLight</code> – The Open Data Standard Light map style provides a detailed basemap
         * for the world suitable for website and mobile application use. The map includes highways major roads, minor
         * roads, railways, water features, cities, parks, landmarks, building footprints, and administrative
         * boundaries.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>VectorOpenDataStandardDark</code> – Open Data Standard Dark is a dark-themed map style that provides a
         * detailed basemap for the world suitable for website and mobile application use. The map includes highways
         * major roads, minor roads, railways, water features, cities, parks, landmarks, building footprints, and
         * administrative boundaries.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>VectorOpenDataVisualizationLight</code> – The Open Data Visualization Light map style is a light-themed
         * style with muted colors and fewer features that aids in understanding overlaid data.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>VectorOpenDataVisualizationDark</code> – The Open Data Visualization Dark map style is a dark-themed
         * style with muted colors and fewer features that aids in understanding overlaid data.
         * </p>
         * </li>
         * </ul>
         * 
         * @param style
         *        Specifies the map style selected from an available data provider.</p>
         *        <p>
         *        Valid <a href="https://docs.aws.amazon.com/location/previous/developerguide/esri.html">Esri map
         *        styles</a>:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>VectorEsriDarkGrayCanvas</code> – The Esri Dark Gray Canvas map style. A vector basemap with a
         *        dark gray, neutral background with minimal colors, labels, and features that's designed to draw
         *        attention to your thematic content.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RasterEsriImagery</code> – The Esri Imagery map style. A raster basemap that provides one meter
         *        or better satellite and aerial imagery in many parts of the world and lower resolution satellite
         *        imagery worldwide.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>VectorEsriLightGrayCanvas</code> – The Esri Light Gray Canvas map style, which provides a
         *        detailed vector basemap with a light gray, neutral background style with minimal colors, labels, and
         *        features that's designed to draw attention to your thematic content.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>VectorEsriTopographic</code> – The Esri Light map style, which provides a detailed vector
         *        basemap with a classic Esri map style.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>VectorEsriStreets</code> – The Esri Street Map style, which provides a detailed vector basemap
         *        for the world symbolized with a classic Esri street map style. The vector tile layer is similar in
         *        content and style to the World Street Map raster map.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>VectorEsriNavigation</code> – The Esri Navigation map style, which provides a detailed basemap
         *        for the world symbolized with a custom navigation map style that's designed for use during the day in
         *        mobile devices.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Valid <a href="https://docs.aws.amazon.com/location/previous/developerguide/HERE.html">HERE
         *        Technologies map styles</a>:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>VectorHereContrast</code> – The HERE Contrast (Berlin) map style is a high contrast detailed
         *        base map of the world that blends 3D and 2D rendering.
         *        </p>
         *        <note>
         *        <p>
         *        The <code>VectorHereContrast</code> style has been renamed from <code>VectorHereBerlin</code>.
         *        <code>VectorHereBerlin</code> has been deprecated, but will continue to work in applications that use
         *        it.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        <code>VectorHereExplore</code> – A default HERE map style containing a neutral, global map and its
         *        features including roads, buildings, landmarks, and water features. It also now includes a fully
         *        designed map of Japan.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>VectorHereExploreTruck</code> – A global map containing truck restrictions and attributes (e.g.
         *        width / height / HAZMAT) symbolized with highlighted segments and icons on top of HERE Explore to
         *        support use cases within transport and logistics.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RasterHereExploreSatellite</code> – A global map containing high resolution satellite imagery.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>HybridHereExploreSatellite</code> – A global map displaying the road network, street names, and
         *        city labels over satellite imagery. This style will automatically retrieve both raster and vector
         *        tiles, and your charges will be based on total tiles retrieved.
         *        </p>
         *        <note>
         *        <p>
         *        Hybrid styles use both vector and raster tiles when rendering the map that you see. This means that
         *        more tiles are retrieved than when using either vector or raster tiles alone. Your charges will
         *        include all tiles retrieved.
         *        </p>
         *        </note></li>
         *        </ul>
         *        <p>
         *        Valid <a href="https://docs.aws.amazon.com/location/previous/developerguide/grab.html">GrabMaps map
         *        styles</a>:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>VectorGrabStandardLight</code> – The Grab Standard Light map style provides a basemap with
         *        detailed land use coloring, area names, roads, landmarks, and points of interest covering Southeast
         *        Asia.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>VectorGrabStandardDark</code> – The Grab Standard Dark map style provides a dark variation of
         *        the standard basemap covering Southeast Asia.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        Grab provides maps only for countries in Southeast Asia, and is only available in the Asia Pacific
         *        (Singapore) Region (<code>ap-southeast-1</code>). For more information, see <a
         *        href="https://docs.aws.amazon.com/location/previous/developerguide/grab.html#grab-coverage-area"
         *        >GrabMaps countries and area covered</a>.
         *        </p>
         *        </note>
         *        <p>
         *        Valid <a href="https://docs.aws.amazon.com/location/previous/developerguide/open-data.html">Open Data
         *        map styles</a>:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>VectorOpenDataStandardLight</code> – The Open Data Standard Light map style provides a detailed
         *        basemap for the world suitable for website and mobile application use. The map includes highways major
         *        roads, minor roads, railways, water features, cities, parks, landmarks, building footprints, and
         *        administrative boundaries.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>VectorOpenDataStandardDark</code> – Open Data Standard Dark is a dark-themed map style that
         *        provides a detailed basemap for the world suitable for website and mobile application use. The map
         *        includes highways major roads, minor roads, railways, water features, cities, parks, landmarks,
         *        building footprints, and administrative boundaries.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>VectorOpenDataVisualizationLight</code> – The Open Data Visualization Light map style is a
         *        light-themed style with muted colors and fewer features that aids in understanding overlaid data.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>VectorOpenDataVisualizationDark</code> – The Open Data Visualization Dark map style is a
         *        dark-themed style with muted colors and fewer features that aids in understanding overlaid data.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder style(String style);

        /**
         * <p>
         * Specifies the political view for the style. Leave unset to not use a political view, or, for styles that
         * support specific political views, you can choose a view, such as <code>IND</code> for the Indian view.
         * </p>
         * <p>
         * Default is unset.
         * </p>
         * <note>
         * <p>
         * Not all map resources or styles support political view styles. See <a
         * href="https://docs.aws.amazon.com/location/previous/developerguide/map-concepts.html#political-views"
         * >Political views</a> for more information.
         * </p>
         * </note>
         * 
         * @param politicalView
         *        Specifies the political view for the style. Leave unset to not use a political view, or, for styles
         *        that support specific political views, you can choose a view, such as <code>IND</code> for the Indian
         *        view.</p>
         *        <p>
         *        Default is unset.
         *        </p>
         *        <note>
         *        <p>
         *        Not all map resources or styles support political view styles. See <a href=
         *        "https://docs.aws.amazon.com/location/previous/developerguide/map-concepts.html#political-views"
         *        >Political views</a> for more information.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder politicalView(String politicalView);

        /**
         * <p>
         * Specifies the custom layers for the style. Leave unset to not enable any custom layer, or, for styles that
         * support custom layers, you can enable layer(s), such as POI layer for the VectorEsriNavigation style. Default
         * is <code>unset</code>.
         * </p>
         * <note>
         * <p>
         * Not all map resources or styles support custom layers. See Custom Layers for more information.
         * </p>
         * </note>
         * 
         * @param customLayers
         *        Specifies the custom layers for the style. Leave unset to not enable any custom layer, or, for styles
         *        that support custom layers, you can enable layer(s), such as POI layer for the VectorEsriNavigation
         *        style. Default is <code>unset</code>.</p> <note>
         *        <p>
         *        Not all map resources or styles support custom layers. See Custom Layers for more information.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customLayers(Collection<String> customLayers);

        /**
         * <p>
         * Specifies the custom layers for the style. Leave unset to not enable any custom layer, or, for styles that
         * support custom layers, you can enable layer(s), such as POI layer for the VectorEsriNavigation style. Default
         * is <code>unset</code>.
         * </p>
         * <note>
         * <p>
         * Not all map resources or styles support custom layers. See Custom Layers for more information.
         * </p>
         * </note>
         * 
         * @param customLayers
         *        Specifies the custom layers for the style. Leave unset to not enable any custom layer, or, for styles
         *        that support custom layers, you can enable layer(s), such as POI layer for the VectorEsriNavigation
         *        style. Default is <code>unset</code>.</p> <note>
         *        <p>
         *        Not all map resources or styles support custom layers. See Custom Layers for more information.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customLayers(String... customLayers);
    }

    static final class BuilderImpl implements Builder {
        private String style;

        private String politicalView;

        private List<String> customLayers = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(MapConfiguration model) {
            style(model.style);
            politicalView(model.politicalView);
            customLayers(model.customLayers);
        }

        public final String getStyle() {
            return style;
        }

        public final void setStyle(String style) {
            this.style = style;
        }

        @Override
        public final Builder style(String style) {
            this.style = style;
            return this;
        }

        public final String getPoliticalView() {
            return politicalView;
        }

        public final void setPoliticalView(String politicalView) {
            this.politicalView = politicalView;
        }

        @Override
        public final Builder politicalView(String politicalView) {
            this.politicalView = politicalView;
            return this;
        }

        public final Collection<String> getCustomLayers() {
            if (customLayers instanceof SdkAutoConstructList) {
                return null;
            }
            return customLayers;
        }

        public final void setCustomLayers(Collection<String> customLayers) {
            this.customLayers = CustomLayerListCopier.copy(customLayers);
        }

        @Override
        public final Builder customLayers(Collection<String> customLayers) {
            this.customLayers = CustomLayerListCopier.copy(customLayers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder customLayers(String... customLayers) {
            customLayers(Arrays.asList(customLayers));
            return this;
        }

        @Override
        public MapConfiguration build() {
            return new MapConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
