/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.location.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateKeyRequest extends LocationRequest implements
        ToCopyableBuilder<CreateKeyRequest.Builder, CreateKeyRequest> {
    private static final SdkField<String> KEY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KeyName").getter(getter(CreateKeyRequest::keyName)).setter(setter(Builder::keyName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyName").build()).build();

    private static final SdkField<ApiKeyRestrictions> RESTRICTIONS_FIELD = SdkField
            .<ApiKeyRestrictions> builder(MarshallingType.SDK_POJO).memberName("Restrictions")
            .getter(getter(CreateKeyRequest::restrictions)).setter(setter(Builder::restrictions))
            .constructor(ApiKeyRestrictions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Restrictions").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(CreateKeyRequest::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<Instant> EXPIRE_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("ExpireTime")
            .getter(getter(CreateKeyRequest::expireTime))
            .setter(setter(Builder::expireTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExpireTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Boolean> NO_EXPIRY_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("NoExpiry").getter(getter(CreateKeyRequest::noExpiry)).setter(setter(Builder::noExpiry))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NoExpiry").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Tags")
            .getter(getter(CreateKeyRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(KEY_NAME_FIELD,
            RESTRICTIONS_FIELD, DESCRIPTION_FIELD, EXPIRE_TIME_FIELD, NO_EXPIRY_FIELD, TAGS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String keyName;

    private final ApiKeyRestrictions restrictions;

    private final String description;

    private final Instant expireTime;

    private final Boolean noExpiry;

    private final Map<String, String> tags;

    private CreateKeyRequest(BuilderImpl builder) {
        super(builder);
        this.keyName = builder.keyName;
        this.restrictions = builder.restrictions;
        this.description = builder.description;
        this.expireTime = builder.expireTime;
        this.noExpiry = builder.noExpiry;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * A custom name for the API key resource.
     * </p>
     * <p>
     * Requirements:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Contain only alphanumeric characters (A–Z, a–z, 0–9), hyphens (-), periods (.), and underscores (_).
     * </p>
     * </li>
     * <li>
     * <p>
     * Must be a unique API key name.
     * </p>
     * </li>
     * <li>
     * <p>
     * No spaces allowed. For example, <code>ExampleAPIKey</code>.
     * </p>
     * </li>
     * </ul>
     * 
     * @return A custom name for the API key resource.</p>
     *         <p>
     *         Requirements:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Contain only alphanumeric characters (A–Z, a–z, 0–9), hyphens (-), periods (.), and underscores (_).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Must be a unique API key name.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         No spaces allowed. For example, <code>ExampleAPIKey</code>.
     *         </p>
     *         </li>
     */
    public final String keyName() {
        return keyName;
    }

    /**
     * <p>
     * The API key restrictions for the API key resource.
     * </p>
     * 
     * @return The API key restrictions for the API key resource.
     */
    public final ApiKeyRestrictions restrictions() {
        return restrictions;
    }

    /**
     * <p>
     * An optional description for the API key resource.
     * </p>
     * 
     * @return An optional description for the API key resource.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The optional timestamp for when the API key resource will expire in <a
     * href="https://www.iso.org/iso-8601-date-and-time-format.html"> ISO 8601</a> format:
     * <code>YYYY-MM-DDThh:mm:ss.sssZ</code>. One of <code>NoExpiry</code> or <code>ExpireTime</code> must be set.
     * </p>
     * 
     * @return The optional timestamp for when the API key resource will expire in <a
     *         href="https://www.iso.org/iso-8601-date-and-time-format.html"> ISO 8601</a> format:
     *         <code>YYYY-MM-DDThh:mm:ss.sssZ</code>. One of <code>NoExpiry</code> or <code>ExpireTime</code> must be
     *         set.
     */
    public final Instant expireTime() {
        return expireTime;
    }

    /**
     * <p>
     * Optionally set to <code>true</code> to set no expiration time for the API key. One of <code>NoExpiry</code> or
     * <code>ExpireTime</code> must be set.
     * </p>
     * 
     * @return Optionally set to <code>true</code> to set no expiration time for the API key. One of
     *         <code>NoExpiry</code> or <code>ExpireTime</code> must be set.
     */
    public final Boolean noExpiry() {
        return noExpiry;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Applies one or more tags to the map resource. A tag is a key-value pair that helps manage, identify, search, and
     * filter your resources by labelling them.
     * </p>
     * <p>
     * Format: <code>"key" : "value"</code>
     * </p>
     * <p>
     * Restrictions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Maximum 50 tags per resource
     * </p>
     * </li>
     * <li>
     * <p>
     * Each resource tag must be unique with a maximum of one value.
     * </p>
     * </li>
     * <li>
     * <p>
     * Maximum key length: 128 Unicode characters in UTF-8
     * </p>
     * </li>
     * <li>
     * <p>
     * Maximum value length: 256 Unicode characters in UTF-8
     * </p>
     * </li>
     * <li>
     * <p>
     * Can use alphanumeric characters (A–Z, a–z, 0–9), and the following characters: + - = . _ : / @.
     * </p>
     * </li>
     * <li>
     * <p>
     * Cannot use "aws:" as a prefix for a key.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return Applies one or more tags to the map resource. A tag is a key-value pair that helps manage, identify,
     *         search, and filter your resources by labelling them.</p>
     *         <p>
     *         Format: <code>"key" : "value"</code>
     *         </p>
     *         <p>
     *         Restrictions:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Maximum 50 tags per resource
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Each resource tag must be unique with a maximum of one value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Maximum key length: 128 Unicode characters in UTF-8
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Maximum value length: 256 Unicode characters in UTF-8
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Can use alphanumeric characters (A–Z, a–z, 0–9), and the following characters: + - = . _ : / @.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Cannot use "aws:" as a prefix for a key.
     *         </p>
     *         </li>
     */
    public final Map<String, String> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(keyName());
        hashCode = 31 * hashCode + Objects.hashCode(restrictions());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(expireTime());
        hashCode = 31 * hashCode + Objects.hashCode(noExpiry());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateKeyRequest)) {
            return false;
        }
        CreateKeyRequest other = (CreateKeyRequest) obj;
        return Objects.equals(keyName(), other.keyName()) && Objects.equals(restrictions(), other.restrictions())
                && Objects.equals(description(), other.description()) && Objects.equals(expireTime(), other.expireTime())
                && Objects.equals(noExpiry(), other.noExpiry()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateKeyRequest").add("KeyName", keyName()).add("Restrictions", restrictions())
                .add("Description", description())
                .add("ExpireTime", expireTime() == null ? null : "*** Sensitive Data Redacted ***").add("NoExpiry", noExpiry())
                .add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "KeyName":
            return Optional.ofNullable(clazz.cast(keyName()));
        case "Restrictions":
            return Optional.ofNullable(clazz.cast(restrictions()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "ExpireTime":
            return Optional.ofNullable(clazz.cast(expireTime()));
        case "NoExpiry":
            return Optional.ofNullable(clazz.cast(noExpiry()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("KeyName", KEY_NAME_FIELD);
        map.put("Restrictions", RESTRICTIONS_FIELD);
        map.put("Description", DESCRIPTION_FIELD);
        map.put("ExpireTime", EXPIRE_TIME_FIELD);
        map.put("NoExpiry", NO_EXPIRY_FIELD);
        map.put("Tags", TAGS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CreateKeyRequest, T> g) {
        return obj -> g.apply((CreateKeyRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends LocationRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateKeyRequest> {
        /**
         * <p>
         * A custom name for the API key resource.
         * </p>
         * <p>
         * Requirements:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Contain only alphanumeric characters (A–Z, a–z, 0–9), hyphens (-), periods (.), and underscores (_).
         * </p>
         * </li>
         * <li>
         * <p>
         * Must be a unique API key name.
         * </p>
         * </li>
         * <li>
         * <p>
         * No spaces allowed. For example, <code>ExampleAPIKey</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param keyName
         *        A custom name for the API key resource.</p>
         *        <p>
         *        Requirements:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Contain only alphanumeric characters (A–Z, a–z, 0–9), hyphens (-), periods (.), and underscores (_).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Must be a unique API key name.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        No spaces allowed. For example, <code>ExampleAPIKey</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keyName(String keyName);

        /**
         * <p>
         * The API key restrictions for the API key resource.
         * </p>
         * 
         * @param restrictions
         *        The API key restrictions for the API key resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder restrictions(ApiKeyRestrictions restrictions);

        /**
         * <p>
         * The API key restrictions for the API key resource.
         * </p>
         * This is a convenience method that creates an instance of the {@link ApiKeyRestrictions.Builder} avoiding the
         * need to create one manually via {@link ApiKeyRestrictions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ApiKeyRestrictions.Builder#build()} is called immediately and its
         * result is passed to {@link #restrictions(ApiKeyRestrictions)}.
         * 
         * @param restrictions
         *        a consumer that will call methods on {@link ApiKeyRestrictions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #restrictions(ApiKeyRestrictions)
         */
        default Builder restrictions(Consumer<ApiKeyRestrictions.Builder> restrictions) {
            return restrictions(ApiKeyRestrictions.builder().applyMutation(restrictions).build());
        }

        /**
         * <p>
         * An optional description for the API key resource.
         * </p>
         * 
         * @param description
         *        An optional description for the API key resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The optional timestamp for when the API key resource will expire in <a
         * href="https://www.iso.org/iso-8601-date-and-time-format.html"> ISO 8601</a> format:
         * <code>YYYY-MM-DDThh:mm:ss.sssZ</code>. One of <code>NoExpiry</code> or <code>ExpireTime</code> must be set.
         * </p>
         * 
         * @param expireTime
         *        The optional timestamp for when the API key resource will expire in <a
         *        href="https://www.iso.org/iso-8601-date-and-time-format.html"> ISO 8601</a> format:
         *        <code>YYYY-MM-DDThh:mm:ss.sssZ</code>. One of <code>NoExpiry</code> or <code>ExpireTime</code> must be
         *        set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder expireTime(Instant expireTime);

        /**
         * <p>
         * Optionally set to <code>true</code> to set no expiration time for the API key. One of <code>NoExpiry</code>
         * or <code>ExpireTime</code> must be set.
         * </p>
         * 
         * @param noExpiry
         *        Optionally set to <code>true</code> to set no expiration time for the API key. One of
         *        <code>NoExpiry</code> or <code>ExpireTime</code> must be set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder noExpiry(Boolean noExpiry);

        /**
         * <p>
         * Applies one or more tags to the map resource. A tag is a key-value pair that helps manage, identify, search,
         * and filter your resources by labelling them.
         * </p>
         * <p>
         * Format: <code>"key" : "value"</code>
         * </p>
         * <p>
         * Restrictions:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Maximum 50 tags per resource
         * </p>
         * </li>
         * <li>
         * <p>
         * Each resource tag must be unique with a maximum of one value.
         * </p>
         * </li>
         * <li>
         * <p>
         * Maximum key length: 128 Unicode characters in UTF-8
         * </p>
         * </li>
         * <li>
         * <p>
         * Maximum value length: 256 Unicode characters in UTF-8
         * </p>
         * </li>
         * <li>
         * <p>
         * Can use alphanumeric characters (A–Z, a–z, 0–9), and the following characters: + - = . _ : / @.
         * </p>
         * </li>
         * <li>
         * <p>
         * Cannot use "aws:" as a prefix for a key.
         * </p>
         * </li>
         * </ul>
         * 
         * @param tags
         *        Applies one or more tags to the map resource. A tag is a key-value pair that helps manage, identify,
         *        search, and filter your resources by labelling them.</p>
         *        <p>
         *        Format: <code>"key" : "value"</code>
         *        </p>
         *        <p>
         *        Restrictions:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Maximum 50 tags per resource
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Each resource tag must be unique with a maximum of one value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Maximum key length: 128 Unicode characters in UTF-8
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Maximum value length: 256 Unicode characters in UTF-8
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Can use alphanumeric characters (A–Z, a–z, 0–9), and the following characters: + - = . _ : / @.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Cannot use "aws:" as a prefix for a key.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends LocationRequest.BuilderImpl implements Builder {
        private String keyName;

        private ApiKeyRestrictions restrictions;

        private String description;

        private Instant expireTime;

        private Boolean noExpiry;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateKeyRequest model) {
            super(model);
            keyName(model.keyName);
            restrictions(model.restrictions);
            description(model.description);
            expireTime(model.expireTime);
            noExpiry(model.noExpiry);
            tags(model.tags);
        }

        public final String getKeyName() {
            return keyName;
        }

        public final void setKeyName(String keyName) {
            this.keyName = keyName;
        }

        @Override
        public final Builder keyName(String keyName) {
            this.keyName = keyName;
            return this;
        }

        public final ApiKeyRestrictions.Builder getRestrictions() {
            return restrictions != null ? restrictions.toBuilder() : null;
        }

        public final void setRestrictions(ApiKeyRestrictions.BuilderImpl restrictions) {
            this.restrictions = restrictions != null ? restrictions.build() : null;
        }

        @Override
        public final Builder restrictions(ApiKeyRestrictions restrictions) {
            this.restrictions = restrictions;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final Instant getExpireTime() {
            return expireTime;
        }

        public final void setExpireTime(Instant expireTime) {
            this.expireTime = expireTime;
        }

        @Override
        public final Builder expireTime(Instant expireTime) {
            this.expireTime = expireTime;
            return this;
        }

        public final Boolean getNoExpiry() {
            return noExpiry;
        }

        public final void setNoExpiry(Boolean noExpiry) {
            this.noExpiry = noExpiry;
        }

        @Override
        public final Builder noExpiry(Boolean noExpiry) {
            this.noExpiry = noExpiry;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateKeyRequest build() {
            return new CreateKeyRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
