/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.location.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The device's position, IP address, and Wi-Fi access points.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DeviceState implements SdkPojo, Serializable, ToCopyableBuilder<DeviceState.Builder, DeviceState> {
    private static final SdkField<String> DEVICE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DeviceId").getter(getter(DeviceState::deviceId)).setter(setter(Builder::deviceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeviceId").build()).build();

    private static final SdkField<Instant> SAMPLE_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("SampleTime")
            .getter(getter(DeviceState::sampleTime))
            .setter(setter(Builder::sampleTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SampleTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<List<Double>> POSITION_FIELD = SdkField
            .<List<Double>> builder(MarshallingType.LIST)
            .memberName("Position")
            .getter(getter(DeviceState::position))
            .setter(setter(Builder::position))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Position").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Double> builder(MarshallingType.DOUBLE)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<PositionalAccuracy> ACCURACY_FIELD = SdkField
            .<PositionalAccuracy> builder(MarshallingType.SDK_POJO).memberName("Accuracy").getter(getter(DeviceState::accuracy))
            .setter(setter(Builder::accuracy)).constructor(PositionalAccuracy::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Accuracy").build()).build();

    private static final SdkField<String> IPV4_ADDRESS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Ipv4Address").getter(getter(DeviceState::ipv4Address)).setter(setter(Builder::ipv4Address))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Ipv4Address").build()).build();

    private static final SdkField<List<WiFiAccessPoint>> WI_FI_ACCESS_POINTS_FIELD = SdkField
            .<List<WiFiAccessPoint>> builder(MarshallingType.LIST)
            .memberName("WiFiAccessPoints")
            .getter(getter(DeviceState::wiFiAccessPoints))
            .setter(setter(Builder::wiFiAccessPoints))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WiFiAccessPoints").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<WiFiAccessPoint> builder(MarshallingType.SDK_POJO)
                                            .constructor(WiFiAccessPoint::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<CellSignals> CELL_SIGNALS_FIELD = SdkField.<CellSignals> builder(MarshallingType.SDK_POJO)
            .memberName("CellSignals").getter(getter(DeviceState::cellSignals)).setter(setter(Builder::cellSignals))
            .constructor(CellSignals::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CellSignals").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections
            .unmodifiableList(Arrays.asList(DEVICE_ID_FIELD, SAMPLE_TIME_FIELD, POSITION_FIELD, ACCURACY_FIELD,
                    IPV4_ADDRESS_FIELD, WI_FI_ACCESS_POINTS_FIELD, CELL_SIGNALS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String deviceId;

    private final Instant sampleTime;

    private final List<Double> position;

    private final PositionalAccuracy accuracy;

    private final String ipv4Address;

    private final List<WiFiAccessPoint> wiFiAccessPoints;

    private final CellSignals cellSignals;

    private DeviceState(BuilderImpl builder) {
        this.deviceId = builder.deviceId;
        this.sampleTime = builder.sampleTime;
        this.position = builder.position;
        this.accuracy = builder.accuracy;
        this.ipv4Address = builder.ipv4Address;
        this.wiFiAccessPoints = builder.wiFiAccessPoints;
        this.cellSignals = builder.cellSignals;
    }

    /**
     * <p>
     * The device identifier.
     * </p>
     * 
     * @return The device identifier.
     */
    public final String deviceId() {
        return deviceId;
    }

    /**
     * <p>
     * The timestamp at which the device's position was determined. Uses <a
     * href="https://www.iso.org/iso-8601-date-and-time-format.html"> ISO 8601 </a> format:
     * <code>YYYY-MM-DDThh:mm:ss.sssZ</code>.
     * </p>
     * 
     * @return The timestamp at which the device's position was determined. Uses <a
     *         href="https://www.iso.org/iso-8601-date-and-time-format.html"> ISO 8601 </a> format:
     *         <code>YYYY-MM-DDThh:mm:ss.sssZ</code>.
     */
    public final Instant sampleTime() {
        return sampleTime;
    }

    /**
     * For responses, this returns true if the service returned a value for the Position property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasPosition() {
        return position != null && !(position instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The last known device position.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPosition} method.
     * </p>
     * 
     * @return The last known device position.
     */
    public final List<Double> position() {
        return position;
    }

    /**
     * Returns the value of the Accuracy property for this object.
     * 
     * @return The value of the Accuracy property for this object.
     */
    public final PositionalAccuracy accuracy() {
        return accuracy;
    }

    /**
     * <p>
     * The device's Ipv4 address.
     * </p>
     * 
     * @return The device's Ipv4 address.
     */
    public final String ipv4Address() {
        return ipv4Address;
    }

    /**
     * For responses, this returns true if the service returned a value for the WiFiAccessPoints property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasWiFiAccessPoints() {
        return wiFiAccessPoints != null && !(wiFiAccessPoints instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Wi-Fi access points the device is using.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasWiFiAccessPoints} method.
     * </p>
     * 
     * @return The Wi-Fi access points the device is using.
     */
    public final List<WiFiAccessPoint> wiFiAccessPoints() {
        return wiFiAccessPoints;
    }

    /**
     * <p>
     * The cellular network infrastructure that the device is connected to.
     * </p>
     * 
     * @return The cellular network infrastructure that the device is connected to.
     */
    public final CellSignals cellSignals() {
        return cellSignals;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(deviceId());
        hashCode = 31 * hashCode + Objects.hashCode(sampleTime());
        hashCode = 31 * hashCode + Objects.hashCode(hasPosition() ? position() : null);
        hashCode = 31 * hashCode + Objects.hashCode(accuracy());
        hashCode = 31 * hashCode + Objects.hashCode(ipv4Address());
        hashCode = 31 * hashCode + Objects.hashCode(hasWiFiAccessPoints() ? wiFiAccessPoints() : null);
        hashCode = 31 * hashCode + Objects.hashCode(cellSignals());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DeviceState)) {
            return false;
        }
        DeviceState other = (DeviceState) obj;
        return Objects.equals(deviceId(), other.deviceId()) && Objects.equals(sampleTime(), other.sampleTime())
                && hasPosition() == other.hasPosition() && Objects.equals(position(), other.position())
                && Objects.equals(accuracy(), other.accuracy()) && Objects.equals(ipv4Address(), other.ipv4Address())
                && hasWiFiAccessPoints() == other.hasWiFiAccessPoints()
                && Objects.equals(wiFiAccessPoints(), other.wiFiAccessPoints())
                && Objects.equals(cellSignals(), other.cellSignals());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DeviceState").add("DeviceId", deviceId())
                .add("SampleTime", sampleTime() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Position", position() == null ? null : "*** Sensitive Data Redacted ***").add("Accuracy", accuracy())
                .add("Ipv4Address", ipv4Address()).add("WiFiAccessPoints", hasWiFiAccessPoints() ? wiFiAccessPoints() : null)
                .add("CellSignals", cellSignals()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DeviceId":
            return Optional.ofNullable(clazz.cast(deviceId()));
        case "SampleTime":
            return Optional.ofNullable(clazz.cast(sampleTime()));
        case "Position":
            return Optional.ofNullable(clazz.cast(position()));
        case "Accuracy":
            return Optional.ofNullable(clazz.cast(accuracy()));
        case "Ipv4Address":
            return Optional.ofNullable(clazz.cast(ipv4Address()));
        case "WiFiAccessPoints":
            return Optional.ofNullable(clazz.cast(wiFiAccessPoints()));
        case "CellSignals":
            return Optional.ofNullable(clazz.cast(cellSignals()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("DeviceId", DEVICE_ID_FIELD);
        map.put("SampleTime", SAMPLE_TIME_FIELD);
        map.put("Position", POSITION_FIELD);
        map.put("Accuracy", ACCURACY_FIELD);
        map.put("Ipv4Address", IPV4_ADDRESS_FIELD);
        map.put("WiFiAccessPoints", WI_FI_ACCESS_POINTS_FIELD);
        map.put("CellSignals", CELL_SIGNALS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<DeviceState, T> g) {
        return obj -> g.apply((DeviceState) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DeviceState> {
        /**
         * <p>
         * The device identifier.
         * </p>
         * 
         * @param deviceId
         *        The device identifier.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deviceId(String deviceId);

        /**
         * <p>
         * The timestamp at which the device's position was determined. Uses <a
         * href="https://www.iso.org/iso-8601-date-and-time-format.html"> ISO 8601 </a> format:
         * <code>YYYY-MM-DDThh:mm:ss.sssZ</code>.
         * </p>
         * 
         * @param sampleTime
         *        The timestamp at which the device's position was determined. Uses <a
         *        href="https://www.iso.org/iso-8601-date-and-time-format.html"> ISO 8601 </a> format:
         *        <code>YYYY-MM-DDThh:mm:ss.sssZ</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sampleTime(Instant sampleTime);

        /**
         * <p>
         * The last known device position.
         * </p>
         * 
         * @param position
         *        The last known device position.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder position(Collection<Double> position);

        /**
         * <p>
         * The last known device position.
         * </p>
         * 
         * @param position
         *        The last known device position.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder position(Double... position);

        /**
         * Sets the value of the Accuracy property for this object.
         *
         * @param accuracy
         *        The new value for the Accuracy property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accuracy(PositionalAccuracy accuracy);

        /**
         * Sets the value of the Accuracy property for this object.
         *
         * This is a convenience method that creates an instance of the {@link PositionalAccuracy.Builder} avoiding the
         * need to create one manually via {@link PositionalAccuracy#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link PositionalAccuracy.Builder#build()} is called immediately and its
         * result is passed to {@link #accuracy(PositionalAccuracy)}.
         * 
         * @param accuracy
         *        a consumer that will call methods on {@link PositionalAccuracy.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #accuracy(PositionalAccuracy)
         */
        default Builder accuracy(Consumer<PositionalAccuracy.Builder> accuracy) {
            return accuracy(PositionalAccuracy.builder().applyMutation(accuracy).build());
        }

        /**
         * <p>
         * The device's Ipv4 address.
         * </p>
         * 
         * @param ipv4Address
         *        The device's Ipv4 address.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipv4Address(String ipv4Address);

        /**
         * <p>
         * The Wi-Fi access points the device is using.
         * </p>
         * 
         * @param wiFiAccessPoints
         *        The Wi-Fi access points the device is using.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder wiFiAccessPoints(Collection<WiFiAccessPoint> wiFiAccessPoints);

        /**
         * <p>
         * The Wi-Fi access points the device is using.
         * </p>
         * 
         * @param wiFiAccessPoints
         *        The Wi-Fi access points the device is using.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder wiFiAccessPoints(WiFiAccessPoint... wiFiAccessPoints);

        /**
         * <p>
         * The Wi-Fi access points the device is using.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.location.model.WiFiAccessPoint.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.location.model.WiFiAccessPoint#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.location.model.WiFiAccessPoint.Builder#build()} is called immediately
         * and its result is passed to {@link #wiFiAccessPoints(List<WiFiAccessPoint>)}.
         * 
         * @param wiFiAccessPoints
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.location.model.WiFiAccessPoint.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #wiFiAccessPoints(java.util.Collection<WiFiAccessPoint>)
         */
        Builder wiFiAccessPoints(Consumer<WiFiAccessPoint.Builder>... wiFiAccessPoints);

        /**
         * <p>
         * The cellular network infrastructure that the device is connected to.
         * </p>
         * 
         * @param cellSignals
         *        The cellular network infrastructure that the device is connected to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cellSignals(CellSignals cellSignals);

        /**
         * <p>
         * The cellular network infrastructure that the device is connected to.
         * </p>
         * This is a convenience method that creates an instance of the {@link CellSignals.Builder} avoiding the need to
         * create one manually via {@link CellSignals#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CellSignals.Builder#build()} is called immediately and its result
         * is passed to {@link #cellSignals(CellSignals)}.
         * 
         * @param cellSignals
         *        a consumer that will call methods on {@link CellSignals.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #cellSignals(CellSignals)
         */
        default Builder cellSignals(Consumer<CellSignals.Builder> cellSignals) {
            return cellSignals(CellSignals.builder().applyMutation(cellSignals).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String deviceId;

        private Instant sampleTime;

        private List<Double> position = DefaultSdkAutoConstructList.getInstance();

        private PositionalAccuracy accuracy;

        private String ipv4Address;

        private List<WiFiAccessPoint> wiFiAccessPoints = DefaultSdkAutoConstructList.getInstance();

        private CellSignals cellSignals;

        private BuilderImpl() {
        }

        private BuilderImpl(DeviceState model) {
            deviceId(model.deviceId);
            sampleTime(model.sampleTime);
            position(model.position);
            accuracy(model.accuracy);
            ipv4Address(model.ipv4Address);
            wiFiAccessPoints(model.wiFiAccessPoints);
            cellSignals(model.cellSignals);
        }

        public final String getDeviceId() {
            return deviceId;
        }

        public final void setDeviceId(String deviceId) {
            this.deviceId = deviceId;
        }

        @Override
        public final Builder deviceId(String deviceId) {
            this.deviceId = deviceId;
            return this;
        }

        public final Instant getSampleTime() {
            return sampleTime;
        }

        public final void setSampleTime(Instant sampleTime) {
            this.sampleTime = sampleTime;
        }

        @Override
        public final Builder sampleTime(Instant sampleTime) {
            this.sampleTime = sampleTime;
            return this;
        }

        public final Collection<Double> getPosition() {
            if (position instanceof SdkAutoConstructList) {
                return null;
            }
            return position;
        }

        public final void setPosition(Collection<Double> position) {
            this.position = PositionCopier.copy(position);
        }

        @Override
        public final Builder position(Collection<Double> position) {
            this.position = PositionCopier.copy(position);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder position(Double... position) {
            position(Arrays.asList(position));
            return this;
        }

        public final PositionalAccuracy.Builder getAccuracy() {
            return accuracy != null ? accuracy.toBuilder() : null;
        }

        public final void setAccuracy(PositionalAccuracy.BuilderImpl accuracy) {
            this.accuracy = accuracy != null ? accuracy.build() : null;
        }

        @Override
        public final Builder accuracy(PositionalAccuracy accuracy) {
            this.accuracy = accuracy;
            return this;
        }

        public final String getIpv4Address() {
            return ipv4Address;
        }

        public final void setIpv4Address(String ipv4Address) {
            this.ipv4Address = ipv4Address;
        }

        @Override
        public final Builder ipv4Address(String ipv4Address) {
            this.ipv4Address = ipv4Address;
            return this;
        }

        public final List<WiFiAccessPoint.Builder> getWiFiAccessPoints() {
            List<WiFiAccessPoint.Builder> result = WiFiAccessPointListCopier.copyToBuilder(this.wiFiAccessPoints);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setWiFiAccessPoints(Collection<WiFiAccessPoint.BuilderImpl> wiFiAccessPoints) {
            this.wiFiAccessPoints = WiFiAccessPointListCopier.copyFromBuilder(wiFiAccessPoints);
        }

        @Override
        public final Builder wiFiAccessPoints(Collection<WiFiAccessPoint> wiFiAccessPoints) {
            this.wiFiAccessPoints = WiFiAccessPointListCopier.copy(wiFiAccessPoints);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder wiFiAccessPoints(WiFiAccessPoint... wiFiAccessPoints) {
            wiFiAccessPoints(Arrays.asList(wiFiAccessPoints));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder wiFiAccessPoints(Consumer<WiFiAccessPoint.Builder>... wiFiAccessPoints) {
            wiFiAccessPoints(Stream.of(wiFiAccessPoints).map(c -> WiFiAccessPoint.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final CellSignals.Builder getCellSignals() {
            return cellSignals != null ? cellSignals.toBuilder() : null;
        }

        public final void setCellSignals(CellSignals.BuilderImpl cellSignals) {
            this.cellSignals = cellSignals != null ? cellSignals.build() : null;
        }

        @Override
        public final Builder cellSignals(CellSignals cellSignals) {
            this.cellSignals = cellSignals;
            return this;
        }

        @Override
        public DeviceState build() {
            return new DeviceState(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
