/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lookoutequipment;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.lookoutequipment.model.AccessDeniedException;
import software.amazon.awssdk.services.lookoutequipment.model.ConflictException;
import software.amazon.awssdk.services.lookoutequipment.model.CreateDatasetRequest;
import software.amazon.awssdk.services.lookoutequipment.model.CreateDatasetResponse;
import software.amazon.awssdk.services.lookoutequipment.model.CreateInferenceSchedulerRequest;
import software.amazon.awssdk.services.lookoutequipment.model.CreateInferenceSchedulerResponse;
import software.amazon.awssdk.services.lookoutequipment.model.CreateModelRequest;
import software.amazon.awssdk.services.lookoutequipment.model.CreateModelResponse;
import software.amazon.awssdk.services.lookoutequipment.model.DeleteDatasetRequest;
import software.amazon.awssdk.services.lookoutequipment.model.DeleteDatasetResponse;
import software.amazon.awssdk.services.lookoutequipment.model.DeleteInferenceSchedulerRequest;
import software.amazon.awssdk.services.lookoutequipment.model.DeleteInferenceSchedulerResponse;
import software.amazon.awssdk.services.lookoutequipment.model.DeleteModelRequest;
import software.amazon.awssdk.services.lookoutequipment.model.DeleteModelResponse;
import software.amazon.awssdk.services.lookoutequipment.model.DescribeDataIngestionJobRequest;
import software.amazon.awssdk.services.lookoutequipment.model.DescribeDataIngestionJobResponse;
import software.amazon.awssdk.services.lookoutequipment.model.DescribeDatasetRequest;
import software.amazon.awssdk.services.lookoutequipment.model.DescribeDatasetResponse;
import software.amazon.awssdk.services.lookoutequipment.model.DescribeInferenceSchedulerRequest;
import software.amazon.awssdk.services.lookoutequipment.model.DescribeInferenceSchedulerResponse;
import software.amazon.awssdk.services.lookoutequipment.model.DescribeModelRequest;
import software.amazon.awssdk.services.lookoutequipment.model.DescribeModelResponse;
import software.amazon.awssdk.services.lookoutequipment.model.InternalServerException;
import software.amazon.awssdk.services.lookoutequipment.model.ListDataIngestionJobsRequest;
import software.amazon.awssdk.services.lookoutequipment.model.ListDataIngestionJobsResponse;
import software.amazon.awssdk.services.lookoutequipment.model.ListDatasetsRequest;
import software.amazon.awssdk.services.lookoutequipment.model.ListDatasetsResponse;
import software.amazon.awssdk.services.lookoutequipment.model.ListInferenceExecutionsRequest;
import software.amazon.awssdk.services.lookoutequipment.model.ListInferenceExecutionsResponse;
import software.amazon.awssdk.services.lookoutequipment.model.ListInferenceSchedulersRequest;
import software.amazon.awssdk.services.lookoutequipment.model.ListInferenceSchedulersResponse;
import software.amazon.awssdk.services.lookoutequipment.model.ListModelsRequest;
import software.amazon.awssdk.services.lookoutequipment.model.ListModelsResponse;
import software.amazon.awssdk.services.lookoutequipment.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.lookoutequipment.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.lookoutequipment.model.LookoutEquipmentException;
import software.amazon.awssdk.services.lookoutequipment.model.LookoutEquipmentRequest;
import software.amazon.awssdk.services.lookoutequipment.model.ResourceNotFoundException;
import software.amazon.awssdk.services.lookoutequipment.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.lookoutequipment.model.StartDataIngestionJobRequest;
import software.amazon.awssdk.services.lookoutequipment.model.StartDataIngestionJobResponse;
import software.amazon.awssdk.services.lookoutequipment.model.StartInferenceSchedulerRequest;
import software.amazon.awssdk.services.lookoutequipment.model.StartInferenceSchedulerResponse;
import software.amazon.awssdk.services.lookoutequipment.model.StopInferenceSchedulerRequest;
import software.amazon.awssdk.services.lookoutequipment.model.StopInferenceSchedulerResponse;
import software.amazon.awssdk.services.lookoutequipment.model.TagResourceRequest;
import software.amazon.awssdk.services.lookoutequipment.model.TagResourceResponse;
import software.amazon.awssdk.services.lookoutequipment.model.ThrottlingException;
import software.amazon.awssdk.services.lookoutequipment.model.UntagResourceRequest;
import software.amazon.awssdk.services.lookoutequipment.model.UntagResourceResponse;
import software.amazon.awssdk.services.lookoutequipment.model.UpdateInferenceSchedulerRequest;
import software.amazon.awssdk.services.lookoutequipment.model.UpdateInferenceSchedulerResponse;
import software.amazon.awssdk.services.lookoutequipment.model.ValidationException;
import software.amazon.awssdk.services.lookoutequipment.paginators.ListDataIngestionJobsPublisher;
import software.amazon.awssdk.services.lookoutequipment.paginators.ListDatasetsPublisher;
import software.amazon.awssdk.services.lookoutequipment.paginators.ListInferenceExecutionsPublisher;
import software.amazon.awssdk.services.lookoutequipment.paginators.ListInferenceSchedulersPublisher;
import software.amazon.awssdk.services.lookoutequipment.paginators.ListModelsPublisher;
import software.amazon.awssdk.services.lookoutequipment.transform.CreateDatasetRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.CreateInferenceSchedulerRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.CreateModelRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.DeleteDatasetRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.DeleteInferenceSchedulerRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.DeleteModelRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.DescribeDataIngestionJobRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.DescribeDatasetRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.DescribeInferenceSchedulerRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.DescribeModelRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.ListDataIngestionJobsRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.ListDatasetsRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.ListInferenceExecutionsRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.ListInferenceSchedulersRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.ListModelsRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.StartDataIngestionJobRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.StartInferenceSchedulerRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.StopInferenceSchedulerRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.lookoutequipment.transform.UpdateInferenceSchedulerRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link LookoutEquipmentAsyncClient}.
 *
 * @see LookoutEquipmentAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultLookoutEquipmentAsyncClient implements LookoutEquipmentAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultLookoutEquipmentAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultLookoutEquipmentAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a container for a collection of data being ingested for analysis. The dataset contains the metadata
     * describing where the data is and what the data actually looks like. In other words, it contains the location of
     * the data source, the data schema, and other information. A dataset also contains any tags associated with the
     * ingested data.
     * </p>
     *
     * @param createDatasetRequest
     * @return A Java Future containing the result of the CreateDataset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy constraints specified by Amazon Lookout for Equipment
     *         or a related AWS service that's being utilized.</li>
     *         <li>ConflictException The request could not be completed due to a conflict with the current state of the
     *         target resource.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ServiceQuotaExceededException Resource limitations have been exceeded.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have access to the
     *         resource.</li>
     *         <li>InternalServerException Processing of the request has failed because of an unknown error, exception
     *         or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutEquipmentException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutEquipmentAsyncClient.CreateDataset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/CreateDataset"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDatasetResponse> createDataset(CreateDatasetRequest createDatasetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDatasetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDataset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDatasetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateDatasetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDatasetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDatasetRequest, CreateDatasetResponse>()
                            .withOperationName("CreateDataset")
                            .withMarshaller(new CreateDatasetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createDatasetRequest));
            CompletableFuture<CreateDatasetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a scheduled inference. Scheduling an inference is setting up a continuous real-time inference plan to
     * analyze new measurement data. When setting up the schedule, you provide an S3 bucket location for the input data,
     * assign it a delimiter between separate entries in the data, set an offset delay if desired, and set the frequency
     * of inferencing. You must also provide an S3 bucket location for the output data.
     * </p>
     *
     * @param createInferenceSchedulerRequest
     * @return A Java Future containing the result of the CreateInferenceScheduler operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException The request could not be completed due to a conflict with the current state of the
     *         target resource.</li>
     *         <li>ResourceNotFoundException The resource requested could not be found. Verify the resource ID and retry
     *         your request.</li>
     *         <li>ValidationException The input fails to satisfy constraints specified by Amazon Lookout for Equipment
     *         or a related AWS service that's being utilized.</li>
     *         <li>ServiceQuotaExceededException Resource limitations have been exceeded.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have access to the
     *         resource.</li>
     *         <li>InternalServerException Processing of the request has failed because of an unknown error, exception
     *         or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutEquipmentException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutEquipmentAsyncClient.CreateInferenceScheduler
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/CreateInferenceScheduler"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateInferenceSchedulerResponse> createInferenceScheduler(
            CreateInferenceSchedulerRequest createInferenceSchedulerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createInferenceSchedulerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateInferenceScheduler");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateInferenceSchedulerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateInferenceSchedulerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateInferenceSchedulerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateInferenceSchedulerRequest, CreateInferenceSchedulerResponse>()
                            .withOperationName("CreateInferenceScheduler")
                            .withMarshaller(new CreateInferenceSchedulerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createInferenceSchedulerRequest));
            CompletableFuture<CreateInferenceSchedulerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an ML model for data inference.
     * </p>
     * <p>
     * A machine-learning (ML) model is a mathematical model that finds patterns in your data. In Amazon Lookout for
     * Equipment, the model learns the patterns of normal behavior and detects abnormal behavior that could be potential
     * equipment failure (or maintenance events). The models are made by analyzing normal data and abnormalities in
     * machine behavior that have already occurred.
     * </p>
     * <p>
     * Your model is trained using a portion of the data from your dataset and uses that data to learn patterns of
     * normal behavior and abnormal patterns that lead to equipment failure. Another portion of the data is used to
     * evaluate the model's accuracy.
     * </p>
     *
     * @param createModelRequest
     * @return A Java Future containing the result of the CreateModel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy constraints specified by Amazon Lookout for Equipment
     *         or a related AWS service that's being utilized.</li>
     *         <li>ConflictException The request could not be completed due to a conflict with the current state of the
     *         target resource.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ServiceQuotaExceededException Resource limitations have been exceeded.</li>
     *         <li>InternalServerException Processing of the request has failed because of an unknown error, exception
     *         or failure.</li>
     *         <li>ResourceNotFoundException The resource requested could not be found. Verify the resource ID and retry
     *         your request.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have access to the
     *         resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutEquipmentException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutEquipmentAsyncClient.CreateModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/CreateModel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateModelResponse> createModel(CreateModelRequest createModelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateModel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateModelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateModelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateModelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateModelRequest, CreateModelResponse>()
                            .withOperationName("CreateModel").withMarshaller(new CreateModelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createModelRequest));
            CompletableFuture<CreateModelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a dataset and associated artifacts. The operation will check to see if any inference scheduler or data
     * ingestion job is currently using the dataset, and if there isn't, the dataset, its metadata, and any associated
     * data stored in S3 will be deleted. This does not affect any models that used this dataset for training and
     * evaluation, but does prevent it from being used in the future.
     * </p>
     *
     * @param deleteDatasetRequest
     * @return A Java Future containing the result of the DeleteDataset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource requested could not be found. Verify the resource ID and retry
     *         your request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException Processing of the request has failed because of an unknown error, exception
     *         or failure.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have access to the
     *         resource.</li>
     *         <li>ConflictException The request could not be completed due to a conflict with the current state of the
     *         target resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutEquipmentException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutEquipmentAsyncClient.DeleteDataset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/DeleteDataset"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDatasetResponse> deleteDataset(DeleteDatasetRequest deleteDatasetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDatasetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDataset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDatasetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteDatasetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDatasetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDatasetRequest, DeleteDatasetResponse>()
                            .withOperationName("DeleteDataset")
                            .withMarshaller(new DeleteDatasetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteDatasetRequest));
            CompletableFuture<DeleteDatasetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an inference scheduler that has been set up. Already processed output results are not affected.
     * </p>
     *
     * @param deleteInferenceSchedulerRequest
     * @return A Java Future containing the result of the DeleteInferenceScheduler operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy constraints specified by Amazon Lookout for Equipment
     *         or a related AWS service that's being utilized.</li>
     *         <li>ResourceNotFoundException The resource requested could not be found. Verify the resource ID and retry
     *         your request.</li>
     *         <li>ConflictException The request could not be completed due to a conflict with the current state of the
     *         target resource.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have access to the
     *         resource.</li>
     *         <li>InternalServerException Processing of the request has failed because of an unknown error, exception
     *         or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutEquipmentException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutEquipmentAsyncClient.DeleteInferenceScheduler
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/DeleteInferenceScheduler"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteInferenceSchedulerResponse> deleteInferenceScheduler(
            DeleteInferenceSchedulerRequest deleteInferenceSchedulerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteInferenceSchedulerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteInferenceScheduler");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteInferenceSchedulerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteInferenceSchedulerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteInferenceSchedulerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteInferenceSchedulerRequest, DeleteInferenceSchedulerResponse>()
                            .withOperationName("DeleteInferenceScheduler")
                            .withMarshaller(new DeleteInferenceSchedulerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteInferenceSchedulerRequest));
            CompletableFuture<DeleteInferenceSchedulerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an ML model currently available for Amazon Lookout for Equipment. This will prevent it from being used
     * with an inference scheduler, even one that is already set up.
     * </p>
     *
     * @param deleteModelRequest
     * @return A Java Future containing the result of the DeleteModel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The resource requested could not be found. Verify the resource ID and retry
     *         your request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>InternalServerException Processing of the request has failed because of an unknown error, exception
     *         or failure.</li>
     *         <li>ConflictException The request could not be completed due to a conflict with the current state of the
     *         target resource.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have access to the
     *         resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutEquipmentException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutEquipmentAsyncClient.DeleteModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/DeleteModel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteModelResponse> deleteModel(DeleteModelRequest deleteModelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteModel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteModelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteModelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteModelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteModelRequest, DeleteModelResponse>()
                            .withOperationName("DeleteModel").withMarshaller(new DeleteModelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteModelRequest));
            CompletableFuture<DeleteModelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides information on a specific data ingestion job such as creation time, dataset ARN, status, and so on.
     * </p>
     *
     * @param describeDataIngestionJobRequest
     * @return A Java Future containing the result of the DescribeDataIngestionJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy constraints specified by Amazon Lookout for Equipment
     *         or a related AWS service that's being utilized.</li>
     *         <li>ResourceNotFoundException The resource requested could not be found. Verify the resource ID and retry
     *         your request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have access to the
     *         resource.</li>
     *         <li>InternalServerException Processing of the request has failed because of an unknown error, exception
     *         or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutEquipmentException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutEquipmentAsyncClient.DescribeDataIngestionJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/DescribeDataIngestionJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDataIngestionJobResponse> describeDataIngestionJob(
            DescribeDataIngestionJobRequest describeDataIngestionJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDataIngestionJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDataIngestionJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDataIngestionJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDataIngestionJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDataIngestionJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDataIngestionJobRequest, DescribeDataIngestionJobResponse>()
                            .withOperationName("DescribeDataIngestionJob")
                            .withMarshaller(new DescribeDataIngestionJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeDataIngestionJobRequest));
            CompletableFuture<DescribeDataIngestionJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides a JSON description of the data that is in each time series dataset, including names, column names, and
     * data types.
     * </p>
     *
     * @param describeDatasetRequest
     * @return A Java Future containing the result of the DescribeDataset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy constraints specified by Amazon Lookout for Equipment
     *         or a related AWS service that's being utilized.</li>
     *         <li>ResourceNotFoundException The resource requested could not be found. Verify the resource ID and retry
     *         your request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have access to the
     *         resource.</li>
     *         <li>InternalServerException Processing of the request has failed because of an unknown error, exception
     *         or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutEquipmentException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutEquipmentAsyncClient.DescribeDataset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/DescribeDataset"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDatasetResponse> describeDataset(DescribeDatasetRequest describeDatasetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDatasetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDataset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDatasetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDatasetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDatasetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDatasetRequest, DescribeDatasetResponse>()
                            .withOperationName("DescribeDataset")
                            .withMarshaller(new DescribeDatasetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeDatasetRequest));
            CompletableFuture<DescribeDatasetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Specifies information about the inference scheduler being used, including name, model, status, and associated
     * metadata
     * </p>
     *
     * @param describeInferenceSchedulerRequest
     * @return A Java Future containing the result of the DescribeInferenceScheduler operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy constraints specified by Amazon Lookout for Equipment
     *         or a related AWS service that's being utilized.</li>
     *         <li>ResourceNotFoundException The resource requested could not be found. Verify the resource ID and retry
     *         your request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have access to the
     *         resource.</li>
     *         <li>InternalServerException Processing of the request has failed because of an unknown error, exception
     *         or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutEquipmentException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutEquipmentAsyncClient.DescribeInferenceScheduler
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/DescribeInferenceScheduler"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeInferenceSchedulerResponse> describeInferenceScheduler(
            DescribeInferenceSchedulerRequest describeInferenceSchedulerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeInferenceSchedulerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeInferenceScheduler");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeInferenceSchedulerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeInferenceSchedulerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeInferenceSchedulerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeInferenceSchedulerRequest, DescribeInferenceSchedulerResponse>()
                            .withOperationName("DescribeInferenceScheduler")
                            .withMarshaller(new DescribeInferenceSchedulerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeInferenceSchedulerRequest));
            CompletableFuture<DescribeInferenceSchedulerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides a JSON containing the overall information about a specific ML model, including model name and ARN,
     * dataset, training and evaluation information, status, and so on.
     * </p>
     *
     * @param describeModelRequest
     * @return A Java Future containing the result of the DescribeModel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy constraints specified by Amazon Lookout for Equipment
     *         or a related AWS service that's being utilized.</li>
     *         <li>ResourceNotFoundException The resource requested could not be found. Verify the resource ID and retry
     *         your request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have access to the
     *         resource.</li>
     *         <li>InternalServerException Processing of the request has failed because of an unknown error, exception
     *         or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutEquipmentException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutEquipmentAsyncClient.DescribeModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/DescribeModel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeModelResponse> describeModel(DescribeModelRequest describeModelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeModel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeModelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeModelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeModelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeModelRequest, DescribeModelResponse>()
                            .withOperationName("DescribeModel")
                            .withMarshaller(new DescribeModelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeModelRequest));
            CompletableFuture<DescribeModelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides a list of all data ingestion jobs, including dataset name and ARN, S3 location of the input data,
     * status, and so on.
     * </p>
     *
     * @param listDataIngestionJobsRequest
     * @return A Java Future containing the result of the ListDataIngestionJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy constraints specified by Amazon Lookout for Equipment
     *         or a related AWS service that's being utilized.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have access to the
     *         resource.</li>
     *         <li>InternalServerException Processing of the request has failed because of an unknown error, exception
     *         or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutEquipmentException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutEquipmentAsyncClient.ListDataIngestionJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/ListDataIngestionJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDataIngestionJobsResponse> listDataIngestionJobs(
            ListDataIngestionJobsRequest listDataIngestionJobsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDataIngestionJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDataIngestionJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDataIngestionJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDataIngestionJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDataIngestionJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDataIngestionJobsRequest, ListDataIngestionJobsResponse>()
                            .withOperationName("ListDataIngestionJobs")
                            .withMarshaller(new ListDataIngestionJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listDataIngestionJobsRequest));
            CompletableFuture<ListDataIngestionJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides a list of all data ingestion jobs, including dataset name and ARN, S3 location of the input data,
     * status, and so on.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listDataIngestionJobs(software.amazon.awssdk.services.lookoutequipment.model.ListDataIngestionJobsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lookoutequipment.paginators.ListDataIngestionJobsPublisher publisher = client.listDataIngestionJobsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lookoutequipment.paginators.ListDataIngestionJobsPublisher publisher = client.listDataIngestionJobsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.lookoutequipment.model.ListDataIngestionJobsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.lookoutequipment.model.ListDataIngestionJobsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDataIngestionJobs(software.amazon.awssdk.services.lookoutequipment.model.ListDataIngestionJobsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listDataIngestionJobsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy constraints specified by Amazon Lookout for Equipment
     *         or a related AWS service that's being utilized.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have access to the
     *         resource.</li>
     *         <li>InternalServerException Processing of the request has failed because of an unknown error, exception
     *         or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutEquipmentException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutEquipmentAsyncClient.ListDataIngestionJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/ListDataIngestionJobs"
     *      target="_top">AWS API Documentation</a>
     */
    public ListDataIngestionJobsPublisher listDataIngestionJobsPaginator(ListDataIngestionJobsRequest listDataIngestionJobsRequest) {
        return new ListDataIngestionJobsPublisher(this, applyPaginatorUserAgent(listDataIngestionJobsRequest));
    }

    /**
     * <p>
     * Lists all datasets currently available in your account, filtering on the dataset name.
     * </p>
     *
     * @param listDatasetsRequest
     * @return A Java Future containing the result of the ListDatasets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy constraints specified by Amazon Lookout for Equipment
     *         or a related AWS service that's being utilized.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have access to the
     *         resource.</li>
     *         <li>InternalServerException Processing of the request has failed because of an unknown error, exception
     *         or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutEquipmentException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutEquipmentAsyncClient.ListDatasets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/ListDatasets" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDatasetsResponse> listDatasets(ListDatasetsRequest listDatasetsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDatasetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDatasets");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDatasetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListDatasetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDatasetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDatasetsRequest, ListDatasetsResponse>()
                            .withOperationName("ListDatasets").withMarshaller(new ListDatasetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listDatasetsRequest));
            CompletableFuture<ListDatasetsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all datasets currently available in your account, filtering on the dataset name.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listDatasets(software.amazon.awssdk.services.lookoutequipment.model.ListDatasetsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lookoutequipment.paginators.ListDatasetsPublisher publisher = client.listDatasetsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lookoutequipment.paginators.ListDatasetsPublisher publisher = client.listDatasetsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.lookoutequipment.model.ListDatasetsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.lookoutequipment.model.ListDatasetsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDatasets(software.amazon.awssdk.services.lookoutequipment.model.ListDatasetsRequest)} operation.</b>
     * </p>
     *
     * @param listDatasetsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy constraints specified by Amazon Lookout for Equipment
     *         or a related AWS service that's being utilized.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have access to the
     *         resource.</li>
     *         <li>InternalServerException Processing of the request has failed because of an unknown error, exception
     *         or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutEquipmentException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutEquipmentAsyncClient.ListDatasets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/ListDatasets" target="_top">AWS
     *      API Documentation</a>
     */
    public ListDatasetsPublisher listDatasetsPaginator(ListDatasetsRequest listDatasetsRequest) {
        return new ListDatasetsPublisher(this, applyPaginatorUserAgent(listDatasetsRequest));
    }

    /**
     * <p>
     * Lists all inference executions that have been performed by the specified inference scheduler.
     * </p>
     *
     * @param listInferenceExecutionsRequest
     * @return A Java Future containing the result of the ListInferenceExecutions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy constraints specified by Amazon Lookout for Equipment
     *         or a related AWS service that's being utilized.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException The resource requested could not be found. Verify the resource ID and retry
     *         your request.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have access to the
     *         resource.</li>
     *         <li>InternalServerException Processing of the request has failed because of an unknown error, exception
     *         or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutEquipmentException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutEquipmentAsyncClient.ListInferenceExecutions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/ListInferenceExecutions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListInferenceExecutionsResponse> listInferenceExecutions(
            ListInferenceExecutionsRequest listInferenceExecutionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInferenceExecutionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInferenceExecutions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListInferenceExecutionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListInferenceExecutionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListInferenceExecutionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListInferenceExecutionsRequest, ListInferenceExecutionsResponse>()
                            .withOperationName("ListInferenceExecutions")
                            .withMarshaller(new ListInferenceExecutionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listInferenceExecutionsRequest));
            CompletableFuture<ListInferenceExecutionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all inference executions that have been performed by the specified inference scheduler.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listInferenceExecutions(software.amazon.awssdk.services.lookoutequipment.model.ListInferenceExecutionsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lookoutequipment.paginators.ListInferenceExecutionsPublisher publisher = client.listInferenceExecutionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lookoutequipment.paginators.ListInferenceExecutionsPublisher publisher = client.listInferenceExecutionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.lookoutequipment.model.ListInferenceExecutionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.lookoutequipment.model.ListInferenceExecutionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listInferenceExecutions(software.amazon.awssdk.services.lookoutequipment.model.ListInferenceExecutionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listInferenceExecutionsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy constraints specified by Amazon Lookout for Equipment
     *         or a related AWS service that's being utilized.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ResourceNotFoundException The resource requested could not be found. Verify the resource ID and retry
     *         your request.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have access to the
     *         resource.</li>
     *         <li>InternalServerException Processing of the request has failed because of an unknown error, exception
     *         or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutEquipmentException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutEquipmentAsyncClient.ListInferenceExecutions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/ListInferenceExecutions"
     *      target="_top">AWS API Documentation</a>
     */
    public ListInferenceExecutionsPublisher listInferenceExecutionsPaginator(
            ListInferenceExecutionsRequest listInferenceExecutionsRequest) {
        return new ListInferenceExecutionsPublisher(this, applyPaginatorUserAgent(listInferenceExecutionsRequest));
    }

    /**
     * <p>
     * Retrieves a list of all inference schedulers currently available for your account.
     * </p>
     *
     * @param listInferenceSchedulersRequest
     * @return A Java Future containing the result of the ListInferenceSchedulers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy constraints specified by Amazon Lookout for Equipment
     *         or a related AWS service that's being utilized.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have access to the
     *         resource.</li>
     *         <li>InternalServerException Processing of the request has failed because of an unknown error, exception
     *         or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutEquipmentException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutEquipmentAsyncClient.ListInferenceSchedulers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/ListInferenceSchedulers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListInferenceSchedulersResponse> listInferenceSchedulers(
            ListInferenceSchedulersRequest listInferenceSchedulersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInferenceSchedulersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInferenceSchedulers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListInferenceSchedulersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListInferenceSchedulersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListInferenceSchedulersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListInferenceSchedulersRequest, ListInferenceSchedulersResponse>()
                            .withOperationName("ListInferenceSchedulers")
                            .withMarshaller(new ListInferenceSchedulersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listInferenceSchedulersRequest));
            CompletableFuture<ListInferenceSchedulersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a list of all inference schedulers currently available for your account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listInferenceSchedulers(software.amazon.awssdk.services.lookoutequipment.model.ListInferenceSchedulersRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lookoutequipment.paginators.ListInferenceSchedulersPublisher publisher = client.listInferenceSchedulersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lookoutequipment.paginators.ListInferenceSchedulersPublisher publisher = client.listInferenceSchedulersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.lookoutequipment.model.ListInferenceSchedulersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.lookoutequipment.model.ListInferenceSchedulersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listInferenceSchedulers(software.amazon.awssdk.services.lookoutequipment.model.ListInferenceSchedulersRequest)}
     * operation.</b>
     * </p>
     *
     * @param listInferenceSchedulersRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy constraints specified by Amazon Lookout for Equipment
     *         or a related AWS service that's being utilized.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have access to the
     *         resource.</li>
     *         <li>InternalServerException Processing of the request has failed because of an unknown error, exception
     *         or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutEquipmentException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutEquipmentAsyncClient.ListInferenceSchedulers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/ListInferenceSchedulers"
     *      target="_top">AWS API Documentation</a>
     */
    public ListInferenceSchedulersPublisher listInferenceSchedulersPaginator(
            ListInferenceSchedulersRequest listInferenceSchedulersRequest) {
        return new ListInferenceSchedulersPublisher(this, applyPaginatorUserAgent(listInferenceSchedulersRequest));
    }

    /**
     * <p>
     * Generates a list of all models in the account, including model name and ARN, dataset, and status.
     * </p>
     *
     * @param listModelsRequest
     * @return A Java Future containing the result of the ListModels operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy constraints specified by Amazon Lookout for Equipment
     *         or a related AWS service that's being utilized.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have access to the
     *         resource.</li>
     *         <li>InternalServerException Processing of the request has failed because of an unknown error, exception
     *         or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutEquipmentException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutEquipmentAsyncClient.ListModels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/ListModels" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListModelsResponse> listModels(ListModelsRequest listModelsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listModelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListModels");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListModelsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListModelsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListModelsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListModelsRequest, ListModelsResponse>().withOperationName("ListModels")
                            .withMarshaller(new ListModelsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listModelsRequest));
            CompletableFuture<ListModelsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Generates a list of all models in the account, including model name and ARN, dataset, and status.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listModels(software.amazon.awssdk.services.lookoutequipment.model.ListModelsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lookoutequipment.paginators.ListModelsPublisher publisher = client.listModelsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lookoutequipment.paginators.ListModelsPublisher publisher = client.listModelsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.lookoutequipment.model.ListModelsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.lookoutequipment.model.ListModelsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listModels(software.amazon.awssdk.services.lookoutequipment.model.ListModelsRequest)} operation.</b>
     * </p>
     *
     * @param listModelsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy constraints specified by Amazon Lookout for Equipment
     *         or a related AWS service that's being utilized.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have access to the
     *         resource.</li>
     *         <li>InternalServerException Processing of the request has failed because of an unknown error, exception
     *         or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutEquipmentException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutEquipmentAsyncClient.ListModels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/ListModels" target="_top">AWS
     *      API Documentation</a>
     */
    public ListModelsPublisher listModelsPaginator(ListModelsRequest listModelsRequest) {
        return new ListModelsPublisher(this, applyPaginatorUserAgent(listModelsRequest));
    }

    /**
     * <p>
     * Lists all the tags for a specified resource, including key and value.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy constraints specified by Amazon Lookout for Equipment
     *         or a related AWS service that's being utilized.</li>
     *         <li>ResourceNotFoundException The resource requested could not be found. Verify the resource ID and retry
     *         your request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have access to the
     *         resource.</li>
     *         <li>InternalServerException Processing of the request has failed because of an unknown error, exception
     *         or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutEquipmentException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutEquipmentAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts a data ingestion job. Amazon Lookout for Equipment returns the job status.
     * </p>
     *
     * @param startDataIngestionJobRequest
     * @return A Java Future containing the result of the StartDataIngestionJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy constraints specified by Amazon Lookout for Equipment
     *         or a related AWS service that's being utilized.</li>
     *         <li>ResourceNotFoundException The resource requested could not be found. Verify the resource ID and retry
     *         your request.</li>
     *         <li>ConflictException The request could not be completed due to a conflict with the current state of the
     *         target resource.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ServiceQuotaExceededException Resource limitations have been exceeded.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have access to the
     *         resource.</li>
     *         <li>InternalServerException Processing of the request has failed because of an unknown error, exception
     *         or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutEquipmentException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutEquipmentAsyncClient.StartDataIngestionJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/StartDataIngestionJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartDataIngestionJobResponse> startDataIngestionJob(
            StartDataIngestionJobRequest startDataIngestionJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startDataIngestionJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartDataIngestionJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartDataIngestionJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartDataIngestionJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartDataIngestionJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartDataIngestionJobRequest, StartDataIngestionJobResponse>()
                            .withOperationName("StartDataIngestionJob")
                            .withMarshaller(new StartDataIngestionJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startDataIngestionJobRequest));
            CompletableFuture<StartDataIngestionJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts an inference scheduler.
     * </p>
     *
     * @param startInferenceSchedulerRequest
     * @return A Java Future containing the result of the StartInferenceScheduler operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy constraints specified by Amazon Lookout for Equipment
     *         or a related AWS service that's being utilized.</li>
     *         <li>ConflictException The request could not be completed due to a conflict with the current state of the
     *         target resource.</li>
     *         <li>ResourceNotFoundException The resource requested could not be found. Verify the resource ID and retry
     *         your request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have access to the
     *         resource.</li>
     *         <li>InternalServerException Processing of the request has failed because of an unknown error, exception
     *         or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutEquipmentException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutEquipmentAsyncClient.StartInferenceScheduler
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/StartInferenceScheduler"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartInferenceSchedulerResponse> startInferenceScheduler(
            StartInferenceSchedulerRequest startInferenceSchedulerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startInferenceSchedulerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartInferenceScheduler");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartInferenceSchedulerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartInferenceSchedulerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartInferenceSchedulerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartInferenceSchedulerRequest, StartInferenceSchedulerResponse>()
                            .withOperationName("StartInferenceScheduler")
                            .withMarshaller(new StartInferenceSchedulerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startInferenceSchedulerRequest));
            CompletableFuture<StartInferenceSchedulerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops an inference scheduler.
     * </p>
     *
     * @param stopInferenceSchedulerRequest
     * @return A Java Future containing the result of the StopInferenceScheduler operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy constraints specified by Amazon Lookout for Equipment
     *         or a related AWS service that's being utilized.</li>
     *         <li>ConflictException The request could not be completed due to a conflict with the current state of the
     *         target resource.</li>
     *         <li>ResourceNotFoundException The resource requested could not be found. Verify the resource ID and retry
     *         your request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have access to the
     *         resource.</li>
     *         <li>InternalServerException Processing of the request has failed because of an unknown error, exception
     *         or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutEquipmentException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutEquipmentAsyncClient.StopInferenceScheduler
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/StopInferenceScheduler"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StopInferenceSchedulerResponse> stopInferenceScheduler(
            StopInferenceSchedulerRequest stopInferenceSchedulerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopInferenceSchedulerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopInferenceScheduler");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopInferenceSchedulerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StopInferenceSchedulerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopInferenceSchedulerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopInferenceSchedulerRequest, StopInferenceSchedulerResponse>()
                            .withOperationName("StopInferenceScheduler")
                            .withMarshaller(new StopInferenceSchedulerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(stopInferenceSchedulerRequest));
            CompletableFuture<StopInferenceSchedulerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Associates a given tag to a resource in your account. A tag is a key-value pair which can be added to an Amazon
     * Lookout for Equipment resource as metadata. Tags can be used for organizing your resources as well as helping you
     * to search and filter by tag. Multiple tags can be added to a resource, either when you create it, or later. Up to
     * 50 tags can be associated with each resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy constraints specified by Amazon Lookout for Equipment
     *         or a related AWS service that's being utilized.</li>
     *         <li>ResourceNotFoundException The resource requested could not be found. Verify the resource ID and retry
     *         your request.</li>
     *         <li>ServiceQuotaExceededException Resource limitations have been exceeded.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have access to the
     *         resource.</li>
     *         <li>InternalServerException Processing of the request has failed because of an unknown error, exception
     *         or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutEquipmentException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutEquipmentAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a specific tag from a given resource. The tag is specified by its key.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy constraints specified by Amazon Lookout for Equipment
     *         or a related AWS service that's being utilized.</li>
     *         <li>ResourceNotFoundException The resource requested could not be found. Verify the resource ID and retry
     *         your request.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have access to the
     *         resource.</li>
     *         <li>InternalServerException Processing of the request has failed because of an unknown error, exception
     *         or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutEquipmentException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutEquipmentAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an inference scheduler.
     * </p>
     *
     * @param updateInferenceSchedulerRequest
     * @return A Java Future containing the result of the UpdateInferenceScheduler operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException The request could not be completed due to a conflict with the current state of the
     *         target resource.</li>
     *         <li>ResourceNotFoundException The resource requested could not be found. Verify the resource ID and retry
     *         your request.</li>
     *         <li>ValidationException The input fails to satisfy constraints specified by Amazon Lookout for Equipment
     *         or a related AWS service that's being utilized.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException The request could not be completed because you do not have access to the
     *         resource.</li>
     *         <li>InternalServerException Processing of the request has failed because of an unknown error, exception
     *         or failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutEquipmentException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutEquipmentAsyncClient.UpdateInferenceScheduler
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutequipment-2020-12-15/UpdateInferenceScheduler"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateInferenceSchedulerResponse> updateInferenceScheduler(
            UpdateInferenceSchedulerRequest updateInferenceSchedulerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateInferenceSchedulerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutEquipment");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateInferenceScheduler");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateInferenceSchedulerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateInferenceSchedulerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateInferenceSchedulerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateInferenceSchedulerRequest, UpdateInferenceSchedulerResponse>()
                            .withOperationName("UpdateInferenceScheduler")
                            .withMarshaller(new UpdateInferenceSchedulerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateInferenceSchedulerRequest));
            CompletableFuture<UpdateInferenceSchedulerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(LookoutEquipmentException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends LookoutEquipmentRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
