/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lookoutequipment.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Gives statistics about how many files have been ingested, and which files have not been ingested, for a particular
 * ingestion job.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class IngestedFilesSummary implements SdkPojo, Serializable,
        ToCopyableBuilder<IngestedFilesSummary.Builder, IngestedFilesSummary> {
    private static final SdkField<Integer> TOTAL_NUMBER_OF_FILES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("TotalNumberOfFiles").getter(getter(IngestedFilesSummary::totalNumberOfFiles))
            .setter(setter(Builder::totalNumberOfFiles))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TotalNumberOfFiles").build())
            .build();

    private static final SdkField<Integer> INGESTED_NUMBER_OF_FILES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("IngestedNumberOfFiles").getter(getter(IngestedFilesSummary::ingestedNumberOfFiles))
            .setter(setter(Builder::ingestedNumberOfFiles))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IngestedNumberOfFiles").build())
            .build();

    private static final SdkField<List<S3Object>> DISCARDED_FILES_FIELD = SdkField
            .<List<S3Object>> builder(MarshallingType.LIST)
            .memberName("DiscardedFiles")
            .getter(getter(IngestedFilesSummary::discardedFiles))
            .setter(setter(Builder::discardedFiles))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DiscardedFiles").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<S3Object> builder(MarshallingType.SDK_POJO)
                                            .constructor(S3Object::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TOTAL_NUMBER_OF_FILES_FIELD,
            INGESTED_NUMBER_OF_FILES_FIELD, DISCARDED_FILES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Integer totalNumberOfFiles;

    private final Integer ingestedNumberOfFiles;

    private final List<S3Object> discardedFiles;

    private IngestedFilesSummary(BuilderImpl builder) {
        this.totalNumberOfFiles = builder.totalNumberOfFiles;
        this.ingestedNumberOfFiles = builder.ingestedNumberOfFiles;
        this.discardedFiles = builder.discardedFiles;
    }

    /**
     * <p>
     * Indicates the total number of files that were submitted for ingestion.
     * </p>
     * 
     * @return Indicates the total number of files that were submitted for ingestion.
     */
    public final Integer totalNumberOfFiles() {
        return totalNumberOfFiles;
    }

    /**
     * <p>
     * Indicates the number of files that were successfully ingested.
     * </p>
     * 
     * @return Indicates the number of files that were successfully ingested.
     */
    public final Integer ingestedNumberOfFiles() {
        return ingestedNumberOfFiles;
    }

    /**
     * For responses, this returns true if the service returned a value for the DiscardedFiles property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasDiscardedFiles() {
        return discardedFiles != null && !(discardedFiles instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Indicates the number of files that were discarded. A file could be discarded because its format is invalid (for
     * example, a jpg or pdf) or not readable.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDiscardedFiles} method.
     * </p>
     * 
     * @return Indicates the number of files that were discarded. A file could be discarded because its format is
     *         invalid (for example, a jpg or pdf) or not readable.
     */
    public final List<S3Object> discardedFiles() {
        return discardedFiles;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(totalNumberOfFiles());
        hashCode = 31 * hashCode + Objects.hashCode(ingestedNumberOfFiles());
        hashCode = 31 * hashCode + Objects.hashCode(hasDiscardedFiles() ? discardedFiles() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof IngestedFilesSummary)) {
            return false;
        }
        IngestedFilesSummary other = (IngestedFilesSummary) obj;
        return Objects.equals(totalNumberOfFiles(), other.totalNumberOfFiles())
                && Objects.equals(ingestedNumberOfFiles(), other.ingestedNumberOfFiles())
                && hasDiscardedFiles() == other.hasDiscardedFiles() && Objects.equals(discardedFiles(), other.discardedFiles());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("IngestedFilesSummary").add("TotalNumberOfFiles", totalNumberOfFiles())
                .add("IngestedNumberOfFiles", ingestedNumberOfFiles())
                .add("DiscardedFiles", hasDiscardedFiles() ? discardedFiles() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TotalNumberOfFiles":
            return Optional.ofNullable(clazz.cast(totalNumberOfFiles()));
        case "IngestedNumberOfFiles":
            return Optional.ofNullable(clazz.cast(ingestedNumberOfFiles()));
        case "DiscardedFiles":
            return Optional.ofNullable(clazz.cast(discardedFiles()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("TotalNumberOfFiles", TOTAL_NUMBER_OF_FILES_FIELD);
        map.put("IngestedNumberOfFiles", INGESTED_NUMBER_OF_FILES_FIELD);
        map.put("DiscardedFiles", DISCARDED_FILES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<IngestedFilesSummary, T> g) {
        return obj -> g.apply((IngestedFilesSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, IngestedFilesSummary> {
        /**
         * <p>
         * Indicates the total number of files that were submitted for ingestion.
         * </p>
         * 
         * @param totalNumberOfFiles
         *        Indicates the total number of files that were submitted for ingestion.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder totalNumberOfFiles(Integer totalNumberOfFiles);

        /**
         * <p>
         * Indicates the number of files that were successfully ingested.
         * </p>
         * 
         * @param ingestedNumberOfFiles
         *        Indicates the number of files that were successfully ingested.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ingestedNumberOfFiles(Integer ingestedNumberOfFiles);

        /**
         * <p>
         * Indicates the number of files that were discarded. A file could be discarded because its format is invalid
         * (for example, a jpg or pdf) or not readable.
         * </p>
         * 
         * @param discardedFiles
         *        Indicates the number of files that were discarded. A file could be discarded because its format is
         *        invalid (for example, a jpg or pdf) or not readable.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder discardedFiles(Collection<S3Object> discardedFiles);

        /**
         * <p>
         * Indicates the number of files that were discarded. A file could be discarded because its format is invalid
         * (for example, a jpg or pdf) or not readable.
         * </p>
         * 
         * @param discardedFiles
         *        Indicates the number of files that were discarded. A file could be discarded because its format is
         *        invalid (for example, a jpg or pdf) or not readable.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder discardedFiles(S3Object... discardedFiles);

        /**
         * <p>
         * Indicates the number of files that were discarded. A file could be discarded because its format is invalid
         * (for example, a jpg or pdf) or not readable.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.lookoutequipment.model.S3Object.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.lookoutequipment.model.S3Object#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.lookoutequipment.model.S3Object.Builder#build()} is called immediately
         * and its result is passed to {@link #discardedFiles(List<S3Object>)}.
         * 
         * @param discardedFiles
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.lookoutequipment.model.S3Object.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #discardedFiles(java.util.Collection<S3Object>)
         */
        Builder discardedFiles(Consumer<S3Object.Builder>... discardedFiles);
    }

    static final class BuilderImpl implements Builder {
        private Integer totalNumberOfFiles;

        private Integer ingestedNumberOfFiles;

        private List<S3Object> discardedFiles = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(IngestedFilesSummary model) {
            totalNumberOfFiles(model.totalNumberOfFiles);
            ingestedNumberOfFiles(model.ingestedNumberOfFiles);
            discardedFiles(model.discardedFiles);
        }

        public final Integer getTotalNumberOfFiles() {
            return totalNumberOfFiles;
        }

        public final void setTotalNumberOfFiles(Integer totalNumberOfFiles) {
            this.totalNumberOfFiles = totalNumberOfFiles;
        }

        @Override
        public final Builder totalNumberOfFiles(Integer totalNumberOfFiles) {
            this.totalNumberOfFiles = totalNumberOfFiles;
            return this;
        }

        public final Integer getIngestedNumberOfFiles() {
            return ingestedNumberOfFiles;
        }

        public final void setIngestedNumberOfFiles(Integer ingestedNumberOfFiles) {
            this.ingestedNumberOfFiles = ingestedNumberOfFiles;
        }

        @Override
        public final Builder ingestedNumberOfFiles(Integer ingestedNumberOfFiles) {
            this.ingestedNumberOfFiles = ingestedNumberOfFiles;
            return this;
        }

        public final List<S3Object.Builder> getDiscardedFiles() {
            List<S3Object.Builder> result = ListOfDiscardedFilesCopier.copyToBuilder(this.discardedFiles);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setDiscardedFiles(Collection<S3Object.BuilderImpl> discardedFiles) {
            this.discardedFiles = ListOfDiscardedFilesCopier.copyFromBuilder(discardedFiles);
        }

        @Override
        public final Builder discardedFiles(Collection<S3Object> discardedFiles) {
            this.discardedFiles = ListOfDiscardedFilesCopier.copy(discardedFiles);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder discardedFiles(S3Object... discardedFiles) {
            discardedFiles(Arrays.asList(discardedFiles));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder discardedFiles(Consumer<S3Object.Builder>... discardedFiles) {
            discardedFiles(Stream.of(discardedFiles).map(c -> S3Object.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public IngestedFilesSummary build() {
            return new IngestedFilesSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
