/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lookoutmetrics;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.lookoutmetrics.model.AccessDeniedException;
import software.amazon.awssdk.services.lookoutmetrics.model.ActivateAnomalyDetectorRequest;
import software.amazon.awssdk.services.lookoutmetrics.model.ActivateAnomalyDetectorResponse;
import software.amazon.awssdk.services.lookoutmetrics.model.BackTestAnomalyDetectorRequest;
import software.amazon.awssdk.services.lookoutmetrics.model.BackTestAnomalyDetectorResponse;
import software.amazon.awssdk.services.lookoutmetrics.model.ConflictException;
import software.amazon.awssdk.services.lookoutmetrics.model.CreateAlertRequest;
import software.amazon.awssdk.services.lookoutmetrics.model.CreateAlertResponse;
import software.amazon.awssdk.services.lookoutmetrics.model.CreateAnomalyDetectorRequest;
import software.amazon.awssdk.services.lookoutmetrics.model.CreateAnomalyDetectorResponse;
import software.amazon.awssdk.services.lookoutmetrics.model.CreateMetricSetRequest;
import software.amazon.awssdk.services.lookoutmetrics.model.CreateMetricSetResponse;
import software.amazon.awssdk.services.lookoutmetrics.model.DeactivateAnomalyDetectorRequest;
import software.amazon.awssdk.services.lookoutmetrics.model.DeactivateAnomalyDetectorResponse;
import software.amazon.awssdk.services.lookoutmetrics.model.DeleteAlertRequest;
import software.amazon.awssdk.services.lookoutmetrics.model.DeleteAlertResponse;
import software.amazon.awssdk.services.lookoutmetrics.model.DeleteAnomalyDetectorRequest;
import software.amazon.awssdk.services.lookoutmetrics.model.DeleteAnomalyDetectorResponse;
import software.amazon.awssdk.services.lookoutmetrics.model.DescribeAlertRequest;
import software.amazon.awssdk.services.lookoutmetrics.model.DescribeAlertResponse;
import software.amazon.awssdk.services.lookoutmetrics.model.DescribeAnomalyDetectionExecutionsRequest;
import software.amazon.awssdk.services.lookoutmetrics.model.DescribeAnomalyDetectionExecutionsResponse;
import software.amazon.awssdk.services.lookoutmetrics.model.DescribeAnomalyDetectorRequest;
import software.amazon.awssdk.services.lookoutmetrics.model.DescribeAnomalyDetectorResponse;
import software.amazon.awssdk.services.lookoutmetrics.model.DescribeMetricSetRequest;
import software.amazon.awssdk.services.lookoutmetrics.model.DescribeMetricSetResponse;
import software.amazon.awssdk.services.lookoutmetrics.model.DetectMetricSetConfigRequest;
import software.amazon.awssdk.services.lookoutmetrics.model.DetectMetricSetConfigResponse;
import software.amazon.awssdk.services.lookoutmetrics.model.GetAnomalyGroupRequest;
import software.amazon.awssdk.services.lookoutmetrics.model.GetAnomalyGroupResponse;
import software.amazon.awssdk.services.lookoutmetrics.model.GetDataQualityMetricsRequest;
import software.amazon.awssdk.services.lookoutmetrics.model.GetDataQualityMetricsResponse;
import software.amazon.awssdk.services.lookoutmetrics.model.GetFeedbackRequest;
import software.amazon.awssdk.services.lookoutmetrics.model.GetFeedbackResponse;
import software.amazon.awssdk.services.lookoutmetrics.model.GetSampleDataRequest;
import software.amazon.awssdk.services.lookoutmetrics.model.GetSampleDataResponse;
import software.amazon.awssdk.services.lookoutmetrics.model.InternalServerException;
import software.amazon.awssdk.services.lookoutmetrics.model.ListAlertsRequest;
import software.amazon.awssdk.services.lookoutmetrics.model.ListAlertsResponse;
import software.amazon.awssdk.services.lookoutmetrics.model.ListAnomalyDetectorsRequest;
import software.amazon.awssdk.services.lookoutmetrics.model.ListAnomalyDetectorsResponse;
import software.amazon.awssdk.services.lookoutmetrics.model.ListAnomalyGroupRelatedMetricsRequest;
import software.amazon.awssdk.services.lookoutmetrics.model.ListAnomalyGroupRelatedMetricsResponse;
import software.amazon.awssdk.services.lookoutmetrics.model.ListAnomalyGroupSummariesRequest;
import software.amazon.awssdk.services.lookoutmetrics.model.ListAnomalyGroupSummariesResponse;
import software.amazon.awssdk.services.lookoutmetrics.model.ListAnomalyGroupTimeSeriesRequest;
import software.amazon.awssdk.services.lookoutmetrics.model.ListAnomalyGroupTimeSeriesResponse;
import software.amazon.awssdk.services.lookoutmetrics.model.ListMetricSetsRequest;
import software.amazon.awssdk.services.lookoutmetrics.model.ListMetricSetsResponse;
import software.amazon.awssdk.services.lookoutmetrics.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.lookoutmetrics.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.lookoutmetrics.model.LookoutMetricsException;
import software.amazon.awssdk.services.lookoutmetrics.model.LookoutMetricsRequest;
import software.amazon.awssdk.services.lookoutmetrics.model.PutFeedbackRequest;
import software.amazon.awssdk.services.lookoutmetrics.model.PutFeedbackResponse;
import software.amazon.awssdk.services.lookoutmetrics.model.ResourceNotFoundException;
import software.amazon.awssdk.services.lookoutmetrics.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.lookoutmetrics.model.TagResourceRequest;
import software.amazon.awssdk.services.lookoutmetrics.model.TagResourceResponse;
import software.amazon.awssdk.services.lookoutmetrics.model.TooManyRequestsException;
import software.amazon.awssdk.services.lookoutmetrics.model.UntagResourceRequest;
import software.amazon.awssdk.services.lookoutmetrics.model.UntagResourceResponse;
import software.amazon.awssdk.services.lookoutmetrics.model.UpdateAlertRequest;
import software.amazon.awssdk.services.lookoutmetrics.model.UpdateAlertResponse;
import software.amazon.awssdk.services.lookoutmetrics.model.UpdateAnomalyDetectorRequest;
import software.amazon.awssdk.services.lookoutmetrics.model.UpdateAnomalyDetectorResponse;
import software.amazon.awssdk.services.lookoutmetrics.model.UpdateMetricSetRequest;
import software.amazon.awssdk.services.lookoutmetrics.model.UpdateMetricSetResponse;
import software.amazon.awssdk.services.lookoutmetrics.model.ValidationException;
import software.amazon.awssdk.services.lookoutmetrics.paginators.DescribeAnomalyDetectionExecutionsPublisher;
import software.amazon.awssdk.services.lookoutmetrics.paginators.GetFeedbackPublisher;
import software.amazon.awssdk.services.lookoutmetrics.paginators.ListAlertsPublisher;
import software.amazon.awssdk.services.lookoutmetrics.paginators.ListAnomalyDetectorsPublisher;
import software.amazon.awssdk.services.lookoutmetrics.paginators.ListAnomalyGroupRelatedMetricsPublisher;
import software.amazon.awssdk.services.lookoutmetrics.paginators.ListAnomalyGroupSummariesPublisher;
import software.amazon.awssdk.services.lookoutmetrics.paginators.ListAnomalyGroupTimeSeriesPublisher;
import software.amazon.awssdk.services.lookoutmetrics.paginators.ListMetricSetsPublisher;
import software.amazon.awssdk.services.lookoutmetrics.transform.ActivateAnomalyDetectorRequestMarshaller;
import software.amazon.awssdk.services.lookoutmetrics.transform.BackTestAnomalyDetectorRequestMarshaller;
import software.amazon.awssdk.services.lookoutmetrics.transform.CreateAlertRequestMarshaller;
import software.amazon.awssdk.services.lookoutmetrics.transform.CreateAnomalyDetectorRequestMarshaller;
import software.amazon.awssdk.services.lookoutmetrics.transform.CreateMetricSetRequestMarshaller;
import software.amazon.awssdk.services.lookoutmetrics.transform.DeactivateAnomalyDetectorRequestMarshaller;
import software.amazon.awssdk.services.lookoutmetrics.transform.DeleteAlertRequestMarshaller;
import software.amazon.awssdk.services.lookoutmetrics.transform.DeleteAnomalyDetectorRequestMarshaller;
import software.amazon.awssdk.services.lookoutmetrics.transform.DescribeAlertRequestMarshaller;
import software.amazon.awssdk.services.lookoutmetrics.transform.DescribeAnomalyDetectionExecutionsRequestMarshaller;
import software.amazon.awssdk.services.lookoutmetrics.transform.DescribeAnomalyDetectorRequestMarshaller;
import software.amazon.awssdk.services.lookoutmetrics.transform.DescribeMetricSetRequestMarshaller;
import software.amazon.awssdk.services.lookoutmetrics.transform.DetectMetricSetConfigRequestMarshaller;
import software.amazon.awssdk.services.lookoutmetrics.transform.GetAnomalyGroupRequestMarshaller;
import software.amazon.awssdk.services.lookoutmetrics.transform.GetDataQualityMetricsRequestMarshaller;
import software.amazon.awssdk.services.lookoutmetrics.transform.GetFeedbackRequestMarshaller;
import software.amazon.awssdk.services.lookoutmetrics.transform.GetSampleDataRequestMarshaller;
import software.amazon.awssdk.services.lookoutmetrics.transform.ListAlertsRequestMarshaller;
import software.amazon.awssdk.services.lookoutmetrics.transform.ListAnomalyDetectorsRequestMarshaller;
import software.amazon.awssdk.services.lookoutmetrics.transform.ListAnomalyGroupRelatedMetricsRequestMarshaller;
import software.amazon.awssdk.services.lookoutmetrics.transform.ListAnomalyGroupSummariesRequestMarshaller;
import software.amazon.awssdk.services.lookoutmetrics.transform.ListAnomalyGroupTimeSeriesRequestMarshaller;
import software.amazon.awssdk.services.lookoutmetrics.transform.ListMetricSetsRequestMarshaller;
import software.amazon.awssdk.services.lookoutmetrics.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.lookoutmetrics.transform.PutFeedbackRequestMarshaller;
import software.amazon.awssdk.services.lookoutmetrics.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.lookoutmetrics.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.lookoutmetrics.transform.UpdateAlertRequestMarshaller;
import software.amazon.awssdk.services.lookoutmetrics.transform.UpdateAnomalyDetectorRequestMarshaller;
import software.amazon.awssdk.services.lookoutmetrics.transform.UpdateMetricSetRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link LookoutMetricsAsyncClient}.
 *
 * @see LookoutMetricsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultLookoutMetricsAsyncClient implements LookoutMetricsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultLookoutMetricsAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultLookoutMetricsAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Activates an anomaly detector.
     * </p>
     *
     * @param activateAnomalyDetectorRequest
     * @return A Java Future containing the result of the ActivateAnomalyDetector operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the AWS service. Check
     *         your input values and try again.</li>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the ARN of the resource and
     *         try again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>TooManyRequestsException The request was denied due to too many requests being submitted at the same
     *         time.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ConflictException There was a conflict processing the request. Try your request again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutMetricsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutMetricsAsyncClient.ActivateAnomalyDetector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutmetrics-2017-07-25/ActivateAnomalyDetector"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ActivateAnomalyDetectorResponse> activateAnomalyDetector(
            ActivateAnomalyDetectorRequest activateAnomalyDetectorRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, activateAnomalyDetectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutMetrics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ActivateAnomalyDetector");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ActivateAnomalyDetectorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ActivateAnomalyDetectorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ActivateAnomalyDetectorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ActivateAnomalyDetectorRequest, ActivateAnomalyDetectorResponse>()
                            .withOperationName("ActivateAnomalyDetector")
                            .withMarshaller(new ActivateAnomalyDetectorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(activateAnomalyDetectorRequest));
            CompletableFuture<ActivateAnomalyDetectorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Runs a backtest for anomaly detection for the specified resource.
     * </p>
     *
     * @param backTestAnomalyDetectorRequest
     * @return A Java Future containing the result of the BackTestAnomalyDetector operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the AWS service. Check
     *         your input values and try again.</li>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the ARN of the resource and
     *         try again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>TooManyRequestsException The request was denied due to too many requests being submitted at the same
     *         time.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutMetricsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutMetricsAsyncClient.BackTestAnomalyDetector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutmetrics-2017-07-25/BackTestAnomalyDetector"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BackTestAnomalyDetectorResponse> backTestAnomalyDetector(
            BackTestAnomalyDetectorRequest backTestAnomalyDetectorRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, backTestAnomalyDetectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutMetrics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BackTestAnomalyDetector");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BackTestAnomalyDetectorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BackTestAnomalyDetectorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BackTestAnomalyDetectorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BackTestAnomalyDetectorRequest, BackTestAnomalyDetectorResponse>()
                            .withOperationName("BackTestAnomalyDetector")
                            .withMarshaller(new BackTestAnomalyDetectorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(backTestAnomalyDetectorRequest));
            CompletableFuture<BackTestAnomalyDetectorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an alert for an anomaly detector.
     * </p>
     *
     * @param createAlertRequest
     * @return A Java Future containing the result of the CreateAlert operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException There was a conflict processing the request. Try your request again.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the AWS service. Check
     *         your input values and try again.</li>
     *         <li>ServiceQuotaExceededException The request exceeded the service's quotas. Check the service quotas and
     *         try again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>TooManyRequestsException The request was denied due to too many requests being submitted at the same
     *         time.</li>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the ARN of the resource and
     *         try again.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutMetricsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutMetricsAsyncClient.CreateAlert
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutmetrics-2017-07-25/CreateAlert" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAlertResponse> createAlert(CreateAlertRequest createAlertRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAlertRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutMetrics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAlert");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAlertResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateAlertResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateAlertResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAlertRequest, CreateAlertResponse>()
                            .withOperationName("CreateAlert").withMarshaller(new CreateAlertRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createAlertRequest));
            CompletableFuture<CreateAlertResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an anomaly detector.
     * </p>
     *
     * @param createAnomalyDetectorRequest
     * @return A Java Future containing the result of the CreateAnomalyDetector operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException There was a conflict processing the request. Try your request again.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the AWS service. Check
     *         your input values and try again.</li>
     *         <li>ServiceQuotaExceededException The request exceeded the service's quotas. Check the service quotas and
     *         try again.</li>
     *         <li>ConflictException There was a conflict processing the request. Try your request again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>TooManyRequestsException The request was denied due to too many requests being submitted at the same
     *         time.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutMetricsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutMetricsAsyncClient.CreateAnomalyDetector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutmetrics-2017-07-25/CreateAnomalyDetector"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAnomalyDetectorResponse> createAnomalyDetector(
            CreateAnomalyDetectorRequest createAnomalyDetectorRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAnomalyDetectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutMetrics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAnomalyDetector");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAnomalyDetectorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateAnomalyDetectorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateAnomalyDetectorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAnomalyDetectorRequest, CreateAnomalyDetectorResponse>()
                            .withOperationName("CreateAnomalyDetector")
                            .withMarshaller(new CreateAnomalyDetectorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createAnomalyDetectorRequest));
            CompletableFuture<CreateAnomalyDetectorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a dataset.
     * </p>
     *
     * @param createMetricSetRequest
     * @return A Java Future containing the result of the CreateMetricSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException There was a conflict processing the request. Try your request again.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the AWS service. Check
     *         your input values and try again.</li>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the ARN of the resource and
     *         try again.</li>
     *         <li>ServiceQuotaExceededException The request exceeded the service's quotas. Check the service quotas and
     *         try again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>TooManyRequestsException The request was denied due to too many requests being submitted at the same
     *         time.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutMetricsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutMetricsAsyncClient.CreateMetricSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutmetrics-2017-07-25/CreateMetricSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateMetricSetResponse> createMetricSet(CreateMetricSetRequest createMetricSetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMetricSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutMetrics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMetricSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateMetricSetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateMetricSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateMetricSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateMetricSetRequest, CreateMetricSetResponse>()
                            .withOperationName("CreateMetricSet")
                            .withMarshaller(new CreateMetricSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createMetricSetRequest));
            CompletableFuture<CreateMetricSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deactivates an anomaly detector.
     * </p>
     *
     * @param deactivateAnomalyDetectorRequest
     * @return A Java Future containing the result of the DeactivateAnomalyDetector operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the AWS service. Check
     *         your input values and try again.</li>
     *         <li>ConflictException There was a conflict processing the request. Try your request again.</li>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the ARN of the resource and
     *         try again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>TooManyRequestsException The request was denied due to too many requests being submitted at the same
     *         time.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutMetricsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutMetricsAsyncClient.DeactivateAnomalyDetector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutmetrics-2017-07-25/DeactivateAnomalyDetector"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeactivateAnomalyDetectorResponse> deactivateAnomalyDetector(
            DeactivateAnomalyDetectorRequest deactivateAnomalyDetectorRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deactivateAnomalyDetectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutMetrics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeactivateAnomalyDetector");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeactivateAnomalyDetectorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeactivateAnomalyDetectorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeactivateAnomalyDetectorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeactivateAnomalyDetectorRequest, DeactivateAnomalyDetectorResponse>()
                            .withOperationName("DeactivateAnomalyDetector")
                            .withMarshaller(new DeactivateAnomalyDetectorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deactivateAnomalyDetectorRequest));
            CompletableFuture<DeactivateAnomalyDetectorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an alert.
     * </p>
     *
     * @param deleteAlertRequest
     * @return A Java Future containing the result of the DeleteAlert operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the ARN of the resource and
     *         try again.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the AWS service. Check
     *         your input values and try again.</li>
     *         <li>TooManyRequestsException The request was denied due to too many requests being submitted at the same
     *         time.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutMetricsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutMetricsAsyncClient.DeleteAlert
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutmetrics-2017-07-25/DeleteAlert" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAlertResponse> deleteAlert(DeleteAlertRequest deleteAlertRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAlertRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutMetrics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAlert");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAlertResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteAlertResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAlertResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAlertRequest, DeleteAlertResponse>()
                            .withOperationName("DeleteAlert").withMarshaller(new DeleteAlertRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteAlertRequest));
            CompletableFuture<DeleteAlertResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a detector. Deleting an anomaly detector will delete all of its corresponding resources including any
     * configured datasets and alerts.
     * </p>
     *
     * @param deleteAnomalyDetectorRequest
     * @return A Java Future containing the result of the DeleteAnomalyDetector operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the ARN of the resource and
     *         try again.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the AWS service. Check
     *         your input values and try again.</li>
     *         <li>ConflictException There was a conflict processing the request. Try your request again.</li>
     *         <li>TooManyRequestsException The request was denied due to too many requests being submitted at the same
     *         time.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutMetricsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutMetricsAsyncClient.DeleteAnomalyDetector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutmetrics-2017-07-25/DeleteAnomalyDetector"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAnomalyDetectorResponse> deleteAnomalyDetector(
            DeleteAnomalyDetectorRequest deleteAnomalyDetectorRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAnomalyDetectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutMetrics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAnomalyDetector");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAnomalyDetectorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteAnomalyDetectorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAnomalyDetectorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAnomalyDetectorRequest, DeleteAnomalyDetectorResponse>()
                            .withOperationName("DeleteAnomalyDetector")
                            .withMarshaller(new DeleteAnomalyDetectorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteAnomalyDetectorRequest));
            CompletableFuture<DeleteAnomalyDetectorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes an alert.
     * </p>
     * <p>
     * Amazon Lookout for Metrics API actions are eventually consistent. If you do a read operation on a resource
     * immediately after creating or modifying it, use retries to allow time for the write operation to complete.
     * </p>
     *
     * @param describeAlertRequest
     * @return A Java Future containing the result of the DescribeAlert operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the ARN of the resource and
     *         try again.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the AWS service. Check
     *         your input values and try again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>TooManyRequestsException The request was denied due to too many requests being submitted at the same
     *         time.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutMetricsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutMetricsAsyncClient.DescribeAlert
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutmetrics-2017-07-25/DescribeAlert" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAlertResponse> describeAlert(DescribeAlertRequest describeAlertRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAlertRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutMetrics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAlert");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAlertResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeAlertResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAlertResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAlertRequest, DescribeAlertResponse>()
                            .withOperationName("DescribeAlert")
                            .withMarshaller(new DescribeAlertRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeAlertRequest));
            CompletableFuture<DescribeAlertResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about the status of the specified anomaly detection jobs.
     * </p>
     *
     * @param describeAnomalyDetectionExecutionsRequest
     * @return A Java Future containing the result of the DescribeAnomalyDetectionExecutions operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the ARN of the resource and
     *         try again.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the AWS service. Check
     *         your input values and try again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>TooManyRequestsException The request was denied due to too many requests being submitted at the same
     *         time.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutMetricsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutMetricsAsyncClient.DescribeAnomalyDetectionExecutions
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/lookoutmetrics-2017-07-25/DescribeAnomalyDetectionExecutions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAnomalyDetectionExecutionsResponse> describeAnomalyDetectionExecutions(
            DescribeAnomalyDetectionExecutionsRequest describeAnomalyDetectionExecutionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeAnomalyDetectionExecutionsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutMetrics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAnomalyDetectionExecutions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAnomalyDetectionExecutionsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeAnomalyDetectionExecutionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAnomalyDetectionExecutionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAnomalyDetectionExecutionsRequest, DescribeAnomalyDetectionExecutionsResponse>()
                            .withOperationName("DescribeAnomalyDetectionExecutions")
                            .withMarshaller(new DescribeAnomalyDetectionExecutionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeAnomalyDetectionExecutionsRequest));
            CompletableFuture<DescribeAnomalyDetectionExecutionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about the status of the specified anomaly detection jobs.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeAnomalyDetectionExecutions(software.amazon.awssdk.services.lookoutmetrics.model.DescribeAnomalyDetectionExecutionsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lookoutmetrics.paginators.DescribeAnomalyDetectionExecutionsPublisher publisher = client.describeAnomalyDetectionExecutionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lookoutmetrics.paginators.DescribeAnomalyDetectionExecutionsPublisher publisher = client.describeAnomalyDetectionExecutionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.lookoutmetrics.model.DescribeAnomalyDetectionExecutionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.lookoutmetrics.model.DescribeAnomalyDetectionExecutionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeAnomalyDetectionExecutions(software.amazon.awssdk.services.lookoutmetrics.model.DescribeAnomalyDetectionExecutionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeAnomalyDetectionExecutionsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the ARN of the resource and
     *         try again.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the AWS service. Check
     *         your input values and try again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>TooManyRequestsException The request was denied due to too many requests being submitted at the same
     *         time.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutMetricsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutMetricsAsyncClient.DescribeAnomalyDetectionExecutions
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/lookoutmetrics-2017-07-25/DescribeAnomalyDetectionExecutions"
     *      target="_top">AWS API Documentation</a>
     */
    public DescribeAnomalyDetectionExecutionsPublisher describeAnomalyDetectionExecutionsPaginator(
            DescribeAnomalyDetectionExecutionsRequest describeAnomalyDetectionExecutionsRequest) {
        return new DescribeAnomalyDetectionExecutionsPublisher(this,
                applyPaginatorUserAgent(describeAnomalyDetectionExecutionsRequest));
    }

    /**
     * <p>
     * Describes a detector.
     * </p>
     * <p>
     * Amazon Lookout for Metrics API actions are eventually consistent. If you do a read operation on a resource
     * immediately after creating or modifying it, use retries to allow time for the write operation to complete.
     * </p>
     *
     * @param describeAnomalyDetectorRequest
     * @return A Java Future containing the result of the DescribeAnomalyDetector operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the ARN of the resource and
     *         try again.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the AWS service. Check
     *         your input values and try again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>TooManyRequestsException The request was denied due to too many requests being submitted at the same
     *         time.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutMetricsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutMetricsAsyncClient.DescribeAnomalyDetector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutmetrics-2017-07-25/DescribeAnomalyDetector"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAnomalyDetectorResponse> describeAnomalyDetector(
            DescribeAnomalyDetectorRequest describeAnomalyDetectorRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAnomalyDetectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutMetrics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAnomalyDetector");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAnomalyDetectorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeAnomalyDetectorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAnomalyDetectorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAnomalyDetectorRequest, DescribeAnomalyDetectorResponse>()
                            .withOperationName("DescribeAnomalyDetector")
                            .withMarshaller(new DescribeAnomalyDetectorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeAnomalyDetectorRequest));
            CompletableFuture<DescribeAnomalyDetectorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes a dataset.
     * </p>
     * <p>
     * Amazon Lookout for Metrics API actions are eventually consistent. If you do a read operation on a resource
     * immediately after creating or modifying it, use retries to allow time for the write operation to complete.
     * </p>
     *
     * @param describeMetricSetRequest
     * @return A Java Future containing the result of the DescribeMetricSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the ARN of the resource and
     *         try again.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the AWS service. Check
     *         your input values and try again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>TooManyRequestsException The request was denied due to too many requests being submitted at the same
     *         time.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutMetricsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutMetricsAsyncClient.DescribeMetricSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutmetrics-2017-07-25/DescribeMetricSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeMetricSetResponse> describeMetricSet(DescribeMetricSetRequest describeMetricSetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeMetricSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutMetrics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeMetricSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeMetricSetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeMetricSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeMetricSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeMetricSetRequest, DescribeMetricSetResponse>()
                            .withOperationName("DescribeMetricSet")
                            .withMarshaller(new DescribeMetricSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeMetricSetRequest));
            CompletableFuture<DescribeMetricSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Detects an Amazon S3 dataset's file format, interval, and offset.
     * </p>
     *
     * @param detectMetricSetConfigRequest
     * @return A Java Future containing the result of the DetectMetricSetConfig operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the ARN of the resource and
     *         try again.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the AWS service. Check
     *         your input values and try again.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>TooManyRequestsException The request was denied due to too many requests being submitted at the same
     *         time.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutMetricsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutMetricsAsyncClient.DetectMetricSetConfig
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutmetrics-2017-07-25/DetectMetricSetConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DetectMetricSetConfigResponse> detectMetricSetConfig(
            DetectMetricSetConfigRequest detectMetricSetConfigRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, detectMetricSetConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutMetrics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DetectMetricSetConfig");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DetectMetricSetConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DetectMetricSetConfigResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DetectMetricSetConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DetectMetricSetConfigRequest, DetectMetricSetConfigResponse>()
                            .withOperationName("DetectMetricSetConfig")
                            .withMarshaller(new DetectMetricSetConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(detectMetricSetConfigRequest));
            CompletableFuture<DetectMetricSetConfigResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns details about a group of anomalous metrics.
     * </p>
     *
     * @param getAnomalyGroupRequest
     * @return A Java Future containing the result of the GetAnomalyGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the ARN of the resource and
     *         try again.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the AWS service. Check
     *         your input values and try again.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>TooManyRequestsException The request was denied due to too many requests being submitted at the same
     *         time.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutMetricsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutMetricsAsyncClient.GetAnomalyGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutmetrics-2017-07-25/GetAnomalyGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAnomalyGroupResponse> getAnomalyGroup(GetAnomalyGroupRequest getAnomalyGroupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAnomalyGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutMetrics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAnomalyGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAnomalyGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetAnomalyGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAnomalyGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAnomalyGroupRequest, GetAnomalyGroupResponse>()
                            .withOperationName("GetAnomalyGroup")
                            .withMarshaller(new GetAnomalyGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getAnomalyGroupRequest));
            CompletableFuture<GetAnomalyGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns details about the requested data quality metrics.
     * </p>
     *
     * @param getDataQualityMetricsRequest
     * @return A Java Future containing the result of the GetDataQualityMetrics operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the ARN of the resource and
     *         try again.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the AWS service. Check
     *         your input values and try again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>TooManyRequestsException The request was denied due to too many requests being submitted at the same
     *         time.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutMetricsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutMetricsAsyncClient.GetDataQualityMetrics
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutmetrics-2017-07-25/GetDataQualityMetrics"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDataQualityMetricsResponse> getDataQualityMetrics(
            GetDataQualityMetricsRequest getDataQualityMetricsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDataQualityMetricsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutMetrics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDataQualityMetrics");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDataQualityMetricsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDataQualityMetricsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDataQualityMetricsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDataQualityMetricsRequest, GetDataQualityMetricsResponse>()
                            .withOperationName("GetDataQualityMetrics")
                            .withMarshaller(new GetDataQualityMetricsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getDataQualityMetricsRequest));
            CompletableFuture<GetDataQualityMetricsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get feedback for an anomaly group.
     * </p>
     *
     * @param getFeedbackRequest
     * @return A Java Future containing the result of the GetFeedback operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the ARN of the resource and
     *         try again.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the AWS service. Check
     *         your input values and try again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>TooManyRequestsException The request was denied due to too many requests being submitted at the same
     *         time.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutMetricsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutMetricsAsyncClient.GetFeedback
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutmetrics-2017-07-25/GetFeedback" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetFeedbackResponse> getFeedback(GetFeedbackRequest getFeedbackRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getFeedbackRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutMetrics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetFeedback");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetFeedbackResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetFeedbackResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetFeedbackResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetFeedbackRequest, GetFeedbackResponse>()
                            .withOperationName("GetFeedback").withMarshaller(new GetFeedbackRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getFeedbackRequest));
            CompletableFuture<GetFeedbackResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Get feedback for an anomaly group.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getFeedback(software.amazon.awssdk.services.lookoutmetrics.model.GetFeedbackRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lookoutmetrics.paginators.GetFeedbackPublisher publisher = client.getFeedbackPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lookoutmetrics.paginators.GetFeedbackPublisher publisher = client.getFeedbackPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.lookoutmetrics.model.GetFeedbackResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.lookoutmetrics.model.GetFeedbackResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getFeedback(software.amazon.awssdk.services.lookoutmetrics.model.GetFeedbackRequest)} operation.</b>
     * </p>
     *
     * @param getFeedbackRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the ARN of the resource and
     *         try again.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the AWS service. Check
     *         your input values and try again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>TooManyRequestsException The request was denied due to too many requests being submitted at the same
     *         time.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutMetricsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutMetricsAsyncClient.GetFeedback
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutmetrics-2017-07-25/GetFeedback" target="_top">AWS
     *      API Documentation</a>
     */
    public GetFeedbackPublisher getFeedbackPaginator(GetFeedbackRequest getFeedbackRequest) {
        return new GetFeedbackPublisher(this, applyPaginatorUserAgent(getFeedbackRequest));
    }

    /**
     * <p>
     * Returns a selection of sample records from an Amazon S3 datasource.
     * </p>
     *
     * @param getSampleDataRequest
     * @return A Java Future containing the result of the GetSampleData operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the AWS service. Check
     *         your input values and try again.</li>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the ARN of the resource and
     *         try again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>TooManyRequestsException The request was denied due to too many requests being submitted at the same
     *         time.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutMetricsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutMetricsAsyncClient.GetSampleData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutmetrics-2017-07-25/GetSampleData" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetSampleDataResponse> getSampleData(GetSampleDataRequest getSampleDataRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSampleDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutMetrics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSampleData");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSampleDataResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetSampleDataResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSampleDataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSampleDataRequest, GetSampleDataResponse>()
                            .withOperationName("GetSampleData")
                            .withMarshaller(new GetSampleDataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getSampleDataRequest));
            CompletableFuture<GetSampleDataResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the alerts attached to a detector.
     * </p>
     * <p>
     * Amazon Lookout for Metrics API actions are eventually consistent. If you do a read operation on a resource
     * immediately after creating or modifying it, use retries to allow time for the write operation to complete.
     * </p>
     *
     * @param listAlertsRequest
     * @return A Java Future containing the result of the ListAlerts operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the ARN of the resource and
     *         try again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>TooManyRequestsException The request was denied due to too many requests being submitted at the same
     *         time.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the AWS service. Check
     *         your input values and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutMetricsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutMetricsAsyncClient.ListAlerts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutmetrics-2017-07-25/ListAlerts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListAlertsResponse> listAlerts(ListAlertsRequest listAlertsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAlertsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutMetrics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAlerts");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAlertsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListAlertsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAlertsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAlertsRequest, ListAlertsResponse>().withOperationName("ListAlerts")
                            .withMarshaller(new ListAlertsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listAlertsRequest));
            CompletableFuture<ListAlertsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the alerts attached to a detector.
     * </p>
     * <p>
     * Amazon Lookout for Metrics API actions are eventually consistent. If you do a read operation on a resource
     * immediately after creating or modifying it, use retries to allow time for the write operation to complete.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listAlerts(software.amazon.awssdk.services.lookoutmetrics.model.ListAlertsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lookoutmetrics.paginators.ListAlertsPublisher publisher = client.listAlertsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lookoutmetrics.paginators.ListAlertsPublisher publisher = client.listAlertsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.lookoutmetrics.model.ListAlertsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.lookoutmetrics.model.ListAlertsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAlerts(software.amazon.awssdk.services.lookoutmetrics.model.ListAlertsRequest)} operation.</b>
     * </p>
     *
     * @param listAlertsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the ARN of the resource and
     *         try again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>TooManyRequestsException The request was denied due to too many requests being submitted at the same
     *         time.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the AWS service. Check
     *         your input values and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutMetricsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutMetricsAsyncClient.ListAlerts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutmetrics-2017-07-25/ListAlerts" target="_top">AWS API
     *      Documentation</a>
     */
    public ListAlertsPublisher listAlertsPaginator(ListAlertsRequest listAlertsRequest) {
        return new ListAlertsPublisher(this, applyPaginatorUserAgent(listAlertsRequest));
    }

    /**
     * <p>
     * Lists the detectors in the current AWS Region.
     * </p>
     * <p>
     * Amazon Lookout for Metrics API actions are eventually consistent. If you do a read operation on a resource
     * immediately after creating or modifying it, use retries to allow time for the write operation to complete.
     * </p>
     *
     * @param listAnomalyDetectorsRequest
     * @return A Java Future containing the result of the ListAnomalyDetectors operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the ARN of the resource and
     *         try again.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the AWS service. Check
     *         your input values and try again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>TooManyRequestsException The request was denied due to too many requests being submitted at the same
     *         time.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutMetricsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutMetricsAsyncClient.ListAnomalyDetectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutmetrics-2017-07-25/ListAnomalyDetectors"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAnomalyDetectorsResponse> listAnomalyDetectors(
            ListAnomalyDetectorsRequest listAnomalyDetectorsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAnomalyDetectorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutMetrics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAnomalyDetectors");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAnomalyDetectorsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAnomalyDetectorsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAnomalyDetectorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAnomalyDetectorsRequest, ListAnomalyDetectorsResponse>()
                            .withOperationName("ListAnomalyDetectors")
                            .withMarshaller(new ListAnomalyDetectorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listAnomalyDetectorsRequest));
            CompletableFuture<ListAnomalyDetectorsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the detectors in the current AWS Region.
     * </p>
     * <p>
     * Amazon Lookout for Metrics API actions are eventually consistent. If you do a read operation on a resource
     * immediately after creating or modifying it, use retries to allow time for the write operation to complete.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAnomalyDetectors(software.amazon.awssdk.services.lookoutmetrics.model.ListAnomalyDetectorsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lookoutmetrics.paginators.ListAnomalyDetectorsPublisher publisher = client.listAnomalyDetectorsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lookoutmetrics.paginators.ListAnomalyDetectorsPublisher publisher = client.listAnomalyDetectorsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.lookoutmetrics.model.ListAnomalyDetectorsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.lookoutmetrics.model.ListAnomalyDetectorsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAnomalyDetectors(software.amazon.awssdk.services.lookoutmetrics.model.ListAnomalyDetectorsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listAnomalyDetectorsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the ARN of the resource and
     *         try again.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the AWS service. Check
     *         your input values and try again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>TooManyRequestsException The request was denied due to too many requests being submitted at the same
     *         time.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutMetricsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutMetricsAsyncClient.ListAnomalyDetectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutmetrics-2017-07-25/ListAnomalyDetectors"
     *      target="_top">AWS API Documentation</a>
     */
    public ListAnomalyDetectorsPublisher listAnomalyDetectorsPaginator(ListAnomalyDetectorsRequest listAnomalyDetectorsRequest) {
        return new ListAnomalyDetectorsPublisher(this, applyPaginatorUserAgent(listAnomalyDetectorsRequest));
    }

    /**
     * <p>
     * Returns a list of measures that are potential causes or effects of an anomaly group.
     * </p>
     *
     * @param listAnomalyGroupRelatedMetricsRequest
     * @return A Java Future containing the result of the ListAnomalyGroupRelatedMetrics operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the ARN of the resource and
     *         try again.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the AWS service. Check
     *         your input values and try again.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>TooManyRequestsException The request was denied due to too many requests being submitted at the same
     *         time.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutMetricsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutMetricsAsyncClient.ListAnomalyGroupRelatedMetrics
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutmetrics-2017-07-25/ListAnomalyGroupRelatedMetrics"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAnomalyGroupRelatedMetricsResponse> listAnomalyGroupRelatedMetrics(
            ListAnomalyGroupRelatedMetricsRequest listAnomalyGroupRelatedMetricsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listAnomalyGroupRelatedMetricsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutMetrics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAnomalyGroupRelatedMetrics");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAnomalyGroupRelatedMetricsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAnomalyGroupRelatedMetricsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAnomalyGroupRelatedMetricsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAnomalyGroupRelatedMetricsRequest, ListAnomalyGroupRelatedMetricsResponse>()
                            .withOperationName("ListAnomalyGroupRelatedMetrics")
                            .withMarshaller(new ListAnomalyGroupRelatedMetricsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listAnomalyGroupRelatedMetricsRequest));
            CompletableFuture<ListAnomalyGroupRelatedMetricsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of measures that are potential causes or effects of an anomaly group.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAnomalyGroupRelatedMetrics(software.amazon.awssdk.services.lookoutmetrics.model.ListAnomalyGroupRelatedMetricsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lookoutmetrics.paginators.ListAnomalyGroupRelatedMetricsPublisher publisher = client.listAnomalyGroupRelatedMetricsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lookoutmetrics.paginators.ListAnomalyGroupRelatedMetricsPublisher publisher = client.listAnomalyGroupRelatedMetricsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.lookoutmetrics.model.ListAnomalyGroupRelatedMetricsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.lookoutmetrics.model.ListAnomalyGroupRelatedMetricsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAnomalyGroupRelatedMetrics(software.amazon.awssdk.services.lookoutmetrics.model.ListAnomalyGroupRelatedMetricsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listAnomalyGroupRelatedMetricsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the ARN of the resource and
     *         try again.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the AWS service. Check
     *         your input values and try again.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>TooManyRequestsException The request was denied due to too many requests being submitted at the same
     *         time.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutMetricsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutMetricsAsyncClient.ListAnomalyGroupRelatedMetrics
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutmetrics-2017-07-25/ListAnomalyGroupRelatedMetrics"
     *      target="_top">AWS API Documentation</a>
     */
    public ListAnomalyGroupRelatedMetricsPublisher listAnomalyGroupRelatedMetricsPaginator(
            ListAnomalyGroupRelatedMetricsRequest listAnomalyGroupRelatedMetricsRequest) {
        return new ListAnomalyGroupRelatedMetricsPublisher(this, applyPaginatorUserAgent(listAnomalyGroupRelatedMetricsRequest));
    }

    /**
     * <p>
     * Returns a list of anomaly groups.
     * </p>
     *
     * @param listAnomalyGroupSummariesRequest
     * @return A Java Future containing the result of the ListAnomalyGroupSummaries operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the ARN of the resource and
     *         try again.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the AWS service. Check
     *         your input values and try again.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>TooManyRequestsException The request was denied due to too many requests being submitted at the same
     *         time.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutMetricsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutMetricsAsyncClient.ListAnomalyGroupSummaries
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutmetrics-2017-07-25/ListAnomalyGroupSummaries"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAnomalyGroupSummariesResponse> listAnomalyGroupSummaries(
            ListAnomalyGroupSummariesRequest listAnomalyGroupSummariesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAnomalyGroupSummariesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutMetrics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAnomalyGroupSummaries");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAnomalyGroupSummariesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAnomalyGroupSummariesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAnomalyGroupSummariesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAnomalyGroupSummariesRequest, ListAnomalyGroupSummariesResponse>()
                            .withOperationName("ListAnomalyGroupSummaries")
                            .withMarshaller(new ListAnomalyGroupSummariesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listAnomalyGroupSummariesRequest));
            CompletableFuture<ListAnomalyGroupSummariesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of anomaly groups.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAnomalyGroupSummaries(software.amazon.awssdk.services.lookoutmetrics.model.ListAnomalyGroupSummariesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lookoutmetrics.paginators.ListAnomalyGroupSummariesPublisher publisher = client.listAnomalyGroupSummariesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lookoutmetrics.paginators.ListAnomalyGroupSummariesPublisher publisher = client.listAnomalyGroupSummariesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.lookoutmetrics.model.ListAnomalyGroupSummariesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.lookoutmetrics.model.ListAnomalyGroupSummariesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAnomalyGroupSummaries(software.amazon.awssdk.services.lookoutmetrics.model.ListAnomalyGroupSummariesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listAnomalyGroupSummariesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the ARN of the resource and
     *         try again.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the AWS service. Check
     *         your input values and try again.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>TooManyRequestsException The request was denied due to too many requests being submitted at the same
     *         time.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutMetricsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutMetricsAsyncClient.ListAnomalyGroupSummaries
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutmetrics-2017-07-25/ListAnomalyGroupSummaries"
     *      target="_top">AWS API Documentation</a>
     */
    public ListAnomalyGroupSummariesPublisher listAnomalyGroupSummariesPaginator(
            ListAnomalyGroupSummariesRequest listAnomalyGroupSummariesRequest) {
        return new ListAnomalyGroupSummariesPublisher(this, applyPaginatorUserAgent(listAnomalyGroupSummariesRequest));
    }

    /**
     * <p>
     * Gets a list of anomalous metrics for a measure in an anomaly group.
     * </p>
     *
     * @param listAnomalyGroupTimeSeriesRequest
     * @return A Java Future containing the result of the ListAnomalyGroupTimeSeries operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the AWS service. Check
     *         your input values and try again.</li>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the ARN of the resource and
     *         try again.</li>
     *         <li>TooManyRequestsException The request was denied due to too many requests being submitted at the same
     *         time.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutMetricsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutMetricsAsyncClient.ListAnomalyGroupTimeSeries
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutmetrics-2017-07-25/ListAnomalyGroupTimeSeries"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAnomalyGroupTimeSeriesResponse> listAnomalyGroupTimeSeries(
            ListAnomalyGroupTimeSeriesRequest listAnomalyGroupTimeSeriesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAnomalyGroupTimeSeriesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutMetrics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAnomalyGroupTimeSeries");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAnomalyGroupTimeSeriesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAnomalyGroupTimeSeriesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAnomalyGroupTimeSeriesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAnomalyGroupTimeSeriesRequest, ListAnomalyGroupTimeSeriesResponse>()
                            .withOperationName("ListAnomalyGroupTimeSeries")
                            .withMarshaller(new ListAnomalyGroupTimeSeriesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listAnomalyGroupTimeSeriesRequest));
            CompletableFuture<ListAnomalyGroupTimeSeriesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a list of anomalous metrics for a measure in an anomaly group.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAnomalyGroupTimeSeries(software.amazon.awssdk.services.lookoutmetrics.model.ListAnomalyGroupTimeSeriesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lookoutmetrics.paginators.ListAnomalyGroupTimeSeriesPublisher publisher = client.listAnomalyGroupTimeSeriesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lookoutmetrics.paginators.ListAnomalyGroupTimeSeriesPublisher publisher = client.listAnomalyGroupTimeSeriesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.lookoutmetrics.model.ListAnomalyGroupTimeSeriesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.lookoutmetrics.model.ListAnomalyGroupTimeSeriesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAnomalyGroupTimeSeries(software.amazon.awssdk.services.lookoutmetrics.model.ListAnomalyGroupTimeSeriesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listAnomalyGroupTimeSeriesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the AWS service. Check
     *         your input values and try again.</li>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the ARN of the resource and
     *         try again.</li>
     *         <li>TooManyRequestsException The request was denied due to too many requests being submitted at the same
     *         time.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutMetricsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutMetricsAsyncClient.ListAnomalyGroupTimeSeries
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutmetrics-2017-07-25/ListAnomalyGroupTimeSeries"
     *      target="_top">AWS API Documentation</a>
     */
    public ListAnomalyGroupTimeSeriesPublisher listAnomalyGroupTimeSeriesPaginator(
            ListAnomalyGroupTimeSeriesRequest listAnomalyGroupTimeSeriesRequest) {
        return new ListAnomalyGroupTimeSeriesPublisher(this, applyPaginatorUserAgent(listAnomalyGroupTimeSeriesRequest));
    }

    /**
     * <p>
     * Lists the datasets in the current AWS Region.
     * </p>
     * <p>
     * Amazon Lookout for Metrics API actions are eventually consistent. If you do a read operation on a resource
     * immediately after creating or modifying it, use retries to allow time for the write operation to complete.
     * </p>
     *
     * @param listMetricSetsRequest
     * @return A Java Future containing the result of the ListMetricSets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the ARN of the resource and
     *         try again.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the AWS service. Check
     *         your input values and try again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>TooManyRequestsException The request was denied due to too many requests being submitted at the same
     *         time.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutMetricsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutMetricsAsyncClient.ListMetricSets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutmetrics-2017-07-25/ListMetricSets" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListMetricSetsResponse> listMetricSets(ListMetricSetsRequest listMetricSetsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMetricSetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutMetrics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMetricSets");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListMetricSetsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListMetricSetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListMetricSetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListMetricSetsRequest, ListMetricSetsResponse>()
                            .withOperationName("ListMetricSets")
                            .withMarshaller(new ListMetricSetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listMetricSetsRequest));
            CompletableFuture<ListMetricSetsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the datasets in the current AWS Region.
     * </p>
     * <p>
     * Amazon Lookout for Metrics API actions are eventually consistent. If you do a read operation on a resource
     * immediately after creating or modifying it, use retries to allow time for the write operation to complete.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listMetricSets(software.amazon.awssdk.services.lookoutmetrics.model.ListMetricSetsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lookoutmetrics.paginators.ListMetricSetsPublisher publisher = client.listMetricSetsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lookoutmetrics.paginators.ListMetricSetsPublisher publisher = client.listMetricSetsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.lookoutmetrics.model.ListMetricSetsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.lookoutmetrics.model.ListMetricSetsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listMetricSets(software.amazon.awssdk.services.lookoutmetrics.model.ListMetricSetsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listMetricSetsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the ARN of the resource and
     *         try again.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the AWS service. Check
     *         your input values and try again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>TooManyRequestsException The request was denied due to too many requests being submitted at the same
     *         time.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutMetricsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutMetricsAsyncClient.ListMetricSets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutmetrics-2017-07-25/ListMetricSets" target="_top">AWS
     *      API Documentation</a>
     */
    public ListMetricSetsPublisher listMetricSetsPaginator(ListMetricSetsRequest listMetricSetsRequest) {
        return new ListMetricSetsPublisher(this, applyPaginatorUserAgent(listMetricSetsRequest));
    }

    /**
     * <p>
     * Gets a list of <a href="https://docs.aws.amazon.com/lookoutmetrics/latest/dev/detectors-tags.html">tags</a> for a
     * detector, dataset, or alert.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the AWS service. Check
     *         your input values and try again.</li>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the ARN of the resource and
     *         try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutMetricsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutMetricsAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutmetrics-2017-07-25/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutMetrics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Add feedback for an anomalous metric.
     * </p>
     *
     * @param putFeedbackRequest
     * @return A Java Future containing the result of the PutFeedback operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the ARN of the resource and
     *         try again.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the AWS service. Check
     *         your input values and try again.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>TooManyRequestsException The request was denied due to too many requests being submitted at the same
     *         time.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutMetricsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutMetricsAsyncClient.PutFeedback
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutmetrics-2017-07-25/PutFeedback" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PutFeedbackResponse> putFeedback(PutFeedbackRequest putFeedbackRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putFeedbackRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutMetrics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutFeedback");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutFeedbackResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutFeedbackResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutFeedbackResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutFeedbackRequest, PutFeedbackResponse>()
                            .withOperationName("PutFeedback").withMarshaller(new PutFeedbackRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putFeedbackRequest));
            CompletableFuture<PutFeedbackResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds <a href="https://docs.aws.amazon.com/lookoutmetrics/latest/dev/detectors-tags.html">tags</a> to a detector,
     * dataset, or alert.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the AWS service. Check
     *         your input values and try again.</li>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the ARN of the resource and
     *         try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutMetricsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutMetricsAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutmetrics-2017-07-25/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutMetrics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes <a href="https://docs.aws.amazon.com/lookoutmetrics/latest/dev/detectors-tags.html">tags</a> from a
     * detector, dataset, or alert.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the AWS service. Check
     *         your input values and try again.</li>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the ARN of the resource and
     *         try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutMetricsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutMetricsAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutmetrics-2017-07-25/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutMetrics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Make changes to an existing alert.
     * </p>
     *
     * @param updateAlertRequest
     * @return A Java Future containing the result of the UpdateAlert operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the AWS service. Check
     *         your input values and try again.</li>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the ARN of the resource and
     *         try again.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>TooManyRequestsException The request was denied due to too many requests being submitted at the same
     *         time.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutMetricsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutMetricsAsyncClient.UpdateAlert
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutmetrics-2017-07-25/UpdateAlert" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateAlertResponse> updateAlert(UpdateAlertRequest updateAlertRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAlertRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutMetrics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAlert");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateAlertResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateAlertResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateAlertResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateAlertRequest, UpdateAlertResponse>()
                            .withOperationName("UpdateAlert").withMarshaller(new UpdateAlertRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateAlertRequest));
            CompletableFuture<UpdateAlertResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a detector. After activation, you can only change a detector's ingestion delay and description.
     * </p>
     *
     * @param updateAnomalyDetectorRequest
     * @return A Java Future containing the result of the UpdateAnomalyDetector operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the ARN of the resource and
     *         try again.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the AWS service. Check
     *         your input values and try again.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>TooManyRequestsException The request was denied due to too many requests being submitted at the same
     *         time.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutMetricsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutMetricsAsyncClient.UpdateAnomalyDetector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutmetrics-2017-07-25/UpdateAnomalyDetector"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateAnomalyDetectorResponse> updateAnomalyDetector(
            UpdateAnomalyDetectorRequest updateAnomalyDetectorRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAnomalyDetectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutMetrics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAnomalyDetector");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateAnomalyDetectorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateAnomalyDetectorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateAnomalyDetectorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateAnomalyDetectorRequest, UpdateAnomalyDetectorResponse>()
                            .withOperationName("UpdateAnomalyDetector")
                            .withMarshaller(new UpdateAnomalyDetectorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateAnomalyDetectorRequest));
            CompletableFuture<UpdateAnomalyDetectorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a dataset.
     * </p>
     *
     * @param updateMetricSetRequest
     * @return A Java Future containing the result of the UpdateMetricSet operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource cannot be found. Check the ARN of the resource and
     *         try again.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by the AWS service. Check
     *         your input values and try again.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>AccessDeniedException You do not have sufficient permissions to perform this action.</li>
     *         <li>TooManyRequestsException The request was denied due to too many requests being submitted at the same
     *         time.</li>
     *         <li>ServiceQuotaExceededException The request exceeded the service's quotas. Check the service quotas and
     *         try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutMetricsException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LookoutMetricsAsyncClient.UpdateMetricSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutmetrics-2017-07-25/UpdateMetricSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateMetricSetResponse> updateMetricSet(UpdateMetricSetRequest updateMetricSetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateMetricSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutMetrics");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateMetricSet");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateMetricSetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateMetricSetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateMetricSetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateMetricSetRequest, UpdateMetricSetResponse>()
                            .withOperationName("UpdateMetricSet")
                            .withMarshaller(new UpdateMetricSetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateMetricSetRequest));
            CompletableFuture<UpdateMetricSetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(LookoutMetricsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends LookoutMetricsRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
