/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lookoutvision;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.auth.signer.AsyncAws4Signer;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.async.AsyncRequestBody;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.runtime.transform.AsyncStreamingRequestMarshaller;
import software.amazon.awssdk.core.signer.Signer;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.lookoutvision.model.AccessDeniedException;
import software.amazon.awssdk.services.lookoutvision.model.ConflictException;
import software.amazon.awssdk.services.lookoutvision.model.CreateDatasetRequest;
import software.amazon.awssdk.services.lookoutvision.model.CreateDatasetResponse;
import software.amazon.awssdk.services.lookoutvision.model.CreateModelRequest;
import software.amazon.awssdk.services.lookoutvision.model.CreateModelResponse;
import software.amazon.awssdk.services.lookoutvision.model.CreateProjectRequest;
import software.amazon.awssdk.services.lookoutvision.model.CreateProjectResponse;
import software.amazon.awssdk.services.lookoutvision.model.DeleteDatasetRequest;
import software.amazon.awssdk.services.lookoutvision.model.DeleteDatasetResponse;
import software.amazon.awssdk.services.lookoutvision.model.DeleteModelRequest;
import software.amazon.awssdk.services.lookoutvision.model.DeleteModelResponse;
import software.amazon.awssdk.services.lookoutvision.model.DeleteProjectRequest;
import software.amazon.awssdk.services.lookoutvision.model.DeleteProjectResponse;
import software.amazon.awssdk.services.lookoutvision.model.DescribeDatasetRequest;
import software.amazon.awssdk.services.lookoutvision.model.DescribeDatasetResponse;
import software.amazon.awssdk.services.lookoutvision.model.DescribeModelRequest;
import software.amazon.awssdk.services.lookoutvision.model.DescribeModelResponse;
import software.amazon.awssdk.services.lookoutvision.model.DescribeProjectRequest;
import software.amazon.awssdk.services.lookoutvision.model.DescribeProjectResponse;
import software.amazon.awssdk.services.lookoutvision.model.DetectAnomaliesRequest;
import software.amazon.awssdk.services.lookoutvision.model.DetectAnomaliesResponse;
import software.amazon.awssdk.services.lookoutvision.model.InternalServerException;
import software.amazon.awssdk.services.lookoutvision.model.ListDatasetEntriesRequest;
import software.amazon.awssdk.services.lookoutvision.model.ListDatasetEntriesResponse;
import software.amazon.awssdk.services.lookoutvision.model.ListModelsRequest;
import software.amazon.awssdk.services.lookoutvision.model.ListModelsResponse;
import software.amazon.awssdk.services.lookoutvision.model.ListProjectsRequest;
import software.amazon.awssdk.services.lookoutvision.model.ListProjectsResponse;
import software.amazon.awssdk.services.lookoutvision.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.lookoutvision.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.lookoutvision.model.LookoutVisionException;
import software.amazon.awssdk.services.lookoutvision.model.LookoutVisionRequest;
import software.amazon.awssdk.services.lookoutvision.model.ResourceNotFoundException;
import software.amazon.awssdk.services.lookoutvision.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.lookoutvision.model.StartModelRequest;
import software.amazon.awssdk.services.lookoutvision.model.StartModelResponse;
import software.amazon.awssdk.services.lookoutvision.model.StopModelRequest;
import software.amazon.awssdk.services.lookoutvision.model.StopModelResponse;
import software.amazon.awssdk.services.lookoutvision.model.TagResourceRequest;
import software.amazon.awssdk.services.lookoutvision.model.TagResourceResponse;
import software.amazon.awssdk.services.lookoutvision.model.ThrottlingException;
import software.amazon.awssdk.services.lookoutvision.model.UntagResourceRequest;
import software.amazon.awssdk.services.lookoutvision.model.UntagResourceResponse;
import software.amazon.awssdk.services.lookoutvision.model.UpdateDatasetEntriesRequest;
import software.amazon.awssdk.services.lookoutvision.model.UpdateDatasetEntriesResponse;
import software.amazon.awssdk.services.lookoutvision.model.ValidationException;
import software.amazon.awssdk.services.lookoutvision.paginators.ListDatasetEntriesPublisher;
import software.amazon.awssdk.services.lookoutvision.paginators.ListModelsPublisher;
import software.amazon.awssdk.services.lookoutvision.paginators.ListProjectsPublisher;
import software.amazon.awssdk.services.lookoutvision.transform.CreateDatasetRequestMarshaller;
import software.amazon.awssdk.services.lookoutvision.transform.CreateModelRequestMarshaller;
import software.amazon.awssdk.services.lookoutvision.transform.CreateProjectRequestMarshaller;
import software.amazon.awssdk.services.lookoutvision.transform.DeleteDatasetRequestMarshaller;
import software.amazon.awssdk.services.lookoutvision.transform.DeleteModelRequestMarshaller;
import software.amazon.awssdk.services.lookoutvision.transform.DeleteProjectRequestMarshaller;
import software.amazon.awssdk.services.lookoutvision.transform.DescribeDatasetRequestMarshaller;
import software.amazon.awssdk.services.lookoutvision.transform.DescribeModelRequestMarshaller;
import software.amazon.awssdk.services.lookoutvision.transform.DescribeProjectRequestMarshaller;
import software.amazon.awssdk.services.lookoutvision.transform.DetectAnomaliesRequestMarshaller;
import software.amazon.awssdk.services.lookoutvision.transform.ListDatasetEntriesRequestMarshaller;
import software.amazon.awssdk.services.lookoutvision.transform.ListModelsRequestMarshaller;
import software.amazon.awssdk.services.lookoutvision.transform.ListProjectsRequestMarshaller;
import software.amazon.awssdk.services.lookoutvision.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.lookoutvision.transform.StartModelRequestMarshaller;
import software.amazon.awssdk.services.lookoutvision.transform.StopModelRequestMarshaller;
import software.amazon.awssdk.services.lookoutvision.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.lookoutvision.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.lookoutvision.transform.UpdateDatasetEntriesRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link LookoutVisionAsyncClient}.
 *
 * @see LookoutVisionAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultLookoutVisionAsyncClient implements LookoutVisionAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultLookoutVisionAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultLookoutVisionAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a new dataset in an Amazon Lookout for Vision project. <code>CreateDataset</code> can create a training
     * or a test dataset from a valid dataset source (<code>DatasetSource</code>).
     * </p>
     * <p>
     * If you want a single dataset project, specify <code>train</code> for the value of <code>DatasetType</code>.
     * </p>
     * <p>
     * To have a project with separate training and test datasets, call <code>CreateDataset</code> twice. On the first
     * call, specify <code>train</code> for the value of <code>DatasetType</code>. On the second call, specify
     * <code>test</code> for the value of <code>DatasetType</code>.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:CreateDataset</code> operation.
     * </p>
     *
     * @param createDatasetRequest
     * @return A Java Future containing the result of the CreateDataset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You are not authorized to perform the action.</li>
     *         <li>InternalServerException Amazon Lookout for Vision experienced a service issue. Try your call again.</li>
     *         <li>ValidationException An input validation error occured. For example, invalid characters in a project
     *         name, or if a pagination token is invalid.</li>
     *         <li>ConflictException The update or deletion of a resource caused an inconsistent state.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>ThrottlingException Amazon Lookout for Vision is temporarily unable to process the request. Try your
     *         call again.</li>
     *         <li>ServiceQuotaExceededException A service quota was exceeded the allowed limit. For more information,
     *         see Limits in Amazon Lookout for Vision in the Amazon Lookout for Vision Developer Guide.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutVisionException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LookoutVisionAsyncClient.CreateDataset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/CreateDataset" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDatasetResponse> createDataset(CreateDatasetRequest createDatasetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDatasetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDataset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDatasetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateDatasetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDatasetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDatasetRequest, CreateDatasetResponse>()
                            .withOperationName("CreateDataset")
                            .withMarshaller(new CreateDatasetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createDatasetRequest));
            CompletableFuture<CreateDatasetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new version of a model within an an Amazon Lookout for Vision project. <code>CreateModel</code> is an
     * asynchronous operation in which Amazon Lookout for Vision trains, tests, and evaluates a new version of a model.
     * </p>
     * <p>
     * To get the current status, check the <code>Status</code> field returned in the response from
     * <a>DescribeModel</a>.
     * </p>
     * <p>
     * If the project has a single dataset, Amazon Lookout for Vision internally splits the dataset to create a training
     * and a test dataset. If the project has a training and a test dataset, Lookout for Vision uses the respective
     * datasets to train and test the model.
     * </p>
     * <p>
     * After training completes, the evaluation metrics are stored at the location specified in
     * <code>OutputConfig</code>.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:CreateModel</code> operation. If you want
     * to tag your model, you also require permission to the <code>lookoutvision:TagResource</code> operation.
     * </p>
     *
     * @param createModelRequest
     * @return A Java Future containing the result of the CreateModel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You are not authorized to perform the action.</li>
     *         <li>InternalServerException Amazon Lookout for Vision experienced a service issue. Try your call again.</li>
     *         <li>ValidationException An input validation error occured. For example, invalid characters in a project
     *         name, or if a pagination token is invalid.</li>
     *         <li>ConflictException The update or deletion of a resource caused an inconsistent state.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>ThrottlingException Amazon Lookout for Vision is temporarily unable to process the request. Try your
     *         call again.</li>
     *         <li>ServiceQuotaExceededException A service quota was exceeded the allowed limit. For more information,
     *         see Limits in Amazon Lookout for Vision in the Amazon Lookout for Vision Developer Guide.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutVisionException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LookoutVisionAsyncClient.CreateModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/CreateModel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateModelResponse> createModel(CreateModelRequest createModelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateModel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateModelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateModelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateModelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateModelRequest, CreateModelResponse>()
                            .withOperationName("CreateModel").withMarshaller(new CreateModelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createModelRequest));
            CompletableFuture<CreateModelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an empty Amazon Lookout for Vision project. After you create the project, add a dataset by calling
     * <a>CreateDataset</a>.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:CreateProject</code> operation.
     * </p>
     *
     * @param createProjectRequest
     * @return A Java Future containing the result of the CreateProject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You are not authorized to perform the action.</li>
     *         <li>InternalServerException Amazon Lookout for Vision experienced a service issue. Try your call again.</li>
     *         <li>ValidationException An input validation error occured. For example, invalid characters in a project
     *         name, or if a pagination token is invalid.</li>
     *         <li>ConflictException The update or deletion of a resource caused an inconsistent state.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>ThrottlingException Amazon Lookout for Vision is temporarily unable to process the request. Try your
     *         call again.</li>
     *         <li>ServiceQuotaExceededException A service quota was exceeded the allowed limit. For more information,
     *         see Limits in Amazon Lookout for Vision in the Amazon Lookout for Vision Developer Guide.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutVisionException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LookoutVisionAsyncClient.CreateProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/CreateProject" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateProjectResponse> createProject(CreateProjectRequest createProjectRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateProject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateProjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateProjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateProjectRequest, CreateProjectResponse>()
                            .withOperationName("CreateProject")
                            .withMarshaller(new CreateProjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createProjectRequest));
            CompletableFuture<CreateProjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an existing Amazon Lookout for Vision <code>dataset</code>.
     * </p>
     * <p>
     * If your the project has a single dataset, you must create a new dataset before you can create a model.
     * </p>
     * <p>
     * If you project has a training dataset and a test dataset consider the following.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you delete the test dataset, your project reverts to a single dataset project. If you then train the model,
     * Amazon Lookout for Vision internally splits the remaining dataset into a training and test dataset.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you delete the training dataset, you must create a training dataset before you can create a model.
     * </p>
     * </li>
     * </ul>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:DeleteDataset</code> operation.
     * </p>
     *
     * @param deleteDatasetRequest
     * @return A Java Future containing the result of the DeleteDataset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You are not authorized to perform the action.</li>
     *         <li>InternalServerException Amazon Lookout for Vision experienced a service issue. Try your call again.</li>
     *         <li>ValidationException An input validation error occured. For example, invalid characters in a project
     *         name, or if a pagination token is invalid.</li>
     *         <li>ConflictException The update or deletion of a resource caused an inconsistent state.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>ThrottlingException Amazon Lookout for Vision is temporarily unable to process the request. Try your
     *         call again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutVisionException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LookoutVisionAsyncClient.DeleteDataset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/DeleteDataset" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDatasetResponse> deleteDataset(DeleteDatasetRequest deleteDatasetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDatasetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDataset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDatasetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteDatasetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDatasetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDatasetRequest, DeleteDatasetResponse>()
                            .withOperationName("DeleteDataset")
                            .withMarshaller(new DeleteDatasetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteDatasetRequest));
            CompletableFuture<DeleteDatasetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an Amazon Lookout for Vision model. You can't delete a running model. To stop a running model, use the
     * <a>StopModel</a> operation.
     * </p>
     * <p>
     * It might take a few seconds to delete a model. To determine if a model has been deleted, call <a>ListProjects</a>
     * and check if the version of the model (<code>ModelVersion</code>) is in the <code>Models</code> array.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:DeleteModel</code> operation.
     * </p>
     *
     * @param deleteModelRequest
     * @return A Java Future containing the result of the DeleteModel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You are not authorized to perform the action.</li>
     *         <li>InternalServerException Amazon Lookout for Vision experienced a service issue. Try your call again.</li>
     *         <li>ValidationException An input validation error occured. For example, invalid characters in a project
     *         name, or if a pagination token is invalid.</li>
     *         <li>ConflictException The update or deletion of a resource caused an inconsistent state.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>ThrottlingException Amazon Lookout for Vision is temporarily unable to process the request. Try your
     *         call again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutVisionException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LookoutVisionAsyncClient.DeleteModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/DeleteModel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteModelResponse> deleteModel(DeleteModelRequest deleteModelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteModel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteModelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteModelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteModelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteModelRequest, DeleteModelResponse>()
                            .withOperationName("DeleteModel").withMarshaller(new DeleteModelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteModelRequest));
            CompletableFuture<DeleteModelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an Amazon Lookout for Vision project.
     * </p>
     * <p>
     * To delete a project, you must first delete each version of the model associated with the project. To delete a
     * model use the <a>DeleteModel</a> operation.
     * </p>
     * <p>
     * You also have to delete the dataset(s) associated with the model. For more information, see <a>DeleteDataset</a>.
     * The images referenced by the training and test datasets aren't deleted.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:DeleteProject</code> operation.
     * </p>
     *
     * @param deleteProjectRequest
     * @return A Java Future containing the result of the DeleteProject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You are not authorized to perform the action.</li>
     *         <li>InternalServerException Amazon Lookout for Vision experienced a service issue. Try your call again.</li>
     *         <li>ValidationException An input validation error occured. For example, invalid characters in a project
     *         name, or if a pagination token is invalid.</li>
     *         <li>ConflictException The update or deletion of a resource caused an inconsistent state.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>ThrottlingException Amazon Lookout for Vision is temporarily unable to process the request. Try your
     *         call again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutVisionException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LookoutVisionAsyncClient.DeleteProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/DeleteProject" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteProjectResponse> deleteProject(DeleteProjectRequest deleteProjectRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteProject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteProjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteProjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteProjectRequest, DeleteProjectResponse>()
                            .withOperationName("DeleteProject")
                            .withMarshaller(new DeleteProjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteProjectRequest));
            CompletableFuture<DeleteProjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describe an Amazon Lookout for Vision dataset.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:DescribeDataset</code> operation.
     * </p>
     *
     * @param describeDatasetRequest
     * @return A Java Future containing the result of the DescribeDataset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You are not authorized to perform the action.</li>
     *         <li>InternalServerException Amazon Lookout for Vision experienced a service issue. Try your call again.</li>
     *         <li>ValidationException An input validation error occured. For example, invalid characters in a project
     *         name, or if a pagination token is invalid.</li>
     *         <li>ConflictException The update or deletion of a resource caused an inconsistent state.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>ThrottlingException Amazon Lookout for Vision is temporarily unable to process the request. Try your
     *         call again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutVisionException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LookoutVisionAsyncClient.DescribeDataset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/DescribeDataset" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDatasetResponse> describeDataset(DescribeDatasetRequest describeDatasetRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDatasetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDataset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDatasetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDatasetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDatasetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDatasetRequest, DescribeDatasetResponse>()
                            .withOperationName("DescribeDataset")
                            .withMarshaller(new DescribeDatasetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeDatasetRequest));
            CompletableFuture<DescribeDatasetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes a version of an Amazon Lookout for Vision model.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:DescribeModel</code> operation.
     * </p>
     *
     * @param describeModelRequest
     * @return A Java Future containing the result of the DescribeModel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You are not authorized to perform the action.</li>
     *         <li>InternalServerException Amazon Lookout for Vision experienced a service issue. Try your call again.</li>
     *         <li>ValidationException An input validation error occured. For example, invalid characters in a project
     *         name, or if a pagination token is invalid.</li>
     *         <li>ConflictException The update or deletion of a resource caused an inconsistent state.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>ThrottlingException Amazon Lookout for Vision is temporarily unable to process the request. Try your
     *         call again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutVisionException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LookoutVisionAsyncClient.DescribeModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/DescribeModel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeModelResponse> describeModel(DescribeModelRequest describeModelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeModel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeModelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeModelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeModelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeModelRequest, DescribeModelResponse>()
                            .withOperationName("DescribeModel")
                            .withMarshaller(new DescribeModelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeModelRequest));
            CompletableFuture<DescribeModelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes an Amazon Lookout for Vision project.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:DescribeProject</code> operation.
     * </p>
     *
     * @param describeProjectRequest
     * @return A Java Future containing the result of the DescribeProject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You are not authorized to perform the action.</li>
     *         <li>InternalServerException Amazon Lookout for Vision experienced a service issue. Try your call again.</li>
     *         <li>ValidationException An input validation error occured. For example, invalid characters in a project
     *         name, or if a pagination token is invalid.</li>
     *         <li>ConflictException The update or deletion of a resource caused an inconsistent state.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>ThrottlingException Amazon Lookout for Vision is temporarily unable to process the request. Try your
     *         call again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutVisionException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LookoutVisionAsyncClient.DescribeProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/DescribeProject" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeProjectResponse> describeProject(DescribeProjectRequest describeProjectRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeProject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeProjectResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeProjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeProjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeProjectRequest, DescribeProjectResponse>()
                            .withOperationName("DescribeProject")
                            .withMarshaller(new DescribeProjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeProjectRequest));
            CompletableFuture<DescribeProjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Detects anomalies in an image that you supply.
     * </p>
     * <p>
     * The response from <code>DetectAnomalies</code> includes a boolean prediction that the image contains one or more
     * anomalies and a confidence value for the prediction.
     * </p>
     * <note>
     * <p>
     * Before calling <code>DetectAnomalies</code>, you must first start your model with the <a>StartModel</a>
     * operation. You are charged for the amount of time, in minutes, that a model runs and for the number of anomaly
     * detection units that your model uses. If you are not using a model, use the <a>StopModel</a> operation to stop
     * your model.
     * </p>
     * </note>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:DetectAnomalies</code> operation.
     * </p>
     *
     * @param detectAnomaliesRequest
     * @param requestBody
     *        Functional interface that can be implemented to produce the request content in a non-blocking manner. The
     *        size of the content is expected to be known up front. See {@link AsyncRequestBody} for specific details on
     *        implementing this interface as well as links to precanned implementations for common scenarios like
     *        uploading from a file. The service documentation for the request content is as follows '
     *        <p>
     *        The unencrypted image bytes that you want to analyze.
     *        </p>
     *        '
     * @return A Java Future containing the result of the DetectAnomalies operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You are not authorized to perform the action.</li>
     *         <li>InternalServerException Amazon Lookout for Vision experienced a service issue. Try your call again.</li>
     *         <li>ValidationException An input validation error occured. For example, invalid characters in a project
     *         name, or if a pagination token is invalid.</li>
     *         <li>ConflictException The update or deletion of a resource caused an inconsistent state.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>ThrottlingException Amazon Lookout for Vision is temporarily unable to process the request. Try your
     *         call again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutVisionException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LookoutVisionAsyncClient.DetectAnomalies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/DetectAnomalies" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DetectAnomaliesResponse> detectAnomalies(DetectAnomaliesRequest detectAnomaliesRequest,
            AsyncRequestBody requestBody) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, detectAnomaliesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DetectAnomalies");
            if (!isSignerOverridden(clientConfiguration)) {
                detectAnomaliesRequest = applySignerOverride(detectAnomaliesRequest, AsyncAws4Signer.create());
            }
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DetectAnomaliesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DetectAnomaliesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DetectAnomaliesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DetectAnomaliesRequest, DetectAnomaliesResponse>()
                            .withOperationName("DetectAnomalies")
                            .withMarshaller(
                                    AsyncStreamingRequestMarshaller.builder()
                                            .delegateMarshaller(new DetectAnomaliesRequestMarshaller(protocolFactory))
                                            .asyncRequestBody(requestBody).requiresLength(true).build())
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withAsyncRequestBody(requestBody)
                            .withInput(detectAnomaliesRequest));
            CompletableFuture<DetectAnomaliesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the JSON Lines within a dataset. An Amazon Lookout for Vision JSON Line contains the anomaly information
     * for a single image, including the image location and the assigned label.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:ListDatasetEntries</code> operation.
     * </p>
     *
     * @param listDatasetEntriesRequest
     * @return A Java Future containing the result of the ListDatasetEntries operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You are not authorized to perform the action.</li>
     *         <li>InternalServerException Amazon Lookout for Vision experienced a service issue. Try your call again.</li>
     *         <li>ValidationException An input validation error occured. For example, invalid characters in a project
     *         name, or if a pagination token is invalid.</li>
     *         <li>ConflictException The update or deletion of a resource caused an inconsistent state.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>ThrottlingException Amazon Lookout for Vision is temporarily unable to process the request. Try your
     *         call again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutVisionException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LookoutVisionAsyncClient.ListDatasetEntries
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/ListDatasetEntries"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDatasetEntriesResponse> listDatasetEntries(ListDatasetEntriesRequest listDatasetEntriesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDatasetEntriesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDatasetEntries");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDatasetEntriesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDatasetEntriesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDatasetEntriesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDatasetEntriesRequest, ListDatasetEntriesResponse>()
                            .withOperationName("ListDatasetEntries")
                            .withMarshaller(new ListDatasetEntriesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listDatasetEntriesRequest));
            CompletableFuture<ListDatasetEntriesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the JSON Lines within a dataset. An Amazon Lookout for Vision JSON Line contains the anomaly information
     * for a single image, including the image location and the assigned label.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:ListDatasetEntries</code> operation.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listDatasetEntries(software.amazon.awssdk.services.lookoutvision.model.ListDatasetEntriesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lookoutvision.paginators.ListDatasetEntriesPublisher publisher = client.listDatasetEntriesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lookoutvision.paginators.ListDatasetEntriesPublisher publisher = client.listDatasetEntriesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.lookoutvision.model.ListDatasetEntriesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.lookoutvision.model.ListDatasetEntriesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDatasetEntries(software.amazon.awssdk.services.lookoutvision.model.ListDatasetEntriesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listDatasetEntriesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You are not authorized to perform the action.</li>
     *         <li>InternalServerException Amazon Lookout for Vision experienced a service issue. Try your call again.</li>
     *         <li>ValidationException An input validation error occured. For example, invalid characters in a project
     *         name, or if a pagination token is invalid.</li>
     *         <li>ConflictException The update or deletion of a resource caused an inconsistent state.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>ThrottlingException Amazon Lookout for Vision is temporarily unable to process the request. Try your
     *         call again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutVisionException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LookoutVisionAsyncClient.ListDatasetEntries
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/ListDatasetEntries"
     *      target="_top">AWS API Documentation</a>
     */
    public ListDatasetEntriesPublisher listDatasetEntriesPaginator(ListDatasetEntriesRequest listDatasetEntriesRequest) {
        return new ListDatasetEntriesPublisher(this, applyPaginatorUserAgent(listDatasetEntriesRequest));
    }

    /**
     * <p>
     * Lists the versions of a model in an Amazon Lookout for Vision project.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:ListModels</code> operation.
     * </p>
     *
     * @param listModelsRequest
     * @return A Java Future containing the result of the ListModels operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You are not authorized to perform the action.</li>
     *         <li>InternalServerException Amazon Lookout for Vision experienced a service issue. Try your call again.</li>
     *         <li>ValidationException An input validation error occured. For example, invalid characters in a project
     *         name, or if a pagination token is invalid.</li>
     *         <li>ConflictException The update or deletion of a resource caused an inconsistent state.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>ThrottlingException Amazon Lookout for Vision is temporarily unable to process the request. Try your
     *         call again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutVisionException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LookoutVisionAsyncClient.ListModels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/ListModels" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListModelsResponse> listModels(ListModelsRequest listModelsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listModelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListModels");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListModelsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListModelsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListModelsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListModelsRequest, ListModelsResponse>().withOperationName("ListModels")
                            .withMarshaller(new ListModelsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listModelsRequest));
            CompletableFuture<ListModelsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the versions of a model in an Amazon Lookout for Vision project.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:ListModels</code> operation.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listModels(software.amazon.awssdk.services.lookoutvision.model.ListModelsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lookoutvision.paginators.ListModelsPublisher publisher = client.listModelsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lookoutvision.paginators.ListModelsPublisher publisher = client.listModelsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.lookoutvision.model.ListModelsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.lookoutvision.model.ListModelsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listModels(software.amazon.awssdk.services.lookoutvision.model.ListModelsRequest)} operation.</b>
     * </p>
     *
     * @param listModelsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You are not authorized to perform the action.</li>
     *         <li>InternalServerException Amazon Lookout for Vision experienced a service issue. Try your call again.</li>
     *         <li>ValidationException An input validation error occured. For example, invalid characters in a project
     *         name, or if a pagination token is invalid.</li>
     *         <li>ConflictException The update or deletion of a resource caused an inconsistent state.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>ThrottlingException Amazon Lookout for Vision is temporarily unable to process the request. Try your
     *         call again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutVisionException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LookoutVisionAsyncClient.ListModels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/ListModels" target="_top">AWS API
     *      Documentation</a>
     */
    public ListModelsPublisher listModelsPaginator(ListModelsRequest listModelsRequest) {
        return new ListModelsPublisher(this, applyPaginatorUserAgent(listModelsRequest));
    }

    /**
     * <p>
     * Lists the Amazon Lookout for Vision projects in your AWS account.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:ListProjects</code> operation.
     * </p>
     *
     * @param listProjectsRequest
     * @return A Java Future containing the result of the ListProjects operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You are not authorized to perform the action.</li>
     *         <li>InternalServerException Amazon Lookout for Vision experienced a service issue. Try your call again.</li>
     *         <li>ValidationException An input validation error occured. For example, invalid characters in a project
     *         name, or if a pagination token is invalid.</li>
     *         <li>ConflictException The update or deletion of a resource caused an inconsistent state.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>ThrottlingException Amazon Lookout for Vision is temporarily unable to process the request. Try your
     *         call again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutVisionException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LookoutVisionAsyncClient.ListProjects
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/ListProjects" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListProjectsResponse> listProjects(ListProjectsRequest listProjectsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listProjectsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProjects");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListProjectsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListProjectsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListProjectsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListProjectsRequest, ListProjectsResponse>()
                            .withOperationName("ListProjects").withMarshaller(new ListProjectsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listProjectsRequest));
            CompletableFuture<ListProjectsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the Amazon Lookout for Vision projects in your AWS account.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:ListProjects</code> operation.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listProjects(software.amazon.awssdk.services.lookoutvision.model.ListProjectsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lookoutvision.paginators.ListProjectsPublisher publisher = client.listProjectsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.lookoutvision.paginators.ListProjectsPublisher publisher = client.listProjectsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.lookoutvision.model.ListProjectsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.lookoutvision.model.ListProjectsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listProjects(software.amazon.awssdk.services.lookoutvision.model.ListProjectsRequest)} operation.</b>
     * </p>
     *
     * @param listProjectsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You are not authorized to perform the action.</li>
     *         <li>InternalServerException Amazon Lookout for Vision experienced a service issue. Try your call again.</li>
     *         <li>ValidationException An input validation error occured. For example, invalid characters in a project
     *         name, or if a pagination token is invalid.</li>
     *         <li>ConflictException The update or deletion of a resource caused an inconsistent state.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>ThrottlingException Amazon Lookout for Vision is temporarily unable to process the request. Try your
     *         call again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutVisionException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LookoutVisionAsyncClient.ListProjects
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/ListProjects" target="_top">AWS
     *      API Documentation</a>
     */
    public ListProjectsPublisher listProjectsPaginator(ListProjectsRequest listProjectsRequest) {
        return new ListProjectsPublisher(this, applyPaginatorUserAgent(listProjectsRequest));
    }

    /**
     * <p>
     * Returns a list of tags attached to the specified Amazon Lookout for Vision model.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:ListTagsForResource</code> operation.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You are not authorized to perform the action.</li>
     *         <li>InternalServerException Amazon Lookout for Vision experienced a service issue. Try your call again.</li>
     *         <li>ValidationException An input validation error occured. For example, invalid characters in a project
     *         name, or if a pagination token is invalid.</li>
     *         <li>ConflictException The update or deletion of a resource caused an inconsistent state.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>ThrottlingException Amazon Lookout for Vision is temporarily unable to process the request. Try your
     *         call again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutVisionException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LookoutVisionAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts the running of the version of an Amazon Lookout for Vision model. Starting a model takes a while to
     * complete. To check the current state of the model, use <a>DescribeModel</a>.
     * </p>
     * <p>
     * A model is ready to use when its status is <code>HOSTED</code>.
     * </p>
     * <p>
     * Once the model is running, you can detect custom labels in new images by calling <a>DetectAnomalies</a>.
     * </p>
     * <note>
     * <p>
     * You are charged for the amount of time that the model is running. To stop a running model, call <a>StopModel</a>.
     * </p>
     * </note>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:StartModel</code> operation.
     * </p>
     *
     * @param startModelRequest
     * @return A Java Future containing the result of the StartModel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You are not authorized to perform the action.</li>
     *         <li>InternalServerException Amazon Lookout for Vision experienced a service issue. Try your call again.</li>
     *         <li>ValidationException An input validation error occured. For example, invalid characters in a project
     *         name, or if a pagination token is invalid.</li>
     *         <li>ConflictException The update or deletion of a resource caused an inconsistent state.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>ThrottlingException Amazon Lookout for Vision is temporarily unable to process the request. Try your
     *         call again.</li>
     *         <li>ServiceQuotaExceededException A service quota was exceeded the allowed limit. For more information,
     *         see Limits in Amazon Lookout for Vision in the Amazon Lookout for Vision Developer Guide.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutVisionException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LookoutVisionAsyncClient.StartModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/StartModel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartModelResponse> startModel(StartModelRequest startModelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartModel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartModelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StartModelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartModelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartModelRequest, StartModelResponse>().withOperationName("StartModel")
                            .withMarshaller(new StartModelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startModelRequest));
            CompletableFuture<StartModelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Stops the hosting of a running model. The operation might take a while to complete. To check the current status,
     * call <a>DescribeModel</a>.
     * </p>
     * <p>
     * After the model hosting stops, the <code>Status</code> of the model is <code>TRAINED</code>.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:StopModel</code> operation.
     * </p>
     *
     * @param stopModelRequest
     * @return A Java Future containing the result of the StopModel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You are not authorized to perform the action.</li>
     *         <li>InternalServerException Amazon Lookout for Vision experienced a service issue. Try your call again.</li>
     *         <li>ValidationException An input validation error occured. For example, invalid characters in a project
     *         name, or if a pagination token is invalid.</li>
     *         <li>ConflictException The update or deletion of a resource caused an inconsistent state.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>ThrottlingException Amazon Lookout for Vision is temporarily unable to process the request. Try your
     *         call again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutVisionException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LookoutVisionAsyncClient.StopModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/StopModel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StopModelResponse> stopModel(StopModelRequest stopModelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopModel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopModelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StopModelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopModelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopModelRequest, StopModelResponse>().withOperationName("StopModel")
                            .withMarshaller(new StopModelRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(stopModelRequest));
            CompletableFuture<StopModelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds one or more key-value tags to an Amazon Lookout for Vision model. For more information, see <i>Tagging a
     * model</i> in the <i>Amazon Lookout for Vision Developer Guide</i>.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:TagResource</code> operation.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You are not authorized to perform the action.</li>
     *         <li>InternalServerException Amazon Lookout for Vision experienced a service issue. Try your call again.</li>
     *         <li>ValidationException An input validation error occured. For example, invalid characters in a project
     *         name, or if a pagination token is invalid.</li>
     *         <li>ConflictException The update or deletion of a resource caused an inconsistent state.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>ThrottlingException Amazon Lookout for Vision is temporarily unable to process the request. Try your
     *         call again.</li>
     *         <li>ServiceQuotaExceededException A service quota was exceeded the allowed limit. For more information,
     *         see Limits in Amazon Lookout for Vision in the Amazon Lookout for Vision Developer Guide.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutVisionException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LookoutVisionAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes one or more tags from an Amazon Lookout for Vision model. For more information, see <i>Tagging a
     * model</i> in the <i>Amazon Lookout for Vision Developer Guide</i>.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:UntagResource</code> operation.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You are not authorized to perform the action.</li>
     *         <li>InternalServerException Amazon Lookout for Vision experienced a service issue. Try your call again.</li>
     *         <li>ValidationException An input validation error occured. For example, invalid characters in a project
     *         name, or if a pagination token is invalid.</li>
     *         <li>ConflictException The update or deletion of a resource caused an inconsistent state.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>ThrottlingException Amazon Lookout for Vision is temporarily unable to process the request. Try your
     *         call again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutVisionException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LookoutVisionAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds one or more JSON Line entries to a dataset. A JSON Line includes information about an image used for
     * training or testing an Amazon Lookout for Vision model. The following is an example JSON Line.
     * </p>
     * <p>
     * Updating a dataset might take a while to complete. To check the current status, call <a>DescribeDataset</a> and
     * check the <code>Status</code> field in the response.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:UpdateDatasetEntries</code> operation.
     * </p>
     *
     * @param updateDatasetEntriesRequest
     * @return A Java Future containing the result of the UpdateDatasetEntries operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You are not authorized to perform the action.</li>
     *         <li>InternalServerException Amazon Lookout for Vision experienced a service issue. Try your call again.</li>
     *         <li>ValidationException An input validation error occured. For example, invalid characters in a project
     *         name, or if a pagination token is invalid.</li>
     *         <li>ConflictException The update or deletion of a resource caused an inconsistent state.</li>
     *         <li>ResourceNotFoundException The resource could not be found.</li>
     *         <li>ThrottlingException Amazon Lookout for Vision is temporarily unable to process the request. Try your
     *         call again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LookoutVisionException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample LookoutVisionAsyncClient.UpdateDatasetEntries
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/UpdateDatasetEntries"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDatasetEntriesResponse> updateDatasetEntries(
            UpdateDatasetEntriesRequest updateDatasetEntriesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDatasetEntriesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDatasetEntries");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDatasetEntriesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDatasetEntriesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDatasetEntriesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDatasetEntriesRequest, UpdateDatasetEntriesResponse>()
                            .withOperationName("UpdateDatasetEntries")
                            .withMarshaller(new UpdateDatasetEntriesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateDatasetEntriesRequest));
            CompletableFuture<UpdateDatasetEntriesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(LookoutVisionException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends LookoutVisionRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private <T extends LookoutVisionRequest> T applySignerOverride(T request, Signer signer) {
        if (request.overrideConfiguration().flatMap(c -> c.signer()).isPresent()) {
            return request;
        }
        Consumer<AwsRequestOverrideConfiguration.Builder> signerOverride = b -> b.signer(signer).build();
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(signerOverride).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(signerOverride).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private static boolean isSignerOverridden(SdkClientConfiguration clientConfiguration) {
        return Boolean.TRUE.equals(clientConfiguration.option(SdkClientOption.SIGNER_OVERRIDDEN));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
