/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lookoutvision;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.runtime.transform.StreamingRequestMarshaller;
import software.amazon.awssdk.core.sync.RequestBody;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.lookoutvision.internal.LookoutVisionServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.lookoutvision.model.AccessDeniedException;
import software.amazon.awssdk.services.lookoutvision.model.ConflictException;
import software.amazon.awssdk.services.lookoutvision.model.CreateDatasetRequest;
import software.amazon.awssdk.services.lookoutvision.model.CreateDatasetResponse;
import software.amazon.awssdk.services.lookoutvision.model.CreateModelRequest;
import software.amazon.awssdk.services.lookoutvision.model.CreateModelResponse;
import software.amazon.awssdk.services.lookoutvision.model.CreateProjectRequest;
import software.amazon.awssdk.services.lookoutvision.model.CreateProjectResponse;
import software.amazon.awssdk.services.lookoutvision.model.DeleteDatasetRequest;
import software.amazon.awssdk.services.lookoutvision.model.DeleteDatasetResponse;
import software.amazon.awssdk.services.lookoutvision.model.DeleteModelRequest;
import software.amazon.awssdk.services.lookoutvision.model.DeleteModelResponse;
import software.amazon.awssdk.services.lookoutvision.model.DeleteProjectRequest;
import software.amazon.awssdk.services.lookoutvision.model.DeleteProjectResponse;
import software.amazon.awssdk.services.lookoutvision.model.DescribeDatasetRequest;
import software.amazon.awssdk.services.lookoutvision.model.DescribeDatasetResponse;
import software.amazon.awssdk.services.lookoutvision.model.DescribeModelPackagingJobRequest;
import software.amazon.awssdk.services.lookoutvision.model.DescribeModelPackagingJobResponse;
import software.amazon.awssdk.services.lookoutvision.model.DescribeModelRequest;
import software.amazon.awssdk.services.lookoutvision.model.DescribeModelResponse;
import software.amazon.awssdk.services.lookoutvision.model.DescribeProjectRequest;
import software.amazon.awssdk.services.lookoutvision.model.DescribeProjectResponse;
import software.amazon.awssdk.services.lookoutvision.model.DetectAnomaliesRequest;
import software.amazon.awssdk.services.lookoutvision.model.DetectAnomaliesResponse;
import software.amazon.awssdk.services.lookoutvision.model.InternalServerException;
import software.amazon.awssdk.services.lookoutvision.model.ListDatasetEntriesRequest;
import software.amazon.awssdk.services.lookoutvision.model.ListDatasetEntriesResponse;
import software.amazon.awssdk.services.lookoutvision.model.ListModelPackagingJobsRequest;
import software.amazon.awssdk.services.lookoutvision.model.ListModelPackagingJobsResponse;
import software.amazon.awssdk.services.lookoutvision.model.ListModelsRequest;
import software.amazon.awssdk.services.lookoutvision.model.ListModelsResponse;
import software.amazon.awssdk.services.lookoutvision.model.ListProjectsRequest;
import software.amazon.awssdk.services.lookoutvision.model.ListProjectsResponse;
import software.amazon.awssdk.services.lookoutvision.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.lookoutvision.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.lookoutvision.model.LookoutVisionException;
import software.amazon.awssdk.services.lookoutvision.model.ResourceNotFoundException;
import software.amazon.awssdk.services.lookoutvision.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.lookoutvision.model.StartModelPackagingJobRequest;
import software.amazon.awssdk.services.lookoutvision.model.StartModelPackagingJobResponse;
import software.amazon.awssdk.services.lookoutvision.model.StartModelRequest;
import software.amazon.awssdk.services.lookoutvision.model.StartModelResponse;
import software.amazon.awssdk.services.lookoutvision.model.StopModelRequest;
import software.amazon.awssdk.services.lookoutvision.model.StopModelResponse;
import software.amazon.awssdk.services.lookoutvision.model.TagResourceRequest;
import software.amazon.awssdk.services.lookoutvision.model.TagResourceResponse;
import software.amazon.awssdk.services.lookoutvision.model.ThrottlingException;
import software.amazon.awssdk.services.lookoutvision.model.UntagResourceRequest;
import software.amazon.awssdk.services.lookoutvision.model.UntagResourceResponse;
import software.amazon.awssdk.services.lookoutvision.model.UpdateDatasetEntriesRequest;
import software.amazon.awssdk.services.lookoutvision.model.UpdateDatasetEntriesResponse;
import software.amazon.awssdk.services.lookoutvision.model.ValidationException;
import software.amazon.awssdk.services.lookoutvision.transform.CreateDatasetRequestMarshaller;
import software.amazon.awssdk.services.lookoutvision.transform.CreateModelRequestMarshaller;
import software.amazon.awssdk.services.lookoutvision.transform.CreateProjectRequestMarshaller;
import software.amazon.awssdk.services.lookoutvision.transform.DeleteDatasetRequestMarshaller;
import software.amazon.awssdk.services.lookoutvision.transform.DeleteModelRequestMarshaller;
import software.amazon.awssdk.services.lookoutvision.transform.DeleteProjectRequestMarshaller;
import software.amazon.awssdk.services.lookoutvision.transform.DescribeDatasetRequestMarshaller;
import software.amazon.awssdk.services.lookoutvision.transform.DescribeModelPackagingJobRequestMarshaller;
import software.amazon.awssdk.services.lookoutvision.transform.DescribeModelRequestMarshaller;
import software.amazon.awssdk.services.lookoutvision.transform.DescribeProjectRequestMarshaller;
import software.amazon.awssdk.services.lookoutvision.transform.DetectAnomaliesRequestMarshaller;
import software.amazon.awssdk.services.lookoutvision.transform.ListDatasetEntriesRequestMarshaller;
import software.amazon.awssdk.services.lookoutvision.transform.ListModelPackagingJobsRequestMarshaller;
import software.amazon.awssdk.services.lookoutvision.transform.ListModelsRequestMarshaller;
import software.amazon.awssdk.services.lookoutvision.transform.ListProjectsRequestMarshaller;
import software.amazon.awssdk.services.lookoutvision.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.lookoutvision.transform.StartModelPackagingJobRequestMarshaller;
import software.amazon.awssdk.services.lookoutvision.transform.StartModelRequestMarshaller;
import software.amazon.awssdk.services.lookoutvision.transform.StopModelRequestMarshaller;
import software.amazon.awssdk.services.lookoutvision.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.lookoutvision.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.lookoutvision.transform.UpdateDatasetEntriesRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link LookoutVisionClient}.
 *
 * @see LookoutVisionClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultLookoutVisionClient implements LookoutVisionClient {
    private static final Logger log = Logger.loggerFor(DefaultLookoutVisionClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultLookoutVisionClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a new dataset in an Amazon Lookout for Vision project. <code>CreateDataset</code> can create a training
     * or a test dataset from a valid dataset source (<code>DatasetSource</code>).
     * </p>
     * <p>
     * If you want a single dataset project, specify <code>train</code> for the value of <code>DatasetType</code>.
     * </p>
     * <p>
     * To have a project with separate training and test datasets, call <code>CreateDataset</code> twice. On the first
     * call, specify <code>train</code> for the value of <code>DatasetType</code>. On the second call, specify
     * <code>test</code> for the value of <code>DatasetType</code>.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:CreateDataset</code> operation.
     * </p>
     *
     * @param createDatasetRequest
     * @return Result of the CreateDataset operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerException
     *         Amazon Lookout for Vision experienced a service issue. Try your call again.
     * @throws ValidationException
     *         An input validation error occured. For example, invalid characters in a project name, or if a pagination
     *         token is invalid.
     * @throws ConflictException
     *         The update or deletion of a resource caused an inconsistent state.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ThrottlingException
     *         Amazon Lookout for Vision is temporarily unable to process the request. Try your call again.
     * @throws ServiceQuotaExceededException
     *         A service quota was exceeded the allowed limit. For more information, see Limits in Amazon Lookout for
     *         Vision in the Amazon Lookout for Vision Developer Guide.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutVisionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutVisionClient.CreateDataset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/CreateDataset" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateDatasetResponse createDataset(CreateDatasetRequest createDatasetRequest) throws AccessDeniedException,
            InternalServerException, ValidationException, ConflictException, ResourceNotFoundException, ThrottlingException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, LookoutVisionException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateDatasetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateDatasetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDatasetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDatasetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDataset");

            return clientHandler.execute(new ClientExecutionParams<CreateDatasetRequest, CreateDatasetResponse>()
                    .withOperationName("CreateDataset").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createDatasetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateDatasetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new version of a model within an an Amazon Lookout for Vision project. <code>CreateModel</code> is an
     * asynchronous operation in which Amazon Lookout for Vision trains, tests, and evaluates a new version of a model.
     * </p>
     * <p>
     * To get the current status, check the <code>Status</code> field returned in the response from
     * <a>DescribeModel</a>.
     * </p>
     * <p>
     * If the project has a single dataset, Amazon Lookout for Vision internally splits the dataset to create a training
     * and a test dataset. If the project has a training and a test dataset, Lookout for Vision uses the respective
     * datasets to train and test the model.
     * </p>
     * <p>
     * After training completes, the evaluation metrics are stored at the location specified in
     * <code>OutputConfig</code>.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:CreateModel</code> operation. If you want
     * to tag your model, you also require permission to the <code>lookoutvision:TagResource</code> operation.
     * </p>
     *
     * @param createModelRequest
     * @return Result of the CreateModel operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerException
     *         Amazon Lookout for Vision experienced a service issue. Try your call again.
     * @throws ValidationException
     *         An input validation error occured. For example, invalid characters in a project name, or if a pagination
     *         token is invalid.
     * @throws ConflictException
     *         The update or deletion of a resource caused an inconsistent state.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ThrottlingException
     *         Amazon Lookout for Vision is temporarily unable to process the request. Try your call again.
     * @throws ServiceQuotaExceededException
     *         A service quota was exceeded the allowed limit. For more information, see Limits in Amazon Lookout for
     *         Vision in the Amazon Lookout for Vision Developer Guide.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutVisionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutVisionClient.CreateModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/CreateModel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateModelResponse createModel(CreateModelRequest createModelRequest) throws AccessDeniedException,
            InternalServerException, ValidationException, ConflictException, ResourceNotFoundException, ThrottlingException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, LookoutVisionException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateModelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateModelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createModelRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateModel");

            return clientHandler.execute(new ClientExecutionParams<CreateModelRequest, CreateModelResponse>()
                    .withOperationName("CreateModel").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(createModelRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateModelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an empty Amazon Lookout for Vision project. After you create the project, add a dataset by calling
     * <a>CreateDataset</a>.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:CreateProject</code> operation.
     * </p>
     *
     * @param createProjectRequest
     * @return Result of the CreateProject operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerException
     *         Amazon Lookout for Vision experienced a service issue. Try your call again.
     * @throws ValidationException
     *         An input validation error occured. For example, invalid characters in a project name, or if a pagination
     *         token is invalid.
     * @throws ConflictException
     *         The update or deletion of a resource caused an inconsistent state.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ThrottlingException
     *         Amazon Lookout for Vision is temporarily unable to process the request. Try your call again.
     * @throws ServiceQuotaExceededException
     *         A service quota was exceeded the allowed limit. For more information, see Limits in Amazon Lookout for
     *         Vision in the Amazon Lookout for Vision Developer Guide.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutVisionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutVisionClient.CreateProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/CreateProject" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateProjectResponse createProject(CreateProjectRequest createProjectRequest) throws AccessDeniedException,
            InternalServerException, ValidationException, ConflictException, ResourceNotFoundException, ThrottlingException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, LookoutVisionException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateProjectResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createProjectRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateProject");

            return clientHandler.execute(new ClientExecutionParams<CreateProjectRequest, CreateProjectResponse>()
                    .withOperationName("CreateProject").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createProjectRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateProjectRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an existing Amazon Lookout for Vision <code>dataset</code>.
     * </p>
     * <p>
     * If your the project has a single dataset, you must create a new dataset before you can create a model.
     * </p>
     * <p>
     * If you project has a training dataset and a test dataset consider the following.
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you delete the test dataset, your project reverts to a single dataset project. If you then train the model,
     * Amazon Lookout for Vision internally splits the remaining dataset into a training and test dataset.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you delete the training dataset, you must create a training dataset before you can create a model.
     * </p>
     * </li>
     * </ul>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:DeleteDataset</code> operation.
     * </p>
     *
     * @param deleteDatasetRequest
     * @return Result of the DeleteDataset operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerException
     *         Amazon Lookout for Vision experienced a service issue. Try your call again.
     * @throws ValidationException
     *         An input validation error occured. For example, invalid characters in a project name, or if a pagination
     *         token is invalid.
     * @throws ConflictException
     *         The update or deletion of a resource caused an inconsistent state.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ThrottlingException
     *         Amazon Lookout for Vision is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutVisionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutVisionClient.DeleteDataset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/DeleteDataset" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteDatasetResponse deleteDataset(DeleteDatasetRequest deleteDatasetRequest) throws AccessDeniedException,
            InternalServerException, ValidationException, ConflictException, ResourceNotFoundException, ThrottlingException,
            AwsServiceException, SdkClientException, LookoutVisionException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteDatasetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteDatasetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteDatasetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDatasetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDataset");

            return clientHandler.execute(new ClientExecutionParams<DeleteDatasetRequest, DeleteDatasetResponse>()
                    .withOperationName("DeleteDataset").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteDatasetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteDatasetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an Amazon Lookout for Vision model. You can't delete a running model. To stop a running model, use the
     * <a>StopModel</a> operation.
     * </p>
     * <p>
     * It might take a few seconds to delete a model. To determine if a model has been deleted, call <a>ListModels</a>
     * and check if the version of the model (<code>ModelVersion</code>) is in the <code>Models</code> array.
     * </p>
     * <p/>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:DeleteModel</code> operation.
     * </p>
     *
     * @param deleteModelRequest
     * @return Result of the DeleteModel operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerException
     *         Amazon Lookout for Vision experienced a service issue. Try your call again.
     * @throws ValidationException
     *         An input validation error occured. For example, invalid characters in a project name, or if a pagination
     *         token is invalid.
     * @throws ConflictException
     *         The update or deletion of a resource caused an inconsistent state.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ThrottlingException
     *         Amazon Lookout for Vision is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutVisionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutVisionClient.DeleteModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/DeleteModel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteModelResponse deleteModel(DeleteModelRequest deleteModelRequest) throws AccessDeniedException,
            InternalServerException, ValidationException, ConflictException, ResourceNotFoundException, ThrottlingException,
            AwsServiceException, SdkClientException, LookoutVisionException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteModelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteModelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteModelRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteModel");

            return clientHandler.execute(new ClientExecutionParams<DeleteModelRequest, DeleteModelResponse>()
                    .withOperationName("DeleteModel").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteModelRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteModelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an Amazon Lookout for Vision project.
     * </p>
     * <p>
     * To delete a project, you must first delete each version of the model associated with the project. To delete a
     * model use the <a>DeleteModel</a> operation.
     * </p>
     * <p>
     * You also have to delete the dataset(s) associated with the model. For more information, see <a>DeleteDataset</a>.
     * The images referenced by the training and test datasets aren't deleted.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:DeleteProject</code> operation.
     * </p>
     *
     * @param deleteProjectRequest
     * @return Result of the DeleteProject operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerException
     *         Amazon Lookout for Vision experienced a service issue. Try your call again.
     * @throws ValidationException
     *         An input validation error occured. For example, invalid characters in a project name, or if a pagination
     *         token is invalid.
     * @throws ConflictException
     *         The update or deletion of a resource caused an inconsistent state.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ThrottlingException
     *         Amazon Lookout for Vision is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutVisionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutVisionClient.DeleteProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/DeleteProject" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteProjectResponse deleteProject(DeleteProjectRequest deleteProjectRequest) throws AccessDeniedException,
            InternalServerException, ValidationException, ConflictException, ResourceNotFoundException, ThrottlingException,
            AwsServiceException, SdkClientException, LookoutVisionException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteProjectResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteProjectRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteProject");

            return clientHandler.execute(new ClientExecutionParams<DeleteProjectRequest, DeleteProjectResponse>()
                    .withOperationName("DeleteProject").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteProjectRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteProjectRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describe an Amazon Lookout for Vision dataset.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:DescribeDataset</code> operation.
     * </p>
     *
     * @param describeDatasetRequest
     * @return Result of the DescribeDataset operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerException
     *         Amazon Lookout for Vision experienced a service issue. Try your call again.
     * @throws ValidationException
     *         An input validation error occured. For example, invalid characters in a project name, or if a pagination
     *         token is invalid.
     * @throws ConflictException
     *         The update or deletion of a resource caused an inconsistent state.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ThrottlingException
     *         Amazon Lookout for Vision is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutVisionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutVisionClient.DescribeDataset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/DescribeDataset" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeDatasetResponse describeDataset(DescribeDatasetRequest describeDatasetRequest) throws AccessDeniedException,
            InternalServerException, ValidationException, ConflictException, ResourceNotFoundException, ThrottlingException,
            AwsServiceException, SdkClientException, LookoutVisionException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeDatasetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeDatasetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeDatasetRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDatasetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDataset");

            return clientHandler.execute(new ClientExecutionParams<DescribeDatasetRequest, DescribeDatasetResponse>()
                    .withOperationName("DescribeDataset").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeDatasetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeDatasetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes a version of an Amazon Lookout for Vision model.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:DescribeModel</code> operation.
     * </p>
     *
     * @param describeModelRequest
     * @return Result of the DescribeModel operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerException
     *         Amazon Lookout for Vision experienced a service issue. Try your call again.
     * @throws ValidationException
     *         An input validation error occured. For example, invalid characters in a project name, or if a pagination
     *         token is invalid.
     * @throws ConflictException
     *         The update or deletion of a resource caused an inconsistent state.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ThrottlingException
     *         Amazon Lookout for Vision is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutVisionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutVisionClient.DescribeModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/DescribeModel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeModelResponse describeModel(DescribeModelRequest describeModelRequest) throws AccessDeniedException,
            InternalServerException, ValidationException, ConflictException, ResourceNotFoundException, ThrottlingException,
            AwsServiceException, SdkClientException, LookoutVisionException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeModelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeModelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeModelRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeModel");

            return clientHandler.execute(new ClientExecutionParams<DescribeModelRequest, DescribeModelResponse>()
                    .withOperationName("DescribeModel").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeModelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeModelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes an Amazon Lookout for Vision model packaging job.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:DescribeModelPackagingJob</code>
     * operation.
     * </p>
     * <p>
     * For more information, see <i>Using your Amazon Lookout for Vision model on an edge device</i> in the Amazon
     * Lookout for Vision Developer Guide.
     * </p>
     *
     * @param describeModelPackagingJobRequest
     * @return Result of the DescribeModelPackagingJob operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerException
     *         Amazon Lookout for Vision experienced a service issue. Try your call again.
     * @throws ValidationException
     *         An input validation error occured. For example, invalid characters in a project name, or if a pagination
     *         token is invalid.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ThrottlingException
     *         Amazon Lookout for Vision is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutVisionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutVisionClient.DescribeModelPackagingJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/DescribeModelPackagingJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeModelPackagingJobResponse describeModelPackagingJob(
            DescribeModelPackagingJobRequest describeModelPackagingJobRequest) throws AccessDeniedException,
            InternalServerException, ValidationException, ResourceNotFoundException, ThrottlingException, AwsServiceException,
            SdkClientException, LookoutVisionException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeModelPackagingJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeModelPackagingJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeModelPackagingJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeModelPackagingJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeModelPackagingJob");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeModelPackagingJobRequest, DescribeModelPackagingJobResponse>()
                            .withOperationName("DescribeModelPackagingJob").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeModelPackagingJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeModelPackagingJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes an Amazon Lookout for Vision project.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:DescribeProject</code> operation.
     * </p>
     *
     * @param describeProjectRequest
     * @return Result of the DescribeProject operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerException
     *         Amazon Lookout for Vision experienced a service issue. Try your call again.
     * @throws ValidationException
     *         An input validation error occured. For example, invalid characters in a project name, or if a pagination
     *         token is invalid.
     * @throws ConflictException
     *         The update or deletion of a resource caused an inconsistent state.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ThrottlingException
     *         Amazon Lookout for Vision is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutVisionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutVisionClient.DescribeProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/DescribeProject" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeProjectResponse describeProject(DescribeProjectRequest describeProjectRequest) throws AccessDeniedException,
            InternalServerException, ValidationException, ConflictException, ResourceNotFoundException, ThrottlingException,
            AwsServiceException, SdkClientException, LookoutVisionException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeProjectResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeProjectRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeProject");

            return clientHandler.execute(new ClientExecutionParams<DescribeProjectRequest, DescribeProjectResponse>()
                    .withOperationName("DescribeProject").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeProjectRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeProjectRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Detects anomalies in an image that you supply.
     * </p>
     * <p>
     * The response from <code>DetectAnomalies</code> includes a boolean prediction that the image contains one or more
     * anomalies and a confidence value for the prediction. If the model is an image segmentation model, the response
     * also includes segmentation information for each type of anomaly found in the image.
     * </p>
     * <note>
     * <p>
     * Before calling <code>DetectAnomalies</code>, you must first start your model with the <a>StartModel</a>
     * operation. You are charged for the amount of time, in minutes, that a model runs and for the number of anomaly
     * detection units that your model uses. If you are not using a model, use the <a>StopModel</a> operation to stop
     * your model.
     * </p>
     * </note>
     * <p>
     * For more information, see <i>Detecting anomalies in an image</i> in the Amazon Lookout for Vision developer
     * guide.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:DetectAnomalies</code> operation.
     * </p>
     *
     * @param detectAnomaliesRequest
     * @param requestBody
     *        The content to send to the service. A {@link RequestBody} can be created using one of several factory
     *        methods for various sources of data. For example, to create a request body from a file you can do the
     *        following.
     * 
     *        <pre>
     * {@code RequestBody.fromFile(new File("myfile.txt"))}
     * </pre>
     * 
     *        See documentation in {@link RequestBody} for additional details and which sources of data are supported.
     *        The service documentation for the request content is as follows '
     *        <p>
     *        The unencrypted image bytes that you want to analyze.
     *        </p>
     *        '
     * @return Result of the DetectAnomalies operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerException
     *         Amazon Lookout for Vision experienced a service issue. Try your call again.
     * @throws ValidationException
     *         An input validation error occured. For example, invalid characters in a project name, or if a pagination
     *         token is invalid.
     * @throws ConflictException
     *         The update or deletion of a resource caused an inconsistent state.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ThrottlingException
     *         Amazon Lookout for Vision is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutVisionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutVisionClient.DetectAnomalies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/DetectAnomalies" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DetectAnomaliesResponse detectAnomalies(DetectAnomaliesRequest detectAnomaliesRequest, RequestBody requestBody)
            throws AccessDeniedException, InternalServerException, ValidationException, ConflictException,
            ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException, LookoutVisionException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DetectAnomaliesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DetectAnomaliesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(detectAnomaliesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, detectAnomaliesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DetectAnomalies");

            return clientHandler.execute(new ClientExecutionParams<DetectAnomaliesRequest, DetectAnomaliesResponse>()
                    .withOperationName("DetectAnomalies")
                    .withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration)
                    .withInput(detectAnomaliesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withRequestBody(requestBody)
                    .withMarshaller(
                            StreamingRequestMarshaller.builder()
                                    .delegateMarshaller(new DetectAnomaliesRequestMarshaller(protocolFactory))
                                    .requestBody(requestBody).requiresLength(true).build()));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the JSON Lines within a dataset. An Amazon Lookout for Vision JSON Line contains the anomaly information
     * for a single image, including the image location and the assigned label.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:ListDatasetEntries</code> operation.
     * </p>
     *
     * @param listDatasetEntriesRequest
     * @return Result of the ListDatasetEntries operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerException
     *         Amazon Lookout for Vision experienced a service issue. Try your call again.
     * @throws ValidationException
     *         An input validation error occured. For example, invalid characters in a project name, or if a pagination
     *         token is invalid.
     * @throws ConflictException
     *         The update or deletion of a resource caused an inconsistent state.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ThrottlingException
     *         Amazon Lookout for Vision is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutVisionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutVisionClient.ListDatasetEntries
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/ListDatasetEntries"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListDatasetEntriesResponse listDatasetEntries(ListDatasetEntriesRequest listDatasetEntriesRequest)
            throws AccessDeniedException, InternalServerException, ValidationException, ConflictException,
            ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException, LookoutVisionException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDatasetEntriesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListDatasetEntriesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDatasetEntriesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDatasetEntriesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDatasetEntries");

            return clientHandler.execute(new ClientExecutionParams<ListDatasetEntriesRequest, ListDatasetEntriesResponse>()
                    .withOperationName("ListDatasetEntries").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listDatasetEntriesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListDatasetEntriesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the model packaging jobs created for an Amazon Lookout for Vision project.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:ListModelPackagingJobs</code> operation.
     * </p>
     * <p>
     * For more information, see <i>Using your Amazon Lookout for Vision model on an edge device</i> in the Amazon
     * Lookout for Vision Developer Guide.
     * </p>
     *
     * @param listModelPackagingJobsRequest
     * @return Result of the ListModelPackagingJobs operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerException
     *         Amazon Lookout for Vision experienced a service issue. Try your call again.
     * @throws ValidationException
     *         An input validation error occured. For example, invalid characters in a project name, or if a pagination
     *         token is invalid.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ThrottlingException
     *         Amazon Lookout for Vision is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutVisionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutVisionClient.ListModelPackagingJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/ListModelPackagingJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListModelPackagingJobsResponse listModelPackagingJobs(ListModelPackagingJobsRequest listModelPackagingJobsRequest)
            throws AccessDeniedException, InternalServerException, ValidationException, ResourceNotFoundException,
            ThrottlingException, AwsServiceException, SdkClientException, LookoutVisionException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListModelPackagingJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListModelPackagingJobsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listModelPackagingJobsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listModelPackagingJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListModelPackagingJobs");

            return clientHandler
                    .execute(new ClientExecutionParams<ListModelPackagingJobsRequest, ListModelPackagingJobsResponse>()
                            .withOperationName("ListModelPackagingJobs").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listModelPackagingJobsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListModelPackagingJobsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the versions of a model in an Amazon Lookout for Vision project.
     * </p>
     * <p>
     * The <code>ListModels</code> operation is eventually consistent. Recent calls to <code>CreateModel</code> might
     * take a while to appear in the response from <code>ListProjects</code>.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:ListModels</code> operation.
     * </p>
     *
     * @param listModelsRequest
     * @return Result of the ListModels operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerException
     *         Amazon Lookout for Vision experienced a service issue. Try your call again.
     * @throws ValidationException
     *         An input validation error occured. For example, invalid characters in a project name, or if a pagination
     *         token is invalid.
     * @throws ConflictException
     *         The update or deletion of a resource caused an inconsistent state.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ThrottlingException
     *         Amazon Lookout for Vision is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutVisionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutVisionClient.ListModels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/ListModels" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListModelsResponse listModels(ListModelsRequest listModelsRequest) throws AccessDeniedException,
            InternalServerException, ValidationException, ConflictException, ResourceNotFoundException, ThrottlingException,
            AwsServiceException, SdkClientException, LookoutVisionException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListModelsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListModelsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listModelsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listModelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListModels");

            return clientHandler.execute(new ClientExecutionParams<ListModelsRequest, ListModelsResponse>()
                    .withOperationName("ListModels").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listModelsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListModelsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the Amazon Lookout for Vision projects in your AWS account that are in the AWS Region in which you call
     * <code>ListProjects</code>.
     * </p>
     * <p>
     * The <code>ListProjects</code> operation is eventually consistent. Recent calls to <code>CreateProject</code> and
     * <code>DeleteProject</code> might take a while to appear in the response from <code>ListProjects</code>.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:ListProjects</code> operation.
     * </p>
     *
     * @param listProjectsRequest
     * @return Result of the ListProjects operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerException
     *         Amazon Lookout for Vision experienced a service issue. Try your call again.
     * @throws ValidationException
     *         An input validation error occured. For example, invalid characters in a project name, or if a pagination
     *         token is invalid.
     * @throws ConflictException
     *         The update or deletion of a resource caused an inconsistent state.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ThrottlingException
     *         Amazon Lookout for Vision is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutVisionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutVisionClient.ListProjects
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/ListProjects" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListProjectsResponse listProjects(ListProjectsRequest listProjectsRequest) throws AccessDeniedException,
            InternalServerException, ValidationException, ConflictException, ResourceNotFoundException, ThrottlingException,
            AwsServiceException, SdkClientException, LookoutVisionException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListProjectsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListProjectsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listProjectsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listProjectsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProjects");

            return clientHandler.execute(new ClientExecutionParams<ListProjectsRequest, ListProjectsResponse>()
                    .withOperationName("ListProjects").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listProjectsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListProjectsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of tags attached to the specified Amazon Lookout for Vision model.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:ListTagsForResource</code> operation.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerException
     *         Amazon Lookout for Vision experienced a service issue. Try your call again.
     * @throws ValidationException
     *         An input validation error occured. For example, invalid characters in a project name, or if a pagination
     *         token is invalid.
     * @throws ConflictException
     *         The update or deletion of a resource caused an inconsistent state.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ThrottlingException
     *         Amazon Lookout for Vision is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutVisionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutVisionClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws AccessDeniedException, InternalServerException, ValidationException, ConflictException,
            ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException, LookoutVisionException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts the running of the version of an Amazon Lookout for Vision model. Starting a model takes a while to
     * complete. To check the current state of the model, use <a>DescribeModel</a>.
     * </p>
     * <p>
     * A model is ready to use when its status is <code>HOSTED</code>.
     * </p>
     * <p>
     * Once the model is running, you can detect custom labels in new images by calling <a>DetectAnomalies</a>.
     * </p>
     * <note>
     * <p>
     * You are charged for the amount of time that the model is running. To stop a running model, call <a>StopModel</a>.
     * </p>
     * </note>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:StartModel</code> operation.
     * </p>
     *
     * @param startModelRequest
     * @return Result of the StartModel operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerException
     *         Amazon Lookout for Vision experienced a service issue. Try your call again.
     * @throws ValidationException
     *         An input validation error occured. For example, invalid characters in a project name, or if a pagination
     *         token is invalid.
     * @throws ConflictException
     *         The update or deletion of a resource caused an inconsistent state.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ThrottlingException
     *         Amazon Lookout for Vision is temporarily unable to process the request. Try your call again.
     * @throws ServiceQuotaExceededException
     *         A service quota was exceeded the allowed limit. For more information, see Limits in Amazon Lookout for
     *         Vision in the Amazon Lookout for Vision Developer Guide.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutVisionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutVisionClient.StartModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/StartModel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartModelResponse startModel(StartModelRequest startModelRequest) throws AccessDeniedException,
            InternalServerException, ValidationException, ConflictException, ResourceNotFoundException, ThrottlingException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, LookoutVisionException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartModelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartModelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startModelRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartModel");

            return clientHandler.execute(new ClientExecutionParams<StartModelRequest, StartModelResponse>()
                    .withOperationName("StartModel").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(startModelRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartModelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts an Amazon Lookout for Vision model packaging job. A model packaging job creates an AWS IoT Greengrass
     * component for a Lookout for Vision model. You can use the component to deploy your model to an edge device
     * managed by Greengrass.
     * </p>
     * <p>
     * Use the <a>DescribeModelPackagingJob</a> API to determine the current status of the job. The model packaging job
     * is complete if the value of <code>Status</code> is <code>SUCCEEDED</code>.
     * </p>
     * <p>
     * To deploy the component to the target device, use the component name and component version with the AWS IoT
     * Greengrass <a
     * href="https://docs.aws.amazon.com/greengrass/v2/APIReference/API_CreateDeployment.html">CreateDeployment</a> API.
     * </p>
     * <p>
     * This operation requires the following permissions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>lookoutvision:StartModelPackagingJob</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>s3:PutObject</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>s3:GetBucketLocation</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>kms:GenerateDataKey</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>greengrass:CreateComponentVersion</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>greengrass:DescribeComponent</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * (Optional) <code>greengrass:TagResource</code>. Only required if you want to tag the component.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <i>Using your Amazon Lookout for Vision model on an edge device</i> in the Amazon
     * Lookout for Vision Developer Guide.
     * </p>
     *
     * @param startModelPackagingJobRequest
     * @return Result of the StartModelPackagingJob operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerException
     *         Amazon Lookout for Vision experienced a service issue. Try your call again.
     * @throws ValidationException
     *         An input validation error occured. For example, invalid characters in a project name, or if a pagination
     *         token is invalid.
     * @throws ConflictException
     *         The update or deletion of a resource caused an inconsistent state.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ThrottlingException
     *         Amazon Lookout for Vision is temporarily unable to process the request. Try your call again.
     * @throws ServiceQuotaExceededException
     *         A service quota was exceeded the allowed limit. For more information, see Limits in Amazon Lookout for
     *         Vision in the Amazon Lookout for Vision Developer Guide.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutVisionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutVisionClient.StartModelPackagingJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/StartModelPackagingJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartModelPackagingJobResponse startModelPackagingJob(StartModelPackagingJobRequest startModelPackagingJobRequest)
            throws AccessDeniedException, InternalServerException, ValidationException, ConflictException,
            ResourceNotFoundException, ThrottlingException, ServiceQuotaExceededException, AwsServiceException,
            SdkClientException, LookoutVisionException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartModelPackagingJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartModelPackagingJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startModelPackagingJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startModelPackagingJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartModelPackagingJob");

            return clientHandler
                    .execute(new ClientExecutionParams<StartModelPackagingJobRequest, StartModelPackagingJobResponse>()
                            .withOperationName("StartModelPackagingJob").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(startModelPackagingJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StartModelPackagingJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stops the hosting of a running model. The operation might take a while to complete. To check the current status,
     * call <a>DescribeModel</a>.
     * </p>
     * <p>
     * After the model hosting stops, the <code>Status</code> of the model is <code>TRAINED</code>.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:StopModel</code> operation.
     * </p>
     *
     * @param stopModelRequest
     * @return Result of the StopModel operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerException
     *         Amazon Lookout for Vision experienced a service issue. Try your call again.
     * @throws ValidationException
     *         An input validation error occured. For example, invalid characters in a project name, or if a pagination
     *         token is invalid.
     * @throws ConflictException
     *         The update or deletion of a resource caused an inconsistent state.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ThrottlingException
     *         Amazon Lookout for Vision is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutVisionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutVisionClient.StopModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/StopModel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StopModelResponse stopModel(StopModelRequest stopModelRequest) throws AccessDeniedException, InternalServerException,
            ValidationException, ConflictException, ResourceNotFoundException, ThrottlingException, AwsServiceException,
            SdkClientException, LookoutVisionException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopModelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StopModelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopModelRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopModel");

            return clientHandler.execute(new ClientExecutionParams<StopModelRequest, StopModelResponse>()
                    .withOperationName("StopModel").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(stopModelRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StopModelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds one or more key-value tags to an Amazon Lookout for Vision model. For more information, see <i>Tagging a
     * model</i> in the <i>Amazon Lookout for Vision Developer Guide</i>.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:TagResource</code> operation.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerException
     *         Amazon Lookout for Vision experienced a service issue. Try your call again.
     * @throws ValidationException
     *         An input validation error occured. For example, invalid characters in a project name, or if a pagination
     *         token is invalid.
     * @throws ConflictException
     *         The update or deletion of a resource caused an inconsistent state.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ThrottlingException
     *         Amazon Lookout for Vision is temporarily unable to process the request. Try your call again.
     * @throws ServiceQuotaExceededException
     *         A service quota was exceeded the allowed limit. For more information, see Limits in Amazon Lookout for
     *         Vision in the Amazon Lookout for Vision Developer Guide.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutVisionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutVisionClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws AccessDeniedException,
            InternalServerException, ValidationException, ConflictException, ResourceNotFoundException, ThrottlingException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, LookoutVisionException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes one or more tags from an Amazon Lookout for Vision model. For more information, see <i>Tagging a
     * model</i> in the <i>Amazon Lookout for Vision Developer Guide</i>.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:UntagResource</code> operation.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerException
     *         Amazon Lookout for Vision experienced a service issue. Try your call again.
     * @throws ValidationException
     *         An input validation error occured. For example, invalid characters in a project name, or if a pagination
     *         token is invalid.
     * @throws ConflictException
     *         The update or deletion of a resource caused an inconsistent state.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ThrottlingException
     *         Amazon Lookout for Vision is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutVisionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutVisionClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws AccessDeniedException,
            InternalServerException, ValidationException, ConflictException, ResourceNotFoundException, ThrottlingException,
            AwsServiceException, SdkClientException, LookoutVisionException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds or updates one or more JSON Line entries in a dataset. A JSON Line includes information about an image used
     * for training or testing an Amazon Lookout for Vision model.
     * </p>
     * <p>
     * To update an existing JSON Line, use the <code>source-ref</code> field to identify the JSON Line. The JSON line
     * that you supply replaces the existing JSON line. Any existing annotations that are not in the new JSON line are
     * removed from the dataset.
     * </p>
     * <p>
     * For more information, see <i>Defining JSON lines for anomaly classification</i> in the Amazon Lookout for Vision
     * Developer Guide.
     * </p>
     * <note>
     * <p>
     * The images you reference in the <code>source-ref</code> field of a JSON line, must be in the same S3 bucket as
     * the existing images in the dataset.
     * </p>
     * </note>
     * <p>
     * Updating a dataset might take a while to complete. To check the current status, call <a>DescribeDataset</a> and
     * check the <code>Status</code> field in the response.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>lookoutvision:UpdateDatasetEntries</code> operation.
     * </p>
     *
     * @param updateDatasetEntriesRequest
     * @return Result of the UpdateDatasetEntries operation returned by the service.
     * @throws AccessDeniedException
     *         You are not authorized to perform the action.
     * @throws InternalServerException
     *         Amazon Lookout for Vision experienced a service issue. Try your call again.
     * @throws ValidationException
     *         An input validation error occured. For example, invalid characters in a project name, or if a pagination
     *         token is invalid.
     * @throws ConflictException
     *         The update or deletion of a resource caused an inconsistent state.
     * @throws ResourceNotFoundException
     *         The resource could not be found.
     * @throws ThrottlingException
     *         Amazon Lookout for Vision is temporarily unable to process the request. Try your call again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws LookoutVisionException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample LookoutVisionClient.UpdateDatasetEntries
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/lookoutvision-2020-11-20/UpdateDatasetEntries"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateDatasetEntriesResponse updateDatasetEntries(UpdateDatasetEntriesRequest updateDatasetEntriesRequest)
            throws AccessDeniedException, InternalServerException, ValidationException, ConflictException,
            ResourceNotFoundException, ThrottlingException, AwsServiceException, SdkClientException, LookoutVisionException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateDatasetEntriesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateDatasetEntriesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateDatasetEntriesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDatasetEntriesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "LookoutVision");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDatasetEntries");

            return clientHandler.execute(new ClientExecutionParams<UpdateDatasetEntriesRequest, UpdateDatasetEntriesResponse>()
                    .withOperationName("UpdateDatasetEntries").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateDatasetEntriesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateDatasetEntriesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        LookoutVisionServiceClientConfigurationBuilder serviceConfigBuilder = new LookoutVisionServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(LookoutVisionException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    @Override
    public final LookoutVisionServiceClientConfiguration serviceClientConfiguration() {
        return new LookoutVisionServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
