/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lookoutvision.model;

import java.io.Serializable;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The prediction results from a call to <a>DetectAnomalies</a>. <code>DetectAnomalyResult</code> includes
 * classification information for the prediction (<code>IsAnomalous</code> and <code>Confidence</code>). If the model
 * you use is an image segementation model, <code>DetectAnomalyResult</code> also includes segmentation information (
 * <code>Anomalies</code> and <code>AnomalyMask</code>). Classification information is calculated separately from
 * segmentation information and you shouldn't assume a relationship between them.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DetectAnomalyResult implements SdkPojo, Serializable,
        ToCopyableBuilder<DetectAnomalyResult.Builder, DetectAnomalyResult> {
    private static final SdkField<ImageSource> SOURCE_FIELD = SdkField.<ImageSource> builder(MarshallingType.SDK_POJO)
            .memberName("Source").getter(getter(DetectAnomalyResult::source)).setter(setter(Builder::source))
            .constructor(ImageSource::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Source").build()).build();

    private static final SdkField<Boolean> IS_ANOMALOUS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("IsAnomalous").getter(getter(DetectAnomalyResult::isAnomalous)).setter(setter(Builder::isAnomalous))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IsAnomalous").build()).build();

    private static final SdkField<Float> CONFIDENCE_FIELD = SdkField.<Float> builder(MarshallingType.FLOAT)
            .memberName("Confidence").getter(getter(DetectAnomalyResult::confidence)).setter(setter(Builder::confidence))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Confidence").build()).build();

    private static final SdkField<List<Anomaly>> ANOMALIES_FIELD = SdkField
            .<List<Anomaly>> builder(MarshallingType.LIST)
            .memberName("Anomalies")
            .getter(getter(DetectAnomalyResult::anomalies))
            .setter(setter(Builder::anomalies))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Anomalies").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Anomaly> builder(MarshallingType.SDK_POJO)
                                            .constructor(Anomaly::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<SdkBytes> ANOMALY_MASK_FIELD = SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .memberName("AnomalyMask").getter(getter(DetectAnomalyResult::anomalyMask)).setter(setter(Builder::anomalyMask))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AnomalyMask").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SOURCE_FIELD,
            IS_ANOMALOUS_FIELD, CONFIDENCE_FIELD, ANOMALIES_FIELD, ANOMALY_MASK_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("Source", SOURCE_FIELD);
                    put("IsAnomalous", IS_ANOMALOUS_FIELD);
                    put("Confidence", CONFIDENCE_FIELD);
                    put("Anomalies", ANOMALIES_FIELD);
                    put("AnomalyMask", ANOMALY_MASK_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final ImageSource source;

    private final Boolean isAnomalous;

    private final Float confidence;

    private final List<Anomaly> anomalies;

    private final SdkBytes anomalyMask;

    private DetectAnomalyResult(BuilderImpl builder) {
        this.source = builder.source;
        this.isAnomalous = builder.isAnomalous;
        this.confidence = builder.confidence;
        this.anomalies = builder.anomalies;
        this.anomalyMask = builder.anomalyMask;
    }

    /**
     * <p>
     * The source of the image that was analyzed. <code>direct</code> means that the images was supplied from the local
     * computer. No other values are supported.
     * </p>
     * 
     * @return The source of the image that was analyzed. <code>direct</code> means that the images was supplied from
     *         the local computer. No other values are supported.
     */
    public final ImageSource source() {
        return source;
    }

    /**
     * <p>
     * True if Amazon Lookout for Vision classifies the image as containing an anomaly, otherwise false.
     * </p>
     * 
     * @return True if Amazon Lookout for Vision classifies the image as containing an anomaly, otherwise false.
     */
    public final Boolean isAnomalous() {
        return isAnomalous;
    }

    /**
     * <p>
     * The confidence that Lookout for Vision has in the accuracy of the classification in <code>IsAnomalous</code>.
     * </p>
     * 
     * @return The confidence that Lookout for Vision has in the accuracy of the classification in
     *         <code>IsAnomalous</code>.
     */
    public final Float confidence() {
        return confidence;
    }

    /**
     * For responses, this returns true if the service returned a value for the Anomalies property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAnomalies() {
        return anomalies != null && !(anomalies instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * If the model is an image segmentation model, <code>Anomalies</code> contains a list of anomaly types found in the
     * image. There is one entry for each type of anomaly found (even if multiple instances of an anomaly type exist on
     * the image). The first element in the list is always an anomaly type representing the image background
     * ('background') and shouldn't be considered an anomaly. Amazon Lookout for Vision automatically add the background
     * anomaly type to the response, and you don't need to declare a background anomaly type in your dataset.
     * </p>
     * <p>
     * If the list has one entry ('background'), no anomalies were found on the image.
     * </p>
     * <p/>
     * <p>
     * An image classification model doesn't return an <code>Anomalies</code> list.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAnomalies} method.
     * </p>
     * 
     * @return If the model is an image segmentation model, <code>Anomalies</code> contains a list of anomaly types
     *         found in the image. There is one entry for each type of anomaly found (even if multiple instances of an
     *         anomaly type exist on the image). The first element in the list is always an anomaly type representing
     *         the image background ('background') and shouldn't be considered an anomaly. Amazon Lookout for Vision
     *         automatically add the background anomaly type to the response, and you don't need to declare a background
     *         anomaly type in your dataset.</p>
     *         <p>
     *         If the list has one entry ('background'), no anomalies were found on the image.
     *         </p>
     *         <p/>
     *         <p>
     *         An image classification model doesn't return an <code>Anomalies</code> list.
     */
    public final List<Anomaly> anomalies() {
        return anomalies;
    }

    /**
     * <p>
     * If the model is an image segmentation model, <code>AnomalyMask</code> contains pixel masks that covers all
     * anomaly types found on the image. Each anomaly type has a different mask color. To map a color to an anomaly
     * type, see the <code>color</code> field of the <a>PixelAnomaly</a> object.
     * </p>
     * <p>
     * An image classification model doesn't return an <code>Anomalies</code> list.
     * </p>
     * 
     * @return If the model is an image segmentation model, <code>AnomalyMask</code> contains pixel masks that covers
     *         all anomaly types found on the image. Each anomaly type has a different mask color. To map a color to an
     *         anomaly type, see the <code>color</code> field of the <a>PixelAnomaly</a> object.</p>
     *         <p>
     *         An image classification model doesn't return an <code>Anomalies</code> list.
     */
    public final SdkBytes anomalyMask() {
        return anomalyMask;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(source());
        hashCode = 31 * hashCode + Objects.hashCode(isAnomalous());
        hashCode = 31 * hashCode + Objects.hashCode(confidence());
        hashCode = 31 * hashCode + Objects.hashCode(hasAnomalies() ? anomalies() : null);
        hashCode = 31 * hashCode + Objects.hashCode(anomalyMask());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DetectAnomalyResult)) {
            return false;
        }
        DetectAnomalyResult other = (DetectAnomalyResult) obj;
        return Objects.equals(source(), other.source()) && Objects.equals(isAnomalous(), other.isAnomalous())
                && Objects.equals(confidence(), other.confidence()) && hasAnomalies() == other.hasAnomalies()
                && Objects.equals(anomalies(), other.anomalies()) && Objects.equals(anomalyMask(), other.anomalyMask());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DetectAnomalyResult").add("Source", source()).add("IsAnomalous", isAnomalous())
                .add("Confidence", confidence()).add("Anomalies", hasAnomalies() ? anomalies() : null)
                .add("AnomalyMask", anomalyMask()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Source":
            return Optional.ofNullable(clazz.cast(source()));
        case "IsAnomalous":
            return Optional.ofNullable(clazz.cast(isAnomalous()));
        case "Confidence":
            return Optional.ofNullable(clazz.cast(confidence()));
        case "Anomalies":
            return Optional.ofNullable(clazz.cast(anomalies()));
        case "AnomalyMask":
            return Optional.ofNullable(clazz.cast(anomalyMask()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<DetectAnomalyResult, T> g) {
        return obj -> g.apply((DetectAnomalyResult) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DetectAnomalyResult> {
        /**
         * <p>
         * The source of the image that was analyzed. <code>direct</code> means that the images was supplied from the
         * local computer. No other values are supported.
         * </p>
         * 
         * @param source
         *        The source of the image that was analyzed. <code>direct</code> means that the images was supplied from
         *        the local computer. No other values are supported.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder source(ImageSource source);

        /**
         * <p>
         * The source of the image that was analyzed. <code>direct</code> means that the images was supplied from the
         * local computer. No other values are supported.
         * </p>
         * This is a convenience method that creates an instance of the {@link ImageSource.Builder} avoiding the need to
         * create one manually via {@link ImageSource#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ImageSource.Builder#build()} is called immediately and its result
         * is passed to {@link #source(ImageSource)}.
         * 
         * @param source
         *        a consumer that will call methods on {@link ImageSource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #source(ImageSource)
         */
        default Builder source(Consumer<ImageSource.Builder> source) {
            return source(ImageSource.builder().applyMutation(source).build());
        }

        /**
         * <p>
         * True if Amazon Lookout for Vision classifies the image as containing an anomaly, otherwise false.
         * </p>
         * 
         * @param isAnomalous
         *        True if Amazon Lookout for Vision classifies the image as containing an anomaly, otherwise false.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isAnomalous(Boolean isAnomalous);

        /**
         * <p>
         * The confidence that Lookout for Vision has in the accuracy of the classification in <code>IsAnomalous</code>.
         * </p>
         * 
         * @param confidence
         *        The confidence that Lookout for Vision has in the accuracy of the classification in
         *        <code>IsAnomalous</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder confidence(Float confidence);

        /**
         * <p>
         * If the model is an image segmentation model, <code>Anomalies</code> contains a list of anomaly types found in
         * the image. There is one entry for each type of anomaly found (even if multiple instances of an anomaly type
         * exist on the image). The first element in the list is always an anomaly type representing the image
         * background ('background') and shouldn't be considered an anomaly. Amazon Lookout for Vision automatically add
         * the background anomaly type to the response, and you don't need to declare a background anomaly type in your
         * dataset.
         * </p>
         * <p>
         * If the list has one entry ('background'), no anomalies were found on the image.
         * </p>
         * <p/>
         * <p>
         * An image classification model doesn't return an <code>Anomalies</code> list.
         * </p>
         * 
         * @param anomalies
         *        If the model is an image segmentation model, <code>Anomalies</code> contains a list of anomaly types
         *        found in the image. There is one entry for each type of anomaly found (even if multiple instances of
         *        an anomaly type exist on the image). The first element in the list is always an anomaly type
         *        representing the image background ('background') and shouldn't be considered an anomaly. Amazon
         *        Lookout for Vision automatically add the background anomaly type to the response, and you don't need
         *        to declare a background anomaly type in your dataset.</p>
         *        <p>
         *        If the list has one entry ('background'), no anomalies were found on the image.
         *        </p>
         *        <p/>
         *        <p>
         *        An image classification model doesn't return an <code>Anomalies</code> list.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder anomalies(Collection<Anomaly> anomalies);

        /**
         * <p>
         * If the model is an image segmentation model, <code>Anomalies</code> contains a list of anomaly types found in
         * the image. There is one entry for each type of anomaly found (even if multiple instances of an anomaly type
         * exist on the image). The first element in the list is always an anomaly type representing the image
         * background ('background') and shouldn't be considered an anomaly. Amazon Lookout for Vision automatically add
         * the background anomaly type to the response, and you don't need to declare a background anomaly type in your
         * dataset.
         * </p>
         * <p>
         * If the list has one entry ('background'), no anomalies were found on the image.
         * </p>
         * <p/>
         * <p>
         * An image classification model doesn't return an <code>Anomalies</code> list.
         * </p>
         * 
         * @param anomalies
         *        If the model is an image segmentation model, <code>Anomalies</code> contains a list of anomaly types
         *        found in the image. There is one entry for each type of anomaly found (even if multiple instances of
         *        an anomaly type exist on the image). The first element in the list is always an anomaly type
         *        representing the image background ('background') and shouldn't be considered an anomaly. Amazon
         *        Lookout for Vision automatically add the background anomaly type to the response, and you don't need
         *        to declare a background anomaly type in your dataset.</p>
         *        <p>
         *        If the list has one entry ('background'), no anomalies were found on the image.
         *        </p>
         *        <p/>
         *        <p>
         *        An image classification model doesn't return an <code>Anomalies</code> list.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder anomalies(Anomaly... anomalies);

        /**
         * <p>
         * If the model is an image segmentation model, <code>Anomalies</code> contains a list of anomaly types found in
         * the image. There is one entry for each type of anomaly found (even if multiple instances of an anomaly type
         * exist on the image). The first element in the list is always an anomaly type representing the image
         * background ('background') and shouldn't be considered an anomaly. Amazon Lookout for Vision automatically add
         * the background anomaly type to the response, and you don't need to declare a background anomaly type in your
         * dataset.
         * </p>
         * <p>
         * If the list has one entry ('background'), no anomalies were found on the image.
         * </p>
         * <p/>
         * <p>
         * An image classification model doesn't return an <code>Anomalies</code> list.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.lookoutvision.model.Anomaly.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.lookoutvision.model.Anomaly#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.lookoutvision.model.Anomaly.Builder#build()} is called immediately and
         * its result is passed to {@link #anomalies(List<Anomaly>)}.
         * 
         * @param anomalies
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.lookoutvision.model.Anomaly.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #anomalies(java.util.Collection<Anomaly>)
         */
        Builder anomalies(Consumer<Anomaly.Builder>... anomalies);

        /**
         * <p>
         * If the model is an image segmentation model, <code>AnomalyMask</code> contains pixel masks that covers all
         * anomaly types found on the image. Each anomaly type has a different mask color. To map a color to an anomaly
         * type, see the <code>color</code> field of the <a>PixelAnomaly</a> object.
         * </p>
         * <p>
         * An image classification model doesn't return an <code>Anomalies</code> list.
         * </p>
         * 
         * @param anomalyMask
         *        If the model is an image segmentation model, <code>AnomalyMask</code> contains pixel masks that covers
         *        all anomaly types found on the image. Each anomaly type has a different mask color. To map a color to
         *        an anomaly type, see the <code>color</code> field of the <a>PixelAnomaly</a> object.</p>
         *        <p>
         *        An image classification model doesn't return an <code>Anomalies</code> list.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder anomalyMask(SdkBytes anomalyMask);
    }

    static final class BuilderImpl implements Builder {
        private ImageSource source;

        private Boolean isAnomalous;

        private Float confidence;

        private List<Anomaly> anomalies = DefaultSdkAutoConstructList.getInstance();

        private SdkBytes anomalyMask;

        private BuilderImpl() {
        }

        private BuilderImpl(DetectAnomalyResult model) {
            source(model.source);
            isAnomalous(model.isAnomalous);
            confidence(model.confidence);
            anomalies(model.anomalies);
            anomalyMask(model.anomalyMask);
        }

        public final ImageSource.Builder getSource() {
            return source != null ? source.toBuilder() : null;
        }

        public final void setSource(ImageSource.BuilderImpl source) {
            this.source = source != null ? source.build() : null;
        }

        @Override
        public final Builder source(ImageSource source) {
            this.source = source;
            return this;
        }

        public final Boolean getIsAnomalous() {
            return isAnomalous;
        }

        public final void setIsAnomalous(Boolean isAnomalous) {
            this.isAnomalous = isAnomalous;
        }

        @Override
        public final Builder isAnomalous(Boolean isAnomalous) {
            this.isAnomalous = isAnomalous;
            return this;
        }

        public final Float getConfidence() {
            return confidence;
        }

        public final void setConfidence(Float confidence) {
            this.confidence = confidence;
        }

        @Override
        public final Builder confidence(Float confidence) {
            this.confidence = confidence;
            return this;
        }

        public final List<Anomaly.Builder> getAnomalies() {
            List<Anomaly.Builder> result = AnomalyListCopier.copyToBuilder(this.anomalies);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAnomalies(Collection<Anomaly.BuilderImpl> anomalies) {
            this.anomalies = AnomalyListCopier.copyFromBuilder(anomalies);
        }

        @Override
        public final Builder anomalies(Collection<Anomaly> anomalies) {
            this.anomalies = AnomalyListCopier.copy(anomalies);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder anomalies(Anomaly... anomalies) {
            anomalies(Arrays.asList(anomalies));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder anomalies(Consumer<Anomaly.Builder>... anomalies) {
            anomalies(Stream.of(anomalies).map(c -> Anomaly.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final ByteBuffer getAnomalyMask() {
            return anomalyMask == null ? null : anomalyMask.asByteBuffer();
        }

        public final void setAnomalyMask(ByteBuffer anomalyMask) {
            anomalyMask(anomalyMask == null ? null : SdkBytes.fromByteBuffer(anomalyMask));
        }

        @Override
        public final Builder anomalyMask(SdkBytes anomalyMask) {
            this.anomalyMask = anomalyMask;
            return this;
        }

        @Override
        public DetectAnomalyResult build() {
            return new DetectAnomalyResult(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
