/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.m2.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class StartBatchJobRequest extends M2Request implements
        ToCopyableBuilder<StartBatchJobRequest.Builder, StartBatchJobRequest> {
    private static final SdkField<String> APPLICATION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("applicationId").getter(getter(StartBatchJobRequest::applicationId))
            .setter(setter(Builder::applicationId))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("applicationId").build()).build();

    private static final SdkField<String> AUTH_SECRETS_MANAGER_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("authSecretsManagerArn").getter(getter(StartBatchJobRequest::authSecretsManagerArn))
            .setter(setter(Builder::authSecretsManagerArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("authSecretsManagerArn").build())
            .build();

    private static final SdkField<BatchJobIdentifier> BATCH_JOB_IDENTIFIER_FIELD = SdkField
            .<BatchJobIdentifier> builder(MarshallingType.SDK_POJO).memberName("batchJobIdentifier")
            .getter(getter(StartBatchJobRequest::batchJobIdentifier)).setter(setter(Builder::batchJobIdentifier))
            .constructor(BatchJobIdentifier::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("batchJobIdentifier").build())
            .build();

    private static final SdkField<Map<String, String>> JOB_PARAMS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("jobParams")
            .getter(getter(StartBatchJobRequest::jobParams))
            .setter(setter(Builder::jobParams))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("jobParams").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(APPLICATION_ID_FIELD,
            AUTH_SECRETS_MANAGER_ARN_FIELD, BATCH_JOB_IDENTIFIER_FIELD, JOB_PARAMS_FIELD));

    private final String applicationId;

    private final String authSecretsManagerArn;

    private final BatchJobIdentifier batchJobIdentifier;

    private final Map<String, String> jobParams;

    private StartBatchJobRequest(BuilderImpl builder) {
        super(builder);
        this.applicationId = builder.applicationId;
        this.authSecretsManagerArn = builder.authSecretsManagerArn;
        this.batchJobIdentifier = builder.batchJobIdentifier;
        this.jobParams = builder.jobParams;
    }

    /**
     * <p>
     * The unique identifier of the application associated with this batch job.
     * </p>
     * 
     * @return The unique identifier of the application associated with this batch job.
     */
    public final String applicationId() {
        return applicationId;
    }

    /**
     * <p>
     * The Amazon Web Services Secrets Manager containing user's credentials for authentication and authorization for
     * Start Batch Job execution operation.
     * </p>
     * 
     * @return The Amazon Web Services Secrets Manager containing user's credentials for authentication and
     *         authorization for Start Batch Job execution operation.
     */
    public final String authSecretsManagerArn() {
        return authSecretsManagerArn;
    }

    /**
     * <p>
     * The unique identifier of the batch job.
     * </p>
     * 
     * @return The unique identifier of the batch job.
     */
    public final BatchJobIdentifier batchJobIdentifier() {
        return batchJobIdentifier;
    }

    /**
     * For responses, this returns true if the service returned a value for the JobParams property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasJobParams() {
        return jobParams != null && !(jobParams instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The collection of batch job parameters. For details about limits for keys and values, see <a
     * href="https://www.ibm.com/docs/en/workload-automation/9.3.0?topic=zos-coding-variables-in-jcl">Coding variables
     * in JCL</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasJobParams} method.
     * </p>
     * 
     * @return The collection of batch job parameters. For details about limits for keys and values, see <a
     *         href="https://www.ibm.com/docs/en/workload-automation/9.3.0?topic=zos-coding-variables-in-jcl">Coding
     *         variables in JCL</a>.
     */
    public final Map<String, String> jobParams() {
        return jobParams;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(applicationId());
        hashCode = 31 * hashCode + Objects.hashCode(authSecretsManagerArn());
        hashCode = 31 * hashCode + Objects.hashCode(batchJobIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(hasJobParams() ? jobParams() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StartBatchJobRequest)) {
            return false;
        }
        StartBatchJobRequest other = (StartBatchJobRequest) obj;
        return Objects.equals(applicationId(), other.applicationId())
                && Objects.equals(authSecretsManagerArn(), other.authSecretsManagerArn())
                && Objects.equals(batchJobIdentifier(), other.batchJobIdentifier()) && hasJobParams() == other.hasJobParams()
                && Objects.equals(jobParams(), other.jobParams());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("StartBatchJobRequest").add("ApplicationId", applicationId())
                .add("AuthSecretsManagerArn", authSecretsManagerArn()).add("BatchJobIdentifier", batchJobIdentifier())
                .add("JobParams", hasJobParams() ? jobParams() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "applicationId":
            return Optional.ofNullable(clazz.cast(applicationId()));
        case "authSecretsManagerArn":
            return Optional.ofNullable(clazz.cast(authSecretsManagerArn()));
        case "batchJobIdentifier":
            return Optional.ofNullable(clazz.cast(batchJobIdentifier()));
        case "jobParams":
            return Optional.ofNullable(clazz.cast(jobParams()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<StartBatchJobRequest, T> g) {
        return obj -> g.apply((StartBatchJobRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends M2Request.Builder, SdkPojo, CopyableBuilder<Builder, StartBatchJobRequest> {
        /**
         * <p>
         * The unique identifier of the application associated with this batch job.
         * </p>
         * 
         * @param applicationId
         *        The unique identifier of the application associated with this batch job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applicationId(String applicationId);

        /**
         * <p>
         * The Amazon Web Services Secrets Manager containing user's credentials for authentication and authorization
         * for Start Batch Job execution operation.
         * </p>
         * 
         * @param authSecretsManagerArn
         *        The Amazon Web Services Secrets Manager containing user's credentials for authentication and
         *        authorization for Start Batch Job execution operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authSecretsManagerArn(String authSecretsManagerArn);

        /**
         * <p>
         * The unique identifier of the batch job.
         * </p>
         * 
         * @param batchJobIdentifier
         *        The unique identifier of the batch job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder batchJobIdentifier(BatchJobIdentifier batchJobIdentifier);

        /**
         * <p>
         * The unique identifier of the batch job.
         * </p>
         * This is a convenience method that creates an instance of the {@link BatchJobIdentifier.Builder} avoiding the
         * need to create one manually via {@link BatchJobIdentifier#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link BatchJobIdentifier.Builder#build()} is called immediately and its
         * result is passed to {@link #batchJobIdentifier(BatchJobIdentifier)}.
         * 
         * @param batchJobIdentifier
         *        a consumer that will call methods on {@link BatchJobIdentifier.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #batchJobIdentifier(BatchJobIdentifier)
         */
        default Builder batchJobIdentifier(Consumer<BatchJobIdentifier.Builder> batchJobIdentifier) {
            return batchJobIdentifier(BatchJobIdentifier.builder().applyMutation(batchJobIdentifier).build());
        }

        /**
         * <p>
         * The collection of batch job parameters. For details about limits for keys and values, see <a
         * href="https://www.ibm.com/docs/en/workload-automation/9.3.0?topic=zos-coding-variables-in-jcl">Coding
         * variables in JCL</a>.
         * </p>
         * 
         * @param jobParams
         *        The collection of batch job parameters. For details about limits for keys and values, see <a
         *        href="https://www.ibm.com/docs/en/workload-automation/9.3.0?topic=zos-coding-variables-in-jcl">Coding
         *        variables in JCL</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobParams(Map<String, String> jobParams);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends M2Request.BuilderImpl implements Builder {
        private String applicationId;

        private String authSecretsManagerArn;

        private BatchJobIdentifier batchJobIdentifier;

        private Map<String, String> jobParams = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(StartBatchJobRequest model) {
            super(model);
            applicationId(model.applicationId);
            authSecretsManagerArn(model.authSecretsManagerArn);
            batchJobIdentifier(model.batchJobIdentifier);
            jobParams(model.jobParams);
        }

        public final String getApplicationId() {
            return applicationId;
        }

        public final void setApplicationId(String applicationId) {
            this.applicationId = applicationId;
        }

        @Override
        public final Builder applicationId(String applicationId) {
            this.applicationId = applicationId;
            return this;
        }

        public final String getAuthSecretsManagerArn() {
            return authSecretsManagerArn;
        }

        public final void setAuthSecretsManagerArn(String authSecretsManagerArn) {
            this.authSecretsManagerArn = authSecretsManagerArn;
        }

        @Override
        public final Builder authSecretsManagerArn(String authSecretsManagerArn) {
            this.authSecretsManagerArn = authSecretsManagerArn;
            return this;
        }

        public final BatchJobIdentifier.Builder getBatchJobIdentifier() {
            return batchJobIdentifier != null ? batchJobIdentifier.toBuilder() : null;
        }

        public final void setBatchJobIdentifier(BatchJobIdentifier.BuilderImpl batchJobIdentifier) {
            this.batchJobIdentifier = batchJobIdentifier != null ? batchJobIdentifier.build() : null;
        }

        @Override
        public final Builder batchJobIdentifier(BatchJobIdentifier batchJobIdentifier) {
            this.batchJobIdentifier = batchJobIdentifier;
            return this;
        }

        public final Map<String, String> getJobParams() {
            if (jobParams instanceof SdkAutoConstructMap) {
                return null;
            }
            return jobParams;
        }

        public final void setJobParams(Map<String, String> jobParams) {
            this.jobParams = BatchJobParametersMapCopier.copy(jobParams);
        }

        @Override
        public final Builder jobParams(Map<String, String> jobParams) {
            this.jobParams = BatchJobParametersMapCopier.copy(jobParams);
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public StartBatchJobRequest build() {
            return new StartBatchJobRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
