/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.machinelearning.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the output of <code>GetEvaluation</code> operation.
 * </p>
 * <p>
 * The content consists of the detailed metadata and data file information and the current status of the
 * <code>Evaluation</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Evaluation implements SdkPojo, Serializable, ToCopyableBuilder<Evaluation.Builder, Evaluation> {
    private static final SdkField<String> EVALUATION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EvaluationId").getter(getter(Evaluation::evaluationId)).setter(setter(Builder::evaluationId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EvaluationId").build()).build();

    private static final SdkField<String> ML_MODEL_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MLModelId").getter(getter(Evaluation::mlModelId)).setter(setter(Builder::mlModelId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MLModelId").build()).build();

    private static final SdkField<String> EVALUATION_DATA_SOURCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EvaluationDataSourceId").getter(getter(Evaluation::evaluationDataSourceId))
            .setter(setter(Builder::evaluationDataSourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EvaluationDataSourceId").build())
            .build();

    private static final SdkField<String> INPUT_DATA_LOCATION_S3_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InputDataLocationS3").getter(getter(Evaluation::inputDataLocationS3))
            .setter(setter(Builder::inputDataLocationS3))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InputDataLocationS3").build())
            .build();

    private static final SdkField<String> CREATED_BY_IAM_USER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CreatedByIamUser").getter(getter(Evaluation::createdByIamUser))
            .setter(setter(Builder::createdByIamUser))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedByIamUser").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedAt").getter(getter(Evaluation::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedAt").build()).build();

    private static final SdkField<Instant> LAST_UPDATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastUpdatedAt").getter(getter(Evaluation::lastUpdatedAt)).setter(setter(Builder::lastUpdatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUpdatedAt").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(Evaluation::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(Evaluation::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<PerformanceMetrics> PERFORMANCE_METRICS_FIELD = SdkField
            .<PerformanceMetrics> builder(MarshallingType.SDK_POJO).memberName("PerformanceMetrics")
            .getter(getter(Evaluation::performanceMetrics)).setter(setter(Builder::performanceMetrics))
            .constructor(PerformanceMetrics::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PerformanceMetrics").build())
            .build();

    private static final SdkField<String> MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Message")
            .getter(getter(Evaluation::message)).setter(setter(Builder::message))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Message").build()).build();

    private static final SdkField<Long> COMPUTE_TIME_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("ComputeTime").getter(getter(Evaluation::computeTime)).setter(setter(Builder::computeTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ComputeTime").build()).build();

    private static final SdkField<Instant> FINISHED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("FinishedAt").getter(getter(Evaluation::finishedAt)).setter(setter(Builder::finishedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FinishedAt").build()).build();

    private static final SdkField<Instant> STARTED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartedAt").getter(getter(Evaluation::startedAt)).setter(setter(Builder::startedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartedAt").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EVALUATION_ID_FIELD,
            ML_MODEL_ID_FIELD, EVALUATION_DATA_SOURCE_ID_FIELD, INPUT_DATA_LOCATION_S3_FIELD, CREATED_BY_IAM_USER_FIELD,
            CREATED_AT_FIELD, LAST_UPDATED_AT_FIELD, NAME_FIELD, STATUS_FIELD, PERFORMANCE_METRICS_FIELD, MESSAGE_FIELD,
            COMPUTE_TIME_FIELD, FINISHED_AT_FIELD, STARTED_AT_FIELD));

    private static final long serialVersionUID = 1L;

    private final String evaluationId;

    private final String mlModelId;

    private final String evaluationDataSourceId;

    private final String inputDataLocationS3;

    private final String createdByIamUser;

    private final Instant createdAt;

    private final Instant lastUpdatedAt;

    private final String name;

    private final String status;

    private final PerformanceMetrics performanceMetrics;

    private final String message;

    private final Long computeTime;

    private final Instant finishedAt;

    private final Instant startedAt;

    private Evaluation(BuilderImpl builder) {
        this.evaluationId = builder.evaluationId;
        this.mlModelId = builder.mlModelId;
        this.evaluationDataSourceId = builder.evaluationDataSourceId;
        this.inputDataLocationS3 = builder.inputDataLocationS3;
        this.createdByIamUser = builder.createdByIamUser;
        this.createdAt = builder.createdAt;
        this.lastUpdatedAt = builder.lastUpdatedAt;
        this.name = builder.name;
        this.status = builder.status;
        this.performanceMetrics = builder.performanceMetrics;
        this.message = builder.message;
        this.computeTime = builder.computeTime;
        this.finishedAt = builder.finishedAt;
        this.startedAt = builder.startedAt;
    }

    /**
     * <p>
     * The ID that is assigned to the <code>Evaluation</code> at creation.
     * </p>
     * 
     * @return The ID that is assigned to the <code>Evaluation</code> at creation.
     */
    public final String evaluationId() {
        return evaluationId;
    }

    /**
     * <p>
     * The ID of the <code>MLModel</code> that is the focus of the evaluation.
     * </p>
     * 
     * @return The ID of the <code>MLModel</code> that is the focus of the evaluation.
     */
    public final String mlModelId() {
        return mlModelId;
    }

    /**
     * <p>
     * The ID of the <code>DataSource</code> that is used to evaluate the <code>MLModel</code>.
     * </p>
     * 
     * @return The ID of the <code>DataSource</code> that is used to evaluate the <code>MLModel</code>.
     */
    public final String evaluationDataSourceId() {
        return evaluationDataSourceId;
    }

    /**
     * <p>
     * The location and name of the data in Amazon Simple Storage Server (Amazon S3) that is used in the evaluation.
     * </p>
     * 
     * @return The location and name of the data in Amazon Simple Storage Server (Amazon S3) that is used in the
     *         evaluation.
     */
    public final String inputDataLocationS3() {
        return inputDataLocationS3;
    }

    /**
     * <p>
     * The AWS user account that invoked the evaluation. The account type can be either an AWS root account or an AWS
     * Identity and Access Management (IAM) user account.
     * </p>
     * 
     * @return The AWS user account that invoked the evaluation. The account type can be either an AWS root account or
     *         an AWS Identity and Access Management (IAM) user account.
     */
    public final String createdByIamUser() {
        return createdByIamUser;
    }

    /**
     * <p>
     * The time that the <code>Evaluation</code> was created. The time is expressed in epoch time.
     * </p>
     * 
     * @return The time that the <code>Evaluation</code> was created. The time is expressed in epoch time.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The time of the most recent edit to the <code>Evaluation</code>. The time is expressed in epoch time.
     * </p>
     * 
     * @return The time of the most recent edit to the <code>Evaluation</code>. The time is expressed in epoch time.
     */
    public final Instant lastUpdatedAt() {
        return lastUpdatedAt;
    }

    /**
     * <p>
     * A user-supplied name or description of the <code>Evaluation</code>.
     * </p>
     * 
     * @return A user-supplied name or description of the <code>Evaluation</code>.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The status of the evaluation. This element can have one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PENDING</code> - Amazon Machine Learning (Amazon ML) submitted a request to evaluate an
     * <code>MLModel</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INPROGRESS</code> - The evaluation is underway.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> - The request to evaluate an <code>MLModel</code> did not run to completion. It is not
     * usable.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>COMPLETED</code> - The evaluation process completed successfully.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETED</code> - The <code>Evaluation</code> is marked as deleted. It is not usable.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link EntityStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the evaluation. This element can have one of the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PENDING</code> - Amazon Machine Learning (Amazon ML) submitted a request to evaluate an
     *         <code>MLModel</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INPROGRESS</code> - The evaluation is underway.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> - The request to evaluate an <code>MLModel</code> did not run to completion. It is
     *         not usable.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>COMPLETED</code> - The evaluation process completed successfully.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETED</code> - The <code>Evaluation</code> is marked as deleted. It is not usable.
     *         </p>
     *         </li>
     * @see EntityStatus
     */
    public final EntityStatus status() {
        return EntityStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the evaluation. This element can have one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PENDING</code> - Amazon Machine Learning (Amazon ML) submitted a request to evaluate an
     * <code>MLModel</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INPROGRESS</code> - The evaluation is underway.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> - The request to evaluate an <code>MLModel</code> did not run to completion. It is not
     * usable.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>COMPLETED</code> - The evaluation process completed successfully.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETED</code> - The <code>Evaluation</code> is marked as deleted. It is not usable.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link EntityStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the evaluation. This element can have one of the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PENDING</code> - Amazon Machine Learning (Amazon ML) submitted a request to evaluate an
     *         <code>MLModel</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INPROGRESS</code> - The evaluation is underway.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> - The request to evaluate an <code>MLModel</code> did not run to completion. It is
     *         not usable.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>COMPLETED</code> - The evaluation process completed successfully.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETED</code> - The <code>Evaluation</code> is marked as deleted. It is not usable.
     *         </p>
     *         </li>
     * @see EntityStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * Measurements of how well the <code>MLModel</code> performed, using observations referenced by the
     * <code>DataSource</code>. One of the following metrics is returned, based on the type of the <code>MLModel</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * BinaryAUC: A binary <code>MLModel</code> uses the Area Under the Curve (AUC) technique to measure performance.
     * </p>
     * </li>
     * <li>
     * <p>
     * RegressionRMSE: A regression <code>MLModel</code> uses the Root Mean Square Error (RMSE) technique to measure
     * performance. RMSE measures the difference between predicted and actual values for a single variable.
     * </p>
     * </li>
     * <li>
     * <p>
     * MulticlassAvgFScore: A multiclass <code>MLModel</code> uses the F1 score technique to measure performance.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information about performance metrics, please see the <a
     * href="https://docs.aws.amazon.com/machine-learning/latest/dg">Amazon Machine Learning Developer Guide</a>.
     * </p>
     * 
     * @return Measurements of how well the <code>MLModel</code> performed, using observations referenced by the
     *         <code>DataSource</code>. One of the following metrics is returned, based on the type of the
     *         <code>MLModel</code>: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         BinaryAUC: A binary <code>MLModel</code> uses the Area Under the Curve (AUC) technique to measure
     *         performance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         RegressionRMSE: A regression <code>MLModel</code> uses the Root Mean Square Error (RMSE) technique to
     *         measure performance. RMSE measures the difference between predicted and actual values for a single
     *         variable.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         MulticlassAvgFScore: A multiclass <code>MLModel</code> uses the F1 score technique to measure
     *         performance.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information about performance metrics, please see the <a
     *         href="https://docs.aws.amazon.com/machine-learning/latest/dg">Amazon Machine Learning Developer
     *         Guide</a>.
     */
    public final PerformanceMetrics performanceMetrics() {
        return performanceMetrics;
    }

    /**
     * <p>
     * A description of the most recent details about evaluating the <code>MLModel</code>.
     * </p>
     * 
     * @return A description of the most recent details about evaluating the <code>MLModel</code>.
     */
    public final String message() {
        return message;
    }

    /**
     * Returns the value of the ComputeTime property for this object.
     * 
     * @return The value of the ComputeTime property for this object.
     */
    public final Long computeTime() {
        return computeTime;
    }

    /**
     * Returns the value of the FinishedAt property for this object.
     * 
     * @return The value of the FinishedAt property for this object.
     */
    public final Instant finishedAt() {
        return finishedAt;
    }

    /**
     * Returns the value of the StartedAt property for this object.
     * 
     * @return The value of the StartedAt property for this object.
     */
    public final Instant startedAt() {
        return startedAt;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(evaluationId());
        hashCode = 31 * hashCode + Objects.hashCode(mlModelId());
        hashCode = 31 * hashCode + Objects.hashCode(evaluationDataSourceId());
        hashCode = 31 * hashCode + Objects.hashCode(inputDataLocationS3());
        hashCode = 31 * hashCode + Objects.hashCode(createdByIamUser());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedAt());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(performanceMetrics());
        hashCode = 31 * hashCode + Objects.hashCode(message());
        hashCode = 31 * hashCode + Objects.hashCode(computeTime());
        hashCode = 31 * hashCode + Objects.hashCode(finishedAt());
        hashCode = 31 * hashCode + Objects.hashCode(startedAt());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Evaluation)) {
            return false;
        }
        Evaluation other = (Evaluation) obj;
        return Objects.equals(evaluationId(), other.evaluationId()) && Objects.equals(mlModelId(), other.mlModelId())
                && Objects.equals(evaluationDataSourceId(), other.evaluationDataSourceId())
                && Objects.equals(inputDataLocationS3(), other.inputDataLocationS3())
                && Objects.equals(createdByIamUser(), other.createdByIamUser()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(lastUpdatedAt(), other.lastUpdatedAt()) && Objects.equals(name(), other.name())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(performanceMetrics(), other.performanceMetrics()) && Objects.equals(message(), other.message())
                && Objects.equals(computeTime(), other.computeTime()) && Objects.equals(finishedAt(), other.finishedAt())
                && Objects.equals(startedAt(), other.startedAt());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Evaluation").add("EvaluationId", evaluationId()).add("MLModelId", mlModelId())
                .add("EvaluationDataSourceId", evaluationDataSourceId()).add("InputDataLocationS3", inputDataLocationS3())
                .add("CreatedByIamUser", createdByIamUser()).add("CreatedAt", createdAt()).add("LastUpdatedAt", lastUpdatedAt())
                .add("Name", name()).add("Status", statusAsString()).add("PerformanceMetrics", performanceMetrics())
                .add("Message", message()).add("ComputeTime", computeTime()).add("FinishedAt", finishedAt())
                .add("StartedAt", startedAt()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "EvaluationId":
            return Optional.ofNullable(clazz.cast(evaluationId()));
        case "MLModelId":
            return Optional.ofNullable(clazz.cast(mlModelId()));
        case "EvaluationDataSourceId":
            return Optional.ofNullable(clazz.cast(evaluationDataSourceId()));
        case "InputDataLocationS3":
            return Optional.ofNullable(clazz.cast(inputDataLocationS3()));
        case "CreatedByIamUser":
            return Optional.ofNullable(clazz.cast(createdByIamUser()));
        case "CreatedAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "LastUpdatedAt":
            return Optional.ofNullable(clazz.cast(lastUpdatedAt()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "PerformanceMetrics":
            return Optional.ofNullable(clazz.cast(performanceMetrics()));
        case "Message":
            return Optional.ofNullable(clazz.cast(message()));
        case "ComputeTime":
            return Optional.ofNullable(clazz.cast(computeTime()));
        case "FinishedAt":
            return Optional.ofNullable(clazz.cast(finishedAt()));
        case "StartedAt":
            return Optional.ofNullable(clazz.cast(startedAt()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Evaluation, T> g) {
        return obj -> g.apply((Evaluation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Evaluation> {
        /**
         * <p>
         * The ID that is assigned to the <code>Evaluation</code> at creation.
         * </p>
         * 
         * @param evaluationId
         *        The ID that is assigned to the <code>Evaluation</code> at creation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder evaluationId(String evaluationId);

        /**
         * <p>
         * The ID of the <code>MLModel</code> that is the focus of the evaluation.
         * </p>
         * 
         * @param mlModelId
         *        The ID of the <code>MLModel</code> that is the focus of the evaluation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mlModelId(String mlModelId);

        /**
         * <p>
         * The ID of the <code>DataSource</code> that is used to evaluate the <code>MLModel</code>.
         * </p>
         * 
         * @param evaluationDataSourceId
         *        The ID of the <code>DataSource</code> that is used to evaluate the <code>MLModel</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder evaluationDataSourceId(String evaluationDataSourceId);

        /**
         * <p>
         * The location and name of the data in Amazon Simple Storage Server (Amazon S3) that is used in the evaluation.
         * </p>
         * 
         * @param inputDataLocationS3
         *        The location and name of the data in Amazon Simple Storage Server (Amazon S3) that is used in the
         *        evaluation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputDataLocationS3(String inputDataLocationS3);

        /**
         * <p>
         * The AWS user account that invoked the evaluation. The account type can be either an AWS root account or an
         * AWS Identity and Access Management (IAM) user account.
         * </p>
         * 
         * @param createdByIamUser
         *        The AWS user account that invoked the evaluation. The account type can be either an AWS root account
         *        or an AWS Identity and Access Management (IAM) user account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdByIamUser(String createdByIamUser);

        /**
         * <p>
         * The time that the <code>Evaluation</code> was created. The time is expressed in epoch time.
         * </p>
         * 
         * @param createdAt
         *        The time that the <code>Evaluation</code> was created. The time is expressed in epoch time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The time of the most recent edit to the <code>Evaluation</code>. The time is expressed in epoch time.
         * </p>
         * 
         * @param lastUpdatedAt
         *        The time of the most recent edit to the <code>Evaluation</code>. The time is expressed in epoch time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedAt(Instant lastUpdatedAt);

        /**
         * <p>
         * A user-supplied name or description of the <code>Evaluation</code>.
         * </p>
         * 
         * @param name
         *        A user-supplied name or description of the <code>Evaluation</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The status of the evaluation. This element can have one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PENDING</code> - Amazon Machine Learning (Amazon ML) submitted a request to evaluate an
         * <code>MLModel</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INPROGRESS</code> - The evaluation is underway.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> - The request to evaluate an <code>MLModel</code> did not run to completion. It is not
         * usable.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>COMPLETED</code> - The evaluation process completed successfully.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETED</code> - The <code>Evaluation</code> is marked as deleted. It is not usable.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the evaluation. This element can have one of the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PENDING</code> - Amazon Machine Learning (Amazon ML) submitted a request to evaluate an
         *        <code>MLModel</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INPROGRESS</code> - The evaluation is underway.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> - The request to evaluate an <code>MLModel</code> did not run to completion. It is
         *        not usable.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>COMPLETED</code> - The evaluation process completed successfully.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETED</code> - The <code>Evaluation</code> is marked as deleted. It is not usable.
         *        </p>
         *        </li>
         * @see EntityStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EntityStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the evaluation. This element can have one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PENDING</code> - Amazon Machine Learning (Amazon ML) submitted a request to evaluate an
         * <code>MLModel</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INPROGRESS</code> - The evaluation is underway.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> - The request to evaluate an <code>MLModel</code> did not run to completion. It is not
         * usable.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>COMPLETED</code> - The evaluation process completed successfully.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETED</code> - The <code>Evaluation</code> is marked as deleted. It is not usable.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the evaluation. This element can have one of the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PENDING</code> - Amazon Machine Learning (Amazon ML) submitted a request to evaluate an
         *        <code>MLModel</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INPROGRESS</code> - The evaluation is underway.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> - The request to evaluate an <code>MLModel</code> did not run to completion. It is
         *        not usable.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>COMPLETED</code> - The evaluation process completed successfully.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETED</code> - The <code>Evaluation</code> is marked as deleted. It is not usable.
         *        </p>
         *        </li>
         * @see EntityStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EntityStatus
         */
        Builder status(EntityStatus status);

        /**
         * <p>
         * Measurements of how well the <code>MLModel</code> performed, using observations referenced by the
         * <code>DataSource</code>. One of the following metrics is returned, based on the type of the
         * <code>MLModel</code>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * BinaryAUC: A binary <code>MLModel</code> uses the Area Under the Curve (AUC) technique to measure
         * performance.
         * </p>
         * </li>
         * <li>
         * <p>
         * RegressionRMSE: A regression <code>MLModel</code> uses the Root Mean Square Error (RMSE) technique to measure
         * performance. RMSE measures the difference between predicted and actual values for a single variable.
         * </p>
         * </li>
         * <li>
         * <p>
         * MulticlassAvgFScore: A multiclass <code>MLModel</code> uses the F1 score technique to measure performance.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information about performance metrics, please see the <a
         * href="https://docs.aws.amazon.com/machine-learning/latest/dg">Amazon Machine Learning Developer Guide</a>.
         * </p>
         * 
         * @param performanceMetrics
         *        Measurements of how well the <code>MLModel</code> performed, using observations referenced by the
         *        <code>DataSource</code>. One of the following metrics is returned, based on the type of the
         *        <code>MLModel</code>: </p>
         *        <ul>
         *        <li>
         *        <p>
         *        BinaryAUC: A binary <code>MLModel</code> uses the Area Under the Curve (AUC) technique to measure
         *        performance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        RegressionRMSE: A regression <code>MLModel</code> uses the Root Mean Square Error (RMSE) technique to
         *        measure performance. RMSE measures the difference between predicted and actual values for a single
         *        variable.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        MulticlassAvgFScore: A multiclass <code>MLModel</code> uses the F1 score technique to measure
         *        performance.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information about performance metrics, please see the <a
         *        href="https://docs.aws.amazon.com/machine-learning/latest/dg">Amazon Machine Learning Developer
         *        Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder performanceMetrics(PerformanceMetrics performanceMetrics);

        /**
         * <p>
         * Measurements of how well the <code>MLModel</code> performed, using observations referenced by the
         * <code>DataSource</code>. One of the following metrics is returned, based on the type of the
         * <code>MLModel</code>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * BinaryAUC: A binary <code>MLModel</code> uses the Area Under the Curve (AUC) technique to measure
         * performance.
         * </p>
         * </li>
         * <li>
         * <p>
         * RegressionRMSE: A regression <code>MLModel</code> uses the Root Mean Square Error (RMSE) technique to measure
         * performance. RMSE measures the difference between predicted and actual values for a single variable.
         * </p>
         * </li>
         * <li>
         * <p>
         * MulticlassAvgFScore: A multiclass <code>MLModel</code> uses the F1 score technique to measure performance.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information about performance metrics, please see the <a
         * href="https://docs.aws.amazon.com/machine-learning/latest/dg">Amazon Machine Learning Developer Guide</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link PerformanceMetrics.Builder} avoiding the need to
         * create one manually via {@link PerformanceMetrics#builder()}.
         *
         * When the {@link Consumer} completes, {@link PerformanceMetrics.Builder#build()} is called immediately and its
         * result is passed to {@link #performanceMetrics(PerformanceMetrics)}.
         * 
         * @param performanceMetrics
         *        a consumer that will call methods on {@link PerformanceMetrics.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #performanceMetrics(PerformanceMetrics)
         */
        default Builder performanceMetrics(Consumer<PerformanceMetrics.Builder> performanceMetrics) {
            return performanceMetrics(PerformanceMetrics.builder().applyMutation(performanceMetrics).build());
        }

        /**
         * <p>
         * A description of the most recent details about evaluating the <code>MLModel</code>.
         * </p>
         * 
         * @param message
         *        A description of the most recent details about evaluating the <code>MLModel</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder message(String message);

        /**
         * Sets the value of the ComputeTime property for this object.
         *
         * @param computeTime
         *        The new value for the ComputeTime property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder computeTime(Long computeTime);

        /**
         * Sets the value of the FinishedAt property for this object.
         *
         * @param finishedAt
         *        The new value for the FinishedAt property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder finishedAt(Instant finishedAt);

        /**
         * Sets the value of the StartedAt property for this object.
         *
         * @param startedAt
         *        The new value for the StartedAt property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startedAt(Instant startedAt);
    }

    static final class BuilderImpl implements Builder {
        private String evaluationId;

        private String mlModelId;

        private String evaluationDataSourceId;

        private String inputDataLocationS3;

        private String createdByIamUser;

        private Instant createdAt;

        private Instant lastUpdatedAt;

        private String name;

        private String status;

        private PerformanceMetrics performanceMetrics;

        private String message;

        private Long computeTime;

        private Instant finishedAt;

        private Instant startedAt;

        private BuilderImpl() {
        }

        private BuilderImpl(Evaluation model) {
            evaluationId(model.evaluationId);
            mlModelId(model.mlModelId);
            evaluationDataSourceId(model.evaluationDataSourceId);
            inputDataLocationS3(model.inputDataLocationS3);
            createdByIamUser(model.createdByIamUser);
            createdAt(model.createdAt);
            lastUpdatedAt(model.lastUpdatedAt);
            name(model.name);
            status(model.status);
            performanceMetrics(model.performanceMetrics);
            message(model.message);
            computeTime(model.computeTime);
            finishedAt(model.finishedAt);
            startedAt(model.startedAt);
        }

        public final String getEvaluationId() {
            return evaluationId;
        }

        @Override
        public final Builder evaluationId(String evaluationId) {
            this.evaluationId = evaluationId;
            return this;
        }

        public final void setEvaluationId(String evaluationId) {
            this.evaluationId = evaluationId;
        }

        public final String getMlModelId() {
            return mlModelId;
        }

        @Override
        public final Builder mlModelId(String mlModelId) {
            this.mlModelId = mlModelId;
            return this;
        }

        public final void setMlModelId(String mlModelId) {
            this.mlModelId = mlModelId;
        }

        public final String getEvaluationDataSourceId() {
            return evaluationDataSourceId;
        }

        @Override
        public final Builder evaluationDataSourceId(String evaluationDataSourceId) {
            this.evaluationDataSourceId = evaluationDataSourceId;
            return this;
        }

        public final void setEvaluationDataSourceId(String evaluationDataSourceId) {
            this.evaluationDataSourceId = evaluationDataSourceId;
        }

        public final String getInputDataLocationS3() {
            return inputDataLocationS3;
        }

        @Override
        public final Builder inputDataLocationS3(String inputDataLocationS3) {
            this.inputDataLocationS3 = inputDataLocationS3;
            return this;
        }

        public final void setInputDataLocationS3(String inputDataLocationS3) {
            this.inputDataLocationS3 = inputDataLocationS3;
        }

        public final String getCreatedByIamUser() {
            return createdByIamUser;
        }

        @Override
        public final Builder createdByIamUser(String createdByIamUser) {
            this.createdByIamUser = createdByIamUser;
            return this;
        }

        public final void setCreatedByIamUser(String createdByIamUser) {
            this.createdByIamUser = createdByIamUser;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        public final Instant getLastUpdatedAt() {
            return lastUpdatedAt;
        }

        @Override
        public final Builder lastUpdatedAt(Instant lastUpdatedAt) {
            this.lastUpdatedAt = lastUpdatedAt;
            return this;
        }

        public final void setLastUpdatedAt(Instant lastUpdatedAt) {
            this.lastUpdatedAt = lastUpdatedAt;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(EntityStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final PerformanceMetrics.Builder getPerformanceMetrics() {
            return performanceMetrics != null ? performanceMetrics.toBuilder() : null;
        }

        @Override
        public final Builder performanceMetrics(PerformanceMetrics performanceMetrics) {
            this.performanceMetrics = performanceMetrics;
            return this;
        }

        public final void setPerformanceMetrics(PerformanceMetrics.BuilderImpl performanceMetrics) {
            this.performanceMetrics = performanceMetrics != null ? performanceMetrics.build() : null;
        }

        public final String getMessage() {
            return message;
        }

        @Override
        public final Builder message(String message) {
            this.message = message;
            return this;
        }

        public final void setMessage(String message) {
            this.message = message;
        }

        public final Long getComputeTime() {
            return computeTime;
        }

        @Override
        public final Builder computeTime(Long computeTime) {
            this.computeTime = computeTime;
            return this;
        }

        public final void setComputeTime(Long computeTime) {
            this.computeTime = computeTime;
        }

        public final Instant getFinishedAt() {
            return finishedAt;
        }

        @Override
        public final Builder finishedAt(Instant finishedAt) {
            this.finishedAt = finishedAt;
            return this;
        }

        public final void setFinishedAt(Instant finishedAt) {
            this.finishedAt = finishedAt;
        }

        public final Instant getStartedAt() {
            return startedAt;
        }

        @Override
        public final Builder startedAt(Instant startedAt) {
            this.startedAt = startedAt;
            return this;
        }

        public final void setStartedAt(Instant startedAt) {
            this.startedAt = startedAt;
        }

        @Override
        public Evaluation build() {
            return new Evaluation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
