/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.machinelearning;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.machinelearning.internal.MachineLearningServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.machinelearning.model.AddTagsRequest;
import software.amazon.awssdk.services.machinelearning.model.AddTagsResponse;
import software.amazon.awssdk.services.machinelearning.model.CreateBatchPredictionRequest;
import software.amazon.awssdk.services.machinelearning.model.CreateBatchPredictionResponse;
import software.amazon.awssdk.services.machinelearning.model.CreateDataSourceFromRdsRequest;
import software.amazon.awssdk.services.machinelearning.model.CreateDataSourceFromRdsResponse;
import software.amazon.awssdk.services.machinelearning.model.CreateDataSourceFromRedshiftRequest;
import software.amazon.awssdk.services.machinelearning.model.CreateDataSourceFromRedshiftResponse;
import software.amazon.awssdk.services.machinelearning.model.CreateDataSourceFromS3Request;
import software.amazon.awssdk.services.machinelearning.model.CreateDataSourceFromS3Response;
import software.amazon.awssdk.services.machinelearning.model.CreateEvaluationRequest;
import software.amazon.awssdk.services.machinelearning.model.CreateEvaluationResponse;
import software.amazon.awssdk.services.machinelearning.model.CreateMlModelRequest;
import software.amazon.awssdk.services.machinelearning.model.CreateMlModelResponse;
import software.amazon.awssdk.services.machinelearning.model.CreateRealtimeEndpointRequest;
import software.amazon.awssdk.services.machinelearning.model.CreateRealtimeEndpointResponse;
import software.amazon.awssdk.services.machinelearning.model.DeleteBatchPredictionRequest;
import software.amazon.awssdk.services.machinelearning.model.DeleteBatchPredictionResponse;
import software.amazon.awssdk.services.machinelearning.model.DeleteDataSourceRequest;
import software.amazon.awssdk.services.machinelearning.model.DeleteDataSourceResponse;
import software.amazon.awssdk.services.machinelearning.model.DeleteEvaluationRequest;
import software.amazon.awssdk.services.machinelearning.model.DeleteEvaluationResponse;
import software.amazon.awssdk.services.machinelearning.model.DeleteMlModelRequest;
import software.amazon.awssdk.services.machinelearning.model.DeleteMlModelResponse;
import software.amazon.awssdk.services.machinelearning.model.DeleteRealtimeEndpointRequest;
import software.amazon.awssdk.services.machinelearning.model.DeleteRealtimeEndpointResponse;
import software.amazon.awssdk.services.machinelearning.model.DeleteTagsRequest;
import software.amazon.awssdk.services.machinelearning.model.DeleteTagsResponse;
import software.amazon.awssdk.services.machinelearning.model.DescribeBatchPredictionsRequest;
import software.amazon.awssdk.services.machinelearning.model.DescribeBatchPredictionsResponse;
import software.amazon.awssdk.services.machinelearning.model.DescribeDataSourcesRequest;
import software.amazon.awssdk.services.machinelearning.model.DescribeDataSourcesResponse;
import software.amazon.awssdk.services.machinelearning.model.DescribeEvaluationsRequest;
import software.amazon.awssdk.services.machinelearning.model.DescribeEvaluationsResponse;
import software.amazon.awssdk.services.machinelearning.model.DescribeMlModelsRequest;
import software.amazon.awssdk.services.machinelearning.model.DescribeMlModelsResponse;
import software.amazon.awssdk.services.machinelearning.model.DescribeTagsRequest;
import software.amazon.awssdk.services.machinelearning.model.DescribeTagsResponse;
import software.amazon.awssdk.services.machinelearning.model.GetBatchPredictionRequest;
import software.amazon.awssdk.services.machinelearning.model.GetBatchPredictionResponse;
import software.amazon.awssdk.services.machinelearning.model.GetDataSourceRequest;
import software.amazon.awssdk.services.machinelearning.model.GetDataSourceResponse;
import software.amazon.awssdk.services.machinelearning.model.GetEvaluationRequest;
import software.amazon.awssdk.services.machinelearning.model.GetEvaluationResponse;
import software.amazon.awssdk.services.machinelearning.model.GetMlModelRequest;
import software.amazon.awssdk.services.machinelearning.model.GetMlModelResponse;
import software.amazon.awssdk.services.machinelearning.model.IdempotentParameterMismatchException;
import software.amazon.awssdk.services.machinelearning.model.InternalServerException;
import software.amazon.awssdk.services.machinelearning.model.InvalidInputException;
import software.amazon.awssdk.services.machinelearning.model.InvalidTagException;
import software.amazon.awssdk.services.machinelearning.model.LimitExceededException;
import software.amazon.awssdk.services.machinelearning.model.MachineLearningException;
import software.amazon.awssdk.services.machinelearning.model.PredictRequest;
import software.amazon.awssdk.services.machinelearning.model.PredictResponse;
import software.amazon.awssdk.services.machinelearning.model.PredictorNotMountedException;
import software.amazon.awssdk.services.machinelearning.model.ResourceNotFoundException;
import software.amazon.awssdk.services.machinelearning.model.TagLimitExceededException;
import software.amazon.awssdk.services.machinelearning.model.UpdateBatchPredictionRequest;
import software.amazon.awssdk.services.machinelearning.model.UpdateBatchPredictionResponse;
import software.amazon.awssdk.services.machinelearning.model.UpdateDataSourceRequest;
import software.amazon.awssdk.services.machinelearning.model.UpdateDataSourceResponse;
import software.amazon.awssdk.services.machinelearning.model.UpdateEvaluationRequest;
import software.amazon.awssdk.services.machinelearning.model.UpdateEvaluationResponse;
import software.amazon.awssdk.services.machinelearning.model.UpdateMlModelRequest;
import software.amazon.awssdk.services.machinelearning.model.UpdateMlModelResponse;
import software.amazon.awssdk.services.machinelearning.transform.AddTagsRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.CreateBatchPredictionRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.CreateDataSourceFromRdsRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.CreateDataSourceFromRedshiftRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.CreateDataSourceFromS3RequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.CreateEvaluationRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.CreateMlModelRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.CreateRealtimeEndpointRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.DeleteBatchPredictionRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.DeleteDataSourceRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.DeleteEvaluationRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.DeleteMlModelRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.DeleteRealtimeEndpointRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.DeleteTagsRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.DescribeBatchPredictionsRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.DescribeDataSourcesRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.DescribeEvaluationsRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.DescribeMlModelsRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.DescribeTagsRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.GetBatchPredictionRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.GetDataSourceRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.GetEvaluationRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.GetMlModelRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.PredictRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.UpdateBatchPredictionRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.UpdateDataSourceRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.UpdateEvaluationRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.transform.UpdateMlModelRequestMarshaller;
import software.amazon.awssdk.services.machinelearning.waiters.MachineLearningAsyncWaiter;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link MachineLearningAsyncClient}.
 *
 * @see MachineLearningAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMachineLearningAsyncClient implements MachineLearningAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultMachineLearningAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final ScheduledExecutorService executorService;

    protected DefaultMachineLearningAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
        this.executorService = clientConfiguration.option(SdkClientOption.SCHEDULED_EXECUTOR_SERVICE);
    }

    /**
     * <p>
     * Adds one or more tags to an object, up to a limit of 10. Each tag consists of a key and an optional value. If you
     * add a tag using a key that is already associated with the ML object, <code>AddTags</code> updates the tag's
     * value.
     * </p>
     *
     * @param addTagsRequest
     * @return A Java Future containing the result of the AddTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidInputException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>InvalidTagException</li>
     *         <li>TagLimitExceededException</li>
     *         <li>ResourceNotFoundException A specified resource cannot be located.</li>
     *         <li>InternalServerException An error on the server occurred when trying to process a request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MachineLearningException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MachineLearningAsyncClient.AddTags
     */
    @Override
    public CompletableFuture<AddTagsResponse> addTags(AddTagsRequest addTagsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(addTagsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddTags");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AddTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    AddTagsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AddTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AddTagsRequest, AddTagsResponse>().withOperationName("AddTags")
                            .withProtocolMetadata(protocolMetadata).withMarshaller(new AddTagsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(addTagsRequest));
            CompletableFuture<AddTagsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Generates predictions for a group of observations. The observations to process exist in one or more data files
     * referenced by a <code>DataSource</code>. This operation creates a new <code>BatchPrediction</code>, and uses an
     * <code>MLModel</code> and the data files referenced by the <code>DataSource</code> as information sources.
     * </p>
     * <p>
     * <code>CreateBatchPrediction</code> is an asynchronous operation. In response to
     * <code>CreateBatchPrediction</code>, Amazon Machine Learning (Amazon ML) immediately returns and sets the
     * <code>BatchPrediction</code> status to <code>PENDING</code>. After the <code>BatchPrediction</code> completes,
     * Amazon ML sets the status to <code>COMPLETED</code>.
     * </p>
     * <p>
     * You can poll for status updates by using the <a>GetBatchPrediction</a> operation and checking the
     * <code>Status</code> parameter of the result. After the <code>COMPLETED</code> status appears, the results are
     * available in the location specified by the <code>OutputUri</code> parameter.
     * </p>
     *
     * @param createBatchPredictionRequest
     * @return A Java Future containing the result of the CreateBatchPrediction operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidInputException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>InternalServerException An error on the server occurred when trying to process a request.</li>
     *         <li>IdempotentParameterMismatchException A second request to use or change an object was not allowed.
     *         This can result from retrying a request using a parameter that was not present in the original request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MachineLearningException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MachineLearningAsyncClient.CreateBatchPrediction
     */
    @Override
    public CompletableFuture<CreateBatchPredictionResponse> createBatchPrediction(
            CreateBatchPredictionRequest createBatchPredictionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createBatchPredictionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createBatchPredictionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBatchPrediction");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateBatchPredictionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateBatchPredictionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateBatchPredictionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateBatchPredictionRequest, CreateBatchPredictionResponse>()
                            .withOperationName("CreateBatchPrediction").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateBatchPredictionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createBatchPredictionRequest));
            CompletableFuture<CreateBatchPredictionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a <code>DataSource</code> object from an <a href="http://aws.amazon.com/rds/"> Amazon Relational Database
     * Service</a> (Amazon RDS). A <code>DataSource</code> references data that can be used to perform
     * <code>CreateMLModel</code>, <code>CreateEvaluation</code>, or <code>CreateBatchPrediction</code> operations.
     * </p>
     * <p>
     * <code>CreateDataSourceFromRDS</code> is an asynchronous operation. In response to
     * <code>CreateDataSourceFromRDS</code>, Amazon Machine Learning (Amazon ML) immediately returns and sets the
     * <code>DataSource</code> status to <code>PENDING</code>. After the <code>DataSource</code> is created and ready
     * for use, Amazon ML sets the <code>Status</code> parameter to <code>COMPLETED</code>. <code>DataSource</code> in
     * the <code>COMPLETED</code> or <code>PENDING</code> state can be used only to perform
     * <code>&gt;CreateMLModel</code>&gt;, <code>CreateEvaluation</code>, or <code>CreateBatchPrediction</code>
     * operations.
     * </p>
     * <p>
     * If Amazon ML cannot accept the input source, it sets the <code>Status</code> parameter to <code>FAILED</code> and
     * includes an error message in the <code>Message</code> attribute of the <code>GetDataSource</code> operation
     * response.
     * </p>
     *
     * @param createDataSourceFromRdsRequest
     * @return A Java Future containing the result of the CreateDataSourceFromRDS operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidInputException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>InternalServerException An error on the server occurred when trying to process a request.</li>
     *         <li>IdempotentParameterMismatchException A second request to use or change an object was not allowed.
     *         This can result from retrying a request using a parameter that was not present in the original request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MachineLearningException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MachineLearningAsyncClient.CreateDataSourceFromRDS
     */
    @Override
    public CompletableFuture<CreateDataSourceFromRdsResponse> createDataSourceFromRDS(
            CreateDataSourceFromRdsRequest createDataSourceFromRdsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDataSourceFromRdsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDataSourceFromRdsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDataSourceFromRDS");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDataSourceFromRdsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDataSourceFromRdsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDataSourceFromRdsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDataSourceFromRdsRequest, CreateDataSourceFromRdsResponse>()
                            .withOperationName("CreateDataSourceFromRDS").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateDataSourceFromRdsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createDataSourceFromRdsRequest));
            CompletableFuture<CreateDataSourceFromRdsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a <code>DataSource</code> from a database hosted on an Amazon Redshift cluster. A <code>DataSource</code>
     * references data that can be used to perform either <code>CreateMLModel</code>, <code>CreateEvaluation</code>, or
     * <code>CreateBatchPrediction</code> operations.
     * </p>
     * <p>
     * <code>CreateDataSourceFromRedshift</code> is an asynchronous operation. In response to
     * <code>CreateDataSourceFromRedshift</code>, Amazon Machine Learning (Amazon ML) immediately returns and sets the
     * <code>DataSource</code> status to <code>PENDING</code>. After the <code>DataSource</code> is created and ready
     * for use, Amazon ML sets the <code>Status</code> parameter to <code>COMPLETED</code>. <code>DataSource</code> in
     * <code>COMPLETED</code> or <code>PENDING</code> states can be used to perform only <code>CreateMLModel</code>,
     * <code>CreateEvaluation</code>, or <code>CreateBatchPrediction</code> operations.
     * </p>
     * <p>
     * If Amazon ML can't accept the input source, it sets the <code>Status</code> parameter to <code>FAILED</code> and
     * includes an error message in the <code>Message</code> attribute of the <code>GetDataSource</code> operation
     * response.
     * </p>
     * <p>
     * The observations should be contained in the database hosted on an Amazon Redshift cluster and should be specified
     * by a <code>SelectSqlQuery</code> query. Amazon ML executes an <code>Unload</code> command in Amazon Redshift to
     * transfer the result set of the <code>SelectSqlQuery</code> query to <code>S3StagingLocation</code>.
     * </p>
     * <p>
     * After the <code>DataSource</code> has been created, it's ready for use in evaluations and batch predictions. If
     * you plan to use the <code>DataSource</code> to train an <code>MLModel</code>, the <code>DataSource</code> also
     * requires a recipe. A recipe describes how each input variable will be used in training an <code>MLModel</code>.
     * Will the variable be included or excluded from training? Will the variable be manipulated; for example, will it
     * be combined with another variable or will it be split apart into word combinations? The recipe provides answers
     * to these questions.
     * </p>
     * <p>
     * You can't change an existing datasource, but you can copy and modify the settings from an existing Amazon
     * Redshift datasource to create a new datasource. To do so, call <code>GetDataSource</code> for an existing
     * datasource and copy the values to a <code>CreateDataSource</code> call. Change the settings that you want to
     * change and make sure that all required fields have the appropriate values.
     * </p>
     *
     * @param createDataSourceFromRedshiftRequest
     * @return A Java Future containing the result of the CreateDataSourceFromRedshift operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidInputException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>InternalServerException An error on the server occurred when trying to process a request.</li>
     *         <li>IdempotentParameterMismatchException A second request to use or change an object was not allowed.
     *         This can result from retrying a request using a parameter that was not present in the original request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MachineLearningException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MachineLearningAsyncClient.CreateDataSourceFromRedshift
     */
    @Override
    public CompletableFuture<CreateDataSourceFromRedshiftResponse> createDataSourceFromRedshift(
            CreateDataSourceFromRedshiftRequest createDataSourceFromRedshiftRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDataSourceFromRedshiftRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDataSourceFromRedshiftRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDataSourceFromRedshift");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDataSourceFromRedshiftResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDataSourceFromRedshiftResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDataSourceFromRedshiftResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDataSourceFromRedshiftRequest, CreateDataSourceFromRedshiftResponse>()
                            .withOperationName("CreateDataSourceFromRedshift").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateDataSourceFromRedshiftRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createDataSourceFromRedshiftRequest));
            CompletableFuture<CreateDataSourceFromRedshiftResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a <code>DataSource</code> object. A <code>DataSource</code> references data that can be used to perform
     * <code>CreateMLModel</code>, <code>CreateEvaluation</code>, or <code>CreateBatchPrediction</code> operations.
     * </p>
     * <p>
     * <code>CreateDataSourceFromS3</code> is an asynchronous operation. In response to
     * <code>CreateDataSourceFromS3</code>, Amazon Machine Learning (Amazon ML) immediately returns and sets the
     * <code>DataSource</code> status to <code>PENDING</code>. After the <code>DataSource</code> has been created and is
     * ready for use, Amazon ML sets the <code>Status</code> parameter to <code>COMPLETED</code>.
     * <code>DataSource</code> in the <code>COMPLETED</code> or <code>PENDING</code> state can be used to perform only
     * <code>CreateMLModel</code>, <code>CreateEvaluation</code> or <code>CreateBatchPrediction</code> operations.
     * </p>
     * <p>
     * If Amazon ML can't accept the input source, it sets the <code>Status</code> parameter to <code>FAILED</code> and
     * includes an error message in the <code>Message</code> attribute of the <code>GetDataSource</code> operation
     * response.
     * </p>
     * <p>
     * The observation data used in a <code>DataSource</code> should be ready to use; that is, it should have a
     * consistent structure, and missing data values should be kept to a minimum. The observation data must reside in
     * one or more .csv files in an Amazon Simple Storage Service (Amazon S3) location, along with a schema that
     * describes the data items by name and type. The same schema must be used for all of the data files referenced by
     * the <code>DataSource</code>.
     * </p>
     * <p>
     * After the <code>DataSource</code> has been created, it's ready to use in evaluations and batch predictions. If
     * you plan to use the <code>DataSource</code> to train an <code>MLModel</code>, the <code>DataSource</code> also
     * needs a recipe. A recipe describes how each input variable will be used in training an <code>MLModel</code>. Will
     * the variable be included or excluded from training? Will the variable be manipulated; for example, will it be
     * combined with another variable or will it be split apart into word combinations? The recipe provides answers to
     * these questions.
     * </p>
     *
     * @param createDataSourceFromS3Request
     * @return A Java Future containing the result of the CreateDataSourceFromS3 operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidInputException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>InternalServerException An error on the server occurred when trying to process a request.</li>
     *         <li>IdempotentParameterMismatchException A second request to use or change an object was not allowed.
     *         This can result from retrying a request using a parameter that was not present in the original request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MachineLearningException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MachineLearningAsyncClient.CreateDataSourceFromS3
     */
    @Override
    public CompletableFuture<CreateDataSourceFromS3Response> createDataSourceFromS3(
            CreateDataSourceFromS3Request createDataSourceFromS3Request) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDataSourceFromS3Request,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDataSourceFromS3Request
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDataSourceFromS3");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDataSourceFromS3Response> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDataSourceFromS3Response::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDataSourceFromS3Response> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDataSourceFromS3Request, CreateDataSourceFromS3Response>()
                            .withOperationName("CreateDataSourceFromS3").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateDataSourceFromS3RequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createDataSourceFromS3Request));
            CompletableFuture<CreateDataSourceFromS3Response> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new <code>Evaluation</code> of an <code>MLModel</code>. An <code>MLModel</code> is evaluated on a set
     * of observations associated to a <code>DataSource</code>. Like a <code>DataSource</code> for an
     * <code>MLModel</code>, the <code>DataSource</code> for an <code>Evaluation</code> contains values for the
     * <code>Target Variable</code>. The <code>Evaluation</code> compares the predicted result for each observation to
     * the actual outcome and provides a summary so that you know how effective the <code>MLModel</code> functions on
     * the test data. Evaluation generates a relevant performance metric, such as BinaryAUC, RegressionRMSE or
     * MulticlassAvgFScore based on the corresponding <code>MLModelType</code>: <code>BINARY</code>,
     * <code>REGRESSION</code> or <code>MULTICLASS</code>.
     * </p>
     * <p>
     * <code>CreateEvaluation</code> is an asynchronous operation. In response to <code>CreateEvaluation</code>, Amazon
     * Machine Learning (Amazon ML) immediately returns and sets the evaluation status to <code>PENDING</code>. After
     * the <code>Evaluation</code> is created and ready for use, Amazon ML sets the status to <code>COMPLETED</code>.
     * </p>
     * <p>
     * You can use the <code>GetEvaluation</code> operation to check progress of the evaluation during the creation
     * operation.
     * </p>
     *
     * @param createEvaluationRequest
     * @return A Java Future containing the result of the CreateEvaluation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidInputException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>InternalServerException An error on the server occurred when trying to process a request.</li>
     *         <li>IdempotentParameterMismatchException A second request to use or change an object was not allowed.
     *         This can result from retrying a request using a parameter that was not present in the original request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MachineLearningException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MachineLearningAsyncClient.CreateEvaluation
     */
    @Override
    public CompletableFuture<CreateEvaluationResponse> createEvaluation(CreateEvaluationRequest createEvaluationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createEvaluationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createEvaluationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateEvaluation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateEvaluationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateEvaluationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateEvaluationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateEvaluationRequest, CreateEvaluationResponse>()
                            .withOperationName("CreateEvaluation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateEvaluationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createEvaluationRequest));
            CompletableFuture<CreateEvaluationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new <code>MLModel</code> using the <code>DataSource</code> and the recipe as information sources.
     * </p>
     * <p>
     * An <code>MLModel</code> is nearly immutable. Users can update only the <code>MLModelName</code> and the
     * <code>ScoreThreshold</code> in an <code>MLModel</code> without creating a new <code>MLModel</code>.
     * </p>
     * <p>
     * <code>CreateMLModel</code> is an asynchronous operation. In response to <code>CreateMLModel</code>, Amazon
     * Machine Learning (Amazon ML) immediately returns and sets the <code>MLModel</code> status to <code>PENDING</code>
     * . After the <code>MLModel</code> has been created and ready is for use, Amazon ML sets the status to
     * <code>COMPLETED</code>.
     * </p>
     * <p>
     * You can use the <code>GetMLModel</code> operation to check the progress of the <code>MLModel</code> during the
     * creation operation.
     * </p>
     * <p>
     * <code>CreateMLModel</code> requires a <code>DataSource</code> with computed statistics, which can be created by
     * setting <code>ComputeStatistics</code> to <code>true</code> in <code>CreateDataSourceFromRDS</code>,
     * <code>CreateDataSourceFromS3</code>, or <code>CreateDataSourceFromRedshift</code> operations.
     * </p>
     *
     * @param createMlModelRequest
     * @return A Java Future containing the result of the CreateMLModel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidInputException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>InternalServerException An error on the server occurred when trying to process a request.</li>
     *         <li>IdempotentParameterMismatchException A second request to use or change an object was not allowed.
     *         This can result from retrying a request using a parameter that was not present in the original request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MachineLearningException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MachineLearningAsyncClient.CreateMLModel
     */
    @Override
    public CompletableFuture<CreateMlModelResponse> createMLModel(CreateMlModelRequest createMlModelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createMlModelRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMlModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMLModel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateMlModelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateMlModelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateMlModelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateMlModelRequest, CreateMlModelResponse>()
                            .withOperationName("CreateMLModel").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateMlModelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createMlModelRequest));
            CompletableFuture<CreateMlModelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a real-time endpoint for the <code>MLModel</code>. The endpoint contains the URI of the
     * <code>MLModel</code>; that is, the location to send real-time prediction requests for the specified
     * <code>MLModel</code>.
     * </p>
     *
     * @param createRealtimeEndpointRequest
     * @return A Java Future containing the result of the CreateRealtimeEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidInputException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>ResourceNotFoundException A specified resource cannot be located.</li>
     *         <li>InternalServerException An error on the server occurred when trying to process a request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MachineLearningException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MachineLearningAsyncClient.CreateRealtimeEndpoint
     */
    @Override
    public CompletableFuture<CreateRealtimeEndpointResponse> createRealtimeEndpoint(
            CreateRealtimeEndpointRequest createRealtimeEndpointRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createRealtimeEndpointRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRealtimeEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRealtimeEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateRealtimeEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateRealtimeEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateRealtimeEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateRealtimeEndpointRequest, CreateRealtimeEndpointResponse>()
                            .withOperationName("CreateRealtimeEndpoint").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateRealtimeEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createRealtimeEndpointRequest));
            CompletableFuture<CreateRealtimeEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Assigns the DELETED status to a <code>BatchPrediction</code>, rendering it unusable.
     * </p>
     * <p>
     * After using the <code>DeleteBatchPrediction</code> operation, you can use the <a>GetBatchPrediction</a> operation
     * to verify that the status of the <code>BatchPrediction</code> changed to DELETED.
     * </p>
     * <p>
     * <b>Caution:</b> The result of the <code>DeleteBatchPrediction</code> operation is irreversible.
     * </p>
     *
     * @param deleteBatchPredictionRequest
     * @return A Java Future containing the result of the DeleteBatchPrediction operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidInputException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>ResourceNotFoundException A specified resource cannot be located.</li>
     *         <li>InternalServerException An error on the server occurred when trying to process a request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MachineLearningException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MachineLearningAsyncClient.DeleteBatchPrediction
     */
    @Override
    public CompletableFuture<DeleteBatchPredictionResponse> deleteBatchPrediction(
            DeleteBatchPredictionRequest deleteBatchPredictionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteBatchPredictionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBatchPredictionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBatchPrediction");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteBatchPredictionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteBatchPredictionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteBatchPredictionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteBatchPredictionRequest, DeleteBatchPredictionResponse>()
                            .withOperationName("DeleteBatchPrediction").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteBatchPredictionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteBatchPredictionRequest));
            CompletableFuture<DeleteBatchPredictionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Assigns the DELETED status to a <code>DataSource</code>, rendering it unusable.
     * </p>
     * <p>
     * After using the <code>DeleteDataSource</code> operation, you can use the <a>GetDataSource</a> operation to verify
     * that the status of the <code>DataSource</code> changed to DELETED.
     * </p>
     * <p>
     * <b>Caution:</b> The results of the <code>DeleteDataSource</code> operation are irreversible.
     * </p>
     *
     * @param deleteDataSourceRequest
     * @return A Java Future containing the result of the DeleteDataSource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidInputException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>ResourceNotFoundException A specified resource cannot be located.</li>
     *         <li>InternalServerException An error on the server occurred when trying to process a request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MachineLearningException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MachineLearningAsyncClient.DeleteDataSource
     */
    @Override
    public CompletableFuture<DeleteDataSourceResponse> deleteDataSource(DeleteDataSourceRequest deleteDataSourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteDataSourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDataSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDataSource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDataSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDataSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDataSourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDataSourceRequest, DeleteDataSourceResponse>()
                            .withOperationName("DeleteDataSource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteDataSourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteDataSourceRequest));
            CompletableFuture<DeleteDataSourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Assigns the <code>DELETED</code> status to an <code>Evaluation</code>, rendering it unusable.
     * </p>
     * <p>
     * After invoking the <code>DeleteEvaluation</code> operation, you can use the <code>GetEvaluation</code> operation
     * to verify that the status of the <code>Evaluation</code> changed to <code>DELETED</code>.
     * </p>
     * <p>
     * <b>Caution:</b> The results of the <code>DeleteEvaluation</code> operation are irreversible.
     * </p>
     *
     * @param deleteEvaluationRequest
     * @return A Java Future containing the result of the DeleteEvaluation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidInputException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>ResourceNotFoundException A specified resource cannot be located.</li>
     *         <li>InternalServerException An error on the server occurred when trying to process a request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MachineLearningException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MachineLearningAsyncClient.DeleteEvaluation
     */
    @Override
    public CompletableFuture<DeleteEvaluationResponse> deleteEvaluation(DeleteEvaluationRequest deleteEvaluationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteEvaluationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteEvaluationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteEvaluation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteEvaluationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteEvaluationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteEvaluationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteEvaluationRequest, DeleteEvaluationResponse>()
                            .withOperationName("DeleteEvaluation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteEvaluationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteEvaluationRequest));
            CompletableFuture<DeleteEvaluationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Assigns the <code>DELETED</code> status to an <code>MLModel</code>, rendering it unusable.
     * </p>
     * <p>
     * After using the <code>DeleteMLModel</code> operation, you can use the <code>GetMLModel</code> operation to verify
     * that the status of the <code>MLModel</code> changed to DELETED.
     * </p>
     * <p>
     * <b>Caution:</b> The result of the <code>DeleteMLModel</code> operation is irreversible.
     * </p>
     *
     * @param deleteMlModelRequest
     * @return A Java Future containing the result of the DeleteMLModel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidInputException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>ResourceNotFoundException A specified resource cannot be located.</li>
     *         <li>InternalServerException An error on the server occurred when trying to process a request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MachineLearningException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MachineLearningAsyncClient.DeleteMLModel
     */
    @Override
    public CompletableFuture<DeleteMlModelResponse> deleteMLModel(DeleteMlModelRequest deleteMlModelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteMlModelRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMlModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMLModel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteMlModelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteMlModelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteMlModelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteMlModelRequest, DeleteMlModelResponse>()
                            .withOperationName("DeleteMLModel").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteMlModelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteMlModelRequest));
            CompletableFuture<DeleteMlModelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a real time endpoint of an <code>MLModel</code>.
     * </p>
     *
     * @param deleteRealtimeEndpointRequest
     * @return A Java Future containing the result of the DeleteRealtimeEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidInputException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>ResourceNotFoundException A specified resource cannot be located.</li>
     *         <li>InternalServerException An error on the server occurred when trying to process a request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MachineLearningException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MachineLearningAsyncClient.DeleteRealtimeEndpoint
     */
    @Override
    public CompletableFuture<DeleteRealtimeEndpointResponse> deleteRealtimeEndpoint(
            DeleteRealtimeEndpointRequest deleteRealtimeEndpointRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteRealtimeEndpointRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRealtimeEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRealtimeEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteRealtimeEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteRealtimeEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteRealtimeEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteRealtimeEndpointRequest, DeleteRealtimeEndpointResponse>()
                            .withOperationName("DeleteRealtimeEndpoint").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteRealtimeEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteRealtimeEndpointRequest));
            CompletableFuture<DeleteRealtimeEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified tags associated with an ML object. After this operation is complete, you can't recover
     * deleted tags.
     * </p>
     * <p>
     * If you specify a tag that doesn't exist, Amazon ML ignores it.
     * </p>
     *
     * @param deleteTagsRequest
     * @return A Java Future containing the result of the DeleteTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidInputException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>InvalidTagException</li>
     *         <li>ResourceNotFoundException A specified resource cannot be located.</li>
     *         <li>InternalServerException An error on the server occurred when trying to process a request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MachineLearningException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MachineLearningAsyncClient.DeleteTags
     */
    @Override
    public CompletableFuture<DeleteTagsResponse> deleteTags(DeleteTagsRequest deleteTagsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteTagsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTags");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteTagsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteTagsRequest, DeleteTagsResponse>().withOperationName("DeleteTags")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteTagsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteTagsRequest));
            CompletableFuture<DeleteTagsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of <code>BatchPrediction</code> operations that match the search criteria in the request.
     * </p>
     *
     * @param describeBatchPredictionsRequest
     * @return A Java Future containing the result of the DescribeBatchPredictions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidInputException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>InternalServerException An error on the server occurred when trying to process a request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MachineLearningException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MachineLearningAsyncClient.DescribeBatchPredictions
     */
    @Override
    public CompletableFuture<DescribeBatchPredictionsResponse> describeBatchPredictions(
            DescribeBatchPredictionsRequest describeBatchPredictionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeBatchPredictionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeBatchPredictionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBatchPredictions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeBatchPredictionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeBatchPredictionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeBatchPredictionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeBatchPredictionsRequest, DescribeBatchPredictionsResponse>()
                            .withOperationName("DescribeBatchPredictions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeBatchPredictionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeBatchPredictionsRequest));
            CompletableFuture<DescribeBatchPredictionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of <code>DataSource</code> that match the search criteria in the request.
     * </p>
     *
     * @param describeDataSourcesRequest
     * @return A Java Future containing the result of the DescribeDataSources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidInputException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>InternalServerException An error on the server occurred when trying to process a request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MachineLearningException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MachineLearningAsyncClient.DescribeDataSources
     */
    @Override
    public CompletableFuture<DescribeDataSourcesResponse> describeDataSources(
            DescribeDataSourcesRequest describeDataSourcesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeDataSourcesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDataSourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDataSources");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDataSourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDataSourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDataSourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDataSourcesRequest, DescribeDataSourcesResponse>()
                            .withOperationName("DescribeDataSources").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeDataSourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeDataSourcesRequest));
            CompletableFuture<DescribeDataSourcesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of <code>DescribeEvaluations</code> that match the search criteria in the request.
     * </p>
     *
     * @param describeEvaluationsRequest
     * @return A Java Future containing the result of the DescribeEvaluations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidInputException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>InternalServerException An error on the server occurred when trying to process a request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MachineLearningException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MachineLearningAsyncClient.DescribeEvaluations
     */
    @Override
    public CompletableFuture<DescribeEvaluationsResponse> describeEvaluations(
            DescribeEvaluationsRequest describeEvaluationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeEvaluationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeEvaluationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEvaluations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeEvaluationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeEvaluationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeEvaluationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeEvaluationsRequest, DescribeEvaluationsResponse>()
                            .withOperationName("DescribeEvaluations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeEvaluationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeEvaluationsRequest));
            CompletableFuture<DescribeEvaluationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of <code>MLModel</code> that match the search criteria in the request.
     * </p>
     *
     * @param describeMlModelsRequest
     * @return A Java Future containing the result of the DescribeMLModels operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidInputException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>InternalServerException An error on the server occurred when trying to process a request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MachineLearningException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MachineLearningAsyncClient.DescribeMLModels
     */
    @Override
    public CompletableFuture<DescribeMlModelsResponse> describeMLModels(DescribeMlModelsRequest describeMlModelsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeMlModelsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeMlModelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeMLModels");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeMlModelsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeMlModelsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeMlModelsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeMlModelsRequest, DescribeMlModelsResponse>()
                            .withOperationName("DescribeMLModels").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeMlModelsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeMlModelsRequest));
            CompletableFuture<DescribeMlModelsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes one or more of the tags for your Amazon ML object.
     * </p>
     *
     * @param describeTagsRequest
     * @return A Java Future containing the result of the DescribeTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidInputException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>ResourceNotFoundException A specified resource cannot be located.</li>
     *         <li>InternalServerException An error on the server occurred when trying to process a request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MachineLearningException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MachineLearningAsyncClient.DescribeTags
     */
    @Override
    public CompletableFuture<DescribeTagsResponse> describeTags(DescribeTagsRequest describeTagsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeTagsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTags");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeTagsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeTagsRequest, DescribeTagsResponse>()
                            .withOperationName("DescribeTags").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeTagsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeTagsRequest));
            CompletableFuture<DescribeTagsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a <code>BatchPrediction</code> that includes detailed metadata, status, and data file information for a
     * <code>Batch Prediction</code> request.
     * </p>
     *
     * @param getBatchPredictionRequest
     * @return A Java Future containing the result of the GetBatchPrediction operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidInputException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>ResourceNotFoundException A specified resource cannot be located.</li>
     *         <li>InternalServerException An error on the server occurred when trying to process a request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MachineLearningException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MachineLearningAsyncClient.GetBatchPrediction
     */
    @Override
    public CompletableFuture<GetBatchPredictionResponse> getBatchPrediction(GetBatchPredictionRequest getBatchPredictionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getBatchPredictionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBatchPredictionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBatchPrediction");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetBatchPredictionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetBatchPredictionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetBatchPredictionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetBatchPredictionRequest, GetBatchPredictionResponse>()
                            .withOperationName("GetBatchPrediction").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetBatchPredictionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getBatchPredictionRequest));
            CompletableFuture<GetBatchPredictionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a <code>DataSource</code> that includes metadata and data file information, as well as the current status
     * of the <code>DataSource</code>.
     * </p>
     * <p>
     * <code>GetDataSource</code> provides results in normal or verbose format. The verbose format adds the schema
     * description and the list of files pointed to by the DataSource to the normal format.
     * </p>
     *
     * @param getDataSourceRequest
     * @return A Java Future containing the result of the GetDataSource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidInputException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>ResourceNotFoundException A specified resource cannot be located.</li>
     *         <li>InternalServerException An error on the server occurred when trying to process a request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MachineLearningException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MachineLearningAsyncClient.GetDataSource
     */
    @Override
    public CompletableFuture<GetDataSourceResponse> getDataSource(GetDataSourceRequest getDataSourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDataSourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDataSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDataSource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDataSourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetDataSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDataSourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDataSourceRequest, GetDataSourceResponse>()
                            .withOperationName("GetDataSource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetDataSourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getDataSourceRequest));
            CompletableFuture<GetDataSourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an <code>Evaluation</code> that includes metadata as well as the current status of the
     * <code>Evaluation</code>.
     * </p>
     *
     * @param getEvaluationRequest
     * @return A Java Future containing the result of the GetEvaluation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidInputException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>ResourceNotFoundException A specified resource cannot be located.</li>
     *         <li>InternalServerException An error on the server occurred when trying to process a request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MachineLearningException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MachineLearningAsyncClient.GetEvaluation
     */
    @Override
    public CompletableFuture<GetEvaluationResponse> getEvaluation(GetEvaluationRequest getEvaluationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getEvaluationRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEvaluationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEvaluation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetEvaluationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetEvaluationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetEvaluationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetEvaluationRequest, GetEvaluationResponse>()
                            .withOperationName("GetEvaluation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetEvaluationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getEvaluationRequest));
            CompletableFuture<GetEvaluationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an <code>MLModel</code> that includes detailed metadata, data source information, and the current status
     * of the <code>MLModel</code>.
     * </p>
     * <p>
     * <code>GetMLModel</code> provides results in normal or verbose format.
     * </p>
     *
     * @param getMlModelRequest
     * @return A Java Future containing the result of the GetMLModel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidInputException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>ResourceNotFoundException A specified resource cannot be located.</li>
     *         <li>InternalServerException An error on the server occurred when trying to process a request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MachineLearningException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MachineLearningAsyncClient.GetMLModel
     */
    @Override
    public CompletableFuture<GetMlModelResponse> getMLModel(GetMlModelRequest getMlModelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getMlModelRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMlModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMLModel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetMlModelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetMlModelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetMlModelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMlModelRequest, GetMlModelResponse>().withOperationName("GetMLModel")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetMlModelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getMlModelRequest));
            CompletableFuture<GetMlModelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Generates a prediction for the observation using the specified <code>ML Model</code>.
     * </p>
     * <p>
     * <b>Note:</b> Not all response parameters will be populated. Whether a response parameter is populated depends on
     * the type of model requested.
     * </p>
     *
     * @param predictRequest
     * @return A Java Future containing the result of the Predict operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidInputException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>ResourceNotFoundException A specified resource cannot be located.</li>
     *         <li>LimitExceededException The subscriber exceeded the maximum number of operations. This exception can
     *         occur when listing objects such as <code>DataSource</code>.</li>
     *         <li>InternalServerException An error on the server occurred when trying to process a request.</li>
     *         <li>PredictorNotMountedException The exception is thrown when a predict request is made to an unmounted
     *         <code>MLModel</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MachineLearningException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MachineLearningAsyncClient.Predict
     */
    @Override
    public CompletableFuture<PredictResponse> predict(PredictRequest predictRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(predictRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, predictRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Predict");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PredictResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PredictResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PredictResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PredictRequest, PredictResponse>().withOperationName("Predict")
                            .withProtocolMetadata(protocolMetadata).withMarshaller(new PredictRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(predictRequest));
            CompletableFuture<PredictResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the <code>BatchPredictionName</code> of a <code>BatchPrediction</code>.
     * </p>
     * <p>
     * You can use the <code>GetBatchPrediction</code> operation to view the contents of the updated data element.
     * </p>
     *
     * @param updateBatchPredictionRequest
     * @return A Java Future containing the result of the UpdateBatchPrediction operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidInputException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>ResourceNotFoundException A specified resource cannot be located.</li>
     *         <li>InternalServerException An error on the server occurred when trying to process a request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MachineLearningException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MachineLearningAsyncClient.UpdateBatchPrediction
     */
    @Override
    public CompletableFuture<UpdateBatchPredictionResponse> updateBatchPrediction(
            UpdateBatchPredictionRequest updateBatchPredictionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateBatchPredictionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateBatchPredictionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateBatchPrediction");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateBatchPredictionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateBatchPredictionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateBatchPredictionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateBatchPredictionRequest, UpdateBatchPredictionResponse>()
                            .withOperationName("UpdateBatchPrediction").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateBatchPredictionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateBatchPredictionRequest));
            CompletableFuture<UpdateBatchPredictionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the <code>DataSourceName</code> of a <code>DataSource</code>.
     * </p>
     * <p>
     * You can use the <code>GetDataSource</code> operation to view the contents of the updated data element.
     * </p>
     *
     * @param updateDataSourceRequest
     * @return A Java Future containing the result of the UpdateDataSource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidInputException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>ResourceNotFoundException A specified resource cannot be located.</li>
     *         <li>InternalServerException An error on the server occurred when trying to process a request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MachineLearningException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MachineLearningAsyncClient.UpdateDataSource
     */
    @Override
    public CompletableFuture<UpdateDataSourceResponse> updateDataSource(UpdateDataSourceRequest updateDataSourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateDataSourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDataSourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDataSource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDataSourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDataSourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDataSourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDataSourceRequest, UpdateDataSourceResponse>()
                            .withOperationName("UpdateDataSource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateDataSourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateDataSourceRequest));
            CompletableFuture<UpdateDataSourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the <code>EvaluationName</code> of an <code>Evaluation</code>.
     * </p>
     * <p>
     * You can use the <code>GetEvaluation</code> operation to view the contents of the updated data element.
     * </p>
     *
     * @param updateEvaluationRequest
     * @return A Java Future containing the result of the UpdateEvaluation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidInputException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>ResourceNotFoundException A specified resource cannot be located.</li>
     *         <li>InternalServerException An error on the server occurred when trying to process a request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MachineLearningException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MachineLearningAsyncClient.UpdateEvaluation
     */
    @Override
    public CompletableFuture<UpdateEvaluationResponse> updateEvaluation(UpdateEvaluationRequest updateEvaluationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateEvaluationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateEvaluationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateEvaluation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateEvaluationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateEvaluationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateEvaluationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateEvaluationRequest, UpdateEvaluationResponse>()
                            .withOperationName("UpdateEvaluation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateEvaluationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateEvaluationRequest));
            CompletableFuture<UpdateEvaluationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the <code>MLModelName</code> and the <code>ScoreThreshold</code> of an <code>MLModel</code>.
     * </p>
     * <p>
     * You can use the <code>GetMLModel</code> operation to view the contents of the updated data element.
     * </p>
     *
     * @param updateMlModelRequest
     * @return A Java Future containing the result of the UpdateMLModel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidInputException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>ResourceNotFoundException A specified resource cannot be located.</li>
     *         <li>InternalServerException An error on the server occurred when trying to process a request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MachineLearningException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample MachineLearningAsyncClient.UpdateMLModel
     */
    @Override
    public CompletableFuture<UpdateMlModelResponse> updateMLModel(UpdateMlModelRequest updateMlModelRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateMlModelRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateMlModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Machine Learning");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateMLModel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateMlModelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateMlModelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateMlModelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateMlModelRequest, UpdateMlModelResponse>()
                            .withOperationName("UpdateMLModel").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateMlModelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateMlModelRequest));
            CompletableFuture<UpdateMlModelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public MachineLearningAsyncWaiter waiter() {
        return MachineLearningAsyncWaiter.builder().client(this).scheduledExecutorService(executorService).build();
    }

    @Override
    public final MachineLearningServiceClientConfiguration serviceClientConfiguration() {
        return new MachineLearningServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MachineLearningException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidTagException")
                                .exceptionBuilderSupplier(InvalidTagException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidInputException")
                                .exceptionBuilderSupplier(InvalidInputException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IdempotentParameterMismatchException")
                                .exceptionBuilderSupplier(IdempotentParameterMismatchException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TagLimitExceededException")
                                .exceptionBuilderSupplier(TagLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PredictorNotMountedException")
                                .exceptionBuilderSupplier(PredictorNotMountedException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        MachineLearningServiceClientConfigurationBuilder serviceConfigBuilder = new MachineLearningServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
