/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.macie.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class DisassociateS3ResourcesRequest extends MacieRequest implements
        ToCopyableBuilder<DisassociateS3ResourcesRequest.Builder, DisassociateS3ResourcesRequest> {
    private static final SdkField<String> MEMBER_ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("memberAccountId").getter(getter(DisassociateS3ResourcesRequest::memberAccountId))
            .setter(setter(Builder::memberAccountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("memberAccountId").build()).build();

    private static final SdkField<List<S3Resource>> ASSOCIATED_S3_RESOURCES_FIELD = SdkField
            .<List<S3Resource>> builder(MarshallingType.LIST)
            .memberName("associatedS3Resources")
            .getter(getter(DisassociateS3ResourcesRequest::associatedS3Resources))
            .setter(setter(Builder::associatedS3Resources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("associatedS3Resources").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<S3Resource> builder(MarshallingType.SDK_POJO)
                                            .constructor(S3Resource::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(MEMBER_ACCOUNT_ID_FIELD,
            ASSOCIATED_S3_RESOURCES_FIELD));

    private final String memberAccountId;

    private final List<S3Resource> associatedS3Resources;

    private DisassociateS3ResourcesRequest(BuilderImpl builder) {
        super(builder);
        this.memberAccountId = builder.memberAccountId;
        this.associatedS3Resources = builder.associatedS3Resources;
    }

    /**
     * <p>
     * The ID of the Amazon Macie Classic member account whose resources you want to remove from being monitored by
     * Amazon Macie Classic.
     * </p>
     * 
     * @return The ID of the Amazon Macie Classic member account whose resources you want to remove from being monitored
     *         by Amazon Macie Classic.
     */
    public String memberAccountId() {
        return memberAccountId;
    }

    /**
     * Returns true if the AssociatedS3Resources property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasAssociatedS3Resources() {
        return associatedS3Resources != null && !(associatedS3Resources instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The S3 resources (buckets or prefixes) that you want to remove from being monitored and classified by Amazon
     * Macie Classic.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAssociatedS3Resources()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The S3 resources (buckets or prefixes) that you want to remove from being monitored and classified by
     *         Amazon Macie Classic.
     */
    public List<S3Resource> associatedS3Resources() {
        return associatedS3Resources;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(memberAccountId());
        hashCode = 31 * hashCode + Objects.hashCode(hasAssociatedS3Resources() ? associatedS3Resources() : null);
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DisassociateS3ResourcesRequest)) {
            return false;
        }
        DisassociateS3ResourcesRequest other = (DisassociateS3ResourcesRequest) obj;
        return Objects.equals(memberAccountId(), other.memberAccountId())
                && hasAssociatedS3Resources() == other.hasAssociatedS3Resources()
                && Objects.equals(associatedS3Resources(), other.associatedS3Resources());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("DisassociateS3ResourcesRequest").add("MemberAccountId", memberAccountId())
                .add("AssociatedS3Resources", hasAssociatedS3Resources() ? associatedS3Resources() : null).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "memberAccountId":
            return Optional.ofNullable(clazz.cast(memberAccountId()));
        case "associatedS3Resources":
            return Optional.ofNullable(clazz.cast(associatedS3Resources()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DisassociateS3ResourcesRequest, T> g) {
        return obj -> g.apply((DisassociateS3ResourcesRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends MacieRequest.Builder, SdkPojo, CopyableBuilder<Builder, DisassociateS3ResourcesRequest> {
        /**
         * <p>
         * The ID of the Amazon Macie Classic member account whose resources you want to remove from being monitored by
         * Amazon Macie Classic.
         * </p>
         * 
         * @param memberAccountId
         *        The ID of the Amazon Macie Classic member account whose resources you want to remove from being
         *        monitored by Amazon Macie Classic.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder memberAccountId(String memberAccountId);

        /**
         * <p>
         * The S3 resources (buckets or prefixes) that you want to remove from being monitored and classified by Amazon
         * Macie Classic.
         * </p>
         * 
         * @param associatedS3Resources
         *        The S3 resources (buckets or prefixes) that you want to remove from being monitored and classified by
         *        Amazon Macie Classic.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associatedS3Resources(Collection<S3Resource> associatedS3Resources);

        /**
         * <p>
         * The S3 resources (buckets or prefixes) that you want to remove from being monitored and classified by Amazon
         * Macie Classic.
         * </p>
         * 
         * @param associatedS3Resources
         *        The S3 resources (buckets or prefixes) that you want to remove from being monitored and classified by
         *        Amazon Macie Classic.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associatedS3Resources(S3Resource... associatedS3Resources);

        /**
         * <p>
         * The S3 resources (buckets or prefixes) that you want to remove from being monitored and classified by Amazon
         * Macie Classic.
         * </p>
         * This is a convenience that creates an instance of the {@link List<S3Resource>.Builder} avoiding the need to
         * create one manually via {@link List<S3Resource>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<S3Resource>.Builder#build()} is called immediately and its
         * result is passed to {@link #associatedS3Resources(List<S3Resource>)}.
         * 
         * @param associatedS3Resources
         *        a consumer that will call methods on {@link List<S3Resource>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #associatedS3Resources(List<S3Resource>)
         */
        Builder associatedS3Resources(Consumer<S3Resource.Builder>... associatedS3Resources);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends MacieRequest.BuilderImpl implements Builder {
        private String memberAccountId;

        private List<S3Resource> associatedS3Resources = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(DisassociateS3ResourcesRequest model) {
            super(model);
            memberAccountId(model.memberAccountId);
            associatedS3Resources(model.associatedS3Resources);
        }

        public final String getMemberAccountId() {
            return memberAccountId;
        }

        @Override
        public final Builder memberAccountId(String memberAccountId) {
            this.memberAccountId = memberAccountId;
            return this;
        }

        public final void setMemberAccountId(String memberAccountId) {
            this.memberAccountId = memberAccountId;
        }

        public final Collection<S3Resource.Builder> getAssociatedS3Resources() {
            if (associatedS3Resources instanceof SdkAutoConstructList) {
                return null;
            }
            return associatedS3Resources != null ? associatedS3Resources.stream().map(S3Resource::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder associatedS3Resources(Collection<S3Resource> associatedS3Resources) {
            this.associatedS3Resources = S3ResourcesCopier.copy(associatedS3Resources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder associatedS3Resources(S3Resource... associatedS3Resources) {
            associatedS3Resources(Arrays.asList(associatedS3Resources));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder associatedS3Resources(Consumer<S3Resource.Builder>... associatedS3Resources) {
            associatedS3Resources(Stream.of(associatedS3Resources).map(c -> S3Resource.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setAssociatedS3Resources(Collection<S3Resource.BuilderImpl> associatedS3Resources) {
            this.associatedS3Resources = S3ResourcesCopier.copyFromBuilder(associatedS3Resources);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public DisassociateS3ResourcesRequest build() {
            return new DisassociateS3ResourcesRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
