/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.macie;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.macie.model.AccessDeniedException;
import software.amazon.awssdk.services.macie.model.AssociateMemberAccountRequest;
import software.amazon.awssdk.services.macie.model.AssociateMemberAccountResponse;
import software.amazon.awssdk.services.macie.model.AssociateS3ResourcesRequest;
import software.amazon.awssdk.services.macie.model.AssociateS3ResourcesResponse;
import software.amazon.awssdk.services.macie.model.DisassociateMemberAccountRequest;
import software.amazon.awssdk.services.macie.model.DisassociateMemberAccountResponse;
import software.amazon.awssdk.services.macie.model.DisassociateS3ResourcesRequest;
import software.amazon.awssdk.services.macie.model.DisassociateS3ResourcesResponse;
import software.amazon.awssdk.services.macie.model.InternalException;
import software.amazon.awssdk.services.macie.model.InvalidInputException;
import software.amazon.awssdk.services.macie.model.LimitExceededException;
import software.amazon.awssdk.services.macie.model.ListMemberAccountsRequest;
import software.amazon.awssdk.services.macie.model.ListMemberAccountsResponse;
import software.amazon.awssdk.services.macie.model.ListS3ResourcesRequest;
import software.amazon.awssdk.services.macie.model.ListS3ResourcesResponse;
import software.amazon.awssdk.services.macie.model.MacieException;
import software.amazon.awssdk.services.macie.model.MacieRequest;
import software.amazon.awssdk.services.macie.model.UpdateS3ResourcesRequest;
import software.amazon.awssdk.services.macie.model.UpdateS3ResourcesResponse;
import software.amazon.awssdk.services.macie.paginators.ListMemberAccountsIterable;
import software.amazon.awssdk.services.macie.paginators.ListS3ResourcesIterable;
import software.amazon.awssdk.services.macie.transform.AssociateMemberAccountRequestMarshaller;
import software.amazon.awssdk.services.macie.transform.AssociateS3ResourcesRequestMarshaller;
import software.amazon.awssdk.services.macie.transform.DisassociateMemberAccountRequestMarshaller;
import software.amazon.awssdk.services.macie.transform.DisassociateS3ResourcesRequestMarshaller;
import software.amazon.awssdk.services.macie.transform.ListMemberAccountsRequestMarshaller;
import software.amazon.awssdk.services.macie.transform.ListS3ResourcesRequestMarshaller;
import software.amazon.awssdk.services.macie.transform.UpdateS3ResourcesRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link MacieClient}.
 *
 * @see MacieClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMacieClient implements MacieClient {
    private static final Logger log = Logger.loggerFor(DefaultMacieClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMacieClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * (Discontinued) Associates a specified Amazon Web Services account with Amazon Macie Classic as a member account.
     * </p>
     *
     * @param associateMemberAccountRequest
     * @return Result of the AssociateMemberAccount operation returned by the service.
     * @throws InvalidInputException
     *         (Discontinued) The request was rejected because an invalid or out-of-range value was supplied for an
     *         input parameter.
     * @throws LimitExceededException
     *         (Discontinued) The request was rejected because it attempted to create resources beyond the current
     *         Amazon Web Services account quotas. The error code describes the quota exceeded.
     * @throws InternalException
     *         (Discontinued) Internal server error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MacieException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MacieClient.AssociateMemberAccount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/macie-2017-12-19/AssociateMemberAccount" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public AssociateMemberAccountResponse associateMemberAccount(AssociateMemberAccountRequest associateMemberAccountRequest)
            throws InvalidInputException, LimitExceededException, InternalException, AwsServiceException, SdkClientException,
            MacieException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateMemberAccountResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AssociateMemberAccountResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateMemberAccountRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Macie");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateMemberAccount");

            return clientHandler
                    .execute(new ClientExecutionParams<AssociateMemberAccountRequest, AssociateMemberAccountResponse>()
                            .withOperationName("AssociateMemberAccount").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(associateMemberAccountRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new AssociateMemberAccountRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * (Discontinued) Associates specified S3 resources with Amazon Macie Classic for monitoring and data
     * classification. If <code>memberAccountId</code> isn't specified, the action associates specified S3 resources
     * with Macie Classic for the current Macie Classic administrator account. If <code>memberAccountId</code> is
     * specified, the action associates specified S3 resources with Macie Classic for the specified member account.
     * </p>
     *
     * @param associateS3ResourcesRequest
     * @return Result of the AssociateS3Resources operation returned by the service.
     * @throws InvalidInputException
     *         (Discontinued) The request was rejected because an invalid or out-of-range value was supplied for an
     *         input parameter.
     * @throws AccessDeniedException
     *         (Discontinued) You do not have required permissions to access the requested resource.
     * @throws LimitExceededException
     *         (Discontinued) The request was rejected because it attempted to create resources beyond the current
     *         Amazon Web Services account quotas. The error code describes the quota exceeded.
     * @throws InternalException
     *         (Discontinued) Internal server error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MacieException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MacieClient.AssociateS3Resources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/macie-2017-12-19/AssociateS3Resources" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public AssociateS3ResourcesResponse associateS3Resources(AssociateS3ResourcesRequest associateS3ResourcesRequest)
            throws InvalidInputException, AccessDeniedException, LimitExceededException, InternalException, AwsServiceException,
            SdkClientException, MacieException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateS3ResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AssociateS3ResourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateS3ResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Macie");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateS3Resources");

            return clientHandler.execute(new ClientExecutionParams<AssociateS3ResourcesRequest, AssociateS3ResourcesResponse>()
                    .withOperationName("AssociateS3Resources").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(associateS3ResourcesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AssociateS3ResourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * (Discontinued) Removes the specified member account from Amazon Macie Classic.
     * </p>
     *
     * @param disassociateMemberAccountRequest
     * @return Result of the DisassociateMemberAccount operation returned by the service.
     * @throws InvalidInputException
     *         (Discontinued) The request was rejected because an invalid or out-of-range value was supplied for an
     *         input parameter.
     * @throws InternalException
     *         (Discontinued) Internal server error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MacieException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MacieClient.DisassociateMemberAccount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/macie-2017-12-19/DisassociateMemberAccount"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateMemberAccountResponse disassociateMemberAccount(
            DisassociateMemberAccountRequest disassociateMemberAccountRequest) throws InvalidInputException, InternalException,
            AwsServiceException, SdkClientException, MacieException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateMemberAccountResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateMemberAccountResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateMemberAccountRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Macie");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateMemberAccount");

            return clientHandler
                    .execute(new ClientExecutionParams<DisassociateMemberAccountRequest, DisassociateMemberAccountResponse>()
                            .withOperationName("DisassociateMemberAccount").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(disassociateMemberAccountRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisassociateMemberAccountRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * (Discontinued) Removes specified S3 resources from being monitored by Amazon Macie Classic. If
     * <code>memberAccountId</code> isn't specified, the action removes specified S3 resources from Macie Classic for
     * the current Macie Classic administrator account. If <code>memberAccountId</code> is specified, the action removes
     * specified S3 resources from Macie Classic for the specified member account.
     * </p>
     *
     * @param disassociateS3ResourcesRequest
     * @return Result of the DisassociateS3Resources operation returned by the service.
     * @throws InvalidInputException
     *         (Discontinued) The request was rejected because an invalid or out-of-range value was supplied for an
     *         input parameter.
     * @throws AccessDeniedException
     *         (Discontinued) You do not have required permissions to access the requested resource.
     * @throws InternalException
     *         (Discontinued) Internal server error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MacieException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MacieClient.DisassociateS3Resources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/macie-2017-12-19/DisassociateS3Resources" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DisassociateS3ResourcesResponse disassociateS3Resources(DisassociateS3ResourcesRequest disassociateS3ResourcesRequest)
            throws InvalidInputException, AccessDeniedException, InternalException, AwsServiceException, SdkClientException,
            MacieException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateS3ResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateS3ResourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateS3ResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Macie");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateS3Resources");

            return clientHandler
                    .execute(new ClientExecutionParams<DisassociateS3ResourcesRequest, DisassociateS3ResourcesResponse>()
                            .withOperationName("DisassociateS3Resources").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(disassociateS3ResourcesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisassociateS3ResourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * (Discontinued) Lists all Amazon Macie Classic member accounts for the current Macie Classic administrator
     * account.
     * </p>
     *
     * @param listMemberAccountsRequest
     * @return Result of the ListMemberAccounts operation returned by the service.
     * @throws InternalException
     *         (Discontinued) Internal server error.
     * @throws InvalidInputException
     *         (Discontinued) The request was rejected because an invalid or out-of-range value was supplied for an
     *         input parameter.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MacieException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MacieClient.ListMemberAccounts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/macie-2017-12-19/ListMemberAccounts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListMemberAccountsResponse listMemberAccounts(ListMemberAccountsRequest listMemberAccountsRequest)
            throws InternalException, InvalidInputException, AwsServiceException, SdkClientException, MacieException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListMemberAccountsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListMemberAccountsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMemberAccountsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Macie");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMemberAccounts");

            return clientHandler.execute(new ClientExecutionParams<ListMemberAccountsRequest, ListMemberAccountsResponse>()
                    .withOperationName("ListMemberAccounts").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listMemberAccountsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListMemberAccountsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * (Discontinued) Lists all Amazon Macie Classic member accounts for the current Macie Classic administrator
     * account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listMemberAccounts(software.amazon.awssdk.services.macie.model.ListMemberAccountsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.macie.paginators.ListMemberAccountsIterable responses = client.listMemberAccountsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.macie.paginators.ListMemberAccountsIterable responses = client
     *             .listMemberAccountsPaginator(request);
     *     for (software.amazon.awssdk.services.macie.model.ListMemberAccountsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.macie.paginators.ListMemberAccountsIterable responses = client.listMemberAccountsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listMemberAccounts(software.amazon.awssdk.services.macie.model.ListMemberAccountsRequest)} operation.</b>
     * </p>
     *
     * @param listMemberAccountsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalException
     *         (Discontinued) Internal server error.
     * @throws InvalidInputException
     *         (Discontinued) The request was rejected because an invalid or out-of-range value was supplied for an
     *         input parameter.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MacieException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MacieClient.ListMemberAccounts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/macie-2017-12-19/ListMemberAccounts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListMemberAccountsIterable listMemberAccountsPaginator(ListMemberAccountsRequest listMemberAccountsRequest)
            throws InternalException, InvalidInputException, AwsServiceException, SdkClientException, MacieException {
        return new ListMemberAccountsIterable(this, applyPaginatorUserAgent(listMemberAccountsRequest));
    }

    /**
     * <p>
     * (Discontinued) Lists all the S3 resources associated with Amazon Macie Classic. If <code>memberAccountId</code>
     * isn't specified, the action lists the S3 resources associated with Macie Classic for the current Macie Classic
     * administrator account. If <code>memberAccountId</code> is specified, the action lists the S3 resources associated
     * with Macie Classic for the specified member account.
     * </p>
     *
     * @param listS3ResourcesRequest
     * @return Result of the ListS3Resources operation returned by the service.
     * @throws InvalidInputException
     *         (Discontinued) The request was rejected because an invalid or out-of-range value was supplied for an
     *         input parameter.
     * @throws AccessDeniedException
     *         (Discontinued) You do not have required permissions to access the requested resource.
     * @throws InternalException
     *         (Discontinued) Internal server error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MacieException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MacieClient.ListS3Resources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/macie-2017-12-19/ListS3Resources" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListS3ResourcesResponse listS3Resources(ListS3ResourcesRequest listS3ResourcesRequest) throws InvalidInputException,
            AccessDeniedException, InternalException, AwsServiceException, SdkClientException, MacieException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListS3ResourcesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListS3ResourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listS3ResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Macie");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListS3Resources");

            return clientHandler.execute(new ClientExecutionParams<ListS3ResourcesRequest, ListS3ResourcesResponse>()
                    .withOperationName("ListS3Resources").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listS3ResourcesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListS3ResourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * (Discontinued) Lists all the S3 resources associated with Amazon Macie Classic. If <code>memberAccountId</code>
     * isn't specified, the action lists the S3 resources associated with Macie Classic for the current Macie Classic
     * administrator account. If <code>memberAccountId</code> is specified, the action lists the S3 resources associated
     * with Macie Classic for the specified member account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listS3Resources(software.amazon.awssdk.services.macie.model.ListS3ResourcesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.macie.paginators.ListS3ResourcesIterable responses = client.listS3ResourcesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.macie.paginators.ListS3ResourcesIterable responses = client.listS3ResourcesPaginator(request);
     *     for (software.amazon.awssdk.services.macie.model.ListS3ResourcesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.macie.paginators.ListS3ResourcesIterable responses = client.listS3ResourcesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listS3Resources(software.amazon.awssdk.services.macie.model.ListS3ResourcesRequest)} operation.</b>
     * </p>
     *
     * @param listS3ResourcesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidInputException
     *         (Discontinued) The request was rejected because an invalid or out-of-range value was supplied for an
     *         input parameter.
     * @throws AccessDeniedException
     *         (Discontinued) You do not have required permissions to access the requested resource.
     * @throws InternalException
     *         (Discontinued) Internal server error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MacieException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MacieClient.ListS3Resources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/macie-2017-12-19/ListS3Resources" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListS3ResourcesIterable listS3ResourcesPaginator(ListS3ResourcesRequest listS3ResourcesRequest)
            throws InvalidInputException, AccessDeniedException, InternalException, AwsServiceException, SdkClientException,
            MacieException {
        return new ListS3ResourcesIterable(this, applyPaginatorUserAgent(listS3ResourcesRequest));
    }

    /**
     * <p>
     * (Discontinued) Updates the classification types for the specified S3 resources. If <code>memberAccountId</code>
     * isn't specified, the action updates the classification types of the S3 resources associated with Amazon Macie
     * Classic for the current Macie Classic administrator account. If <code>memberAccountId</code> is specified, the
     * action updates the classification types of the S3 resources associated with Macie Classic for the specified
     * member account.
     * </p>
     *
     * @param updateS3ResourcesRequest
     * @return Result of the UpdateS3Resources operation returned by the service.
     * @throws InvalidInputException
     *         (Discontinued) The request was rejected because an invalid or out-of-range value was supplied for an
     *         input parameter.
     * @throws AccessDeniedException
     *         (Discontinued) You do not have required permissions to access the requested resource.
     * @throws InternalException
     *         (Discontinued) Internal server error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws MacieException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample MacieClient.UpdateS3Resources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/macie-2017-12-19/UpdateS3Resources" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateS3ResourcesResponse updateS3Resources(UpdateS3ResourcesRequest updateS3ResourcesRequest)
            throws InvalidInputException, AccessDeniedException, InternalException, AwsServiceException, SdkClientException,
            MacieException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateS3ResourcesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateS3ResourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateS3ResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Macie");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateS3Resources");

            return clientHandler.execute(new ClientExecutionParams<UpdateS3ResourcesRequest, UpdateS3ResourcesResponse>()
                    .withOperationName("UpdateS3Resources").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateS3ResourcesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateS3ResourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MacieException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidInputException")
                                .exceptionBuilderSupplier(InvalidInputException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalException")
                                .exceptionBuilderSupplier(InternalException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends MacieRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
