/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.macie;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.macie.model.AccessDeniedException;
import software.amazon.awssdk.services.macie.model.AssociateMemberAccountRequest;
import software.amazon.awssdk.services.macie.model.AssociateMemberAccountResponse;
import software.amazon.awssdk.services.macie.model.AssociateS3ResourcesRequest;
import software.amazon.awssdk.services.macie.model.AssociateS3ResourcesResponse;
import software.amazon.awssdk.services.macie.model.DisassociateMemberAccountRequest;
import software.amazon.awssdk.services.macie.model.DisassociateMemberAccountResponse;
import software.amazon.awssdk.services.macie.model.DisassociateS3ResourcesRequest;
import software.amazon.awssdk.services.macie.model.DisassociateS3ResourcesResponse;
import software.amazon.awssdk.services.macie.model.InternalException;
import software.amazon.awssdk.services.macie.model.InvalidInputException;
import software.amazon.awssdk.services.macie.model.LimitExceededException;
import software.amazon.awssdk.services.macie.model.ListMemberAccountsRequest;
import software.amazon.awssdk.services.macie.model.ListMemberAccountsResponse;
import software.amazon.awssdk.services.macie.model.ListS3ResourcesRequest;
import software.amazon.awssdk.services.macie.model.ListS3ResourcesResponse;
import software.amazon.awssdk.services.macie.model.MacieException;
import software.amazon.awssdk.services.macie.model.MacieRequest;
import software.amazon.awssdk.services.macie.model.UpdateS3ResourcesRequest;
import software.amazon.awssdk.services.macie.model.UpdateS3ResourcesResponse;
import software.amazon.awssdk.services.macie.paginators.ListMemberAccountsPublisher;
import software.amazon.awssdk.services.macie.paginators.ListS3ResourcesPublisher;
import software.amazon.awssdk.services.macie.transform.AssociateMemberAccountRequestMarshaller;
import software.amazon.awssdk.services.macie.transform.AssociateS3ResourcesRequestMarshaller;
import software.amazon.awssdk.services.macie.transform.DisassociateMemberAccountRequestMarshaller;
import software.amazon.awssdk.services.macie.transform.DisassociateS3ResourcesRequestMarshaller;
import software.amazon.awssdk.services.macie.transform.ListMemberAccountsRequestMarshaller;
import software.amazon.awssdk.services.macie.transform.ListS3ResourcesRequestMarshaller;
import software.amazon.awssdk.services.macie.transform.UpdateS3ResourcesRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link MacieAsyncClient}.
 *
 * @see MacieAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMacieAsyncClient implements MacieAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultMacieAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultMacieAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Associates a specified AWS account with Amazon Macie as a member account.
     * </p>
     *
     * @param associateMemberAccountRequest
     * @return A Java Future containing the result of the AssociateMemberAccount operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The request was rejected because an invalid or out-of-range value was supplied
     *         for an input parameter.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current AWS account limits. The error code describes the limit exceeded.</li>
     *         <li>InternalException Internal server error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MacieException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MacieAsyncClient.AssociateMemberAccount
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/macie-2017-12-19/AssociateMemberAccount" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateMemberAccountResponse> associateMemberAccount(
            AssociateMemberAccountRequest associateMemberAccountRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateMemberAccountResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateMemberAccountResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateMemberAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateMemberAccountRequest, AssociateMemberAccountResponse>()
                            .withOperationName("AssociateMemberAccount")
                            .withMarshaller(new AssociateMemberAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(associateMemberAccountRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Associates specified S3 resources with Amazon Macie for monitoring and data classification. If memberAccountId
     * isn't specified, the action associates specified S3 resources with Macie for the current master account. If
     * memberAccountId is specified, the action associates specified S3 resources with Macie for the specified member
     * account.
     * </p>
     *
     * @param associateS3ResourcesRequest
     * @return A Java Future containing the result of the AssociateS3Resources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The request was rejected because an invalid or out-of-range value was supplied
     *         for an input parameter.</li>
     *         <li>AccessDeniedException You do not have required permissions to access the requested resource.</li>
     *         <li>LimitExceededException The request was rejected because it attempted to create resources beyond the
     *         current AWS account limits. The error code describes the limit exceeded.</li>
     *         <li>InternalException Internal server error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MacieException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MacieAsyncClient.AssociateS3Resources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/macie-2017-12-19/AssociateS3Resources" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateS3ResourcesResponse> associateS3Resources(
            AssociateS3ResourcesRequest associateS3ResourcesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateS3ResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateS3ResourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateS3ResourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateS3ResourcesRequest, AssociateS3ResourcesResponse>()
                            .withOperationName("AssociateS3Resources")
                            .withMarshaller(new AssociateS3ResourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(associateS3ResourcesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the specified member account from Amazon Macie.
     * </p>
     *
     * @param disassociateMemberAccountRequest
     * @return A Java Future containing the result of the DisassociateMemberAccount operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The request was rejected because an invalid or out-of-range value was supplied
     *         for an input parameter.</li>
     *         <li>InternalException Internal server error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MacieException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MacieAsyncClient.DisassociateMemberAccount
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/macie-2017-12-19/DisassociateMemberAccount"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateMemberAccountResponse> disassociateMemberAccount(
            DisassociateMemberAccountRequest disassociateMemberAccountRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateMemberAccountResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateMemberAccountResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateMemberAccountResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateMemberAccountRequest, DisassociateMemberAccountResponse>()
                            .withOperationName("DisassociateMemberAccount")
                            .withMarshaller(new DisassociateMemberAccountRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(disassociateMemberAccountRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes specified S3 resources from being monitored by Amazon Macie. If memberAccountId isn't specified, the
     * action removes specified S3 resources from Macie for the current master account. If memberAccountId is specified,
     * the action removes specified S3 resources from Macie for the specified member account.
     * </p>
     *
     * @param disassociateS3ResourcesRequest
     * @return A Java Future containing the result of the DisassociateS3Resources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The request was rejected because an invalid or out-of-range value was supplied
     *         for an input parameter.</li>
     *         <li>AccessDeniedException You do not have required permissions to access the requested resource.</li>
     *         <li>InternalException Internal server error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MacieException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MacieAsyncClient.DisassociateS3Resources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/macie-2017-12-19/DisassociateS3Resources" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateS3ResourcesResponse> disassociateS3Resources(
            DisassociateS3ResourcesRequest disassociateS3ResourcesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateS3ResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateS3ResourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateS3ResourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateS3ResourcesRequest, DisassociateS3ResourcesResponse>()
                            .withOperationName("DisassociateS3Resources")
                            .withMarshaller(new DisassociateS3ResourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(disassociateS3ResourcesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all Amazon Macie member accounts for the current Amazon Macie master account.
     * </p>
     *
     * @param listMemberAccountsRequest
     * @return A Java Future containing the result of the ListMemberAccounts operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because an invalid or out-of-range value was supplied
     *         for an input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MacieException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MacieAsyncClient.ListMemberAccounts
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/macie-2017-12-19/ListMemberAccounts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListMemberAccountsResponse> listMemberAccounts(ListMemberAccountsRequest listMemberAccountsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListMemberAccountsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListMemberAccountsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListMemberAccountsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListMemberAccountsRequest, ListMemberAccountsResponse>()
                            .withOperationName("ListMemberAccounts")
                            .withMarshaller(new ListMemberAccountsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listMemberAccountsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all Amazon Macie member accounts for the current Amazon Macie master account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listMemberAccounts(software.amazon.awssdk.services.macie.model.ListMemberAccountsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.macie.paginators.ListMemberAccountsPublisher publisher = client.listMemberAccountsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.macie.paginators.ListMemberAccountsPublisher publisher = client.listMemberAccountsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.macie.model.ListMemberAccountsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.macie.model.ListMemberAccountsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listMemberAccounts(software.amazon.awssdk.services.macie.model.ListMemberAccountsRequest)} operation.</b>
     * </p>
     *
     * @param listMemberAccountsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalException Internal server error.</li>
     *         <li>InvalidInputException The request was rejected because an invalid or out-of-range value was supplied
     *         for an input parameter.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MacieException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MacieAsyncClient.ListMemberAccounts
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/macie-2017-12-19/ListMemberAccounts" target="_top">AWS API
     *      Documentation</a>
     */
    public ListMemberAccountsPublisher listMemberAccountsPaginator(ListMemberAccountsRequest listMemberAccountsRequest) {
        return new ListMemberAccountsPublisher(this, applyPaginatorUserAgent(listMemberAccountsRequest));
    }

    /**
     * <p>
     * Lists all the S3 resources associated with Amazon Macie. If memberAccountId isn't specified, the action lists the
     * S3 resources associated with Amazon Macie for the current master account. If memberAccountId is specified, the
     * action lists the S3 resources associated with Amazon Macie for the specified member account.
     * </p>
     *
     * @param listS3ResourcesRequest
     * @return A Java Future containing the result of the ListS3Resources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The request was rejected because an invalid or out-of-range value was supplied
     *         for an input parameter.</li>
     *         <li>AccessDeniedException You do not have required permissions to access the requested resource.</li>
     *         <li>InternalException Internal server error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MacieException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MacieAsyncClient.ListS3Resources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/macie-2017-12-19/ListS3Resources" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListS3ResourcesResponse> listS3Resources(ListS3ResourcesRequest listS3ResourcesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListS3ResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListS3ResourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListS3ResourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListS3ResourcesRequest, ListS3ResourcesResponse>()
                            .withOperationName("ListS3Resources")
                            .withMarshaller(new ListS3ResourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listS3ResourcesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the S3 resources associated with Amazon Macie. If memberAccountId isn't specified, the action lists the
     * S3 resources associated with Amazon Macie for the current master account. If memberAccountId is specified, the
     * action lists the S3 resources associated with Amazon Macie for the specified member account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listS3Resources(software.amazon.awssdk.services.macie.model.ListS3ResourcesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.macie.paginators.ListS3ResourcesPublisher publisher = client.listS3ResourcesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.macie.paginators.ListS3ResourcesPublisher publisher = client.listS3ResourcesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.macie.model.ListS3ResourcesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.macie.model.ListS3ResourcesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listS3Resources(software.amazon.awssdk.services.macie.model.ListS3ResourcesRequest)} operation.</b>
     * </p>
     *
     * @param listS3ResourcesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The request was rejected because an invalid or out-of-range value was supplied
     *         for an input parameter.</li>
     *         <li>AccessDeniedException You do not have required permissions to access the requested resource.</li>
     *         <li>InternalException Internal server error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MacieException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MacieAsyncClient.ListS3Resources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/macie-2017-12-19/ListS3Resources" target="_top">AWS API
     *      Documentation</a>
     */
    public ListS3ResourcesPublisher listS3ResourcesPaginator(ListS3ResourcesRequest listS3ResourcesRequest) {
        return new ListS3ResourcesPublisher(this, applyPaginatorUserAgent(listS3ResourcesRequest));
    }

    /**
     * <p>
     * Updates the classification types for the specified S3 resources. If memberAccountId isn't specified, the action
     * updates the classification types of the S3 resources associated with Amazon Macie for the current master account.
     * If memberAccountId is specified, the action updates the classification types of the S3 resources associated with
     * Amazon Macie for the specified member account.
     * </p>
     *
     * @param updateS3ResourcesRequest
     * @return A Java Future containing the result of the UpdateS3Resources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidInputException The request was rejected because an invalid or out-of-range value was supplied
     *         for an input parameter.</li>
     *         <li>AccessDeniedException You do not have required permissions to access the requested resource.</li>
     *         <li>InternalException Internal server error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MacieException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MacieAsyncClient.UpdateS3Resources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/macie-2017-12-19/UpdateS3Resources" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateS3ResourcesResponse> updateS3Resources(UpdateS3ResourcesRequest updateS3ResourcesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateS3ResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateS3ResourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateS3ResourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateS3ResourcesRequest, UpdateS3ResourcesResponse>()
                            .withOperationName("UpdateS3Resources")
                            .withMarshaller(new UpdateS3ResourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateS3ResourcesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MacieException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidInputException")
                                .exceptionBuilderSupplier(InvalidInputException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalException")
                                .exceptionBuilderSupplier(InternalException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build());
    }

    private <T extends MacieRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
