/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.macie2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides detailed information about a sensitive data finding, including the types and number of occurrences of the
 * sensitive data that was found.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ClassificationResult implements SdkPojo, Serializable,
        ToCopyableBuilder<ClassificationResult.Builder, ClassificationResult> {
    private static final SdkField<CustomDataIdentifiers> CUSTOM_DATA_IDENTIFIERS_FIELD = SdkField
            .<CustomDataIdentifiers> builder(MarshallingType.SDK_POJO)
            .getter(getter(ClassificationResult::customDataIdentifiers)).setter(setter(Builder::customDataIdentifiers))
            .constructor(CustomDataIdentifiers::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("customDataIdentifiers").build())
            .build();

    private static final SdkField<String> MIME_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ClassificationResult::mimeType)).setter(setter(Builder::mimeType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("mimeType").build()).build();

    private static final SdkField<List<SensitiveDataItem>> SENSITIVE_DATA_FIELD = SdkField
            .<List<SensitiveDataItem>> builder(MarshallingType.LIST)
            .getter(getter(ClassificationResult::sensitiveData))
            .setter(setter(Builder::sensitiveData))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sensitiveData").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<SensitiveDataItem> builder(MarshallingType.SDK_POJO)
                                            .constructor(SensitiveDataItem::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Long> SIZE_CLASSIFIED_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .getter(getter(ClassificationResult::sizeClassified)).setter(setter(Builder::sizeClassified))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sizeClassified").build()).build();

    private static final SdkField<ClassificationResultStatus> STATUS_FIELD = SdkField
            .<ClassificationResultStatus> builder(MarshallingType.SDK_POJO).getter(getter(ClassificationResult::status))
            .setter(setter(Builder::status)).constructor(ClassificationResultStatus::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CUSTOM_DATA_IDENTIFIERS_FIELD,
            MIME_TYPE_FIELD, SENSITIVE_DATA_FIELD, SIZE_CLASSIFIED_FIELD, STATUS_FIELD));

    private static final long serialVersionUID = 1L;

    private final CustomDataIdentifiers customDataIdentifiers;

    private final String mimeType;

    private final List<SensitiveDataItem> sensitiveData;

    private final Long sizeClassified;

    private final ClassificationResultStatus status;

    private ClassificationResult(BuilderImpl builder) {
        this.customDataIdentifiers = builder.customDataIdentifiers;
        this.mimeType = builder.mimeType;
        this.sensitiveData = builder.sensitiveData;
        this.sizeClassified = builder.sizeClassified;
        this.status = builder.status;
    }

    /**
     * <p>
     * The number of occurrences of the data that produced the finding, and the custom data identifiers that detected
     * the data.
     * </p>
     * 
     * @return The number of occurrences of the data that produced the finding, and the custom data identifiers that
     *         detected the data.
     */
    public CustomDataIdentifiers customDataIdentifiers() {
        return customDataIdentifiers;
    }

    /**
     * <p>
     * The type of content, expressed as a MIME type, that the finding applies to. For example, application/gzip, for a
     * GNU Gzip compressed archive file, or application/pdf, for an Adobe PDF file.
     * </p>
     * 
     * @return The type of content, expressed as a MIME type, that the finding applies to. For example,
     *         application/gzip, for a GNU Gzip compressed archive file, or application/pdf, for an Adobe PDF file.
     */
    public String mimeType() {
        return mimeType;
    }

    /**
     * Returns true if the SensitiveData property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasSensitiveData() {
        return sensitiveData != null && !(sensitiveData instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The category and number of occurrences of the sensitive data that produced the finding.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSensitiveData()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The category and number of occurrences of the sensitive data that produced the finding.
     */
    public List<SensitiveDataItem> sensitiveData() {
        return sensitiveData;
    }

    /**
     * <p>
     * The total size, in bytes, of the data that the finding applies to.
     * </p>
     * 
     * @return The total size, in bytes, of the data that the finding applies to.
     */
    public Long sizeClassified() {
        return sizeClassified;
    }

    /**
     * <p>
     * The status of the finding.
     * </p>
     * 
     * @return The status of the finding.
     */
    public ClassificationResultStatus status() {
        return status;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(customDataIdentifiers());
        hashCode = 31 * hashCode + Objects.hashCode(mimeType());
        hashCode = 31 * hashCode + Objects.hashCode(sensitiveData());
        hashCode = 31 * hashCode + Objects.hashCode(sizeClassified());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ClassificationResult)) {
            return false;
        }
        ClassificationResult other = (ClassificationResult) obj;
        return Objects.equals(customDataIdentifiers(), other.customDataIdentifiers())
                && Objects.equals(mimeType(), other.mimeType()) && Objects.equals(sensitiveData(), other.sensitiveData())
                && Objects.equals(sizeClassified(), other.sizeClassified()) && Objects.equals(status(), other.status());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ClassificationResult").add("CustomDataIdentifiers", customDataIdentifiers())
                .add("MimeType", mimeType()).add("SensitiveData", sensitiveData()).add("SizeClassified", sizeClassified())
                .add("Status", status()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "customDataIdentifiers":
            return Optional.ofNullable(clazz.cast(customDataIdentifiers()));
        case "mimeType":
            return Optional.ofNullable(clazz.cast(mimeType()));
        case "sensitiveData":
            return Optional.ofNullable(clazz.cast(sensitiveData()));
        case "sizeClassified":
            return Optional.ofNullable(clazz.cast(sizeClassified()));
        case "status":
            return Optional.ofNullable(clazz.cast(status()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ClassificationResult, T> g) {
        return obj -> g.apply((ClassificationResult) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ClassificationResult> {
        /**
         * <p>
         * The number of occurrences of the data that produced the finding, and the custom data identifiers that
         * detected the data.
         * </p>
         * 
         * @param customDataIdentifiers
         *        The number of occurrences of the data that produced the finding, and the custom data identifiers that
         *        detected the data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customDataIdentifiers(CustomDataIdentifiers customDataIdentifiers);

        /**
         * <p>
         * The number of occurrences of the data that produced the finding, and the custom data identifiers that
         * detected the data.
         * </p>
         * This is a convenience that creates an instance of the {@link CustomDataIdentifiers.Builder} avoiding the need
         * to create one manually via {@link CustomDataIdentifiers#builder()}.
         *
         * When the {@link Consumer} completes, {@link CustomDataIdentifiers.Builder#build()} is called immediately and
         * its result is passed to {@link #customDataIdentifiers(CustomDataIdentifiers)}.
         * 
         * @param customDataIdentifiers
         *        a consumer that will call methods on {@link CustomDataIdentifiers.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #customDataIdentifiers(CustomDataIdentifiers)
         */
        default Builder customDataIdentifiers(Consumer<CustomDataIdentifiers.Builder> customDataIdentifiers) {
            return customDataIdentifiers(CustomDataIdentifiers.builder().applyMutation(customDataIdentifiers).build());
        }

        /**
         * <p>
         * The type of content, expressed as a MIME type, that the finding applies to. For example, application/gzip,
         * for a GNU Gzip compressed archive file, or application/pdf, for an Adobe PDF file.
         * </p>
         * 
         * @param mimeType
         *        The type of content, expressed as a MIME type, that the finding applies to. For example,
         *        application/gzip, for a GNU Gzip compressed archive file, or application/pdf, for an Adobe PDF file.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mimeType(String mimeType);

        /**
         * <p>
         * The category and number of occurrences of the sensitive data that produced the finding.
         * </p>
         * 
         * @param sensitiveData
         *        The category and number of occurrences of the sensitive data that produced the finding.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sensitiveData(Collection<SensitiveDataItem> sensitiveData);

        /**
         * <p>
         * The category and number of occurrences of the sensitive data that produced the finding.
         * </p>
         * 
         * @param sensitiveData
         *        The category and number of occurrences of the sensitive data that produced the finding.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sensitiveData(SensitiveDataItem... sensitiveData);

        /**
         * <p>
         * The category and number of occurrences of the sensitive data that produced the finding.
         * </p>
         * This is a convenience that creates an instance of the {@link List<SensitiveDataItem>.Builder} avoiding the
         * need to create one manually via {@link List<SensitiveDataItem>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<SensitiveDataItem>.Builder#build()} is called immediately
         * and its result is passed to {@link #sensitiveData(List<SensitiveDataItem>)}.
         * 
         * @param sensitiveData
         *        a consumer that will call methods on {@link List<SensitiveDataItem>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sensitiveData(List<SensitiveDataItem>)
         */
        Builder sensitiveData(Consumer<SensitiveDataItem.Builder>... sensitiveData);

        /**
         * <p>
         * The total size, in bytes, of the data that the finding applies to.
         * </p>
         * 
         * @param sizeClassified
         *        The total size, in bytes, of the data that the finding applies to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sizeClassified(Long sizeClassified);

        /**
         * <p>
         * The status of the finding.
         * </p>
         * 
         * @param status
         *        The status of the finding.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(ClassificationResultStatus status);

        /**
         * <p>
         * The status of the finding.
         * </p>
         * This is a convenience that creates an instance of the {@link ClassificationResultStatus.Builder} avoiding the
         * need to create one manually via {@link ClassificationResultStatus#builder()}.
         *
         * When the {@link Consumer} completes, {@link ClassificationResultStatus.Builder#build()} is called immediately
         * and its result is passed to {@link #status(ClassificationResultStatus)}.
         * 
         * @param status
         *        a consumer that will call methods on {@link ClassificationResultStatus.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #status(ClassificationResultStatus)
         */
        default Builder status(Consumer<ClassificationResultStatus.Builder> status) {
            return status(ClassificationResultStatus.builder().applyMutation(status).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private CustomDataIdentifiers customDataIdentifiers;

        private String mimeType;

        private List<SensitiveDataItem> sensitiveData = DefaultSdkAutoConstructList.getInstance();

        private Long sizeClassified;

        private ClassificationResultStatus status;

        private BuilderImpl() {
        }

        private BuilderImpl(ClassificationResult model) {
            customDataIdentifiers(model.customDataIdentifiers);
            mimeType(model.mimeType);
            sensitiveData(model.sensitiveData);
            sizeClassified(model.sizeClassified);
            status(model.status);
        }

        public final CustomDataIdentifiers.Builder getCustomDataIdentifiers() {
            return customDataIdentifiers != null ? customDataIdentifiers.toBuilder() : null;
        }

        @Override
        public final Builder customDataIdentifiers(CustomDataIdentifiers customDataIdentifiers) {
            this.customDataIdentifiers = customDataIdentifiers;
            return this;
        }

        public final void setCustomDataIdentifiers(CustomDataIdentifiers.BuilderImpl customDataIdentifiers) {
            this.customDataIdentifiers = customDataIdentifiers != null ? customDataIdentifiers.build() : null;
        }

        public final String getMimeType() {
            return mimeType;
        }

        @Override
        public final Builder mimeType(String mimeType) {
            this.mimeType = mimeType;
            return this;
        }

        public final void setMimeType(String mimeType) {
            this.mimeType = mimeType;
        }

        public final Collection<SensitiveDataItem.Builder> getSensitiveData() {
            return sensitiveData != null ? sensitiveData.stream().map(SensitiveDataItem::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder sensitiveData(Collection<SensitiveDataItem> sensitiveData) {
            this.sensitiveData = SensitiveDataCopier.copy(sensitiveData);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sensitiveData(SensitiveDataItem... sensitiveData) {
            sensitiveData(Arrays.asList(sensitiveData));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sensitiveData(Consumer<SensitiveDataItem.Builder>... sensitiveData) {
            sensitiveData(Stream.of(sensitiveData).map(c -> SensitiveDataItem.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setSensitiveData(Collection<SensitiveDataItem.BuilderImpl> sensitiveData) {
            this.sensitiveData = SensitiveDataCopier.copyFromBuilder(sensitiveData);
        }

        public final Long getSizeClassified() {
            return sizeClassified;
        }

        @Override
        public final Builder sizeClassified(Long sizeClassified) {
            this.sizeClassified = sizeClassified;
            return this;
        }

        public final void setSizeClassified(Long sizeClassified) {
            this.sizeClassified = sizeClassified;
        }

        public final ClassificationResultStatus.Builder getStatus() {
            return status != null ? status.toBuilder() : null;
        }

        @Override
        public final Builder status(ClassificationResultStatus status) {
            this.status = status;
            return this;
        }

        public final void setStatus(ClassificationResultStatus.BuilderImpl status) {
            this.status = status != null ? status.build() : null;
        }

        @Override
        public ClassificationResult build() {
            return new ClassificationResult(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
