/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.macie2.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides information about an S3 bucket that a finding applies to.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class S3Bucket implements SdkPojo, Serializable, ToCopyableBuilder<S3Bucket.Builder, S3Bucket> {
    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(S3Bucket::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(S3Bucket::createdAt))
            .setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<ServerSideEncryption> DEFAULT_SERVER_SIDE_ENCRYPTION_FIELD = SdkField
            .<ServerSideEncryption> builder(MarshallingType.SDK_POJO)
            .getter(getter(S3Bucket::defaultServerSideEncryption))
            .setter(setter(Builder::defaultServerSideEncryption))
            .constructor(ServerSideEncryption::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("defaultServerSideEncryption")
                    .build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(S3Bucket::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<S3BucketOwner> OWNER_FIELD = SdkField.<S3BucketOwner> builder(MarshallingType.SDK_POJO)
            .getter(getter(S3Bucket::owner)).setter(setter(Builder::owner)).constructor(S3BucketOwner::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("owner").build()).build();

    private static final SdkField<BucketPublicAccess> PUBLIC_ACCESS_FIELD = SdkField
            .<BucketPublicAccess> builder(MarshallingType.SDK_POJO).getter(getter(S3Bucket::publicAccess))
            .setter(setter(Builder::publicAccess)).constructor(BucketPublicAccess::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("publicAccess").build()).build();

    private static final SdkField<List<KeyValuePair>> TAGS_FIELD = SdkField
            .<List<KeyValuePair>> builder(MarshallingType.LIST)
            .getter(getter(S3Bucket::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<KeyValuePair> builder(MarshallingType.SDK_POJO)
                                            .constructor(KeyValuePair::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ARN_FIELD, CREATED_AT_FIELD,
            DEFAULT_SERVER_SIDE_ENCRYPTION_FIELD, NAME_FIELD, OWNER_FIELD, PUBLIC_ACCESS_FIELD, TAGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String arn;

    private final Instant createdAt;

    private final ServerSideEncryption defaultServerSideEncryption;

    private final String name;

    private final S3BucketOwner owner;

    private final BucketPublicAccess publicAccess;

    private final List<KeyValuePair> tags;

    private S3Bucket(BuilderImpl builder) {
        this.arn = builder.arn;
        this.createdAt = builder.createdAt;
        this.defaultServerSideEncryption = builder.defaultServerSideEncryption;
        this.name = builder.name;
        this.owner = builder.owner;
        this.publicAccess = builder.publicAccess;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the bucket.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the bucket.
     */
    public String arn() {
        return arn;
    }

    /**
     * <p>
     * The date and time, in UTC and extended ISO 8601 format, when the bucket was created.
     * </p>
     * 
     * @return The date and time, in UTC and extended ISO 8601 format, when the bucket was created.
     */
    public Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The server-side encryption settings for the bucket.
     * </p>
     * 
     * @return The server-side encryption settings for the bucket.
     */
    public ServerSideEncryption defaultServerSideEncryption() {
        return defaultServerSideEncryption;
    }

    /**
     * <p>
     * The name of the bucket.
     * </p>
     * 
     * @return The name of the bucket.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The display name and account identifier for the user who owns the bucket.
     * </p>
     * 
     * @return The display name and account identifier for the user who owns the bucket.
     */
    public S3BucketOwner owner() {
        return owner;
    }

    /**
     * <p>
     * The permissions settings that determine whether the bucket is publicly accessible.
     * </p>
     * 
     * @return The permissions settings that determine whether the bucket is publicly accessible.
     */
    public BucketPublicAccess publicAccess() {
        return publicAccess;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags that are associated with the bucket.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The tags that are associated with the bucket.
     */
    public List<KeyValuePair> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(defaultServerSideEncryption());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(owner());
        hashCode = 31 * hashCode + Objects.hashCode(publicAccess());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof S3Bucket)) {
            return false;
        }
        S3Bucket other = (S3Bucket) obj;
        return Objects.equals(arn(), other.arn()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(defaultServerSideEncryption(), other.defaultServerSideEncryption())
                && Objects.equals(name(), other.name()) && Objects.equals(owner(), other.owner())
                && Objects.equals(publicAccess(), other.publicAccess()) && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("S3Bucket").add("Arn", arn()).add("CreatedAt", createdAt())
                .add("DefaultServerSideEncryption", defaultServerSideEncryption()).add("Name", name()).add("Owner", owner())
                .add("PublicAccess", publicAccess()).add("Tags", tags()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "defaultServerSideEncryption":
            return Optional.ofNullable(clazz.cast(defaultServerSideEncryption()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "owner":
            return Optional.ofNullable(clazz.cast(owner()));
        case "publicAccess":
            return Optional.ofNullable(clazz.cast(publicAccess()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<S3Bucket, T> g) {
        return obj -> g.apply((S3Bucket) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, S3Bucket> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the bucket.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The date and time, in UTC and extended ISO 8601 format, when the bucket was created.
         * </p>
         * 
         * @param createdAt
         *        The date and time, in UTC and extended ISO 8601 format, when the bucket was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The server-side encryption settings for the bucket.
         * </p>
         * 
         * @param defaultServerSideEncryption
         *        The server-side encryption settings for the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultServerSideEncryption(ServerSideEncryption defaultServerSideEncryption);

        /**
         * <p>
         * The server-side encryption settings for the bucket.
         * </p>
         * This is a convenience that creates an instance of the {@link ServerSideEncryption.Builder} avoiding the need
         * to create one manually via {@link ServerSideEncryption#builder()}.
         *
         * When the {@link Consumer} completes, {@link ServerSideEncryption.Builder#build()} is called immediately and
         * its result is passed to {@link #defaultServerSideEncryption(ServerSideEncryption)}.
         * 
         * @param defaultServerSideEncryption
         *        a consumer that will call methods on {@link ServerSideEncryption.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #defaultServerSideEncryption(ServerSideEncryption)
         */
        default Builder defaultServerSideEncryption(Consumer<ServerSideEncryption.Builder> defaultServerSideEncryption) {
            return defaultServerSideEncryption(ServerSideEncryption.builder().applyMutation(defaultServerSideEncryption).build());
        }

        /**
         * <p>
         * The name of the bucket.
         * </p>
         * 
         * @param name
         *        The name of the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The display name and account identifier for the user who owns the bucket.
         * </p>
         * 
         * @param owner
         *        The display name and account identifier for the user who owns the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder owner(S3BucketOwner owner);

        /**
         * <p>
         * The display name and account identifier for the user who owns the bucket.
         * </p>
         * This is a convenience that creates an instance of the {@link S3BucketOwner.Builder} avoiding the need to
         * create one manually via {@link S3BucketOwner#builder()}.
         *
         * When the {@link Consumer} completes, {@link S3BucketOwner.Builder#build()} is called immediately and its
         * result is passed to {@link #owner(S3BucketOwner)}.
         * 
         * @param owner
         *        a consumer that will call methods on {@link S3BucketOwner.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #owner(S3BucketOwner)
         */
        default Builder owner(Consumer<S3BucketOwner.Builder> owner) {
            return owner(S3BucketOwner.builder().applyMutation(owner).build());
        }

        /**
         * <p>
         * The permissions settings that determine whether the bucket is publicly accessible.
         * </p>
         * 
         * @param publicAccess
         *        The permissions settings that determine whether the bucket is publicly accessible.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder publicAccess(BucketPublicAccess publicAccess);

        /**
         * <p>
         * The permissions settings that determine whether the bucket is publicly accessible.
         * </p>
         * This is a convenience that creates an instance of the {@link BucketPublicAccess.Builder} avoiding the need to
         * create one manually via {@link BucketPublicAccess#builder()}.
         *
         * When the {@link Consumer} completes, {@link BucketPublicAccess.Builder#build()} is called immediately and its
         * result is passed to {@link #publicAccess(BucketPublicAccess)}.
         * 
         * @param publicAccess
         *        a consumer that will call methods on {@link BucketPublicAccess.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #publicAccess(BucketPublicAccess)
         */
        default Builder publicAccess(Consumer<BucketPublicAccess.Builder> publicAccess) {
            return publicAccess(BucketPublicAccess.builder().applyMutation(publicAccess).build());
        }

        /**
         * <p>
         * The tags that are associated with the bucket.
         * </p>
         * 
         * @param tags
         *        The tags that are associated with the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<KeyValuePair> tags);

        /**
         * <p>
         * The tags that are associated with the bucket.
         * </p>
         * 
         * @param tags
         *        The tags that are associated with the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(KeyValuePair... tags);

        /**
         * <p>
         * The tags that are associated with the bucket.
         * </p>
         * This is a convenience that creates an instance of the {@link List<KeyValuePair>.Builder} avoiding the need to
         * create one manually via {@link List<KeyValuePair>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<KeyValuePair>.Builder#build()} is called immediately and its
         * result is passed to {@link #tags(List<KeyValuePair>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<KeyValuePair>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<KeyValuePair>)
         */
        Builder tags(Consumer<KeyValuePair.Builder>... tags);
    }

    static final class BuilderImpl implements Builder {
        private String arn;

        private Instant createdAt;

        private ServerSideEncryption defaultServerSideEncryption;

        private String name;

        private S3BucketOwner owner;

        private BucketPublicAccess publicAccess;

        private List<KeyValuePair> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(S3Bucket model) {
            arn(model.arn);
            createdAt(model.createdAt);
            defaultServerSideEncryption(model.defaultServerSideEncryption);
            name(model.name);
            owner(model.owner);
            publicAccess(model.publicAccess);
            tags(model.tags);
        }

        public final String getArn() {
            return arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        public final ServerSideEncryption.Builder getDefaultServerSideEncryption() {
            return defaultServerSideEncryption != null ? defaultServerSideEncryption.toBuilder() : null;
        }

        @Override
        public final Builder defaultServerSideEncryption(ServerSideEncryption defaultServerSideEncryption) {
            this.defaultServerSideEncryption = defaultServerSideEncryption;
            return this;
        }

        public final void setDefaultServerSideEncryption(ServerSideEncryption.BuilderImpl defaultServerSideEncryption) {
            this.defaultServerSideEncryption = defaultServerSideEncryption != null ? defaultServerSideEncryption.build() : null;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final S3BucketOwner.Builder getOwner() {
            return owner != null ? owner.toBuilder() : null;
        }

        @Override
        public final Builder owner(S3BucketOwner owner) {
            this.owner = owner;
            return this;
        }

        public final void setOwner(S3BucketOwner.BuilderImpl owner) {
            this.owner = owner != null ? owner.build() : null;
        }

        public final BucketPublicAccess.Builder getPublicAccess() {
            return publicAccess != null ? publicAccess.toBuilder() : null;
        }

        @Override
        public final Builder publicAccess(BucketPublicAccess publicAccess) {
            this.publicAccess = publicAccess;
            return this;
        }

        public final void setPublicAccess(BucketPublicAccess.BuilderImpl publicAccess) {
            this.publicAccess = publicAccess != null ? publicAccess.build() : null;
        }

        public final Collection<KeyValuePair.Builder> getTags() {
            return tags != null ? tags.stream().map(KeyValuePair::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<KeyValuePair> tags) {
            this.tags = KeyValuePairListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(KeyValuePair... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<KeyValuePair.Builder>... tags) {
            tags(Stream.of(tags).map(c -> KeyValuePair.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<KeyValuePair.BuilderImpl> tags) {
            this.tags = KeyValuePairListCopier.copyFromBuilder(tags);
        }

        @Override
        public S3Bucket build() {
            return new S3Bucket(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
