/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.macie2.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides information about an S3 object that a finding applies to.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class S3Object implements SdkPojo, Serializable, ToCopyableBuilder<S3Object.Builder, S3Object> {
    private static final SdkField<String> BUCKET_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(S3Object::bucketArn)).setter(setter(Builder::bucketArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("bucketArn").build()).build();

    private static final SdkField<String> E_TAG_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(S3Object::eTag)).setter(setter(Builder::eTag))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("eTag").build()).build();

    private static final SdkField<String> EXTENSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(S3Object::extension)).setter(setter(Builder::extension))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("extension").build()).build();

    private static final SdkField<String> KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(S3Object::key)).setter(setter(Builder::key))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("key").build()).build();

    private static final SdkField<Instant> LAST_MODIFIED_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(S3Object::lastModified))
            .setter(setter(Builder::lastModified))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastModified").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(S3Object::path)).setter(setter(Builder::path))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("path").build()).build();

    private static final SdkField<Boolean> PUBLIC_ACCESS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(S3Object::publicAccess)).setter(setter(Builder::publicAccess))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("publicAccess").build()).build();

    private static final SdkField<ServerSideEncryption> SERVER_SIDE_ENCRYPTION_FIELD = SdkField
            .<ServerSideEncryption> builder(MarshallingType.SDK_POJO).getter(getter(S3Object::serverSideEncryption))
            .setter(setter(Builder::serverSideEncryption)).constructor(ServerSideEncryption::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("serverSideEncryption").build())
            .build();

    private static final SdkField<Long> SIZE_FIELD = SdkField.<Long> builder(MarshallingType.LONG).getter(getter(S3Object::size))
            .setter(setter(Builder::size))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("size").build()).build();

    private static final SdkField<String> STORAGE_CLASS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(S3Object::storageClassAsString)).setter(setter(Builder::storageClass))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("storageClass").build()).build();

    private static final SdkField<List<KeyValuePair>> TAGS_FIELD = SdkField
            .<List<KeyValuePair>> builder(MarshallingType.LIST)
            .getter(getter(S3Object::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<KeyValuePair> builder(MarshallingType.SDK_POJO)
                                            .constructor(KeyValuePair::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> VERSION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(S3Object::versionId)).setter(setter(Builder::versionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("versionId").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BUCKET_ARN_FIELD, E_TAG_FIELD,
            EXTENSION_FIELD, KEY_FIELD, LAST_MODIFIED_FIELD, PATH_FIELD, PUBLIC_ACCESS_FIELD, SERVER_SIDE_ENCRYPTION_FIELD,
            SIZE_FIELD, STORAGE_CLASS_FIELD, TAGS_FIELD, VERSION_ID_FIELD));

    private static final long serialVersionUID = 1L;

    private final String bucketArn;

    private final String eTag;

    private final String extension;

    private final String key;

    private final Instant lastModified;

    private final String path;

    private final Boolean publicAccess;

    private final ServerSideEncryption serverSideEncryption;

    private final Long size;

    private final String storageClass;

    private final List<KeyValuePair> tags;

    private final String versionId;

    private S3Object(BuilderImpl builder) {
        this.bucketArn = builder.bucketArn;
        this.eTag = builder.eTag;
        this.extension = builder.extension;
        this.key = builder.key;
        this.lastModified = builder.lastModified;
        this.path = builder.path;
        this.publicAccess = builder.publicAccess;
        this.serverSideEncryption = builder.serverSideEncryption;
        this.size = builder.size;
        this.storageClass = builder.storageClass;
        this.tags = builder.tags;
        this.versionId = builder.versionId;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the bucket that contains the object.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the bucket that contains the object.
     */
    public String bucketArn() {
        return bucketArn;
    }

    /**
     * <p>
     * The entity tag (ETag) that identifies the affected version of the object. If the object was overwritten or
     * changed after Amazon Macie produced the finding, this value might be different from the current ETag for the
     * object.
     * </p>
     * 
     * @return The entity tag (ETag) that identifies the affected version of the object. If the object was overwritten
     *         or changed after Amazon Macie produced the finding, this value might be different from the current ETag
     *         for the object.
     */
    public String eTag() {
        return eTag;
    }

    /**
     * <p>
     * The file extension of the object. If the object doesn't have a file extension, this value is "".
     * </p>
     * 
     * @return The file extension of the object. If the object doesn't have a file extension, this value is "".
     */
    public String extension() {
        return extension;
    }

    /**
     * <p>
     * The full key (name) that's assigned to the object.
     * </p>
     * 
     * @return The full key (name) that's assigned to the object.
     */
    public String key() {
        return key;
    }

    /**
     * <p>
     * The date and time, in UTC and extended ISO 8601 format, when the object was last modified.
     * </p>
     * 
     * @return The date and time, in UTC and extended ISO 8601 format, when the object was last modified.
     */
    public Instant lastModified() {
        return lastModified;
    }

    /**
     * <p>
     * The path to the object, including the full key (name).
     * </p>
     * 
     * @return The path to the object, including the full key (name).
     */
    public String path() {
        return path;
    }

    /**
     * <p>
     * Specifies whether the object is publicly accessible due to the combination of permissions settings that apply to
     * the object.
     * </p>
     * 
     * @return Specifies whether the object is publicly accessible due to the combination of permissions settings that
     *         apply to the object.
     */
    public Boolean publicAccess() {
        return publicAccess;
    }

    /**
     * <p>
     * The server-side encryption settings for the object.
     * </p>
     * 
     * @return The server-side encryption settings for the object.
     */
    public ServerSideEncryption serverSideEncryption() {
        return serverSideEncryption;
    }

    /**
     * <p>
     * The total storage size, in bytes, of the object.
     * </p>
     * 
     * @return The total storage size, in bytes, of the object.
     */
    public Long size() {
        return size;
    }

    /**
     * <p>
     * The storage class of the object.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #storageClass} will
     * return {@link StorageClass#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #storageClassAsString}.
     * </p>
     * 
     * @return The storage class of the object.
     * @see StorageClass
     */
    public StorageClass storageClass() {
        return StorageClass.fromValue(storageClass);
    }

    /**
     * <p>
     * The storage class of the object.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #storageClass} will
     * return {@link StorageClass#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #storageClassAsString}.
     * </p>
     * 
     * @return The storage class of the object.
     * @see StorageClass
     */
    public String storageClassAsString() {
        return storageClass;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags that are associated with the object.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The tags that are associated with the object.
     */
    public List<KeyValuePair> tags() {
        return tags;
    }

    /**
     * <p>
     * The identifier for the affected version of the object.
     * </p>
     * 
     * @return The identifier for the affected version of the object.
     */
    public String versionId() {
        return versionId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(bucketArn());
        hashCode = 31 * hashCode + Objects.hashCode(eTag());
        hashCode = 31 * hashCode + Objects.hashCode(extension());
        hashCode = 31 * hashCode + Objects.hashCode(key());
        hashCode = 31 * hashCode + Objects.hashCode(lastModified());
        hashCode = 31 * hashCode + Objects.hashCode(path());
        hashCode = 31 * hashCode + Objects.hashCode(publicAccess());
        hashCode = 31 * hashCode + Objects.hashCode(serverSideEncryption());
        hashCode = 31 * hashCode + Objects.hashCode(size());
        hashCode = 31 * hashCode + Objects.hashCode(storageClassAsString());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        hashCode = 31 * hashCode + Objects.hashCode(versionId());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof S3Object)) {
            return false;
        }
        S3Object other = (S3Object) obj;
        return Objects.equals(bucketArn(), other.bucketArn()) && Objects.equals(eTag(), other.eTag())
                && Objects.equals(extension(), other.extension()) && Objects.equals(key(), other.key())
                && Objects.equals(lastModified(), other.lastModified()) && Objects.equals(path(), other.path())
                && Objects.equals(publicAccess(), other.publicAccess())
                && Objects.equals(serverSideEncryption(), other.serverSideEncryption()) && Objects.equals(size(), other.size())
                && Objects.equals(storageClassAsString(), other.storageClassAsString()) && Objects.equals(tags(), other.tags())
                && Objects.equals(versionId(), other.versionId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("S3Object").add("BucketArn", bucketArn()).add("ETag", eTag()).add("Extension", extension())
                .add("Key", key()).add("LastModified", lastModified()).add("Path", path()).add("PublicAccess", publicAccess())
                .add("ServerSideEncryption", serverSideEncryption()).add("Size", size())
                .add("StorageClass", storageClassAsString()).add("Tags", tags()).add("VersionId", versionId()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "bucketArn":
            return Optional.ofNullable(clazz.cast(bucketArn()));
        case "eTag":
            return Optional.ofNullable(clazz.cast(eTag()));
        case "extension":
            return Optional.ofNullable(clazz.cast(extension()));
        case "key":
            return Optional.ofNullable(clazz.cast(key()));
        case "lastModified":
            return Optional.ofNullable(clazz.cast(lastModified()));
        case "path":
            return Optional.ofNullable(clazz.cast(path()));
        case "publicAccess":
            return Optional.ofNullable(clazz.cast(publicAccess()));
        case "serverSideEncryption":
            return Optional.ofNullable(clazz.cast(serverSideEncryption()));
        case "size":
            return Optional.ofNullable(clazz.cast(size()));
        case "storageClass":
            return Optional.ofNullable(clazz.cast(storageClassAsString()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "versionId":
            return Optional.ofNullable(clazz.cast(versionId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<S3Object, T> g) {
        return obj -> g.apply((S3Object) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, S3Object> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the bucket that contains the object.
         * </p>
         * 
         * @param bucketArn
         *        The Amazon Resource Name (ARN) of the bucket that contains the object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucketArn(String bucketArn);

        /**
         * <p>
         * The entity tag (ETag) that identifies the affected version of the object. If the object was overwritten or
         * changed after Amazon Macie produced the finding, this value might be different from the current ETag for the
         * object.
         * </p>
         * 
         * @param eTag
         *        The entity tag (ETag) that identifies the affected version of the object. If the object was
         *        overwritten or changed after Amazon Macie produced the finding, this value might be different from the
         *        current ETag for the object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eTag(String eTag);

        /**
         * <p>
         * The file extension of the object. If the object doesn't have a file extension, this value is "".
         * </p>
         * 
         * @param extension
         *        The file extension of the object. If the object doesn't have a file extension, this value is "".
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder extension(String extension);

        /**
         * <p>
         * The full key (name) that's assigned to the object.
         * </p>
         * 
         * @param key
         *        The full key (name) that's assigned to the object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder key(String key);

        /**
         * <p>
         * The date and time, in UTC and extended ISO 8601 format, when the object was last modified.
         * </p>
         * 
         * @param lastModified
         *        The date and time, in UTC and extended ISO 8601 format, when the object was last modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModified(Instant lastModified);

        /**
         * <p>
         * The path to the object, including the full key (name).
         * </p>
         * 
         * @param path
         *        The path to the object, including the full key (name).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder path(String path);

        /**
         * <p>
         * Specifies whether the object is publicly accessible due to the combination of permissions settings that apply
         * to the object.
         * </p>
         * 
         * @param publicAccess
         *        Specifies whether the object is publicly accessible due to the combination of permissions settings
         *        that apply to the object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder publicAccess(Boolean publicAccess);

        /**
         * <p>
         * The server-side encryption settings for the object.
         * </p>
         * 
         * @param serverSideEncryption
         *        The server-side encryption settings for the object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serverSideEncryption(ServerSideEncryption serverSideEncryption);

        /**
         * <p>
         * The server-side encryption settings for the object.
         * </p>
         * This is a convenience that creates an instance of the {@link ServerSideEncryption.Builder} avoiding the need
         * to create one manually via {@link ServerSideEncryption#builder()}.
         *
         * When the {@link Consumer} completes, {@link ServerSideEncryption.Builder#build()} is called immediately and
         * its result is passed to {@link #serverSideEncryption(ServerSideEncryption)}.
         * 
         * @param serverSideEncryption
         *        a consumer that will call methods on {@link ServerSideEncryption.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #serverSideEncryption(ServerSideEncryption)
         */
        default Builder serverSideEncryption(Consumer<ServerSideEncryption.Builder> serverSideEncryption) {
            return serverSideEncryption(ServerSideEncryption.builder().applyMutation(serverSideEncryption).build());
        }

        /**
         * <p>
         * The total storage size, in bytes, of the object.
         * </p>
         * 
         * @param size
         *        The total storage size, in bytes, of the object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder size(Long size);

        /**
         * <p>
         * The storage class of the object.
         * </p>
         * 
         * @param storageClass
         *        The storage class of the object.
         * @see StorageClass
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StorageClass
         */
        Builder storageClass(String storageClass);

        /**
         * <p>
         * The storage class of the object.
         * </p>
         * 
         * @param storageClass
         *        The storage class of the object.
         * @see StorageClass
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StorageClass
         */
        Builder storageClass(StorageClass storageClass);

        /**
         * <p>
         * The tags that are associated with the object.
         * </p>
         * 
         * @param tags
         *        The tags that are associated with the object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<KeyValuePair> tags);

        /**
         * <p>
         * The tags that are associated with the object.
         * </p>
         * 
         * @param tags
         *        The tags that are associated with the object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(KeyValuePair... tags);

        /**
         * <p>
         * The tags that are associated with the object.
         * </p>
         * This is a convenience that creates an instance of the {@link List<KeyValuePair>.Builder} avoiding the need to
         * create one manually via {@link List<KeyValuePair>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<KeyValuePair>.Builder#build()} is called immediately and its
         * result is passed to {@link #tags(List<KeyValuePair>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<KeyValuePair>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<KeyValuePair>)
         */
        Builder tags(Consumer<KeyValuePair.Builder>... tags);

        /**
         * <p>
         * The identifier for the affected version of the object.
         * </p>
         * 
         * @param versionId
         *        The identifier for the affected version of the object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionId(String versionId);
    }

    static final class BuilderImpl implements Builder {
        private String bucketArn;

        private String eTag;

        private String extension;

        private String key;

        private Instant lastModified;

        private String path;

        private Boolean publicAccess;

        private ServerSideEncryption serverSideEncryption;

        private Long size;

        private String storageClass;

        private List<KeyValuePair> tags = DefaultSdkAutoConstructList.getInstance();

        private String versionId;

        private BuilderImpl() {
        }

        private BuilderImpl(S3Object model) {
            bucketArn(model.bucketArn);
            eTag(model.eTag);
            extension(model.extension);
            key(model.key);
            lastModified(model.lastModified);
            path(model.path);
            publicAccess(model.publicAccess);
            serverSideEncryption(model.serverSideEncryption);
            size(model.size);
            storageClass(model.storageClass);
            tags(model.tags);
            versionId(model.versionId);
        }

        public final String getBucketArn() {
            return bucketArn;
        }

        @Override
        public final Builder bucketArn(String bucketArn) {
            this.bucketArn = bucketArn;
            return this;
        }

        public final void setBucketArn(String bucketArn) {
            this.bucketArn = bucketArn;
        }

        public final String getETag() {
            return eTag;
        }

        @Override
        public final Builder eTag(String eTag) {
            this.eTag = eTag;
            return this;
        }

        public final void setETag(String eTag) {
            this.eTag = eTag;
        }

        public final String getExtension() {
            return extension;
        }

        @Override
        public final Builder extension(String extension) {
            this.extension = extension;
            return this;
        }

        public final void setExtension(String extension) {
            this.extension = extension;
        }

        public final String getKey() {
            return key;
        }

        @Override
        public final Builder key(String key) {
            this.key = key;
            return this;
        }

        public final void setKey(String key) {
            this.key = key;
        }

        public final Instant getLastModified() {
            return lastModified;
        }

        @Override
        public final Builder lastModified(Instant lastModified) {
            this.lastModified = lastModified;
            return this;
        }

        public final void setLastModified(Instant lastModified) {
            this.lastModified = lastModified;
        }

        public final String getPath() {
            return path;
        }

        @Override
        public final Builder path(String path) {
            this.path = path;
            return this;
        }

        public final void setPath(String path) {
            this.path = path;
        }

        public final Boolean getPublicAccess() {
            return publicAccess;
        }

        @Override
        public final Builder publicAccess(Boolean publicAccess) {
            this.publicAccess = publicAccess;
            return this;
        }

        public final void setPublicAccess(Boolean publicAccess) {
            this.publicAccess = publicAccess;
        }

        public final ServerSideEncryption.Builder getServerSideEncryption() {
            return serverSideEncryption != null ? serverSideEncryption.toBuilder() : null;
        }

        @Override
        public final Builder serverSideEncryption(ServerSideEncryption serverSideEncryption) {
            this.serverSideEncryption = serverSideEncryption;
            return this;
        }

        public final void setServerSideEncryption(ServerSideEncryption.BuilderImpl serverSideEncryption) {
            this.serverSideEncryption = serverSideEncryption != null ? serverSideEncryption.build() : null;
        }

        public final Long getSize() {
            return size;
        }

        @Override
        public final Builder size(Long size) {
            this.size = size;
            return this;
        }

        public final void setSize(Long size) {
            this.size = size;
        }

        public final String getStorageClass() {
            return storageClass;
        }

        @Override
        public final Builder storageClass(String storageClass) {
            this.storageClass = storageClass;
            return this;
        }

        @Override
        public final Builder storageClass(StorageClass storageClass) {
            this.storageClass(storageClass == null ? null : storageClass.toString());
            return this;
        }

        public final void setStorageClass(String storageClass) {
            this.storageClass = storageClass;
        }

        public final Collection<KeyValuePair.Builder> getTags() {
            return tags != null ? tags.stream().map(KeyValuePair::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<KeyValuePair> tags) {
            this.tags = KeyValuePairListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(KeyValuePair... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<KeyValuePair.Builder>... tags) {
            tags(Stream.of(tags).map(c -> KeyValuePair.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<KeyValuePair.BuilderImpl> tags) {
            this.tags = KeyValuePairListCopier.copyFromBuilder(tags);
        }

        public final String getVersionId() {
            return versionId;
        }

        @Override
        public final Builder versionId(String versionId) {
            this.versionId = versionId;
            return this;
        }

        public final void setVersionId(String versionId) {
            this.versionId = versionId;
        }

        @Override
        public S3Object build() {
            return new S3Object(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
