/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.macie2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides information about the category, types, and occurrences of sensitive data that produced a sensitive data
 * finding.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SensitiveDataItem implements SdkPojo, Serializable,
        ToCopyableBuilder<SensitiveDataItem.Builder, SensitiveDataItem> {
    private static final SdkField<String> CATEGORY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("category").getter(getter(SensitiveDataItem::categoryAsString)).setter(setter(Builder::category))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("category").build()).build();

    private static final SdkField<List<DefaultDetection>> DETECTIONS_FIELD = SdkField
            .<List<DefaultDetection>> builder(MarshallingType.LIST)
            .memberName("detections")
            .getter(getter(SensitiveDataItem::detections))
            .setter(setter(Builder::detections))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("detections").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DefaultDetection> builder(MarshallingType.SDK_POJO)
                                            .constructor(DefaultDetection::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Long> TOTAL_COUNT_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("totalCount").getter(getter(SensitiveDataItem::totalCount)).setter(setter(Builder::totalCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("totalCount").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CATEGORY_FIELD,
            DETECTIONS_FIELD, TOTAL_COUNT_FIELD));

    private static final long serialVersionUID = 1L;

    private final String category;

    private final List<DefaultDetection> detections;

    private final Long totalCount;

    private SensitiveDataItem(BuilderImpl builder) {
        this.category = builder.category;
        this.detections = builder.detections;
        this.totalCount = builder.totalCount;
    }

    /**
     * <p>
     * The category of sensitive data that was detected. For example: CREDENTIALS, for credentials data such as private
     * keys or AWS secret keys; FINANCIAL_INFORMATION, for financial data such as credit card numbers; or,
     * PERSONAL_INFORMATION, for personal health information, such as health insurance identification numbers, or
     * personally identifiable information, such as driver's license identification numbers.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #category} will
     * return {@link SensitiveDataItemCategory#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #categoryAsString}.
     * </p>
     * 
     * @return The category of sensitive data that was detected. For example: CREDENTIALS, for credentials data such as
     *         private keys or AWS secret keys; FINANCIAL_INFORMATION, for financial data such as credit card numbers;
     *         or, PERSONAL_INFORMATION, for personal health information, such as health insurance identification
     *         numbers, or personally identifiable information, such as driver's license identification numbers.
     * @see SensitiveDataItemCategory
     */
    public final SensitiveDataItemCategory category() {
        return SensitiveDataItemCategory.fromValue(category);
    }

    /**
     * <p>
     * The category of sensitive data that was detected. For example: CREDENTIALS, for credentials data such as private
     * keys or AWS secret keys; FINANCIAL_INFORMATION, for financial data such as credit card numbers; or,
     * PERSONAL_INFORMATION, for personal health information, such as health insurance identification numbers, or
     * personally identifiable information, such as driver's license identification numbers.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #category} will
     * return {@link SensitiveDataItemCategory#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #categoryAsString}.
     * </p>
     * 
     * @return The category of sensitive data that was detected. For example: CREDENTIALS, for credentials data such as
     *         private keys or AWS secret keys; FINANCIAL_INFORMATION, for financial data such as credit card numbers;
     *         or, PERSONAL_INFORMATION, for personal health information, such as health insurance identification
     *         numbers, or personally identifiable information, such as driver's license identification numbers.
     * @see SensitiveDataItemCategory
     */
    public final String categoryAsString() {
        return category;
    }

    /**
     * Returns true if the Detections property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasDetections() {
        return detections != null && !(detections instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of objects, one for each type of sensitive data that was detected. Each object reports the number of
     * occurrences of a specific type of sensitive data that was detected, and the location of up to 15 of those
     * occurrences.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasDetections()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of objects, one for each type of sensitive data that was detected. Each object reports the
     *         number of occurrences of a specific type of sensitive data that was detected, and the location of up to
     *         15 of those occurrences.
     */
    public final List<DefaultDetection> detections() {
        return detections;
    }

    /**
     * <p>
     * The total number of occurrences of the sensitive data that was detected.
     * </p>
     * 
     * @return The total number of occurrences of the sensitive data that was detected.
     */
    public final Long totalCount() {
        return totalCount;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(categoryAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasDetections() ? detections() : null);
        hashCode = 31 * hashCode + Objects.hashCode(totalCount());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SensitiveDataItem)) {
            return false;
        }
        SensitiveDataItem other = (SensitiveDataItem) obj;
        return Objects.equals(categoryAsString(), other.categoryAsString()) && hasDetections() == other.hasDetections()
                && Objects.equals(detections(), other.detections()) && Objects.equals(totalCount(), other.totalCount());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SensitiveDataItem").add("Category", categoryAsString())
                .add("Detections", hasDetections() ? detections() : null).add("TotalCount", totalCount()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "category":
            return Optional.ofNullable(clazz.cast(categoryAsString()));
        case "detections":
            return Optional.ofNullable(clazz.cast(detections()));
        case "totalCount":
            return Optional.ofNullable(clazz.cast(totalCount()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SensitiveDataItem, T> g) {
        return obj -> g.apply((SensitiveDataItem) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SensitiveDataItem> {
        /**
         * <p>
         * The category of sensitive data that was detected. For example: CREDENTIALS, for credentials data such as
         * private keys or AWS secret keys; FINANCIAL_INFORMATION, for financial data such as credit card numbers; or,
         * PERSONAL_INFORMATION, for personal health information, such as health insurance identification numbers, or
         * personally identifiable information, such as driver's license identification numbers.
         * </p>
         * 
         * @param category
         *        The category of sensitive data that was detected. For example: CREDENTIALS, for credentials data such
         *        as private keys or AWS secret keys; FINANCIAL_INFORMATION, for financial data such as credit card
         *        numbers; or, PERSONAL_INFORMATION, for personal health information, such as health insurance
         *        identification numbers, or personally identifiable information, such as driver's license
         *        identification numbers.
         * @see SensitiveDataItemCategory
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SensitiveDataItemCategory
         */
        Builder category(String category);

        /**
         * <p>
         * The category of sensitive data that was detected. For example: CREDENTIALS, for credentials data such as
         * private keys or AWS secret keys; FINANCIAL_INFORMATION, for financial data such as credit card numbers; or,
         * PERSONAL_INFORMATION, for personal health information, such as health insurance identification numbers, or
         * personally identifiable information, such as driver's license identification numbers.
         * </p>
         * 
         * @param category
         *        The category of sensitive data that was detected. For example: CREDENTIALS, for credentials data such
         *        as private keys or AWS secret keys; FINANCIAL_INFORMATION, for financial data such as credit card
         *        numbers; or, PERSONAL_INFORMATION, for personal health information, such as health insurance
         *        identification numbers, or personally identifiable information, such as driver's license
         *        identification numbers.
         * @see SensitiveDataItemCategory
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SensitiveDataItemCategory
         */
        Builder category(SensitiveDataItemCategory category);

        /**
         * <p>
         * An array of objects, one for each type of sensitive data that was detected. Each object reports the number of
         * occurrences of a specific type of sensitive data that was detected, and the location of up to 15 of those
         * occurrences.
         * </p>
         * 
         * @param detections
         *        An array of objects, one for each type of sensitive data that was detected. Each object reports the
         *        number of occurrences of a specific type of sensitive data that was detected, and the location of up
         *        to 15 of those occurrences.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder detections(Collection<DefaultDetection> detections);

        /**
         * <p>
         * An array of objects, one for each type of sensitive data that was detected. Each object reports the number of
         * occurrences of a specific type of sensitive data that was detected, and the location of up to 15 of those
         * occurrences.
         * </p>
         * 
         * @param detections
         *        An array of objects, one for each type of sensitive data that was detected. Each object reports the
         *        number of occurrences of a specific type of sensitive data that was detected, and the location of up
         *        to 15 of those occurrences.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder detections(DefaultDetection... detections);

        /**
         * <p>
         * An array of objects, one for each type of sensitive data that was detected. Each object reports the number of
         * occurrences of a specific type of sensitive data that was detected, and the location of up to 15 of those
         * occurrences.
         * </p>
         * This is a convenience that creates an instance of the {@link List<DefaultDetection>.Builder} avoiding the
         * need to create one manually via {@link List<DefaultDetection>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<DefaultDetection>.Builder#build()} is called immediately and
         * its result is passed to {@link #detections(List<DefaultDetection>)}.
         * 
         * @param detections
         *        a consumer that will call methods on {@link List<DefaultDetection>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #detections(List<DefaultDetection>)
         */
        Builder detections(Consumer<DefaultDetection.Builder>... detections);

        /**
         * <p>
         * The total number of occurrences of the sensitive data that was detected.
         * </p>
         * 
         * @param totalCount
         *        The total number of occurrences of the sensitive data that was detected.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder totalCount(Long totalCount);
    }

    static final class BuilderImpl implements Builder {
        private String category;

        private List<DefaultDetection> detections = DefaultSdkAutoConstructList.getInstance();

        private Long totalCount;

        private BuilderImpl() {
        }

        private BuilderImpl(SensitiveDataItem model) {
            category(model.category);
            detections(model.detections);
            totalCount(model.totalCount);
        }

        public final String getCategory() {
            return category;
        }

        @Override
        public final Builder category(String category) {
            this.category = category;
            return this;
        }

        @Override
        public final Builder category(SensitiveDataItemCategory category) {
            this.category(category == null ? null : category.toString());
            return this;
        }

        public final void setCategory(String category) {
            this.category = category;
        }

        public final List<DefaultDetection.Builder> getDetections() {
            List<DefaultDetection.Builder> result = DefaultDetectionsCopier.copyToBuilder(this.detections);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder detections(Collection<DefaultDetection> detections) {
            this.detections = DefaultDetectionsCopier.copy(detections);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder detections(DefaultDetection... detections) {
            detections(Arrays.asList(detections));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder detections(Consumer<DefaultDetection.Builder>... detections) {
            detections(Stream.of(detections).map(c -> DefaultDetection.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setDetections(Collection<DefaultDetection.BuilderImpl> detections) {
            this.detections = DefaultDetectionsCopier.copyFromBuilder(detections);
        }

        public final Long getTotalCount() {
            return totalCount;
        }

        @Override
        public final Builder totalCount(Long totalCount) {
            this.totalCount = totalCount;
            return this;
        }

        public final void setTotalCount(Long totalCount) {
            this.totalCount = totalCount;
        }

        @Override
        public SensitiveDataItem build() {
            return new SensitiveDataItem(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
