/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.macie2.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides information about when a classification job was paused. For a one-time job, this object also specifies when
 * the job will expire and be cancelled if it isn't resumed. For a recurring job, this object also specifies when the
 * paused job run will expire and be cancelled if it isn't resumed. This object is present only if a job's current
 * status (jobStatus) is USER_PAUSED. The information in this object applies only to a job that was paused while it had
 * a status of RUNNING.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class UserPausedDetails implements SdkPojo, Serializable,
        ToCopyableBuilder<UserPausedDetails.Builder, UserPausedDetails> {
    private static final SdkField<Instant> JOB_EXPIRES_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("jobExpiresAt")
            .getter(getter(UserPausedDetails::jobExpiresAt))
            .setter(setter(Builder::jobExpiresAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("jobExpiresAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> JOB_IMMINENT_EXPIRATION_HEALTH_EVENT_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("jobImminentExpirationHealthEventArn")
            .getter(getter(UserPausedDetails::jobImminentExpirationHealthEventArn))
            .setter(setter(Builder::jobImminentExpirationHealthEventArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("jobImminentExpirationHealthEventArn").build()).build();

    private static final SdkField<Instant> JOB_PAUSED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("jobPausedAt")
            .getter(getter(UserPausedDetails::jobPausedAt))
            .setter(setter(Builder::jobPausedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("jobPausedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(JOB_EXPIRES_AT_FIELD,
            JOB_IMMINENT_EXPIRATION_HEALTH_EVENT_ARN_FIELD, JOB_PAUSED_AT_FIELD));

    private static final long serialVersionUID = 1L;

    private final Instant jobExpiresAt;

    private final String jobImminentExpirationHealthEventArn;

    private final Instant jobPausedAt;

    private UserPausedDetails(BuilderImpl builder) {
        this.jobExpiresAt = builder.jobExpiresAt;
        this.jobImminentExpirationHealthEventArn = builder.jobImminentExpirationHealthEventArn;
        this.jobPausedAt = builder.jobPausedAt;
    }

    /**
     * <p>
     * The date and time, in UTC and extended ISO 8601 format, when the job or job run will expire and be cancelled if
     * you don't resume it first.
     * </p>
     * 
     * @return The date and time, in UTC and extended ISO 8601 format, when the job or job run will expire and be
     *         cancelled if you don't resume it first.
     */
    public final Instant jobExpiresAt() {
        return jobExpiresAt;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the Health event that Amazon Macie sent to notify you of the job or job run's
     * pending expiration and cancellation. This value is null if a job has been paused for less than 23 days.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the Health event that Amazon Macie sent to notify you of the job or job
     *         run's pending expiration and cancellation. This value is null if a job has been paused for less than 23
     *         days.
     */
    public final String jobImminentExpirationHealthEventArn() {
        return jobImminentExpirationHealthEventArn;
    }

    /**
     * <p>
     * The date and time, in UTC and extended ISO 8601 format, when you paused the job.
     * </p>
     * 
     * @return The date and time, in UTC and extended ISO 8601 format, when you paused the job.
     */
    public final Instant jobPausedAt() {
        return jobPausedAt;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(jobExpiresAt());
        hashCode = 31 * hashCode + Objects.hashCode(jobImminentExpirationHealthEventArn());
        hashCode = 31 * hashCode + Objects.hashCode(jobPausedAt());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UserPausedDetails)) {
            return false;
        }
        UserPausedDetails other = (UserPausedDetails) obj;
        return Objects.equals(jobExpiresAt(), other.jobExpiresAt())
                && Objects.equals(jobImminentExpirationHealthEventArn(), other.jobImminentExpirationHealthEventArn())
                && Objects.equals(jobPausedAt(), other.jobPausedAt());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("UserPausedDetails").add("JobExpiresAt", jobExpiresAt())
                .add("JobImminentExpirationHealthEventArn", jobImminentExpirationHealthEventArn())
                .add("JobPausedAt", jobPausedAt()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "jobExpiresAt":
            return Optional.ofNullable(clazz.cast(jobExpiresAt()));
        case "jobImminentExpirationHealthEventArn":
            return Optional.ofNullable(clazz.cast(jobImminentExpirationHealthEventArn()));
        case "jobPausedAt":
            return Optional.ofNullable(clazz.cast(jobPausedAt()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<UserPausedDetails, T> g) {
        return obj -> g.apply((UserPausedDetails) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, UserPausedDetails> {
        /**
         * <p>
         * The date and time, in UTC and extended ISO 8601 format, when the job or job run will expire and be cancelled
         * if you don't resume it first.
         * </p>
         * 
         * @param jobExpiresAt
         *        The date and time, in UTC and extended ISO 8601 format, when the job or job run will expire and be
         *        cancelled if you don't resume it first.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobExpiresAt(Instant jobExpiresAt);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the Health event that Amazon Macie sent to notify you of the job or job
         * run's pending expiration and cancellation. This value is null if a job has been paused for less than 23 days.
         * </p>
         * 
         * @param jobImminentExpirationHealthEventArn
         *        The Amazon Resource Name (ARN) of the Health event that Amazon Macie sent to notify you of the job or
         *        job run's pending expiration and cancellation. This value is null if a job has been paused for less
         *        than 23 days.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobImminentExpirationHealthEventArn(String jobImminentExpirationHealthEventArn);

        /**
         * <p>
         * The date and time, in UTC and extended ISO 8601 format, when you paused the job.
         * </p>
         * 
         * @param jobPausedAt
         *        The date and time, in UTC and extended ISO 8601 format, when you paused the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobPausedAt(Instant jobPausedAt);
    }

    static final class BuilderImpl implements Builder {
        private Instant jobExpiresAt;

        private String jobImminentExpirationHealthEventArn;

        private Instant jobPausedAt;

        private BuilderImpl() {
        }

        private BuilderImpl(UserPausedDetails model) {
            jobExpiresAt(model.jobExpiresAt);
            jobImminentExpirationHealthEventArn(model.jobImminentExpirationHealthEventArn);
            jobPausedAt(model.jobPausedAt);
        }

        public final Instant getJobExpiresAt() {
            return jobExpiresAt;
        }

        @Override
        public final Builder jobExpiresAt(Instant jobExpiresAt) {
            this.jobExpiresAt = jobExpiresAt;
            return this;
        }

        public final void setJobExpiresAt(Instant jobExpiresAt) {
            this.jobExpiresAt = jobExpiresAt;
        }

        public final String getJobImminentExpirationHealthEventArn() {
            return jobImminentExpirationHealthEventArn;
        }

        @Override
        public final Builder jobImminentExpirationHealthEventArn(String jobImminentExpirationHealthEventArn) {
            this.jobImminentExpirationHealthEventArn = jobImminentExpirationHealthEventArn;
            return this;
        }

        public final void setJobImminentExpirationHealthEventArn(String jobImminentExpirationHealthEventArn) {
            this.jobImminentExpirationHealthEventArn = jobImminentExpirationHealthEventArn;
        }

        public final Instant getJobPausedAt() {
            return jobPausedAt;
        }

        @Override
        public final Builder jobPausedAt(Instant jobPausedAt) {
            this.jobPausedAt = jobPausedAt;
            return this;
        }

        public final void setJobPausedAt(Instant jobPausedAt) {
            this.jobPausedAt = jobPausedAt;
        }

        @Override
        public UserPausedDetails build() {
            return new UserPausedDetails(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
