/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.macie2.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides information about an account that's associated with an Amazon Macie administrator account.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Member implements SdkPojo, Serializable, ToCopyableBuilder<Member.Builder, Member> {
    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("accountId").getter(getter(Member::accountId)).setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("accountId").build()).build();

    private static final SdkField<String> ADMINISTRATOR_ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("administratorAccountId").getter(getter(Member::administratorAccountId))
            .setter(setter(Builder::administratorAccountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("administratorAccountId").build())
            .build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(Member::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<String> EMAIL_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("email")
            .getter(getter(Member::email)).setter(setter(Builder::email))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("email").build()).build();

    private static final SdkField<Instant> INVITED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("invitedAt")
            .getter(getter(Member::invitedAt))
            .setter(setter(Builder::invitedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("invitedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> MASTER_ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("masterAccountId").getter(getter(Member::masterAccountId)).setter(setter(Builder::masterAccountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("masterAccountId").build()).build();

    private static final SdkField<String> RELATIONSHIP_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("relationshipStatus").getter(getter(Member::relationshipStatusAsString))
            .setter(setter(Builder::relationshipStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("relationshipStatus").build())
            .build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(Member::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Instant> UPDATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("updatedAt")
            .getter(getter(Member::updatedAt))
            .setter(setter(Builder::updatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("updatedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACCOUNT_ID_FIELD,
            ADMINISTRATOR_ACCOUNT_ID_FIELD, ARN_FIELD, EMAIL_FIELD, INVITED_AT_FIELD, MASTER_ACCOUNT_ID_FIELD,
            RELATIONSHIP_STATUS_FIELD, TAGS_FIELD, UPDATED_AT_FIELD));

    private static final long serialVersionUID = 1L;

    private final String accountId;

    private final String administratorAccountId;

    private final String arn;

    private final String email;

    private final Instant invitedAt;

    private final String masterAccountId;

    private final String relationshipStatus;

    private final Map<String, String> tags;

    private final Instant updatedAt;

    private Member(BuilderImpl builder) {
        this.accountId = builder.accountId;
        this.administratorAccountId = builder.administratorAccountId;
        this.arn = builder.arn;
        this.email = builder.email;
        this.invitedAt = builder.invitedAt;
        this.masterAccountId = builder.masterAccountId;
        this.relationshipStatus = builder.relationshipStatus;
        this.tags = builder.tags;
        this.updatedAt = builder.updatedAt;
    }

    /**
     * <p>
     * The Amazon Web Services account ID for the account.
     * </p>
     * 
     * @return The Amazon Web Services account ID for the account.
     */
    public final String accountId() {
        return accountId;
    }

    /**
     * <p>
     * The Amazon Web Services account ID for the administrator account.
     * </p>
     * 
     * @return The Amazon Web Services account ID for the administrator account.
     */
    public final String administratorAccountId() {
        return administratorAccountId;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the account.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the account.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The email address for the account.
     * </p>
     * 
     * @return The email address for the account.
     */
    public final String email() {
        return email;
    }

    /**
     * <p>
     * The date and time, in UTC and extended ISO 8601 format, when an Amazon Macie membership invitation was last sent
     * to the account. This value is null if an invitation hasn't been sent to the account.
     * </p>
     * 
     * @return The date and time, in UTC and extended ISO 8601 format, when an Amazon Macie membership invitation was
     *         last sent to the account. This value is null if an invitation hasn't been sent to the account.
     */
    public final Instant invitedAt() {
        return invitedAt;
    }

    /**
     * <p>
     * (Deprecated) The Amazon Web Services account ID for the administrator account. This property has been replaced by
     * the administratorAccountId property and is retained only for backward compatibility.
     * </p>
     * 
     * @return (Deprecated) The Amazon Web Services account ID for the administrator account. This property has been
     *         replaced by the administratorAccountId property and is retained only for backward compatibility.
     */
    public final String masterAccountId() {
        return masterAccountId;
    }

    /**
     * <p>
     * The current status of the relationship between the account and the administrator account.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #relationshipStatus} will return {@link RelationshipStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #relationshipStatusAsString}.
     * </p>
     * 
     * @return The current status of the relationship between the account and the administrator account.
     * @see RelationshipStatus
     */
    public final RelationshipStatus relationshipStatus() {
        return RelationshipStatus.fromValue(relationshipStatus);
    }

    /**
     * <p>
     * The current status of the relationship between the account and the administrator account.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #relationshipStatus} will return {@link RelationshipStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #relationshipStatusAsString}.
     * </p>
     * 
     * @return The current status of the relationship between the account and the administrator account.
     * @see RelationshipStatus
     */
    public final String relationshipStatusAsString() {
        return relationshipStatus;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A map of key-value pairs that identifies the tags (keys and values) that are associated with the account in
     * Amazon Macie.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return A map of key-value pairs that identifies the tags (keys and values) that are associated with the account
     *         in Amazon Macie.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * The date and time, in UTC and extended ISO 8601 format, of the most recent change to the status of the
     * relationship between the account and the administrator account.
     * </p>
     * 
     * @return The date and time, in UTC and extended ISO 8601 format, of the most recent change to the status of the
     *         relationship between the account and the administrator account.
     */
    public final Instant updatedAt() {
        return updatedAt;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(administratorAccountId());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(email());
        hashCode = 31 * hashCode + Objects.hashCode(invitedAt());
        hashCode = 31 * hashCode + Objects.hashCode(masterAccountId());
        hashCode = 31 * hashCode + Objects.hashCode(relationshipStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(updatedAt());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Member)) {
            return false;
        }
        Member other = (Member) obj;
        return Objects.equals(accountId(), other.accountId())
                && Objects.equals(administratorAccountId(), other.administratorAccountId()) && Objects.equals(arn(), other.arn())
                && Objects.equals(email(), other.email()) && Objects.equals(invitedAt(), other.invitedAt())
                && Objects.equals(masterAccountId(), other.masterAccountId())
                && Objects.equals(relationshipStatusAsString(), other.relationshipStatusAsString())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags())
                && Objects.equals(updatedAt(), other.updatedAt());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Member").add("AccountId", accountId()).add("AdministratorAccountId", administratorAccountId())
                .add("Arn", arn()).add("Email", email()).add("InvitedAt", invitedAt()).add("MasterAccountId", masterAccountId())
                .add("RelationshipStatus", relationshipStatusAsString()).add("Tags", hasTags() ? tags() : null)
                .add("UpdatedAt", updatedAt()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "accountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "administratorAccountId":
            return Optional.ofNullable(clazz.cast(administratorAccountId()));
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "email":
            return Optional.ofNullable(clazz.cast(email()));
        case "invitedAt":
            return Optional.ofNullable(clazz.cast(invitedAt()));
        case "masterAccountId":
            return Optional.ofNullable(clazz.cast(masterAccountId()));
        case "relationshipStatus":
            return Optional.ofNullable(clazz.cast(relationshipStatusAsString()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "updatedAt":
            return Optional.ofNullable(clazz.cast(updatedAt()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Member, T> g) {
        return obj -> g.apply((Member) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Member> {
        /**
         * <p>
         * The Amazon Web Services account ID for the account.
         * </p>
         * 
         * @param accountId
         *        The Amazon Web Services account ID for the account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * The Amazon Web Services account ID for the administrator account.
         * </p>
         * 
         * @param administratorAccountId
         *        The Amazon Web Services account ID for the administrator account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder administratorAccountId(String administratorAccountId);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the account.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The email address for the account.
         * </p>
         * 
         * @param email
         *        The email address for the account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder email(String email);

        /**
         * <p>
         * The date and time, in UTC and extended ISO 8601 format, when an Amazon Macie membership invitation was last
         * sent to the account. This value is null if an invitation hasn't been sent to the account.
         * </p>
         * 
         * @param invitedAt
         *        The date and time, in UTC and extended ISO 8601 format, when an Amazon Macie membership invitation was
         *        last sent to the account. This value is null if an invitation hasn't been sent to the account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder invitedAt(Instant invitedAt);

        /**
         * <p>
         * (Deprecated) The Amazon Web Services account ID for the administrator account. This property has been
         * replaced by the administratorAccountId property and is retained only for backward compatibility.
         * </p>
         * 
         * @param masterAccountId
         *        (Deprecated) The Amazon Web Services account ID for the administrator account. This property has been
         *        replaced by the administratorAccountId property and is retained only for backward compatibility.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder masterAccountId(String masterAccountId);

        /**
         * <p>
         * The current status of the relationship between the account and the administrator account.
         * </p>
         * 
         * @param relationshipStatus
         *        The current status of the relationship between the account and the administrator account.
         * @see RelationshipStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RelationshipStatus
         */
        Builder relationshipStatus(String relationshipStatus);

        /**
         * <p>
         * The current status of the relationship between the account and the administrator account.
         * </p>
         * 
         * @param relationshipStatus
         *        The current status of the relationship between the account and the administrator account.
         * @see RelationshipStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RelationshipStatus
         */
        Builder relationshipStatus(RelationshipStatus relationshipStatus);

        /**
         * <p>
         * A map of key-value pairs that identifies the tags (keys and values) that are associated with the account in
         * Amazon Macie.
         * </p>
         * 
         * @param tags
         *        A map of key-value pairs that identifies the tags (keys and values) that are associated with the
         *        account in Amazon Macie.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * The date and time, in UTC and extended ISO 8601 format, of the most recent change to the status of the
         * relationship between the account and the administrator account.
         * </p>
         * 
         * @param updatedAt
         *        The date and time, in UTC and extended ISO 8601 format, of the most recent change to the status of the
         *        relationship between the account and the administrator account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updatedAt(Instant updatedAt);
    }

    static final class BuilderImpl implements Builder {
        private String accountId;

        private String administratorAccountId;

        private String arn;

        private String email;

        private Instant invitedAt;

        private String masterAccountId;

        private String relationshipStatus;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private Instant updatedAt;

        private BuilderImpl() {
        }

        private BuilderImpl(Member model) {
            accountId(model.accountId);
            administratorAccountId(model.administratorAccountId);
            arn(model.arn);
            email(model.email);
            invitedAt(model.invitedAt);
            masterAccountId(model.masterAccountId);
            relationshipStatus(model.relationshipStatus);
            tags(model.tags);
            updatedAt(model.updatedAt);
        }

        public final String getAccountId() {
            return accountId;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final String getAdministratorAccountId() {
            return administratorAccountId;
        }

        public final void setAdministratorAccountId(String administratorAccountId) {
            this.administratorAccountId = administratorAccountId;
        }

        @Override
        public final Builder administratorAccountId(String administratorAccountId) {
            this.administratorAccountId = administratorAccountId;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getEmail() {
            return email;
        }

        public final void setEmail(String email) {
            this.email = email;
        }

        @Override
        public final Builder email(String email) {
            this.email = email;
            return this;
        }

        public final Instant getInvitedAt() {
            return invitedAt;
        }

        public final void setInvitedAt(Instant invitedAt) {
            this.invitedAt = invitedAt;
        }

        @Override
        public final Builder invitedAt(Instant invitedAt) {
            this.invitedAt = invitedAt;
            return this;
        }

        public final String getMasterAccountId() {
            return masterAccountId;
        }

        public final void setMasterAccountId(String masterAccountId) {
            this.masterAccountId = masterAccountId;
        }

        @Override
        public final Builder masterAccountId(String masterAccountId) {
            this.masterAccountId = masterAccountId;
            return this;
        }

        public final String getRelationshipStatus() {
            return relationshipStatus;
        }

        public final void setRelationshipStatus(String relationshipStatus) {
            this.relationshipStatus = relationshipStatus;
        }

        @Override
        public final Builder relationshipStatus(String relationshipStatus) {
            this.relationshipStatus = relationshipStatus;
            return this;
        }

        @Override
        public final Builder relationshipStatus(RelationshipStatus relationshipStatus) {
            this.relationshipStatus(relationshipStatus == null ? null : relationshipStatus.toString());
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
            return this;
        }

        public final Instant getUpdatedAt() {
            return updatedAt;
        }

        public final void setUpdatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
        }

        @Override
        public final Builder updatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
            return this;
        }

        @Override
        public Member build() {
            return new Member(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
