/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.macie2.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides information about an Amazon Macie membership invitation that was received by an account.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Invitation implements SdkPojo, Serializable, ToCopyableBuilder<Invitation.Builder, Invitation> {
    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("accountId").getter(getter(Invitation::accountId)).setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("accountId").build()).build();

    private static final SdkField<String> INVITATION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("invitationId").getter(getter(Invitation::invitationId)).setter(setter(Builder::invitationId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("invitationId").build()).build();

    private static final SdkField<Instant> INVITED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("invitedAt")
            .getter(getter(Invitation::invitedAt))
            .setter(setter(Builder::invitedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("invitedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> RELATIONSHIP_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("relationshipStatus").getter(getter(Invitation::relationshipStatusAsString))
            .setter(setter(Builder::relationshipStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("relationshipStatus").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACCOUNT_ID_FIELD,
            INVITATION_ID_FIELD, INVITED_AT_FIELD, RELATIONSHIP_STATUS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String accountId;

    private final String invitationId;

    private final Instant invitedAt;

    private final String relationshipStatus;

    private Invitation(BuilderImpl builder) {
        this.accountId = builder.accountId;
        this.invitationId = builder.invitationId;
        this.invitedAt = builder.invitedAt;
        this.relationshipStatus = builder.relationshipStatus;
    }

    /**
     * <p>
     * The Amazon Web Services account ID for the account that sent the invitation.
     * </p>
     * 
     * @return The Amazon Web Services account ID for the account that sent the invitation.
     */
    public final String accountId() {
        return accountId;
    }

    /**
     * <p>
     * The unique identifier for the invitation. Amazon Macie uses this identifier to validate the inviter account with
     * the invitee account.
     * </p>
     * 
     * @return The unique identifier for the invitation. Amazon Macie uses this identifier to validate the inviter
     *         account with the invitee account.
     */
    public final String invitationId() {
        return invitationId;
    }

    /**
     * <p>
     * The date and time, in UTC and extended ISO 8601 format, when the invitation was sent.
     * </p>
     * 
     * @return The date and time, in UTC and extended ISO 8601 format, when the invitation was sent.
     */
    public final Instant invitedAt() {
        return invitedAt;
    }

    /**
     * <p>
     * The status of the relationship between the account that sent the invitation (<i>inviter account</i>) and the
     * account that received the invitation (<i>invitee account</i>).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #relationshipStatus} will return {@link RelationshipStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #relationshipStatusAsString}.
     * </p>
     * 
     * @return The status of the relationship between the account that sent the invitation (<i>inviter account</i>) and
     *         the account that received the invitation (<i>invitee account</i>).
     * @see RelationshipStatus
     */
    public final RelationshipStatus relationshipStatus() {
        return RelationshipStatus.fromValue(relationshipStatus);
    }

    /**
     * <p>
     * The status of the relationship between the account that sent the invitation (<i>inviter account</i>) and the
     * account that received the invitation (<i>invitee account</i>).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #relationshipStatus} will return {@link RelationshipStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #relationshipStatusAsString}.
     * </p>
     * 
     * @return The status of the relationship between the account that sent the invitation (<i>inviter account</i>) and
     *         the account that received the invitation (<i>invitee account</i>).
     * @see RelationshipStatus
     */
    public final String relationshipStatusAsString() {
        return relationshipStatus;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(invitationId());
        hashCode = 31 * hashCode + Objects.hashCode(invitedAt());
        hashCode = 31 * hashCode + Objects.hashCode(relationshipStatusAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Invitation)) {
            return false;
        }
        Invitation other = (Invitation) obj;
        return Objects.equals(accountId(), other.accountId()) && Objects.equals(invitationId(), other.invitationId())
                && Objects.equals(invitedAt(), other.invitedAt())
                && Objects.equals(relationshipStatusAsString(), other.relationshipStatusAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Invitation").add("AccountId", accountId()).add("InvitationId", invitationId())
                .add("InvitedAt", invitedAt()).add("RelationshipStatus", relationshipStatusAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "accountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "invitationId":
            return Optional.ofNullable(clazz.cast(invitationId()));
        case "invitedAt":
            return Optional.ofNullable(clazz.cast(invitedAt()));
        case "relationshipStatus":
            return Optional.ofNullable(clazz.cast(relationshipStatusAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Invitation, T> g) {
        return obj -> g.apply((Invitation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Invitation> {
        /**
         * <p>
         * The Amazon Web Services account ID for the account that sent the invitation.
         * </p>
         * 
         * @param accountId
         *        The Amazon Web Services account ID for the account that sent the invitation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * The unique identifier for the invitation. Amazon Macie uses this identifier to validate the inviter account
         * with the invitee account.
         * </p>
         * 
         * @param invitationId
         *        The unique identifier for the invitation. Amazon Macie uses this identifier to validate the inviter
         *        account with the invitee account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder invitationId(String invitationId);

        /**
         * <p>
         * The date and time, in UTC and extended ISO 8601 format, when the invitation was sent.
         * </p>
         * 
         * @param invitedAt
         *        The date and time, in UTC and extended ISO 8601 format, when the invitation was sent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder invitedAt(Instant invitedAt);

        /**
         * <p>
         * The status of the relationship between the account that sent the invitation (<i>inviter account</i>) and the
         * account that received the invitation (<i>invitee account</i>).
         * </p>
         * 
         * @param relationshipStatus
         *        The status of the relationship between the account that sent the invitation (<i>inviter account</i>)
         *        and the account that received the invitation (<i>invitee account</i>).
         * @see RelationshipStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RelationshipStatus
         */
        Builder relationshipStatus(String relationshipStatus);

        /**
         * <p>
         * The status of the relationship between the account that sent the invitation (<i>inviter account</i>) and the
         * account that received the invitation (<i>invitee account</i>).
         * </p>
         * 
         * @param relationshipStatus
         *        The status of the relationship between the account that sent the invitation (<i>inviter account</i>)
         *        and the account that received the invitation (<i>invitee account</i>).
         * @see RelationshipStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RelationshipStatus
         */
        Builder relationshipStatus(RelationshipStatus relationshipStatus);
    }

    static final class BuilderImpl implements Builder {
        private String accountId;

        private String invitationId;

        private Instant invitedAt;

        private String relationshipStatus;

        private BuilderImpl() {
        }

        private BuilderImpl(Invitation model) {
            accountId(model.accountId);
            invitationId(model.invitationId);
            invitedAt(model.invitedAt);
            relationshipStatus(model.relationshipStatus);
        }

        public final String getAccountId() {
            return accountId;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        @Override
        @Transient
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final String getInvitationId() {
            return invitationId;
        }

        public final void setInvitationId(String invitationId) {
            this.invitationId = invitationId;
        }

        @Override
        @Transient
        public final Builder invitationId(String invitationId) {
            this.invitationId = invitationId;
            return this;
        }

        public final Instant getInvitedAt() {
            return invitedAt;
        }

        public final void setInvitedAt(Instant invitedAt) {
            this.invitedAt = invitedAt;
        }

        @Override
        @Transient
        public final Builder invitedAt(Instant invitedAt) {
            this.invitedAt = invitedAt;
            return this;
        }

        public final String getRelationshipStatus() {
            return relationshipStatus;
        }

        public final void setRelationshipStatus(String relationshipStatus) {
            this.relationshipStatus = relationshipStatus;
        }

        @Override
        @Transient
        public final Builder relationshipStatus(String relationshipStatus) {
            this.relationshipStatus = relationshipStatus;
            return this;
        }

        @Override
        @Transient
        public final Builder relationshipStatus(RelationshipStatus relationshipStatus) {
            this.relationshipStatus(relationshipStatus == null ? null : relationshipStatus.toString());
            return this;
        }

        @Override
        public Invitation build() {
            return new Invitation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
