/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.macie2.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides information about the S3 bucket that a finding applies to.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class S3Bucket implements SdkPojo, Serializable, ToCopyableBuilder<S3Bucket.Builder, S3Bucket> {
    private static final SdkField<String> ALLOWS_UNENCRYPTED_OBJECT_UPLOADS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("allowsUnencryptedObjectUploads")
            .getter(getter(S3Bucket::allowsUnencryptedObjectUploadsAsString))
            .setter(setter(Builder::allowsUnencryptedObjectUploads))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("allowsUnencryptedObjectUploads")
                    .build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(S3Bucket::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt")
            .getter(getter(S3Bucket::createdAt))
            .setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<ServerSideEncryption> DEFAULT_SERVER_SIDE_ENCRYPTION_FIELD = SdkField
            .<ServerSideEncryption> builder(MarshallingType.SDK_POJO)
            .memberName("defaultServerSideEncryption")
            .getter(getter(S3Bucket::defaultServerSideEncryption))
            .setter(setter(Builder::defaultServerSideEncryption))
            .constructor(ServerSideEncryption::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("defaultServerSideEncryption")
                    .build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(S3Bucket::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<S3BucketOwner> OWNER_FIELD = SdkField.<S3BucketOwner> builder(MarshallingType.SDK_POJO)
            .memberName("owner").getter(getter(S3Bucket::owner)).setter(setter(Builder::owner))
            .constructor(S3BucketOwner::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("owner").build()).build();

    private static final SdkField<BucketPublicAccess> PUBLIC_ACCESS_FIELD = SdkField
            .<BucketPublicAccess> builder(MarshallingType.SDK_POJO).memberName("publicAccess")
            .getter(getter(S3Bucket::publicAccess)).setter(setter(Builder::publicAccess))
            .constructor(BucketPublicAccess::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("publicAccess").build()).build();

    private static final SdkField<List<KeyValuePair>> TAGS_FIELD = SdkField
            .<List<KeyValuePair>> builder(MarshallingType.LIST)
            .memberName("tags")
            .getter(getter(S3Bucket::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<KeyValuePair> builder(MarshallingType.SDK_POJO)
                                            .constructor(KeyValuePair::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            ALLOWS_UNENCRYPTED_OBJECT_UPLOADS_FIELD, ARN_FIELD, CREATED_AT_FIELD, DEFAULT_SERVER_SIDE_ENCRYPTION_FIELD,
            NAME_FIELD, OWNER_FIELD, PUBLIC_ACCESS_FIELD, TAGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String allowsUnencryptedObjectUploads;

    private final String arn;

    private final Instant createdAt;

    private final ServerSideEncryption defaultServerSideEncryption;

    private final String name;

    private final S3BucketOwner owner;

    private final BucketPublicAccess publicAccess;

    private final List<KeyValuePair> tags;

    private S3Bucket(BuilderImpl builder) {
        this.allowsUnencryptedObjectUploads = builder.allowsUnencryptedObjectUploads;
        this.arn = builder.arn;
        this.createdAt = builder.createdAt;
        this.defaultServerSideEncryption = builder.defaultServerSideEncryption;
        this.name = builder.name;
        this.owner = builder.owner;
        this.publicAccess = builder.publicAccess;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * Specifies whether the bucket policy for the bucket requires server-side encryption of objects when objects are
     * uploaded to the bucket. Possible values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * FALSE - The bucket policy requires server-side encryption of new objects. PutObject requests must include the
     * x-amz-server-side-encryption header and the value for that header must be AES256 or aws:kms.
     * </p>
     * </li>
     * <li>
     * <p>
     * TRUE - The bucket doesn't have a bucket policy or it has a bucket policy that doesn't require server-side
     * encryption of new objects. If a bucket policy exists, it doesn't require PutObject requests to include the
     * x-amz-server-side-encryption header and it doesn't require the value for that header to be AES256 or aws:kms.
     * </p>
     * </li>
     * <li>
     * <p>
     * UNKNOWN - Amazon Macie can't determine whether the bucket policy requires server-side encryption of objects.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #allowsUnencryptedObjectUploads} will return {@link AllowsUnencryptedObjectUploads#UNKNOWN_TO_SDK_VERSION}
     * . The raw value returned by the service is available from {@link #allowsUnencryptedObjectUploadsAsString}.
     * </p>
     * 
     * @return Specifies whether the bucket policy for the bucket requires server-side encryption of objects when
     *         objects are uploaded to the bucket. Possible values are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         FALSE - The bucket policy requires server-side encryption of new objects. PutObject requests must include
     *         the x-amz-server-side-encryption header and the value for that header must be AES256 or aws:kms.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         TRUE - The bucket doesn't have a bucket policy or it has a bucket policy that doesn't require server-side
     *         encryption of new objects. If a bucket policy exists, it doesn't require PutObject requests to include
     *         the x-amz-server-side-encryption header and it doesn't require the value for that header to be AES256 or
     *         aws:kms.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         UNKNOWN - Amazon Macie can't determine whether the bucket policy requires server-side encryption of
     *         objects.
     *         </p>
     *         </li>
     * @see AllowsUnencryptedObjectUploads
     */
    public final AllowsUnencryptedObjectUploads allowsUnencryptedObjectUploads() {
        return AllowsUnencryptedObjectUploads.fromValue(allowsUnencryptedObjectUploads);
    }

    /**
     * <p>
     * Specifies whether the bucket policy for the bucket requires server-side encryption of objects when objects are
     * uploaded to the bucket. Possible values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * FALSE - The bucket policy requires server-side encryption of new objects. PutObject requests must include the
     * x-amz-server-side-encryption header and the value for that header must be AES256 or aws:kms.
     * </p>
     * </li>
     * <li>
     * <p>
     * TRUE - The bucket doesn't have a bucket policy or it has a bucket policy that doesn't require server-side
     * encryption of new objects. If a bucket policy exists, it doesn't require PutObject requests to include the
     * x-amz-server-side-encryption header and it doesn't require the value for that header to be AES256 or aws:kms.
     * </p>
     * </li>
     * <li>
     * <p>
     * UNKNOWN - Amazon Macie can't determine whether the bucket policy requires server-side encryption of objects.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #allowsUnencryptedObjectUploads} will return {@link AllowsUnencryptedObjectUploads#UNKNOWN_TO_SDK_VERSION}
     * . The raw value returned by the service is available from {@link #allowsUnencryptedObjectUploadsAsString}.
     * </p>
     * 
     * @return Specifies whether the bucket policy for the bucket requires server-side encryption of objects when
     *         objects are uploaded to the bucket. Possible values are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         FALSE - The bucket policy requires server-side encryption of new objects. PutObject requests must include
     *         the x-amz-server-side-encryption header and the value for that header must be AES256 or aws:kms.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         TRUE - The bucket doesn't have a bucket policy or it has a bucket policy that doesn't require server-side
     *         encryption of new objects. If a bucket policy exists, it doesn't require PutObject requests to include
     *         the x-amz-server-side-encryption header and it doesn't require the value for that header to be AES256 or
     *         aws:kms.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         UNKNOWN - Amazon Macie can't determine whether the bucket policy requires server-side encryption of
     *         objects.
     *         </p>
     *         </li>
     * @see AllowsUnencryptedObjectUploads
     */
    public final String allowsUnencryptedObjectUploadsAsString() {
        return allowsUnencryptedObjectUploads;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the bucket.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the bucket.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The date and time, in UTC and extended ISO 8601 format, when the bucket was created.
     * </p>
     * 
     * @return The date and time, in UTC and extended ISO 8601 format, when the bucket was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The type of server-side encryption that's used by default to encrypt objects in the bucket.
     * </p>
     * 
     * @return The type of server-side encryption that's used by default to encrypt objects in the bucket.
     */
    public final ServerSideEncryption defaultServerSideEncryption() {
        return defaultServerSideEncryption;
    }

    /**
     * <p>
     * The name of the bucket.
     * </p>
     * 
     * @return The name of the bucket.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The display name and Amazon Web Services account ID for the user who owns the bucket.
     * </p>
     * 
     * @return The display name and Amazon Web Services account ID for the user who owns the bucket.
     */
    public final S3BucketOwner owner() {
        return owner;
    }

    /**
     * <p>
     * The permissions settings that determine whether the bucket is publicly accessible.
     * </p>
     * 
     * @return The permissions settings that determine whether the bucket is publicly accessible.
     */
    public final BucketPublicAccess publicAccess() {
        return publicAccess;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags that are associated with the bucket.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags that are associated with the bucket.
     */
    public final List<KeyValuePair> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(allowsUnencryptedObjectUploadsAsString());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(defaultServerSideEncryption());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(owner());
        hashCode = 31 * hashCode + Objects.hashCode(publicAccess());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof S3Bucket)) {
            return false;
        }
        S3Bucket other = (S3Bucket) obj;
        return Objects.equals(allowsUnencryptedObjectUploadsAsString(), other.allowsUnencryptedObjectUploadsAsString())
                && Objects.equals(arn(), other.arn()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(defaultServerSideEncryption(), other.defaultServerSideEncryption())
                && Objects.equals(name(), other.name()) && Objects.equals(owner(), other.owner())
                && Objects.equals(publicAccess(), other.publicAccess()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("S3Bucket").add("AllowsUnencryptedObjectUploads", allowsUnencryptedObjectUploadsAsString())
                .add("Arn", arn()).add("CreatedAt", createdAt())
                .add("DefaultServerSideEncryption", defaultServerSideEncryption()).add("Name", name()).add("Owner", owner())
                .add("PublicAccess", publicAccess()).add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "allowsUnencryptedObjectUploads":
            return Optional.ofNullable(clazz.cast(allowsUnencryptedObjectUploadsAsString()));
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "defaultServerSideEncryption":
            return Optional.ofNullable(clazz.cast(defaultServerSideEncryption()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "owner":
            return Optional.ofNullable(clazz.cast(owner()));
        case "publicAccess":
            return Optional.ofNullable(clazz.cast(publicAccess()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<S3Bucket, T> g) {
        return obj -> g.apply((S3Bucket) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, S3Bucket> {
        /**
         * <p>
         * Specifies whether the bucket policy for the bucket requires server-side encryption of objects when objects
         * are uploaded to the bucket. Possible values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * FALSE - The bucket policy requires server-side encryption of new objects. PutObject requests must include the
         * x-amz-server-side-encryption header and the value for that header must be AES256 or aws:kms.
         * </p>
         * </li>
         * <li>
         * <p>
         * TRUE - The bucket doesn't have a bucket policy or it has a bucket policy that doesn't require server-side
         * encryption of new objects. If a bucket policy exists, it doesn't require PutObject requests to include the
         * x-amz-server-side-encryption header and it doesn't require the value for that header to be AES256 or aws:kms.
         * </p>
         * </li>
         * <li>
         * <p>
         * UNKNOWN - Amazon Macie can't determine whether the bucket policy requires server-side encryption of objects.
         * </p>
         * </li>
         * </ul>
         * 
         * @param allowsUnencryptedObjectUploads
         *        Specifies whether the bucket policy for the bucket requires server-side encryption of objects when
         *        objects are uploaded to the bucket. Possible values are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        FALSE - The bucket policy requires server-side encryption of new objects. PutObject requests must
         *        include the x-amz-server-side-encryption header and the value for that header must be AES256 or
         *        aws:kms.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        TRUE - The bucket doesn't have a bucket policy or it has a bucket policy that doesn't require
         *        server-side encryption of new objects. If a bucket policy exists, it doesn't require PutObject
         *        requests to include the x-amz-server-side-encryption header and it doesn't require the value for that
         *        header to be AES256 or aws:kms.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        UNKNOWN - Amazon Macie can't determine whether the bucket policy requires server-side encryption of
         *        objects.
         *        </p>
         *        </li>
         * @see AllowsUnencryptedObjectUploads
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AllowsUnencryptedObjectUploads
         */
        Builder allowsUnencryptedObjectUploads(String allowsUnencryptedObjectUploads);

        /**
         * <p>
         * Specifies whether the bucket policy for the bucket requires server-side encryption of objects when objects
         * are uploaded to the bucket. Possible values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * FALSE - The bucket policy requires server-side encryption of new objects. PutObject requests must include the
         * x-amz-server-side-encryption header and the value for that header must be AES256 or aws:kms.
         * </p>
         * </li>
         * <li>
         * <p>
         * TRUE - The bucket doesn't have a bucket policy or it has a bucket policy that doesn't require server-side
         * encryption of new objects. If a bucket policy exists, it doesn't require PutObject requests to include the
         * x-amz-server-side-encryption header and it doesn't require the value for that header to be AES256 or aws:kms.
         * </p>
         * </li>
         * <li>
         * <p>
         * UNKNOWN - Amazon Macie can't determine whether the bucket policy requires server-side encryption of objects.
         * </p>
         * </li>
         * </ul>
         * 
         * @param allowsUnencryptedObjectUploads
         *        Specifies whether the bucket policy for the bucket requires server-side encryption of objects when
         *        objects are uploaded to the bucket. Possible values are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        FALSE - The bucket policy requires server-side encryption of new objects. PutObject requests must
         *        include the x-amz-server-side-encryption header and the value for that header must be AES256 or
         *        aws:kms.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        TRUE - The bucket doesn't have a bucket policy or it has a bucket policy that doesn't require
         *        server-side encryption of new objects. If a bucket policy exists, it doesn't require PutObject
         *        requests to include the x-amz-server-side-encryption header and it doesn't require the value for that
         *        header to be AES256 or aws:kms.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        UNKNOWN - Amazon Macie can't determine whether the bucket policy requires server-side encryption of
         *        objects.
         *        </p>
         *        </li>
         * @see AllowsUnencryptedObjectUploads
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AllowsUnencryptedObjectUploads
         */
        Builder allowsUnencryptedObjectUploads(AllowsUnencryptedObjectUploads allowsUnencryptedObjectUploads);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the bucket.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The date and time, in UTC and extended ISO 8601 format, when the bucket was created.
         * </p>
         * 
         * @param createdAt
         *        The date and time, in UTC and extended ISO 8601 format, when the bucket was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The type of server-side encryption that's used by default to encrypt objects in the bucket.
         * </p>
         * 
         * @param defaultServerSideEncryption
         *        The type of server-side encryption that's used by default to encrypt objects in the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultServerSideEncryption(ServerSideEncryption defaultServerSideEncryption);

        /**
         * <p>
         * The type of server-side encryption that's used by default to encrypt objects in the bucket.
         * </p>
         * This is a convenience that creates an instance of the {@link ServerSideEncryption.Builder} avoiding the need
         * to create one manually via {@link ServerSideEncryption#builder()}.
         *
         * When the {@link Consumer} completes, {@link ServerSideEncryption.Builder#build()} is called immediately and
         * its result is passed to {@link #defaultServerSideEncryption(ServerSideEncryption)}.
         * 
         * @param defaultServerSideEncryption
         *        a consumer that will call methods on {@link ServerSideEncryption.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #defaultServerSideEncryption(ServerSideEncryption)
         */
        default Builder defaultServerSideEncryption(Consumer<ServerSideEncryption.Builder> defaultServerSideEncryption) {
            return defaultServerSideEncryption(ServerSideEncryption.builder().applyMutation(defaultServerSideEncryption).build());
        }

        /**
         * <p>
         * The name of the bucket.
         * </p>
         * 
         * @param name
         *        The name of the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The display name and Amazon Web Services account ID for the user who owns the bucket.
         * </p>
         * 
         * @param owner
         *        The display name and Amazon Web Services account ID for the user who owns the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder owner(S3BucketOwner owner);

        /**
         * <p>
         * The display name and Amazon Web Services account ID for the user who owns the bucket.
         * </p>
         * This is a convenience that creates an instance of the {@link S3BucketOwner.Builder} avoiding the need to
         * create one manually via {@link S3BucketOwner#builder()}.
         *
         * When the {@link Consumer} completes, {@link S3BucketOwner.Builder#build()} is called immediately and its
         * result is passed to {@link #owner(S3BucketOwner)}.
         * 
         * @param owner
         *        a consumer that will call methods on {@link S3BucketOwner.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #owner(S3BucketOwner)
         */
        default Builder owner(Consumer<S3BucketOwner.Builder> owner) {
            return owner(S3BucketOwner.builder().applyMutation(owner).build());
        }

        /**
         * <p>
         * The permissions settings that determine whether the bucket is publicly accessible.
         * </p>
         * 
         * @param publicAccess
         *        The permissions settings that determine whether the bucket is publicly accessible.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder publicAccess(BucketPublicAccess publicAccess);

        /**
         * <p>
         * The permissions settings that determine whether the bucket is publicly accessible.
         * </p>
         * This is a convenience that creates an instance of the {@link BucketPublicAccess.Builder} avoiding the need to
         * create one manually via {@link BucketPublicAccess#builder()}.
         *
         * When the {@link Consumer} completes, {@link BucketPublicAccess.Builder#build()} is called immediately and its
         * result is passed to {@link #publicAccess(BucketPublicAccess)}.
         * 
         * @param publicAccess
         *        a consumer that will call methods on {@link BucketPublicAccess.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #publicAccess(BucketPublicAccess)
         */
        default Builder publicAccess(Consumer<BucketPublicAccess.Builder> publicAccess) {
            return publicAccess(BucketPublicAccess.builder().applyMutation(publicAccess).build());
        }

        /**
         * <p>
         * The tags that are associated with the bucket.
         * </p>
         * 
         * @param tags
         *        The tags that are associated with the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<KeyValuePair> tags);

        /**
         * <p>
         * The tags that are associated with the bucket.
         * </p>
         * 
         * @param tags
         *        The tags that are associated with the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(KeyValuePair... tags);

        /**
         * <p>
         * The tags that are associated with the bucket.
         * </p>
         * This is a convenience that creates an instance of the {@link List<KeyValuePair>.Builder} avoiding the need to
         * create one manually via {@link List<KeyValuePair>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<KeyValuePair>.Builder#build()} is called immediately and its
         * result is passed to {@link #tags(List<KeyValuePair>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<KeyValuePair>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<KeyValuePair>)
         */
        Builder tags(Consumer<KeyValuePair.Builder>... tags);
    }

    static final class BuilderImpl implements Builder {
        private String allowsUnencryptedObjectUploads;

        private String arn;

        private Instant createdAt;

        private ServerSideEncryption defaultServerSideEncryption;

        private String name;

        private S3BucketOwner owner;

        private BucketPublicAccess publicAccess;

        private List<KeyValuePair> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(S3Bucket model) {
            allowsUnencryptedObjectUploads(model.allowsUnencryptedObjectUploads);
            arn(model.arn);
            createdAt(model.createdAt);
            defaultServerSideEncryption(model.defaultServerSideEncryption);
            name(model.name);
            owner(model.owner);
            publicAccess(model.publicAccess);
            tags(model.tags);
        }

        public final String getAllowsUnencryptedObjectUploads() {
            return allowsUnencryptedObjectUploads;
        }

        public final void setAllowsUnencryptedObjectUploads(String allowsUnencryptedObjectUploads) {
            this.allowsUnencryptedObjectUploads = allowsUnencryptedObjectUploads;
        }

        @Override
        @Transient
        public final Builder allowsUnencryptedObjectUploads(String allowsUnencryptedObjectUploads) {
            this.allowsUnencryptedObjectUploads = allowsUnencryptedObjectUploads;
            return this;
        }

        @Override
        @Transient
        public final Builder allowsUnencryptedObjectUploads(AllowsUnencryptedObjectUploads allowsUnencryptedObjectUploads) {
            this.allowsUnencryptedObjectUploads(allowsUnencryptedObjectUploads == null ? null : allowsUnencryptedObjectUploads
                    .toString());
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        @Transient
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        @Transient
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final ServerSideEncryption.Builder getDefaultServerSideEncryption() {
            return defaultServerSideEncryption != null ? defaultServerSideEncryption.toBuilder() : null;
        }

        public final void setDefaultServerSideEncryption(ServerSideEncryption.BuilderImpl defaultServerSideEncryption) {
            this.defaultServerSideEncryption = defaultServerSideEncryption != null ? defaultServerSideEncryption.build() : null;
        }

        @Override
        @Transient
        public final Builder defaultServerSideEncryption(ServerSideEncryption defaultServerSideEncryption) {
            this.defaultServerSideEncryption = defaultServerSideEncryption;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        @Transient
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final S3BucketOwner.Builder getOwner() {
            return owner != null ? owner.toBuilder() : null;
        }

        public final void setOwner(S3BucketOwner.BuilderImpl owner) {
            this.owner = owner != null ? owner.build() : null;
        }

        @Override
        @Transient
        public final Builder owner(S3BucketOwner owner) {
            this.owner = owner;
            return this;
        }

        public final BucketPublicAccess.Builder getPublicAccess() {
            return publicAccess != null ? publicAccess.toBuilder() : null;
        }

        public final void setPublicAccess(BucketPublicAccess.BuilderImpl publicAccess) {
            this.publicAccess = publicAccess != null ? publicAccess.build() : null;
        }

        @Override
        @Transient
        public final Builder publicAccess(BucketPublicAccess publicAccess) {
            this.publicAccess = publicAccess;
            return this;
        }

        public final List<KeyValuePair.Builder> getTags() {
            List<KeyValuePair.Builder> result = KeyValuePairListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<KeyValuePair.BuilderImpl> tags) {
            this.tags = KeyValuePairListCopier.copyFromBuilder(tags);
        }

        @Override
        @Transient
        public final Builder tags(Collection<KeyValuePair> tags) {
            this.tags = KeyValuePairListCopier.copy(tags);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder tags(KeyValuePair... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder tags(Consumer<KeyValuePair.Builder>... tags) {
            tags(Stream.of(tags).map(c -> KeyValuePair.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public S3Bucket build() {
            return new S3Bucket(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
