/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.macie2.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides information about an S3 bucket that Amazon Macie monitors and analyzes.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class BucketMetadata implements SdkPojo, Serializable, ToCopyableBuilder<BucketMetadata.Builder, BucketMetadata> {
    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("accountId").getter(getter(BucketMetadata::accountId)).setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("accountId").build()).build();

    private static final SdkField<String> ALLOWS_UNENCRYPTED_OBJECT_UPLOADS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("allowsUnencryptedObjectUploads")
            .getter(getter(BucketMetadata::allowsUnencryptedObjectUploadsAsString))
            .setter(setter(Builder::allowsUnencryptedObjectUploads))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("allowsUnencryptedObjectUploads")
                    .build()).build();

    private static final SdkField<String> BUCKET_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("bucketArn").getter(getter(BucketMetadata::bucketArn)).setter(setter(Builder::bucketArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("bucketArn").build()).build();

    private static final SdkField<Instant> BUCKET_CREATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("bucketCreatedAt")
            .getter(getter(BucketMetadata::bucketCreatedAt))
            .setter(setter(Builder::bucketCreatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("bucketCreatedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> BUCKET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("bucketName").getter(getter(BucketMetadata::bucketName)).setter(setter(Builder::bucketName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("bucketName").build()).build();

    private static final SdkField<Long> CLASSIFIABLE_OBJECT_COUNT_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("classifiableObjectCount").getter(getter(BucketMetadata::classifiableObjectCount))
            .setter(setter(Builder::classifiableObjectCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("classifiableObjectCount").build())
            .build();

    private static final SdkField<Long> CLASSIFIABLE_SIZE_IN_BYTES_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("classifiableSizeInBytes").getter(getter(BucketMetadata::classifiableSizeInBytes))
            .setter(setter(Builder::classifiableSizeInBytes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("classifiableSizeInBytes").build())
            .build();

    private static final SdkField<JobDetails> JOB_DETAILS_FIELD = SdkField.<JobDetails> builder(MarshallingType.SDK_POJO)
            .memberName("jobDetails").getter(getter(BucketMetadata::jobDetails)).setter(setter(Builder::jobDetails))
            .constructor(JobDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("jobDetails").build()).build();

    private static final SdkField<Instant> LAST_UPDATED_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastUpdated")
            .getter(getter(BucketMetadata::lastUpdated))
            .setter(setter(Builder::lastUpdated))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastUpdated").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Long> OBJECT_COUNT_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("objectCount").getter(getter(BucketMetadata::objectCount)).setter(setter(Builder::objectCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("objectCount").build()).build();

    private static final SdkField<ObjectCountByEncryptionType> OBJECT_COUNT_BY_ENCRYPTION_TYPE_FIELD = SdkField
            .<ObjectCountByEncryptionType> builder(MarshallingType.SDK_POJO)
            .memberName("objectCountByEncryptionType")
            .getter(getter(BucketMetadata::objectCountByEncryptionType))
            .setter(setter(Builder::objectCountByEncryptionType))
            .constructor(ObjectCountByEncryptionType::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("objectCountByEncryptionType")
                    .build()).build();

    private static final SdkField<BucketPublicAccess> PUBLIC_ACCESS_FIELD = SdkField
            .<BucketPublicAccess> builder(MarshallingType.SDK_POJO).memberName("publicAccess")
            .getter(getter(BucketMetadata::publicAccess)).setter(setter(Builder::publicAccess))
            .constructor(BucketPublicAccess::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("publicAccess").build()).build();

    private static final SdkField<String> REGION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("region")
            .getter(getter(BucketMetadata::region)).setter(setter(Builder::region))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("region").build()).build();

    private static final SdkField<ReplicationDetails> REPLICATION_DETAILS_FIELD = SdkField
            .<ReplicationDetails> builder(MarshallingType.SDK_POJO).memberName("replicationDetails")
            .getter(getter(BucketMetadata::replicationDetails)).setter(setter(Builder::replicationDetails))
            .constructor(ReplicationDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("replicationDetails").build())
            .build();

    private static final SdkField<BucketServerSideEncryption> SERVER_SIDE_ENCRYPTION_FIELD = SdkField
            .<BucketServerSideEncryption> builder(MarshallingType.SDK_POJO).memberName("serverSideEncryption")
            .getter(getter(BucketMetadata::serverSideEncryption)).setter(setter(Builder::serverSideEncryption))
            .constructor(BucketServerSideEncryption::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("serverSideEncryption").build())
            .build();

    private static final SdkField<String> SHARED_ACCESS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("sharedAccess").getter(getter(BucketMetadata::sharedAccessAsString))
            .setter(setter(Builder::sharedAccess))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sharedAccess").build()).build();

    private static final SdkField<Long> SIZE_IN_BYTES_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("sizeInBytes").getter(getter(BucketMetadata::sizeInBytes)).setter(setter(Builder::sizeInBytes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sizeInBytes").build()).build();

    private static final SdkField<Long> SIZE_IN_BYTES_COMPRESSED_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("sizeInBytesCompressed").getter(getter(BucketMetadata::sizeInBytesCompressed))
            .setter(setter(Builder::sizeInBytesCompressed))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sizeInBytesCompressed").build())
            .build();

    private static final SdkField<List<KeyValuePair>> TAGS_FIELD = SdkField
            .<List<KeyValuePair>> builder(MarshallingType.LIST)
            .memberName("tags")
            .getter(getter(BucketMetadata::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<KeyValuePair> builder(MarshallingType.SDK_POJO)
                                            .constructor(KeyValuePair::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<ObjectLevelStatistics> UNCLASSIFIABLE_OBJECT_COUNT_FIELD = SdkField
            .<ObjectLevelStatistics> builder(MarshallingType.SDK_POJO).memberName("unclassifiableObjectCount")
            .getter(getter(BucketMetadata::unclassifiableObjectCount)).setter(setter(Builder::unclassifiableObjectCount))
            .constructor(ObjectLevelStatistics::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("unclassifiableObjectCount").build())
            .build();

    private static final SdkField<ObjectLevelStatistics> UNCLASSIFIABLE_OBJECT_SIZE_IN_BYTES_FIELD = SdkField
            .<ObjectLevelStatistics> builder(MarshallingType.SDK_POJO)
            .memberName("unclassifiableObjectSizeInBytes")
            .getter(getter(BucketMetadata::unclassifiableObjectSizeInBytes))
            .setter(setter(Builder::unclassifiableObjectSizeInBytes))
            .constructor(ObjectLevelStatistics::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("unclassifiableObjectSizeInBytes")
                    .build()).build();

    private static final SdkField<Boolean> VERSIONING_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("versioning").getter(getter(BucketMetadata::versioning)).setter(setter(Builder::versioning))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("versioning").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACCOUNT_ID_FIELD,
            ALLOWS_UNENCRYPTED_OBJECT_UPLOADS_FIELD, BUCKET_ARN_FIELD, BUCKET_CREATED_AT_FIELD, BUCKET_NAME_FIELD,
            CLASSIFIABLE_OBJECT_COUNT_FIELD, CLASSIFIABLE_SIZE_IN_BYTES_FIELD, JOB_DETAILS_FIELD, LAST_UPDATED_FIELD,
            OBJECT_COUNT_FIELD, OBJECT_COUNT_BY_ENCRYPTION_TYPE_FIELD, PUBLIC_ACCESS_FIELD, REGION_FIELD,
            REPLICATION_DETAILS_FIELD, SERVER_SIDE_ENCRYPTION_FIELD, SHARED_ACCESS_FIELD, SIZE_IN_BYTES_FIELD,
            SIZE_IN_BYTES_COMPRESSED_FIELD, TAGS_FIELD, UNCLASSIFIABLE_OBJECT_COUNT_FIELD,
            UNCLASSIFIABLE_OBJECT_SIZE_IN_BYTES_FIELD, VERSIONING_FIELD));

    private static final long serialVersionUID = 1L;

    private final String accountId;

    private final String allowsUnencryptedObjectUploads;

    private final String bucketArn;

    private final Instant bucketCreatedAt;

    private final String bucketName;

    private final Long classifiableObjectCount;

    private final Long classifiableSizeInBytes;

    private final JobDetails jobDetails;

    private final Instant lastUpdated;

    private final Long objectCount;

    private final ObjectCountByEncryptionType objectCountByEncryptionType;

    private final BucketPublicAccess publicAccess;

    private final String region;

    private final ReplicationDetails replicationDetails;

    private final BucketServerSideEncryption serverSideEncryption;

    private final String sharedAccess;

    private final Long sizeInBytes;

    private final Long sizeInBytesCompressed;

    private final List<KeyValuePair> tags;

    private final ObjectLevelStatistics unclassifiableObjectCount;

    private final ObjectLevelStatistics unclassifiableObjectSizeInBytes;

    private final Boolean versioning;

    private BucketMetadata(BuilderImpl builder) {
        this.accountId = builder.accountId;
        this.allowsUnencryptedObjectUploads = builder.allowsUnencryptedObjectUploads;
        this.bucketArn = builder.bucketArn;
        this.bucketCreatedAt = builder.bucketCreatedAt;
        this.bucketName = builder.bucketName;
        this.classifiableObjectCount = builder.classifiableObjectCount;
        this.classifiableSizeInBytes = builder.classifiableSizeInBytes;
        this.jobDetails = builder.jobDetails;
        this.lastUpdated = builder.lastUpdated;
        this.objectCount = builder.objectCount;
        this.objectCountByEncryptionType = builder.objectCountByEncryptionType;
        this.publicAccess = builder.publicAccess;
        this.region = builder.region;
        this.replicationDetails = builder.replicationDetails;
        this.serverSideEncryption = builder.serverSideEncryption;
        this.sharedAccess = builder.sharedAccess;
        this.sizeInBytes = builder.sizeInBytes;
        this.sizeInBytesCompressed = builder.sizeInBytesCompressed;
        this.tags = builder.tags;
        this.unclassifiableObjectCount = builder.unclassifiableObjectCount;
        this.unclassifiableObjectSizeInBytes = builder.unclassifiableObjectSizeInBytes;
        this.versioning = builder.versioning;
    }

    /**
     * <p>
     * The unique identifier for the Amazon Web Services account that owns the bucket.
     * </p>
     * 
     * @return The unique identifier for the Amazon Web Services account that owns the bucket.
     */
    public final String accountId() {
        return accountId;
    }

    /**
     * <p>
     * Specifies whether the bucket policy for the bucket requires server-side encryption of objects when objects are
     * uploaded to the bucket. Possible values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * FALSE - The bucket policy requires server-side encryption of new objects. PutObject requests must include the
     * x-amz-server-side-encryption header and the value for that header must be AES256 or aws:kms.
     * </p>
     * </li>
     * <li>
     * <p>
     * TRUE - The bucket doesn't have a bucket policy or it has a bucket policy that doesn't require server-side
     * encryption of new objects. If a bucket policy exists, it doesn't require PutObject requests to include the
     * x-amz-server-side-encryption header and it doesn't require the value for that header to be AES256 or aws:kms.
     * </p>
     * </li>
     * <li>
     * <p>
     * UNKNOWN - Amazon Macie can't determine whether the bucket policy requires server-side encryption of new objects.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #allowsUnencryptedObjectUploads} will return {@link AllowsUnencryptedObjectUploads#UNKNOWN_TO_SDK_VERSION}
     * . The raw value returned by the service is available from {@link #allowsUnencryptedObjectUploadsAsString}.
     * </p>
     * 
     * @return Specifies whether the bucket policy for the bucket requires server-side encryption of objects when
     *         objects are uploaded to the bucket. Possible values are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         FALSE - The bucket policy requires server-side encryption of new objects. PutObject requests must include
     *         the x-amz-server-side-encryption header and the value for that header must be AES256 or aws:kms.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         TRUE - The bucket doesn't have a bucket policy or it has a bucket policy that doesn't require server-side
     *         encryption of new objects. If a bucket policy exists, it doesn't require PutObject requests to include
     *         the x-amz-server-side-encryption header and it doesn't require the value for that header to be AES256 or
     *         aws:kms.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         UNKNOWN - Amazon Macie can't determine whether the bucket policy requires server-side encryption of new
     *         objects.
     *         </p>
     *         </li>
     * @see AllowsUnencryptedObjectUploads
     */
    public final AllowsUnencryptedObjectUploads allowsUnencryptedObjectUploads() {
        return AllowsUnencryptedObjectUploads.fromValue(allowsUnencryptedObjectUploads);
    }

    /**
     * <p>
     * Specifies whether the bucket policy for the bucket requires server-side encryption of objects when objects are
     * uploaded to the bucket. Possible values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * FALSE - The bucket policy requires server-side encryption of new objects. PutObject requests must include the
     * x-amz-server-side-encryption header and the value for that header must be AES256 or aws:kms.
     * </p>
     * </li>
     * <li>
     * <p>
     * TRUE - The bucket doesn't have a bucket policy or it has a bucket policy that doesn't require server-side
     * encryption of new objects. If a bucket policy exists, it doesn't require PutObject requests to include the
     * x-amz-server-side-encryption header and it doesn't require the value for that header to be AES256 or aws:kms.
     * </p>
     * </li>
     * <li>
     * <p>
     * UNKNOWN - Amazon Macie can't determine whether the bucket policy requires server-side encryption of new objects.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #allowsUnencryptedObjectUploads} will return {@link AllowsUnencryptedObjectUploads#UNKNOWN_TO_SDK_VERSION}
     * . The raw value returned by the service is available from {@link #allowsUnencryptedObjectUploadsAsString}.
     * </p>
     * 
     * @return Specifies whether the bucket policy for the bucket requires server-side encryption of objects when
     *         objects are uploaded to the bucket. Possible values are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         FALSE - The bucket policy requires server-side encryption of new objects. PutObject requests must include
     *         the x-amz-server-side-encryption header and the value for that header must be AES256 or aws:kms.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         TRUE - The bucket doesn't have a bucket policy or it has a bucket policy that doesn't require server-side
     *         encryption of new objects. If a bucket policy exists, it doesn't require PutObject requests to include
     *         the x-amz-server-side-encryption header and it doesn't require the value for that header to be AES256 or
     *         aws:kms.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         UNKNOWN - Amazon Macie can't determine whether the bucket policy requires server-side encryption of new
     *         objects.
     *         </p>
     *         </li>
     * @see AllowsUnencryptedObjectUploads
     */
    public final String allowsUnencryptedObjectUploadsAsString() {
        return allowsUnencryptedObjectUploads;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the bucket.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the bucket.
     */
    public final String bucketArn() {
        return bucketArn;
    }

    /**
     * <p>
     * The date and time, in UTC and extended ISO 8601 format, when the bucket was created.
     * </p>
     * 
     * @return The date and time, in UTC and extended ISO 8601 format, when the bucket was created.
     */
    public final Instant bucketCreatedAt() {
        return bucketCreatedAt;
    }

    /**
     * <p>
     * The name of the bucket.
     * </p>
     * 
     * @return The name of the bucket.
     */
    public final String bucketName() {
        return bucketName;
    }

    /**
     * <p>
     * The total number of objects that Amazon Macie can analyze in the bucket. These objects use a supported storage
     * class and have a file name extension for a supported file or storage format.
     * </p>
     * 
     * @return The total number of objects that Amazon Macie can analyze in the bucket. These objects use a supported
     *         storage class and have a file name extension for a supported file or storage format.
     */
    public final Long classifiableObjectCount() {
        return classifiableObjectCount;
    }

    /**
     * <p>
     * The total storage size, in bytes, of the objects that Amazon Macie can analyze in the bucket. These objects use a
     * supported storage class and have a file name extension for a supported file or storage format.
     * </p>
     * <p>
     * If versioning is enabled for the bucket, Macie calculates this value based on the size of the latest version of
     * each applicable object in the bucket. This value doesn't reflect the storage size of all versions of each
     * applicable object in the bucket.
     * </p>
     * 
     * @return The total storage size, in bytes, of the objects that Amazon Macie can analyze in the bucket. These
     *         objects use a supported storage class and have a file name extension for a supported file or storage
     *         format.</p>
     *         <p>
     *         If versioning is enabled for the bucket, Macie calculates this value based on the size of the latest
     *         version of each applicable object in the bucket. This value doesn't reflect the storage size of all
     *         versions of each applicable object in the bucket.
     */
    public final Long classifiableSizeInBytes() {
        return classifiableSizeInBytes;
    }

    /**
     * <p>
     * Specifies whether any one-time or recurring classification jobs are configured to analyze data in the bucket,
     * and, if so, the details of the job that ran most recently.
     * </p>
     * 
     * @return Specifies whether any one-time or recurring classification jobs are configured to analyze data in the
     *         bucket, and, if so, the details of the job that ran most recently.
     */
    public final JobDetails jobDetails() {
        return jobDetails;
    }

    /**
     * <p>
     * The date and time, in UTC and extended ISO 8601 format, when Amazon Macie most recently retrieved both bucket and
     * object metadata from Amazon S3 for the bucket.
     * </p>
     * 
     * @return The date and time, in UTC and extended ISO 8601 format, when Amazon Macie most recently retrieved both
     *         bucket and object metadata from Amazon S3 for the bucket.
     */
    public final Instant lastUpdated() {
        return lastUpdated;
    }

    /**
     * <p>
     * The total number of objects in the bucket.
     * </p>
     * 
     * @return The total number of objects in the bucket.
     */
    public final Long objectCount() {
        return objectCount;
    }

    /**
     * <p>
     * The total number of objects that are in the bucket, grouped by server-side encryption type. This includes a
     * grouping that reports the total number of objects that aren't encrypted or use client-side encryption.
     * </p>
     * 
     * @return The total number of objects that are in the bucket, grouped by server-side encryption type. This includes
     *         a grouping that reports the total number of objects that aren't encrypted or use client-side encryption.
     */
    public final ObjectCountByEncryptionType objectCountByEncryptionType() {
        return objectCountByEncryptionType;
    }

    /**
     * <p>
     * Specifies whether the bucket is publicly accessible due to the combination of permissions settings that apply to
     * the bucket, and provides information about those settings.
     * </p>
     * 
     * @return Specifies whether the bucket is publicly accessible due to the combination of permissions settings that
     *         apply to the bucket, and provides information about those settings.
     */
    public final BucketPublicAccess publicAccess() {
        return publicAccess;
    }

    /**
     * <p>
     * The Amazon Web Services Region that hosts the bucket.
     * </p>
     * 
     * @return The Amazon Web Services Region that hosts the bucket.
     */
    public final String region() {
        return region;
    }

    /**
     * <p>
     * Specifies whether the bucket is configured to replicate one or more objects to buckets for other Amazon Web
     * Services accounts and, if so, which accounts.
     * </p>
     * 
     * @return Specifies whether the bucket is configured to replicate one or more objects to buckets for other Amazon
     *         Web Services accounts and, if so, which accounts.
     */
    public final ReplicationDetails replicationDetails() {
        return replicationDetails;
    }

    /**
     * <p>
     * Specifies whether the bucket encrypts new objects by default and, if so, the type of server-side encryption
     * that's used.
     * </p>
     * 
     * @return Specifies whether the bucket encrypts new objects by default and, if so, the type of server-side
     *         encryption that's used.
     */
    public final BucketServerSideEncryption serverSideEncryption() {
        return serverSideEncryption;
    }

    /**
     * <p>
     * Specifies whether the bucket is shared with another Amazon Web Services account. Possible values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * EXTERNAL - The bucket is shared with an Amazon Web Services account that isn't part of the same Amazon Macie
     * organization.
     * </p>
     * </li>
     * <li>
     * <p>
     * INTERNAL - The bucket is shared with an Amazon Web Services account that's part of the same Amazon Macie
     * organization.
     * </p>
     * </li>
     * <li>
     * <p>
     * NOT_SHARED - The bucket isn't shared with other Amazon Web Services accounts.
     * </p>
     * </li>
     * <li>
     * <p>
     * UNKNOWN - Amazon Macie wasn't able to evaluate the shared access settings for the bucket.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #sharedAccess} will
     * return {@link SharedAccess#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #sharedAccessAsString}.
     * </p>
     * 
     * @return <p>
     *         Specifies whether the bucket is shared with another Amazon Web Services account. Possible values are:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         EXTERNAL - The bucket is shared with an Amazon Web Services account that isn't part of the same Amazon
     *         Macie organization.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         INTERNAL - The bucket is shared with an Amazon Web Services account that's part of the same Amazon Macie
     *         organization.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         NOT_SHARED - The bucket isn't shared with other Amazon Web Services accounts.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         UNKNOWN - Amazon Macie wasn't able to evaluate the shared access settings for the bucket.
     *         </p>
     *         </li>
     *         </ul>
     * @see SharedAccess
     */
    public final SharedAccess sharedAccess() {
        return SharedAccess.fromValue(sharedAccess);
    }

    /**
     * <p>
     * Specifies whether the bucket is shared with another Amazon Web Services account. Possible values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * EXTERNAL - The bucket is shared with an Amazon Web Services account that isn't part of the same Amazon Macie
     * organization.
     * </p>
     * </li>
     * <li>
     * <p>
     * INTERNAL - The bucket is shared with an Amazon Web Services account that's part of the same Amazon Macie
     * organization.
     * </p>
     * </li>
     * <li>
     * <p>
     * NOT_SHARED - The bucket isn't shared with other Amazon Web Services accounts.
     * </p>
     * </li>
     * <li>
     * <p>
     * UNKNOWN - Amazon Macie wasn't able to evaluate the shared access settings for the bucket.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #sharedAccess} will
     * return {@link SharedAccess#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #sharedAccessAsString}.
     * </p>
     * 
     * @return <p>
     *         Specifies whether the bucket is shared with another Amazon Web Services account. Possible values are:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         EXTERNAL - The bucket is shared with an Amazon Web Services account that isn't part of the same Amazon
     *         Macie organization.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         INTERNAL - The bucket is shared with an Amazon Web Services account that's part of the same Amazon Macie
     *         organization.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         NOT_SHARED - The bucket isn't shared with other Amazon Web Services accounts.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         UNKNOWN - Amazon Macie wasn't able to evaluate the shared access settings for the bucket.
     *         </p>
     *         </li>
     *         </ul>
     * @see SharedAccess
     */
    public final String sharedAccessAsString() {
        return sharedAccess;
    }

    /**
     * <p>
     * The total storage size, in bytes, of the bucket.
     * </p>
     * <p>
     * If versioning is enabled for the bucket, Amazon Macie calculates this value based on the size of the latest
     * version of each object in the bucket. This value doesn't reflect the storage size of all versions of each object
     * in the bucket.
     * </p>
     * 
     * @return The total storage size, in bytes, of the bucket.</p>
     *         <p>
     *         If versioning is enabled for the bucket, Amazon Macie calculates this value based on the size of the
     *         latest version of each object in the bucket. This value doesn't reflect the storage size of all versions
     *         of each object in the bucket.
     */
    public final Long sizeInBytes() {
        return sizeInBytes;
    }

    /**
     * <p>
     * The total storage size, in bytes, of the objects that are compressed (.gz, .gzip, .zip) files in the bucket.
     * </p>
     * <p>
     * If versioning is enabled for the bucket, Macie calculates this value based on the size of the latest version of
     * each applicable object in the bucket. This value doesn't reflect the storage size of all versions of each
     * applicable object in the bucket.
     * </p>
     * 
     * @return The total storage size, in bytes, of the objects that are compressed (.gz, .gzip, .zip) files in the
     *         bucket.</p>
     *         <p>
     *         If versioning is enabled for the bucket, Macie calculates this value based on the size of the latest
     *         version of each applicable object in the bucket. This value doesn't reflect the storage size of all
     *         versions of each applicable object in the bucket.
     */
    public final Long sizeInBytesCompressed() {
        return sizeInBytesCompressed;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array that specifies the tags (keys and values) that are associated with the bucket.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return An array that specifies the tags (keys and values) that are associated with the bucket.
     */
    public final List<KeyValuePair> tags() {
        return tags;
    }

    /**
     * <p>
     * The total number of objects that Amazon Macie can't analyze in the bucket. These objects don't use a supported
     * storage class or don't have a file name extension for a supported file or storage format.
     * </p>
     * 
     * @return The total number of objects that Amazon Macie can't analyze in the bucket. These objects don't use a
     *         supported storage class or don't have a file name extension for a supported file or storage format.
     */
    public final ObjectLevelStatistics unclassifiableObjectCount() {
        return unclassifiableObjectCount;
    }

    /**
     * <p>
     * The total storage size, in bytes, of the objects that Amazon Macie can't analyze in the bucket. These objects
     * don't use a supported storage class or don't have a file name extension for a supported file or storage format.
     * </p>
     * 
     * @return The total storage size, in bytes, of the objects that Amazon Macie can't analyze in the bucket. These
     *         objects don't use a supported storage class or don't have a file name extension for a supported file or
     *         storage format.
     */
    public final ObjectLevelStatistics unclassifiableObjectSizeInBytes() {
        return unclassifiableObjectSizeInBytes;
    }

    /**
     * <p>
     * Specifies whether versioning is enabled for the bucket.
     * </p>
     * 
     * @return Specifies whether versioning is enabled for the bucket.
     */
    public final Boolean versioning() {
        return versioning;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(allowsUnencryptedObjectUploadsAsString());
        hashCode = 31 * hashCode + Objects.hashCode(bucketArn());
        hashCode = 31 * hashCode + Objects.hashCode(bucketCreatedAt());
        hashCode = 31 * hashCode + Objects.hashCode(bucketName());
        hashCode = 31 * hashCode + Objects.hashCode(classifiableObjectCount());
        hashCode = 31 * hashCode + Objects.hashCode(classifiableSizeInBytes());
        hashCode = 31 * hashCode + Objects.hashCode(jobDetails());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdated());
        hashCode = 31 * hashCode + Objects.hashCode(objectCount());
        hashCode = 31 * hashCode + Objects.hashCode(objectCountByEncryptionType());
        hashCode = 31 * hashCode + Objects.hashCode(publicAccess());
        hashCode = 31 * hashCode + Objects.hashCode(region());
        hashCode = 31 * hashCode + Objects.hashCode(replicationDetails());
        hashCode = 31 * hashCode + Objects.hashCode(serverSideEncryption());
        hashCode = 31 * hashCode + Objects.hashCode(sharedAccessAsString());
        hashCode = 31 * hashCode + Objects.hashCode(sizeInBytes());
        hashCode = 31 * hashCode + Objects.hashCode(sizeInBytesCompressed());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(unclassifiableObjectCount());
        hashCode = 31 * hashCode + Objects.hashCode(unclassifiableObjectSizeInBytes());
        hashCode = 31 * hashCode + Objects.hashCode(versioning());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof BucketMetadata)) {
            return false;
        }
        BucketMetadata other = (BucketMetadata) obj;
        return Objects.equals(accountId(), other.accountId())
                && Objects.equals(allowsUnencryptedObjectUploadsAsString(), other.allowsUnencryptedObjectUploadsAsString())
                && Objects.equals(bucketArn(), other.bucketArn()) && Objects.equals(bucketCreatedAt(), other.bucketCreatedAt())
                && Objects.equals(bucketName(), other.bucketName())
                && Objects.equals(classifiableObjectCount(), other.classifiableObjectCount())
                && Objects.equals(classifiableSizeInBytes(), other.classifiableSizeInBytes())
                && Objects.equals(jobDetails(), other.jobDetails()) && Objects.equals(lastUpdated(), other.lastUpdated())
                && Objects.equals(objectCount(), other.objectCount())
                && Objects.equals(objectCountByEncryptionType(), other.objectCountByEncryptionType())
                && Objects.equals(publicAccess(), other.publicAccess()) && Objects.equals(region(), other.region())
                && Objects.equals(replicationDetails(), other.replicationDetails())
                && Objects.equals(serverSideEncryption(), other.serverSideEncryption())
                && Objects.equals(sharedAccessAsString(), other.sharedAccessAsString())
                && Objects.equals(sizeInBytes(), other.sizeInBytes())
                && Objects.equals(sizeInBytesCompressed(), other.sizeInBytesCompressed()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags())
                && Objects.equals(unclassifiableObjectCount(), other.unclassifiableObjectCount())
                && Objects.equals(unclassifiableObjectSizeInBytes(), other.unclassifiableObjectSizeInBytes())
                && Objects.equals(versioning(), other.versioning());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("BucketMetadata").add("AccountId", accountId())
                .add("AllowsUnencryptedObjectUploads", allowsUnencryptedObjectUploadsAsString()).add("BucketArn", bucketArn())
                .add("BucketCreatedAt", bucketCreatedAt()).add("BucketName", bucketName())
                .add("ClassifiableObjectCount", classifiableObjectCount())
                .add("ClassifiableSizeInBytes", classifiableSizeInBytes()).add("JobDetails", jobDetails())
                .add("LastUpdated", lastUpdated()).add("ObjectCount", objectCount())
                .add("ObjectCountByEncryptionType", objectCountByEncryptionType()).add("PublicAccess", publicAccess())
                .add("Region", region()).add("ReplicationDetails", replicationDetails())
                .add("ServerSideEncryption", serverSideEncryption()).add("SharedAccess", sharedAccessAsString())
                .add("SizeInBytes", sizeInBytes()).add("SizeInBytesCompressed", sizeInBytesCompressed())
                .add("Tags", hasTags() ? tags() : null).add("UnclassifiableObjectCount", unclassifiableObjectCount())
                .add("UnclassifiableObjectSizeInBytes", unclassifiableObjectSizeInBytes()).add("Versioning", versioning())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "accountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "allowsUnencryptedObjectUploads":
            return Optional.ofNullable(clazz.cast(allowsUnencryptedObjectUploadsAsString()));
        case "bucketArn":
            return Optional.ofNullable(clazz.cast(bucketArn()));
        case "bucketCreatedAt":
            return Optional.ofNullable(clazz.cast(bucketCreatedAt()));
        case "bucketName":
            return Optional.ofNullable(clazz.cast(bucketName()));
        case "classifiableObjectCount":
            return Optional.ofNullable(clazz.cast(classifiableObjectCount()));
        case "classifiableSizeInBytes":
            return Optional.ofNullable(clazz.cast(classifiableSizeInBytes()));
        case "jobDetails":
            return Optional.ofNullable(clazz.cast(jobDetails()));
        case "lastUpdated":
            return Optional.ofNullable(clazz.cast(lastUpdated()));
        case "objectCount":
            return Optional.ofNullable(clazz.cast(objectCount()));
        case "objectCountByEncryptionType":
            return Optional.ofNullable(clazz.cast(objectCountByEncryptionType()));
        case "publicAccess":
            return Optional.ofNullable(clazz.cast(publicAccess()));
        case "region":
            return Optional.ofNullable(clazz.cast(region()));
        case "replicationDetails":
            return Optional.ofNullable(clazz.cast(replicationDetails()));
        case "serverSideEncryption":
            return Optional.ofNullable(clazz.cast(serverSideEncryption()));
        case "sharedAccess":
            return Optional.ofNullable(clazz.cast(sharedAccessAsString()));
        case "sizeInBytes":
            return Optional.ofNullable(clazz.cast(sizeInBytes()));
        case "sizeInBytesCompressed":
            return Optional.ofNullable(clazz.cast(sizeInBytesCompressed()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "unclassifiableObjectCount":
            return Optional.ofNullable(clazz.cast(unclassifiableObjectCount()));
        case "unclassifiableObjectSizeInBytes":
            return Optional.ofNullable(clazz.cast(unclassifiableObjectSizeInBytes()));
        case "versioning":
            return Optional.ofNullable(clazz.cast(versioning()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<BucketMetadata, T> g) {
        return obj -> g.apply((BucketMetadata) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, BucketMetadata> {
        /**
         * <p>
         * The unique identifier for the Amazon Web Services account that owns the bucket.
         * </p>
         * 
         * @param accountId
         *        The unique identifier for the Amazon Web Services account that owns the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * Specifies whether the bucket policy for the bucket requires server-side encryption of objects when objects
         * are uploaded to the bucket. Possible values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * FALSE - The bucket policy requires server-side encryption of new objects. PutObject requests must include the
         * x-amz-server-side-encryption header and the value for that header must be AES256 or aws:kms.
         * </p>
         * </li>
         * <li>
         * <p>
         * TRUE - The bucket doesn't have a bucket policy or it has a bucket policy that doesn't require server-side
         * encryption of new objects. If a bucket policy exists, it doesn't require PutObject requests to include the
         * x-amz-server-side-encryption header and it doesn't require the value for that header to be AES256 or aws:kms.
         * </p>
         * </li>
         * <li>
         * <p>
         * UNKNOWN - Amazon Macie can't determine whether the bucket policy requires server-side encryption of new
         * objects.
         * </p>
         * </li>
         * </ul>
         * 
         * @param allowsUnencryptedObjectUploads
         *        Specifies whether the bucket policy for the bucket requires server-side encryption of objects when
         *        objects are uploaded to the bucket. Possible values are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        FALSE - The bucket policy requires server-side encryption of new objects. PutObject requests must
         *        include the x-amz-server-side-encryption header and the value for that header must be AES256 or
         *        aws:kms.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        TRUE - The bucket doesn't have a bucket policy or it has a bucket policy that doesn't require
         *        server-side encryption of new objects. If a bucket policy exists, it doesn't require PutObject
         *        requests to include the x-amz-server-side-encryption header and it doesn't require the value for that
         *        header to be AES256 or aws:kms.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        UNKNOWN - Amazon Macie can't determine whether the bucket policy requires server-side encryption of
         *        new objects.
         *        </p>
         *        </li>
         * @see AllowsUnencryptedObjectUploads
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AllowsUnencryptedObjectUploads
         */
        Builder allowsUnencryptedObjectUploads(String allowsUnencryptedObjectUploads);

        /**
         * <p>
         * Specifies whether the bucket policy for the bucket requires server-side encryption of objects when objects
         * are uploaded to the bucket. Possible values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * FALSE - The bucket policy requires server-side encryption of new objects. PutObject requests must include the
         * x-amz-server-side-encryption header and the value for that header must be AES256 or aws:kms.
         * </p>
         * </li>
         * <li>
         * <p>
         * TRUE - The bucket doesn't have a bucket policy or it has a bucket policy that doesn't require server-side
         * encryption of new objects. If a bucket policy exists, it doesn't require PutObject requests to include the
         * x-amz-server-side-encryption header and it doesn't require the value for that header to be AES256 or aws:kms.
         * </p>
         * </li>
         * <li>
         * <p>
         * UNKNOWN - Amazon Macie can't determine whether the bucket policy requires server-side encryption of new
         * objects.
         * </p>
         * </li>
         * </ul>
         * 
         * @param allowsUnencryptedObjectUploads
         *        Specifies whether the bucket policy for the bucket requires server-side encryption of objects when
         *        objects are uploaded to the bucket. Possible values are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        FALSE - The bucket policy requires server-side encryption of new objects. PutObject requests must
         *        include the x-amz-server-side-encryption header and the value for that header must be AES256 or
         *        aws:kms.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        TRUE - The bucket doesn't have a bucket policy or it has a bucket policy that doesn't require
         *        server-side encryption of new objects. If a bucket policy exists, it doesn't require PutObject
         *        requests to include the x-amz-server-side-encryption header and it doesn't require the value for that
         *        header to be AES256 or aws:kms.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        UNKNOWN - Amazon Macie can't determine whether the bucket policy requires server-side encryption of
         *        new objects.
         *        </p>
         *        </li>
         * @see AllowsUnencryptedObjectUploads
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AllowsUnencryptedObjectUploads
         */
        Builder allowsUnencryptedObjectUploads(AllowsUnencryptedObjectUploads allowsUnencryptedObjectUploads);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the bucket.
         * </p>
         * 
         * @param bucketArn
         *        The Amazon Resource Name (ARN) of the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucketArn(String bucketArn);

        /**
         * <p>
         * The date and time, in UTC and extended ISO 8601 format, when the bucket was created.
         * </p>
         * 
         * @param bucketCreatedAt
         *        The date and time, in UTC and extended ISO 8601 format, when the bucket was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucketCreatedAt(Instant bucketCreatedAt);

        /**
         * <p>
         * The name of the bucket.
         * </p>
         * 
         * @param bucketName
         *        The name of the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucketName(String bucketName);

        /**
         * <p>
         * The total number of objects that Amazon Macie can analyze in the bucket. These objects use a supported
         * storage class and have a file name extension for a supported file or storage format.
         * </p>
         * 
         * @param classifiableObjectCount
         *        The total number of objects that Amazon Macie can analyze in the bucket. These objects use a supported
         *        storage class and have a file name extension for a supported file or storage format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder classifiableObjectCount(Long classifiableObjectCount);

        /**
         * <p>
         * The total storage size, in bytes, of the objects that Amazon Macie can analyze in the bucket. These objects
         * use a supported storage class and have a file name extension for a supported file or storage format.
         * </p>
         * <p>
         * If versioning is enabled for the bucket, Macie calculates this value based on the size of the latest version
         * of each applicable object in the bucket. This value doesn't reflect the storage size of all versions of each
         * applicable object in the bucket.
         * </p>
         * 
         * @param classifiableSizeInBytes
         *        The total storage size, in bytes, of the objects that Amazon Macie can analyze in the bucket. These
         *        objects use a supported storage class and have a file name extension for a supported file or storage
         *        format.</p>
         *        <p>
         *        If versioning is enabled for the bucket, Macie calculates this value based on the size of the latest
         *        version of each applicable object in the bucket. This value doesn't reflect the storage size of all
         *        versions of each applicable object in the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder classifiableSizeInBytes(Long classifiableSizeInBytes);

        /**
         * <p>
         * Specifies whether any one-time or recurring classification jobs are configured to analyze data in the bucket,
         * and, if so, the details of the job that ran most recently.
         * </p>
         * 
         * @param jobDetails
         *        Specifies whether any one-time or recurring classification jobs are configured to analyze data in the
         *        bucket, and, if so, the details of the job that ran most recently.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobDetails(JobDetails jobDetails);

        /**
         * <p>
         * Specifies whether any one-time or recurring classification jobs are configured to analyze data in the bucket,
         * and, if so, the details of the job that ran most recently.
         * </p>
         * This is a convenience that creates an instance of the {@link JobDetails.Builder} avoiding the need to create
         * one manually via {@link JobDetails#builder()}.
         *
         * When the {@link Consumer} completes, {@link JobDetails.Builder#build()} is called immediately and its result
         * is passed to {@link #jobDetails(JobDetails)}.
         * 
         * @param jobDetails
         *        a consumer that will call methods on {@link JobDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #jobDetails(JobDetails)
         */
        default Builder jobDetails(Consumer<JobDetails.Builder> jobDetails) {
            return jobDetails(JobDetails.builder().applyMutation(jobDetails).build());
        }

        /**
         * <p>
         * The date and time, in UTC and extended ISO 8601 format, when Amazon Macie most recently retrieved both bucket
         * and object metadata from Amazon S3 for the bucket.
         * </p>
         * 
         * @param lastUpdated
         *        The date and time, in UTC and extended ISO 8601 format, when Amazon Macie most recently retrieved both
         *        bucket and object metadata from Amazon S3 for the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdated(Instant lastUpdated);

        /**
         * <p>
         * The total number of objects in the bucket.
         * </p>
         * 
         * @param objectCount
         *        The total number of objects in the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder objectCount(Long objectCount);

        /**
         * <p>
         * The total number of objects that are in the bucket, grouped by server-side encryption type. This includes a
         * grouping that reports the total number of objects that aren't encrypted or use client-side encryption.
         * </p>
         * 
         * @param objectCountByEncryptionType
         *        The total number of objects that are in the bucket, grouped by server-side encryption type. This
         *        includes a grouping that reports the total number of objects that aren't encrypted or use client-side
         *        encryption.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder objectCountByEncryptionType(ObjectCountByEncryptionType objectCountByEncryptionType);

        /**
         * <p>
         * The total number of objects that are in the bucket, grouped by server-side encryption type. This includes a
         * grouping that reports the total number of objects that aren't encrypted or use client-side encryption.
         * </p>
         * This is a convenience that creates an instance of the {@link ObjectCountByEncryptionType.Builder} avoiding
         * the need to create one manually via {@link ObjectCountByEncryptionType#builder()}.
         *
         * When the {@link Consumer} completes, {@link ObjectCountByEncryptionType.Builder#build()} is called
         * immediately and its result is passed to {@link #objectCountByEncryptionType(ObjectCountByEncryptionType)}.
         * 
         * @param objectCountByEncryptionType
         *        a consumer that will call methods on {@link ObjectCountByEncryptionType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #objectCountByEncryptionType(ObjectCountByEncryptionType)
         */
        default Builder objectCountByEncryptionType(Consumer<ObjectCountByEncryptionType.Builder> objectCountByEncryptionType) {
            return objectCountByEncryptionType(ObjectCountByEncryptionType.builder().applyMutation(objectCountByEncryptionType)
                    .build());
        }

        /**
         * <p>
         * Specifies whether the bucket is publicly accessible due to the combination of permissions settings that apply
         * to the bucket, and provides information about those settings.
         * </p>
         * 
         * @param publicAccess
         *        Specifies whether the bucket is publicly accessible due to the combination of permissions settings
         *        that apply to the bucket, and provides information about those settings.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder publicAccess(BucketPublicAccess publicAccess);

        /**
         * <p>
         * Specifies whether the bucket is publicly accessible due to the combination of permissions settings that apply
         * to the bucket, and provides information about those settings.
         * </p>
         * This is a convenience that creates an instance of the {@link BucketPublicAccess.Builder} avoiding the need to
         * create one manually via {@link BucketPublicAccess#builder()}.
         *
         * When the {@link Consumer} completes, {@link BucketPublicAccess.Builder#build()} is called immediately and its
         * result is passed to {@link #publicAccess(BucketPublicAccess)}.
         * 
         * @param publicAccess
         *        a consumer that will call methods on {@link BucketPublicAccess.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #publicAccess(BucketPublicAccess)
         */
        default Builder publicAccess(Consumer<BucketPublicAccess.Builder> publicAccess) {
            return publicAccess(BucketPublicAccess.builder().applyMutation(publicAccess).build());
        }

        /**
         * <p>
         * The Amazon Web Services Region that hosts the bucket.
         * </p>
         * 
         * @param region
         *        The Amazon Web Services Region that hosts the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder region(String region);

        /**
         * <p>
         * Specifies whether the bucket is configured to replicate one or more objects to buckets for other Amazon Web
         * Services accounts and, if so, which accounts.
         * </p>
         * 
         * @param replicationDetails
         *        Specifies whether the bucket is configured to replicate one or more objects to buckets for other
         *        Amazon Web Services accounts and, if so, which accounts.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replicationDetails(ReplicationDetails replicationDetails);

        /**
         * <p>
         * Specifies whether the bucket is configured to replicate one or more objects to buckets for other Amazon Web
         * Services accounts and, if so, which accounts.
         * </p>
         * This is a convenience that creates an instance of the {@link ReplicationDetails.Builder} avoiding the need to
         * create one manually via {@link ReplicationDetails#builder()}.
         *
         * When the {@link Consumer} completes, {@link ReplicationDetails.Builder#build()} is called immediately and its
         * result is passed to {@link #replicationDetails(ReplicationDetails)}.
         * 
         * @param replicationDetails
         *        a consumer that will call methods on {@link ReplicationDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #replicationDetails(ReplicationDetails)
         */
        default Builder replicationDetails(Consumer<ReplicationDetails.Builder> replicationDetails) {
            return replicationDetails(ReplicationDetails.builder().applyMutation(replicationDetails).build());
        }

        /**
         * <p>
         * Specifies whether the bucket encrypts new objects by default and, if so, the type of server-side encryption
         * that's used.
         * </p>
         * 
         * @param serverSideEncryption
         *        Specifies whether the bucket encrypts new objects by default and, if so, the type of server-side
         *        encryption that's used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serverSideEncryption(BucketServerSideEncryption serverSideEncryption);

        /**
         * <p>
         * Specifies whether the bucket encrypts new objects by default and, if so, the type of server-side encryption
         * that's used.
         * </p>
         * This is a convenience that creates an instance of the {@link BucketServerSideEncryption.Builder} avoiding the
         * need to create one manually via {@link BucketServerSideEncryption#builder()}.
         *
         * When the {@link Consumer} completes, {@link BucketServerSideEncryption.Builder#build()} is called immediately
         * and its result is passed to {@link #serverSideEncryption(BucketServerSideEncryption)}.
         * 
         * @param serverSideEncryption
         *        a consumer that will call methods on {@link BucketServerSideEncryption.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #serverSideEncryption(BucketServerSideEncryption)
         */
        default Builder serverSideEncryption(Consumer<BucketServerSideEncryption.Builder> serverSideEncryption) {
            return serverSideEncryption(BucketServerSideEncryption.builder().applyMutation(serverSideEncryption).build());
        }

        /**
         * <p>
         * Specifies whether the bucket is shared with another Amazon Web Services account. Possible values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * EXTERNAL - The bucket is shared with an Amazon Web Services account that isn't part of the same Amazon Macie
         * organization.
         * </p>
         * </li>
         * <li>
         * <p>
         * INTERNAL - The bucket is shared with an Amazon Web Services account that's part of the same Amazon Macie
         * organization.
         * </p>
         * </li>
         * <li>
         * <p>
         * NOT_SHARED - The bucket isn't shared with other Amazon Web Services accounts.
         * </p>
         * </li>
         * <li>
         * <p>
         * UNKNOWN - Amazon Macie wasn't able to evaluate the shared access settings for the bucket.
         * </p>
         * </li>
         * </ul>
         * 
         * @param sharedAccess
         *        <p>
         *        Specifies whether the bucket is shared with another Amazon Web Services account. Possible values are:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        EXTERNAL - The bucket is shared with an Amazon Web Services account that isn't part of the same Amazon
         *        Macie organization.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        INTERNAL - The bucket is shared with an Amazon Web Services account that's part of the same Amazon
         *        Macie organization.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        NOT_SHARED - The bucket isn't shared with other Amazon Web Services accounts.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        UNKNOWN - Amazon Macie wasn't able to evaluate the shared access settings for the bucket.
         *        </p>
         *        </li>
         *        </ul>
         * @see SharedAccess
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SharedAccess
         */
        Builder sharedAccess(String sharedAccess);

        /**
         * <p>
         * Specifies whether the bucket is shared with another Amazon Web Services account. Possible values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * EXTERNAL - The bucket is shared with an Amazon Web Services account that isn't part of the same Amazon Macie
         * organization.
         * </p>
         * </li>
         * <li>
         * <p>
         * INTERNAL - The bucket is shared with an Amazon Web Services account that's part of the same Amazon Macie
         * organization.
         * </p>
         * </li>
         * <li>
         * <p>
         * NOT_SHARED - The bucket isn't shared with other Amazon Web Services accounts.
         * </p>
         * </li>
         * <li>
         * <p>
         * UNKNOWN - Amazon Macie wasn't able to evaluate the shared access settings for the bucket.
         * </p>
         * </li>
         * </ul>
         * 
         * @param sharedAccess
         *        <p>
         *        Specifies whether the bucket is shared with another Amazon Web Services account. Possible values are:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        EXTERNAL - The bucket is shared with an Amazon Web Services account that isn't part of the same Amazon
         *        Macie organization.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        INTERNAL - The bucket is shared with an Amazon Web Services account that's part of the same Amazon
         *        Macie organization.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        NOT_SHARED - The bucket isn't shared with other Amazon Web Services accounts.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        UNKNOWN - Amazon Macie wasn't able to evaluate the shared access settings for the bucket.
         *        </p>
         *        </li>
         *        </ul>
         * @see SharedAccess
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SharedAccess
         */
        Builder sharedAccess(SharedAccess sharedAccess);

        /**
         * <p>
         * The total storage size, in bytes, of the bucket.
         * </p>
         * <p>
         * If versioning is enabled for the bucket, Amazon Macie calculates this value based on the size of the latest
         * version of each object in the bucket. This value doesn't reflect the storage size of all versions of each
         * object in the bucket.
         * </p>
         * 
         * @param sizeInBytes
         *        The total storage size, in bytes, of the bucket.</p>
         *        <p>
         *        If versioning is enabled for the bucket, Amazon Macie calculates this value based on the size of the
         *        latest version of each object in the bucket. This value doesn't reflect the storage size of all
         *        versions of each object in the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sizeInBytes(Long sizeInBytes);

        /**
         * <p>
         * The total storage size, in bytes, of the objects that are compressed (.gz, .gzip, .zip) files in the bucket.
         * </p>
         * <p>
         * If versioning is enabled for the bucket, Macie calculates this value based on the size of the latest version
         * of each applicable object in the bucket. This value doesn't reflect the storage size of all versions of each
         * applicable object in the bucket.
         * </p>
         * 
         * @param sizeInBytesCompressed
         *        The total storage size, in bytes, of the objects that are compressed (.gz, .gzip, .zip) files in the
         *        bucket.</p>
         *        <p>
         *        If versioning is enabled for the bucket, Macie calculates this value based on the size of the latest
         *        version of each applicable object in the bucket. This value doesn't reflect the storage size of all
         *        versions of each applicable object in the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sizeInBytesCompressed(Long sizeInBytesCompressed);

        /**
         * <p>
         * An array that specifies the tags (keys and values) that are associated with the bucket.
         * </p>
         * 
         * @param tags
         *        An array that specifies the tags (keys and values) that are associated with the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<KeyValuePair> tags);

        /**
         * <p>
         * An array that specifies the tags (keys and values) that are associated with the bucket.
         * </p>
         * 
         * @param tags
         *        An array that specifies the tags (keys and values) that are associated with the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(KeyValuePair... tags);

        /**
         * <p>
         * An array that specifies the tags (keys and values) that are associated with the bucket.
         * </p>
         * This is a convenience that creates an instance of the {@link List<KeyValuePair>.Builder} avoiding the need to
         * create one manually via {@link List<KeyValuePair>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<KeyValuePair>.Builder#build()} is called immediately and its
         * result is passed to {@link #tags(List<KeyValuePair>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<KeyValuePair>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<KeyValuePair>)
         */
        Builder tags(Consumer<KeyValuePair.Builder>... tags);

        /**
         * <p>
         * The total number of objects that Amazon Macie can't analyze in the bucket. These objects don't use a
         * supported storage class or don't have a file name extension for a supported file or storage format.
         * </p>
         * 
         * @param unclassifiableObjectCount
         *        The total number of objects that Amazon Macie can't analyze in the bucket. These objects don't use a
         *        supported storage class or don't have a file name extension for a supported file or storage format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder unclassifiableObjectCount(ObjectLevelStatistics unclassifiableObjectCount);

        /**
         * <p>
         * The total number of objects that Amazon Macie can't analyze in the bucket. These objects don't use a
         * supported storage class or don't have a file name extension for a supported file or storage format.
         * </p>
         * This is a convenience that creates an instance of the {@link ObjectLevelStatistics.Builder} avoiding the need
         * to create one manually via {@link ObjectLevelStatistics#builder()}.
         *
         * When the {@link Consumer} completes, {@link ObjectLevelStatistics.Builder#build()} is called immediately and
         * its result is passed to {@link #unclassifiableObjectCount(ObjectLevelStatistics)}.
         * 
         * @param unclassifiableObjectCount
         *        a consumer that will call methods on {@link ObjectLevelStatistics.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #unclassifiableObjectCount(ObjectLevelStatistics)
         */
        default Builder unclassifiableObjectCount(Consumer<ObjectLevelStatistics.Builder> unclassifiableObjectCount) {
            return unclassifiableObjectCount(ObjectLevelStatistics.builder().applyMutation(unclassifiableObjectCount).build());
        }

        /**
         * <p>
         * The total storage size, in bytes, of the objects that Amazon Macie can't analyze in the bucket. These objects
         * don't use a supported storage class or don't have a file name extension for a supported file or storage
         * format.
         * </p>
         * 
         * @param unclassifiableObjectSizeInBytes
         *        The total storage size, in bytes, of the objects that Amazon Macie can't analyze in the bucket. These
         *        objects don't use a supported storage class or don't have a file name extension for a supported file
         *        or storage format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder unclassifiableObjectSizeInBytes(ObjectLevelStatistics unclassifiableObjectSizeInBytes);

        /**
         * <p>
         * The total storage size, in bytes, of the objects that Amazon Macie can't analyze in the bucket. These objects
         * don't use a supported storage class or don't have a file name extension for a supported file or storage
         * format.
         * </p>
         * This is a convenience that creates an instance of the {@link ObjectLevelStatistics.Builder} avoiding the need
         * to create one manually via {@link ObjectLevelStatistics#builder()}.
         *
         * When the {@link Consumer} completes, {@link ObjectLevelStatistics.Builder#build()} is called immediately and
         * its result is passed to {@link #unclassifiableObjectSizeInBytes(ObjectLevelStatistics)}.
         * 
         * @param unclassifiableObjectSizeInBytes
         *        a consumer that will call methods on {@link ObjectLevelStatistics.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #unclassifiableObjectSizeInBytes(ObjectLevelStatistics)
         */
        default Builder unclassifiableObjectSizeInBytes(Consumer<ObjectLevelStatistics.Builder> unclassifiableObjectSizeInBytes) {
            return unclassifiableObjectSizeInBytes(ObjectLevelStatistics.builder().applyMutation(unclassifiableObjectSizeInBytes)
                    .build());
        }

        /**
         * <p>
         * Specifies whether versioning is enabled for the bucket.
         * </p>
         * 
         * @param versioning
         *        Specifies whether versioning is enabled for the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versioning(Boolean versioning);
    }

    static final class BuilderImpl implements Builder {
        private String accountId;

        private String allowsUnencryptedObjectUploads;

        private String bucketArn;

        private Instant bucketCreatedAt;

        private String bucketName;

        private Long classifiableObjectCount;

        private Long classifiableSizeInBytes;

        private JobDetails jobDetails;

        private Instant lastUpdated;

        private Long objectCount;

        private ObjectCountByEncryptionType objectCountByEncryptionType;

        private BucketPublicAccess publicAccess;

        private String region;

        private ReplicationDetails replicationDetails;

        private BucketServerSideEncryption serverSideEncryption;

        private String sharedAccess;

        private Long sizeInBytes;

        private Long sizeInBytesCompressed;

        private List<KeyValuePair> tags = DefaultSdkAutoConstructList.getInstance();

        private ObjectLevelStatistics unclassifiableObjectCount;

        private ObjectLevelStatistics unclassifiableObjectSizeInBytes;

        private Boolean versioning;

        private BuilderImpl() {
        }

        private BuilderImpl(BucketMetadata model) {
            accountId(model.accountId);
            allowsUnencryptedObjectUploads(model.allowsUnencryptedObjectUploads);
            bucketArn(model.bucketArn);
            bucketCreatedAt(model.bucketCreatedAt);
            bucketName(model.bucketName);
            classifiableObjectCount(model.classifiableObjectCount);
            classifiableSizeInBytes(model.classifiableSizeInBytes);
            jobDetails(model.jobDetails);
            lastUpdated(model.lastUpdated);
            objectCount(model.objectCount);
            objectCountByEncryptionType(model.objectCountByEncryptionType);
            publicAccess(model.publicAccess);
            region(model.region);
            replicationDetails(model.replicationDetails);
            serverSideEncryption(model.serverSideEncryption);
            sharedAccess(model.sharedAccess);
            sizeInBytes(model.sizeInBytes);
            sizeInBytesCompressed(model.sizeInBytesCompressed);
            tags(model.tags);
            unclassifiableObjectCount(model.unclassifiableObjectCount);
            unclassifiableObjectSizeInBytes(model.unclassifiableObjectSizeInBytes);
            versioning(model.versioning);
        }

        public final String getAccountId() {
            return accountId;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        @Override
        @Transient
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final String getAllowsUnencryptedObjectUploads() {
            return allowsUnencryptedObjectUploads;
        }

        public final void setAllowsUnencryptedObjectUploads(String allowsUnencryptedObjectUploads) {
            this.allowsUnencryptedObjectUploads = allowsUnencryptedObjectUploads;
        }

        @Override
        @Transient
        public final Builder allowsUnencryptedObjectUploads(String allowsUnencryptedObjectUploads) {
            this.allowsUnencryptedObjectUploads = allowsUnencryptedObjectUploads;
            return this;
        }

        @Override
        @Transient
        public final Builder allowsUnencryptedObjectUploads(AllowsUnencryptedObjectUploads allowsUnencryptedObjectUploads) {
            this.allowsUnencryptedObjectUploads(allowsUnencryptedObjectUploads == null ? null : allowsUnencryptedObjectUploads
                    .toString());
            return this;
        }

        public final String getBucketArn() {
            return bucketArn;
        }

        public final void setBucketArn(String bucketArn) {
            this.bucketArn = bucketArn;
        }

        @Override
        @Transient
        public final Builder bucketArn(String bucketArn) {
            this.bucketArn = bucketArn;
            return this;
        }

        public final Instant getBucketCreatedAt() {
            return bucketCreatedAt;
        }

        public final void setBucketCreatedAt(Instant bucketCreatedAt) {
            this.bucketCreatedAt = bucketCreatedAt;
        }

        @Override
        @Transient
        public final Builder bucketCreatedAt(Instant bucketCreatedAt) {
            this.bucketCreatedAt = bucketCreatedAt;
            return this;
        }

        public final String getBucketName() {
            return bucketName;
        }

        public final void setBucketName(String bucketName) {
            this.bucketName = bucketName;
        }

        @Override
        @Transient
        public final Builder bucketName(String bucketName) {
            this.bucketName = bucketName;
            return this;
        }

        public final Long getClassifiableObjectCount() {
            return classifiableObjectCount;
        }

        public final void setClassifiableObjectCount(Long classifiableObjectCount) {
            this.classifiableObjectCount = classifiableObjectCount;
        }

        @Override
        @Transient
        public final Builder classifiableObjectCount(Long classifiableObjectCount) {
            this.classifiableObjectCount = classifiableObjectCount;
            return this;
        }

        public final Long getClassifiableSizeInBytes() {
            return classifiableSizeInBytes;
        }

        public final void setClassifiableSizeInBytes(Long classifiableSizeInBytes) {
            this.classifiableSizeInBytes = classifiableSizeInBytes;
        }

        @Override
        @Transient
        public final Builder classifiableSizeInBytes(Long classifiableSizeInBytes) {
            this.classifiableSizeInBytes = classifiableSizeInBytes;
            return this;
        }

        public final JobDetails.Builder getJobDetails() {
            return jobDetails != null ? jobDetails.toBuilder() : null;
        }

        public final void setJobDetails(JobDetails.BuilderImpl jobDetails) {
            this.jobDetails = jobDetails != null ? jobDetails.build() : null;
        }

        @Override
        @Transient
        public final Builder jobDetails(JobDetails jobDetails) {
            this.jobDetails = jobDetails;
            return this;
        }

        public final Instant getLastUpdated() {
            return lastUpdated;
        }

        public final void setLastUpdated(Instant lastUpdated) {
            this.lastUpdated = lastUpdated;
        }

        @Override
        @Transient
        public final Builder lastUpdated(Instant lastUpdated) {
            this.lastUpdated = lastUpdated;
            return this;
        }

        public final Long getObjectCount() {
            return objectCount;
        }

        public final void setObjectCount(Long objectCount) {
            this.objectCount = objectCount;
        }

        @Override
        @Transient
        public final Builder objectCount(Long objectCount) {
            this.objectCount = objectCount;
            return this;
        }

        public final ObjectCountByEncryptionType.Builder getObjectCountByEncryptionType() {
            return objectCountByEncryptionType != null ? objectCountByEncryptionType.toBuilder() : null;
        }

        public final void setObjectCountByEncryptionType(ObjectCountByEncryptionType.BuilderImpl objectCountByEncryptionType) {
            this.objectCountByEncryptionType = objectCountByEncryptionType != null ? objectCountByEncryptionType.build() : null;
        }

        @Override
        @Transient
        public final Builder objectCountByEncryptionType(ObjectCountByEncryptionType objectCountByEncryptionType) {
            this.objectCountByEncryptionType = objectCountByEncryptionType;
            return this;
        }

        public final BucketPublicAccess.Builder getPublicAccess() {
            return publicAccess != null ? publicAccess.toBuilder() : null;
        }

        public final void setPublicAccess(BucketPublicAccess.BuilderImpl publicAccess) {
            this.publicAccess = publicAccess != null ? publicAccess.build() : null;
        }

        @Override
        @Transient
        public final Builder publicAccess(BucketPublicAccess publicAccess) {
            this.publicAccess = publicAccess;
            return this;
        }

        public final String getRegion() {
            return region;
        }

        public final void setRegion(String region) {
            this.region = region;
        }

        @Override
        @Transient
        public final Builder region(String region) {
            this.region = region;
            return this;
        }

        public final ReplicationDetails.Builder getReplicationDetails() {
            return replicationDetails != null ? replicationDetails.toBuilder() : null;
        }

        public final void setReplicationDetails(ReplicationDetails.BuilderImpl replicationDetails) {
            this.replicationDetails = replicationDetails != null ? replicationDetails.build() : null;
        }

        @Override
        @Transient
        public final Builder replicationDetails(ReplicationDetails replicationDetails) {
            this.replicationDetails = replicationDetails;
            return this;
        }

        public final BucketServerSideEncryption.Builder getServerSideEncryption() {
            return serverSideEncryption != null ? serverSideEncryption.toBuilder() : null;
        }

        public final void setServerSideEncryption(BucketServerSideEncryption.BuilderImpl serverSideEncryption) {
            this.serverSideEncryption = serverSideEncryption != null ? serverSideEncryption.build() : null;
        }

        @Override
        @Transient
        public final Builder serverSideEncryption(BucketServerSideEncryption serverSideEncryption) {
            this.serverSideEncryption = serverSideEncryption;
            return this;
        }

        public final String getSharedAccess() {
            return sharedAccess;
        }

        public final void setSharedAccess(String sharedAccess) {
            this.sharedAccess = sharedAccess;
        }

        @Override
        @Transient
        public final Builder sharedAccess(String sharedAccess) {
            this.sharedAccess = sharedAccess;
            return this;
        }

        @Override
        @Transient
        public final Builder sharedAccess(SharedAccess sharedAccess) {
            this.sharedAccess(sharedAccess == null ? null : sharedAccess.toString());
            return this;
        }

        public final Long getSizeInBytes() {
            return sizeInBytes;
        }

        public final void setSizeInBytes(Long sizeInBytes) {
            this.sizeInBytes = sizeInBytes;
        }

        @Override
        @Transient
        public final Builder sizeInBytes(Long sizeInBytes) {
            this.sizeInBytes = sizeInBytes;
            return this;
        }

        public final Long getSizeInBytesCompressed() {
            return sizeInBytesCompressed;
        }

        public final void setSizeInBytesCompressed(Long sizeInBytesCompressed) {
            this.sizeInBytesCompressed = sizeInBytesCompressed;
        }

        @Override
        @Transient
        public final Builder sizeInBytesCompressed(Long sizeInBytesCompressed) {
            this.sizeInBytesCompressed = sizeInBytesCompressed;
            return this;
        }

        public final List<KeyValuePair.Builder> getTags() {
            List<KeyValuePair.Builder> result = ___listOfKeyValuePairCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<KeyValuePair.BuilderImpl> tags) {
            this.tags = ___listOfKeyValuePairCopier.copyFromBuilder(tags);
        }

        @Override
        @Transient
        public final Builder tags(Collection<KeyValuePair> tags) {
            this.tags = ___listOfKeyValuePairCopier.copy(tags);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder tags(KeyValuePair... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder tags(Consumer<KeyValuePair.Builder>... tags) {
            tags(Stream.of(tags).map(c -> KeyValuePair.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final ObjectLevelStatistics.Builder getUnclassifiableObjectCount() {
            return unclassifiableObjectCount != null ? unclassifiableObjectCount.toBuilder() : null;
        }

        public final void setUnclassifiableObjectCount(ObjectLevelStatistics.BuilderImpl unclassifiableObjectCount) {
            this.unclassifiableObjectCount = unclassifiableObjectCount != null ? unclassifiableObjectCount.build() : null;
        }

        @Override
        @Transient
        public final Builder unclassifiableObjectCount(ObjectLevelStatistics unclassifiableObjectCount) {
            this.unclassifiableObjectCount = unclassifiableObjectCount;
            return this;
        }

        public final ObjectLevelStatistics.Builder getUnclassifiableObjectSizeInBytes() {
            return unclassifiableObjectSizeInBytes != null ? unclassifiableObjectSizeInBytes.toBuilder() : null;
        }

        public final void setUnclassifiableObjectSizeInBytes(ObjectLevelStatistics.BuilderImpl unclassifiableObjectSizeInBytes) {
            this.unclassifiableObjectSizeInBytes = unclassifiableObjectSizeInBytes != null ? unclassifiableObjectSizeInBytes
                    .build() : null;
        }

        @Override
        @Transient
        public final Builder unclassifiableObjectSizeInBytes(ObjectLevelStatistics unclassifiableObjectSizeInBytes) {
            this.unclassifiableObjectSizeInBytes = unclassifiableObjectSizeInBytes;
            return this;
        }

        public final Boolean getVersioning() {
            return versioning;
        }

        public final void setVersioning(Boolean versioning) {
            this.versioning = versioning;
        }

        @Override
        @Transient
        public final Builder versioning(Boolean versioning) {
            this.versioning = versioning;
            return this;
        }

        @Override
        public BucketMetadata build() {
            return new BucketMetadata(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
