/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.macie2.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides information about settings that define whether one or more objects in an S3 bucket are replicated to S3
 * buckets for other Amazon Web Services accounts and, if so, which accounts.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ReplicationDetails implements SdkPojo, Serializable,
        ToCopyableBuilder<ReplicationDetails.Builder, ReplicationDetails> {
    private static final SdkField<Boolean> REPLICATED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("replicated").getter(getter(ReplicationDetails::replicated)).setter(setter(Builder::replicated))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("replicated").build()).build();

    private static final SdkField<Boolean> REPLICATED_EXTERNALLY_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("replicatedExternally").getter(getter(ReplicationDetails::replicatedExternally))
            .setter(setter(Builder::replicatedExternally))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("replicatedExternally").build())
            .build();

    private static final SdkField<List<String>> REPLICATION_ACCOUNTS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("replicationAccounts")
            .getter(getter(ReplicationDetails::replicationAccounts))
            .setter(setter(Builder::replicationAccounts))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("replicationAccounts").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(REPLICATED_FIELD,
            REPLICATED_EXTERNALLY_FIELD, REPLICATION_ACCOUNTS_FIELD));

    private static final long serialVersionUID = 1L;

    private final Boolean replicated;

    private final Boolean replicatedExternally;

    private final List<String> replicationAccounts;

    private ReplicationDetails(BuilderImpl builder) {
        this.replicated = builder.replicated;
        this.replicatedExternally = builder.replicatedExternally;
        this.replicationAccounts = builder.replicationAccounts;
    }

    /**
     * <p>
     * Specifies whether the bucket is configured to replicate one or more objects to any destination.
     * </p>
     * 
     * @return Specifies whether the bucket is configured to replicate one or more objects to any destination.
     */
    public final Boolean replicated() {
        return replicated;
    }

    /**
     * <p>
     * Specifies whether the bucket is configured to replicate one or more objects to an Amazon Web Services account
     * that isn't part of the same Amazon Macie organization.
     * </p>
     * 
     * @return Specifies whether the bucket is configured to replicate one or more objects to an Amazon Web Services
     *         account that isn't part of the same Amazon Macie organization.
     */
    public final Boolean replicatedExternally() {
        return replicatedExternally;
    }

    /**
     * For responses, this returns true if the service returned a value for the ReplicationAccounts property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasReplicationAccounts() {
        return replicationAccounts != null && !(replicationAccounts instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of Amazon Web Services account IDs, one for each Amazon Web Services account that the bucket is
     * configured to replicate one or more objects to.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasReplicationAccounts} method.
     * </p>
     * 
     * @return An array of Amazon Web Services account IDs, one for each Amazon Web Services account that the bucket is
     *         configured to replicate one or more objects to.
     */
    public final List<String> replicationAccounts() {
        return replicationAccounts;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(replicated());
        hashCode = 31 * hashCode + Objects.hashCode(replicatedExternally());
        hashCode = 31 * hashCode + Objects.hashCode(hasReplicationAccounts() ? replicationAccounts() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ReplicationDetails)) {
            return false;
        }
        ReplicationDetails other = (ReplicationDetails) obj;
        return Objects.equals(replicated(), other.replicated())
                && Objects.equals(replicatedExternally(), other.replicatedExternally())
                && hasReplicationAccounts() == other.hasReplicationAccounts()
                && Objects.equals(replicationAccounts(), other.replicationAccounts());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ReplicationDetails").add("Replicated", replicated())
                .add("ReplicatedExternally", replicatedExternally())
                .add("ReplicationAccounts", hasReplicationAccounts() ? replicationAccounts() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "replicated":
            return Optional.ofNullable(clazz.cast(replicated()));
        case "replicatedExternally":
            return Optional.ofNullable(clazz.cast(replicatedExternally()));
        case "replicationAccounts":
            return Optional.ofNullable(clazz.cast(replicationAccounts()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ReplicationDetails, T> g) {
        return obj -> g.apply((ReplicationDetails) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ReplicationDetails> {
        /**
         * <p>
         * Specifies whether the bucket is configured to replicate one or more objects to any destination.
         * </p>
         * 
         * @param replicated
         *        Specifies whether the bucket is configured to replicate one or more objects to any destination.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replicated(Boolean replicated);

        /**
         * <p>
         * Specifies whether the bucket is configured to replicate one or more objects to an Amazon Web Services account
         * that isn't part of the same Amazon Macie organization.
         * </p>
         * 
         * @param replicatedExternally
         *        Specifies whether the bucket is configured to replicate one or more objects to an Amazon Web Services
         *        account that isn't part of the same Amazon Macie organization.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replicatedExternally(Boolean replicatedExternally);

        /**
         * <p>
         * An array of Amazon Web Services account IDs, one for each Amazon Web Services account that the bucket is
         * configured to replicate one or more objects to.
         * </p>
         * 
         * @param replicationAccounts
         *        An array of Amazon Web Services account IDs, one for each Amazon Web Services account that the bucket
         *        is configured to replicate one or more objects to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replicationAccounts(Collection<String> replicationAccounts);

        /**
         * <p>
         * An array of Amazon Web Services account IDs, one for each Amazon Web Services account that the bucket is
         * configured to replicate one or more objects to.
         * </p>
         * 
         * @param replicationAccounts
         *        An array of Amazon Web Services account IDs, one for each Amazon Web Services account that the bucket
         *        is configured to replicate one or more objects to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replicationAccounts(String... replicationAccounts);
    }

    static final class BuilderImpl implements Builder {
        private Boolean replicated;

        private Boolean replicatedExternally;

        private List<String> replicationAccounts = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ReplicationDetails model) {
            replicated(model.replicated);
            replicatedExternally(model.replicatedExternally);
            replicationAccounts(model.replicationAccounts);
        }

        public final Boolean getReplicated() {
            return replicated;
        }

        public final void setReplicated(Boolean replicated) {
            this.replicated = replicated;
        }

        @Override
        @Transient
        public final Builder replicated(Boolean replicated) {
            this.replicated = replicated;
            return this;
        }

        public final Boolean getReplicatedExternally() {
            return replicatedExternally;
        }

        public final void setReplicatedExternally(Boolean replicatedExternally) {
            this.replicatedExternally = replicatedExternally;
        }

        @Override
        @Transient
        public final Builder replicatedExternally(Boolean replicatedExternally) {
            this.replicatedExternally = replicatedExternally;
            return this;
        }

        public final Collection<String> getReplicationAccounts() {
            if (replicationAccounts instanceof SdkAutoConstructList) {
                return null;
            }
            return replicationAccounts;
        }

        public final void setReplicationAccounts(Collection<String> replicationAccounts) {
            this.replicationAccounts = ___listOf__stringCopier.copy(replicationAccounts);
        }

        @Override
        @Transient
        public final Builder replicationAccounts(Collection<String> replicationAccounts) {
            this.replicationAccounts = ___listOf__stringCopier.copy(replicationAccounts);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder replicationAccounts(String... replicationAccounts) {
            replicationAccounts(Arrays.asList(replicationAccounts));
            return this;
        }

        @Override
        public ReplicationDetails build() {
            return new ReplicationDetails(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
