/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.macie2.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides information about a classification job, including the current status of the job.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class JobSummary implements SdkPojo, Serializable, ToCopyableBuilder<JobSummary.Builder, JobSummary> {
    private static final SdkField<S3BucketCriteriaForJob> BUCKET_CRITERIA_FIELD = SdkField
            .<S3BucketCriteriaForJob> builder(MarshallingType.SDK_POJO).memberName("bucketCriteria")
            .getter(getter(JobSummary::bucketCriteria)).setter(setter(Builder::bucketCriteria))
            .constructor(S3BucketCriteriaForJob::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("bucketCriteria").build()).build();

    private static final SdkField<List<S3BucketDefinitionForJob>> BUCKET_DEFINITIONS_FIELD = SdkField
            .<List<S3BucketDefinitionForJob>> builder(MarshallingType.LIST)
            .memberName("bucketDefinitions")
            .getter(getter(JobSummary::bucketDefinitions))
            .setter(setter(Builder::bucketDefinitions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("bucketDefinitions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<S3BucketDefinitionForJob> builder(MarshallingType.SDK_POJO)
                                            .constructor(S3BucketDefinitionForJob::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt")
            .getter(getter(JobSummary::createdAt))
            .setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> JOB_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("jobId")
            .getter(getter(JobSummary::jobId)).setter(setter(Builder::jobId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("jobId").build()).build();

    private static final SdkField<String> JOB_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("jobStatus").getter(getter(JobSummary::jobStatusAsString)).setter(setter(Builder::jobStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("jobStatus").build()).build();

    private static final SdkField<String> JOB_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("jobType").getter(getter(JobSummary::jobTypeAsString)).setter(setter(Builder::jobType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("jobType").build()).build();

    private static final SdkField<LastRunErrorStatus> LAST_RUN_ERROR_STATUS_FIELD = SdkField
            .<LastRunErrorStatus> builder(MarshallingType.SDK_POJO).memberName("lastRunErrorStatus")
            .getter(getter(JobSummary::lastRunErrorStatus)).setter(setter(Builder::lastRunErrorStatus))
            .constructor(LastRunErrorStatus::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastRunErrorStatus").build())
            .build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(JobSummary::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<UserPausedDetails> USER_PAUSED_DETAILS_FIELD = SdkField
            .<UserPausedDetails> builder(MarshallingType.SDK_POJO).memberName("userPausedDetails")
            .getter(getter(JobSummary::userPausedDetails)).setter(setter(Builder::userPausedDetails))
            .constructor(UserPausedDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("userPausedDetails").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BUCKET_CRITERIA_FIELD,
            BUCKET_DEFINITIONS_FIELD, CREATED_AT_FIELD, JOB_ID_FIELD, JOB_STATUS_FIELD, JOB_TYPE_FIELD,
            LAST_RUN_ERROR_STATUS_FIELD, NAME_FIELD, USER_PAUSED_DETAILS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("bucketCriteria", BUCKET_CRITERIA_FIELD);
                    put("bucketDefinitions", BUCKET_DEFINITIONS_FIELD);
                    put("createdAt", CREATED_AT_FIELD);
                    put("jobId", JOB_ID_FIELD);
                    put("jobStatus", JOB_STATUS_FIELD);
                    put("jobType", JOB_TYPE_FIELD);
                    put("lastRunErrorStatus", LAST_RUN_ERROR_STATUS_FIELD);
                    put("name", NAME_FIELD);
                    put("userPausedDetails", USER_PAUSED_DETAILS_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final S3BucketCriteriaForJob bucketCriteria;

    private final List<S3BucketDefinitionForJob> bucketDefinitions;

    private final Instant createdAt;

    private final String jobId;

    private final String jobStatus;

    private final String jobType;

    private final LastRunErrorStatus lastRunErrorStatus;

    private final String name;

    private final UserPausedDetails userPausedDetails;

    private JobSummary(BuilderImpl builder) {
        this.bucketCriteria = builder.bucketCriteria;
        this.bucketDefinitions = builder.bucketDefinitions;
        this.createdAt = builder.createdAt;
        this.jobId = builder.jobId;
        this.jobStatus = builder.jobStatus;
        this.jobType = builder.jobType;
        this.lastRunErrorStatus = builder.lastRunErrorStatus;
        this.name = builder.name;
        this.userPausedDetails = builder.userPausedDetails;
    }

    /**
     * <p>
     * The property- and tag-based conditions that determine which S3 buckets are included or excluded from the job's
     * analysis. Each time the job runs, the job uses these criteria to determine which buckets to analyze. A job's
     * definition can contain a bucketCriteria object or a bucketDefinitions array, not both.
     * </p>
     * 
     * @return The property- and tag-based conditions that determine which S3 buckets are included or excluded from the
     *         job's analysis. Each time the job runs, the job uses these criteria to determine which buckets to
     *         analyze. A job's definition can contain a bucketCriteria object or a bucketDefinitions array, not both.
     */
    public final S3BucketCriteriaForJob bucketCriteria() {
        return bucketCriteria;
    }

    /**
     * For responses, this returns true if the service returned a value for the BucketDefinitions property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasBucketDefinitions() {
        return bucketDefinitions != null && !(bucketDefinitions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of objects, one for each Amazon Web Services account that owns specific S3 buckets for the job to
     * analyze. Each object specifies the account ID for an account and one or more buckets to analyze for that account.
     * A job's definition can contain a bucketDefinitions array or a bucketCriteria object, not both.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasBucketDefinitions} method.
     * </p>
     * 
     * @return An array of objects, one for each Amazon Web Services account that owns specific S3 buckets for the job
     *         to analyze. Each object specifies the account ID for an account and one or more buckets to analyze for
     *         that account. A job's definition can contain a bucketDefinitions array or a bucketCriteria object, not
     *         both.
     */
    public final List<S3BucketDefinitionForJob> bucketDefinitions() {
        return bucketDefinitions;
    }

    /**
     * <p>
     * The date and time, in UTC and extended ISO 8601 format, when the job was created.
     * </p>
     * 
     * @return The date and time, in UTC and extended ISO 8601 format, when the job was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The unique identifier for the job.
     * </p>
     * 
     * @return The unique identifier for the job.
     */
    public final String jobId() {
        return jobId;
    }

    /**
     * <p>
     * The current status of the job. Possible values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * CANCELLED - You cancelled the job or, if it's a one-time job, you paused the job and didn't resume it within 30
     * days.
     * </p>
     * </li>
     * <li>
     * <p>
     * COMPLETE - For a one-time job, Amazon Macie finished processing the data specified for the job. This value
     * doesn't apply to recurring jobs.
     * </p>
     * </li>
     * <li>
     * <p>
     * IDLE - For a recurring job, the previous scheduled run is complete and the next scheduled run is pending. This
     * value doesn't apply to one-time jobs.
     * </p>
     * </li>
     * <li>
     * <p>
     * PAUSED - Macie started running the job but additional processing would exceed the monthly sensitive data
     * discovery quota for your account or one or more member accounts that the job analyzes data for.
     * </p>
     * </li>
     * <li>
     * <p>
     * RUNNING - For a one-time job, the job is in progress. For a recurring job, a scheduled run is in progress.
     * </p>
     * </li>
     * <li>
     * <p>
     * USER_PAUSED - You paused the job. If you paused the job while it had a status of RUNNING and you don't resume it
     * within 30 days of pausing it, the job or job run will expire and be cancelled, depending on the job's type. To
     * check the expiration date, refer to the UserPausedDetails.jobExpiresAt property.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #jobStatus} will
     * return {@link JobStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #jobStatusAsString}.
     * </p>
     * 
     * @return The current status of the job. Possible values are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         CANCELLED - You cancelled the job or, if it's a one-time job, you paused the job and didn't resume it
     *         within 30 days.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         COMPLETE - For a one-time job, Amazon Macie finished processing the data specified for the job. This
     *         value doesn't apply to recurring jobs.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         IDLE - For a recurring job, the previous scheduled run is complete and the next scheduled run is pending.
     *         This value doesn't apply to one-time jobs.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         PAUSED - Macie started running the job but additional processing would exceed the monthly sensitive data
     *         discovery quota for your account or one or more member accounts that the job analyzes data for.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         RUNNING - For a one-time job, the job is in progress. For a recurring job, a scheduled run is in
     *         progress.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         USER_PAUSED - You paused the job. If you paused the job while it had a status of RUNNING and you don't
     *         resume it within 30 days of pausing it, the job or job run will expire and be cancelled, depending on the
     *         job's type. To check the expiration date, refer to the UserPausedDetails.jobExpiresAt property.
     *         </p>
     *         </li>
     * @see JobStatus
     */
    public final JobStatus jobStatus() {
        return JobStatus.fromValue(jobStatus);
    }

    /**
     * <p>
     * The current status of the job. Possible values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * CANCELLED - You cancelled the job or, if it's a one-time job, you paused the job and didn't resume it within 30
     * days.
     * </p>
     * </li>
     * <li>
     * <p>
     * COMPLETE - For a one-time job, Amazon Macie finished processing the data specified for the job. This value
     * doesn't apply to recurring jobs.
     * </p>
     * </li>
     * <li>
     * <p>
     * IDLE - For a recurring job, the previous scheduled run is complete and the next scheduled run is pending. This
     * value doesn't apply to one-time jobs.
     * </p>
     * </li>
     * <li>
     * <p>
     * PAUSED - Macie started running the job but additional processing would exceed the monthly sensitive data
     * discovery quota for your account or one or more member accounts that the job analyzes data for.
     * </p>
     * </li>
     * <li>
     * <p>
     * RUNNING - For a one-time job, the job is in progress. For a recurring job, a scheduled run is in progress.
     * </p>
     * </li>
     * <li>
     * <p>
     * USER_PAUSED - You paused the job. If you paused the job while it had a status of RUNNING and you don't resume it
     * within 30 days of pausing it, the job or job run will expire and be cancelled, depending on the job's type. To
     * check the expiration date, refer to the UserPausedDetails.jobExpiresAt property.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #jobStatus} will
     * return {@link JobStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #jobStatusAsString}.
     * </p>
     * 
     * @return The current status of the job. Possible values are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         CANCELLED - You cancelled the job or, if it's a one-time job, you paused the job and didn't resume it
     *         within 30 days.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         COMPLETE - For a one-time job, Amazon Macie finished processing the data specified for the job. This
     *         value doesn't apply to recurring jobs.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         IDLE - For a recurring job, the previous scheduled run is complete and the next scheduled run is pending.
     *         This value doesn't apply to one-time jobs.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         PAUSED - Macie started running the job but additional processing would exceed the monthly sensitive data
     *         discovery quota for your account or one or more member accounts that the job analyzes data for.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         RUNNING - For a one-time job, the job is in progress. For a recurring job, a scheduled run is in
     *         progress.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         USER_PAUSED - You paused the job. If you paused the job while it had a status of RUNNING and you don't
     *         resume it within 30 days of pausing it, the job or job run will expire and be cancelled, depending on the
     *         job's type. To check the expiration date, refer to the UserPausedDetails.jobExpiresAt property.
     *         </p>
     *         </li>
     * @see JobStatus
     */
    public final String jobStatusAsString() {
        return jobStatus;
    }

    /**
     * <p>
     * The schedule for running the job. Possible values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * ONE_TIME - The job runs only once.
     * </p>
     * </li>
     * <li>
     * <p>
     * SCHEDULED - The job runs on a daily, weekly, or monthly basis.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #jobType} will
     * return {@link JobType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #jobTypeAsString}.
     * </p>
     * 
     * @return The schedule for running the job. Possible values are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         ONE_TIME - The job runs only once.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         SCHEDULED - The job runs on a daily, weekly, or monthly basis.
     *         </p>
     *         </li>
     * @see JobType
     */
    public final JobType jobType() {
        return JobType.fromValue(jobType);
    }

    /**
     * <p>
     * The schedule for running the job. Possible values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * ONE_TIME - The job runs only once.
     * </p>
     * </li>
     * <li>
     * <p>
     * SCHEDULED - The job runs on a daily, weekly, or monthly basis.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #jobType} will
     * return {@link JobType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #jobTypeAsString}.
     * </p>
     * 
     * @return The schedule for running the job. Possible values are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         ONE_TIME - The job runs only once.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         SCHEDULED - The job runs on a daily, weekly, or monthly basis.
     *         </p>
     *         </li>
     * @see JobType
     */
    public final String jobTypeAsString() {
        return jobType;
    }

    /**
     * <p>
     * Specifies whether any account- or bucket-level access errors occurred when the job ran. For a recurring job, this
     * value indicates the error status of the job's most recent run.
     * </p>
     * 
     * @return Specifies whether any account- or bucket-level access errors occurred when the job ran. For a recurring
     *         job, this value indicates the error status of the job's most recent run.
     */
    public final LastRunErrorStatus lastRunErrorStatus() {
        return lastRunErrorStatus;
    }

    /**
     * <p>
     * The custom name of the job.
     * </p>
     * 
     * @return The custom name of the job.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * If the current status of the job is USER_PAUSED, specifies when the job was paused and when the job or job run
     * will expire and be cancelled if it isn't resumed. This value is present only if the value for jobStatus is
     * USER_PAUSED.
     * </p>
     * 
     * @return If the current status of the job is USER_PAUSED, specifies when the job was paused and when the job or
     *         job run will expire and be cancelled if it isn't resumed. This value is present only if the value for
     *         jobStatus is USER_PAUSED.
     */
    public final UserPausedDetails userPausedDetails() {
        return userPausedDetails;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(bucketCriteria());
        hashCode = 31 * hashCode + Objects.hashCode(hasBucketDefinitions() ? bucketDefinitions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(jobId());
        hashCode = 31 * hashCode + Objects.hashCode(jobStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(jobTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(lastRunErrorStatus());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(userPausedDetails());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof JobSummary)) {
            return false;
        }
        JobSummary other = (JobSummary) obj;
        return Objects.equals(bucketCriteria(), other.bucketCriteria()) && hasBucketDefinitions() == other.hasBucketDefinitions()
                && Objects.equals(bucketDefinitions(), other.bucketDefinitions())
                && Objects.equals(createdAt(), other.createdAt()) && Objects.equals(jobId(), other.jobId())
                && Objects.equals(jobStatusAsString(), other.jobStatusAsString())
                && Objects.equals(jobTypeAsString(), other.jobTypeAsString())
                && Objects.equals(lastRunErrorStatus(), other.lastRunErrorStatus()) && Objects.equals(name(), other.name())
                && Objects.equals(userPausedDetails(), other.userPausedDetails());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("JobSummary").add("BucketCriteria", bucketCriteria())
                .add("BucketDefinitions", hasBucketDefinitions() ? bucketDefinitions() : null).add("CreatedAt", createdAt())
                .add("JobId", jobId()).add("JobStatus", jobStatusAsString()).add("JobType", jobTypeAsString())
                .add("LastRunErrorStatus", lastRunErrorStatus()).add("Name", name())
                .add("UserPausedDetails", userPausedDetails()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "bucketCriteria":
            return Optional.ofNullable(clazz.cast(bucketCriteria()));
        case "bucketDefinitions":
            return Optional.ofNullable(clazz.cast(bucketDefinitions()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "jobId":
            return Optional.ofNullable(clazz.cast(jobId()));
        case "jobStatus":
            return Optional.ofNullable(clazz.cast(jobStatusAsString()));
        case "jobType":
            return Optional.ofNullable(clazz.cast(jobTypeAsString()));
        case "lastRunErrorStatus":
            return Optional.ofNullable(clazz.cast(lastRunErrorStatus()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "userPausedDetails":
            return Optional.ofNullable(clazz.cast(userPausedDetails()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<JobSummary, T> g) {
        return obj -> g.apply((JobSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, JobSummary> {
        /**
         * <p>
         * The property- and tag-based conditions that determine which S3 buckets are included or excluded from the
         * job's analysis. Each time the job runs, the job uses these criteria to determine which buckets to analyze. A
         * job's definition can contain a bucketCriteria object or a bucketDefinitions array, not both.
         * </p>
         * 
         * @param bucketCriteria
         *        The property- and tag-based conditions that determine which S3 buckets are included or excluded from
         *        the job's analysis. Each time the job runs, the job uses these criteria to determine which buckets to
         *        analyze. A job's definition can contain a bucketCriteria object or a bucketDefinitions array, not
         *        both.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucketCriteria(S3BucketCriteriaForJob bucketCriteria);

        /**
         * <p>
         * The property- and tag-based conditions that determine which S3 buckets are included or excluded from the
         * job's analysis. Each time the job runs, the job uses these criteria to determine which buckets to analyze. A
         * job's definition can contain a bucketCriteria object or a bucketDefinitions array, not both.
         * </p>
         * This is a convenience method that creates an instance of the {@link S3BucketCriteriaForJob.Builder} avoiding
         * the need to create one manually via {@link S3BucketCriteriaForJob#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link S3BucketCriteriaForJob.Builder#build()} is called immediately and
         * its result is passed to {@link #bucketCriteria(S3BucketCriteriaForJob)}.
         * 
         * @param bucketCriteria
         *        a consumer that will call methods on {@link S3BucketCriteriaForJob.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #bucketCriteria(S3BucketCriteriaForJob)
         */
        default Builder bucketCriteria(Consumer<S3BucketCriteriaForJob.Builder> bucketCriteria) {
            return bucketCriteria(S3BucketCriteriaForJob.builder().applyMutation(bucketCriteria).build());
        }

        /**
         * <p>
         * An array of objects, one for each Amazon Web Services account that owns specific S3 buckets for the job to
         * analyze. Each object specifies the account ID for an account and one or more buckets to analyze for that
         * account. A job's definition can contain a bucketDefinitions array or a bucketCriteria object, not both.
         * </p>
         * 
         * @param bucketDefinitions
         *        An array of objects, one for each Amazon Web Services account that owns specific S3 buckets for the
         *        job to analyze. Each object specifies the account ID for an account and one or more buckets to analyze
         *        for that account. A job's definition can contain a bucketDefinitions array or a bucketCriteria object,
         *        not both.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucketDefinitions(Collection<S3BucketDefinitionForJob> bucketDefinitions);

        /**
         * <p>
         * An array of objects, one for each Amazon Web Services account that owns specific S3 buckets for the job to
         * analyze. Each object specifies the account ID for an account and one or more buckets to analyze for that
         * account. A job's definition can contain a bucketDefinitions array or a bucketCriteria object, not both.
         * </p>
         * 
         * @param bucketDefinitions
         *        An array of objects, one for each Amazon Web Services account that owns specific S3 buckets for the
         *        job to analyze. Each object specifies the account ID for an account and one or more buckets to analyze
         *        for that account. A job's definition can contain a bucketDefinitions array or a bucketCriteria object,
         *        not both.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucketDefinitions(S3BucketDefinitionForJob... bucketDefinitions);

        /**
         * <p>
         * An array of objects, one for each Amazon Web Services account that owns specific S3 buckets for the job to
         * analyze. Each object specifies the account ID for an account and one or more buckets to analyze for that
         * account. A job's definition can contain a bucketDefinitions array or a bucketCriteria object, not both.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.macie2.model.S3BucketDefinitionForJob.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.macie2.model.S3BucketDefinitionForJob#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.macie2.model.S3BucketDefinitionForJob.Builder#build()} is called
         * immediately and its result is passed to {@link #bucketDefinitions(List<S3BucketDefinitionForJob>)}.
         * 
         * @param bucketDefinitions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.macie2.model.S3BucketDefinitionForJob.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #bucketDefinitions(java.util.Collection<S3BucketDefinitionForJob>)
         */
        Builder bucketDefinitions(Consumer<S3BucketDefinitionForJob.Builder>... bucketDefinitions);

        /**
         * <p>
         * The date and time, in UTC and extended ISO 8601 format, when the job was created.
         * </p>
         * 
         * @param createdAt
         *        The date and time, in UTC and extended ISO 8601 format, when the job was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The unique identifier for the job.
         * </p>
         * 
         * @param jobId
         *        The unique identifier for the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobId(String jobId);

        /**
         * <p>
         * The current status of the job. Possible values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * CANCELLED - You cancelled the job or, if it's a one-time job, you paused the job and didn't resume it within
         * 30 days.
         * </p>
         * </li>
         * <li>
         * <p>
         * COMPLETE - For a one-time job, Amazon Macie finished processing the data specified for the job. This value
         * doesn't apply to recurring jobs.
         * </p>
         * </li>
         * <li>
         * <p>
         * IDLE - For a recurring job, the previous scheduled run is complete and the next scheduled run is pending.
         * This value doesn't apply to one-time jobs.
         * </p>
         * </li>
         * <li>
         * <p>
         * PAUSED - Macie started running the job but additional processing would exceed the monthly sensitive data
         * discovery quota for your account or one or more member accounts that the job analyzes data for.
         * </p>
         * </li>
         * <li>
         * <p>
         * RUNNING - For a one-time job, the job is in progress. For a recurring job, a scheduled run is in progress.
         * </p>
         * </li>
         * <li>
         * <p>
         * USER_PAUSED - You paused the job. If you paused the job while it had a status of RUNNING and you don't resume
         * it within 30 days of pausing it, the job or job run will expire and be cancelled, depending on the job's
         * type. To check the expiration date, refer to the UserPausedDetails.jobExpiresAt property.
         * </p>
         * </li>
         * </ul>
         * 
         * @param jobStatus
         *        The current status of the job. Possible values are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        CANCELLED - You cancelled the job or, if it's a one-time job, you paused the job and didn't resume it
         *        within 30 days.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        COMPLETE - For a one-time job, Amazon Macie finished processing the data specified for the job. This
         *        value doesn't apply to recurring jobs.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        IDLE - For a recurring job, the previous scheduled run is complete and the next scheduled run is
         *        pending. This value doesn't apply to one-time jobs.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        PAUSED - Macie started running the job but additional processing would exceed the monthly sensitive
         *        data discovery quota for your account or one or more member accounts that the job analyzes data for.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        RUNNING - For a one-time job, the job is in progress. For a recurring job, a scheduled run is in
         *        progress.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        USER_PAUSED - You paused the job. If you paused the job while it had a status of RUNNING and you don't
         *        resume it within 30 days of pausing it, the job or job run will expire and be cancelled, depending on
         *        the job's type. To check the expiration date, refer to the UserPausedDetails.jobExpiresAt property.
         *        </p>
         *        </li>
         * @see JobStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobStatus
         */
        Builder jobStatus(String jobStatus);

        /**
         * <p>
         * The current status of the job. Possible values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * CANCELLED - You cancelled the job or, if it's a one-time job, you paused the job and didn't resume it within
         * 30 days.
         * </p>
         * </li>
         * <li>
         * <p>
         * COMPLETE - For a one-time job, Amazon Macie finished processing the data specified for the job. This value
         * doesn't apply to recurring jobs.
         * </p>
         * </li>
         * <li>
         * <p>
         * IDLE - For a recurring job, the previous scheduled run is complete and the next scheduled run is pending.
         * This value doesn't apply to one-time jobs.
         * </p>
         * </li>
         * <li>
         * <p>
         * PAUSED - Macie started running the job but additional processing would exceed the monthly sensitive data
         * discovery quota for your account or one or more member accounts that the job analyzes data for.
         * </p>
         * </li>
         * <li>
         * <p>
         * RUNNING - For a one-time job, the job is in progress. For a recurring job, a scheduled run is in progress.
         * </p>
         * </li>
         * <li>
         * <p>
         * USER_PAUSED - You paused the job. If you paused the job while it had a status of RUNNING and you don't resume
         * it within 30 days of pausing it, the job or job run will expire and be cancelled, depending on the job's
         * type. To check the expiration date, refer to the UserPausedDetails.jobExpiresAt property.
         * </p>
         * </li>
         * </ul>
         * 
         * @param jobStatus
         *        The current status of the job. Possible values are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        CANCELLED - You cancelled the job or, if it's a one-time job, you paused the job and didn't resume it
         *        within 30 days.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        COMPLETE - For a one-time job, Amazon Macie finished processing the data specified for the job. This
         *        value doesn't apply to recurring jobs.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        IDLE - For a recurring job, the previous scheduled run is complete and the next scheduled run is
         *        pending. This value doesn't apply to one-time jobs.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        PAUSED - Macie started running the job but additional processing would exceed the monthly sensitive
         *        data discovery quota for your account or one or more member accounts that the job analyzes data for.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        RUNNING - For a one-time job, the job is in progress. For a recurring job, a scheduled run is in
         *        progress.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        USER_PAUSED - You paused the job. If you paused the job while it had a status of RUNNING and you don't
         *        resume it within 30 days of pausing it, the job or job run will expire and be cancelled, depending on
         *        the job's type. To check the expiration date, refer to the UserPausedDetails.jobExpiresAt property.
         *        </p>
         *        </li>
         * @see JobStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobStatus
         */
        Builder jobStatus(JobStatus jobStatus);

        /**
         * <p>
         * The schedule for running the job. Possible values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * ONE_TIME - The job runs only once.
         * </p>
         * </li>
         * <li>
         * <p>
         * SCHEDULED - The job runs on a daily, weekly, or monthly basis.
         * </p>
         * </li>
         * </ul>
         * 
         * @param jobType
         *        The schedule for running the job. Possible values are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        ONE_TIME - The job runs only once.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        SCHEDULED - The job runs on a daily, weekly, or monthly basis.
         *        </p>
         *        </li>
         * @see JobType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobType
         */
        Builder jobType(String jobType);

        /**
         * <p>
         * The schedule for running the job. Possible values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * ONE_TIME - The job runs only once.
         * </p>
         * </li>
         * <li>
         * <p>
         * SCHEDULED - The job runs on a daily, weekly, or monthly basis.
         * </p>
         * </li>
         * </ul>
         * 
         * @param jobType
         *        The schedule for running the job. Possible values are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        ONE_TIME - The job runs only once.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        SCHEDULED - The job runs on a daily, weekly, or monthly basis.
         *        </p>
         *        </li>
         * @see JobType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobType
         */
        Builder jobType(JobType jobType);

        /**
         * <p>
         * Specifies whether any account- or bucket-level access errors occurred when the job ran. For a recurring job,
         * this value indicates the error status of the job's most recent run.
         * </p>
         * 
         * @param lastRunErrorStatus
         *        Specifies whether any account- or bucket-level access errors occurred when the job ran. For a
         *        recurring job, this value indicates the error status of the job's most recent run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastRunErrorStatus(LastRunErrorStatus lastRunErrorStatus);

        /**
         * <p>
         * Specifies whether any account- or bucket-level access errors occurred when the job ran. For a recurring job,
         * this value indicates the error status of the job's most recent run.
         * </p>
         * This is a convenience method that creates an instance of the {@link LastRunErrorStatus.Builder} avoiding the
         * need to create one manually via {@link LastRunErrorStatus#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link LastRunErrorStatus.Builder#build()} is called immediately and its
         * result is passed to {@link #lastRunErrorStatus(LastRunErrorStatus)}.
         * 
         * @param lastRunErrorStatus
         *        a consumer that will call methods on {@link LastRunErrorStatus.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #lastRunErrorStatus(LastRunErrorStatus)
         */
        default Builder lastRunErrorStatus(Consumer<LastRunErrorStatus.Builder> lastRunErrorStatus) {
            return lastRunErrorStatus(LastRunErrorStatus.builder().applyMutation(lastRunErrorStatus).build());
        }

        /**
         * <p>
         * The custom name of the job.
         * </p>
         * 
         * @param name
         *        The custom name of the job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * If the current status of the job is USER_PAUSED, specifies when the job was paused and when the job or job
         * run will expire and be cancelled if it isn't resumed. This value is present only if the value for jobStatus
         * is USER_PAUSED.
         * </p>
         * 
         * @param userPausedDetails
         *        If the current status of the job is USER_PAUSED, specifies when the job was paused and when the job or
         *        job run will expire and be cancelled if it isn't resumed. This value is present only if the value for
         *        jobStatus is USER_PAUSED.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userPausedDetails(UserPausedDetails userPausedDetails);

        /**
         * <p>
         * If the current status of the job is USER_PAUSED, specifies when the job was paused and when the job or job
         * run will expire and be cancelled if it isn't resumed. This value is present only if the value for jobStatus
         * is USER_PAUSED.
         * </p>
         * This is a convenience method that creates an instance of the {@link UserPausedDetails.Builder} avoiding the
         * need to create one manually via {@link UserPausedDetails#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link UserPausedDetails.Builder#build()} is called immediately and its
         * result is passed to {@link #userPausedDetails(UserPausedDetails)}.
         * 
         * @param userPausedDetails
         *        a consumer that will call methods on {@link UserPausedDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #userPausedDetails(UserPausedDetails)
         */
        default Builder userPausedDetails(Consumer<UserPausedDetails.Builder> userPausedDetails) {
            return userPausedDetails(UserPausedDetails.builder().applyMutation(userPausedDetails).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private S3BucketCriteriaForJob bucketCriteria;

        private List<S3BucketDefinitionForJob> bucketDefinitions = DefaultSdkAutoConstructList.getInstance();

        private Instant createdAt;

        private String jobId;

        private String jobStatus;

        private String jobType;

        private LastRunErrorStatus lastRunErrorStatus;

        private String name;

        private UserPausedDetails userPausedDetails;

        private BuilderImpl() {
        }

        private BuilderImpl(JobSummary model) {
            bucketCriteria(model.bucketCriteria);
            bucketDefinitions(model.bucketDefinitions);
            createdAt(model.createdAt);
            jobId(model.jobId);
            jobStatus(model.jobStatus);
            jobType(model.jobType);
            lastRunErrorStatus(model.lastRunErrorStatus);
            name(model.name);
            userPausedDetails(model.userPausedDetails);
        }

        public final S3BucketCriteriaForJob.Builder getBucketCriteria() {
            return bucketCriteria != null ? bucketCriteria.toBuilder() : null;
        }

        public final void setBucketCriteria(S3BucketCriteriaForJob.BuilderImpl bucketCriteria) {
            this.bucketCriteria = bucketCriteria != null ? bucketCriteria.build() : null;
        }

        @Override
        public final Builder bucketCriteria(S3BucketCriteriaForJob bucketCriteria) {
            this.bucketCriteria = bucketCriteria;
            return this;
        }

        public final List<S3BucketDefinitionForJob.Builder> getBucketDefinitions() {
            List<S3BucketDefinitionForJob.Builder> result = ___listOfS3BucketDefinitionForJobCopier
                    .copyToBuilder(this.bucketDefinitions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setBucketDefinitions(Collection<S3BucketDefinitionForJob.BuilderImpl> bucketDefinitions) {
            this.bucketDefinitions = ___listOfS3BucketDefinitionForJobCopier.copyFromBuilder(bucketDefinitions);
        }

        @Override
        public final Builder bucketDefinitions(Collection<S3BucketDefinitionForJob> bucketDefinitions) {
            this.bucketDefinitions = ___listOfS3BucketDefinitionForJobCopier.copy(bucketDefinitions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder bucketDefinitions(S3BucketDefinitionForJob... bucketDefinitions) {
            bucketDefinitions(Arrays.asList(bucketDefinitions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder bucketDefinitions(Consumer<S3BucketDefinitionForJob.Builder>... bucketDefinitions) {
            bucketDefinitions(Stream.of(bucketDefinitions).map(c -> S3BucketDefinitionForJob.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final String getJobId() {
            return jobId;
        }

        public final void setJobId(String jobId) {
            this.jobId = jobId;
        }

        @Override
        public final Builder jobId(String jobId) {
            this.jobId = jobId;
            return this;
        }

        public final String getJobStatus() {
            return jobStatus;
        }

        public final void setJobStatus(String jobStatus) {
            this.jobStatus = jobStatus;
        }

        @Override
        public final Builder jobStatus(String jobStatus) {
            this.jobStatus = jobStatus;
            return this;
        }

        @Override
        public final Builder jobStatus(JobStatus jobStatus) {
            this.jobStatus(jobStatus == null ? null : jobStatus.toString());
            return this;
        }

        public final String getJobType() {
            return jobType;
        }

        public final void setJobType(String jobType) {
            this.jobType = jobType;
        }

        @Override
        public final Builder jobType(String jobType) {
            this.jobType = jobType;
            return this;
        }

        @Override
        public final Builder jobType(JobType jobType) {
            this.jobType(jobType == null ? null : jobType.toString());
            return this;
        }

        public final LastRunErrorStatus.Builder getLastRunErrorStatus() {
            return lastRunErrorStatus != null ? lastRunErrorStatus.toBuilder() : null;
        }

        public final void setLastRunErrorStatus(LastRunErrorStatus.BuilderImpl lastRunErrorStatus) {
            this.lastRunErrorStatus = lastRunErrorStatus != null ? lastRunErrorStatus.build() : null;
        }

        @Override
        public final Builder lastRunErrorStatus(LastRunErrorStatus lastRunErrorStatus) {
            this.lastRunErrorStatus = lastRunErrorStatus;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final UserPausedDetails.Builder getUserPausedDetails() {
            return userPausedDetails != null ? userPausedDetails.toBuilder() : null;
        }

        public final void setUserPausedDetails(UserPausedDetails.BuilderImpl userPausedDetails) {
            this.userPausedDetails = userPausedDetails != null ? userPausedDetails.build() : null;
        }

        @Override
        public final Builder userPausedDetails(UserPausedDetails userPausedDetails) {
            this.userPausedDetails = userPausedDetails;
            return this;
        }

        @Override
        public JobSummary build() {
            return new JobSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
