/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.macie2.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides statistical data and other information about an S3 bucket that Amazon Macie monitors and analyzes for your
 * account. By default, object count and storage size values include data for object parts that are the result of
 * incomplete multipart uploads. For more information, see <a
 * href="https://docs.aws.amazon.com/macie/latest/user/monitoring-s3-how-it-works.html">How Macie monitors Amazon S3
 * data security</a> in the <i>Amazon Macie User Guide</i>.
 * </p>
 * <p>
 * If an error occurs when Macie attempts to retrieve and process information about the bucket or the bucket's objects,
 * the value for most of these properties is null. Key exceptions are accountId and bucketName. To identify the cause of
 * the error, refer to the errorCode and errorMessage values.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MatchingBucket implements SdkPojo, Serializable, ToCopyableBuilder<MatchingBucket.Builder, MatchingBucket> {
    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("accountId").getter(getter(MatchingBucket::accountId)).setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("accountId").build()).build();

    private static final SdkField<String> AUTOMATED_DISCOVERY_MONITORING_STATUS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("automatedDiscoveryMonitoringStatus")
            .getter(getter(MatchingBucket::automatedDiscoveryMonitoringStatusAsString))
            .setter(setter(Builder::automatedDiscoveryMonitoringStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("automatedDiscoveryMonitoringStatus")
                    .build()).build();

    private static final SdkField<String> BUCKET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("bucketName").getter(getter(MatchingBucket::bucketName)).setter(setter(Builder::bucketName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("bucketName").build()).build();

    private static final SdkField<Long> CLASSIFIABLE_OBJECT_COUNT_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("classifiableObjectCount").getter(getter(MatchingBucket::classifiableObjectCount))
            .setter(setter(Builder::classifiableObjectCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("classifiableObjectCount").build())
            .build();

    private static final SdkField<Long> CLASSIFIABLE_SIZE_IN_BYTES_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("classifiableSizeInBytes").getter(getter(MatchingBucket::classifiableSizeInBytes))
            .setter(setter(Builder::classifiableSizeInBytes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("classifiableSizeInBytes").build())
            .build();

    private static final SdkField<String> ERROR_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("errorCode").getter(getter(MatchingBucket::errorCodeAsString)).setter(setter(Builder::errorCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("errorCode").build()).build();

    private static final SdkField<String> ERROR_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("errorMessage").getter(getter(MatchingBucket::errorMessage)).setter(setter(Builder::errorMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("errorMessage").build()).build();

    private static final SdkField<JobDetails> JOB_DETAILS_FIELD = SdkField.<JobDetails> builder(MarshallingType.SDK_POJO)
            .memberName("jobDetails").getter(getter(MatchingBucket::jobDetails)).setter(setter(Builder::jobDetails))
            .constructor(JobDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("jobDetails").build()).build();

    private static final SdkField<Instant> LAST_AUTOMATED_DISCOVERY_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastAutomatedDiscoveryTime")
            .getter(getter(MatchingBucket::lastAutomatedDiscoveryTime))
            .setter(setter(Builder::lastAutomatedDiscoveryTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastAutomatedDiscoveryTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Long> OBJECT_COUNT_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("objectCount").getter(getter(MatchingBucket::objectCount)).setter(setter(Builder::objectCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("objectCount").build()).build();

    private static final SdkField<ObjectCountByEncryptionType> OBJECT_COUNT_BY_ENCRYPTION_TYPE_FIELD = SdkField
            .<ObjectCountByEncryptionType> builder(MarshallingType.SDK_POJO)
            .memberName("objectCountByEncryptionType")
            .getter(getter(MatchingBucket::objectCountByEncryptionType))
            .setter(setter(Builder::objectCountByEncryptionType))
            .constructor(ObjectCountByEncryptionType::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("objectCountByEncryptionType")
                    .build()).build();

    private static final SdkField<Integer> SENSITIVITY_SCORE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("sensitivityScore").getter(getter(MatchingBucket::sensitivityScore))
            .setter(setter(Builder::sensitivityScore))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sensitivityScore").build()).build();

    private static final SdkField<Long> SIZE_IN_BYTES_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("sizeInBytes").getter(getter(MatchingBucket::sizeInBytes)).setter(setter(Builder::sizeInBytes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sizeInBytes").build()).build();

    private static final SdkField<Long> SIZE_IN_BYTES_COMPRESSED_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("sizeInBytesCompressed").getter(getter(MatchingBucket::sizeInBytesCompressed))
            .setter(setter(Builder::sizeInBytesCompressed))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sizeInBytesCompressed").build())
            .build();

    private static final SdkField<ObjectLevelStatistics> UNCLASSIFIABLE_OBJECT_COUNT_FIELD = SdkField
            .<ObjectLevelStatistics> builder(MarshallingType.SDK_POJO).memberName("unclassifiableObjectCount")
            .getter(getter(MatchingBucket::unclassifiableObjectCount)).setter(setter(Builder::unclassifiableObjectCount))
            .constructor(ObjectLevelStatistics::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("unclassifiableObjectCount").build())
            .build();

    private static final SdkField<ObjectLevelStatistics> UNCLASSIFIABLE_OBJECT_SIZE_IN_BYTES_FIELD = SdkField
            .<ObjectLevelStatistics> builder(MarshallingType.SDK_POJO)
            .memberName("unclassifiableObjectSizeInBytes")
            .getter(getter(MatchingBucket::unclassifiableObjectSizeInBytes))
            .setter(setter(Builder::unclassifiableObjectSizeInBytes))
            .constructor(ObjectLevelStatistics::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("unclassifiableObjectSizeInBytes")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACCOUNT_ID_FIELD,
            AUTOMATED_DISCOVERY_MONITORING_STATUS_FIELD, BUCKET_NAME_FIELD, CLASSIFIABLE_OBJECT_COUNT_FIELD,
            CLASSIFIABLE_SIZE_IN_BYTES_FIELD, ERROR_CODE_FIELD, ERROR_MESSAGE_FIELD, JOB_DETAILS_FIELD,
            LAST_AUTOMATED_DISCOVERY_TIME_FIELD, OBJECT_COUNT_FIELD, OBJECT_COUNT_BY_ENCRYPTION_TYPE_FIELD,
            SENSITIVITY_SCORE_FIELD, SIZE_IN_BYTES_FIELD, SIZE_IN_BYTES_COMPRESSED_FIELD, UNCLASSIFIABLE_OBJECT_COUNT_FIELD,
            UNCLASSIFIABLE_OBJECT_SIZE_IN_BYTES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("accountId", ACCOUNT_ID_FIELD);
                    put("automatedDiscoveryMonitoringStatus", AUTOMATED_DISCOVERY_MONITORING_STATUS_FIELD);
                    put("bucketName", BUCKET_NAME_FIELD);
                    put("classifiableObjectCount", CLASSIFIABLE_OBJECT_COUNT_FIELD);
                    put("classifiableSizeInBytes", CLASSIFIABLE_SIZE_IN_BYTES_FIELD);
                    put("errorCode", ERROR_CODE_FIELD);
                    put("errorMessage", ERROR_MESSAGE_FIELD);
                    put("jobDetails", JOB_DETAILS_FIELD);
                    put("lastAutomatedDiscoveryTime", LAST_AUTOMATED_DISCOVERY_TIME_FIELD);
                    put("objectCount", OBJECT_COUNT_FIELD);
                    put("objectCountByEncryptionType", OBJECT_COUNT_BY_ENCRYPTION_TYPE_FIELD);
                    put("sensitivityScore", SENSITIVITY_SCORE_FIELD);
                    put("sizeInBytes", SIZE_IN_BYTES_FIELD);
                    put("sizeInBytesCompressed", SIZE_IN_BYTES_COMPRESSED_FIELD);
                    put("unclassifiableObjectCount", UNCLASSIFIABLE_OBJECT_COUNT_FIELD);
                    put("unclassifiableObjectSizeInBytes", UNCLASSIFIABLE_OBJECT_SIZE_IN_BYTES_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String accountId;

    private final String automatedDiscoveryMonitoringStatus;

    private final String bucketName;

    private final Long classifiableObjectCount;

    private final Long classifiableSizeInBytes;

    private final String errorCode;

    private final String errorMessage;

    private final JobDetails jobDetails;

    private final Instant lastAutomatedDiscoveryTime;

    private final Long objectCount;

    private final ObjectCountByEncryptionType objectCountByEncryptionType;

    private final Integer sensitivityScore;

    private final Long sizeInBytes;

    private final Long sizeInBytesCompressed;

    private final ObjectLevelStatistics unclassifiableObjectCount;

    private final ObjectLevelStatistics unclassifiableObjectSizeInBytes;

    private MatchingBucket(BuilderImpl builder) {
        this.accountId = builder.accountId;
        this.automatedDiscoveryMonitoringStatus = builder.automatedDiscoveryMonitoringStatus;
        this.bucketName = builder.bucketName;
        this.classifiableObjectCount = builder.classifiableObjectCount;
        this.classifiableSizeInBytes = builder.classifiableSizeInBytes;
        this.errorCode = builder.errorCode;
        this.errorMessage = builder.errorMessage;
        this.jobDetails = builder.jobDetails;
        this.lastAutomatedDiscoveryTime = builder.lastAutomatedDiscoveryTime;
        this.objectCount = builder.objectCount;
        this.objectCountByEncryptionType = builder.objectCountByEncryptionType;
        this.sensitivityScore = builder.sensitivityScore;
        this.sizeInBytes = builder.sizeInBytes;
        this.sizeInBytesCompressed = builder.sizeInBytesCompressed;
        this.unclassifiableObjectCount = builder.unclassifiableObjectCount;
        this.unclassifiableObjectSizeInBytes = builder.unclassifiableObjectSizeInBytes;
    }

    /**
     * <p>
     * The unique identifier for the Amazon Web Services account that owns the bucket.
     * </p>
     * 
     * @return The unique identifier for the Amazon Web Services account that owns the bucket.
     */
    public final String accountId() {
        return accountId;
    }

    /**
     * <p>
     * Specifies whether automated sensitive data discovery is currently configured to analyze objects in the bucket.
     * Possible values are: MONITORED, the bucket is included in analyses; and, NOT_MONITORED, the bucket is excluded
     * from analyses. If automated sensitive data discovery is disabled for your account, this value is NOT_MONITORED.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #automatedDiscoveryMonitoringStatus} will return
     * {@link AutomatedDiscoveryMonitoringStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #automatedDiscoveryMonitoringStatusAsString}.
     * </p>
     * 
     * @return Specifies whether automated sensitive data discovery is currently configured to analyze objects in the
     *         bucket. Possible values are: MONITORED, the bucket is included in analyses; and, NOT_MONITORED, the
     *         bucket is excluded from analyses. If automated sensitive data discovery is disabled for your account,
     *         this value is NOT_MONITORED.
     * @see AutomatedDiscoveryMonitoringStatus
     */
    public final AutomatedDiscoveryMonitoringStatus automatedDiscoveryMonitoringStatus() {
        return AutomatedDiscoveryMonitoringStatus.fromValue(automatedDiscoveryMonitoringStatus);
    }

    /**
     * <p>
     * Specifies whether automated sensitive data discovery is currently configured to analyze objects in the bucket.
     * Possible values are: MONITORED, the bucket is included in analyses; and, NOT_MONITORED, the bucket is excluded
     * from analyses. If automated sensitive data discovery is disabled for your account, this value is NOT_MONITORED.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #automatedDiscoveryMonitoringStatus} will return
     * {@link AutomatedDiscoveryMonitoringStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #automatedDiscoveryMonitoringStatusAsString}.
     * </p>
     * 
     * @return Specifies whether automated sensitive data discovery is currently configured to analyze objects in the
     *         bucket. Possible values are: MONITORED, the bucket is included in analyses; and, NOT_MONITORED, the
     *         bucket is excluded from analyses. If automated sensitive data discovery is disabled for your account,
     *         this value is NOT_MONITORED.
     * @see AutomatedDiscoveryMonitoringStatus
     */
    public final String automatedDiscoveryMonitoringStatusAsString() {
        return automatedDiscoveryMonitoringStatus;
    }

    /**
     * <p>
     * The name of the bucket.
     * </p>
     * 
     * @return The name of the bucket.
     */
    public final String bucketName() {
        return bucketName;
    }

    /**
     * <p>
     * The total number of objects that Amazon Macie can analyze in the bucket. These objects use a supported storage
     * class and have a file name extension for a supported file or storage format.
     * </p>
     * 
     * @return The total number of objects that Amazon Macie can analyze in the bucket. These objects use a supported
     *         storage class and have a file name extension for a supported file or storage format.
     */
    public final Long classifiableObjectCount() {
        return classifiableObjectCount;
    }

    /**
     * <p>
     * The total storage size, in bytes, of the objects that Amazon Macie can analyze in the bucket. These objects use a
     * supported storage class and have a file name extension for a supported file or storage format.
     * </p>
     * <p>
     * If versioning is enabled for the bucket, Macie calculates this value based on the size of the latest version of
     * each applicable object in the bucket. This value doesn't reflect the storage size of all versions of each
     * applicable object in the bucket.
     * </p>
     * 
     * @return The total storage size, in bytes, of the objects that Amazon Macie can analyze in the bucket. These
     *         objects use a supported storage class and have a file name extension for a supported file or storage
     *         format.</p>
     *         <p>
     *         If versioning is enabled for the bucket, Macie calculates this value based on the size of the latest
     *         version of each applicable object in the bucket. This value doesn't reflect the storage size of all
     *         versions of each applicable object in the bucket.
     */
    public final Long classifiableSizeInBytes() {
        return classifiableSizeInBytes;
    }

    /**
     * <p>
     * The error code for an error that prevented Amazon Macie from retrieving and processing information about the
     * bucket and the bucket's objects. If this value is ACCESS_DENIED, Macie doesn't have permission to retrieve the
     * information. For example, the bucket has a restrictive bucket policy and Amazon S3 denied the request. If this
     * value is null, Macie was able to retrieve and process the information.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #errorCode} will
     * return {@link BucketMetadataErrorCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #errorCodeAsString}.
     * </p>
     * 
     * @return The error code for an error that prevented Amazon Macie from retrieving and processing information about
     *         the bucket and the bucket's objects. If this value is ACCESS_DENIED, Macie doesn't have permission to
     *         retrieve the information. For example, the bucket has a restrictive bucket policy and Amazon S3 denied
     *         the request. If this value is null, Macie was able to retrieve and process the information.
     * @see BucketMetadataErrorCode
     */
    public final BucketMetadataErrorCode errorCode() {
        return BucketMetadataErrorCode.fromValue(errorCode);
    }

    /**
     * <p>
     * The error code for an error that prevented Amazon Macie from retrieving and processing information about the
     * bucket and the bucket's objects. If this value is ACCESS_DENIED, Macie doesn't have permission to retrieve the
     * information. For example, the bucket has a restrictive bucket policy and Amazon S3 denied the request. If this
     * value is null, Macie was able to retrieve and process the information.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #errorCode} will
     * return {@link BucketMetadataErrorCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #errorCodeAsString}.
     * </p>
     * 
     * @return The error code for an error that prevented Amazon Macie from retrieving and processing information about
     *         the bucket and the bucket's objects. If this value is ACCESS_DENIED, Macie doesn't have permission to
     *         retrieve the information. For example, the bucket has a restrictive bucket policy and Amazon S3 denied
     *         the request. If this value is null, Macie was able to retrieve and process the information.
     * @see BucketMetadataErrorCode
     */
    public final String errorCodeAsString() {
        return errorCode;
    }

    /**
     * <p>
     * A brief description of the error (errorCode) that prevented Amazon Macie from retrieving and processing
     * information about the bucket and the bucket's objects. This value is null if Macie was able to retrieve and
     * process the information.
     * </p>
     * 
     * @return A brief description of the error (errorCode) that prevented Amazon Macie from retrieving and processing
     *         information about the bucket and the bucket's objects. This value is null if Macie was able to retrieve
     *         and process the information.
     */
    public final String errorMessage() {
        return errorMessage;
    }

    /**
     * <p>
     * Specifies whether any one-time or recurring classification jobs are configured to analyze objects in the bucket,
     * and, if so, the details of the job that ran most recently.
     * </p>
     * 
     * @return Specifies whether any one-time or recurring classification jobs are configured to analyze objects in the
     *         bucket, and, if so, the details of the job that ran most recently.
     */
    public final JobDetails jobDetails() {
        return jobDetails;
    }

    /**
     * <p>
     * The date and time, in UTC and extended ISO 8601 format, when Amazon Macie most recently analyzed objects in the
     * bucket while performing automated sensitive data discovery. This value is null if automated sensitive data
     * discovery is disabled for your account.
     * </p>
     * 
     * @return The date and time, in UTC and extended ISO 8601 format, when Amazon Macie most recently analyzed objects
     *         in the bucket while performing automated sensitive data discovery. This value is null if automated
     *         sensitive data discovery is disabled for your account.
     */
    public final Instant lastAutomatedDiscoveryTime() {
        return lastAutomatedDiscoveryTime;
    }

    /**
     * <p>
     * The total number of objects in the bucket.
     * </p>
     * 
     * @return The total number of objects in the bucket.
     */
    public final Long objectCount() {
        return objectCount;
    }

    /**
     * <p>
     * The total number of objects in the bucket, grouped by server-side encryption type. This includes a grouping that
     * reports the total number of objects that aren't encrypted or use client-side encryption.
     * </p>
     * 
     * @return The total number of objects in the bucket, grouped by server-side encryption type. This includes a
     *         grouping that reports the total number of objects that aren't encrypted or use client-side encryption.
     */
    public final ObjectCountByEncryptionType objectCountByEncryptionType() {
        return objectCountByEncryptionType;
    }

    /**
     * <p>
     * The sensitivity score for the bucket, ranging from -1 (classification error) to 100 (sensitive).
     * </p>
     * <p>
     * If automated sensitive data discovery has never been enabled for your account or it’s been disabled for your
     * organization or your standalone account for more than 30 days, possible values are: 1, the bucket is empty; or,
     * 50, the bucket stores objects but it’s been excluded from recent analyses.
     * </p>
     * 
     * @return The sensitivity score for the bucket, ranging from -1 (classification error) to 100 (sensitive).</p>
     *         <p>
     *         If automated sensitive data discovery has never been enabled for your account or it’s been disabled for
     *         your organization or your standalone account for more than 30 days, possible values are: 1, the bucket is
     *         empty; or, 50, the bucket stores objects but it’s been excluded from recent analyses.
     */
    public final Integer sensitivityScore() {
        return sensitivityScore;
    }

    /**
     * <p>
     * The total storage size, in bytes, of the bucket.
     * </p>
     * <p>
     * If versioning is enabled for the bucket, Amazon Macie calculates this value based on the size of the latest
     * version of each object in the bucket. This value doesn't reflect the storage size of all versions of each object
     * in the bucket.
     * </p>
     * 
     * @return The total storage size, in bytes, of the bucket.</p>
     *         <p>
     *         If versioning is enabled for the bucket, Amazon Macie calculates this value based on the size of the
     *         latest version of each object in the bucket. This value doesn't reflect the storage size of all versions
     *         of each object in the bucket.
     */
    public final Long sizeInBytes() {
        return sizeInBytes;
    }

    /**
     * <p>
     * The total storage size, in bytes, of the objects that are compressed (.gz, .gzip, .zip) files in the bucket.
     * </p>
     * <p>
     * If versioning is enabled for the bucket, Amazon Macie calculates this value based on the size of the latest
     * version of each applicable object in the bucket. This value doesn't reflect the storage size of all versions of
     * each applicable object in the bucket.
     * </p>
     * 
     * @return The total storage size, in bytes, of the objects that are compressed (.gz, .gzip, .zip) files in the
     *         bucket.</p>
     *         <p>
     *         If versioning is enabled for the bucket, Amazon Macie calculates this value based on the size of the
     *         latest version of each applicable object in the bucket. This value doesn't reflect the storage size of
     *         all versions of each applicable object in the bucket.
     */
    public final Long sizeInBytesCompressed() {
        return sizeInBytesCompressed;
    }

    /**
     * <p>
     * The total number of objects that Amazon Macie can't analyze in the bucket. These objects don't use a supported
     * storage class or don't have a file name extension for a supported file or storage format.
     * </p>
     * 
     * @return The total number of objects that Amazon Macie can't analyze in the bucket. These objects don't use a
     *         supported storage class or don't have a file name extension for a supported file or storage format.
     */
    public final ObjectLevelStatistics unclassifiableObjectCount() {
        return unclassifiableObjectCount;
    }

    /**
     * <p>
     * The total storage size, in bytes, of the objects that Amazon Macie can't analyze in the bucket. These objects
     * don't use a supported storage class or don't have a file name extension for a supported file or storage format.
     * </p>
     * 
     * @return The total storage size, in bytes, of the objects that Amazon Macie can't analyze in the bucket. These
     *         objects don't use a supported storage class or don't have a file name extension for a supported file or
     *         storage format.
     */
    public final ObjectLevelStatistics unclassifiableObjectSizeInBytes() {
        return unclassifiableObjectSizeInBytes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(automatedDiscoveryMonitoringStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(bucketName());
        hashCode = 31 * hashCode + Objects.hashCode(classifiableObjectCount());
        hashCode = 31 * hashCode + Objects.hashCode(classifiableSizeInBytes());
        hashCode = 31 * hashCode + Objects.hashCode(errorCodeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(errorMessage());
        hashCode = 31 * hashCode + Objects.hashCode(jobDetails());
        hashCode = 31 * hashCode + Objects.hashCode(lastAutomatedDiscoveryTime());
        hashCode = 31 * hashCode + Objects.hashCode(objectCount());
        hashCode = 31 * hashCode + Objects.hashCode(objectCountByEncryptionType());
        hashCode = 31 * hashCode + Objects.hashCode(sensitivityScore());
        hashCode = 31 * hashCode + Objects.hashCode(sizeInBytes());
        hashCode = 31 * hashCode + Objects.hashCode(sizeInBytesCompressed());
        hashCode = 31 * hashCode + Objects.hashCode(unclassifiableObjectCount());
        hashCode = 31 * hashCode + Objects.hashCode(unclassifiableObjectSizeInBytes());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MatchingBucket)) {
            return false;
        }
        MatchingBucket other = (MatchingBucket) obj;
        return Objects.equals(accountId(), other.accountId())
                && Objects.equals(automatedDiscoveryMonitoringStatusAsString(),
                        other.automatedDiscoveryMonitoringStatusAsString()) && Objects.equals(bucketName(), other.bucketName())
                && Objects.equals(classifiableObjectCount(), other.classifiableObjectCount())
                && Objects.equals(classifiableSizeInBytes(), other.classifiableSizeInBytes())
                && Objects.equals(errorCodeAsString(), other.errorCodeAsString())
                && Objects.equals(errorMessage(), other.errorMessage()) && Objects.equals(jobDetails(), other.jobDetails())
                && Objects.equals(lastAutomatedDiscoveryTime(), other.lastAutomatedDiscoveryTime())
                && Objects.equals(objectCount(), other.objectCount())
                && Objects.equals(objectCountByEncryptionType(), other.objectCountByEncryptionType())
                && Objects.equals(sensitivityScore(), other.sensitivityScore())
                && Objects.equals(sizeInBytes(), other.sizeInBytes())
                && Objects.equals(sizeInBytesCompressed(), other.sizeInBytesCompressed())
                && Objects.equals(unclassifiableObjectCount(), other.unclassifiableObjectCount())
                && Objects.equals(unclassifiableObjectSizeInBytes(), other.unclassifiableObjectSizeInBytes());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("MatchingBucket").add("AccountId", accountId())
                .add("AutomatedDiscoveryMonitoringStatus", automatedDiscoveryMonitoringStatusAsString())
                .add("BucketName", bucketName()).add("ClassifiableObjectCount", classifiableObjectCount())
                .add("ClassifiableSizeInBytes", classifiableSizeInBytes()).add("ErrorCode", errorCodeAsString())
                .add("ErrorMessage", errorMessage()).add("JobDetails", jobDetails())
                .add("LastAutomatedDiscoveryTime", lastAutomatedDiscoveryTime()).add("ObjectCount", objectCount())
                .add("ObjectCountByEncryptionType", objectCountByEncryptionType()).add("SensitivityScore", sensitivityScore())
                .add("SizeInBytes", sizeInBytes()).add("SizeInBytesCompressed", sizeInBytesCompressed())
                .add("UnclassifiableObjectCount", unclassifiableObjectCount())
                .add("UnclassifiableObjectSizeInBytes", unclassifiableObjectSizeInBytes()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "accountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "automatedDiscoveryMonitoringStatus":
            return Optional.ofNullable(clazz.cast(automatedDiscoveryMonitoringStatusAsString()));
        case "bucketName":
            return Optional.ofNullable(clazz.cast(bucketName()));
        case "classifiableObjectCount":
            return Optional.ofNullable(clazz.cast(classifiableObjectCount()));
        case "classifiableSizeInBytes":
            return Optional.ofNullable(clazz.cast(classifiableSizeInBytes()));
        case "errorCode":
            return Optional.ofNullable(clazz.cast(errorCodeAsString()));
        case "errorMessage":
            return Optional.ofNullable(clazz.cast(errorMessage()));
        case "jobDetails":
            return Optional.ofNullable(clazz.cast(jobDetails()));
        case "lastAutomatedDiscoveryTime":
            return Optional.ofNullable(clazz.cast(lastAutomatedDiscoveryTime()));
        case "objectCount":
            return Optional.ofNullable(clazz.cast(objectCount()));
        case "objectCountByEncryptionType":
            return Optional.ofNullable(clazz.cast(objectCountByEncryptionType()));
        case "sensitivityScore":
            return Optional.ofNullable(clazz.cast(sensitivityScore()));
        case "sizeInBytes":
            return Optional.ofNullable(clazz.cast(sizeInBytes()));
        case "sizeInBytesCompressed":
            return Optional.ofNullable(clazz.cast(sizeInBytesCompressed()));
        case "unclassifiableObjectCount":
            return Optional.ofNullable(clazz.cast(unclassifiableObjectCount()));
        case "unclassifiableObjectSizeInBytes":
            return Optional.ofNullable(clazz.cast(unclassifiableObjectSizeInBytes()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<MatchingBucket, T> g) {
        return obj -> g.apply((MatchingBucket) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MatchingBucket> {
        /**
         * <p>
         * The unique identifier for the Amazon Web Services account that owns the bucket.
         * </p>
         * 
         * @param accountId
         *        The unique identifier for the Amazon Web Services account that owns the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * Specifies whether automated sensitive data discovery is currently configured to analyze objects in the
         * bucket. Possible values are: MONITORED, the bucket is included in analyses; and, NOT_MONITORED, the bucket is
         * excluded from analyses. If automated sensitive data discovery is disabled for your account, this value is
         * NOT_MONITORED.
         * </p>
         * 
         * @param automatedDiscoveryMonitoringStatus
         *        Specifies whether automated sensitive data discovery is currently configured to analyze objects in the
         *        bucket. Possible values are: MONITORED, the bucket is included in analyses; and, NOT_MONITORED, the
         *        bucket is excluded from analyses. If automated sensitive data discovery is disabled for your account,
         *        this value is NOT_MONITORED.
         * @see AutomatedDiscoveryMonitoringStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AutomatedDiscoveryMonitoringStatus
         */
        Builder automatedDiscoveryMonitoringStatus(String automatedDiscoveryMonitoringStatus);

        /**
         * <p>
         * Specifies whether automated sensitive data discovery is currently configured to analyze objects in the
         * bucket. Possible values are: MONITORED, the bucket is included in analyses; and, NOT_MONITORED, the bucket is
         * excluded from analyses. If automated sensitive data discovery is disabled for your account, this value is
         * NOT_MONITORED.
         * </p>
         * 
         * @param automatedDiscoveryMonitoringStatus
         *        Specifies whether automated sensitive data discovery is currently configured to analyze objects in the
         *        bucket. Possible values are: MONITORED, the bucket is included in analyses; and, NOT_MONITORED, the
         *        bucket is excluded from analyses. If automated sensitive data discovery is disabled for your account,
         *        this value is NOT_MONITORED.
         * @see AutomatedDiscoveryMonitoringStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AutomatedDiscoveryMonitoringStatus
         */
        Builder automatedDiscoveryMonitoringStatus(AutomatedDiscoveryMonitoringStatus automatedDiscoveryMonitoringStatus);

        /**
         * <p>
         * The name of the bucket.
         * </p>
         * 
         * @param bucketName
         *        The name of the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucketName(String bucketName);

        /**
         * <p>
         * The total number of objects that Amazon Macie can analyze in the bucket. These objects use a supported
         * storage class and have a file name extension for a supported file or storage format.
         * </p>
         * 
         * @param classifiableObjectCount
         *        The total number of objects that Amazon Macie can analyze in the bucket. These objects use a supported
         *        storage class and have a file name extension for a supported file or storage format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder classifiableObjectCount(Long classifiableObjectCount);

        /**
         * <p>
         * The total storage size, in bytes, of the objects that Amazon Macie can analyze in the bucket. These objects
         * use a supported storage class and have a file name extension for a supported file or storage format.
         * </p>
         * <p>
         * If versioning is enabled for the bucket, Macie calculates this value based on the size of the latest version
         * of each applicable object in the bucket. This value doesn't reflect the storage size of all versions of each
         * applicable object in the bucket.
         * </p>
         * 
         * @param classifiableSizeInBytes
         *        The total storage size, in bytes, of the objects that Amazon Macie can analyze in the bucket. These
         *        objects use a supported storage class and have a file name extension for a supported file or storage
         *        format.</p>
         *        <p>
         *        If versioning is enabled for the bucket, Macie calculates this value based on the size of the latest
         *        version of each applicable object in the bucket. This value doesn't reflect the storage size of all
         *        versions of each applicable object in the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder classifiableSizeInBytes(Long classifiableSizeInBytes);

        /**
         * <p>
         * The error code for an error that prevented Amazon Macie from retrieving and processing information about the
         * bucket and the bucket's objects. If this value is ACCESS_DENIED, Macie doesn't have permission to retrieve
         * the information. For example, the bucket has a restrictive bucket policy and Amazon S3 denied the request. If
         * this value is null, Macie was able to retrieve and process the information.
         * </p>
         * 
         * @param errorCode
         *        The error code for an error that prevented Amazon Macie from retrieving and processing information
         *        about the bucket and the bucket's objects. If this value is ACCESS_DENIED, Macie doesn't have
         *        permission to retrieve the information. For example, the bucket has a restrictive bucket policy and
         *        Amazon S3 denied the request. If this value is null, Macie was able to retrieve and process the
         *        information.
         * @see BucketMetadataErrorCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BucketMetadataErrorCode
         */
        Builder errorCode(String errorCode);

        /**
         * <p>
         * The error code for an error that prevented Amazon Macie from retrieving and processing information about the
         * bucket and the bucket's objects. If this value is ACCESS_DENIED, Macie doesn't have permission to retrieve
         * the information. For example, the bucket has a restrictive bucket policy and Amazon S3 denied the request. If
         * this value is null, Macie was able to retrieve and process the information.
         * </p>
         * 
         * @param errorCode
         *        The error code for an error that prevented Amazon Macie from retrieving and processing information
         *        about the bucket and the bucket's objects. If this value is ACCESS_DENIED, Macie doesn't have
         *        permission to retrieve the information. For example, the bucket has a restrictive bucket policy and
         *        Amazon S3 denied the request. If this value is null, Macie was able to retrieve and process the
         *        information.
         * @see BucketMetadataErrorCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BucketMetadataErrorCode
         */
        Builder errorCode(BucketMetadataErrorCode errorCode);

        /**
         * <p>
         * A brief description of the error (errorCode) that prevented Amazon Macie from retrieving and processing
         * information about the bucket and the bucket's objects. This value is null if Macie was able to retrieve and
         * process the information.
         * </p>
         * 
         * @param errorMessage
         *        A brief description of the error (errorCode) that prevented Amazon Macie from retrieving and
         *        processing information about the bucket and the bucket's objects. This value is null if Macie was able
         *        to retrieve and process the information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder errorMessage(String errorMessage);

        /**
         * <p>
         * Specifies whether any one-time or recurring classification jobs are configured to analyze objects in the
         * bucket, and, if so, the details of the job that ran most recently.
         * </p>
         * 
         * @param jobDetails
         *        Specifies whether any one-time or recurring classification jobs are configured to analyze objects in
         *        the bucket, and, if so, the details of the job that ran most recently.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobDetails(JobDetails jobDetails);

        /**
         * <p>
         * Specifies whether any one-time or recurring classification jobs are configured to analyze objects in the
         * bucket, and, if so, the details of the job that ran most recently.
         * </p>
         * This is a convenience method that creates an instance of the {@link JobDetails.Builder} avoiding the need to
         * create one manually via {@link JobDetails#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link JobDetails.Builder#build()} is called immediately and its result
         * is passed to {@link #jobDetails(JobDetails)}.
         * 
         * @param jobDetails
         *        a consumer that will call methods on {@link JobDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #jobDetails(JobDetails)
         */
        default Builder jobDetails(Consumer<JobDetails.Builder> jobDetails) {
            return jobDetails(JobDetails.builder().applyMutation(jobDetails).build());
        }

        /**
         * <p>
         * The date and time, in UTC and extended ISO 8601 format, when Amazon Macie most recently analyzed objects in
         * the bucket while performing automated sensitive data discovery. This value is null if automated sensitive
         * data discovery is disabled for your account.
         * </p>
         * 
         * @param lastAutomatedDiscoveryTime
         *        The date and time, in UTC and extended ISO 8601 format, when Amazon Macie most recently analyzed
         *        objects in the bucket while performing automated sensitive data discovery. This value is null if
         *        automated sensitive data discovery is disabled for your account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastAutomatedDiscoveryTime(Instant lastAutomatedDiscoveryTime);

        /**
         * <p>
         * The total number of objects in the bucket.
         * </p>
         * 
         * @param objectCount
         *        The total number of objects in the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder objectCount(Long objectCount);

        /**
         * <p>
         * The total number of objects in the bucket, grouped by server-side encryption type. This includes a grouping
         * that reports the total number of objects that aren't encrypted or use client-side encryption.
         * </p>
         * 
         * @param objectCountByEncryptionType
         *        The total number of objects in the bucket, grouped by server-side encryption type. This includes a
         *        grouping that reports the total number of objects that aren't encrypted or use client-side encryption.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder objectCountByEncryptionType(ObjectCountByEncryptionType objectCountByEncryptionType);

        /**
         * <p>
         * The total number of objects in the bucket, grouped by server-side encryption type. This includes a grouping
         * that reports the total number of objects that aren't encrypted or use client-side encryption.
         * </p>
         * This is a convenience method that creates an instance of the {@link ObjectCountByEncryptionType.Builder}
         * avoiding the need to create one manually via {@link ObjectCountByEncryptionType#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ObjectCountByEncryptionType.Builder#build()} is called
         * immediately and its result is passed to {@link #objectCountByEncryptionType(ObjectCountByEncryptionType)}.
         * 
         * @param objectCountByEncryptionType
         *        a consumer that will call methods on {@link ObjectCountByEncryptionType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #objectCountByEncryptionType(ObjectCountByEncryptionType)
         */
        default Builder objectCountByEncryptionType(Consumer<ObjectCountByEncryptionType.Builder> objectCountByEncryptionType) {
            return objectCountByEncryptionType(ObjectCountByEncryptionType.builder().applyMutation(objectCountByEncryptionType)
                    .build());
        }

        /**
         * <p>
         * The sensitivity score for the bucket, ranging from -1 (classification error) to 100 (sensitive).
         * </p>
         * <p>
         * If automated sensitive data discovery has never been enabled for your account or it’s been disabled for your
         * organization or your standalone account for more than 30 days, possible values are: 1, the bucket is empty;
         * or, 50, the bucket stores objects but it’s been excluded from recent analyses.
         * </p>
         * 
         * @param sensitivityScore
         *        The sensitivity score for the bucket, ranging from -1 (classification error) to 100 (sensitive).</p>
         *        <p>
         *        If automated sensitive data discovery has never been enabled for your account or it’s been disabled
         *        for your organization or your standalone account for more than 30 days, possible values are: 1, the
         *        bucket is empty; or, 50, the bucket stores objects but it’s been excluded from recent analyses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sensitivityScore(Integer sensitivityScore);

        /**
         * <p>
         * The total storage size, in bytes, of the bucket.
         * </p>
         * <p>
         * If versioning is enabled for the bucket, Amazon Macie calculates this value based on the size of the latest
         * version of each object in the bucket. This value doesn't reflect the storage size of all versions of each
         * object in the bucket.
         * </p>
         * 
         * @param sizeInBytes
         *        The total storage size, in bytes, of the bucket.</p>
         *        <p>
         *        If versioning is enabled for the bucket, Amazon Macie calculates this value based on the size of the
         *        latest version of each object in the bucket. This value doesn't reflect the storage size of all
         *        versions of each object in the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sizeInBytes(Long sizeInBytes);

        /**
         * <p>
         * The total storage size, in bytes, of the objects that are compressed (.gz, .gzip, .zip) files in the bucket.
         * </p>
         * <p>
         * If versioning is enabled for the bucket, Amazon Macie calculates this value based on the size of the latest
         * version of each applicable object in the bucket. This value doesn't reflect the storage size of all versions
         * of each applicable object in the bucket.
         * </p>
         * 
         * @param sizeInBytesCompressed
         *        The total storage size, in bytes, of the objects that are compressed (.gz, .gzip, .zip) files in the
         *        bucket.</p>
         *        <p>
         *        If versioning is enabled for the bucket, Amazon Macie calculates this value based on the size of the
         *        latest version of each applicable object in the bucket. This value doesn't reflect the storage size of
         *        all versions of each applicable object in the bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sizeInBytesCompressed(Long sizeInBytesCompressed);

        /**
         * <p>
         * The total number of objects that Amazon Macie can't analyze in the bucket. These objects don't use a
         * supported storage class or don't have a file name extension for a supported file or storage format.
         * </p>
         * 
         * @param unclassifiableObjectCount
         *        The total number of objects that Amazon Macie can't analyze in the bucket. These objects don't use a
         *        supported storage class or don't have a file name extension for a supported file or storage format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder unclassifiableObjectCount(ObjectLevelStatistics unclassifiableObjectCount);

        /**
         * <p>
         * The total number of objects that Amazon Macie can't analyze in the bucket. These objects don't use a
         * supported storage class or don't have a file name extension for a supported file or storage format.
         * </p>
         * This is a convenience method that creates an instance of the {@link ObjectLevelStatistics.Builder} avoiding
         * the need to create one manually via {@link ObjectLevelStatistics#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ObjectLevelStatistics.Builder#build()} is called immediately and
         * its result is passed to {@link #unclassifiableObjectCount(ObjectLevelStatistics)}.
         * 
         * @param unclassifiableObjectCount
         *        a consumer that will call methods on {@link ObjectLevelStatistics.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #unclassifiableObjectCount(ObjectLevelStatistics)
         */
        default Builder unclassifiableObjectCount(Consumer<ObjectLevelStatistics.Builder> unclassifiableObjectCount) {
            return unclassifiableObjectCount(ObjectLevelStatistics.builder().applyMutation(unclassifiableObjectCount).build());
        }

        /**
         * <p>
         * The total storage size, in bytes, of the objects that Amazon Macie can't analyze in the bucket. These objects
         * don't use a supported storage class or don't have a file name extension for a supported file or storage
         * format.
         * </p>
         * 
         * @param unclassifiableObjectSizeInBytes
         *        The total storage size, in bytes, of the objects that Amazon Macie can't analyze in the bucket. These
         *        objects don't use a supported storage class or don't have a file name extension for a supported file
         *        or storage format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder unclassifiableObjectSizeInBytes(ObjectLevelStatistics unclassifiableObjectSizeInBytes);

        /**
         * <p>
         * The total storage size, in bytes, of the objects that Amazon Macie can't analyze in the bucket. These objects
         * don't use a supported storage class or don't have a file name extension for a supported file or storage
         * format.
         * </p>
         * This is a convenience method that creates an instance of the {@link ObjectLevelStatistics.Builder} avoiding
         * the need to create one manually via {@link ObjectLevelStatistics#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ObjectLevelStatistics.Builder#build()} is called immediately and
         * its result is passed to {@link #unclassifiableObjectSizeInBytes(ObjectLevelStatistics)}.
         * 
         * @param unclassifiableObjectSizeInBytes
         *        a consumer that will call methods on {@link ObjectLevelStatistics.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #unclassifiableObjectSizeInBytes(ObjectLevelStatistics)
         */
        default Builder unclassifiableObjectSizeInBytes(Consumer<ObjectLevelStatistics.Builder> unclassifiableObjectSizeInBytes) {
            return unclassifiableObjectSizeInBytes(ObjectLevelStatistics.builder().applyMutation(unclassifiableObjectSizeInBytes)
                    .build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String accountId;

        private String automatedDiscoveryMonitoringStatus;

        private String bucketName;

        private Long classifiableObjectCount;

        private Long classifiableSizeInBytes;

        private String errorCode;

        private String errorMessage;

        private JobDetails jobDetails;

        private Instant lastAutomatedDiscoveryTime;

        private Long objectCount;

        private ObjectCountByEncryptionType objectCountByEncryptionType;

        private Integer sensitivityScore;

        private Long sizeInBytes;

        private Long sizeInBytesCompressed;

        private ObjectLevelStatistics unclassifiableObjectCount;

        private ObjectLevelStatistics unclassifiableObjectSizeInBytes;

        private BuilderImpl() {
        }

        private BuilderImpl(MatchingBucket model) {
            accountId(model.accountId);
            automatedDiscoveryMonitoringStatus(model.automatedDiscoveryMonitoringStatus);
            bucketName(model.bucketName);
            classifiableObjectCount(model.classifiableObjectCount);
            classifiableSizeInBytes(model.classifiableSizeInBytes);
            errorCode(model.errorCode);
            errorMessage(model.errorMessage);
            jobDetails(model.jobDetails);
            lastAutomatedDiscoveryTime(model.lastAutomatedDiscoveryTime);
            objectCount(model.objectCount);
            objectCountByEncryptionType(model.objectCountByEncryptionType);
            sensitivityScore(model.sensitivityScore);
            sizeInBytes(model.sizeInBytes);
            sizeInBytesCompressed(model.sizeInBytesCompressed);
            unclassifiableObjectCount(model.unclassifiableObjectCount);
            unclassifiableObjectSizeInBytes(model.unclassifiableObjectSizeInBytes);
        }

        public final String getAccountId() {
            return accountId;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final String getAutomatedDiscoveryMonitoringStatus() {
            return automatedDiscoveryMonitoringStatus;
        }

        public final void setAutomatedDiscoveryMonitoringStatus(String automatedDiscoveryMonitoringStatus) {
            this.automatedDiscoveryMonitoringStatus = automatedDiscoveryMonitoringStatus;
        }

        @Override
        public final Builder automatedDiscoveryMonitoringStatus(String automatedDiscoveryMonitoringStatus) {
            this.automatedDiscoveryMonitoringStatus = automatedDiscoveryMonitoringStatus;
            return this;
        }

        @Override
        public final Builder automatedDiscoveryMonitoringStatus(
                AutomatedDiscoveryMonitoringStatus automatedDiscoveryMonitoringStatus) {
            this.automatedDiscoveryMonitoringStatus(automatedDiscoveryMonitoringStatus == null ? null
                    : automatedDiscoveryMonitoringStatus.toString());
            return this;
        }

        public final String getBucketName() {
            return bucketName;
        }

        public final void setBucketName(String bucketName) {
            this.bucketName = bucketName;
        }

        @Override
        public final Builder bucketName(String bucketName) {
            this.bucketName = bucketName;
            return this;
        }

        public final Long getClassifiableObjectCount() {
            return classifiableObjectCount;
        }

        public final void setClassifiableObjectCount(Long classifiableObjectCount) {
            this.classifiableObjectCount = classifiableObjectCount;
        }

        @Override
        public final Builder classifiableObjectCount(Long classifiableObjectCount) {
            this.classifiableObjectCount = classifiableObjectCount;
            return this;
        }

        public final Long getClassifiableSizeInBytes() {
            return classifiableSizeInBytes;
        }

        public final void setClassifiableSizeInBytes(Long classifiableSizeInBytes) {
            this.classifiableSizeInBytes = classifiableSizeInBytes;
        }

        @Override
        public final Builder classifiableSizeInBytes(Long classifiableSizeInBytes) {
            this.classifiableSizeInBytes = classifiableSizeInBytes;
            return this;
        }

        public final String getErrorCode() {
            return errorCode;
        }

        public final void setErrorCode(String errorCode) {
            this.errorCode = errorCode;
        }

        @Override
        public final Builder errorCode(String errorCode) {
            this.errorCode = errorCode;
            return this;
        }

        @Override
        public final Builder errorCode(BucketMetadataErrorCode errorCode) {
            this.errorCode(errorCode == null ? null : errorCode.toString());
            return this;
        }

        public final String getErrorMessage() {
            return errorMessage;
        }

        public final void setErrorMessage(String errorMessage) {
            this.errorMessage = errorMessage;
        }

        @Override
        public final Builder errorMessage(String errorMessage) {
            this.errorMessage = errorMessage;
            return this;
        }

        public final JobDetails.Builder getJobDetails() {
            return jobDetails != null ? jobDetails.toBuilder() : null;
        }

        public final void setJobDetails(JobDetails.BuilderImpl jobDetails) {
            this.jobDetails = jobDetails != null ? jobDetails.build() : null;
        }

        @Override
        public final Builder jobDetails(JobDetails jobDetails) {
            this.jobDetails = jobDetails;
            return this;
        }

        public final Instant getLastAutomatedDiscoveryTime() {
            return lastAutomatedDiscoveryTime;
        }

        public final void setLastAutomatedDiscoveryTime(Instant lastAutomatedDiscoveryTime) {
            this.lastAutomatedDiscoveryTime = lastAutomatedDiscoveryTime;
        }

        @Override
        public final Builder lastAutomatedDiscoveryTime(Instant lastAutomatedDiscoveryTime) {
            this.lastAutomatedDiscoveryTime = lastAutomatedDiscoveryTime;
            return this;
        }

        public final Long getObjectCount() {
            return objectCount;
        }

        public final void setObjectCount(Long objectCount) {
            this.objectCount = objectCount;
        }

        @Override
        public final Builder objectCount(Long objectCount) {
            this.objectCount = objectCount;
            return this;
        }

        public final ObjectCountByEncryptionType.Builder getObjectCountByEncryptionType() {
            return objectCountByEncryptionType != null ? objectCountByEncryptionType.toBuilder() : null;
        }

        public final void setObjectCountByEncryptionType(ObjectCountByEncryptionType.BuilderImpl objectCountByEncryptionType) {
            this.objectCountByEncryptionType = objectCountByEncryptionType != null ? objectCountByEncryptionType.build() : null;
        }

        @Override
        public final Builder objectCountByEncryptionType(ObjectCountByEncryptionType objectCountByEncryptionType) {
            this.objectCountByEncryptionType = objectCountByEncryptionType;
            return this;
        }

        public final Integer getSensitivityScore() {
            return sensitivityScore;
        }

        public final void setSensitivityScore(Integer sensitivityScore) {
            this.sensitivityScore = sensitivityScore;
        }

        @Override
        public final Builder sensitivityScore(Integer sensitivityScore) {
            this.sensitivityScore = sensitivityScore;
            return this;
        }

        public final Long getSizeInBytes() {
            return sizeInBytes;
        }

        public final void setSizeInBytes(Long sizeInBytes) {
            this.sizeInBytes = sizeInBytes;
        }

        @Override
        public final Builder sizeInBytes(Long sizeInBytes) {
            this.sizeInBytes = sizeInBytes;
            return this;
        }

        public final Long getSizeInBytesCompressed() {
            return sizeInBytesCompressed;
        }

        public final void setSizeInBytesCompressed(Long sizeInBytesCompressed) {
            this.sizeInBytesCompressed = sizeInBytesCompressed;
        }

        @Override
        public final Builder sizeInBytesCompressed(Long sizeInBytesCompressed) {
            this.sizeInBytesCompressed = sizeInBytesCompressed;
            return this;
        }

        public final ObjectLevelStatistics.Builder getUnclassifiableObjectCount() {
            return unclassifiableObjectCount != null ? unclassifiableObjectCount.toBuilder() : null;
        }

        public final void setUnclassifiableObjectCount(ObjectLevelStatistics.BuilderImpl unclassifiableObjectCount) {
            this.unclassifiableObjectCount = unclassifiableObjectCount != null ? unclassifiableObjectCount.build() : null;
        }

        @Override
        public final Builder unclassifiableObjectCount(ObjectLevelStatistics unclassifiableObjectCount) {
            this.unclassifiableObjectCount = unclassifiableObjectCount;
            return this;
        }

        public final ObjectLevelStatistics.Builder getUnclassifiableObjectSizeInBytes() {
            return unclassifiableObjectSizeInBytes != null ? unclassifiableObjectSizeInBytes.toBuilder() : null;
        }

        public final void setUnclassifiableObjectSizeInBytes(ObjectLevelStatistics.BuilderImpl unclassifiableObjectSizeInBytes) {
            this.unclassifiableObjectSizeInBytes = unclassifiableObjectSizeInBytes != null ? unclassifiableObjectSizeInBytes
                    .build() : null;
        }

        @Override
        public final Builder unclassifiableObjectSizeInBytes(ObjectLevelStatistics unclassifiableObjectSizeInBytes) {
            this.unclassifiableObjectSizeInBytes = unclassifiableObjectSizeInBytes;
            return this;
        }

        @Override
        public MatchingBucket build() {
            return new MatchingBucket(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
